/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 


/*
 * otAfterBoot.c  -- remove RCS lock from all CRs after server boot
 *
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <time.h>
#include <varargs.h>


#ifndef LINT
static char RCSid_otAfterBoot[] =
    "$RCSfile: otAfterBoot.c,v $ $Revision: 1.1.4.2 $ $Date: 1993/12/01 20:50:35 $";
#endif

int     doOnePrj();
int	readTopNumber(); /* get the highest number for a project */
void	unlockOneCR();
void    logMsg();       /* place an informational message in the log file */
void    error();	/* fatal error msg */ 

#define SHORTSTR 20
#define PATHLEN 100
#define COMMAND 500

static  char  logFile[PATHLEN];  /* Unlock log file full path-name */
static  char  projBase[PATHLEN];  /* Base directory path */
static  char  rcsPath[PATHLEN];   /* rcs commands location */
static  FILE *lfp;          /* otUnlock.log log file */



/*
 *                      m a i n
 *
 *  For each project from the list of projects passed via command line:
 *    - go through d00-d99/d00-d99 directories to find locked CRs.
 *      For each locked CR:
 *	  - get the owner of the lock
 *	  - su to the owner
 *        - remove RCS lock
 *        - update otUnlock.log file
 */
int 
main(argc, argv, envp)
int argc;
char ** argv;
char ** envp;
{
    char *projList, *prj;	/* list of projects to go through */
    struct stat stat_buf;


/* Parse command line */

    if (argc != 3) 
	error("Usage: %s -p<list of projets> <base directory path>\n", argv[0]);
    
    if ( (argv[1][0] != '-') || (argv[1][1] != 'p') || (argv[2][0] != '/') ) 
	error("Usage: %s -p<list of projets> <base directory path>\n", argv[0]);
    
    if ( !(projList = (char *)malloc(strlen(argv[1] + 1))) ) 
	error(stderr, "Malloc error - list of projects (%s)\n", argv[1]);

    strcpy(projList, &argv[1][2]);
    strcpy(projBase, argv[2]);
    

/* Open otUnlock.log file */

    sprintf(logFile, "%s/otUnlock.log", projBase); 
    if ((lfp = fopen(logFile, "a")) == NULL) 
        error("can't append to log file (%s)\n", logFile);
    
/* Place "start" message into the log file */
    
    logMsg("========== started otAfterBoot for '%s' projects", projList);


/* Locate RCS commands */

    if ( !((stat("/usr/local/bin/rcs", &stat_buf) == 0) && 
		(stat_buf.st_mode & S_IFREG)) )  {
	logMsg("***** no rcs commands in the /usr/local/bin *****", "");
	error("no rcs command in the /usr/local/bin\n");
    }
    strcpy(rcsPath, "/usr/local/bin");


/* Get next project name */

    for (prj=strtok(projList, ","); prj; prj=strtok(0, ",") )  {
	if (doOnePrj(prj)) 
	    logMsg("***** error processing '%s' project *****", prj);
    }

    logMsg("--------------------------- otAfterBoot completed", "");
    free(projList);

    exit(0);
}


int
doOnePrj(proj)
char * proj;
{
    register char *cp;
    long  topNumber, ti;
    int   k, oloop, iloop;
    char  path[PATHLEN];
    char  oneCRnum[SHORTSTR];
    char  fullpath[PATHLEN];
    DIR  * dirp;
    struct dirent *dp;
    struct stat statbuf;

    int errCode = 0;

    logMsg("Processing '%s' project", proj);

/* Get the highest CR number for a project */

    if (errCode = readTopNumber(proj, &topNumber)) {
	logMsg("***** can't read number file for '%s' project *****", proj);
	return errCode;
    }
    logMsg("                         (topNumber = %d)", topNumber); 


/* Get directory for the highest number */

    ti = topNumber - ((topNumber / 10000) * 10000);
/*
    oloop = topNumber / 10000;
    iloop = ti / 100;
*/
    k = topNumber / 100;
    oloop = k / 100;
    iloop = k - oloop*100;

/* Loop through data directories */

    for (;  (oloop >= 00) && !errCode;  oloop--)  {

	for (;  (iloop >= 00) && !errCode;  iloop--)  {

	    sprintf(path, "%s/%s/d%02d/d%02d", projBase, proj, oloop, iloop);

	/* Go down directory to find locked files */

	    if ( (dirp = opendir(path)) == NULL) {
		logMsg("***** opendir() error on '%s' directory", path);
		continue;
	    }
	    
	    while (dp = readdir(dirp))  {
		if ( (strncmp(dp->d_name, "c", 1)) ||
		     (!strstr(dp->d_name, ".edit")) || 
		     (strlen(dp->d_name) != 12) ) 
		    continue;

		sprintf(fullpath, "%s/%s", path, dp->d_name);
		strcpy(oneCRnum, dp->d_name);
		if (cp = strrchr(oneCRnum, '.')) 
		    *cp = 0;
		
		logMsg("  %s", oneCRnum);

	    /* Get the owner of the lock */

		if (stat(fullpath, &statbuf) == -1) {
		    logMsg("  *** stat() error %d ***", errno);
		    continue;
		}

		unlockOneCR(oneCRnum, statbuf, path);
	    }

	    if (dirp)
		closedir(dirp);
	}

	iloop = 99; /* initialize for the next oloop loop */
    }
    return errCode;
}



void
unlockOneCR(aCR, sbuf, dname)
char * aCR;
struct stat sbuf;
char * dname;
{
    char *userName;
    int childpid, childstat;
    char  command[COMMAND];

    struct passwd *getpwuid(), *pw;


    if ( (childpid = fork()) == -1) {
	logMsg("  ***  fork() error %d ***", errno);
	return;
    }

    else if (childpid == 0) {			/* ***** child ***** */

	char  logEnv[PATHLEN];	  /* do NOT reuse logEnv and userEnv buffers */
	char  userEnv[PATHLEN];	  /* it's putenv() requirement !!! */

    /* Set group and user id's to the owner of a file */

	if (setgid(sbuf.st_gid) == -1) {
	    logMsg("  *** setgid() error %d ***", errno);
	    exit(1);
	}
	if (setuid(sbuf.st_uid) == -1) {
	    logMsg("  *** setuid() error %d ***", errno);
	    exit(1);
        }

    /* Set LOGNAME and USER to the owner of a file -- to make RCS happy */

	if (!(pw = getpwuid(sbuf.st_uid))) {
	    logMsg("  *** getpwuid() error %d ***", errno);
	    exit(1);
	}

	userName = pw->pw_name;
	sprintf(command, "uid is %d, euid is %d, gid is %d, owner is %s", 
				getuid(), geteuid(), getgid(), userName);
	logMsg("   (%s)", command);

	sprintf(logEnv, "LOGNAME=%s", userName);
	sprintf(userEnv, "USER=%s", userName);
	if ( (putenv(logEnv)) || (putenv(userEnv)) ) {
	    logMsg("  *** putenv() error ***", "");
	    exit(1);
	}

    /* And finally unlock CR */

	sprintf(command, "cd %s; %s/rcs -u -q %s; rm -f %s*; %s/co -q %s",
	    		dname, rcsPath, aCR, aCR, rcsPath, aCR);

	if (system(command))  {
	   logMsg("  *** system() error processing file '%s' ***", aCR);	
	   exit(1);
	}

	exit(0); 
    }
    else {					/* ***** parent ***** */

    /* Let parent process wait for the child */

    	if (wait(&childstat) != childpid) 
	    logMsg("  *** wait() error ***", "");
    
        return;
    }
}



int
readTopNumber(proj, number)
char * proj;
long	* number;
{
    char TheNumber[PATHLEN];
    FILE *fp;
    int i = 0;

    int errCode = 0;

    sprintf(TheNumber, "%s/%s/number", projBase, proj);

    if ( ((fp = fopen(TheNumber, "r")) == NULL) ||
	 	    	    !(i = fread(TheNumber, 1, PATHLEN, fp)) ) 
	errCode = 1;
    
    fclose(fp);

    if (!errCode)
        *number = atol(TheNumber);

    return errCode;
}



/*  logMsg()  --  place a message to the log file
 *
 */
void
logMsg(msg, arg)
char * msg;     /* message to send */
char * arg;     /* an optional arg */
{
    long timer;
    struct tm * timeptr;
    char times[PATHLEN];

    timer = time(0L);
    timeptr = localtime(&timer);
    fprintf(lfp, "(%d/%d/%d %d:%d): ", timeptr->tm_mon +1, timeptr->tm_mday,
            	    timeptr->tm_year, timeptr->tm_hour, timeptr->tm_min);
    fprintf(lfp, msg, arg);
    fprintf(lfp, "\n");
    fflush(lfp);

    return;
}



/* output fatal error message and exit */
void
error(fmt, va_alist)
char *fmt;
va_dcl
{
    va_list args;

    va_start(args);
    vfprintf(stderr, fmt, args);
    va_end(args);

    exit(1);
}
