/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */

#ifndef TRUE
#define TRUE (1 == 1)
#endif
#ifndef FALSE
#define FALSE (1 == 0)
#endif

#define SUCCESS	    0
#define FAILURE     -1
#define EOS         '\000'

#define NDBM	    1	    /* compile in support for ndbm */

#define NAMELEN     100     /* name length */
#define PATHLEN     100     /* pathname length */
#define SHORTSTR    50      /* short string */
#define COMMAND     500     /* max length of command */
#define LONGVALUE   5000    /* special long info lines */

#define MAXTEMPLATE 80     /* max number of records in a template */
#define MAXNOTES    500     /* max number of notes in a template */
#define MAXNDATA    1000    /* max number of notedata lines */
#define MAXREGEXP   10000   /* maximum length of regular expr. */
#define MAXPATHDIR  50      /* max. # of direc-s in OT_DBPATH */
#define MAXPROJECTS 50      /* max. # of projects */

#define MAXNUMCRDIGITS 7    /* max # of digits in a CR number */
#define MAXNUMSTAT  20      /* max # of status values */
#define MAXNUMVAL   8       /* max # of priority, severity values */
#define NOTEBLOCKSIZE 10000 /* size of chunks to allocate for notes */
#define MAXLISTLEN  20000   /* max length of list (user-defined mainly) */
#define MAXKWD	    7	    /* maximum len. of notedata keyword field */
#define MAXDATELEN  9	    /* length of mm/dd/yy */

#define MAXTIME     8	    /* maximum len. of time/date str. in notedata */
#define MAXNDHEADER 35

#define STATLEN     20      /* status value length 			*/

#define FIELDWIDTH 42   /* available space in created template for field and opts */
#define FIELDSWIDTH 30  /* available space in full report template for field */

#define MAXWIDTH    512 /* max width for a one liner report */

/* strings and places */

#define BASE "/project/ot"

#define HEADERDB   "headdb"	/* header database filename */
#define HISTORYDB  "histdb"	/* history database filename */

#define CRSEPARATOR "\01\02\03\04\05\06\07" /* separates bugs in a Sum file */

#define OKRESPONSE "ok\n"       /* good response from the server */
#define BADRESPONSE "ng\n"      /* generic bad response from the server */

#define NUM_RANGE_CHARS "-<>"
#define NUM_COMP_CHARS  ",-<>"
#define DATE_COMP_CHARS "-<>"
#define DATE_CHARS "+-<>/."
#define OPT_CHARS "()[],"

#define DEGREE_MANY_CHAR '+'


/*  ifdefs for ULTRIX  */
#ifdef ultrix
#define strdup(s)   strcpy((char*)malloc(strlen(s)+1),s)
#endif



/* typedefs and structures */
typedef enum { TYPE_LIST, TYPE_TEXT, TYPE_MAILNAME, TYPE_DATE,
    TYPE_IDNUM, TYPE_NUM, TYPE_ENUM, TYPE_NULL} otType;
		
typedef enum { OPT_NONE, OPT_MAND, OPT_OPTL } otOpt;

typedef enum { DEGREE_NONE, DEGREE_ONE, DEGREE_MANY } otDegr;

typedef enum { Sig_Default, Sig_Ignore, Sig_Catch } sigType;

typedef enum
{
    CLOSE,	/* derived operation */
    DELETE,	/* delete existing object */
    ENTER, 	/* enter new object */
    INITIAL,	/* initialization value */
    UPDATE, 	/* update existing object */
    VALIDATE,	/* validate only object or template file */
    TEMPLATE,	/* return an initialized template */
    QUERY	/* query option */
} OTOperation;

typedef enum
{
    INIT,       /* initial value -- no interface type is specified */
    MOTIF,      /* Motif UI */
    CLI         /* command line interface */
} OTUI;


typedef enum
{
    RESRVED,	/* can only be written by ot internals  */
    USRVSBL,	/* may be visible to users in reports   */
    UNPRNTD	/* unprinted during full report queries */
} OTNDType;

#define	OT_SUCCESS_BIT		0x0000	/* general success level	     */
#define	OT_VALID_BIT		0x1000  /* invalid template - things user    */
					/* can fix			     */
#define OT_WARN_BIT		0x2000  /* project definition file missing   */
					/* unknown host			     */
					/* <error requiring reconfiguration> */
#define OT_SYSTEM_BIT	        0x4000  /* program should exit		     */

#define OT_INVAL( id )		( (int)(id) & OT_VALID_BIT )
#define OT_WARN( id )		( (int)(id) & OT_WARN_BIT )
#define OT_SYSTEM( id )		( (int)(id) & OT_SYSTEM_BIT )

/*
 * Error code severity levels:
 *
 * - success
 * - validation errors
 * - warning level - configuration errors
 * - system error level - exit
 */

typedef enum {
    OT_SUCCESS		 = OT_SUCCESS_BIT,
    OT_TEMPLATE_UNCHANGED,
    OT_OBJECT_CREATED,
    OT_OBJECT_UPDATED,
    OT_OBJECT_DELETED,
    OT_TRANSMISSION_CONCLUDED,
    OT_NOMATCH,
    OT_NO_OBJECT_MATCH,
    OT_TCL_NOPROC,
    OT_INFORMATION,

    OT_TEMPLATE_READ     = OT_VALID_BIT,
    OT_NUMBER_PLACE_HOLDER,
    OT_INVALID,
    OT_MANDATORY_FIELD,
    OT_NOT_MULTIPLE,
    OT_ILLEGAL_DATE_FORMAT,
    OT_ILLEGAL_DATE_COMPAR,
    OT_NO_CLOSING_NOTE,
    OT_TEMPLATE_LONG,
    OT_NOTES_TOOMANY,
    OT_NDATA_TOOMANY,
    OT_FIELDNAME_NOT_FOUND,
    OT_IDNUMFIELD_NOT_FOUND,
    OT_NO_OPERATION,
    OT_INVALID_BYLINE,
    OT_INVALID_FIELD_DATE,
    OT_INVALID_FIELD_ENUM,
    OT_INVALID_FIELD_ENUMTYPE,
    OT_INVALID_FIELD_NUM,
    OT_INVALID_FIELD_IDNUM,
    OT_INVALID_FIELD_ELEM,
    OT_LHS_INVALID,
    OT_LHS_MANDATORY,
    OT_LHS_MANDATORY_USER,
    OT_LHS_ONE_IDNUM,
    OT_LHS_TYPE,
    OT_LHS_OPT,
    OT_LHS_DEGREE,
    OT_LHS_BOTHVAL_MISSING,
    OT_LHS_VAL_MISSING,
    OT_LHS_NOTPRESENT,
    OT_UNKNOWN_SORT_FLD,
    OT_NO_SORT_PIPE,
    OT_DUPLICATE_FIELD,
    OT_FIELDNAME_LONG,
    OT_NDKEYWORD_LONG,
    OT_NO_CR_IN_DBFILE,

    OT_TCL		 = OT_WARN_BIT,
    OT_TCL_CALLER,
    OT_PROJECT_NOT_FOUND,
    OT_PROJECT_READ,
    OT_PROJECT_TYPE_FILENAME,
    OT_PROJECT_NAME_CONFLICT,
    OT_OBJECT_NOT_FOUND,
    OT_TCLPROJDEF,
    OT_TCLCMDSTRING,
    OT_TCLCMDFILE,
    OT_UNKNOWN_PROJECT_FLD,
    OT_PROJECT_SYNTAX,
    OT_TYPE_FILE,
    OT_TYPE_FILE_CONTENTS,
    OT_ILLEGAL_OPERATION,
    OT_ILLEGAL_NOTIFICATION,
    OT_ILLEGAL_APPEND,
    OT_METATEMPLATE_READ,
    OT_METATEMPLATE_LONG,
    OT_METATEMPLATE_EOL,
    OT_OTRC,
    OT_UNKNOWN_HOST,
    OT_SOCKET,
    OT_CONNECT,
    
    OT_HOST_UNSPECIFIED,
    OT_PORT_UNSPECIFIED,
    OT_SOCKET_OPEN,
    OT_LOG_APPEND,
    OT_MAKE_SUM,
    OT_TEMPLATE_NAME,
    OT_GETPWUID,
    OT_WHO_ARE_YOU,
    OT_TEMPLATE_EDIT,
    OT_TEMPLATE_FIND,
    OT_NUMBER_HIGH,
    OT_USERVAL_TOO_LONG,
    OT_TOO_MANY_PROJECTS,
    OT_NOT_RESERVED_NDATA,
    OT_OTQM_CORRUPT_QI,
    OT_OTQM_QI_ERROR,
    OT_OTQM_DB_EOF,

    OT_RCS_CO,
    OT_TEMPLATE_COPY_TMP,
    OT_TEMPLATE_MOVE_DB,
    OT_TEMPLATE_COPY_DB,
    OT_RCS_CI,
    OT_RCS_ADMIN,
    OT_TEMPLATE_UPDATE,
    OT_RCS_DELTA,
    OT_GET_NUMBER,
    OT_OPENDIR_LOCATION,
    OT_RCS_DELETE,
    OT_RCS_UNLOCK,
    OT_TEMPLATE_WRITE,
    OT_NO_DIR,
    OT_OTQM_QLIST,
    OT_FILE_OPEN,
    OT_RENAME_FILE,
    OT_DBFILE_TOP_NUM,

    OT_MALLOC            = OT_SYSTEM_BIT,
    OT_MALLOC_LOCATION,
    OT_PCLOSE,
    OT_STAT,
    OT_POPEN,
    OT_SETRLIMIT,
    OT_OTTYPE,
    OT_REGEXP,
    OT_NEED_FULL_PATH,
    OT_CREATE_DIR,
    OT_CREATE_LOCK_FILE,
    OT_REMOVE_LOCK_FILE,
    OT_CREATE_FILE,

    OT_NEED_PROJECT,
    OT_VALIDATE_USAGE,
    OT_APPEND_USAGE,
    OT_DELETE_USAGE,
    OT_DIFFERENT_PROJ,
    OT_ENTER_NUMBER,
    OT_DIFFERENT_NUMBER,
    OT_UPDATE_NUMBER ,
    OT_NUMBER_FILE_READ,
    OT_MULTIPLE_PROJECTS ,
    OT_MULTIPLE_ALTERNATE,
    OT_ILLEGAL_NUMERIC_SEARCH,
    OT_MULTIPLE_RANGES,
    OT_ILLEGAL_RANGE,
    OT_ILLEGAL_ID_NUMBER,
    OT_OPEN_INPUT,
    OT_ILLEGAL_SEARCH_SYNTAX,
    OT_UNKNOWN_FIELD,
    OT_ONE_FILENAME_ONLY,
    OT_NOT_REGULAR_FILE,
    OT_UNKNOWN_LAYOUT,
    OT_OTCLI_USAGE,
    OT_OTB_CLI_USAGE,
    OT_OTQM_USAGE,
    OT_CLI_ZERO_NUMBER,
    OT_ARGUMENT_NO_VALUE,
    OT_OTMAKESUM_USAGE,
    OT_FULL_PATH,
    OT_OTMAKESUM_SUMMARY,
    OT_SUMFILE_READ,
    OT_SUMFILE_RENAME,
    OT_OTQM_FATAL_ERROR,
    OT_INTERNAL_ERROR,
    OT_INTERNAL_REGEXP,
    OT_GENERAL_ERROR,
    OT_MAX } OTErr ;

typedef int bool;

typedef struct {
    char *  s;              /* original string */
    char    comparison;     /*  -, <, > or nothing */
    int     month, day, year;
    int     days;
} OTDateComparison;

struct _OTCompMail
{
    char *  compName;	/* component name */
    char *  unames;	/* notification mail names for that component */
    struct _OTCompMail * next; /* point at the next one */
};

typedef struct _OTCompMail OTCompMail;


typedef struct
{			        /* project-specific enumerated type */
    char *typename;             /* type name (e.g., 'company') */
    char *filename;             /* absolute pathname of file containing list */
    char *list;                 /* list of elements in type */
} OTEnumType;

typedef struct
{
    char    field[NAMELEN];
    char    opts[NAMELEN];
    otType  type;
    otOpt   optionality;
    otDegr  degree;
    char    defs[NAMELEN];
    char    abbr[NAMELEN];
    char    head[NAMELEN];
    char    width[SHORTSTR];
    char    line[COMMAND];
    bool    unassigned;
    char    list[COMMAND];
    char    range[COMMAND];
} OTMetaTemplate;

typedef struct
{
    char    name[NAMELEN];
    char    proj_leader[NAMELEN];
    char    object_name[NAMELEN]; 
    char    proj_dir[NAMELEN];
    char    blogMlist[NAMELEN];
    bool    blogSubmitter;
    char   *server_fields;
    OTCompMail * cMlist;
    char    server_host[NAMELEN];
    int     server_port;
    OTEnumType *enum_types;
    char    odexmcli[PATHLEN];
    bool    notify;
    char    compliance[NAMELEN];
    int     block_silence;
    char    default_layout[LONGVALUE];
    char    all_layout[LONGVALUE];
    OTMetaTemplate *otmp;
    char   *projTclCode;
} OTProject;

typedef struct
{
    char *   field;
    char *   list;
    char *   value;
    unsigned type : 8;		/* assigned from otType */
    unsigned optionality : 4;	/* assigned from otOpt  */
    unsigned degree : 4;	/* assigned from otDegr */
} OTHeaderField;

typedef struct
{		/* for holding a template note */
    char    who[NAMELEN];       /* note: person */
    char    date[NAMELEN];      /* note: date */
    char    sens[NAMELEN];      /* note: sensitivity */
    char   *text;               /* text of note */
} OTNote;

#define ND_HIST 0x00001		/* is a history line 	 	*/
#define ND_LUPD 0x00002		/* is a last update line 	*/
#define ND_ZZZZ 0x00004		/* bit reserved for future use  */
#define ND_USER 0x00008		/* is a user-contributed line   */

#define ND_RESV 0x00010		/* is a reserved word		*/
#define ND_VISB 0x00020		/* is visible in reports        */
#define ND_XXXX 0x00040		/* TBD 				*/
#define ND_YYYY 0x00080		/* TBD 				*/

#define ND_ISHISTORY(  id )     ( (int)(id) & ND_HIST )
#define ND_ISLUPDATE( id )	( (int)(id) & ND_LUPD )
#define ND_ISUSER( id )		( (int)(id) & ND_USER )
#define ND_ISRESERVED( id )	( (int)(id) & ND_RESV )
#define ND_ISVISIBLE( id )	( (int)(id) & ND_VISB )

#define ND_HISTORY_KWD		"HISTORY"
#define ND_LUPDATE_KWD		"LUPDATE"

typedef struct
{
    char      *nd_kwd;		/* Only used for user-contributed lines */
    char      *nd_agent;	/* user name  			   */
    char      *nd_field;	/* field name (HISTORY lines only)	   */
    char      *nd_previous;	/* previous value (HIST only)	   */
    char      *nd_current;	/* current value (HIST or user-contrib.)*/
    char       nd_time[MAXTIME]; /* hh:mm:ss				   */
    unsigned   nd_flags: 8;     /* Stores data type, display properties */
    unsigned   nd_weekday : 4;  /* Day of week/Sun='\001', Mon='\002'.. */
    unsigned   nd_mon: 4;       /* Month as char: Jan='001', etc        */
    unsigned   nd_day : 8;      /* Integer value as char (day)          */
    unsigned   nd_year : 8;     /* Integer value as year (year)         */

} OTNotedata ;

typedef struct
{
    OTHeaderField * tr;
    OTNote * notep;
    OTNotedata * ndp;
} OTTemplate;

typedef struct
{
    char    dir[NAMELEN];
    char    name[NAMELEN];
} OTFilen;


typedef struct
{
    OTUI    pcb_ui;	   /* user interface type */
    char *  pcb_uName;     /* username (OT_USER, USER, LOGNAME, getpwuid)*/
    long    pcb_CRNumber;  /* current value for a CR number */
    OTOperation pcb_operation; /* current (derived) value for an operation */
    bool    pcb_imTheServer; /* networked/non-networked server */
    bool    pcb_nopened;   /* network connection opened */
    bool    pcb_lopened;   /* local connection opened */
    FILE *  pcb_efp;	   /* error file pointer -- log file */
    char    pcb_logPid[NAMELEN];  /* our pid (for log messages) */
    char    pcb_rcsCmd[PATHLEN];  /* RCS commands directory path */
    bool    pcb_notify;    /* current value for the notification indicator   */
			   /* (derived from cb->cb_notify and 		     */
    			   /* proj->block_silence) 			     */
    bool    pcb_forceUpdate; /* force an update */
    bool    pcb_quiet;	 /* do it quiet TCL indicator */
    bool    pcb_tclBegin;  /* initializing pattern for TCL scripts */
    bool    pcb_tclEnd;    /* concluding pattern for TCL scripts */
    bool    pcb_xmitEnd;   /* true when server has sent null */
    int	    pcb_sumPrior[MAXNUMVAL]; /* for special summary reports */
    int	    pcb_sumSever[MAXNUMVAL]; /* for special summary reports */
    int     pcb_sumStat[MAXNUMVAL]; /* for special summary reports */
    char    pcb_statName[MAXNUMSTAT][STATLEN]; /*  -- '' -- */
    int     pcb_bugCount;	 /* current bug (matching) count */
    char    pcb_filterNum[COMMAND];	/* list of CRs to look for */
    FILE *  pcb_sortPipe;	 /* pipe for the sort */
    long    pcb_topNumber;	 /* project's top object number */
    int	    pcb_CRmatch;	 /* matching CR found */
    OTProject  * pcb_project;    /* pointer to "current" project */
    OTTemplate * pcb_template;   /* pointer to "current" template */
    Tcl_Interp * pcb_interp;     /* TCL interpreter */
    FILE    *pcb_fp;		 /* Temporary holder for server connection  */
    OpenFile *pcb_fileIn;	 /* Input for server (client)		    */
    char      pcb_fileInName[SHORTSTR]; /* Name of server input		    */
    OpenFile *pcb_fileOut;	 /* Output for server (client)		    */
    char      pcb_fileOutName[SHORTSTR]; /* Name of server output	    */
    OpenFile *pcb_kwikPixSock;	 /* Socket to otKwikPix server		    */
    char      pcb_kwikPix[SHORTSTR]; /* Name of kwikpix file                */
} OTPrivateCB;

typedef struct
{
    OTTemplate * ecb_origStruct;
    OTTemplate * ecb_tStruct;
    OTTemplate * ecb_blankStruct;
    OTFilen    * ecb_fname;
} OTEnterCB;

typedef struct
{
    bool   qcb_skipHeader;
    bool   qcb_count;
    bool   qcb_summary;
    bool   qcb_tabSeparate;
    bool   qcb_fullText;
    bool   qcb_historyLines;
    bool   qcb_readNote;
    bool   qcb_readHistory;
    bool   qcb_keywordMatchFound;
    int	   qcb_outwidth;
    FILE * qcb_outputFile;
    char * qcb_sortField;
    char * qcb_nsens;
    char * qcb_keyInText;
    long   qcb_loRange;
    long   qcb_hiRange;
    OTMetaTemplate * qcb_filter;
    char * qcb_kwikPixMatches;
    char * qcb_nextMatch;
} OTQueryCB;

typedef struct
{
    char      * cb_project;
    char      * cb_altTname;
    long        cb_CRNumber;
    OTOperation cb_operation;
    bool        cb_notify;
    bool	cb_append;
    char      * cb_templFile;
    char      * cb_dbpath;	  /* OT_DBPATH environment variable contents */
    char      * cb_layout;	  /* OT_LAYOUT environment variable contents */
    char      * cb_prompt;	  /* OT_PROMPT environment variable contents */
    char      * cb_tclString;	  /* TCL update string */
    char      * cb_tclFile;	  /* TCL file to pass to server */
    char      * cb_tclOtrc;	  /* contents of otrc file */
    OTPrivateCB * cb_pcb;
    OTQueryCB * cb_qcb;
    OTEnterCB * cb_ecb;
} OTControlBlock;

/* globals */

extern char     gotCRfile[];    /* name of gotten CR file */
extern char     gotCRdir[];     /* directory of gotten CR file */
extern char     rmTfile[];      /* name of file to remove on signal */
extern OTProject * otProj;      /* project structure */
extern OTControlBlock * otCB;	/* control block structure */

/* getopt() details -- internal variables -- shouldn't these be in a
 * system include file somewhere consistent? */

 extern char *   optarg;         /*  points to start of flag's parameter */
 extern int      optind;         /* index to next arg */
 extern int      opterr;         /* error message controller */
 extern int      optopt;         /* real option found */


/*    R O U T I N E S    */


/*
 * OT initial initialization.
 * This function allocates space for the \fIOTControlBlock\fR control block
 * structure, calls \fIotClearControlBlock()\fR function to clear it, calls 
 * initialization for the TCL interpreter \fIotTclInitInterp\fR and stores 
 * the pointer to it in the \fIOTControlBlock\fR structure.
 * Then it allocates space for the \fIOTPrivateCB\fR control block structure
 * and calls \fIotMallocQCB\fR and \fIotMallocECB\fR to allocate space for 
 * the \fIOTEnterCB\fR and \fIOTQueryCB\fR control block structures. It also
 * stores pointers to these three control structures in the 
 * \fIOTControlBlock\fR structure. 
 * After that it reads environment variables \fIOT_PROJECT, OT_LAYOUT\fR and
 * \fIOT_DBPATH\fR into the control block structure, calls 
 * \fIotGetUserName()\fR to put user name into the control block structure. 
 * Then it invokes \fIotPreparseInput()\fR function to get project and 
 * alternate template names and stores them in the control block structure. 
 * Function \fIotSetCBMember()\fR is used whenever any member of the 
 * control block structure needs to be set/reset.
 * Returns OT error status.
 */
OTErr
otInitialInit _ANSI_ARGS_((int    *argc,
	char  **argv));


/* Free control block.
 * This function frees the memory used by the \fIOTControlBlock\fR
 * structure along with the memory used by \fIOTPrivateCB, OTQueryCB\fR and 
 * \fIOTEnterCB\fR structures pointed by the \fIOTControlBlock\fR
 * structure.
 */
void
otFreeControlBlock();


/* 
 * Validate operation initialization.
 * This function initializes values in the \fIOTEnterCB\fR block, checks
 * an input information and calls \fIotInitTemplate()\fR to initialize
 * \fIotCB->cb_ecb->ecb_origStruct\fR and \fIotCB->cb_ecb->ecb_blankStruct\fR
 * template structures.
 * Returns OT error status.
 */
OTErr
otInitValidate();


/*
 * Enter operation initialization.
 * This function initializes values in the \fIOTEnterCB\fR block, checks 
 * an input information and calls \fIotInitTemplate()\fR to initialize 
 * \fIotCB->cb_ecb->ecb_origStruct\fR and \fIotCB->cb_ecb->ecb_blankStruct\fR
 * template structures.  If 
 * Returns OT error status.
 */
OTErr
otInitEnter _ANSI_ARGS_((char *tmpfile));

/*
 * Update operation preparation.
 * This function calls \fIotInitUpdate()\fR to initialize control block 
 * structures for an update operation, calls internal functions to check input 
 * information for potential conflicts and after that calls 
 * \fIotGetObjectForUpdate()\fR to check out the object and return
 * the name of the file for edit into provided by the caller local variable
 * (tmpfile).
 * Returns OT error status.
 */
OTErr
otPrepareForUpdate _ANSI_ARGS_((char * tmpfile));     /* filename for edit */


/*
 * Clean after \fIEnter\fR operation
 * This function clears all values and frees dynamically allocated
 * memory after an \fIEnter\fR operation
 */
void
otCleanAfterEnter _ANSI_ARGS_((OTErr errCode));	/* OT error status returned by an enter operation */


/*
 * Clean after \fIUpdate\fR operation
 * This function clears all values and frees dynamically allocated
 * memory after an \fIUpdate\fR operation.
 */
OTErr
otCleanAfterUpdate _ANSI_ARGS_((OTErr errCode));	/* OT error status returned by an update operation */


/*
 * Clean after \fIDelete\fR operation
 * This function clears all values and frees dynamically allocated
 * memory after an \fIDelete\fR operation.
 */
void
otCleanAfterDelete _ANSI_ARGS_((OTErr errCode));	/* OT error status returned by a delete operation */


/*
 * Clean after \fIQuery\fR operation
 * This function clears all values and frees dynamically allocated
 * memory after an \fIQuery\fR operation.
 */
void
otCleanAfterQuery();

/*
 * Clean after \fIValidate\fR operation
 * This function clears all values and frees dynamically allocated
 * memory after an \fIValidate\fR operation.
 */
void
otCleanAfterValidate();


/*
 * Clean after \fITemplate\fR operation
 * This function clears all values and frees dynamically allocated
 * memory after an \fITemplate\fR operation.
 */
void
otCleanAfterTemplate();


/*
 * Allocate memory for template structure.
 * This function allocates memory for a template structure of \fIOTTemplate\fR
 * type.
 * Returns an error status.
 */
OTErr
otMallocTemplate _ANSI_ARGS_((OTTemplate **tStruct)); /* pointer to pointer to template */


/*
 * Free memory from template structure.
 * This function frees the memory used by the template structure.
 */
void
otFreeTemplate _ANSI_ARGS_((OTTemplate *tStruct));  /* template structure to free */

/*
 * Free memory from project structure.
 * This function frees the memory used by the project structure.
 */
void
otFreeProject _ANSI_ARGS_((OTProject *pStruct));

/*
 * Free memory from metatemplate structure.
 * This function frees the memory used by the metatemplate structure.
 */
void
otFreeMetaTemplate _ANSI_ARGS_((OTMetaTemplate *mStruct));  /* metatemplate structure to free */


/*
 * Read project.
 * This function handles reading and validating project definition 
 * information.  
 * First, it calls internal function \fIverifyProjectName()\fR to verify 
 * project name. 
 * Then it calls internal function \fIreadProject()\fR to read project 
 * information named by project into project definition structure. 
 * After that it validates project name in the project definition file. 
 * Finally it calls \fIotSetMetaTemplate\fR to read default metatemplate
 * file into \fIOTMetaTemplate\fR type structure and store the pointer to that 
 * structure in the project definition structure. 
 * This function also invokes \fIotSetCBMember\fR function to set members 
 * to the control block structure pertinent to the project.
 * Returns an error status.
 */
OTErr
otReadProject _ANSI_ARGS_((char *  project));      	    /* project name */

/*
 * Set metatemplate information.
 * This function reads metatemplate file into \fIOTMetaTemplate\fR type 
 * structure and store the pointer to that structure in the project 
 * definition structure. 
 * It also calls \fIotTclInitCmds()\fR function to initialize TCL commands.
 * Returns zero on success, and -1 on failure.
 */
OTErr
otSetMetaTemplate _ANSI_ARGS_((char * tDefFile));		/* template definition file name */


/*
 * Get initialized blank template.
 * This function initializes blank OTTemplate template structure based on 
 * the contents of the OTMetaTemplate structure in the project definition
 * structure. 
 * It also stores a pointer to this structure in the \fIecb_blankStruct\fR
 * member of the \fIOTEnterCB\fR.
 * OT template operation (\fIot -t\fR) calls \fIotInitTemplate()\fR as a first 
 * step of getting a template for the project.
 * Returns an error status.
 */
OTErr
otInitTemplate();

/*
 * Read a template file into a template structure from a file stream.
 * This function allocates memory for the template, stores a pointer to
 * this memory in the template arguments and reads a template from the 
 * file stream, placing a corresponding representation in the template
 * structure.
 * Returns OT_TEMPLATE_LONG if the template exceeds the maximum number
 * of lines, OT_MALLOC_LOCATION if there is a memory allocation error,
 * OT_SUCCESS if a template has been read and OT_TRANSMISSION_CONCLUDED
 * if there is nothing more to read.
 * Returns an error status.
 */
OTErr
otReadTemplateFromFile _ANSI_ARGS_((OTTemplate **tStruct      /* pointer to pointer to template  */,
        FILE        *fp           /* file pointer                    */,
	bool         saveNote	  /* if true, save note in structure */,
	bool         saveNotedata /* if true, save notedata          */ ));

/*
 * Read a template file into a template structure from a file by name.
 * This function allocates memory for the template, stores a pointer to
 * this memory in the template arguments and reads a template from the named
 * file, placing a corresponding representation in the template
 * structure.
 * Returns OT_TEMPLATE_LONG if the template exceeds the maximum number
 * of lines, OT_MALLOC_LOCATION if there is a memory allocation error,
 * OT_SUCCESS if a template has been read and OT_TRANSMISSION_CONCLUDED
 * if there is nothing more to read.
 * Returns an error status.
 */
OTErr
otReadTemplateFromFilename _ANSI_ARGS_((OTTemplate **tStruct      /* pointer to pointer to template  */,
        char        *filename     /* input filename                  */,
	bool         saveNote	  /* if true, save note in structure */,
	bool         saveNotedata /* if true, save notedata          */ ));
	

/*
 * Read a note or series of notes into a template structure.
 * This function allocates memory for the note, stores a pointer to this 
 * memory and reads the note(s) from either the file or from the string,
 * placing a corresponding representation in the template structure.
 * Returns OT_TEMPLATE_LONG if the template exceeds the maximum number
 * of lines, OT_MALLOC_LOCATION if there is a memory allocation error,
 * OT_SUCCESS if a template has been read and OT_TRANSMISSION_CONCLUDED
 * if there is nothing more to read.
 * Returns an error status.
 */
OTErr
otReadNotes _ANSI_ARGS_((OTTemplate  *tStruct      /* pointer to pointer to template  */,
	FILE        *fp		  /* input filename                  */,
	bool         saveNote	  /* if true, save note in structure */,
	bool         saveNotedata /* if true, save notedata          */ ));

/*
 * Write template structure to a template file by name.
 * This function writes the contents of an \fIOTTemplate\fR template structure 
 * to the template file.
 * Returns an error status.
 */
OTErr
otWriteTemplateToFilename _ANSI_ARGS_((OTTemplate *tStruct      /* template definition structure */,
	char       *filename     /* output file name */ ));

/*
 * Write template structure to a template file by file pointer.
 * This function writes the contents of an \fIOTTemplate\fR template structure 
 * to the template file.
 * Returns an error status.
 */
OTErr
otWriteTemplateToFile _ANSI_ARGS_((OTTemplate *tStruct      /* template definition structure */,
        FILE       *fp           /* file pointer                  */));

/*
 * Check out an object for update.
 * This function calls internal function \fIotFormName()\fR to forms RCS 
 * filename for the object. Then it calls internal function \fIotLockCRFile\fR
 * to handle RCS check out operation. It sets global variables \fIgotCRfile 
 * gotCRdir\fR and \fIrmTfile\fR. 
 * Finally it calls \fIotReadTemplateFromFilename()\fR function to create and 
 * populate OTTemplate structure for the object. 
 * This function also stores a pointer to the \fIOTTemplate\fR structure in 
 * the \fIecb_origStruct\fR member and pointer to the \fIOTFilen\fR structure 
 * in the \fIecb_fname\fR member of the \fIOTEnterCB\fR structure. The
 * name of the file for edit is stored in the provided \fItemplfile\fR variable.
 * Returns an error status.
 */
OTErr 
otGetObjectForUpdate _ANSI_ARGS_((char * tmpfile));     /* file to edit */


/*
 * Delete object.
 * This function corresponds to OT delete operation (\fIot -d\fR). 
 * It calls internal function \fIotChkInputForDelete()\fR to verify an input 
 * for conflicts and a number of internal functions including 
 * \fIotDeleteCRFile()\fR to handle the deletion of an object. 
 * Returns an error status.
 */
OTErr
otDeleteObject();


/*
 * Validate data for types and values.
 * This function calls internal function \fIValidateData\fR for each header
 * field to performs type-specific data validation routine. 
 * \fIOTControlBlock\fR points to \fIOTEnterCB\fR control block, which
 * contains pointers to both template structures (before and after the update).
 * Returns TRUE or FALSE.
 */
OTErr
otValidateData();

/*
 * Validate data dependencies.
 * This function validates data dependencies (conditional functionalities
 * parsed by TCL from project definition file, .otrc file, etc..).
 * This function also validates notes and makes sure that a new note is 
 * added if status is changing to \fIclosed\fR for OT \fIupdate\fR operation.
 * \fIOTControlBlock\fR points to \fIOTEnterCB\fR control block, which
 * contains a pointer to the updated template definition structure.
 * Returns TRUE or FALSE.
 */
OTErr
otValidateDepend();

/*
 * Validate template format.
 * This function validates the format of the user input template as follows.
 * Any mandatory field in the metatemplate missing in the user template
 * causes an error, unless the project definition file has
 * "strict lhs compliance" set to off.
 * Each field in the user template with a name which is identical to a
 * field name in the metatemplate must have parenthesis/bracket/degree 
 * information identical to that of the corresponding metatemplate field.  
 * If it doesn't, it causes an error, unless the project file has "strict
 * lhs compliance" set to off.
 * Note that the user template may have fields not present in the metatemplate.
 * That is never an error.
 * If a field in the user template and a field in the metatemplate have
 * the same name but different types and the project file has "strict
 * lhs compliance" turned off, then make sure that if the
 * user template left hand side is a list, then the original 
 * left hand side for the corresponding field (i.e., the left hand side 
 * of the template which the user is updating) is a list and 
 * each of the user template left hand side's
 * elements is a member of the either the original left hand side list or
 * the metatemplate left hand side elements.  With strict compliance, the 
 * elements of the user left hand side list must be the same as the 
 * metatemplate lhs list, although they need not be in the same order.
 */
OTErr
otValidateLhs();

/*
 * Post process data validation and update.
 * This function is called for ENTER and UPDATE operations on a server from
 * the \fIcmdCommit()\fR. It evaluates the \fIpostProcess\fR TCL procedure. 
 * \fIotPostProcess\fR is invoked after successfull regular validation.
 */
OTErr
otPostProcess();

/*
 * Check the updates.
 * This function calls internal function \fItemplate_changed()\fR to check 
 * if the template was modified during an edit session. If the template 
 * remains unchanged, and \fIforce update\fR indicator is not set, it calls 
 * internal function \fIrmRCSlock()\fR to unlock the RCS file for the object 
 * (\fIupdate\fR OT operation case only).
 * \fIOTControlBlock\fR points to \fIOTEnterCB\fR control block, which
 * contains pointers to both template structures (before and after the update).
 * Returns zero on success, and -1 on failure.
 */
OTErr
otTemplateChanged();


/*
 * Unlock an object
 * This function calls internal function \fIrmRCSlock()\fR to unlock
 * the RCS file for an object. 
 * RCS directory and filename are stored in global variables 
 * \fIgotCRdir\fR and \fIgotCRfile\fR.
 * Returns zero on success, and -1 on failure.
 */
int
otUnlockObject();


/*
 * Store new object under RCS.
 * This function calls \fIotGetnCRnum\fR to get next object number for a 
 * project, stores it in the template structure, calls 
 * \fIotWriteTemplateToFile\fR to write final version of the template 
 * structure to a template file.
 * Then it calls internal function \fIotAdminCRFile()\fR to put new object 
 * file under RCS, and internal function \fIotNotify()\fR
 * to fork a process to update corresponding Sum file and to send 
 * notification mail.
 * Returns OT error status.
 */
OTErr
otEnterObject _ANSI_ARGS_((char *tempfile));


/*
 * Store updated object under RCS.
 * For an interractive update this function writes an updated template 
 * structure to a temporary file the name of which is passed by a caller. After 
 * that it calls internal function \fIotFormDeltamsg()\fR to form the list
 * of changed header fields and notes for the notification mail and object 
 * history lines. Then it calls internal function \fIotDeltaCRFile()\fR to 
 * update the object file under RCS. After all of that
 * it calls internal function \fIotNotify()\fR to fork a process to
 * update corresponding Sum file and to form and send notification mail.
 * \fIOTEnterCB\fR control block has to contain pointers to the \fIOTFilen\fR
 * structure with the RCS filename and to the template definition structures 
 * before and after the update.
 * Returns OT error status.
 */
OTErr
otUpdateObject _ANSI_ARGS_((char *tmpfile));  /* template file after an update */	


/*
 * Copy metatemplate structure.
 * This function allocates space for a new \fIOTMetaTemplate\fR structure
 * and makes a duplicate of the \fIinMetaStruct\fR structure into it.
 * Returns a pointer to the output metatemplate structure on success, a null
 * pointer (0) on failure.
 */
OTErr
otDupMetaTemplate _ANSI_ARGS_((OTMetaTemplate  *inMetaStruct  /* structure to copy from */,
	OTMetaTemplate **outMetaStruct));	  /* structure returned     */

/*
 * Copy template structure.
 * This function allocates space for a new \fIOTTemplate\fR structure
 * and makes a duplicate of the \fIinTemplate\fR structure into it.
 * Returns a pointer to the output template structure.
 */
OTErr
otDupTemplate _ANSI_ARGS_((OTTemplate  *inTemplate 	  /* structure to copy from */,
	OTTemplate **outTemplate));	  /* structure returned     */


/*
 * Get header field value.
 * This function finds the value of a header field given the header field 
 * label.
 * Returns a pointer to the value on success, a null pointer (0) on failure.
 */
char *
otGetHeaderFieldValue _ANSI_ARGS_((char * headerFieldLabel    /* header field label */,
	OTTemplate  * tStruct));	    /* template structure */

/*
 * Set header field value.
 * This function sets the value of a header field given the header field
 * label.
 * Returns informative status.
 */
OTErr
otSetHeaderFieldValue _ANSI_ARGS_((char * headerFieldLabel     /* header field label */,
	char * headerFieldValue	    /* header field value */,
	OTTemplate * tStruct));        /* template structure */


/*
 * Get list of projects.
 * This function gets a list of projects in the user's current \fIOT_DBPATH\fR.
 * The result is dynamically allocated and must be freed by the caller.
 * Returns OT error status.
 */
OTErr
otGetProjectInfo _ANSI_ARGS_((char **projects)); /* pointer to project list */


/* 
 * Set filter value.
 * This function sets the value of a header field given the header field
 * label in the metatemplate structure for query search criteria
 * Returns OT error status.
 */
OTErr
otSetFilterValue _ANSI_ARGS_((char * label	/* header field label */,
	char * value			/* header field value */,
	OTMetaTemplate * filterTemplate));	/* metatemplate filter structure */

/*
 * Query operation initialization.
 * This function calls \fIotDupMetaTemplate()\fR to initialize meta template
 * structure for filter values, internal function \fIotFormStatusArray()\fR 
 * to form an array of status names for summary, clears filter values and
 * assigns initial value for the layout string in the \fIotCB->cb_layout\fR.
 * Returns OT error status.
 */
OTErr
otInitQuery();

/*
 * Additional setup required after the project is read.
 * This function performs additional setup which requires project and
 * template information being read.
 * It calls \fIotSetupSort()\fR to perform the setup for the sort and
 * \fIotReadTopNumber()\fR to read project's top number
 * Returns OT error status.
 */
OTErr
otSetupQuery();


/*
 * Set control block structure member value.
 * This function sets the value to a control block structure member.
 * It finds the member of the structure, the type of data and stores the
 * data according to it's type.
 * Returns OT error status.
 */
OTErr
otSetCBMember _ANSI_ARGS_((char * label	/* control block structure member label */,
	char * value));	/* control block structure member value */


/*
 * Set query control block structure member value.
 * This function sets the value to a query control block structure member.
 * It finds the member of the structure, the type of data and stores the
 * data according to it's type.
 * Returns OT error status.
 */
OTErr
otSetQCBMember _ANSI_ARGS_((char * label	/* control block structure member label */,
	char * value));	/* control block structure member value */


/*
 * Set the \fIlayout\fR control block structure member.
 * This function sets the \fIlayout\fR and related to it \fIoutwidth\fR 
 * in accordance to the value passed to the function via the argument.
 * Returns OT error status.
 */
OTErr
otSetQlayout _ANSI_ARGS_((char	* value));


/* 
 * Set the \fIfull text\fR query control block structure member.
 * This function sets \fIfull text\fR and related \fIskip header\fR 
 * query control block structure members in accordance to the value passed 
 * to the function via the argument.
 * Returns OT error status.
 */
OTErr
otSetQfullText _ANSI_ARGS_((char    * value));

/*
 * Set the \fIcount\fR and \fIsummary\fR query control block structure member.
 * This function sets \fIcount\fR and \fIsummary\fR query control block 
 * structure members in accordance to the value passed to the function via 
 * the argument.
 * Returns OT error status.
 */
OTErr
otSetQmetrics _ANSI_ARGS_((char    * value));


/*
 * Set the \fIskipHeader, fullText\fR and \fIhistoryLines\fR query control 
 * block structure members.
 * This function sets \fIskipHeader, fullText\fR and \fIhistoryLines\fR query 
 * control block structure members in accordance to the value passed to the 
 * function via the argument.
 * Returns OT error status.
 */
OTErr
otSetQhistory _ANSI_ARGS_((char    * value));


/*
 * Set the \fICRNumber\fR private control block structure member.
 * This function sets \fICRNumber\fR private control block structure member
 * and related members of the query filter structure.
 * Returns OT error status.
 */
OTErr
otSetQCRNumber _ANSI_ARGS_((char    * value));

/*
 * Parse and set filter values for query function.
 * This function processes list of field=value pairs passed via command line
 * argument and calls \fIotSetFilterValue()\fR to set the \fIqcb_filter\fR 
 * metatemplate structure members.
 * Returns OT error status.
 */
OTErr
otSetQselect _ANSI_ARGS_((char    * optarg));


/*
 * Set Tcl EvalFile\fR
 * This function opens and specified file and sets it for the Tcl Eval
 * Returns OT error status.
 */
OTErr
otSetTclEvalFile _ANSI_ARGS_((char    * value));



/* 
 * Put posted message.
 * This function places message in to the \fImessage\fR buffer.
 * The arguments are: an error code of type OTErr and a variable
 * number of arguments
 */
void
otPutPostedMessage();

/*
 * Get posted message.
 * This function retrieves posted message.
 * Returns a pointer to the message buffer on success, a null pointer (0) 
 * on failure.
 */
char *
otGetPostedMessage();


/*
 * Get an environment variable value.
 * This function retrieves an enviroment variable value.
 * Returns an enviroment variable, substitute a default if not found
 */
char *
otGetValue _ANSI_ARGS_((char * variable            /* variable to search for */,
	char * defvalue));            /* default to subs if no env variable */

/*
 * Display help message
 * This function displayes the help message for the ot_bugs command
 * line interface
 */
void
otDoHelp();


/*
 * Display detailed  help message
 * This function displayes the detailed help message for the ot_bugs command
 * line interface
 */
void
otDoHelpDetails();


/*
 * Set trace level for session.
 * This function sets the trace level based on the value of "OT_DEBUG".
 */
void
otTraceLevel();

/*
 * Get the base directory for a project in the search path.
 * This function obtains the directory location of a project.
 * Returns an error status.
 */
OTErr
otGetProjectBase _ANSI_ARGS_((char	* prj		/* name of project */,
        char   ** prjdir));	/* name of project directory */


#define COPY_STRING( from , to , err )				\
    if ( otCopyString( ( from ) , ( to ) ) ) { 			\
	otPutPostedMessage( OT_MALLOC_LOCATION, ( err ) ); 	\
	return OT_MALLOC_LOCATION;				\
    }

#define FREEANDZERO( a )	\
    if ( ( a ) ) {		\
	(void)free( ( a ) );	\
	( a ) = 0;		\
    }


/*
 * Use the DEBUG print statements like this:
 *
 *      DBUG_MIN((stderr, "Returned %d\n", i));     / * minimum debugging * /
 *      DBUG_MED((stderr, "Returned %d\n", i));     / * medium debugging * /
 *      DBUG_MAX((stderr, "Returned %d\n", i));     / * maximum debugging * /
 *
 * Note that this is a macro and must therefore be one line unless the newline
 * is escaped.  Trace level determined by content of env variable OT_DEBUG.
 */

#define D_MIN 1
#define D_MED 2
#define D_MAX 3

/* #define OT_DEBUG TRUE */

#ifdef OT_DEBUG

#define DBUG_MIN(print)  if (otTracing >= D_MIN) { fprintf print; } else
#define DBUG_MED(print)  if (otTracing >= D_MED) { fprintf print; } else
#define DBUG_MAX(print)  if (otTracing >= D_MAX) { fprintf print; } else

#else

#define DBUG_MIN(print)
#define DBUG_MED(print)
#define DBUG_MAX(print)

#endif


#ifdef MAINPROGRAM
#define DECLARE /**/
#else
#define DECLARE extern
#endif


DECLARE int otTracing;
