/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */

#ifndef LINT
static char RCSid_otConvert[] =
    "$RCSfile: otConvert.c,v $ $Revision: 1.1.7.2 $ $Date: 1994/01/05 20:06:13 $";
#endif

/*
 * otConvert.c - convert a CR in ASCII format to Tcl.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <tcl.h>
#include <tclInt.h>

#include "ot.h"
#include "otInt.h"

OTErr otWriteTemplateToTcl();
OTErr otConvertToFieldCmd();
OTErr otGenQueryContext();
OTErr otQueryEstimateStringLength();
OTErr otWriteFileToString();

OTErr
otWriteTemplateToTcl(pjp, tp, cpp, origFlag)
OTProject *pjp;     
OTTemplate *tp;
char **cpp;
bool origFlag;
{
    OTErr err = OT_SUCCESS;
    register OTHeaderField *hfp;
    register OTNotedata *ndp;
    register OTNote *np;
    register char *cp;
    register OTMetaTemplate *mtp, *tmp_mtp;
    bool useFieldCmd;
    int len, result, estLen, scanFlags, stat;
    char *origStr, *val, *kwd, *cvtVal;
    char *cvtAgent, *cvtField, *cvtPrevious, *cvtCurrent, *label;
    char flag;
    char *tName, *oName, *dName;
    Tcl_Interp *interp = otCB->cb_pcb->pcb_interp;

    if ( !tp || !(hfp = tp->tr) ) {
	otPutPostedMessage( OT_INTERNAL_ERROR,
	    "otWriteTemplateToTcl(): template pointer null");
	return OT_INTERNAL_ERROR;
    }

    len = 0;
    if (origFlag)
	origStr = "orig ";
    else
	origStr = "";

    /*
     * Estimate size of TCL strings required to represent the header fields
     * which match fields in the metatemplate: add up field name, 'set'
     * option, value and semicolon command terminator.
     */
    for ( mtp = pjp->otmp; mtp->field && *mtp->field; mtp++ ) {

	if ( (*mtp->field == '!') || !mtp->abbr )
	    continue;

	if (err = otConvertToFieldCmd(&useFieldCmd, pjp, 0, mtp->abbr))
	    return err;

	if ( !useFieldCmd ) {
	    len += strlen( mtp->abbr );

	    if ( origFlag )
		len += sizeof(" orig");
	    len += sizeof(" set ");

	    if ( val = otGetHeaderFieldValue(mtp->field, tp) ) {
		estLen = Tcl_ScanElement( val, &scanFlags );
		DBUG_MED((stderr, "estimated length of %s is %d\n", val, estLen));
		len += estLen;
	    } else
		len += sizeof("{}");
	}
	len += sizeof("; ");
    }
    DBUG_MED((stderr, "total of metatemplate header is %d\n", len));

    /*
     * Estimate size of TCL strings required to represent the header fields
     * which do NOT match fields in the metatemplate.
     */
    for ( ; hfp && hfp->field && *hfp->field; hfp++ ) {

	if ( err = otConvertToFieldCmd(&useFieldCmd, pjp, hfp->field, 0) )
	    return err;

	/*
	 * Either the field name is not associated with an abbreviation, or
	 * the abbreviation is not recognized by the server.
	 */
	if ( useFieldCmd ) {
	    /*
	     * First command:
	     *
	     *  field [orig] myFieldName [append] { type option degree }
	     */
	    len += sizeof("field append ");
	    if (origFlag)
		len += sizeof("orig ");
	    /*
	     * TODO: use Tcl_{Scan,Convert}Element() to estimate the converted
	     * length of the following strings.
	     */
	    len += strlen(hfp->field);
	    if ( hfp->list )
		len += strlen(hfp->list);

	    /*
	     * Second command:
	     *
	     *  field [orig] myFieldName set <value>
	     */
	    len += sizeof("field set ");
	    if (origFlag)
		len += sizeof("orig ");
	    len += strlen(hfp->field);
	    
	    if (hfp->value) {
		estLen = Tcl_ScanElement( hfp->value, &scanFlags );
		DBUG_MED((stderr, "estimated length of %s is %d\n", hfp->value, estLen));
		len += estLen;
	    } else
		len += sizeof(" {}");
	    len += sizeof("; ");
	}
    }
    DBUG_MED((stderr, "total of non-metatemplate header is %d\n", len));

    len += sizeof("notedata ");
    if (origFlag) {
	len += sizeof("orig ");
    }

    for ( ndp = tp->ndp; ndp && ndp->nd_flags; ndp++ ) {

        /*
	 * Allocate room for list of form
	 *     0 Thu 05/27/93 13:38:01 HISTORY pnh stat open closed 
	 *
	 * Again, strings are not limited in length in OT so we need to
	 * determine their dimensions dynamically.
	 */
	len += sizeof(char);				    /* flag 	    */
	len += sizeof("Mon");
	len += sizeof("mm/dd/yy") + sizeof("00:00:00") + sizeof("HISTORY");


	len += (ndp->nd_agent)    ? strlen(ndp->nd_agent)    : 3;
	len += (ndp->nd_field)    ? strlen(ndp->nd_field)    : 3; 
	len += (ndp->nd_previous) ? strlen(ndp->nd_previous) : 3;
	len += (ndp->nd_current)  ? strlen(ndp->nd_current)  : 3;

    }

    /*
     * Estimate size of note data.  Right now, who, date and sens fields
     * are also in the note itself, so just compute the length of the note.
     */
    for ( np = tp->notep; np && np->text; np++ ) {
	estLen = Tcl_ScanElement( np->text, &scanFlags );
	DBUG_MED((stderr, "estimated length of note %s is %d\n", np->text, estLen));
	len += estLen;
	if (origFlag)
	    len += sizeof("orig ");
    }

    DBUG_MED((stderr, "length for allocation is %d\n", len));

    /*
     * The doubling of the approximate length is patterned after the 
     * action of the Tcl_ScanElement() routine.
     */
    if ( !(*cpp = calloc(len * 2, sizeof(char))) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "otWriteTemplateToTcl()");
	return OT_MALLOC_LOCATION;
    }
    cp = *cpp;

    sprintf(cp, "clear %sheader; clear %snotes; \n", origStr, origStr);
    cp += strlen(cp);

    for ( mtp = pjp->otmp; mtp->field[0]; mtp++ ) {
	if ( (mtp->field[0] == '!') || (mtp->abbr[0] == 0) )
	    continue;
	if (err = otConvertToFieldCmd(&useFieldCmd, pjp, 0, mtp->abbr))
	    return err;

	if ( !useFieldCmd ) {
	    if ( val = otGetHeaderFieldValue(mtp->field, tp) ) {
		estLen = Tcl_ScanElement( val, &scanFlags );
		if ( !(cvtVal = malloc(estLen)) ) {
		    otPutPostedMessage(OT_MALLOC_LOCATION,
		        "otWriteTemplateToTcl()");
		    return(OT_MALLOC_LOCATION);
		}
		result = Tcl_ConvertElement(val, cvtVal, scanFlags);

		sprintf(cp, "%s %sset ", mtp->abbr, origStr);
		cp += strlen(cp);

		strcat(cp, cvtVal);
		strcat(cp, "; ");
		cp += strlen(cp);

		free(cvtVal);
	    } else {
		sprintf(cp, "%s %sset {}; ", mtp->abbr, origStr);
		cp += strlen(cp);
	    }
	    DBUG_MED((stderr, "Converted string is |%s|\n", *cpp));
        }
    }

    for ( hfp = tp->tr; hfp && hfp->field && *hfp->field; hfp++ ) {

	if ( err = otConvertToFieldCmd(&useFieldCmd, pjp, hfp->field, 0) )
	    return err;

	if ( useFieldCmd ) {
	    /*
	     * First command:
	     *  field [orig] myFieldName append { type option degree }
	     */
	    estLen = Tcl_ScanElement( hfp->field, &scanFlags );
	    if ( !(cvtVal = malloc(estLen)) ) {
		otPutPostedMessage(OT_MALLOC_LOCATION, "otWriteTemplateToTcl()");
		return(OT_MALLOC_LOCATION);
	    }
	    result = Tcl_ConvertElement(hfp->field, cvtVal, scanFlags);
	    sprintf(cp, "field %s%s append ", origStr, cvtVal);
	    free(cvtVal);
	    cp += strlen(cp);

	    oName = (hfp->optionality == OPT_MAND) ? "mandatory" : "optional";
	    dName = (hfp->degree == DEGREE_MANY) ? "many" : "one";

	    tName = otGetTypeName(hfp->type);
	    if ( !tName || !*tName ) {
		sprintf(cp, " { {} %s %s }\n", oName, dName);
		cp += strlen(cp);
	    } else if ( !strcmp(tName, "enum") ) {
		estLen = Tcl_ScanElement( hfp->list, &scanFlags );
		if ( estLen && !(cvtVal = malloc(estLen)) ) {
		    otPutPostedMessage(OT_MALLOC_LOCATION, "otWriteTemplateToTcl()");
		    return(OT_MALLOC_LOCATION);
		}
		result = Tcl_ConvertElement(hfp->list, cvtVal, scanFlags);
		sprintf(cp, " { %s %s %s }\n", cvtVal, oName, dName);
		cp += strlen(cp);
		free(cvtVal);
	    } else {
		sprintf(cp, " { %s %s %s }\n", tName, oName, dName);
		cp += strlen(cp);
	    }


	    /*
	     * Second command:
	     * field [orig] myFieldName set {value}
	     */
	    estLen = Tcl_ScanElement( hfp->field, &scanFlags );
	    if ( !(cvtVal = malloc(estLen)) ) {
		otPutPostedMessage(OT_MALLOC_LOCATION, "otWriteTemplateToTcl()");
		return(OT_MALLOC_LOCATION);
	    }
	    result = Tcl_ConvertElement(hfp->field, cvtVal, scanFlags);
	    sprintf(cp, "field %s%s set ", origStr, cvtVal);
	    free(cvtVal);
	    cp += strlen(cp);
	    
	    if (hfp->value) {
		estLen = Tcl_ScanElement( hfp->value, &scanFlags );
		if ( !(cvtVal = malloc(estLen)) ) {
		    otPutPostedMessage(OT_MALLOC_LOCATION,
		        "otWriteTemplateToTcl()");
		    return(OT_MALLOC_LOCATION);
		}
		result = Tcl_ConvertElement(hfp->value, cvtVal, scanFlags);
		strcat(cp, cvtVal);
		strcat(cp, "; ");
		cp += strlen(cp);
		free(cvtVal);
	    } else {
		sprintf(cp, " {}; ");
		cp += strlen(cp);
	    }
	}
    }

    sprintf(cp, "\n");
    cp += strlen(cp);

    /*
     * Write notedata as a command in the form
     *
     * notedataBuild { {first history (or other notedata) line} {next} {etc} }
     */
    ndp = tp->ndp;
    if ( ndp && ndp->nd_flags ) {
	sprintf(cp, "notedataBuild %s{ ", origStr);
	cp += strlen(cp);
    }
    for ( ; ndp && ndp->nd_flags; ndp++ ) {

	if (      ND_ISRESERVED( ndp->nd_flags ) )
	    flag = '0';
	else if ( ND_ISVISIBLE( ndp->nd_flags ) )
	    flag = '1';
	else
	    flag = '2';

	if ( ND_ISHISTORY( ndp->nd_flags ) )
	    label = "HISTORY";
        else if ( ND_ISLUPDATE(ndp->nd_flags) )
	    label = "LUPDATE";
	else
	    label = ndp->nd_kwd;

	sprintf(cp, "{ %c %s %02d/%02d/%02d %c%c%c%c%c%c%c%c %s ",
	    flag, dayNameForNumber(ndp->nd_weekday),
	    ndp->nd_mon, ndp->nd_day, ndp->nd_year,
	    ndp->nd_time[0], ndp->nd_time[1], ndp->nd_time[2],
	    ndp->nd_time[3], ndp->nd_time[4], ndp->nd_time[5],
	    ndp->nd_time[6], ndp->nd_time[7], label);
	cp += strlen(cp);

	cvtAgent = cvtField = cvtPrevious = cvtCurrent = 0;
	if ( ndp->nd_agent ) {
            estLen = Tcl_ScanElement( ndp->nd_agent, &scanFlags );
	    if ( !(cvtAgent = malloc(estLen)) ) {
		otPutPostedMessage(OT_MALLOC_LOCATION,
		    "otWriteTemplateToTcl()");
		return(OT_MALLOC_LOCATION);
	    }
	    result = Tcl_ConvertElement(ndp->nd_agent, cvtAgent, scanFlags);
	}

	if ( ndp->nd_field ) {
	    estLen = Tcl_ScanElement( ndp->nd_field, &scanFlags );
	    if ( !(cvtField = malloc(estLen)) ) {
		otPutPostedMessage(OT_MALLOC_LOCATION,
		    "otWriteTemplateToTcl()");
		return(OT_MALLOC_LOCATION);
	    }
	    result = Tcl_ConvertElement(ndp->nd_field, cvtField, scanFlags);
	}

	if ( ndp->nd_previous ) {
	    estLen = Tcl_ScanElement( ndp->nd_previous, &scanFlags );
	    if ( !(cvtPrevious = malloc(estLen)) ) {
		otPutPostedMessage(OT_MALLOC_LOCATION,
		    "otWriteTemplateToTcl()");
		return(OT_MALLOC_LOCATION);
	    }
	    result = Tcl_ConvertElement(ndp->nd_previous, cvtPrevious,
		scanFlags);
	}

	if ( ndp->nd_current ) {
	    estLen = Tcl_ScanElement( ndp->nd_current, &scanFlags );
	    if ( !(cvtCurrent = malloc(estLen)) ) {
		otPutPostedMessage(OT_MALLOC_LOCATION,
		    "otWriteTemplateToTcl()");
		return(OT_MALLOC_LOCATION);
	    }
	    result = Tcl_ConvertElement(ndp->nd_current, cvtCurrent, scanFlags);
	}

        sprintf(cp, "%s %s ", 
	    cvtAgent    == 0 ? "{}" : cvtAgent, 
	    cvtField    == 0 ? "{}" : cvtField);
	cp += strlen(cp);

	if (!cvtPrevious || !strcmp(cvtPrevious, "NULL")) {
	    sprintf(cp, "{} ");
	} else {
	    strcat(cp, cvtPrevious);
	    strcat(cp, " ");
	}
	cp += strlen(cp);

	if (!cvtCurrent || !strcmp(cvtCurrent, "NULL")) {
	    sprintf(cp, "{} } ");
	} else {
	    strcat(cp, cvtCurrent);
	    strcat(cp, " } ");
	}
	cp += strlen(cp);

	if (cvtAgent)
	    free(cvtAgent);
	if (cvtField)
	    free(cvtField);
	if (cvtPrevious)
	    free(cvtPrevious);
	if (cvtCurrent)
	    free(cvtCurrent);

	cp += strlen(cp);
    }

    if ( tp->ndp && tp->ndp->nd_flags ) {
	sprintf(cp, " }\n");
	cp += strlen(cp);
    }

    sprintf(cp, "\n");
    cp += strlen(cp);

    for ( np = tp->notep; np && np->text; np++ ) {
	estLen = Tcl_ScanElement(np->text, &scanFlags);
	if ( !(cvtVal = malloc(estLen)) ) {
	    otPutPostedMessage(OT_MALLOC_LOCATION, "otWriteTemplateToTcl()");
	    return OT_MALLOC_LOCATION;
	}
	result = Tcl_ConvertElement(np->text, cvtVal, scanFlags);

	sprintf(cp, "note %sappend ", origStr);
	cp += strlen(cp);

	strcat(cp, cvtVal);
	strcat(cp, "\n");
	cp += strlen(cp);
	free(cvtVal);
    }

    DBUG_MED((stderr, "Converted string is |%s|\n", *cpp));
    return err;
}


/*
 *                  o t C o n v e r t T o F i e l d C m d
 *
 * This is required because OT still supports 'alternative' metatemplates.
 * These are metatemplates with a set of header fields which is a 
 * SUPERSET of the project metatemplate (the metatemplate stored on the
 * server host for the project).
 *
 * When the server calls this routine:
 *
 * If the server metatemplate does not have an abbreviation for this
 * field name, then return TRUE: the TCL generated for this field will use
 * the 'field' command.  Otherwise return FALSE: the server generates a
 * 'keyword' command (e.g. 'stat set open').
 *
 * When the client calls this routine:
 *
 * If the abbreviation in the client metatemplate for this field name is not
 * an element in the list of the server's abbreviations, as stored in
 * the OTProject server_fields member, then return TRUE: the client uses
 * the 'field' command.  Otherwise return FALSE: the client generates a
 * 'keyword' command.
 * 
 */
OTErr
otConvertToFieldCmd(useFieldCmd, pjp, fldName, abbr)
bool *useFieldCmd;
OTProject *pjp;
char *fldName;
char *abbr;
{
    char *sargv[3];
    char *kwd, *fnm;
    ClientData notUsed;
    OTErr err = OT_SUCCESS;
    OTPrivateCB *pcp = otCB->cb_pcb;
    Tcl_Interp *interp = pcp->pcb_interp;
    char cmdBuf[LONGVALUE];

    sargv[0] = "lsearch";
    *useFieldCmd = FALSE;

    notUsed = (ClientData)0;
    if ( !abbr && !fldName ) {
	otPutPostedMessage( OT_INTERNAL_ERROR, "argument errors in otConvertToFieldCmd()" );
	return OT_INTERNAL_ERROR;
    }

    if ( pcp->pcb_imTheServer ) {
	if ( fldName ) {
	    if ( !(kwd = otGetFieldNameAbbr(fldName, pjp)) ) {
		*useFieldCmd = TRUE;
		return err;
	    }
	}
	if ( abbr ) {
	    if ( !(fnm = otGetAbbrFieldName(abbr, pjp)) ) {
		*useFieldCmd = TRUE;
		return err;
	    }
	}
    } else {
        if ( abbr ) 
	    sargv[2] = abbr;
	else {
	    if ( !(sargv[2] = otGetFieldNameAbbr(fldName, pjp)) ) {
		*useFieldCmd = TRUE;
		return err;
	    }
	}

	if (!pjp->server_fields) {
	    otPutPostedMessage( OT_INTERNAL_ERROR,
		"no server fields to process" );
	    return OT_INTERNAL_ERROR;
	} else
	    sargv[1] = pjp->server_fields;

	if ( Tcl_LsearchCmd(notUsed, interp, 3, (char **) sargv) ) {
	    otPutPostedMessage( OT_INTERNAL_ERROR,
		"otConvertToFieldCmd(): error in lsearch" );
	    return OT_INTERNAL_ERROR;
	}

	if ( !strcmp(interp->result, "-1") )
	    *useFieldCmd = TRUE;
    }

    return err;

}


/*
 * otGenQueryContext() - generate a TCL string reflecting the current context.
 */
OTErr otGenQueryContext(ctxtString)
char **ctxtString;
{
    OTErr err = OT_SUCCESS;
    bool filterHasValues;
    int len = 0, rcLen;
    int sargc;
    int filterValuesLen;
    char *sargv[10];
    char *cp, *tmpStr, *filterVal;
    OTQueryCB *qcb = otCB->cb_qcb;
    OTPrivateCB *pcb = otCB->cb_pcb;
    OTMetaTemplate *tmp_mtp;

#define ESTIMATE_LENGTH( field )  if ( ( field ) ) len += strlen( (field) )

#define APPEND_FROM_SHORTVAL( label, member )				\
    sprintf(cp, "{ %s %s } ", ( label ), ( member ) ? "1" : "0");	\
    cp += strlen(cp)

#define APPEND_FROM_CB( label , member )				\
    sargv[0] = ( ( label ) );						\
    if ( ! ( member ) )							\
	sargv[1] = "";							\
    else								\
	sargv[1] = ( ( member ) );					\
    sargv[2] = (char *)NULL;						\
    sargc = 2;								\
    tmpStr = Tcl_Merge(sargc, sargv);					\
    sprintf(cp, "{ %s } ", tmpStr);					\
    cp += strlen(cp);							\
    free(tmpStr)

    /*
     * Estimate the 'variable length' fields' total in bytes.
     */
    ESTIMATE_LENGTH( otCB->cb_layout         );
    ESTIMATE_LENGTH( qcb->qcb_keyInText      );
    ESTIMATE_LENGTH( otCB->cb_tclString      );
    ESTIMATE_LENGTH( otCB->cb_qcb->qcb_nsens );

    if ( err = otQueryEstimateStringLength( &filterHasValues, qcb->qcb_filter,
	&filterValuesLen ) ) {
	return err;
    }

    if ( !(*ctxtString = malloc(LONGVALUE + len + filterValuesLen)) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "otGenQueryContext()");
	return OT_MALLOC_LOCATION;
    }
    cp = *ctxtString;

    
    sprintf(cp, "control { ");
    cp += strlen(cp);

    APPEND_FROM_SHORTVAL( "skipHeader",	  qcb->qcb_skipHeader   );
    APPEND_FROM_SHORTVAL( "count",	  qcb->qcb_count        );
    APPEND_FROM_SHORTVAL( "summary",	  qcb->qcb_summary      );
    APPEND_FROM_SHORTVAL( "tabSeparate",  qcb->qcb_tabSeparate  );
    APPEND_FROM_SHORTVAL( "fullText",	  qcb->qcb_fullText     );
    APPEND_FROM_SHORTVAL( "historyLines", qcb->qcb_historyLines );
    APPEND_FROM_SHORTVAL( "quiet",	  pcb->pcb_quiet        );

    APPEND_FROM_CB( "layout",		  otCB->cb_layout	);
    APPEND_FROM_CB( "keyInText",	  qcb->qcb_keyInText	);
    APPEND_FROM_CB( "nsens",		  qcb->qcb_nsens	);
    APPEND_FROM_CB( "tclFile",		  otCB->cb_tclFile      );
    APPEND_FROM_CB( "tclOtrc",		  otCB->cb_tclOtrc      );

    sprintf(cp, "{ outwidth %d } ", qcb->qcb_outwidth);
    cp += strlen(cp);

    if ( otCB->cb_tclString ) {
	APPEND_FROM_CB( "tclString", otCB->cb_tclString	 );
    } else {

	if ( filterHasValues ) {
	    bool isFirstClause;
	    char *tclXlat, *tclXlatbase;
	    
	    if ( !(tclXlat = calloc(filterValuesLen + LONGVALUE, sizeof(char))) ) {
		otPutPostedMessage(OT_MALLOC_LOCATION, "otGenQueryContext()");
		return OT_MALLOC_LOCATION;
	    }
	    tclXlatbase = tclXlat;
	    isFirstClause = TRUE;
	    
	    /*
	     * if ranges are present, start off with the TCL xlation of these.
	     */

	    for ( tmp_mtp = qcb->qcb_filter; tmp_mtp->field[0]; tmp_mtp++ ) {

		if ( (tmp_mtp->type == TYPE_IDNUM) &&
		         otCB->cb_qcb->qcb_hiRange &&
			 otCB->cb_qcb->qcb_loRange && 
			 otCB->cb_qcb->qcb_hiRange != 1000000 ) {

		    sprintf(tclXlat, 
			    "%s[expr \"[%s] <= %d\"] && [expr \"[%s] >= %d\"]",
			    isFirstClause ? "" : " && ",
			    tmp_mtp->abbr, otCB->cb_qcb->qcb_hiRange,
			    tmp_mtp->abbr, otCB->cb_qcb->qcb_loRange);
		    tclXlat += strlen(tclXlat);
		    isFirstClause = FALSE;
		} else {
		    if ( tmp_mtp->line[0] ) {
		        sargc = 1;
			sargv[0] = tmp_mtp->line;
			sargv[1] = (char *)NULL;
			filterVal = Tcl_Merge(sargc, sargv);
			sprintf(tclXlat, "%s[%s %s]",
			    isFirstClause ? "" : " && ",
			    tmp_mtp->abbr, filterVal);
			free(filterVal);
		    } else if ( tmp_mtp->unassigned )
			sprintf(tclXlat, "%s[%s {}]",
			    isFirstClause ? "" : " && ",
			    tmp_mtp->abbr);
		    tclXlat += strlen(tclXlat);
		    if ( tmp_mtp->line[0] || tmp_mtp->unassigned )
			isFirstClause = FALSE;
		}
	    }
	    
	    APPEND_FROM_CB( "tclString", tclXlatbase );
	    free(tclXlatbase);
	} else {
	    sprintf(cp, "{ tclString {} } ");
	    cp += strlen(cp);
	}
    }
    sprintf(cp, " }");

    return err;

}

OTErr
otQueryEstimateStringLength(filterHasValues, mt, len)
bool *filterHasValues;
OTMetaTemplate *mt;
int *len;
{
    OTErr err = OT_SUCCESS;
    OTMetaTemplate *tmp_mtp;
    int valueLen;

    *filterHasValues = FALSE;
    *len = 0;
    tmp_mtp = mt;

    for ( ; tmp_mtp->field[0]; tmp_mtp++ ) {
        valueLen = 0;
	if ( tmp_mtp->type == TYPE_IDNUM && tmp_mtp->range )
	    valueLen += strlen(tmp_mtp->range);
	if ( tmp_mtp->line[0] ) {
	    *filterHasValues = TRUE;
	    valueLen += strlen(tmp_mtp->line);
	} else if ( tmp_mtp->unassigned == TRUE ) {
	    *filterHasValues = TRUE;
	    valueLen += 2; /* length of "" */
	}
	if ( valueLen )
	    *len += valueLen + strlen(tmp_mtp->abbr);
    }

    if ( otCB->cb_pcb->pcb_filterNum ) {
	/*
	 * This pcb_filterNum value would NOT be in any mtp->line array,
	 * because it would be zeroed out in otSetFilterValue() [otUtil.c].
	 */
	*len += SHORTSTR;
    } else if ( otCB->cb_qcb->qcb_loRange || otCB->cb_qcb->qcb_hiRange ) {
	*len += SHORTSTR;
    } else if ( otCB->cb_qcb->qcb_nsens ) {
	*len += SHORTSTR;
    }

    return err;

}

/*
 * otWriteFileToString(char *fn, char **str) - store file contents as string
 * Allocates string (which caller must free) and stores contents of file
 * fn in it.
 */

OTErr otWriteFileToString(fn, str)
char *fn;
char **str;
{
    OTErr err = OT_SUCCESS;
    struct stat statbuf;
    int fd, n;

    *str = 0;
    if ( stat(fn, &statbuf) < 0 ) {
	err = OT_STAT;
    } else if ( !(*str = calloc(statbuf.st_size + 1, sizeof(char))) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "otWriteFileToString()");
	err = OT_MALLOC_LOCATION;
    } else if ( (fd = open(fn, O_RDONLY)) < 0 ) {
	otPutPostedMessage(OT_OPEN_INPUT, fn);
	err = OT_OPEN_INPUT;
    } else if ( (n = read(fd, *str, statbuf.st_size)) != statbuf.st_size ) {
	close(fd);
	otPutPostedMessage(OT_OPEN_INPUT, fn);
	err = OT_OPEN_INPUT;
    } else
	close(fd);

    if ( err && *str )
	free(*str);

    return err;

}
