/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */

/*
 *		otEnter.c
 *
 */


#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"



#ifndef LINT
static char RCSid_otEnter[] =
    "$RCSfile: otEnter.c,v $ $Revision: 1.1.6.2 $ $Date: 1994/01/25 19:27:32 $";

#endif


/*
 *		List of functions defined in the otEnter.c
 *
 *	otEnterObject()
 *
 *
 *
 *
 */

OTErr otAdminCRFile();
OTErr otFormEnteredHist();
char *doUnLock();


OTErr
otEnterObject(tmpfile)
char    *tmpfile;	/* template file in /tmp */
{
    long CRNumber;
    char aCRnum[10];
    OTFilen * fname;
    char name[NAMELEN];
    char *subject_line;
    char *cp1;

    OTProject *proj = otCB->cb_pcb->pcb_project;
    OTErr errCode = OT_SUCCESS;

    /* Calls server to get new object number */

    if (errCode = otGetnCRnum(proj, &CRNumber)) 
	return errCode;

    /* Places object number into template structure */

    sprintf(aCRnum, "%-6d", CRNumber);
    if (errCode = otSetIDNumValue(aCRnum, otCB->cb_ecb->ecb_tStruct))
	return errCode;

    /* Write final version of a template structure back to the file */

    if (errCode = otWriteTemplateToFilename(otCB->cb_ecb->ecb_tStruct, tmpfile))
	return errCode;

/********/
    if (errCode = otFormEnteredHist(tmpfile))
	return errCode;
/*******/

    /* Create an edit version of the CR filename */
    if (errCode = otFormName(proj, CRNumber, &fname)) {
	if (fname)
	    free(fname);
	return errCode;
    }

    otCB->cb_ecb->ecb_fname = fname;
    strcat(fname->name, ".edit");

    /* Got template ready -- create new obj */

    if ((errCode = otAdminCRFile(fname, tmpfile, proj)) == OT_OBJECT_CREATED) {
	errCode = OT_SUCCESS;
    }
    
    return errCode;
}



OTErr
otGetnCRnum(proj, CRNumber)
OTProject * proj;
long * CRNumber;
{
    register char * aCRnum;
    char request[NAMELEN];
    char *getCRNumber();

    OTErr errCode = OT_SUCCESS;

    aCRnum = getCRNumber(otCB->cb_pcb->pcb_uName, proj->name);

    *CRNumber = atol(aCRnum);        /* get binary representation */

    if (*CRNumber <= 0) {
	otPutPostedMessage(OT_GET_NUMBER, proj->object_name);
	errCode = OT_GET_NUMBER;
    }

    return errCode;
}



/* Form the "entered" history line */
OTErr
otFormEnteredHist(tmpfile)
char    *tmpfile;		/* template file */
{
    FILE *tfp;
    long timer;
    struct tm *timeptr;
    char times[NAMELEN];

    OTProject * proj = otCB->cb_pcb->pcb_project;

    OTErr errCode = OT_SUCCESS;

    if ((tfp = fopen(tmpfile, "a+")) == NULL ) {
	otPutPostedMessage(OT_TEMPLATE_UPDATE, tmpfile, proj->object_name);
	return OT_TEMPLATE_UPDATE;
    }

    timer = time(0L);
    timeptr = localtime(&timer);
    strcpy(times, asctime(timeptr));
    times[strlen(times) - 1] = 0;


    DBUG_MED((stderr, "Adding the Created history line\n"));

    fprintf(tfp, "+0HISTORY %s %s\tCreated\n", times, otCB->cb_pcb->pcb_uName);
    fclose(tfp);

    /*
     * Put the new information in the OTNotedata structure.  (This routine
     * could be rewritten to deposit the history info directly in ecb_tStruct's
     * notedata area.
     */
    if ( otCB->cb_ecb->ecb_tStruct )
	otFreeTemplate(otCB->cb_ecb->ecb_tStruct);

    errCode = otReadTemplateFromFilename( &(otCB->cb_ecb->ecb_tStruct),
	tmpfile, TRUE, TRUE );

    return errCode;
}


/*
 *                            a d m i n C R F i l e
 *
 * Create first version of CR file (admin).
 *
 * Arguments:
 *  CRFile      = CR file to lock for editing.
 *  tfile       = template file with new content.
 *  proj        = ptr to project definition structure.
 *
 * Returns status from deltaCRFile().
 */

OTErr
otAdminCRFile(CRFile, tfile, proj)
OTFilen * CRFile;
char * tfile;
OTProject *proj;
{
    register int i;
    OTFilen newFile;
    char msg[LONGVALUE];

    OTErr errCode = OT_SUCCESS;

    /* form OTFilen structure for the file name holding an object */
    otFormFilen(&newFile, tfile);

    sprintf(msg, "new %s", proj->object_name);

    if ((errCode = otDeltaCRFile(CRFile, newFile, msg, TRUE, proj)) 
		    == OT_RCS_CI) {
	otPutPostedMessage(OT_RCS_ADMIN, proj->object_name, CRFile->name);
	errCode = OT_RCS_ADMIN;
    }
    return errCode;
}

/*
 *                           o t R e q u e s t E n t e r
 *
 * This is the call of the client.
 */
/* how will we know where the existing temp file is? */
OTErr
otRequestEnter()
{
    int status;
    char prjCmd[LONGVALUE], fldCmd[LONGVALUE];
    char name[NAMELEN];
    char *tclString;
    OTPrivateCB *pcp;
    OTProject *pjp;
    OTTemplate *ott_template;
    OTFilen fname;
    OTErr err;
    bool orig;
    Tcl_Interp *interp;
    char *xString, *newCR;

    /*
     * via otRemoteTcl...
     *
     * . if OT_DBPATH is set, send it
     * project set <projName>
     * enter
     * send template in TclString format
     * send <tclString> (user's -x argument)
     * send commit
     * send tclString 'the command' so we refresh the CR in ecb_tStruct...
     */

    newCR = 0;
    orig = FALSE;
    pcp = otCB->cb_pcb;
    ott_template = otCB->cb_ecb->ecb_tStruct;
    interp = pcp->pcb_interp;

    xString = otCB->cb_tclString;

    if ( pjp = pcp->pcb_project ) {
	sprintf(prjCmd, "project set %s\n", pjp->name);

	if ( status = otRemoteTcl(prjCmd) ) {
	    otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	    return OT_GENERAL_ERROR;
	}
	DBUG_MIN((stderr, "sent >> %s\n", prjCmd));

	sprintf(fldCmd, "project fields\n");
	if ( status = otRemoteTcl(fldCmd) ) {
	    otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	    return OT_GENERAL_ERROR;
	}
	DBUG_MIN((stderr, "sent >> %s\n", fldCmd));
	if ( otCopyString( interp->result, &(pjp->server_fields)) ) {
	    otPutPostedMessage(OT_MALLOC_LOCATION, "otRequestEnter()");
	    return OT_MALLOC_LOCATION;
	}
	DBUG_MIN((stderr, "received >> %s\n", pjp->server_fields));

    } else {
	otPutPostedMessage(OT_NEED_PROJECT);
	return OT_NEED_PROJECT;
    }

    if (status = otRemoteTcl("enter\n")) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_GENERAL_ERROR;
    }
    DBUG_MIN((stderr, "sent >> enter\n"));


    if ( err = otWriteTemplateToTcl(pjp, ott_template, &tclString, orig) ) {
	return err;
    }

    /* TODO.
     * what error processing required here - we need to send exit to the server
     * if we can't build any tcl strings.
     */

    status = otRemoteTcl(tclString);
    DBUG_MIN((stderr, "sent >> %s\n", tclString));

    free(tclString);
    if (status) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_GENERAL_ERROR;
    }

    if ( xString ) {
	if (status = otRemoteTcl(xString)) {
	    otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	    return OT_GENERAL_ERROR;
	}
	DBUG_MIN((stderr, "sent >> %s\n", xString));
    }

    if (status = otRemoteTcl("commit\n")) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_GENERAL_ERROR;
    } else if ( *interp->result ) {
	otPutPostedMessage(OT_INVALID, interp->result);
	fprintf(stderr, "%s\n", otGetPostedMessage() );
/*	return OT_INVALID; */
    }
    DBUG_MIN((stderr, "sent >> commit\n"));

    if (status = otRemoteTcl("tclString\n")) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_GENERAL_ERROR;
    } else {
        if ( otCopyString(interp->result, &newCR) ) {
	    otPutPostedMessage(OT_MALLOC_LOCATION, "otRequestEnter()");
	    return OT_MALLOC_LOCATION;
	}

	DBUG_MIN((stderr, "received >> %s\n", newCR));
	err = otTclEval(newCR);
	free(newCR);
	if (err)
	    return err;
    }

    return OT_SUCCESS;

}

/*                         g e t C R N u m b e r
 *
 *  Get the next CR number for the named project
 */

char *
getCRNumber(who, msg)
char *  who;    /* who made the request */
char *  msg;    /* request sent by client -- project name */
{
    register int i;
    register long CRNumber;
    char file[PATHLEN];
    char lockFile[PATHLEN];
    static char aCRnum[NAMELEN];
    char next_aCRnum[NAMELEN];
    char * omode;
    char * prjdir = 0;
    FILE * ifile;
    struct stat statbuf;

    OTErr errCode = OT_SUCCESS;

    /* Directory place and name of number file */

    sprintf(aCRnum, "0");
    
    if (errCode = otGetProjectBase(msg, &prjdir)) {
	otPutPostedMessage(OT_PROJECT_NOT_FOUND, msg);
	logWarn("%s", otGetPostedMessage());
	return aCRnum;
    }
    sprintf(file, "%s/%s/number", prjdir, msg);
    if (prjdir)
	free(prjdir);

    strcpy(lockFile, file);             /* and lock file */
    strcat(lockFile, ".lock");

    /* Get exclusive access to number file */

    if (!makeLock(who, lockFile, 60)) {     /* create lock */
	logWarn("couldn't create lock file %s", lockFile);
	return aCRnum;
    }

    /* Got lock -- read and update CR number */

    if (stat(file, &statbuf))
	omode = "w+";
    else
	omode = "r+";

    if ((ifile = fopen(file, omode)) == NULL) {
	logWarn("can't get number file -- errno %d", errno);
	return aCRnum;	
    }

    if (!strcmp(omode, "w+")) {
	chmod(file, 0666);
    }

    if ((i = fread(aCRnum, 1, NAMELEN, ifile)) == 0) {
	logWarn("Initializing number file: see tools group if not first entry", "");
	CRNumber = 1;
    } else
	CRNumber = atol(aCRnum);        /* get binary representation */

    sprintf(next_aCRnum, "%06ld\n", CRNumber+1);

    rewind(ifile);              /* reset file */

    if (fwrite(next_aCRnum, strlen(next_aCRnum), 1, ifile) != 1)    {
	logWarn("failed writing new number -- errno %d", errno);
	sprintf(aCRnum, "0");
	return aCRnum;
    }

    fsync(ifile);
    fclose(ifile);

/* done - remove lock file */

    doUnLock(who, lockFile);

    return aCRnum;
}

/*  makeLock() -- low level routine to actually create a lock file
 *
 */
bool
makeLock(who, file, waitCount)
char * who;     /* who wants to make the lock */
char * file;    /* name of lock file */
int waitCount;  /* count of how long to wait for lock */
{
    register int i, fd;
    char msg[NAMELEN];


/* exclusive open lock */

    for (i=0;  i < waitCount;  i++) {

	if ((fd = open(file, O_WRONLY | O_EXCL | O_CREAT, 0777)) >= 0)  {
	    chmod(file, 0666);
	    sprintf(msg, "locked by: %s\n", who);
	    write(fd, msg, strlen(msg));
#if DEBUG
	    logWarn("created lock file %s", file);
#endif
	    close(fd);          /* don't need to keep it open */
	    return TRUE;
	}

	if (i % 10) {
	    sprintf(msg, "waiting for lock on %s, errno %d", file, errno);
	    logWarn(msg);
	}
	sleep(1);
    }

    logWarn("couldn't make lock file %s -- see tools group", file);
    return FALSE;
}


/*
 *                               d o U n L o c k
 *
 * Destroy a lock file
 */


char *
doUnLock(who, msg)
char *  who;    /* who made the request */
char *  msg;    /* request sent by client */
{

    if (!removeLock(msg))
        logWarn("couldn't unlink lock file %s", msg);
    else    {
#ifdef DEBUG
        logWarn("unlinked lock file %s", msg);
#endif

    }

    return OKRESPONSE;
}


/*  removeLock() -- low level routine to delete a lock file
 *
 */
bool
removeLock(file)
char * file;    /* name of lock file */
{
    return (unlink(file) == 0);
}
