/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */

/*
 * otTcl.c --
 *
 *	Contains the functions associated with OT TCL commands.
 */

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/mman.h>
#include <string.h>
#include <pwd.h>
#include <time.h>

#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"

#include "dproto.h"

#ifndef LINT
static char RCSid_otTcl[] =
    "$RCSfile: otTcl.c,v $ $Revision: 1.1.9.7 $ $Date: 1994/03/10 21:02:27 $";
#endif

OTErr otTclInitInterp();
OTErr otTclInitCmds();
OTErr otTclEvalProc();
OTErr otTclEval();
OTErr otQueryNextFull();
OTErr otQueryOneCR();

int  cmdFieldValue();
int  cmdNoteValue();
int  cmdBeginEnd();
int  cmdForce();
int  cmdQuiet();
int  cmdUserName();
int  cmdDelete();
int  cmdUnlock();
int  cmdProject();
int  cmdValidate();
int  cmdControl();
int  cmdEnter();
int  cmdUpdate();
int  cmdClear();
int  cmdCommit();
int  cmdField();
int  cmdDelta();
int  cmdTclString();
int  cmdNotedata();
int  cmdNotedataBuildClient();
int  cmdNotedataBuildServer();
int  cmdQuery();
int  cmdKwikQuery();
int  cmdTrue();

/*
 * Eventually move these to the private control block.
 */
static char *nsens_val;
static char tclTemp[LONGVALUE];



/*
 *                        o t T c l I n i t I n t e r p
 *
 * Initialize the TCL interpreter.
 */
OTErr
otTclInitInterp()
{
    char *cp, *rcfn, *fileStr;
    int res;
    OTErr err;
    OTPrivateCB *pvp;
    Tcl_Interp *interp;

    pvp = otCB->cb_pcb;
    err = OT_SUCCESS;
    if ( !pvp->pcb_interp )
        interp = pvp->pcb_interp = Tcl_CreateInterp();
    else
        interp = pvp->pcb_interp;

    pvp->pcb_tclBegin = TRUE;
    pvp->pcb_tclEnd = FALSE;

    Tcl_CreateCommand(interp, "true", cmdTrue, (ClientData)"true",
	(Tcl_CmdDeleteProc *)0);

    Tcl_CreateCommand(interp, "project", cmdProject, (ClientData)"project",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "begin", cmdBeginEnd,   (ClientData)"begin",
        (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "end",   cmdBeginEnd,   (ClientData)"end",
        (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "force", cmdForce,      (ClientData)"force",
        (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "quiet", cmdQuiet,      (ClientData)"quiet",
        (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "userName", cmdUserName, (ClientData)"userName",
        (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "validate", cmdValidate, (ClientData)"validate",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "enter", cmdEnter, (ClientData)"enter",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "update", cmdUpdate, (ClientData)"update",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "clear", cmdClear, (ClientData)"clear",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "delete", cmdDelete, (ClientData)"delete",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "unlock", cmdUnlock, (ClientData)"unlock",
	(Tcl_CmdDeleteProc *)0);

    Tcl_CreateCommand(interp, "commit", cmdCommit, (ClientData)"commit",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "field", cmdField, (ClientData)"field",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "control", cmdControl, (ClientData)"control",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "delta", cmdDelta, (ClientData)"delta",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "notedata", cmdNotedata, (ClientData)"notedata",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "notedataBuild",  cmdNotedataBuildClient,
	(ClientData)"notedataBuild",  (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "tclString", cmdTclString,
	(ClientData)"tclString", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "query", cmdQuery, (ClientData) "query",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "kwikQuery", cmdKwikQuery,
	(ClientData) "kwikQuery", (Tcl_CmdDeleteProc *)0);

    /*
     * Read .otrc first so the user cannot redefine special procedures.
     */
    cp = getenv("HOME");
    if ( cp ) {
        rcfn = (char *)malloc(strlen(cp) + strlen("/.otrc") + 1);
        if ( !rcfn ) {
	    otPutPostedMessage( OT_MALLOC_LOCATION, "otTclInitInterp()" );
	    return OT_MALLOC_LOCATION;
	}
	strcpy(rcfn, cp);	
	strcat(rcfn, "/.otrc");

	err = otWriteFileToString(rcfn, &fileStr);
	if ( err ) {
	    if ( err == OT_STAT )
		err = OT_SUCCESS;
	} else {
	    err = otSetCBMember("tclOtrc", fileStr);

	    if ((res = Tcl_Eval(interp, fileStr)) == TCL_ERROR) {
	        otPutPostedMessage( OT_TCL, interp->result );
		err = OT_TCL;
	    }
	    free(fileStr);
	}
	free(rcfn);

    }
    return err;

}


/*
 * otTclInitCmds - initialize TCL commands.
 */
OTErr
otTclInitCmds()
{
    int i;
    OTProject *otProj;
    OTMetaTemplate *mTemplate;
    Tcl_Interp *interp;
    OTErr cmdErr = OT_SUCCESS;
    OTControlBlock *cb;

    cb = otCB;
    interp = cb->cb_pcb->pcb_interp;
    otProj = cb->cb_pcb->pcb_project;
    if ( otProj ) {
	mTemplate = otProj->otmp;

	/* What happens when we have more than one project?	 */
	for(i = 0; mTemplate[i].field[0]; i++) {
	    if (mTemplate[i].abbr[0])
		Tcl_CreateCommand(interp, mTemplate[i].abbr, cmdFieldValue,
		    (ClientData) &(mTemplate[i]), (Tcl_CmdDeleteProc *)0);
	}

	Tcl_CreateCommand(interp, "note",  cmdNoteValue,  (ClientData)"note",
	    (Tcl_CmdDeleteProc *)0);
    }
    else {
        otPutPostedMessage( OT_INTERNAL_ERROR, "otTclInitCmds" );
	cmdErr = OT_INTERNAL_ERROR;
    }
    return cmdErr;

}

/*
 *
 * otTclEvalProc
 *
 * Error returns:
 *    - OT_MALLOC_LOCATION
 *    - OT_TCL_CALLER
 *    - OT_TCL_NOPROC
 *    - OT_SUCCESS
 */
OTErr otTclEvalProc(procName)
char *procName;
{
    char *tclStr;
    int tclRes;
    OTErr tclStat;
    Tcl_Interp *interp;

    interp = otCB->cb_pcb->pcb_interp;
    tclStat = OT_SUCCESS;
    /*
     * Is procName defined by the user in the project definition file?
     */
    if ( !(tclStr = (char *)malloc(NAMELEN)) ) {
	otPutPostedMessage( OT_MALLOC_LOCATION, "otTclEvalProc()" );
	return OT_MALLOC_LOCATION;
    }
    strcpy(tclStr, "info commands ");
    strcat(tclStr, procName);
    tclRes = Tcl_Eval( interp, tclStr );
    if ( tclRes != TCL_OK ) {
	otPutPostedMessage( OT_TCL_CALLER, tclStr, interp->result );
	tclStat = OT_TCL_CALLER;
    }
    else if ( !strcmp(interp->result, procName) ) {
	strcpy(tclStr, procName);
	tclRes = Tcl_Eval( interp, tclStr );
	if ( tclRes != TCL_OK ) {
	    otPutPostedMessage( OT_TCL_CALLER, procName, interp->result );
	    tclStat = OT_TCL_CALLER;
	}
    }
    else 
	tclStat = OT_TCL_NOPROC;
    free(tclStr);
    return tclStat;

}



/*
 * Evaluate the default TCL string in the default interpreter's environment.
 */
OTErr
otTclEval(tclStr)
char *tclStr;
{
    int res;
    OTErr errCode = OT_SUCCESS;
    Tcl_Interp *interp = otCB->cb_pcb->pcb_interp;

    DBUG_MED((stderr, "Tcl_Eval to interpret %s\n", tclStr));

    if ( (res = Tcl_Eval(interp, tclStr)) != TCL_OK ) {
	otPutPostedMessage( OT_TCL, interp->result );
	errCode = OT_TCL;
    }
    DBUG_MED((stderr, "Tcl_Eval --> %d/(%s)\n", res, interp->result));
    return errCode;

}



int
cmdFieldValue(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    char *fieldname, *query_val, *tmpl_val, *cp;
    otType fieldtype;
    OTErr compStatus;
    OTTemplate *ott_template;
    register OTHeaderField *hfp;
    int i, stat;
    bool orig=FALSE;
    OTControlBlock *cb = otCB;
    OTProject *prjp = cb->cb_pcb->pcb_project;
   
    DBUG_MED((stderr, "cmdFieldValue argc = %d, argv[0] = %s\n", argc, argv[0]));

    /*
     * The argument "orig" refers to the original template in an update
     * operation.  Copy the other arguments down and set ott_template to 
     * the origStruct.  We cannot "set" the fields of this template.
     * 
     * We need to do something different for OTQueryCB.
     */ 
    if (cb->cb_operation == QUERY) 
	ott_template = cb->cb_pcb->pcb_template;
    else {
	if ( (argc > 1) && (argv[1][0] == 'o') && !strcmp(argv[1], "orig") ) {
	    ott_template = cb->cb_ecb->ecb_origStruct;
	    for ( i=1 ; i < argc-1; i++ )
		argv[i] = argv[i+1];
	    argv[argc-1] = 0;
	    argc--;
	    orig = TRUE;
	} else
	    ott_template = cb->cb_ecb->ecb_tStruct;
    }

    /*
     * If there is no current buffer, return (unless this is an 'info'
     * command, then it is unnecessary).
     */
    if (!ott_template && !( (argc == 3) && (!strcmp(argv[1], "info")) ) ) {
	interp->result = "";
	return TCL_OK;
    }

    fieldname = ((OTMetaTemplate *)clientData)->field;
    fieldtype = ((OTMetaTemplate *)clientData)->type;
    DBUG_MED((stderr, "fieldname %s - fieldtype %d\n", fieldname, fieldtype));
	
    if (argc == 1) {
	if ( !(tmpl_val = otGetHeaderFieldValue(fieldname, ott_template )) )
	    Tcl_SetResult(interp, "", TCL_VOLATILE);
	else 
	    Tcl_SetResult(interp, tmpl_val, TCL_VOLATILE);
	return TCL_OK;
    }
    if (argc == 2) {
	query_val = argv[1];
	tmpl_val = 0;
	for (hfp = ott_template->tr; hfp && hfp->field && *hfp->field; hfp++)
	    if ( (*fieldname == *hfp->field) && !strcmp(fieldname,
		hfp->field) )
		tmpl_val = hfp->value;

	if ( !*query_val && !tmpl_val)
	    stat = 1; 
	else if ( (!*query_val && tmpl_val) || (*query_val && !tmpl_val) )
	    stat = 0;
	else {
	    compStatus = otCompare(tmpl_val, fieldtype, query_val);
	    if ( compStatus && compStatus != OT_NOMATCH )
	        compStatus = OT_NOMATCH;
	    stat = (compStatus == OT_SUCCESS) ? 1 : 0;
	}

        DBUG_MED((stderr, "Result of Comp? is %d\n", stat));

	if ( stat )
	    interp->result = "1";
	else
	    interp->result = "0";
	
	return TCL_OK;
    }

    if (argc == 3) {

	DBUG_MED((stderr, "cmdFieldValue argv[1] = %s\n", argv[1]));
	/*
	 * Support an assignment operator for project specific fields.
	 */
	if (!strcmp(argv[1], "set")) {
	    if (!otSetHeaderFieldValue(fieldname, argv[2], ott_template))
		return TCL_OK;
	    else {
		Tcl_SetResult(interp, "error in assignment", TCL_VOLATILE);
		return TCL_ERROR;
	    }
	} else if (!strcmp(argv[1], "info"))  { 

	    cp = argv[2];
	    if ( *cp == 'l' && !strcmp(cp, "label")) {
		Tcl_SetResult(interp, ((OTMetaTemplate *)clientData)->field,
		    TCL_VOLATILE);
		return TCL_OK;
	    } else if ( *cp == 't' && !strcmp(cp, "type")) {
		/*
		 * Just return what is in "list" member (which could be
		 * enumerated typeNAME, not elements.  These would be 
		 * returned in "list" subcommand.
		 */
		char *tname;
		switch ( ((OTMetaTemplate *)clientData)->type ) {
		    case TYPE_LIST:
			tname = "list";
			break;
		    case TYPE_TEXT:
			tname = "text";
			break;
		    case TYPE_MAILNAME:
			tname = "mailname";
			break;
		    case TYPE_DATE:
			tname = "date";
			break;
		    case TYPE_IDNUM:
			tname = "idnum";
			break;
		    case TYPE_NUM:
			tname = "num";
			break;
		    case TYPE_ENUM:
			tname = ((OTMetaTemplate *)clientData)->list;
			break;
		    case TYPE_NULL:
			tname = "null";
			break;
		    default:
			tname = "";
			break;
		}
		Tcl_SetResult(interp, tname, TCL_VOLATILE);
		return TCL_OK;
	    } else if ( *cp == 'o' && !strcmp(cp, "optionality")) {
		char *oname;

		switch ( ((OTMetaTemplate *)clientData)->optionality ) {
		    case OPT_NONE:
			oname = "";
			break;
		    case OPT_MAND:
			oname = "mandatory";
			break;
		    case OPT_OPTL:
			oname = "optional";
			break;
		    default:
			oname = "";
			break;
		}
		Tcl_SetResult(interp, oname, TCL_VOLATILE);
		return TCL_OK;
	    } else if ( *cp == 'd' && !strcmp(cp, "degree")) {
		char *dname;

		switch ( ((OTMetaTemplate *)clientData)->degree ) {
		    case DEGREE_NONE:
			dname = "";
			break;
		    case DEGREE_ONE:
			dname = "one";
			break;
		    case DEGREE_MANY:
			dname = "many";
			break;
		    default:
			dname = "";
			break;
		}
		Tcl_SetResult(interp, dname, TCL_VOLATILE);
		return TCL_OK;
	    } else if ( *cp == 'd' && !strcmp(cp, "defs")) {
		Tcl_SetResult(interp, ((OTMetaTemplate *)clientData)->defs,
		    TCL_VOLATILE);
		return TCL_OK;
	    } else if ( *cp == 'h' && !strcmp(cp, "head")) {
		Tcl_SetResult(interp, ((OTMetaTemplate *)clientData)->head,
		    TCL_VOLATILE);
		return TCL_OK;
	    } else if ( *cp == 'w' && !strcmp(cp, "width")) {
		Tcl_SetResult(interp, ((OTMetaTemplate *)clientData)->width,
		    TCL_VOLATILE);
		return TCL_OK;
	    } else if ( *cp == 'l' && !strcmp(cp, "list")) {
		int stat;
		char *sargv[3];
		char *elist;
		ClientData dummy;

		sargv[0] = "split";
		sargv[2] = ",";
		if ( (elist = otGetEnumtypeList( ((OTMetaTemplate *)clientData)->list, prjp)) ) {
		    DBUG_MED((stderr, "list = %s\n", elist));

		    sargv[1] = elist;
		    sargv[3] = 0;
		    stat = Tcl_SplitCmd(dummy, interp, 3, (char **)sargv);
		    return stat;
		} else {
		    DBUG_MED((stderr, "Enum list = %s\n", ((OTMetaTemplate *)clientData)->list));
		    sargv[1] = ((OTMetaTemplate *)clientData)->list;
		    sargv[3] = 0;
		    stat = Tcl_SplitCmd(dummy, interp, 3, (char **)sargv);
		    return stat;
		}
	    } else if ( *cp == 'l' && !strcmp(cp, "listFile")) {
		/*
		 * What file contains the list of elements?  look in
		 * otCB->cb_project->enum_types[n].filename
		 */
		int i;
		char *fname = "";
		char *list = (char *)((OTMetaTemplate *)clientData)->list;
	    
		for (i=0; !*fname && prjp->enum_types[i].typename != 0 ; i++)
		    if ( !strcmp(list, prjp->enum_types[i].typename) )
			fname = prjp->enum_types[i].filename;
		Tcl_SetResult(interp, fname, TCL_VOLATILE);
		return TCL_OK;
	    }
	}
    }

    return TCL_OK;
}


int
cmdNoteValue(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    int n, count, len, i;
    char *np;
    OTTemplate *ott_template;
    OTControlBlock *cb;
    bool orig;
    OTErr appendErr;

    cb = otCB;
    if ( (cb->cb_operation == QUERY) && cb->cb_pcb->pcb_tclBegin )
	cb->cb_qcb->qcb_readNote = TRUE;

    /*
     * The argument "orig" refers to the original template in an update
     * operation.  Copy the other arguments down and set ott_template to 
     * the origStruct.  We cannot "set" the fields of this template.
     * 
     * We need to do something different for OTQueryCB.
     */ 
    if (cb->cb_operation == QUERY) 
	ott_template = cb->cb_pcb->pcb_template;
    else {
	if ( (argc > 1) && (!strcmp(argv[1], "orig")) ) {
	    ott_template = cb->cb_ecb->ecb_origStruct;
	    for ( i=1 ; i < argc-1; i++ )
		argv[i] = argv[i+1];
	    argv[argc-1] = 0;
	    argc--;
	    orig = TRUE;
	} 
	else
	    ott_template = cb->cb_ecb->ecb_tStruct;
    }

    if ( !ott_template || !(ott_template->notep) ) {
	Tcl_SetResult(interp, "", TCL_VOLATILE);
	return TCL_OK;
    }

    if ( argc == 1 ) {
        len = 0;

        for (n = 0; (ott_template->notep[n].who[0]) && (n < MAXNOTES); n++)
	    len += strlen(ott_template->notep[n].text) + 2;

	if ( len == 0 ) {
	    Tcl_SetResult(interp, "", TCL_VOLATILE);
	    return TCL_OK;
	}

	np = (char *)calloc(len + 1, 1);

        for (n = 0; (ott_template->notep[n].who[0]) && (n < MAXNOTES); n++)
	    if (ott_template->notep[n].text) {

		if(nsens_val && strcmp(nsens_val, ott_template->notep[n].sens))
		    continue;

		strcat(np, ott_template->notep[n].text);
		strcat(np, "\n\n");

	    }
	Tcl_SetResult(interp, np, TCL_DYNAMIC);
	return TCL_OK;
    }
    else if (argc == 2 && !strcmp("count", argv[1])) {
	count = 0;
	for (n = 0; (ott_template->notep[n].who[0]) && (n < MAXNOTES); n++)
	    count++;	    
	if ( !(np = malloc(NAMELEN)) ) {
	    otPutPostedMessage( OT_MALLOC_LOCATION, "cmdNoteValue" );
	    return OT_MALLOC_LOCATION ;
	}
	sprintf(np, "%d", count);
	Tcl_SetResult(interp, np, TCL_DYNAMIC);
	return TCL_OK;
    }
    else if (argc == 2 && argv[1][0] && 
	    (strspn(argv[1], "0123456789") == strlen(argv[1]))) {
	sscanf(argv[1], "%d", &count);
	DBUG_MED((stderr, "note %d\n", count));

	for (n = 0; (ott_template->notep[n].who[0]) && (n < count) && (n < MAXNOTES); n++);
	if (!ott_template->notep[n].text) {
	    Tcl_SetResult(interp, "no note found", TCL_VOLATILE);
	    return TCL_OK;
	}
	else {
	    Tcl_SetResult(interp, ott_template->notep[n].text, TCL_VOLATILE);
	    return TCL_OK;
	}	    
    }
    else if ( argc == 3 && (!strcmp(argv[1], "nsens")) ) {
	nsens_val = strdup(argv[2]);
	Tcl_SetResult(interp, "", TCL_VOLATILE);
	return TCL_OK;
    }
    else if ( argc == 3 && (!strcmp(argv[1], "clear")) && (!strcmp(argv[2], "all")) ) {
	register OTNote *tmp_np = ott_template->notep;

	if (tmp_np) {
	    for (; *tmp_np->who; tmp_np++) {
		if (tmp_np->text)
		    free( tmp_np->text );
		memset(tmp_np, 0, sizeof(OTNote));
	    }
	}	
	Tcl_SetResult(interp, "", TCL_VOLATILE);
	return TCL_OK;
	/*
	 * Later, clean up and add "note clear n", where n is the number of the
	 * note to delete.
	 */
    }
    else if ( argc > 2 && (!strcmp(argv[1], "append")) ) {

	len = 0;
        for ( i = 2; i < argc; i++) {
	    len += strlen(argv[i]) + 2;
	    DBUG_MED((stderr, "i = %d  len = %d\n", i, len));
	}

	if ( len == 0 ) {
	    Tcl_SetResult(interp, "", TCL_VOLATILE);
	    return TCL_OK;
	}

	np = (char *)calloc(len + 1, 1);
	for ( i = 2; i < argc; i++) {
	    DBUG_MED((stderr, "note append %s\n", argv[i]));
	    strcat(np, argv[i]);
	}

#ifdef notdef
	if (ott_template->notep)
	    freeNotes(ott_template->notep);
	if ( !( ott_template->notep = 
	    (OTNote *)calloc(MAXNOTES, sizeof(OTNote))) ) {

	    otPutPostedMessage( OT_MALLOC_LOCATION, "otMallocTemplate" );
	    Tcl_SetResult( interp, otGetPostedMessage(), TCL_VOLATILE );
	    return TCL_ERROR;
	} 
#endif

	appendErr = otAppendNoteToTemplateFromString( ott_template, np, TRUE, TRUE ); 
	free(np);
	if ( appendErr ) {
	    DBUG_MED((stderr, "note set failed\n"));
	    Tcl_SetResult( interp, otGetPostedMessage(), TCL_VOLATILE );
	    return TCL_ERROR;
	} else {
	    DBUG_MED((stderr, "note set succeeded\n"));
            Tcl_SetResult(interp, "", TCL_VOLATILE);
	    return TCL_OK;
        }
    }
    else if ( argc == 2 ) {
	char *startp, *nextp, *tp, *query_val;
	int oneMatch = 0;

	query_val = argv[1];

	for (n = 0; (ott_template->notep[n].who[0]) && (n < MAXNOTES); n++) {

	    if (nsens_val && strcmp(nsens_val, ott_template->notep[n].sens))
		continue;

	    tp = strdup(ott_template->notep[n].text);
	    startp = nextp = tp;

	    while ( nextp && *nextp && !oneMatch ) {
		startp = nextp;
		/*
		 * Break into separate lines for strstr().
		 * Actually not necessary but we may need to use regexp()
		 * again.
		 */
		nextp = strchr(startp, '\n');
		if (nextp) {
		    *nextp = '\000';
		    nextp++;
		}
    
		if ( strstr(startp, query_val) ) 
		    oneMatch = 1;
	    }
	    free(tp);
	}
	DBUG_MED((stderr, "Result of note search is %d\n", oneMatch));
	if (oneMatch == 1)
	    Tcl_SetResult(interp, "1", TCL_VOLATILE);
	else
	    Tcl_SetResult(interp, "0", TCL_VOLATILE);
        return TCL_OK;
    } else {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " [append|nsens|count|<index>|clear|<qry_val>]\"", (char *)NULL);
	return TCL_ERROR;
    }
}


int
cmdTrue(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{

    interp->result = "1";
    return TCL_OK;

}

/*
 *                             c m d B e g i n E n d
 *
 *
 */
int
cmdBeginEnd(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    char *status;
    bool tclBegin, tclEnd;
    OTControlBlock *cb;

    cb = otCB;
    tclBegin = cb->cb_pcb->pcb_tclBegin;
    tclEnd = cb->cb_pcb->pcb_tclEnd;
    if (!strcmp(argv[0], "begin"))
        status = (tclBegin == TRUE) ? "1" : "0";
    else if (!strcmp(argv[0], "end"))
        status = (tclEnd == TRUE) ? "1" : "0";

    Tcl_SetResult(interp, status, TCL_VOLATILE);
    return TCL_OK;
}

/*
 *                                 c m d F o r c e
 *
 * Defines command to force an update, even if the user has not altered
 * the data.  This is for template updates based on a changed metatemplate.
 */
int
cmdForce(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    char *result;
    int status;
    OTPrivateCB *pcb;
    OTControlBlock *cb;

    cb = otCB;
    pcb = cb->cb_pcb;

    if (argc >= 2 && (!strcmp("update", argv[1]))) {
        /*
	 * "force update" print current setting.
	 */
	if (argc == 2) {
	    result = (pcb->pcb_forceUpdate == TRUE) ? "1" : "0";
	    status = TCL_OK;
	}
	else {    
	    if (!strcmp("1", argv[2])) {
		pcb->pcb_forceUpdate = TRUE;
		result = "";
		status = TCL_OK;
	    }
	    else if (!strcmp("0", argv[2])) {
	        pcb->pcb_forceUpdate = FALSE;
	        result = "";
		status = TCL_OK;
	    }
	    else {
		result = "force update keyword %s must be 1 or 0";
	        status = TCL_ERROR;
	    }
	}
    }
    else
	result = "error: command should be force update [1|0]";

    Tcl_SetResult(interp, result, TCL_VOLATILE);
    return status;

}


/*
 *                                 c m d Q u i e t
 *
 * Defines command to suppress default message from ot_bugs.
 * For customized reports.
 */
int
cmdQuiet(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    char *result;
    int status;
    OTPrivateCB *pcb;
    OTControlBlock *cb;

    cb = otCB;
    pcb = cb->cb_pcb;
    /*
     * "quiet" - print current setting.
     */
    if (argc == 1) {
	result = (pcb->pcb_quiet == TRUE) ? "1" : "0";
	status = TCL_OK;
    }
    else {    
	if (!strcmp("1", argv[1])) {
	    pcb->pcb_quiet = TRUE;
	    result = "";
	    status = TCL_OK;
	}
	else if (!strcmp("0", argv[1])) {
	    pcb->pcb_quiet = FALSE;
	    result = "";
	    status = TCL_OK;
	}
	else {
	    result = "quiet keyword %s must be 1 or 0";
	    status = TCL_ERROR;
	}
    }

    Tcl_SetResult(interp, result, TCL_VOLATILE);
    return status;

}


/*
 *                           c m d U s e r N a m e
 *
 * Return the current username.
 */
int
cmdUserName(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{

    if ( (argc == 3) && (!strcmp(argv[1], "set")) ) {
	if ( otCopyString(argv[2], &(otCB->cb_pcb->pcb_uName) ) ) {
	    Tcl_SetResult(interp, "Error in allocation", TCL_STATIC);
	    return TCL_ERROR;
	}
    } else {
	Tcl_SetResult( interp, otCB->cb_pcb->pcb_uName, TCL_VOLATILE );
    }
    return TCL_OK;

}
/*
 *                           c m d D e l e t e 
 *
 * Delete a CR.
 */
int
cmdDelete(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    OTErr errCode = OT_SUCCESS;

    if ( argc != 2 ) {
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
	    " recordNum", (char *)NULL );
	return TCL_ERROR;
    }

    otCB->cb_operation = otCB->cb_pcb->pcb_operation = DELETE;
    otCB->cb_CRNumber = otCB->cb_pcb->pcb_CRNumber = atol(argv[1]);

    errCode = otDeleteObject();
    if ( errCode ) {
	Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	return TCL_ERROR;
    }

    Tcl_SetResult( interp, "", TCL_VOLATILE );
    return TCL_OK;

}


/*
 *                           c m d U n l o c k
 *
 * Unlock a CR.
 */
int
cmdUnlock(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    OTProject * proj = otCB->cb_pcb->pcb_project; 	
    OTErr errCode;
    OTFilen * fname;

    if ( argc > 2 ) {
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
	    (char *)NULL );
	return TCL_ERROR;
    }

    if ( argc == 2 ) {
	otCB->cb_CRNumber = otCB->cb_pcb->pcb_CRNumber = atol(argv[1]);
	errCode = otFormName(proj, otCB->cb_CRNumber, &fname);
	if (errCode == OT_SUCCESS) {
	    strcpy(gotCRdir, fname->dir);
	    strcpy(gotCRfile, fname->name);
	}
	free(fname);
        if (errCode) {
	    Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	    return TCL_ERROR;
	}
    }

    if ( gotCRfile[0] ) {
	if ( !otUnlockObject() ) {
	    otPutPostedMessage(OT_RCS_UNLOCK, gotCRfile);
	    Tcl_AppendResult(interp, otGetPostedMessage(), (char *)NULL);
	    return TCL_ERROR;
	}
	else {
	    gotCRfile[0] = 0;
	    *interp->result = 0;
	}
    } else
	*interp->result = 0;

    return TCL_OK;

}


/*
 *                           c m d P r o j e c t
 *
 * Manipulate project information.
 * Commands:
 *      project list		return list of commands
 *	project set name	return list
 */
int
cmdProject(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    char *cp;
    OTErr prjErr;

    if ( argc == 1 ) {
	/*
	 * This will return the current project.  Stub for now.
	 */
        if (otCB->cb_project)
	    Tcl_SetResult( interp, otCB->cb_project, TCL_VOLATILE );
	else 
	    Tcl_SetResult( interp, "", TCL_VOLATILE );
	return TCL_OK;
    }

    if (argc == 2) {
        if ( !otCB->cb_pcb ) {
	    Tcl_SetResult( interp, "", TCL_VOLATILE );
	} else if ( argv[1][0] == 'l' && !strcmp(argv[1], "list") ) {
	    char *pp;

	    /*
	     * Bug in otGetProjectInfo() when there is a non-existing
	     * directory in OT_DBPATH.
	     */
	    if ( otGetProjectInfo(&pp) ) {
		Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
		return TCL_ERROR;
	    } else {
		Tcl_SetResult( interp, pp, TCL_DYNAMIC );
	        return TCL_OK;
	    }
	} else if (argv[1][0] == 'f' && !(strcmp(argv[1], "fields"))) {
	    OTMetaTemplate *mtp;
	
	    mtp = otCB->cb_pcb->pcb_project->otmp;

	    for( ; *mtp->field; mtp++)
		if ( *mtp->field != '!')
		    Tcl_AppendElement( interp, mtp->abbr );
	    return TCL_OK;
	} else if (argv[1][0] == 'p' && !(strcmp(argv[1], "projectLeader"))) {
	    Tcl_SetResult( interp, otCB->cb_pcb->pcb_project->proj_leader,
		TCL_VOLATILE );
	    return TCL_OK;
	} else if (argv[1][0] == 'o' && !(strcmp(argv[1], "objectName")))    {
	    Tcl_SetResult( interp, otCB->cb_pcb->pcb_project->object_name,
		TCL_VOLATILE );
	    return TCL_OK;
	} else if (argv[1][0] == 'p' && !(strcmp(argv[1], "projectDirectory"))) {
	    Tcl_SetResult( interp, otCB->cb_pcb->pcb_project->proj_dir, 
		TCL_VOLATILE );
	    return TCL_OK;
	} else if (argv[1][0] == 's' && !(strcmp(argv[1], "serverHost"))) {
	    Tcl_SetResult( interp, otCB->cb_pcb->pcb_project->server_host,
		TCL_VOLATILE );
	    return TCL_OK;
	} else if (argv[1][0] == 's' && !(strcmp(argv[1], "serverPort"))) {
	    char port[LONGVALUE];

	    sprintf(port, "%d", otCB->cb_pcb->pcb_project->server_port);
	    Tcl_SetResult(interp, port, TCL_VOLATILE);
	    return TCL_OK;

	} else if (argv[1][0] == 'n' && !(strcmp(argv[1], "notify"))) {
	    Tcl_SetResult( interp, otCB->cb_pcb->pcb_project->notify?"1":"0",
		TCL_VOLATILE );
	    return TCL_OK;
	} else if (argv[1][0] == 'c' && !(strcmp(argv[1], "compliance"))) {
	    Tcl_SetResult( interp, otCB->cb_pcb->pcb_project->compliance,
		TCL_VOLATILE );
	    return TCL_OK;
	} else if (argv[1][0] == 'b' && !(strcmp(argv[1], "blockSilence"))) {
	    Tcl_SetResult( interp, 
		otCB->cb_pcb->pcb_project->block_silence ? "1" : "0",
		TCL_VOLATILE );
	    return TCL_OK;
	} else {
	    Tcl_SetResult( interp, "", TCL_VOLATILE );
	    return TCL_OK;
	}
    }

    if ( argc == 3 ) {
	if ( !strcmp(argv[1], "set" ) ) {
	    prjErr = otReadProject(argv[2]);
	    if ( prjErr ) {
		cp = otGetPostedMessage();
		(void)Tcl_SetResult(interp, cp, TCL_VOLATILE);
		return TCL_OK;
	    } else {
		if (otCB->cb_project) {
		    (void)free(otCB->cb_project);
		    otCB->cb_project = strdup(argv[2]);
		}
		(void)Tcl_SetResult( interp, "", TCL_VOLATILE );
		return TCL_OK;
	    }
	} else {
	    Tcl_SetResult( interp, "", TCL_VOLATILE );
	    return TCL_OK;
	}
    }

    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	" [set]\"", (char *)NULL);
    return TCL_ERROR;

}


/*
 *                           c m d V a l i d a t e
 *
 * Validate current template.
 */
int
cmdValidate(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    OTErr errCode;
    bool valid, sysErr;

    errCode = otValidateLhs();
    if ( errCode )
	valid = FALSE;
    if ( OT_WARN(errCode) || OT_SYSTEM(errCode) )
	sysErr = TRUE;

    DBUG_MED((stderr, "validate lhs = %d\n", valid));
    if ( !sysErr ) {
	errCode = otValidateData();
	if (errCode)
	    valid = FALSE;
	if ( OT_WARN(errCode) || OT_SYSTEM(errCode) ) {
	    sysErr = TRUE;
	}
    }

    DBUG_MED((stderr, "validate data = %d\n", valid));
    if ( !sysErr ) {
	errCode = otValidateDepend();
	if (errCode)
	    valid = FALSE;
	if ( OT_WARN(errCode) || OT_SYSTEM(errCode) ) {
	    sysErr = TRUE;
	}
    }
	
    if ( valid == FALSE ) {
	Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	return TCL_OK;
    } else {
	Tcl_SetResult( interp, "", TCL_VOLATILE );
	return TCL_OK;
    }

}


/*
 *                           c m d C o n t r o l
 *
 * Get control block information.
 */
int
cmdControl(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    OTErr err = OT_SUCCESS;
    int cntrlCount, attrCount, i;
    char *lp, *valp;
    char **cntrlArgv, **attrArgv;

    cntrlArgv = attrArgv = 0;
    Tcl_ResetResult(interp);

    /*
     *
     */
    if ( argc != 2 ) {
	Tcl_SetResult(interp,
	    "usage: control { {attribute value} {attribute value} ... }",
	    TCL_VOLATILE);
	return TCL_ERROR;
    }

    if ( Tcl_SplitList(interp, argv[1], &cntrlCount, &cntrlArgv) ) {
	Tcl_SetResult(interp, "Error in control argument", TCL_VOLATILE);
	return TCL_ERROR;
    }

    if ( !strcmp(cntrlArgv[0], "{}") ) {
	cntrlArgv[0] = 0;
	cntrlCount = 0;
    }

    for ( i = 0; !err && i < cntrlCount; i++ ) {

	if ( Tcl_SplitList(interp, cntrlArgv[i], &attrCount, &attrArgv) ) {
	    Tcl_SetResult(interp, "Error in control argument", TCL_VOLATILE);
	    return TCL_ERROR;
	}

	lp = attrArgv[0];

	if ( !strcmp(lp, "skipHeader")  || !strcmp(lp, "count") 	||
	     !strcmp(lp, "fullText")    || !strcmp(lp, "historyLines")  ||
	     !strcmp(lp, "tabSeparate") || !strcmp(lp, "summary") ) {

	    valp = attrArgv[1];
	    
	    if ( !strcmp(valp, "TRUE") || !strcmp(valp, "1") ) {
		otSetQCBMember( lp, "TRUE" );
	    } else if ( !strcmp(valp, "FALSE") || !strcmp(valp, "0") ) {
		otSetQCBMember( lp, "FALSE" );
	    } else {
		Tcl_AppendResult(interp, lp, "takes TRUE or FALSE\n",
		    (char *)NULL);
	    }

	} else if ( !strcmp(lp, "quiet") ) {
	    valp = attrArgv[1];

	    if ( !strcmp(valp, "TRUE") || !strcmp(valp, "1") ) {
		otCB->cb_pcb->pcb_quiet = 1;
	    } else if ( !strcmp(valp, "FALSE") || !strcmp(valp, "0") ) {
		otCB->cb_pcb->pcb_quiet = 0;
	    } else {
		Tcl_AppendResult(interp, lp, "takes TRUE or FALSE\n",
		    (char *)NULL);
	    }

	} else if ( !strcmp(lp, "keyInText") || !strcmp(lp, "outwidth") ||
		    !strcmp(lp, "nsens")    ) {
	    valp = attrArgv[1];

	    if ( !strcmp(valp, "{}") || !strcmp(valp, "\"\"") || !valp )
		valp = "";

	    err = otSetQCBMember( lp, valp );
	    if ( err )
		Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);

	} else if ( !strcmp(lp, "layout")    || !strcmp(lp, "operation")
		 || !strcmp(lp, "tclString") || !strcmp(lp, "tclOtrc")
		 || !strcmp(lp, "tclFile") ) {
	    valp = attrArgv[1];

	    if ( !strcmp(valp, "{}") || !strcmp(valp, "\"\"") || !valp )
		valp = "";

	    err = otSetCBMember( lp, valp );
	    if ( err )
		Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	}

	if (attrArgv)
	    free(attrArgv);
    }

    if (cntrlArgv)
	free(cntrlArgv);

    if ( err )
	return TCL_ERROR;

    Tcl_AppendResult( interp, "", (char *)NULL );
    return TCL_OK;

}

/*
 *                           c m d E n t e r
 *
 * Initiate an enter operation.
 */
int
cmdEnter(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    char tmpfile[PATHLEN];
    OTTemplate * tStruct;
    OTErr errCode = OT_SUCCESS;

/* Initialize OTEnterCB and blank template structure. Process
   cb_append and cb_templFile. */

    otCB->cb_operation = otCB->cb_pcb->pcb_operation = ENTER;
    /*
     * Check for value for cb_project, cb_pcb->pcb_project.
     */

    otEFreeMemory();

    if ( !(errCode = otInitEnter(tmpfile)) ) {
	/*
	 * We must make the values of the new template available to the
	 * TCL primitives so they may be displayed or manipulated.  If there
	 * are any value alterations through tclString or template 
	 * specification, read it in again.  Otherwise dup the original.
	 */
        if ( otCB->cb_append || otCB->cb_templFile ) {
	    errCode = otReadTemplateFromFilename(&tStruct, tmpfile, TRUE, FALSE);
	    if (errCode)
		otFreeTemplate(tStruct);
	    else
		otCB->cb_ecb->ecb_tStruct = tStruct;
	}
	else {
	    errCode = otDupTemplate( otCB->cb_ecb->ecb_origStruct,
		&(otCB->cb_ecb->ecb_tStruct) );
	}
    }

    if (errCode) {
	Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	return TCL_ERROR;
    }
    else {
        strcpy(tclTemp, tmpfile);
	Tcl_SetResult( interp, "", TCL_VOLATILE );
	return TCL_OK;
    }

}


/*
 *                           c m d U p d a t e
 *
 * Update a CR.
 */
int
cmdUpdate(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    char tmpfile[PATHLEN];

    OTErr errCode = OT_SUCCESS;

    if ( argc != 2 ) {
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
	    " recordNum", (char *)NULL );
	return TCL_ERROR;

    } else {

	otEFreeMemory();

        /*
	 * Additionally, what value needs to be placed in the control block
	 * for CR number?
	 */
	otCB->cb_operation = otCB->cb_pcb->pcb_operation = UPDATE;
	otCB->cb_CRNumber = atol(argv[1]);

	/*
	 * Initialize enter control block, check input for conflicts, 
	 * get an Object for an update
	 */
	errCode = otPrepareForUpdate(tmpfile);
	if (errCode) {
	    Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	    return TCL_ERROR;
	} else {
	    strcpy(tclTemp, tmpfile);
	    errCode = otCollateTemplate(&(otCB->cb_ecb->ecb_origStruct));
	    if ( errCode && (errCode != OT_TEMPLATE_UNCHANGED) ) {
		Tcl_SetResult( interp, otGetPostedMessage(), TCL_VOLATILE );
		return TCL_ERROR;
	    }
	    errCode = otDupTemplate( otCB->cb_ecb->ecb_origStruct,
		&(otCB->cb_ecb->ecb_tStruct) );
	    if ( errCode ) {
		Tcl_SetResult( interp, otGetPostedMessage(), TCL_VOLATILE );
		return TCL_ERROR;
	    } else {
		Tcl_SetResult( interp, "", TCL_VOLATILE );
		return TCL_OK;
	    }
	}
    }

}

/*
 *                                c m d C l e a r
 *
 *
 */
int
cmdClear(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    OTErr err;
    OTTemplate *ott_template;
    OTMetaTemplate *mtp;
    OTHeaderField *hfp;
    OTNote *np;
    OTProject *pjp;
    bool orig = FALSE;
    int i, j;

    pjp = otCB->cb_pcb->pcb_project;

    if ( (argc > 1) && !strcmp(argv[1], "orig") ) {
	ott_template = otCB->cb_ecb->ecb_origStruct;
	for ( i = 1 ; i < argc-1; i++ )
	    argv[i] = argv[i+1];
	argv[argc-1] = 0;
	argc--;
	orig = TRUE;
    } else
	ott_template = otCB->cb_ecb->ecb_tStruct;

    if ( argc == 2 ) {
	if ( !strcmp(argv[1], "header") ) {
	    if (ott_template && ott_template->tr) {

		hfp = ott_template->tr;

		for(i=0; i< MAXTEMPLATE; i++) {
		    if (hfp[i].value) {
			free( hfp[i].value );
			hfp[i].value = 0;
		    }
		    if (hfp[i].field)
			hfp[i].field[0] = 0;
		    if (hfp[i].list)
			hfp[i].list[0] = 0;
		    hfp[i].type = 0;
		    hfp[i].optionality = 0;
		    hfp[i].degree = 0;
		}

		err = otDupMetaTemplate( pjp->otmp, &mtp);
		if ( err )
		    return err;

		for (i=0; mtp[i].field[0]; i++) {
		    if ( mtp[i].field[0] == '!') {
			fixStr(mtp[i].line, pjp->name);
			continue;
		    }
		    fixStr(mtp[i].field, pjp->name);
		    fixStr(mtp[i].opts,  pjp->name);
		    fixStr(mtp[i].abbr,  pjp->name);
		    fixStr(mtp[i].head,  pjp->name);
		}

		/*
		 * Add field, type, etc, default value from metatemplate.
		 */
		for (i=0, j=0;  mtp[i].field[0];  i++) {
		    if (mtp[i].field[0] == '!') {
			if (mtp[i].line[0] == '[')
			    break;
			else
			    continue;
		    }
		    strcpy(hfp[j].field, mtp[i].field);
		    if ( mtp[i].list )
			strcpy(hfp[j].list, mtp[i].list);
		    hfp[j].type = mtp[i].type;
		    hfp[j].optionality = mtp[i].optionality;
		    hfp[j].degree = mtp[i].degree;

		    j++;
		}
		otFreeMetaTemplate(mtp);
	    }
	} else if ( !strcmp(argv[1], "notes") ) {
	    if (ott_template && ott_template->notep) {
		np = ott_template->notep;
		for(i=0; i<MAXNOTES; i++) {
		    if (np[i].text)
			free(np[i].text);
		}
		memset(np, 0, sizeof(OTNote) * MAXNOTES);
	    }
	} else {
	    Tcl_SetResult(interp, "usage: clear [orig] [header] [notes]", TCL_STATIC);
	    return TCL_ERROR;
	}
    }
    return TCL_OK;

}

/*
 *                           c m d C o m m i t
 *
 * Commit a transaction.
 */
int
cmdCommit(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    bool justClosed, valid;
    char *origstat, *currstat;
    long id = 0;
    OTErr errCode, sysErr, postErr;
    OTTemplate *tStruct = otCB->cb_ecb->ecb_tStruct;
    OTTemplate *origStruct = otCB->cb_ecb->ecb_origStruct;
    OTOperation  operation = otCB->cb_operation;
    OTProject * pStruct = otCB->cb_pcb->pcb_project;
    long        CRNumber = otCB->cb_pcb->pcb_CRNumber;
    char *tmpfile;
    char qmpath[PATHLEN];
    char command[COMMAND];

    tmpfile = tclTemp;
    if (!*tmpfile) {
	tmpfile[0] = 0;
	tmpnam(tmpfile);
	strcpy(rmTfile, tmpfile);
    }

    errCode = OT_SUCCESS;
    postErr = OT_SUCCESS;
    sysErr = FALSE;

    if ( (operation == ENTER) || (operation == UPDATE) ) {
	DBUG_MED((stderr, "cmdCommit ENTER, UPDATE\n"));
	valid = TRUE;
	justClosed = FALSE;
	/*
	 * No call to otReadTemplateFromFile() here: everything is
	 * in otCB->cb_ecb->ecb_tStruct.
	 * 
	 * This call to otCollateTemplate() used to be conditionalized
	 * with
	 *
	 * if (otCB->cb_templFile || otCB->cb_tclString) 
	 * 
	 * but now that has been removed, since files are no longer used
	 * as CR representations and therefore no type/deg/opt info is
	 * presented.
	 */
	
	errCode = otCollateTemplate( &(otCB->cb_ecb->ecb_tStruct) );
	if ( OT_WARN( errCode ) || OT_SYSTEM( errCode ) )
	    valid = FALSE;
	else
	    errCode = OT_SUCCESS;

	if (valid && otCB->cb_tclString) {
	    errCode = otTclEval(otCB->cb_tclString);
	    if (errCode)
		valid = FALSE;
	}

	DBUG_MED((stderr, "cmdCommit ENTER, valid = %d\n", valid));
	if (valid) {
	    errCode = otTemplateChanged();

	    if ( OT_WARN(errCode) || OT_SYSTEM(errCode) )
	        valid = FALSE;

	    if ((errCode != OT_TEMPLATE_UNCHANGED) || otCB->cb_pcb->pcb_forceUpdate )
		valid = TRUE;
	    else
		valid = FALSE;
	}
	
	DBUG_MED((stderr, "cmdCommit ENTER template changed, valid = %d\n", valid));
	if ( valid == FALSE ) {
	    Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	    return TCL_ERROR;
	}

	if ( (operation == ENTER) && (otGetIDNumValue(tStruct) != 1000000) ) {
	    valid = FALSE;
	    otPutPostedMessage(OT_NUMBER_PLACE_HOLDER, pStruct->object_name);
	}

	DBUG_MED((stderr, "cmdCommit ENTER, idnum check = %d\n", valid));

	if (operation == UPDATE) {
	    id = otGetIDNumValue(tStruct);
	    if ( !id || (CRNumber != id) ) {
		if (id == 1000000)
		    id = 0;
		valid = FALSE;
		otPutPostedMessage(OT_DIFFERENT_NUMBER, pStruct->object_name, 
		    		id, pStruct->object_name, CRNumber);
	    }
	}

	DBUG_MIN((stderr, "cmdCommit: validation\n"));

	/*
	 * Note that simple validation errors will continue to accumulate
	 * so all errors are detected on one pass: only warnings and system
	 * errors cause a break.
	 */

	errCode = otValidateLhs();
	if ( errCode )
	    valid = FALSE;
	if ( OT_WARN(errCode) || OT_SYSTEM(errCode) ) {
	    sysErr = TRUE;
	}

	DBUG_MED((stderr, "cmdCommit ENTER, validate lhs = %d\n", valid));

	if ( !sysErr ) {
	    errCode = otValidateData();
	    if (errCode)
		valid = FALSE;
	    if ( OT_WARN(errCode) || OT_SYSTEM(errCode) ) {
		sysErr = TRUE;
	    }
	}

	DBUG_MED((stderr, "cmdCommit ENTER, validate data = %d\n", valid));

	if ( !sysErr ) {
	    errCode = otValidateDepend();
	    if (errCode)
		valid = FALSE;
	    if ( OT_WARN(errCode) || OT_SYSTEM(errCode) ) {
	        sysErr = TRUE;
	    }
	}
	
	DBUG_MED((stderr, "cmdCommit ENTER, validate depend = %d\n", valid));

	if ( !sysErr ) {
	    postErr = otPostProcess();
	    if ( postErr && (postErr != OT_INFORMATION) )
		valid = FALSE;
	    if ( OT_WARN(postErr) || OT_SYSTEM(postErr) )
		sysErr = TRUE;
	}

	DBUG_MED((stderr, "cmdCommit ENTER, post process = %d\n", valid));

	if ( valid == FALSE ) {
	    Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	    return TCL_ERROR;
	} 
	currstat = otGetHeaderFieldValue( "Status", tStruct );
	if (currstat) {
	    if ( (operation == ENTER) && ( !strcmp(currstat, "closed") ) )
		justClosed = TRUE;
	    if ( operation == UPDATE ) {
		origstat = otGetHeaderFieldValue( "Status", origStruct );
		if ( origstat ) {
		    if (strcmp(origstat, "closed") &&
			!strcmp(currstat, "closed"))
			justClosed = TRUE;
		}
	    }
	}

	if (justClosed && valid)
	    otCB->cb_pcb->pcb_operation = CLOSE; /*spec. "derived" operation*/
	/*
	 * We're going to ignore most all signals from now on...
	 */
	otCatchSignals(1);

	if (operation == ENTER) {
	    errCode = otEnterObject(tmpfile); 
	    DBUG_MED((stderr, "cmdCommit ENTER, enter object = %d\n", errCode));

	    if (!errCode)
		errCode = otBuildQueueItem(tStruct);
	    tclTemp[0] = 0;
	    otCatchSignals(2);
	    otCleanup();
	}
	else if (operation == UPDATE) {
	    /*
	     * Store an object under RCS, call otNotify and otMakeSum.
	     */
	    errCode = otUpdateObject(tmpfile); 
	    DBUG_MED((stderr, "cmdCommit ENTER, enter object = %d\n", errCode));

	    if (!errCode)
		errCode = otBuildQueueItem(tStruct);
	    tclTemp[0] = 0;
	    otCatchSignals(2);
	    otCleanup();
	}

        /*
         * Start the otqm     otqm -p<proj>
         */

	if ( !errCode ) {
	    memset(qmpath, 0, PATHLEN);

#ifdef NATA
	if (envpath = getenv("PATH"))  {
	    locateCmd("otqm", envpath, qmpath);
	    DBUG_MIN((stderr, "cmdCommit locateCmd otqm path is %s\n", qmpath));
	}
	if (!qmpath[0])  {
	    locateCmd("otqm", "/usr/local/bin", qmpath);
	    DBUG_MIN((stderr, "cmdCommit locateCmd otqm path is %s\n", qmpath));
	}
#endif
	    locateCmd("otqm", "/etc", qmpath);
	    DBUG_MIN((stderr, "cmdCommit locateCmd otqm path is %s\n", qmpath));

	    if (!qmpath[0]) {
		logWarn("%s: otqm command not on a server",
		    otCB->cb_pcb->pcb_uName ? otCB->cb_pcb->pcb_uName : "ot");
		DBUG_MIN((stderr, "cmdCommit otqm command not on a server\n"));

/*** temporary line  ( NATA ) ***/
/* strcpy(qmpath, "/usr/sandbox/otSbox/obj/at386_osf1/bin"); */

	    }

	    if (qmpath[0]) {
		sprintf(command, "(%s/otqm -p%s) 2>/dev/null",
		    qmpath, otCB->cb_project);
		if (!fork())    {
		    if (system(command)) {
			/* put an error message */
			logWarn("%s: call to otqm failed", 
			    otCB->cb_pcb->pcb_uName ? otCB->cb_pcb->pcb_uName : "ot");
			DBUG_MIN((stderr, "cmdCommit system call to otqm failed\n"));
		    }
		    exit(0);
		}
	    }
	}

	otCatchSignals(0);
    }

    if (errCode || postErr) 
	Tcl_SetResult( interp, otGetPostedMessage(), TCL_VOLATILE );
    else
	Tcl_SetResult( interp, "", TCL_VOLATILE );

    if (errCode)
	return TCL_ERROR;
    else 
	return TCL_OK;


}


/*
 *                           c m d F i e l d
 *
 * Give or assign information w/r/t a field not in the metatemplate.
 *
 * field [orig] fieldName
 * field [orig] fieldName compare type <pattern>
 * field [orig] fieldName append { text mandatory many } # type option degree
 * field [orig] fieldName append { text optional  one }
 * field [orig] fieldName append { stat,closed optional one }
 * field [orig] fieldName set    <value>
 * field [orig] fieldName delete
 */
int
cmdField(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    bool orig;
    OTTemplate *ott_template;
    char *fieldName, *option;
    OTHeaderField *hfp1, *hfp2;
    OTErr compStatus;
    int i, stat, todArgc;
    char *typDegOpt, *val;
    char **todArgv;
    otType tmptype;

    if (otCB->cb_operation == QUERY) 
	ott_template = otCB->cb_pcb->pcb_template;
    else {
	if ( argc > 2 && !strcmp(argv[1], "orig") ) {
	    ott_template = otCB->cb_ecb->ecb_origStruct;
	    for ( i = 1; i < argc-1; i++ ) {
		argv[i] = argv[i+1];
	    }
	    argv[argc-1] = 0;
	    argc--;
	    orig = TRUE;
	} else
	    ott_template = otCB->cb_ecb->ecb_tStruct;
    }

    fieldName = argv[1];
    if ( argc == 2 ) {
	/*
	 * Query: just return the value.
	 */
	if ( !(val = otGetHeaderFieldValue( fieldName, ott_template )) )
	    interp->result[0] = 0;
	else
	    Tcl_SetResult(interp, val, TCL_VOLATILE);
	return TCL_OK;
    } else if ( argc > 2 ) {
	option = argv[2];

	if (	    !strcmp(option, "compare" ) ) {	/* COMPARE */
	    bool queryPatternSet;

	    val = otGetHeaderFieldValue( fieldName, ott_template );

	    tmptype = otGetType(argv[3]);
	    if ( (tmptype == TYPE_NULL) || (tmptype == TYPE_ENUM) )
		tmptype = TYPE_ENUM;

	    queryPatternSet = argv[4][0] && 
		( (argv[4][0] != '{') || (argv[4][1] != '}') );
	    if ( !val && !queryPatternSet)
		stat = 1; 
	    else if ( ( val && !queryPatternSet ) || ( !val && queryPatternSet ) )
		stat = 0;
	    else {
		compStatus = otCompare(val, tmptype, argv[4]);
		stat = (compStatus == OT_SUCCESS) ? 1 : 0;
	    }

	    if ( stat )
		interp->result = "1";
	    else
		interp->result = "0";
	    return TCL_OK;

	} else if ( !strcmp(option, "append") ) {	/* APPEND */
	    typDegOpt = argv[3];
	    /*
	     * Find end of header fields.
	     */
	    hfp1 = ott_template->tr;
	    while ( hfp1->field && *hfp1->field )
		hfp1++;
	    /*
	     * Set field, type, optionality, degree.
	     */
	    if ( otCopyString(fieldName, &(hfp1->field)) ) {
		Tcl_SetResult(interp, "memory allocation error", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if ( Tcl_SplitList(interp, typDegOpt, &todArgc, &todArgv) ) {
		Tcl_SetResult(interp, "Error in type argument", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if ( !strcmp(todArgv[0], "{}") )
		todArgv[0] = 0;

	    if ( (hfp1->type = otGetType(todArgv[0])) == TYPE_ENUM ) {
		if ( otCopyString(todArgv[0], &(hfp1->list)) ) {
		    Tcl_SetResult(interp, "memory allocation error",
			TCL_STATIC);
		    return TCL_ERROR;
		}
	    }
	    
	    if ( !strcmp(todArgv[1], "mandatory") )
		hfp1->optionality = OPT_MAND;
	    else
		hfp1->optionality = OPT_OPTL;

	    if ( !strcmp(todArgv[2], "many") )
		hfp1->degree = DEGREE_MANY;
	    else
	        hfp1->degree = DEGREE_ONE;

	} else if ( !strcmp(option, "delete") ) {	/* DELETE */
	    hfp1 = ott_template->tr;
	    while ( hfp1->field && strcmp(hfp1->field, fieldName) )
		hfp1++;
	    if ( hfp1->field && *hfp1->field ) {
		hfp2 = hfp1 + 1;

		if (hfp1->field)
		    free(hfp1->field);
		if (hfp1->list)
		    free(hfp1->list);
		if (hfp1->value)
		    free(hfp1->value);

		memset(hfp1, 0, sizeof(OTHeaderField));
		while ( hfp2->field && *hfp2->field ) {
		    memcpy(hfp1, hfp2, sizeof(OTHeaderField));
		    hfp1++;
		    hfp2++;
		}
		memset(hfp2, 0, sizeof(OTHeaderField));
	    }
	} else if ( !strcmp(option, "set") ) {		/* SET */
	    if (!otSetHeaderFieldValue(fieldName, argv[3], ott_template))
		return TCL_OK;
	    else {
		Tcl_SetResult(interp, "error in assignment", TCL_VOLATILE);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_SetResult(interp,
		"Usage: field [orig] fieldName [append] [set] [delete]", TCL_STATIC);
	    return TCL_ERROR;
	}

    }

    Tcl_SetResult( interp, "", TCL_VOLATILE );
    return TCL_OK;

}



/*
 *                                c m d D e l t a
 *
 *	            delta date agent fieldName oldVal newValue
 *
 * delta's arguments act as matching expressions for the information cached in 
 * the notedata buffer for the current template.  The delta command returns
 * all history lines for this template such that
 *
 * - date (which may be a range) matches its date, as defined by OT's
 *   date matching rules
 * AND
 * - the fieldName matches the fieldName of the string (exact match required)
 *   which is actually the keyword of the field which changed
 * AND
 * - oldVal is a match, where match is defined relative to the field's
 *   OT type, to the "previous" value in the history line
 * AND
 * - newVal is a match, where match is defined relative to the field's
 *   OT type, to the "new" value in the history line
 *
 * The delta command recognizes a null string as a match, e.g.
 *
 *	delta "" "" stat "" closed
 *
 * returns all history lines such that they reflect a change in the status
 * from open to closed.
 *
 * Also,
 *
 *	delta 12/01/92-. "" stat "" ""
 *
 * returns all history lines for the current template which reflect a change
 * in status from Dec. 1, 1992 until today, regardless of their value.
 * 
 * These history lines are returned in a TCL list type, so 
 *
 *	puts stdout [llength [delta 12/01/92-. "" stat "" ""]]
 *
 * would print the number of history lines matching the conditions mentioned
 * above.
 *
 * The 'delta append' command is only used as an intermediate representation
 * when sending the CR to update from the server to the client.  The append
 * flag is not recognized by the server.
 */
int
cmdDelta(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    OTControlBlock *cb = otCB;
    bool orig = FALSE;
    otType type;
    int sargc, targc;
    int i, j, k;
    char *sargv[6], *targv[MAXNDATA];
    char *rval;
    char *matchList[MAXNDATA];
    char *usrDate, *usrAgent, *usrField, *usrOldVal, *usrNewVal;
    char *templateAgent, *templateField, *templateOldVal,
	*templateNewVal;
    char templateDate[MAXDATELEN];
    bool dateMatch, agentMatch, fieldMatch, oldValMatch, newValMatch;
    OTTemplate *ott_template;
    OTNotedata *ndp;

    i = j = k = 0;

    if ( argc != 6 ) {
	Tcl_SetResult( interp,
	    "Usage: delta date agent keyword/fieldName oldVal newVal",
	    TCL_STATIC );
	return TCL_ERROR;
    }

    if (cb->cb_operation == QUERY) {
	if ( !(ott_template = otCB->cb_pcb->pcb_template) ) {
	    Tcl_SetResult( interp, "", TCL_VOLATILE );
	    return TCL_OK;
	}
    }
    else {
	if ( (argc > 1) && !strcmp(argv[1], "orig") ) {
	    ott_template = cb->cb_ecb->ecb_origStruct;
	    for ( i = 1 ; i < argc-1; i++ )
		argv[i] = argv[i+1];
	    argv[argc-1] = 0;
	    argc--;
	    orig = TRUE;
	} else
	    ott_template = cb->cb_ecb->ecb_tStruct;
    }

    DBUG_MED((stderr, "usrDate = |%s|, usrAgent = |%s|, usrField = |%s|, usrOldVal = |%s|, usrNewVal = |%s|\n", argv[1], argv[2], argv[3], argv[4], argv[5]));

    /*
     * These trims are really necessary.  TCL coders could say
     *
     *	delta { } { } { } { } { }
     *
     * thinking they are asking for all HISTORY lines and they would get
     * argv[1], etc equal to " ".
     */
    for( i = 0; i < argc; i++)
        otTrimWs(argv[i]);

    usrDate   = argv[1];
    usrAgent  = argv[2];
    usrField  = argv[3];
    usrOldVal = argv[4];
    usrNewVal = argv[5];

    for ( ndp = ott_template->ndp; ndp && ndp->nd_flags; ndp++ ) {
        /*
	 * Compare current template against user criteria.
	 */
	dateMatch = agentMatch = fieldMatch = oldValMatch = newValMatch = FALSE;
        if ( ! ND_ISHISTORY(ndp->nd_flags) )
	    continue;
	else {
	    sprintf(templateDate, "%02d/%02d/%02d", (int)ndp->nd_mon,
		(int)ndp->nd_day, (int)ndp->nd_year);
	    templateAgent = ndp->nd_agent;
	    templateField = ndp->nd_field;

	    /*
	     * Determine the type of the data changed.  That means
	     * 		... stat: open -> closed
	     * matches 'open' as a list element, while
	     *		... rdate: 12/01/92 -> 12/02/92
	     * matches 12/01/92 as a date.
	     */
	    if ( !templateField || (templateField && !*templateField) ) {
		type = TYPE_LIST;
	    } else if ( (type = otGetAbbrFieldType(templateField, 
		otCB->cb_pcb->pcb_project)) == -1 ) {
		type = TYPE_LIST;
	    }

	    /*
	     * To match "NULL" in HISTORY line, use "NULL" (!)
	     */
	    templateOldVal = ndp->nd_previous;
	    templateNewVal = ndp->nd_current;

	    DBUG_MED((stderr, "templateDate = |%s|, templateAgent = |%s|, templateField = |%s|, templateOldVal = |%s|, templateNewVal = |%s|\n", templateDate, templateAgent, templateField, templateOldVal, templateNewVal));

	    if ( !*usrDate || !otCompDate(templateDate, usrDate) )
		dateMatch = TRUE;

	    if ( !*usrAgent || ( templateAgent && (*templateAgent == *usrAgent) &&
		!strcmp(templateAgent, usrAgent) ) )
		agentMatch = TRUE;

	    if ( !*usrField || ( templateField && (*templateField == *usrField) &&
		!strcmp(templateField, usrField) ) )
	        fieldMatch = TRUE;

	    if ( !*usrOldVal || ( templateOldVal && !otCompare(templateOldVal, type, usrOldVal) ) )
		oldValMatch = TRUE;
	    if ( !*usrNewVal || ( templateNewVal && !otCompare(templateNewVal, type, usrNewVal) ) )
		newValMatch = TRUE;

	    if ( dateMatch && agentMatch && fieldMatch && oldValMatch && newValMatch ) {
		sargv[0] = templateDate;
		sargv[1] = templateAgent;
		sargv[2] = templateField;
		sargv[3] = templateOldVal;
		sargv[4] = templateNewVal;
		sargv[5] = 0;
		sargc = 5;
		matchList[k++] = Tcl_Merge(sargc, sargv);
	    }
	}
    }

    /*
     * Will de-allocation really be done automatically by TCL?
     */
    if ( k ) {
	targc = k;
	for( j = 0; j < k; j++ ) {
	    targv[j] = matchList[j];
	}
	rval = Tcl_Merge(targc, targv);
	Tcl_SetResult( interp, rval, TCL_DYNAMIC );
    } else {
	Tcl_SetResult( interp, "", TCL_DYNAMIC );
    }

    return TCL_OK;
}

/*
 *                          c m d T c l S t r i n g
 *
 */
int
cmdTclString(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    OTErr err;
    OTTemplate *ott_template;
    bool orig = FALSE;
    int i;
    char *tclString;

    if ( (argc > 1) && !strcmp(argv[1], "orig") ) {
	ott_template = otCB->cb_ecb->ecb_origStruct;
	for ( i = 1 ; i < argc-1; i++ )
	    argv[i] = argv[i+1];
	argv[argc-1] = 0;
	argc--;
	orig = TRUE;
    } else
	ott_template = otCB->cb_ecb->ecb_tStruct;

    err = otWriteTemplateToTcl(otCB->cb_pcb->pcb_project, ott_template,
	&tclString, orig);
    if ( err ) {
	Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	return TCL_ERROR;
    } else {
	Tcl_SetResult(interp, tclString, TCL_DYNAMIC);
	return TCL_OK;
    }

}


/*
 *                           c m d N o t e d a t a
 *
 *		    set res [notedata "+BSUBMIT" {} {} {} ]
 *
 * <1>	notedata [orig] date agent keyword text
 * <2>	notedata [orig] [index] date agent keyword text
 * <3>	notedata [orig] <linenum>
 * <4>	notedata [orig] append keyword text
 * <5>	notedata [orig] set <linenum> keyword text
 *
 * - In the first form of the command ( <1> ), the notedata command returns 
 * all notedata lines for this template such that 
 *
 *	- keyword matches the keyword if specified, otherwise match etc
 *	- date matches the date (using OT date matching expression)
 *	- agent matches the agent
 *	- text matches the string which follows (regular expression match)
 *
 * As in delta, notedata recognizes a null string as a match, so
 *
 *	notedata +BSUBMIT "" "" ""
 *
 * matches any notedata line w/ the keyword +BSUBMIT.
 *
 * - In the second form of the command ( <2> ), the first argument is the
 * word 'index'.  The procedure returns the indices of matching notedata
 * lines, where 0 is the index of the first line.
 *
 * - In the third form of the command ( <3> ), the first argument
 * is a digit n: the command then returns the n-th notedata line.
 *
 * - In the fourth form of the command ( <4> ), the command appends the
 * keyword and text to the end of the notedata for the current template.  It
 * is used to add a "notedata" line, such as the +BSUBMIT line.  (The date
 * and agent are added automatically).
 *
 * - In the fifth form of the command ( <5> ), the command replaces the n-th
 * notedata line with the keyword and text given.
 */
int
cmdNotedata(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    OTErr err;
    OTControlBlock *cb = otCB;
    bool orig = FALSE;
    bool index = FALSE;
    int sargc;
    register int i, j, k;
    int linenum, tmpday;
    char *sargv[6];
    char *rval;
    char *matchList[MAXNDATA];
    char *usrDate, *usrAgent, *usrKwd, *usrText;
    char *templateAgent, *templateKwd, *templateText;
    char templateDate[MAXDATELEN];
    bool dateMatch, agentMatch, kwdMatch, textMatch;
    OTTemplate *ott_template;
    OTNotedata *ndp;
    OTPrivateCB *pvp;
    char buf[LONGVALUE];
    long timer;
    struct tm *timeptr;
    char times[NAMELEN];
    char tmpweekday[NAMELEN], tmpmon[NAMELEN];
    char tmptime[NAMELEN], tmpyear[NAMELEN];
    int yr;

    pvp = otCB->cb_pcb;
    i = j = k = 0;

    buf[0] = 0;

    timer = time(0L);
    timeptr = localtime(&timer);
    strcpy(times, asctime(timeptr));
    times[strlen(times) - 1] = 0;

    if ( sscanf(times, "%3s %3s %d %8s %4s", tmpweekday, tmpmon, &tmpday,
	tmptime, tmpyear) != 5 ) {
	Tcl_SetResult( interp, "time storage error", TCL_VOLATILE );
	return TCL_ERROR;
    }

    if (cb->cb_operation == QUERY) {
	if ( !(ott_template = pvp->pcb_template) ) {
	    Tcl_SetResult( interp, "", TCL_VOLATILE );
	    return TCL_OK;
	}
    } else if ( !strcmp(argv[1], "orig") ) {
	if ( argc < 3 ) {
	    Tcl_SetResult( interp, 
		"wrong # args: should be \"notedata orig [index | append | set <linenum>] [date agent] type keyword text\"", TCL_STATIC );
	    return TCL_ERROR;
	} else {
	    ott_template = cb->cb_ecb->ecb_origStruct;
	    for ( i = 1; i < argc-1; i++ )
		argv[i] = argv[i+1];
	    argv[argc-1] = 0;
	    argc--;
	    orig = TRUE;
	} 
    } else
	ott_template = cb->cb_ecb->ecb_tStruct;

    /*
     * Command parse order:
     * - look for 'append'
     * - look for 'set'
     * - look for command w/ 2 arguments
     */

    if ( argv[1][0] == 'd' && !strcmp(argv[1], "delete") ) {

        if (orig) {
	    Tcl_SetResult(interp, "original notedata line cannot be deleted\"",
		TCL_VOLATILE);
	    return TCL_OK;
	} else if ( argc != 3 ) {
	    Tcl_SetResult(interp, 
		"wrong # args: should be \"notedata delete linenum\"",
		TCL_VOLATILE);
	    return TCL_ERROR;
	} else {
	    linenum = atoi(argv[2]);
	    ndp = ott_template->ndp;

	    for (i = 0; (i < linenum) && (i < MAXNDATA) && ndp;	i++, ndp++ );

	    if ( linenum == i ) {
	        if ( ND_ISRESERVED(ndp->nd_flags) ) {
		    Tcl_SetResult(interp, "cannot delete reserved word",
			TCL_VOLATILE);
		    return TCL_ERROR;
		} else {
		    FREEANDZERO( ndp->nd_kwd );
		    FREEANDZERO( ndp->nd_agent );
		    FREEANDZERO( ndp->nd_field );
		    FREEANDZERO( ndp->nd_previous );
		    FREEANDZERO( ndp->nd_current );
		    do {
			memcpy( ndp, ndp + 1, sizeof(OTNotedata));
		    } while ( (++ndp)->nd_flags );
		    memset( ndp, 0, sizeof(OTNotedata) );
		    
		    Tcl_SetResult(interp, "", TCL_VOLATILE);
		    return TCL_OK;
		}
	    } else {
		Tcl_SetResult(interp, "line not found", TCL_VOLATILE);
		return TCL_OK;
	    }
	}
    } else if ( argv[1][0] == 'a' && !strcmp(argv[1], "append") ) {
        if (orig) {
	    Tcl_SetResult(interp, "original notedata line not assignable",
		TCL_VOLATILE);
	    return TCL_OK;
	} else if ( argc != 4 ) {
	    Tcl_SetResult(interp, 
		"wrong # args: should be \"notedata append keyword text\"",
		TCL_VOLATILE);
	    return TCL_ERROR;
	} else {
	    otTrimWs(argv[2]);
	    otTrimWs(argv[3]);

	    usrKwd = argv[2];
	    usrText = argv[3];
	    
	    /*
	     * '1' is used here because notedata allows appends or sets
	     * user-visible (and not reserved) keywords and data.
	     */
	    if ( usrText && *usrText )
		sprintf(buf, "+1%s %s %s %s", usrKwd, times, 
		    pvp->pcb_uName, usrText);
	    else 
		sprintf(buf, "+1%s %s %s", usrKwd, times, pvp->pcb_uName);
	    err = otAppendNotedataToTemplateFromString(ott_template, buf);
	    if ( err ) {
		Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
		return TCL_ERROR;
	    } else {
		Tcl_SetResult(interp, "", TCL_VOLATILE);
		return TCL_OK;
	    }
	}
    } else if (!strcmp(argv[1], "set")) {
	/*
	 * notedata [orig] set <linenum> keyword text
	 */
        if (orig) {
	    Tcl_SetResult(interp, "original notedata line not assignable",
		TCL_VOLATILE);
	    return TCL_OK;
	} else if ( !strcmp(argv[1], "set") && (argc != 5) ) {
	    Tcl_SetResult(interp, 
	       "wrong # args: should be \"notedata set linenum keyword text\"",
		TCL_VOLATILE);
	    return TCL_ERROR;
	} else {
	    usrKwd = argv[3];
	    usrText = argv[4];

	    linenum = atoi(argv[2]);
	    ndp = ott_template->ndp;
	    
	    for ( i = 0, ndp = ott_template->ndp ; (i < MAXNDATA) && ndp; 
		 i++, ndp++ ) {
	        if ( linenum == i ) {

		    FREEANDZERO( ndp->nd_kwd );
		    FREEANDZERO( ndp->nd_agent );
		    FREEANDZERO( ndp->nd_field );
		    FREEANDZERO( ndp->nd_previous );
		    FREEANDZERO( ndp->nd_current );

		    /*
		     * For the moment, all user-contributed lines are
		     * visible in full reports.
		     */
		    ndp->nd_flags = ND_USER | ND_VISB;
		    ndp->nd_weekday = (char)dayNumberForName(tmpweekday);
		    ndp->nd_mon = (char)monthNumberForName(tmpmon);
		    ndp->nd_day = (char)tmpday;

		    yr = atoi(&tmpyear[2]);
		    yr = (yr > 1900) ? (yr - 1900) : yr;
		    ndp->nd_year = yr;
		    memcpy(ndp->nd_time, tmptime, MAXTIME);
		    err = otCopyString(usrKwd, &(ndp->nd_kwd));
		    if ( err ) {
			Tcl_SetResult(interp, "memory allocation error",
			    TCL_VOLATILE);
			return TCL_ERROR;
		    }
		    err = otCopyString(pvp->pcb_uName, &(ndp->nd_agent));
		    if ( err ) {
			Tcl_SetResult(interp, "memory allocation error",
			    TCL_VOLATILE);
			return TCL_ERROR;
		    }
		    err = otCopyString(usrText, &(ndp->nd_current));
		    if ( err ) {
			Tcl_SetResult(interp, "memory allocation error",
			    TCL_VOLATILE);
			return TCL_ERROR;
		    }
		    Tcl_SetResult(interp, "", TCL_VOLATILE);
		    return TCL_OK;
		}
	    }
	    Tcl_SetResult(interp, "not found", TCL_VOLATILE);
	    return TCL_OK;
	}
    } else if ( argc == 2 ) {
	/*
	 * notedata <linenum>
	 * fetches linenum'th notedata entry
	 */

	linenum = atoi(argv[1]);
	for ( i = 0, ndp = ott_template->ndp ; (i < MAXNDATA) && ndp; 
	    i++, ndp++ ) {
	    if ( linenum == i ) {
		sprintf(templateDate, "%02d/%02d/%02d", (int)ndp->nd_mon,
		    (int)ndp->nd_day, (int)ndp->nd_year);
		sargv[0] = templateDate;
		sargv[1] = ndp->nd_agent;
		sargv[2] = ndp->nd_kwd;
		sargv[3] = ndp->nd_current;
		sargv[4] = 0;
		sargc = 4;
		matchList[k++] = Tcl_Merge(sargc, sargv);
		Tcl_SetResult(interp, "not found", TCL_VOLATILE);
		return TCL_OK;
	    }
	}
	Tcl_SetResult(interp, "not found", TCL_VOLATILE);
	return TCL_OK;

    } else {
	/*
	 * notedata [orig] [index] date agent keyword text
	 *  - Get notedata entries which match keyword/data/agent/text
	 *    and return the entries themselves, or their indices if
	 *    'index' is indicated.
	 */
	if ( (argc > 1) && !strcmp(argv[1], "index") ) {
	    for( i = 1; i < argc - 1; i++ )
		argv[i] = argv[i+1];
	    argv[argc-1] = 0;
	    argc--;
	    index = TRUE;
	}

	for (i = 0, ndp = ott_template->ndp; (i < MAXNDATA) && ndp && 
	     ndp->nd_flags; i++, ndp++) {

	    /*
	     * Compare current notedata line against user criteria.
	     */
	    dateMatch = agentMatch = kwdMatch = textMatch = FALSE;
	    if ( ND_ISHISTORY(ndp->nd_flags) )
		continue;
	    else {
		sprintf(templateDate, "%02d/%02d/%02d", (int)ndp->nd_mon,
		    (int)ndp->nd_day, (int)ndp->nd_year);
		templateAgent = ndp->nd_agent;
		templateKwd = ndp->nd_kwd;
		/*
		 * To match "NULL" in HISTORY line, use "NULL" (!)
		 */
		templateText = ndp->nd_current;

		for( j = 1; j < argc; j++)
		    otTrimWs(argv[j]);

		usrDate  = argv[1]; 
		usrAgent = argv[2];
		usrKwd   = argv[3];
		usrText  = argv[4];

DBUG_MED((stderr, "templateDate = |%s|, templateAgent = |%s|, templateKwd = |%s|, templateText = |%s|\n", templateDate, templateAgent, templateKwd, templateText));

		if ( !*usrDate || !otCompDate(templateDate, usrDate) )
		    dateMatch = TRUE;

		if ( !*usrAgent || ( (*templateAgent == *usrAgent) &&
		    !strcmp(templateAgent, usrAgent) ) )
		    agentMatch = TRUE;
		if ( !*usrKwd || ( (*templateKwd == *usrKwd) &&
		    !strcmp(templateKwd, usrKwd) ) )
		    kwdMatch = TRUE;

		if ( !*usrText || !otCompare(templateText, TYPE_TEXT, usrText) )
		    textMatch = TRUE;

		if ( dateMatch && agentMatch && kwdMatch && textMatch ) {
		    if ( !index ) {
			sargv[0] = templateDate;
			sargv[1] = templateAgent;
			sargv[2] = templateKwd;
			sargv[3] = templateText;
			sargv[4] = 0;
			sargc = 4;
			matchList[k++] = Tcl_Merge(sargc, sargv);
		    } else {
			sprintf(buf, "%d", i);
			matchList[k++] = strdup(buf);
		    }
		}
	    }
	}
	matchList[k] = 0;
	if ( k ) {
	    rval = Tcl_Merge(k, matchList);
	    for (i = 0; i < k; i++)
		if (matchList[i])
		    free(matchList[i]);

	    Tcl_SetResult( interp, rval, TCL_DYNAMIC );
	} else {
	    Tcl_SetResult( interp, "", TCL_STATIC );
	}

    }

    return TCL_OK;

}

/*
 * The client must receive and analyze 'notedataBuild' commands, which contain
 * all the information in the notedata lines so this may be displayed by the
 * user interface.  Note that the user may not edit the lines in any way in 
 * the user interface: this may only be done by a TCL command.  This was the
 * only way to keep the notedata lines from becoming trashed in the UI.
 *
 * This also illustrates two different techniques for parsing TCL lists:
 * the TclSplitList() routine, and the TclFindElement() routine.
 *
 * Syntax:
 * notedataBuild [orig] list, where list is composed of elements of the
 * following structure:
 * <flags> weekday date time kwd agent fieldName previous current
 *
 * where flags = ?
 *	   weekday
 *	   date is date in typical OT format mm/dd/yy
 * 	   time is time in form hh:mm:ss
 * 	   kwd is keyword, as in "HISTORY", "BSUBMIT"
 *       agent is 
 *       field name (HISTORY only?)
 *	   previous is previous value (HISTORY only, "" otherwise)
 *	   current is current value (HISTORY OR user-contributed keyword)
 */
int cmdNotedataBuildClient(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    int result, size, parenthesized, sargc, i;
    int mo, da, yr;
    char *p, *element, *buf;
    char flag;
    char **sargvp;
    OTTemplate *ott_template;
    OTNotedata *base_ndp, *ndp;
    OTErr err;
    bool orig = FALSE;
    
    if ( argc > 2 && !strcmp(argv[1], "orig") ) {
	ott_template = otCB->cb_ecb->ecb_origStruct;
	for ( i = 1; i < argc-1; i++ )
	    argv[i] = argv[i+1];
	argv[argc-1] = 0;
	argc--;
	orig = TRUE;
    } else 
	ott_template = otCB->cb_ecb->ecb_tStruct;
        
    if ( argc != 2 ) {
	Tcl_SetResult(interp, "Usage: notedataBuild [orig] list", TCL_STATIC);
	return TCL_ERROR;
    }

    ndp = (OTNotedata *)calloc(MAXNDATA, sizeof(OTNotedata));
    if ( !ndp ) {
	Tcl_SetResult(interp, "calloc() failure in cmdNotedataBuildClient",
	    TCL_STATIC);
	return TCL_ERROR;
    }

    base_ndp = ndp;
    /*
     * Parse through list.
     */
    p = argv[1];
    while ( 1 ) {

	if (*p == 0)
	    break;
	if ( TclFindElement(interp, p, &element, &p, &size, &parenthesized)
	    != TCL_OK )
	    break;

	if ( size ) {
	    if ( !(buf = calloc(size + 1, sizeof(char))) ) {
		Tcl_SetResult(interp, "error in malloc()", TCL_STATIC);
		return TCL_ERROR;
	    } else {
		if (parenthesized)
		    memcpy((VOID *) buf, (VOID *) element, size);
		else
		    TclCopyAndCollapse(size, element, buf);
	    }

	    if ( (result = Tcl_SplitList(interp, buf, &sargc, &sargvp)) != 
		TCL_OK )
	        goto fieldsDone;

	    if ( sargc != 9 ) {
	        Tcl_SetResult(interp,
		    "notedata elements contain: flag weekday date time kwd agent fieldName previous current", TCL_STATIC);
		result = TCL_ERROR;
		goto fieldsDone;
	    }

	    i = 0;
	    flag = sargvp[i][0];

	    i++;
	    ndp->nd_weekday = (char)dayNumberForName(sargvp[i]);
	    
	    i++;
	    if (sscanf(sargvp[i], "%02d/%02d/%02d", &mo, &da, &yr) != 3) {
	        Tcl_SetResult(interp, "notedataBuild: date error", 
		    TCL_STATIC);
		result = TCL_ERROR;
		goto fieldsDone;
	    }
	    ndp->nd_mon = (char)mo;
	    ndp->nd_day = (char)da;
	    ndp->nd_year = (char)yr;
		
	    i++;
	    strncpy(ndp->nd_time, sargvp[i], MAXTIME);
	    
	    i++;
	    err = otCopyString(sargvp[i], &(ndp->nd_kwd));
	    if ( err ) {
	        Tcl_SetResult(interp, "notedataBuild: keyword copy failed",
		    TCL_STATIC);
		result = TCL_ERROR;
		goto fieldsDone;
	    }
		
	    i++;
	    err = otCopyString(sargvp[i], &(ndp->nd_agent));
	    if ( err ) {
	        Tcl_SetResult(interp, "notedataBuild: agent copy failed",
		    TCL_STATIC);
		result = TCL_ERROR;
		goto fieldsDone;
	    }
		
	    i++;
	    err = otCopyString(sargvp[i], &(ndp->nd_field));
	    if ( err ) {
	        Tcl_SetResult(interp, "notedataBuild: field copy failed",
		    TCL_STATIC);
		result = TCL_ERROR;
		goto fieldsDone;
	    }

	    i++;
	    err = otCopyString(sargvp[i], &(ndp->nd_previous));
	    if ( err ) {
	        Tcl_SetResult(interp, 
		"notedataBuild: previous value copy failed", TCL_STATIC);
		result = TCL_ERROR;
		goto fieldsDone;
	    }
		    
	    i++;
	    err = otCopyString(sargvp[i], &(ndp->nd_current));
	    if ( err ) {
	        Tcl_SetResult(interp, 
		    "notedataBuild: current valuecopy failed", TCL_STATIC);
		result = TCL_ERROR;
		goto fieldsDone;
	    }

fieldsDone:
	    if ( result != TCL_OK ) {
		free(buf);
		free(ndp);
		free((char *)sargvp);
		return TCL_ERROR;
	    }

	    switch (flag) {
		/*
		 * 0 - a reserved word --> always visible
		 * 1 - user-contributed and visible in reports
		 * 2 - user-contributed but invisible
		 */
		case '0':  ndp->nd_flags |= ND_RESV;
		 	   break;
		case '2':  ndp->nd_flags |= ND_USER;
		           break;
		case '1':
		default:
			   ndp->nd_flags |= ND_VISB | ND_USER;
			   break;
	    }

	    if ( !strcmp(ndp->nd_kwd, ND_HISTORY_KWD) ) {
		ndp->nd_flags |= ND_HIST;
		free(ndp->nd_kwd);
		ndp->nd_kwd = 0;
	    } else if ( !strcmp(ndp->nd_kwd, ND_LUPDATE_KWD) ) {
		ndp->nd_flags |= ND_LUPD;
		free(ndp->nd_kwd);
		ndp->nd_kwd = 0;
	    }

	    if (ndp > (base_ndp + MAXNDATA)) {
		Tcl_SetResult(interp, "template has too many notedata lines",
		    TCL_STATIC);
		return TCL_ERROR;
	    }
	    else
		ndp++;

	    free((char *)sargvp);
	    free(buf);

	}
    }

    freeNotedata(ott_template->ndp);
    ott_template->ndp = base_ndp;
    interp->result[0] = 0;
    return TCL_OK;

}


/*
 * In the server, the notedataBuild command returns a null string: it is
 * a stub, since the server builds all its notedata, including the history
 * lines, from explicit changes from the orig structure to the current one
 * (in the case of HISTORY), and via 'notedata append' commands issued by a
 * user.
 */
int
cmdNotedataBuildServer(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{

    interp->result = "";
    return TCL_OK;

}


#define SHORT_REPORT_BUFFER 100
#define LONG_REPORT_BUFFER   10

int
cmdQuery(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    ClientData notUsed;
    OTTemplate *tp = 0;
    OTPrivateCB *pvp;
    OTQueryCB *qcb;
    OTProject *prjp;
    OTErr err;
    FILE *fp;
    OpenFile *filePtr;
    char hdrBuf[LONGVALUE], kwikPath[LONGVALUE], connFile[SHORTSTR];
    char *crStr, *ctxtString, *cp, *tclStr;
    int crNum, i, sargc, devnull, stdoutFd, stderrFd, result;
    char *sargv[5];
    char tclOpenCommand[LONGVALUE];

    pvp = otCB->cb_pcb;
    qcb = otCB->cb_qcb;
    otCB->cb_operation = QUERY;
    prjp = pvp->pcb_project;
    if ( !prjp ) {
	Tcl_SetResult(interp, "no project specified", TCL_VOLATILE);
	return TCL_ERROR;
    }
    if ( argc > 2 ) {
	Tcl_SetResult(interp, "usage: query [next | end | <number>]",
	    TCL_STATIC);
	return TCL_ERROR;
    }
    
    if ( argc == 1 ) {

	if (qcb->qcb_filter) {
	    otFreeMetaTemplate(qcb->qcb_filter);
	    qcb->qcb_filter = 0;
	}
	err = otInitQuery();
	if ( err ) {
	    Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	    return TCL_ERROR;
	}

	/*
	 * Evaluate the tclString, which may have a procedure in its begin
	 * clause that sets historyLines to TRUE.  We have to redirect any
	 * stdout written by the procedure to /dev/null - temporarily.
	 */
	stdoutFd = dup(1);
	stderrFd = dup(2);
	devnull = open("/dev/null", O_RDWR, 0);
	dup2(devnull, 1);
	dup2(devnull, 2);
	if (devnull > 2)
	    close(devnull);

	pvp->pcb_tclBegin = 1;
	pvp->pcb_tclEnd = 0;
	pvp->pcb_template = 0;

#ifdef SUPPORT_FOR_USER_TCL
	if ( otCB->cb_tclOtrc ) {
	    tclStr = strdup(otCB->cb_tclOtrc);
	    (void)Tcl_Eval(interp, tclStr);
	    free(tclStr);
	}

	if ( otCB->cb_tclFile ) {
	    tclStr = strdup(otCB->cb_tclFile);
	    (void)Tcl_Eval(interp, tclStr);
	    free(tclStr);
	}
#endif

	if ( otCB->cb_tclString ) {
	    tclStr = strdup(otCB->cb_tclString);
	    (void)Tcl_ExprBoolean(interp, tclStr, &result);
	    free(tclStr);
	}

	dup2(stdoutFd, 1);
	dup2(stderrFd, 2);
	if (stdoutFd > 2)  /* necessary ? */
	    close(stdoutFd);
	if (stderrFd > 2)
	    close(stderrFd);

	pvp->pcb_kwikPixSock = 0;

	if ( !qcb->qcb_historyLines ) {
	    /*
	     * Create connection to otKwikPix server.  
	     * Store the socket returned in the private control block.
	     */
	    sargc = 2;
	    sprintf(kwikPath, "%s/%s/socket", prjp->proj_dir, prjp->name);
	    sargv[0] = "connect";
	    sargv[1] = kwikPath;
	    sargv[2] = 0;

	    if ( Tdp_ConnectCmd(notUsed, interp, sargc, sargv) == TCL_OK ) {
		strcpy(connFile, interp->result);
    
		/*
		 * The return value from Tdp_ConnectCmd() has an appended
		 * status field  which we must remove.
		 */
		for (cp = connFile; *cp; cp++)
		    if (*cp == ' ') {
			*cp = 0;
			break;
		    }
		if (Tdp_GetOpenFile(interp, connFile, 1, 0, &filePtr) == TCL_OK) {
		    strcpy(pvp->pcb_kwikPix, connFile);
		    pvp->pcb_kwikPixSock = filePtr;
		}
	    }
	}

	if ( qcb->qcb_historyLines || !pvp->pcb_kwikPixSock ) {
	    sargc = 3;
	    sargv[0] = "open";
	    sargv[1] = tclOpenCommand;
	    sargv[2] = "r+";
	    sargv[3] = 0;
	    sprintf(tclOpenCommand, "|/etc/otSlowPix -p%s", prjp->name);

	    if ( !Tcl_OpenCmd(notUsed, interp, sargc, (char **)sargv) ) {
		strcpy(pvp->pcb_kwikPix, interp->result);
		if (Tdp_GetOpenFile(interp, interp->result, 1, 0, &filePtr) == TCL_OK)
		    pvp->pcb_kwikPixSock = filePtr;
	    }
	}

	if ( ! pvp->pcb_kwikPixSock ) {
	    Tcl_SetResult(interp, "connect to KwikPix failed", TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_ResetResult(interp);
	/*
	 * Generate a string reflecting the current query context (search
	 * criteria, print layout options etc) and send to otKwikPix.
 	 */
	if ( !(err = otGenQueryContext(&ctxtString)) ) {
	    logWarn("otd to otKwikPix: %s\n", ctxtString);

	    if ( !(fp = pvp->pcb_kwikPixSock->f2) )
		fp = pvp->pcb_kwikPixSock->f;
	    fprintf(fp, "%s\n", ctxtString);
	    fflush(fp);
	    free( ctxtString );

	    /*
	     * Now, just read from kwikpix.
	     */
	    fp = pvp->pcb_kwikPixSock->f;
	    /*
	     * If the query requires access of notes, then just get a list
	     * of CR numbers from otKwikPix representing matches, store it.
	     */
	    if ( !pvp->pcb_quiet && !qcb->qcb_count && !qcb->qcb_summary &&
		(qcb->qcb_fullText || qcb->qcb_keyInText) ) {

		err = otReadStringFromFileHandle( &(qcb->qcb_kwikPixMatches),
		    /* pvp->pcb_kwikPixSock*/ fp, "\n", 0, FALSE);
	        if (err) {
		    Tcl_SetResult(interp, otGetPostedMessage(), TCL_STATIC);
		    return TCL_ERROR;
		} else {
		    if ( qcb->qcb_kwikPixMatches && 
			*qcb->qcb_kwikPixMatches == '\n' ) {
			free(qcb->qcb_kwikPixMatches);
			qcb->qcb_kwikPixMatches = 0;
		    }
		    qcb->qcb_nextMatch = qcb->qcb_kwikPixMatches;
		}
		
	    }
	    /*
	     * Otherwise, send the header for a short report.
	     * Alternatively, in the case of no matches for full reports,
	     * emit error message here.
	     */
	    if ( !qcb->qcb_fullText && !qcb->qcb_summary && !qcb->qcb_count &&
		!qcb->qcb_skipHeader && !pvp->pcb_quiet ) {
		hdrBuf[0] = 0;
		otCB->cb_pcb->pcb_template = tp;
		err = otPrintOneLinerToString(tp, prjp->otmp, TRUE, hdrBuf);
		Tcl_SetResult(interp, hdrBuf, TCL_VOLATILE);
	    }

	    if ( qcb->qcb_fullText && qcb->qcb_nextMatch &&
		!*qcb->qcb_nextMatch && !qcb->qcb_summary && !qcb->qcb_count &&
		!pvp->pcb_quiet ) {
		Tcl_ResetResult(interp);
	    }
	}

    } else if ( argc == 2 ) {
	if        ( !strcmp(argv[1], "next") ) {

	    qcb->qcb_keywordMatchFound = FALSE;

	    Tcl_ResetResult(interp);
	    if ( !qcb->qcb_count && !qcb->qcb_summary && 
		(qcb->qcb_fullText || qcb->qcb_keyInText) ) { /* FULL */

		for (i = 0; i < LONG_REPORT_BUFFER; ) {
		    err = otQueryNextFull(&crStr);
		    if ( err ) {
			Tcl_SetResult(interp, otGetPostedMessage(),
			    TCL_VOLATILE);
			return TCL_ERROR;
		    } else if ( !crStr ) {   /* end of reports */
			break;
		    } else if ( ! *crStr ) {
		        free(crStr);
			continue;
		    } else {
		        qcb->qcb_keywordMatchFound = TRUE;
			Tcl_AppendResult(interp, crStr, 0);
			if ( crStr )
			    free(crStr);
			i++;
		    }
		}
		if ((i < LONG_REPORT_BUFFER) && !qcb->qcb_keywordMatchFound) {
		    Tcl_ResetResult(interp);
		}

	    } else {					     /* SHORT */
		if ( !pvp->pcb_kwikPixSock ) {
		    *interp->result = 0;
		    return TCL_OK;
		} else
		    fp = pvp->pcb_kwikPixSock->f;

		err = OT_SUCCESS;
		for (i = 0; !err && i < SHORT_REPORT_BUFFER; i++) {
		    err = otReadStringFromFileHandle(&cp, fp, "\n", 0, FALSE);
		    if (!err) {
			Tcl_AppendResult(interp, cp, "\n", 0);
		    } else if ( err != OT_TRANSMISSION_CONCLUDED ) {
			Tcl_SetResult(interp, "i/o error", TCL_STATIC);
			return TCL_ERROR;
		    }
		    if (cp)
			free(cp);
		}
	    }

	} else if ( !strcmp(argv[1], "end") ) {

	    if ( qcb->qcb_kwikPixMatches ) {
		free( qcb->qcb_kwikPixMatches );
		qcb->qcb_kwikPixMatches = 0;
	    }
	    qcb->qcb_nextMatch = 0;
	    interp->result = "";

	    if ( pvp->pcb_kwikPixSock ) {

		if ( pvp->pcb_kwikPixSock->numPids ) {
		    sargc = 2;
		    sargv[0] = "close";
		    sargv[1] = pvp->pcb_kwikPix;
		    sargv[2] = 0;

		    (void)Tcl_CloseCmd(notUsed, interp, sargc,(char **)sargv);
		    pvp->pcb_kwikPixSock = 0;
		}
	    }
	}
	else	{	/* query one CR */
	
	    crNum = atoi(argv[1]);
	    if ( crNum <= 0 ) {
		otPutPostedMessage(OT_CLI_ZERO_NUMBER, "-n <number>");
		Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
		return TCL_ERROR;
	    }

	    err = otInitQuery();
	    if ( err ) {
		Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
		return TCL_ERROR;
	    }

	    /* Get the header for the short report */

	    hdrBuf[0] = 0;
	    if (!qcb->qcb_fullText && !qcb->qcb_skipHeader && !pvp->pcb_quiet) {


		otCB->cb_pcb->pcb_template = tp;
		err = otPrintOneLinerToString(tp, prjp->otmp, TRUE, hdrBuf);
		if (err) {
		    Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
		    return TCL_ERROR;
		}
	    }
		
	    if ( !pvp->pcb_quiet) {
		err = otQueryOneCR(crNum, &crStr);
		if (err) {
		    Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
		    return TCL_ERROR;
		}
		Tcl_SetResult(interp, hdrBuf, TCL_VOLATILE);
		Tcl_AppendResult(interp, crStr, 0);
		if ( crStr )
		    free(crStr);
	    }
	}

    }
    return TCL_OK;

}


#define READBUF		    512
#define KWIKPIXBUF	  30000

int
cmdKwikQuery(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    ClientData notUsed;
    OTTemplate *tp = 0;
    OTPrivateCB *pvp;
    OTQueryCB *qcb;
    OTProject *prjp;
    OTErr err;
    FILE *fp;
    OpenFile *filePtr;
    char hdrBuf[LONGVALUE], kwikPath[LONGVALUE], connFile[SHORTSTR];
    char hdrDbfile[LONGVALUE];
    char buf[READBUF];
    char *crStr, *ctxtString, *cp, *dp, *ep;
    int sargc, n, fd, devnull, stdoutFd, stderrFd, result;
    int status;
    char *sargv[5];
    char tclOpenCommand[LONGVALUE], illegFields[LONGVALUE];
    struct stat statbuf;
    unsigned long headerBytes;
    bool kwikPixServer;
#ifndef NORLIMIT
    struct rlimit rl;
#endif


    kwikPixServer = TRUE;
    pvp = otCB->cb_pcb;
    qcb = otCB->cb_qcb;
    otCB->cb_operation = QUERY;
    prjp = pvp->pcb_project;
    if ( !prjp ) {
	Tcl_SetResult(interp, "no project specified", TCL_VOLATILE);
	return TCL_ERROR;
    }
    if ( argc > 2 ) {
	Tcl_SetResult(interp, "usage: query [next | end | <number>]",
	    TCL_STATIC);
	return TCL_ERROR;
    }
    
    if ( argc == 1 ) {

	if (qcb->qcb_filter) {
	    otFreeMetaTemplate(qcb->qcb_filter);
	    qcb->qcb_filter = 0;
	}
	err = otInitQuery();
	if ( err ) {
	    Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	    return TCL_ERROR;
	}

	/*
	 * Evaluate the tclString, which may have a procedure in its begin
	 * clause that sets historyLines to TRUE.  We have to redirect any
	 * stdout written by the procedure to /dev/null - temporarily.
	 */
	stdoutFd = dup(1);
	stderrFd = dup(2);
	devnull = open("/dev/null", O_RDWR, 0);
	dup2(devnull, 1);
	dup2(devnull, 2);
	close(devnull);

#ifdef SUPPORT_FOR_USER_TCL
	if ( otCB->cb_tclOtrc ) {
	    tclStr = strdup(otCB->cb_tclOtrc);
	    (void)Tcl_Eval(interp, tclStr);
	    free(tclStr);
	}

	if ( otCB->cb_tclFile ) {
	    tclStr = strdup(otCB->cb_tclFile);
	    (void)Tcl_Eval(interp, tclStr);
	    free(tclStr);
	}
#endif

	if ( otCB->cb_tclString ) {
	    pvp->pcb_tclBegin = 1;
	    pvp->pcb_tclEnd = 0;
	    pvp->pcb_template = 0;
	    Tcl_ExprBoolean(pvp->pcb_interp, otCB->cb_tclString, &result);
	    /*
	     * Without the following line we get a core dump.  The nuances
	     * of TCL return values should be better documented.
	     */
	    Tcl_ResetResult(interp); 
	}


	/*
	 * Add code here to call the 'begin' pattern for each of the fields
	 * in otCB->cb_layout which do not appear in the metatemplate.
	 */

	result = 0;
	illegFields[0] = 0;
	dp = strdup(otCB->cb_layout);
	for (cp = strtok(dp, ","); cp; cp = strtok(NULL, ",") ) {
	    char *colp;

	    colp = strchr(cp, ':');
	    if ( colp )
		*colp = 0;
	    if ( !otGetAbbrFieldName(cp, prjp) ) {
		result = Tcl_Eval(interp, cp);
		if ( result != TCL_OK ) {
		    if ( illegFields[0] )
			strcat(illegFields, ", ");
		    strcat(illegFields, cp);
		}
	    }
	}
	free(dp);
		 
	dup2(stdoutFd, 1);
	dup2(stderrFd, 2);
	close(stdoutFd);
	close(stderrFd);

	if (result) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, illegFields, ": no such field", NULL);
	    return TCL_ERROR;
	}

	err = otGenQueryContext(&ctxtString);
	if ( err ) {
	    Tcl_SetResult(interp, otGetPostedMessage(), TCL_VOLATILE);
	    return TCL_ERROR;
	}
	logWarn("otd to otKwikPix: %s\n", ctxtString);

	/*
	 * Insert code here to insure that the historyLines variable has
	 * been set (if necessary).
	 */
	pvp->pcb_kwikPixSock = 0;
	qcb->qcb_nextMatch = 0;

	if ( !qcb->qcb_historyLines ) {
	    /*
	     * Create connection to otKwikPix server.  
	     * Store the socket returned in the private control block.
	     */
	    sargc = 2;
	    sprintf(kwikPath, "%s/%s/socket", prjp->proj_dir, prjp->name);
	    sargv[0] = "connect";
	    sargv[1] = kwikPath;
	    sargv[2] = 0;

	    if ( Tdp_ConnectCmd(notUsed, interp, sargc, sargv) == TCL_OK ) {
		strcpy(connFile, interp->result);
    
		/*
		 * The return value from Tdp_ConnectCmd() has an appended
		 * status field  which we must remove.
		 */
		for (cp = connFile; *cp; cp++)
		    if (*cp == ' ') {
			*cp = 0;
			break;
		    }
		if (Tdp_GetOpenFile(interp, connFile, 1, 0, &filePtr) == TCL_OK) {
		    strcpy(pvp->pcb_kwikPix, connFile);
		    pvp->pcb_kwikPixSock = filePtr;
		}
	    }
	}

	if ( qcb->qcb_historyLines || !pvp->pcb_kwikPixSock ) {
	    kwikPixServer = FALSE;
	    sargc = 3;
	    sargv[0] = "open";
	    sargv[1] = tclOpenCommand;
	    sargv[2] = "r+";
	    sargv[3] = 0;
	    sprintf(tclOpenCommand, "|/etc/otSlowPix -p%s", prjp->name);

	    if ( !Tcl_OpenCmd(notUsed, interp, sargc, (char **)sargv) ) {
		strcpy(pvp->pcb_kwikPix, interp->result);
		if (Tdp_GetOpenFile(interp, interp->result, 1, 0, &filePtr) == TCL_OK)
		    pvp->pcb_kwikPixSock = filePtr;
	    } else {
		interp->result = "connect to storing server failed";
		return TCL_ERROR;
	    }
	    
	}

	if ( ! pvp->pcb_kwikPixSock ) {
	    Tcl_SetResult(interp, "connect to KwikPix failed", TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_ResetResult(interp);

	if ( !(fp = pvp->pcb_kwikPixSock->f2) )
	    fp = pvp->pcb_kwikPixSock->f;
	fprintf(fp, "%s\n", ctxtString);
	fflush(fp);
	free( ctxtString );

	/*
	 * Allocate memory for data read from ot*Pix.  The heuristic is
	 * four times the header file.
	 */
	sprintf(hdrDbfile, "%s/%s/%s", prjp->proj_dir, otCB->cb_project,
	    HEADERDB);
	if ( stat(hdrDbfile, &statbuf) ) {
	    Tcl_SetResult(interp, "stat error on header file", TCL_STATIC);
	    return TCL_ERROR;
	}
	headerBytes = statbuf.st_size * 4;
	
#ifndef NORLIMIT
	if ( status = getrlimit(RLIMIT_DATA, &rl) ) {
	    otPutPostedMessage(OT_SETRLIMIT);
	    return OT_SETRLIMIT;
	}

	logWarn("limit DATA - cur: %ld - max: %ld\n", rl.rlim_cur,rl.rlim_max);
	if ( rl.rlim_cur < (4 * headerBytes) ) {
	    logWarn("rl.rlim_cur < headerBytes\n");
	    rl.rlim_cur = (headerBytes * 4) > rl.rlim_max ? 
		rl.rlim_max : (headerBytes * 4);
	    if ( (status = setrlimit(RLIMIT_DATA, &rl)) < 0 ) {
		otPutPostedMessage(OT_SETRLIMIT);
		Tcl_SetResult(interp, otGetPostedMessage(), TCL_STATIC);
		return OT_SETRLIMIT;
	    }
	}
#endif

	qcb->qcb_kwikPixMatches = malloc( headerBytes );
	if ( !qcb->qcb_kwikPixMatches && headerBytes ) {
	    Tcl_SetResult(interp, "malloc error in kwikQuery", TCL_STATIC);
	    return TCL_ERROR;
	}

	if (!pvp->pcb_quiet && !qcb->qcb_count &&
	    !qcb->qcb_summary && (qcb->qcb_fullText || qcb->qcb_keyInText) ) {
	    /*
	     * Now, just read from otKwikPix or otSlowPix.
	     */
	    fd = fileno(pvp->pcb_kwikPixSock->f);
	    cp = qcb->qcb_kwikPixMatches;
	    *cp = 0;
	    while ( (n = read(fd, cp, KWIKPIXBUF)) > 0 ) {
		cp += n;
		*cp = 0;
		if ( *(cp-1) == '\000' )
		    break;
	    }
    
	    sargc = 2;
	    sargv[0] = "close";
	    sargv[1] = pvp->pcb_kwikPix;
	    sargv[2] = 0;

	    (void)Tcl_CloseCmd(notUsed, interp, sargc,(char **)sargv);
	    pvp->pcb_kwikPixSock = 0;
	}

	/*
	 * If the query requires access of notes, then just get a list
	 * of CR numbers from otKwikPix representing matches, store it.
	 * Note that 'quiet', 'count', 'summary' options require a single
	 * pass through the CR header info, so we don't want any numbers
	 * returned here.
	 */
	if ( !pvp->pcb_quiet && !qcb->qcb_count && !qcb->qcb_summary &&
	    (qcb->qcb_fullText || qcb->qcb_keyInText) ) {

	    if ( qcb->qcb_kwikPixMatches && 
		*qcb->qcb_kwikPixMatches == '\n' ) {
		free(qcb->qcb_kwikPixMatches);
		qcb->qcb_kwikPixMatches = 0;
	    }
	    qcb->qcb_nextMatch = qcb->qcb_kwikPixMatches;
	}

	/*
	 * Otherwise, send the header for a short report.
	 * Alternatively, in the case of no matches for full reports,
	 * emit error message here.
	 */
	if ( !pvp->pcb_quiet && !qcb->qcb_count && !qcb->qcb_summary && 
	     !qcb->qcb_fullText && !qcb->qcb_skipHeader ) {
	    hdrBuf[0] = 0;
	    otCB->cb_pcb->pcb_template = tp;
	    err = otPrintOneLinerToString(tp, prjp->otmp, TRUE, hdrBuf);
	    Tcl_SetResult(interp, hdrBuf, TCL_VOLATILE);
	}

	if ( qcb->qcb_fullText && qcb->qcb_nextMatch &&
	    !*qcb->qcb_nextMatch && !qcb->qcb_summary && !qcb->qcb_count &&
	    !pvp->pcb_quiet ) {

	    /* No match */
	}

    } else if ( argc == 2 ) {
	if        ( !strcmp(argv[1], "next") ) {

	    if ( !qcb->qcb_count && !qcb->qcb_summary && 
		(qcb->qcb_fullText || qcb->qcb_keyInText) ) { /* FULL */

		qcb->qcb_keywordMatchFound = FALSE;

	        /*
		 * In this instance either otKwikPix or otSlowPix have
		 * provided the numbers and they are used to generate a full
		 * report.
		 */
		while ( 1 ) {
		    crStr = 0;
		    err = otQueryNextFull(&crStr);
		    if ( err ) {
			fprintf(stdout, "%s\n", otGetPostedMessage());
		    } else if ( !crStr ) {		
			break;
		    } else {
		        if ( *crStr ) {
			    qcb->qcb_keywordMatchFound = TRUE;
			    fprintf(stdout, "%s", crStr);
			}
			free(crStr);
		    }
		}
		if ( qcb->qcb_keywordMatchFound == FALSE ) {
		    /* No match */
		}
		fflush(stdout);
	    } else {					     /* SHORT */

		if ( pvp->pcb_kwikPixSock ) {
		    /*
		     * If this is non-zero, then we did not succeed in
		     * connecting to otKwikPix and had to settle for otSlowPix
		     * instead.  We want to send back info as we receive it
		     * rather than making the user wait (as in the 'kwikQuery'
		     * command code for storing otKwikPix output.
		     */
		    fd = fileno(pvp->pcb_kwikPixSock->f);
		    while ( (n = read(fd, buf, sizeof(buf))) > 0 ) {
			write(1, buf, n);
			if ( buf[n-1] == '\000' )
			    break;
		    }
		    sargc = 2;
		    sargv[0] = "close";
		    sargv[1] = pvp->pcb_kwikPix;
		    sargv[2] = 0;

		    (void)Tcl_CloseCmd(notUsed, interp, sargc,(char **)sargv);
		    pvp->pcb_kwikPixSock = 0;

		} else {
		    /*
		     * Walk through the data obtained from otKwikPix and
		     * return it to the client.
		     */
		    cp = qcb->qcb_kwikPixMatches;
		    ep = cp + strlen(cp);
		    while ( *cp ) {
		        if ( (cp + KWIKPIXBUF) > ep )  
			    n = write(1, cp, strlen(cp));
			else
			    n = write(1, cp, KWIKPIXBUF);
			cp += n;
		    }
		}
	    }
	}

	write(1, "\000", 1);
	qcb->qcb_nextMatch = 0;
	if ( qcb->qcb_kwikPixMatches ) {
	    free( qcb->qcb_kwikPixMatches );
	    qcb->qcb_kwikPixMatches = 0;
	}

#ifdef notdef
	/*
	 * Final handshake.
	 */
	read(0, buf, 1);
	exit(0);
#endif

    }
    return TCL_OK;


}

OTErr
otQueryNextFull(crStr)
char **crStr;
{
    OTErr err;
    OTPrivateCB *pvp;
    OTQueryCB *qcb;
    OTNote *notep;
    OTProject *prjp;
    int k, hi, lo, crnum;
    char filename[MAXPATHLEN], crNum[SHORTSTR];
    register char *cp, *crNump = crNum;
    struct stat statbuf;
    OTTemplate *tp;

    tp = 0;
    qcb = otCB->cb_qcb;
    err = OT_SUCCESS;
    pvp = otCB->cb_pcb;
    if ( !(prjp = pvp->pcb_project) ) {
	otPutPostedMessage( OT_NEED_PROJECT );
	return OT_NEED_PROJECT;
    }

    /*
     * For next CR in otCB->cb_qcb->qcb_kwikPixMatches, obtain CR.
     * Allocate memory and return in crStr.
     */
    if ( !(cp = qcb->qcb_nextMatch) )
	return OT_SUCCESS;

    if ( ! (*cp) ) {
        *crStr = 0;
    } else {
	while( isdigit(*cp) )
	    *crNump++ = *cp++;
	*crNump = 0;
	while( *cp && !isdigit(*cp) )
	    cp++;
	qcb->qcb_nextMatch = cp;

	crnum = atoi(crNum);
/*
	hi = crnum / 10000;
	lo = crnum / 100;
*/
	k = crnum / 100;
	hi = k / 100;
	lo = k - hi*100;

	sprintf(filename, "%s/%s/d%02d/d%02d/c%06d", prjp->proj_dir,
	    prjp->name,	hi, lo, crnum);

	/*
	 * For performance, allocate size of current CR file w/ fudge factor
	 * since full report is same size as original CR plus mailname
	 * expansion.
	 */
	if (stat(filename, &statbuf) != 0) {
	    otPutPostedMessage(OT_OBJECT_NOT_FOUND, prjp->object_name, crnum);
	    return OT_OBJECT_NOT_FOUND;
	}
	if ( !(*crStr = malloc( statbuf.st_size * 2 )) ) {
	    otPutPostedMessage( OT_MALLOC_LOCATION, "otQueryNextFull()" );
	    return OT_MALLOC_LOCATION;
	}
	**crStr = 0;

	err = otReadTemplateFromFilename( &tp, filename, TRUE, TRUE );
	if ( err == OT_TEMPLATE_READ )
	    return OT_SUCCESS;

	if ( qcb->qcb_keyInText ) {
	    if ( tp->notep ) {

		pvp->pcb_CRmatch = 0;
		for (notep = tp->notep; !(pvp->pcb_CRmatch) && notep->who &&
		    *notep->who; notep++) {

		    if (qcb->qcb_nsens && strcmp(qcb->qcb_nsens, notep->sens))
			continue;
		    if ( strstr(notep->text, qcb->qcb_keyInText) ) 
			pvp->pcb_CRmatch = 1;  
		}

		if (pvp->pcb_CRmatch) {
		    otCB->cb_pcb->pcb_template = tp;
		    if ( !pvp->pcb_quiet ) {
			if ( qcb->qcb_fullText )
			    err = otPrintFullReportToString(tp, *crStr);
			else
			    err = otPrintOneLinerToString(tp, prjp->otmp,
				FALSE, *crStr);
		    }
		}

	    }
	} else {
	    if ( !OT_WARN(err) && !OT_SYSTEM(err) ) {
		otCB->cb_pcb->pcb_template = tp;
		
		if (!pvp->pcb_quiet) {
		    if ( qcb->qcb_fullText )
/* HERE */		err = otPrintFullReportToString(tp, *crStr);
		    else
			err = otPrintOneLinerToString(tp, prjp->otmp, FALSE,
			    *crStr);
		}

	    }
	}
    }

    if ( tp )
	otFreeTemplate(tp);
    return err;

}


OTErr
otQueryOneCR(crnum, crStr)
int  crnum;
char **crStr;
{
    OTErr err;
    OTPrivateCB *pvp;
    OTQueryCB *qcb;
    OTProject *prjp;
    int k, hi, lo;
    char filename[MAXPATHLEN];
    struct stat statbuf;
    OTTemplate *tp;

    tp = 0;
    qcb = otCB->cb_qcb;
    err = OT_SUCCESS;
    pvp = otCB->cb_pcb;
    if ( !(prjp = pvp->pcb_project) ) {
	otPutPostedMessage( OT_NEED_PROJECT );
	return OT_NEED_PROJECT;
    }

    /*
     * Allocate memory and return in crStr.
     */
/*
    hi = crnum / 10000;
    lo = crnum / 100;
*/
    k = crnum / 100;
    hi = k / 100;
    lo = k - hi*100;


    sprintf(filename, "%s/%s/d%02d/d%02d/c%06d", prjp->proj_dir,
        prjp->name, hi, lo, crnum);

    /*
     * For performance, allocate size of current CR file w/ fudge factor
     * since full report is same size as original CR plus mailname
     * expansion.
     */
    if (stat(filename, &statbuf) != 0) {
        otPutPostedMessage(OT_OBJECT_NOT_FOUND, prjp->object_name, crnum);
        return OT_OBJECT_NOT_FOUND;
    }
    if ( !(*crStr = malloc( statbuf.st_size * 2 )) ) {
        otPutPostedMessage( OT_MALLOC_LOCATION, "otQueryNextFull()" );
        return OT_MALLOC_LOCATION;
    }
    **crStr = 0;

    err = otReadTemplateFromFilename( &tp, filename, TRUE, TRUE );
    if ( err == OT_TEMPLATE_READ )
        return OT_SUCCESS;

    if ( !OT_WARN(err) && !OT_SYSTEM(err) ) {
	otCB->cb_pcb->pcb_template = tp;
	if ( qcb->qcb_fullText )
	    err = otPrintFullReportToString(tp, *crStr);
	else
	    err = otPrintOneLinerToString(tp, prjp->otmp, FALSE, *crStr);
    }

    if ( tp )
	otFreeTemplate(tp);
    return err;

}

