/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */

/*
 * otTemplate.c
 */


/*
#include <stdlib.h>
*/
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"

#ifndef LINT
static char RCSid_otTemplate[] =
    "$RCSfile: otTemplate.c,v $ $Revision: 1.1.7.3 $ $Date: 1994/01/19 18:10:32 $";

#endif

OTErr  otReadTemplateFromFilename();
OTErr  otReadTemplateFromFile();
OTErr  otReadHeaderFromFile();
OTErr  otReadNotes();
OTErr  readByline();
OTErr  otAppendHeaderFieldToTemplateFromString();
OTErr  otAppendNoteToTemplateFromString();
OTErr  otAppendNotedataToTemplateFromString();

OTErr  otWriteTemplateToFilename();
OTErr  otWriteTemplateToFile();

OTErr  otWriteHeaderFieldToString();
OTErr  otWriteNotedataToString();

OTErr  otTemplateChanged();
OTErr  otNotesCmp();
OTErr  otNotedataCmp();

OTErr  otInitTemplate();
void   fixStr();
OTErr  otCollateTemplate();

void   rinse();
void   rinseTemplate();

char  *otGetHeaderFieldValue();
OTErr  otSetHeaderFieldValue();
OTErr  otSetIDNumValue();
long   otGetIDNumValue();
OTErr  getOptString();
void   getTypeOptDeg();
char  *otGetAbbrFieldName();
otType otGetAbbrFieldType();
char  *otGetFieldNameAbbr();


OTErr  otMallocTemplate();
void   otFreeTemplate();
void   freeNotes();
void   freeNotedata();
OTErr  otDupTemplate();




/*
 *            o t R e a d T e m p l a t e F r o m F i l e n a m e
 */
OTErr
otReadTemplateFromFilename( tStruct, filename, saveNote, saveNotedata)
OTTemplate **tStruct;
char *filename;
bool saveNote;
bool saveNotedata;
{
    FILE *fp;
    OTErr readErr;

    if ( !(filename && *filename && (fp = fopen( filename, "r" ))) ) {
	otPutPostedMessage( OT_TEMPLATE_READ, 
	    (filename || *filename) ? filename : "null" );
	return OT_TEMPLATE_READ;
    }

    if ( readErr = otReadTemplateFromFile( tStruct, fp, saveNote,
	saveNotedata ) )
        return readErr;

    fclose(fp);
    return OT_SUCCESS;

}

/*
 *                o t R e a d T e m p l a t e F r o m F i l e
 */
OTErr
otReadTemplateFromFile( tpp, fp, saveNote, saveNotedata)
OTTemplate **tpp;
FILE *fp;
bool saveNote;
bool saveNotedata;
{
    OTErr headerErr, mallocErr, noteErr;

    if ( mallocErr = otMallocTemplate( tpp ) )
        return mallocErr;

    if ( (headerErr = otReadHeaderFromFile(*tpp, fp)) != OT_SUCCESS ) {
	switch ( headerErr ) {
	    case OT_TEMPLATE_LONG:
	    case OT_MALLOC_LOCATION:
	    case OT_TRANSMISSION_CONCLUDED: 
		otFreeTemplate(*tpp);
		return headerErr;
	    default:
		/* no default action */
		;
	}
    }

    if ( (noteErr = otReadNotes(*tpp, fp, saveNote, saveNotedata)) != OT_SUCCESS ) {
	switch ( noteErr ) {
	    case OT_TEMPLATE_LONG:
	    case OT_MALLOC_LOCATION:
		otFreeTemplate(*tpp);
		return noteErr;
	    case OT_TRANSMISSION_CONCLUDED: 
	    default:
		/* no default action */
		;
	}
    }

    return headerErr;

}


/*
 *                o t R e a d H e a d e r F r o m F i l e 
 *
 * OT_TRANSMISSION_CONCLUDED - Only return this if 'nothing' has been
 *			       read.  Otherwise return OT_SUCCESS.
 * OT_SUCCESS		     - Found a note or the CRSEPARATOR.
 *			       Return OT_SUCCESS
 * OT_TEMPLATE_LONG	     - Number of header lines exceeds limit.
 * OT_MALLOC_LOCATION	     - Return OT_MALLOC_LOCATION.
 */
OTErr
otReadHeaderFromFile(tp, fp)
OTTemplate *tp;
FILE *fp;
{
    register OTHeaderField *hfp;
    OTHeaderField *start;
    OTErr headerErr   = OT_SUCCESS;
    OTErr appendErr   = OT_SUCCESS;
    char *cp;

    if ( otCB->cb_pcb->pcb_nopened && otCB->cb_pcb->pcb_xmitEnd == TRUE )
        return OT_TRANSMISSION_CONCLUDED;

    if ( headerErr = otReadStringFromFileHandle(&cp, fp, 0, "\001[", TRUE) )
	switch ( headerErr ) {
	    case OT_TRANSMISSION_CONCLUDED:
		/*
		 * Return OT_TRANSMISSION_CONCLUDED but only if 
		 * otAppendHeaderFieldToTemplateFromString() does not make 
		 * tp bigger.
		 */
		break;		
	    case OT_MALLOC_LOCATION:
	    case OT_INTERNAL_ERROR:
		return headerErr;

	    default:
		/* no default action */
	      	;
	}

    DBUG_MAX((stderr, "otReadStringFromFileHandle: %s\n", cp));

    if ( appendErr = otAppendHeaderFieldToTemplateFromString(tp, cp) ) {
	switch ( appendErr ) {
	    case OT_TEMPLATE_LONG:
	    case OT_MALLOC_LOCATION:
		{ if ( *cp )
		      (void)free(cp);
		  return appendErr;
		  break;
		}
	    default:
		break;
	}
    }

    if ( cp )
	(void)free(cp);

#ifdef notdef
/*
 * This code causes errors because, in the case of CRs w/out notes, it
 * reads the CR separator before it passes control to otReadNotes().
 */

    if ( headerErr != OT_TRANSMISSION_CONCLUDED ) {
        ch = getc(fp);
	if ( (ch == EOF) || !ch ) {
	    headerErr = OT_TRANSMISSION_CONCLUDED;
	} else if (ch == '\001') {
	    do {
		ch = getc(fp);
	    } while ( (ch != EOF) && ch && (ch != '\n') );

	    if ( (ch == EOF) || !ch )
		headerErr = OT_TRANSMISSION_CONCLUDED;
	} else {
	    if (ungetc(ch, fp) != ch) {
		otPutPostedMessage(OT_INTERNAL_ERROR, "ungetc()");
		headerErr = OT_INTERNAL_ERROR;
	    }
	}
    }
#endif

    start = tp->tr;
    for ( hfp = start; hfp->field && *hfp->field; hfp++);

    if ( headerErr == OT_TRANSMISSION_CONCLUDED ) {

	if (hfp == start) {
	    return OT_TRANSMISSION_CONCLUDED;
	} else {
	    if ( otCB->cb_pcb->pcb_nopened )
		otCB->cb_pcb->pcb_xmitEnd = TRUE;
	    else 
	        return OT_SUCCESS;
	}

    } else 
	return headerErr;

}


/*
 *                         o t R e a d N o t e s
 *
 * OT_TRANSMISSION_CONCLUDED - Only return this if 'nothing' has been
 *			       read.  Otherwise return OT_SUCCESS.
 * OT_SUCCESS		     - Found a note or the CRSEPARATOR.
 *			       Return OT_SUCCESS
 * OT_NOTES_TOOMANY	     - Number of notes exceeds limit.
 * OT_MALLOC_LOCATION	     - Return OT_MALLOC_LOCATION.
 */
OTErr
otReadNotes(tp, fp, saveNote, saveNotedata)
OTTemplate *tp;
FILE *fp;
bool saveNote;
bool saveNotedata;
{
    register int ch;
    char *cp;
    OTErr noteErr, appendErr;

    if ( otCB->cb_pcb->pcb_nopened && otCB->cb_pcb->pcb_xmitEnd == TRUE )
        return OT_TRANSMISSION_CONCLUDED;

    if ( noteErr = otReadStringFromFileHandle(&cp, fp, 0, "\001", TRUE) ) {
	switch ( noteErr ) {
	    case OT_TRANSMISSION_CONCLUDED:
	  	if ( strlen(cp) != 0 )
		    break;
	    case OT_MALLOC_LOCATION:
	    case OT_INTERNAL_ERROR:
		(void)free(cp);
		return noteErr;

	    default:
		/* no default action */
		;
	}
    }

    DBUG_MAX((stderr, "otReadStringFromFileHandle: %s\n", cp));
    if ( appendErr = otAppendNoteToTemplateFromString(tp, cp, saveNote, saveNotedata) ) {
	switch ( appendErr ) {
	    case OT_TEMPLATE_LONG:
	    case OT_MALLOC_LOCATION:
		{ if ( cp )
		    (void)free(cp);
		  return appendErr;
		  break;
		}
	    default:
		break;
	}
    }

    if ( cp )    
	(void)free(cp);

    if ( !otCB->cb_pcb->pcb_xmitEnd ) {
	ch = getc(fp);
	if ( (ch == EOF) || !ch ) {
	    noteErr = OT_TRANSMISSION_CONCLUDED;
	} else if (ch == '\001') {
	    do {
		ch = getc(fp);
	    } while ( (ch != EOF) && ch && (ch != '\n') );

	    if ( (ch == EOF) || !ch )
		noteErr = OT_TRANSMISSION_CONCLUDED;
	} else {
	    if (ungetc(ch, fp) != ch) {
		otPutPostedMessage(OT_INTERNAL_ERROR, "ungetc()");
		noteErr = OT_INTERNAL_ERROR;
	    }
	}
    }

    return noteErr;


}


/*
 *                          r e a d B y l i n e
 *
 * Reads line of form
 *
 *   [pnh 03/19/92 public]
 *
 */
OTErr readByline( bp, wp, dp, sp)
char *bp;
char *wp;
char *dp;
char *sp;
{
    OTErr err;
    OTDateComparison dc;
    char *datep = dp;

    err = OT_SUCCESS;
    DBUG_MED((stderr, "readByline start |%s|\n", bp));
    if ( bp && *bp == '[') {

        bp++;
        while ( isspace(*bp) && (*bp != '\n') )
	    bp++;

	while ( *bp && !isspace(*bp) && *bp != '\n' && *bp != ']' )
	    *wp++ = *bp++;
	*wp = 0;

	while (isspace(*bp) && (*bp != '\n'))
	    bp++;

	while (*bp && !isspace(*bp) && *bp != ']' && *bp != '\n')
	    *dp++ = *bp++;
	*dp = 0;

	if ( otParseDate(datep, &dc) )
	    err = OT_INVALID_BYLINE;

	while (isspace(*bp) && *bp != '\n')
	    bp++;

	while (*bp && !isspace(*bp) && *bp != ']' && *bp != '\n') 
	    *sp++ = *bp++;
	*sp = 0;

    }
    DBUG_MED((stderr, "readByline end\n"));
    return err;

}



OTErr
otAppendHeaderFieldToTemplateFromString(tp, cp)
OTTemplate *tp;
char *cp;
{
    bool endOfString;
    OTErr headerErr;
    OTHeaderField *start;
    register char *bolp, *t;
    register OTHeaderField *hfp;

    endOfString = FALSE;
    /*
     * Move tp to next free header field slot in template.
     */
    start = tp->tr;
    for ( hfp = start; hfp->field && *hfp->field; hfp++) {

	if ( hfp >= start + MAXTEMPLATE ) {
	    otPutPostedMessage( OT_TEMPLATE_LONG );
	    headerErr = OT_TEMPLATE_LONG;
	    continue;
	}
    }

    headerErr = OT_SUCCESS;
    while ( !endOfString && !headerErr) {

        if ( hfp >= tp->tr + MAXTEMPLATE ) {
	    otPutPostedMessage( OT_TEMPLATE_LONG );
	    headerErr = OT_TEMPLATE_LONG;
	    continue;
	}

	bolp = cp;
	if (*cp == '#') {
	    if ( !(cp = strchr(cp, '\n')) )
		endOfString = TRUE;
	    else
		cp++;
	    continue;
	}

	if (*cp == '\n') {
	    cp++;
	    continue;
	}

	if ( hfp->field == 0 ) {
	    if ( !(hfp->field = calloc(NAMELEN, sizeof(char))) ) {
		otPutPostedMessage( OT_MALLOC_LOCATION, "otAppendHeaderFieldToTemplateFromString");
		headerErr = OT_MALLOC_LOCATION;
	    }
	}

	for ( t = hfp->field; *cp && (*cp != '\n') && (*cp != ':') ; )
	    if ( (cp - bolp) >= NAMELEN ) {
		*t = 0;
		cp++;
	    } else
		*t++ = *cp++;

	if ( headerErr )
	    continue;

	*t = 0;
	if ( !*cp )
	    endOfString = TRUE;

	cp--;
	if ( (*cp != ' ' && *cp != ')' && *cp != ']' && *cp != '+') ||
	    (*(cp+1) == '\000') || (*(cp+1) == '\n') ) {
	    /*
	     * A line contains a header label, beginning a new field, when 
	     * it contains a colon UNLESS the colon is NOT immediately
	     * preceded by a delimiter character or space, where a
	     * delimiter character is one of ] ) +
	     */
	    if ( hfp > start ) {

		hfp--;
		if ( !(cp = strchr(cp, '\n')) )
		    endOfString = TRUE;
		else 
		    *cp++ = 0;

	        if ( otAppendString( "\n", &hfp->value ) ||
	             otAppendString( bolp, &hfp->value ) ) {
		    otPutPostedMessage(OT_MALLOC_LOCATION,
			"otAppendHeaderFieldToTemplateFromString()");
		    headerErr = OT_MALLOC_LOCATION;
		    continue;
		}

		hfp++;
		memset(hfp->field, 0, sizeof(hfp->field));
		continue;
	    }
        }
	cp++;
	
	getTypeOptDeg( hfp, cp - bolp );

	if (*cp == ':') {
	    char *eov;

	    cp++;
	    while ( isspace(*cp) && (*cp != '\n') )
		cp++;

	    if ( !(eov = strchr(cp, '\n') ) )
		endOfString = TRUE;
	    else
		*eov = 0;

	    if (*cp) {
		if ( otCopyString( cp, &(hfp->value) ) ) {
		    otPutPostedMessage( OT_MALLOC_LOCATION,
		        "otAppendHeaderFieldToTemplateFromString()" );
		    headerErr = OT_MALLOC_LOCATION;
		    continue;
		}
	    } else {
		hfp->value = 0;
	      }
	    cp = eov + 1;
	}

	DBUG_MED((stderr, "readTemplate: field %s type %d list %s\n", hfp->field, hfp->type, hfp->list));
	DBUG_MED((stderr, "\toptionality %d degree %d\n", hfp->optionality, hfp->degree));
	DBUG_MED((stderr, "\tvalue = %s\n", hfp->value));
	DBUG_MED((stderr, "Appended value |%s|\n", bolp));
	hfp++;
    }

    for (hfp = tp->tr; hfp->field && *hfp->field; hfp++)
       if (hfp->value) {
	   otTrimWs(hfp->value);
           if ( *hfp->value == 0 ) {
	       free(hfp->value);
	       hfp->value = 0;
	   }
	}

    return headerErr;

}


OTErr
otAppendNoteToTemplateFromString(tp, cp, saveNote, saveNotedata)
OTTemplate *tp;
char *cp;
bool saveNote;
bool saveNotedata;
{
    register OTNote *startnp, *np;
    register int i;
    register char *bolp, *startstr, *tmpp;
    bool endOfString;
    OTErr dateErr, noteErr, ndErr;
    long noteLength, usedNoteLength;
    char value[LONGVALUE];
    char name[LONGVALUE], date[LONGVALUE], sens[LONGVALUE];

    usedNoteLength = 0;
    startstr = cp;
    startnp = tp->notep;
    for( np = startnp; (np+1)->text; np++ )
	if ( np >= tp->notep + MAXNOTES ) {
	    otPutPostedMessage( OT_NOTES_TOOMANY );	    
	    noteErr = OT_NOTES_TOOMANY;
	    continue;
	}

    noteErr = OT_SUCCESS;
    endOfString = FALSE;
    memset(value, 0, LONGVALUE);

    while (!endOfString && !noteErr) {

	if ( np >= tp->notep + MAXNOTES ) {
	    otPutPostedMessage( OT_NOTES_TOOMANY );	    
	    noteErr = OT_NOTES_TOOMANY;
	    continue;
	}

	bolp = cp;
	
	if ( !(cp = strchr(cp, '\n')) )
	    endOfString = TRUE;
	else {
	    *cp++ = 0;
#ifdef notdef
	    if (bolp == cp) {
		cp++;
		continue;
	    }
	    else
	    *cp++ = 0;
#endif
	}
	DBUG_MED((stderr, "|%s| (%c/%X) is bolp before call\n", bolp, *bolp, bolp));

	if ( (*bolp == '+') && *(bolp+1) &&
	     ( isdigit(*(bolp+1)) || (isupper(*(bolp+1))) ) ) {

	    /* 
	     * Later add the condition
	     *
	     * && isdigit( *(bolp + 1) )  ) {
	     *
	     * when all CRs have been converted to have a digit with the
	     * notedata
	     */

	    if ( tmpp = bolp ) {
		*(cp-1) = '\n';
		do {
		    if (*tmpp == '[')
			break;
		    if ( tmpp = strchr(tmpp, '\n') )
			tmpp++;
		    else
			tmpp += strlen(tmpp);
		} while (*tmpp);
		if (!*tmpp)
		    endOfString = TRUE;
		if (*tmpp == '[')
		    *(tmpp-1) = 0;
		if (cp != tmpp) {
		    cp = tmpp;
		    otTrimWs(bolp);
		}
	    }

	    /*
	     * Change this.  Read down to eos or '[' in the first column.
	     */
	    if ( ndErr = otAppendNotedataToTemplateFromString(tp, bolp) )
		return ndErr;	
	    else
		continue; 

	} else if ( *bolp == '[' ) {
	    /*
	     * This line is an author line if there is a date after a name.
	     * The date search, [0-9/]*, is heuristic.  
	     */
	    if ( dateErr = readByline(bolp, name, date, sens) ) {
		if ( (np == tp->notep) && !(tp->notep->text) ) {
		    /*
		     * If the first dateline in a note is "bad", we still need
		     * to create a fake one so we can allocate space for it and
		     * fill the header line areas.
		     */
		    if ( !(np->text = (char *)malloc(NOTEBLOCKSIZE)) ) {
			otPutPostedMessage(OT_MALLOC_LOCATION,
			    "otReadTemplate()");
			noteErr = OT_MALLOC_LOCATION;
			continue;
		    }
		    noteLength = NOTEBLOCKSIZE;
		    usedNoteLength = 0;

		    (void)strcpy(np->who, "firstnote");
		    (void)strcpy(np->date, "01/01/93");
		    (void)strcpy(np->sens, "public");

		    cp = bolp;
		    bolp = "[firstnote 01/01/93 public]";
		}
	    } else {
	        if (saveNote) {

		    if ( np->text )
			np++;

		    if ( !(np->text = (char *)malloc(NOTEBLOCKSIZE)) ) {
			otPutPostedMessage(OT_MALLOC_LOCATION,
			    "otReadTemplate()");
			noteErr = OT_MALLOC_LOCATION;
			continue;
		    }
		    noteLength = NOTEBLOCKSIZE;
		    usedNoteLength = 0;
		    (void)strcpy(np->who, name);
		    (void)strcpy(np->date, date);
		    (void)strcpy(np->sens, sens);

		    DBUG_MED((stderr, "otAppendNoteToTemplateFromString: got new note, who '%s' date '%s' sens '%s'\n", np->who, np->date, np->sens));
		}
	    }
	}

	if (saveNote)	{
	    /*
	     * This is a line to append to note.
	     */
	    i = usedNoteLength + strlen(bolp) + 2;
	    while ( (i > noteLength) && (noteErr == OT_SUCCESS) ) {
		noteLength += NOTEBLOCKSIZE;	/* another block */
		DBUG_MIN((stderr, "otAppendNoteToTemplateFromString: realloc'ing to %d target %d\n", noteLength, i));
		if ( !(np->text = (char *)realloc(np->text, noteLength)) ) {
		    otPutPostedMessage(OT_MALLOC_LOCATION, "otAppendNoteToTemplateFromString()");
		    noteErr = OT_MALLOC_LOCATION;
		}
	    }
	    if ( noteErr == OT_SUCCESS ) {
		strcpy(&(np->text[usedNoteLength]), bolp);
		strcat(&(np->text[usedNoteLength]), "\n");
		usedNoteLength = i - 1;
	    }
	}

    } /* while */

    for(np=tp->notep; *np->who; np++)
	otTrimWs(np->text);

    return noteErr;



}


/*
 * otAppendNotedataToTemplateFromString accepts strings of the following 
 * form:
 *
 * +0HISTORY Wed Jul 15 12:30:06 1992 uar	added/changed [fred 3/10/92 public]
 * +0HISTORY Mon Dec 14 15:53:06 1992 uar	prior: 2 -> 4
 * +0HISTORY Wed Sep 23 11:30:51 1992 uar   Transarc Status: export -> impor 
 * +1BSUBMIT Mon Jan 25 11:25:36 1993 uar
 * +2TARCNUM Wed Sep 23 11:30:51 1992 uar   2233
 *
 * This expects the last character of the string to be null, not preceded
 * by newline.
 */
OTErr
otAppendNotedataToTemplateFromString(tp, cp)
OTTemplate *tp;
char *cp;
{
    register char *bolp, *t;
    register int i;
    register OTNotedata *tndp;
    OTErr ndErr;
    char *startstr, *fnp, *curp;
    int yr, tmpday;
    char tmpweekday[NAMELEN], tmpmon[NAMELEN];
    char tmptime[MAXTIME+1], tmpyear[NAMELEN], tmpkwd[NAMELEN];
    char tmpagent[NAMELEN];
    bool endOfString;

    startstr = cp;
    endOfString = FALSE;
    ndErr = OT_SUCCESS;
    /*
     * Append after last of the current notedata lines.
     */
    for( tndp = tp->ndp; !ndErr && tndp->nd_flags; tndp++ )
	if ( tndp >= tp->ndp + MAXNDATA )
	    ndErr = OT_NDATA_TOOMANY;

    while (!endOfString && !ndErr) {

	/*
	 * Make bolp point to beginning of current line.
	 */
	bolp = cp;

	/*
	 * The string argument to this function may be one or more lines of
	 * notedata information.  Null-terminate the current line.
	 */

        if ( !(cp = strchr(cp, '\n')) )
	    endOfString = TRUE;
	else
	    *cp++ = 0;

	if ( *bolp != '+' ) {
	    /*
	     * HISTORY lines may cross over multiple lines if the previous 
	     * or current values of the fields changed were 'multi-line' 
	     * (i.e., contained newlines).  If the first character of the
	     * line is not '+' and the HISTORY bit is set, then the current
	     * input line is a continuation either of the field value before
	     * the change or after the change.
	     */

	    otTrimWs(bolp);
	    if ( !*bolp )
		continue;
	    if ( (tndp->nd_current == 0) && ND_ISHISTORY( tndp->nd_flags ) ) {
		if ( ndErr = otAppendString( bolp, &(tndp->nd_previous) ) )
		    continue;
	    } else if ( ndErr = otAppendString( bolp, &(tndp->nd_current) ) )
		continue;
	} else {
	    /*
	     * If this is not the first notedata line to be parsed, then
	     * increment the notedata pointer to the next available slot.
	     */
	    if ( startstr != bolp )
		tndp++;

	    /*
	     * Point beyond the '+' character in column 0.
	     */
	    t = bolp + 1;

	    if ( sscanf(t, "%s %3s %3s %d %8s %4s %99s", tmpkwd, tmpweekday,
		tmpmon,	&tmpday, tmptime, tmpyear, tmpagent) == 7 ) {

	        /*
		 * The first character of tmpkwd should now be a flag character
		 * ('0', '1' or '2').  If not, then we assume for now that 
		 * it's an old HISTORY line.
		 */
		tndp->nd_kwd = &(tmpkwd[1]);
		if ( isdigit(tmpkwd[0]) ) {
		    switch (tmpkwd[0]) {
			/*
			 * 0 - a reserved word --> always visible
			 * 1 - user-contributed and visible in reports
			 * 2 - user-contributed but invisible
			 */
			case '0':  tndp->nd_flags |= ND_RESV;
				   break;
			case '2':  tndp->nd_flags |= ND_USER;
				   break;
			case '1':
			default:   tndp->nd_flags |= ND_VISB | ND_USER;
				   break;
		    }
		}
		else {
		    /*
		     * This is here to handle the conversion from +HISTORY to
		     * +0HISTORY.  We need to be able to read the old lines.
		     */
		    tndp->nd_flags = ND_RESV | ND_HIST;
		}

		/*
		 * Mark bits for the reserved words (no need to save word if
		 * it's reserved).
		 */
		if ( ND_ISRESERVED( tndp->nd_flags ) ) {
		    if (        !strcmp(ND_HISTORY_KWD, tndp->nd_kwd) ) {
			tndp->nd_flags |= ND_HIST;
		    } else if ( !strcmp(ND_LUPDATE_KWD, tndp->nd_kwd) ) {
			tndp->nd_flags |= ND_LUPD;
		    } else {
		      /*
		       * Fix this when all CRs have converted HISTORY lines.
		       *
		       * otPutPostedMessage(OT_NOT_RESERVED_NDATA,
		       *     tndp->nd_kwd);
		       * ndErr = OT_NOT_RESERVED_NDATA;
		       * continue;
		       */
		    }
		    tndp->nd_kwd = 0;
		} else {
		    int len;

		    tndp->nd_kwd = 0;
    		    if ( (len = strlen(&tmpkwd[1])) > MAXKWD ) {
			otPutPostedMessage(OT_NDKEYWORD_LONG, tmpkwd);
			ndErr = OT_NDKEYWORD_LONG;
			continue;
		    } else if (ndErr = otCopyString(&(tmpkwd[1]),
			&(tndp->nd_kwd))) {
		        otPutPostedMessage(OT_MALLOC_LOCATION,
			    "otAppendNotedataToTemplateFromString()");
			continue;			
		    }
		}

		if ( ndErr = otCopyString(tmpagent, &(tndp->nd_agent)) ) {
		    otPutPostedMessage(OT_MALLOC_LOCATION,
			"otAppendNotedataToTemplateFromString()");	       
		    continue;
		}

		tndp->nd_weekday = dayNumberForName(tmpweekday);
		/*
		 * Store mon, day, year and time.
		 */
		tndp->nd_mon = monthNumberForName(tmpmon);
		tndp->nd_day = tmpday;

		yr = atoi(&(tmpyear[2]));
		yr = (yr > 1900) ? (yr - 1900) : yr;
		tndp->nd_year = yr;
		
		memcpy(tndp->nd_time, tmptime, MAXTIME);

		/* 
		 * Skip past at least 7 'words':
		 *      1     2   3  4     5       6   7
		 * +0HISTORY Thu Oct 29 14:17:58 1992 tmt	stat:
		 *
		 * then move past spaces to fieldname (or text).  If there
		 * is no text (as in e.g. LUPDATE) then continue.
		 */

		for( i = 0; *t && i < 7; t++)
		    if ( isspace(*t) && !isspace( *(t-1) ) )
			i++;
		while ( *t && isspace(*t) )
		    t++;
		if ( !*t )
		    continue;

		/*
		 * If this is a HISTORY line, then try to fill fields
		 * "nd_field" and "nd_previous".
		 */
		if ( ND_ISHISTORY( tndp->nd_flags ) ) {
		    tndp->nd_field = 0;

		    if (!strncmp(t, "added", 5) || !strncmp(t, "Created", 7)) {

		        if ( ndErr = otCopyString(t, &(tndp->nd_field)) ) {
			   otPutPostedMessage(OT_MALLOC_LOCATION, 
				"otAppendNotedataToTemplateFromString()");
			   continue;
			}
			if (tndp >= tp->ndp + MAXNDATA) 
			    ndErr = OT_NDATA_TOOMANY;
			continue;
		    } else if ( fnp = strchr(t, ':') ) {
			*fnp++ = 0;

			if ( ndErr = otCopyString(t, &(tndp->nd_field)) ) {
			    otPutPostedMessage(OT_MALLOC_LOCATION,
				"otAppendNotedataToTemplateFromString()");
			    continue;
			}
			otTrimWs(tndp->nd_field);
			while ( *fnp && isspace(*fnp) )
			    fnp++;
			t = fnp;

			if ( curp = strstr(t, "->") ) {
			    *curp++ = 0;
			    *curp++;
			    tndp->nd_previous = 0;
			    if ( !strncmp(t, "NULL", 4) ) {

			    } else if (ndErr = otAppendString(t, &(tndp->nd_previous))){
				otPutPostedMessage(OT_MALLOC_LOCATION,
				    "otAppendNotedataToTemplateFromString()");
				continue;
			    }
			    if ( tndp->nd_previous )
				otTrimWs(tndp->nd_previous);

			    while ( *curp && isspace(*curp) )
				curp++;
			    t = curp;
			}
		    }
		}

		/*
		 * HISTORY lines save their 'after' (as in 'before and after')
		 * value in the nd_current member.  All other notedata lines
		 * (such as 'BSUBMIT') save all their data in nd_current.
		 */
		if ( !ndErr ) {
		    if ( !strncmp(t, "NULL", 4) ) {
			tndp->nd_current = 0;
		    } else if (ndErr = otAppendString(t, &(tndp->nd_current))) {
			otPutPostedMessage(OT_MALLOC_LOCATION,
			    "otAppendNotedataToTemplateFromString()");
			continue;
		    }
		    if ( tndp->nd_current )
			otTrimWs(tndp->nd_current);
		}
	    }
	}
	if (tndp >= tp->ndp + MAXNDATA) 
	    ndErr = OT_NDATA_TOOMANY;
    }
    DBUG_MED((stderr, "otAppendNotedata.."));

    DBUG_MED((stderr, "otApp kwd = %s\tflags = %o\n", tndp->nd_kwd ? tndp->nd_kwd : "null" , tndp->nd_flags));
    DBUG_MED((stderr, "otApp weekday = |%o|\tfield = |%s| \n", tndp->nd_weekday, tndp->nd_field ? tndp->nd_field : ""));
    DBUG_MED((stderr, "otApp month = |%o|\t day = |%o| year = |%o|\n", tndp->nd_mon, tndp->nd_day, tndp->nd_year));

    DBUG_MED((stderr, "otApp agent = %s\n", tndp->nd_agent));
    DBUG_MED((stderr, "otApp previous: |%s|\tcurrent: |%s|\n", tndp->nd_previous ? tndp->nd_previous : "", tndp->nd_current ? tndp->nd_current : "" ));

    return ndErr;

}




/*
 *                 o t W r i t e T e m p l a t e T o F i l e n a m e
 */
OTErr
otWriteTemplateToFilename(tStruct, filename)
OTTemplate  * tStruct;	/* template definition structure */
char        * filename;	/* output file name */
{
    FILE *fp;
    OTErr wrtErr;

    /*
     * Open the new output template 
     */
    if ((fp = fopen(filename, "w")) == NULL) {
	otPutPostedMessage( OT_TEMPLATE_WRITE, filename );
	return OT_TEMPLATE_WRITE;
    }
    wrtErr = otWriteTemplateToFile(tStruct, fp);
    (void)fclose(fp);
    return wrtErr;

}


/*
 *                  o t W r i t e T e m p l a t e T o F i l e
 */

OTErr
otWriteTemplateToFile(tp, fp)
OTTemplate * tp;	/* template definition structure */
FILE       * fp;	/* output file name */
{
    register OTMetaTemplate *mtp;
    register OTHeaderField *hfp;
    register OTNote *np;
    register OTNotedata *tndp, *ndp;
    char *cp, *kwd;
    OTErr malErr;
    OTHeaderField hf;
    struct stat statBuf;
    OTProject * pjp;	/* project definition structure */

    pjp = otCB->cb_pcb->pcb_project;
    mtp = pjp->otmp;

    rinseTemplate(tp);
    DBUG_MED((stderr, "otWriteTemplateToFile for proj %s\n", pjp->name));

    if ( fstat( fileno(fp), &statBuf ) != 0 ) {
	otPutPostedMessage(OT_TEMPLATE_WRITE, "<otWriteTemplateToFile>");
	return OT_TEMPLATE_WRITE;
    }

    for (;  *mtp->field; mtp++) {
      	DBUG_MAX((stderr, "field |%s| line |%s|\n", mtp->field, mtp->line));

	if ( (*mtp->field == '!') && (*mtp->line != '[') ) {
	    fprintf(fp, "%s\n", mtp->line);
	    continue;
	}
	if ( *mtp->field == '!' )
	    continue;

	for ( hfp = tp->tr; hfp->field && *hfp->field; hfp++ ) 
	    if (*mtp->field == *hfp->field && !strcmp(mtp->field, hfp->field))
		break;

	if ( !hfp->field || !*hfp->field ) {
	    hf.field = 0;
	    if ( otCopyString( mtp->field, &(hf.field) ) ) {
		otPutPostedMessage( OT_MALLOC_LOCATION,
		    "otWriteTemplateToFile()" );
		return OT_MALLOC_LOCATION;
	    }
	    hf.value = 0;
	    hfp = &hf;
	}
	if ( malErr = otWriteHeaderFieldToString(pjp, hfp, &cp) )
	    return malErr;

	fprintf(fp, "%s\n", cp);
	(void)free(cp);
    }

    /*
     * Now pick up fields which don't match the metatemplate.
     */
    for(hfp = tp->tr; hfp->field && *hfp->field; hfp++) {
	if ( !(kwd = otGetFieldNameAbbr(hfp->field, pjp)) ) {
	    if ( malErr = otWriteHeaderFieldToString( pjp, hfp, &cp ) )
		return malErr;

	    fprintf(fp, "%s\n", cp);
	    (void)free(cp);
        }
    }

    /*
     * Write the notes.
     */
    if (np = tp->notep)
	for ( ; *np->who ; np++)
	    if (*np->text) {
		fprintf(fp, "%s\n\n", np->text);
	    }
    fprintf(fp, "\n");

    /*
     * Write the notedata.
     */

    if (tndp = tp->ndp) {
	if ( malErr = otWriteNotedataToString( tp, &cp ) )
	    return malErr;

	if ( cp ) {
	    fprintf(fp, "%s", cp);
	    free(cp);
	}
    }

    DBUG_MED((stderr, "end otWriteTemplate for proj %s\n", pjp->name));
    return OT_SUCCESS;

}


OTErr otWriteHeaderFieldToString(pjp, hfp, s)
OTProject *pjp;
OTHeaderField *hfp;
char **s;
{
    register OTMetaTemplate *mtp;
    register int optslen, colonpos, j, k;
    int slen, valuelen;
    OTErr optErr;
    char tmp_opts[LONGVALUE];
    char *opts, *cp;
    bool inMeta = TRUE;

    DBUG_MED((stderr, "otWriteHeaderFieldToString |%s|%d|\n", hfp->field, hfp->type));

    for( mtp = pjp->otmp; *mtp->field; mtp++ ) {
	if ( *mtp->field == *hfp->field && !strcmp(mtp->field, hfp->field) )
	    break;
    }
    if ( !*mtp->field )
	inMeta = FALSE;

    if ( inMeta ) {
	optslen = strlen(mtp->opts);
	if (strlen(mtp->field) + optslen > FIELDWIDTH)
	    colonpos = strlen(mtp->field) + optslen;
	else
	    colonpos = FIELDWIDTH;
	opts = mtp->opts;
    } else {
	tmp_opts[0] = 0;
	if ( optErr = getOptString( hfp, tmp_opts ) ) {
	    return optErr;
	}
	optslen = strlen(tmp_opts);

 	if (strlen(hfp->field) + optslen > FIELDWIDTH)
	    colonpos = strlen(hfp->field) + optslen;
	else
	    colonpos = FIELDWIDTH;
	opts = tmp_opts;
    }

    if (hfp->value)
        valuelen = strlen(hfp->value);
    else
        valuelen = 0;
    slen = colonpos + valuelen + 20;

    if ( !(cp = malloc(slen+3)) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "otWriteHeaderFieldToString()");
	return OT_MALLOC_LOCATION;
    }

    *s = cp;
    strcpy(cp, hfp->field);
    for (j = strlen(cp);  j <= FIELDWIDTH;  j++)
	    cp[j] = ' ';

    for (j=colonpos, k=strlen(opts)-1;  k >= 0;  j--, k--)
	cp[j] = opts[k];
    cp[colonpos+1] = ':';
    cp[colonpos+2] = ' ';
    cp[colonpos+3] = 0;

    if ( hfp->value )
	strcat(cp, hfp->value );
    /*
     * This statement would trim trailing white space.  It has been
     * commented out by user request.
     * 
     * else 
     *	   cp[colonpos+2] = 0;
     */

   return OT_SUCCESS;

}


/*
 * General rule in strings: no newlines (since they are record terminators)
 * and no nulls, since they terminate strings.
 */
OTErr otWriteNotedataToString(tp, val)
OTTemplate *tp;
char **val;
{
    register OTNotedata *ndp;
    register char *cp;
    int len;
    char *label;

    /*
     * Estimate the mem size for storing the notedata associated
     * w/ this CR and allocate it.
     */
    ndp = tp->ndp;
    if ( !ndp->nd_flags ) {
	*val = 0;
        return OT_SUCCESS;
    }

    for (len = 0; ndp->nd_flags; ndp++) {
        len += sizeof(OTNotedata);
	if ( ndp->nd_kwd )
	    len += strlen(ndp->nd_kwd);
	if ( ndp->nd_agent )
	    len += strlen(ndp->nd_agent);
	if ( ndp->nd_field )
	    len += strlen(ndp->nd_field);
	if ( ndp->nd_previous )
	    len += strlen(ndp->nd_previous);
	if ( ndp->nd_current )
	    len += strlen(ndp->nd_current);
	/*
	 * Add a fudge factor for spaces, '->' in the case of history lines,
	 * etc.
	 */
	len += 100;
    }

    if ( !(*val = calloc(len, sizeof(char))) && len ) {
	otPutPostedMessage(OT_MALLOC_LOCATION,
	    "otWriteNotedataToString()");
	return OT_MALLOC_LOCATION;
    }
    cp = *val;

    /*
     * Write the fields in the order in which they will be
     * loaded into OTNotedata.
     */
    for ( ndp = tp->ndp; ndp->nd_flags; ndp++ ) {

	*cp++ = '+';
	if (      ND_ISRESERVED( ndp->nd_flags ) )
	    *cp++ = '0';
	else if ( ND_ISVISIBLE( ndp->nd_flags ) )
	    *cp++ = '1';
	else
	    *cp++ = '2';

	if ( ND_ISHISTORY( ndp->nd_flags ) ) {
	    sprintf(cp, "HISTORY %s %s %2d %c%c%c%c%c%c%c%c 19%2d %s\t%s",
		dayNameForNumber(ndp->nd_weekday),
		monthNameForNumber(ndp->nd_mon), ndp->nd_day,
		ndp->nd_time[0], ndp->nd_time[1], ndp->nd_time[2],
		ndp->nd_time[3], ndp->nd_time[4], ndp->nd_time[5],
		ndp->nd_time[6], ndp->nd_time[7],
		ndp->nd_year, ndp->nd_agent, ndp->nd_field);
	    cp += strlen(cp);

	    if ( ndp->nd_previous || ndp->nd_current ) {
		sprintf(cp, ": ");
		cp += strlen(cp);

		if ( ndp->nd_previous )
		    strcat(cp, ndp->nd_previous);
		else
		    sprintf(cp, "NULL");
		cp += strlen(cp);

		sprintf(cp, " -> ");
		cp += strlen(cp);

		if ( ndp->nd_current )
		    strcat(cp, ndp->nd_current);
		else
		    sprintf(cp, "NULL");
		cp += strlen(cp);
	    }
	    sprintf(cp, "\n");
	    cp += strlen(cp);
	} else {
	    /*
	     * For all other cases, just print nd_current value.
	     */
	    if ( ND_ISLUPDATE( ndp->nd_flags ) )
		label = "LUPDATE";
	    else if ( ND_ISUSER( ndp->nd_flags ) )
		label = ndp->nd_kwd;
	    else
	        label = "(kwd)";

	    sprintf(cp, "%s %s %s %2d %c%c%c%c%c%c%c%c 19%2d %s",
		label,
		dayNameForNumber(ndp->nd_weekday),
		monthNameForNumber(ndp->nd_mon), ndp->nd_day, 
		ndp->nd_time[0], ndp->nd_time[1], ndp->nd_time[2],
		ndp->nd_time[3], ndp->nd_time[4], ndp->nd_time[5], 
		ndp->nd_time[6], ndp->nd_time[7],
		ndp->nd_year, ndp->nd_agent);
	    cp += strlen(cp);

	    if ( ndp->nd_current ) {
		strcat(cp, "\t");
		cp += strlen(cp);
		strcat(cp, ndp->nd_current);
		cp += strlen(cp);
	    }
	    sprintf(cp, "\n");
	    cp += strlen(cp);
	}
    }
    *cp = 0;

    return OT_SUCCESS;

}


/*
 *                      o t T e m p l a t e C h a n g e d
 */
OTErr otTemplateChanged()
{
    register OTHeaderField *orighfp, *hfp;
    OTErr noteErr;
    OTTemplate *origtp, *tp;
    char *tp_val, *origtp_val;
    bool change = FALSE;
    OTNote *newnotep;
    OTControlBlock *control;
    OTProject *prjp;
    OTOperation operation;
    int i;

    DBUG_MED((stderr, "otTemplateChanged\n"));

    if ( otCB->cb_pcb->pcb_forceUpdate ) {
	return OT_SUCCESS;
    }
    control = otCB;
    prjp = control->cb_pcb->pcb_project;
    operation = control->cb_operation;
    if (operation == UPDATE) {
	origtp = control->cb_ecb->ecb_origStruct;
	tp     = control->cb_ecb->ecb_tStruct;
    }
    else if (operation == ENTER ) {
	origtp = control->cb_ecb->ecb_blankStruct;
	tp     = control->cb_ecb->ecb_tStruct;
    }
    else if (operation == VALIDATE ) {
	origtp = control->cb_ecb->ecb_blankStruct;
	tp     = control->cb_ecb->ecb_tStruct;
    }

    for (orighfp = origtp->tr; orighfp->field && *orighfp->field; orighfp++) {
        /*
	 * If both original and updated fields are null, then no change
	 */
        DBUG_MED((stderr, "field:old:new\t%s:%s:%s\n", orighfp->field, orighfp->value, otGetHeaderFieldValue(orighfp->field, tp)));

	if (!(tp_val = otGetHeaderFieldValue(orighfp->field, tp)) && !orighfp->value)
	    continue;
	else {
	    /* If only one of the original/updated values is null, or if 
	     * neither are null and they don't match, then there has been a 
	     * change.
	     */
	    if ( (!orighfp->value && tp_val) || (orighfp->value && !tp_val) ||
		strcmp(orighfp->value, tp_val))
		change = TRUE;
	}
    }

    for (hfp = tp->tr; hfp->field && *hfp->field; hfp++) {
        /*
	 * If both original and updated fields are null, then no change
	 */
	if (!(origtp_val = otGetHeaderFieldValue(hfp->field, origtp))
	    && !hfp->value)
	    continue;
	else {
	    /*
	     * If only one of the original/updated values is null, or if 
	     * neither are null and they don't match, then there has been a
	     * change.
	     */
	    if ( (!hfp->value && origtp_val) || (hfp->value && !origtp_val) ||
		strcmp(hfp->value, origtp_val))
		change = TRUE;
	}
    }

    DBUG_MED((stderr, "otTemplateChanged header: change == %s\n", change == TRUE ? "true" : "false"));

    if ( noteErr = otNotesCmp(origtp->notep, tp->notep, &newnotep))
        return noteErr;

    if ( newnotep ) {
	change = TRUE;
	DBUG_MED((stderr, "otTemplateChanged note: change == %s\n", change == TRUE ? "true" : "false"));
	freeNotes(newnotep);
    }

    if ( !otNotedataCmp( origtp->ndp, tp->ndp ) )
        change = TRUE;

    /*
     * If the template remains unchanged, and force update indicator is
     * not set, it calls internal function rmRCSlock() to unlock the RCS
     * file for the object update OT operation case only).
     */
    if ( ! change ) {
        otPutPostedMessage( OT_TEMPLATE_UNCHANGED, prjp->object_name );
        return OT_TEMPLATE_UNCHANGED;
    }
    else
        return OT_SUCCESS;

}



/* 
 * otNotesCmp - compare two strings of notes
 *
 * For each note in the list of notes pointed to by "new", 
 * try to find it in the list pointed to by "old".  If it is not
 * found in the old section, then copy the whole thing to a new
 * OTNote.  Return all the note sections in the "new" list
 * not found in the "old" list.
 *
 * Do the same for all notes in the list of notes pointed to by "old" 
 * NOT found in the list pointed to by "new" (i.e., return bylines for
 * all deleted notes).
 */
OTErr otNotesCmp( old, new, diff )
OTNote *old;
OTNote *new;
OTNote **diff;
{
    int i, j, k, len;
    bool match;
    OTNote *np;

    *diff = 0;
    if ( !new ) {
        return OT_SUCCESS;
    }
    if ( !old ) {
	*diff = new;
        return OT_SUCCESS;
    }

    k = 0;
    np = 0;
    for (i = 0; (new[i].who[0]) && (i < MAXNOTES); i++) {
        DBUG_MED((stderr, "new[%d].who = %s\n", i, new[i].who));
	match = FALSE;
	for (j = 0; (old[j].who[0]) && (match == FALSE) && 
	    (j < MAXNOTES); j++)
	    if ( new[i].text && old[j].text ) {

	        DBUG_MED((stderr, "new[%d].text = %s\n", i, new[i].text));
                DBUG_MED((stderr, "old[%d].text = %s\n", j, old[j].text));
	      
		if (!strcmp(new[i].text, old[j].text)) {
		    DBUG_MED((stderr, "strcmp indicates match\n"));
		    match = TRUE;
		}
	    }

	DBUG_MED((stderr, "Match is %s\n", match == TRUE ? "true" : "false"));
	if (match == FALSE) {
	    if ( k == 0 ) {
		if ( (*diff  = (OTNote *)calloc(MAXNOTES, sizeof(OTNote)))
		    == NULL ) {
		    otPutPostedMessage( OT_MALLOC_LOCATION, "notescmp()" );
		    return OT_MALLOC_LOCATION;
		}
	    }

	    strcpy((*diff)[k].who, new[i].who);
	    strcpy((*diff)[k].date, new[i].date);
	    strcpy((*diff)[k].sens, new[i].sens);

	    len = strlen(new[i].text) + 1;
	    if ( ((*diff)[k].text = (char *)malloc(len+3)) == NULL ) {
	        otPutPostedMessage( OT_MALLOC_LOCATION, "notescmp()" );
		return OT_MALLOC_LOCATION;
	    }
	    strcpy((*diff)[k].text, new[i].text);
	    k++;
	}
    }

    for (i = 0; (old[i].who[0]) && (i < MAXNOTES); i++) {
        DBUG_MED((stderr, "old[%d].who = %s\n", i, old[i].who));
	match = FALSE;
	for (j = 0; (new[j].who[0]) && (match == FALSE) && 
	    (j < MAXNOTES); j++)
	    if ( old[i].text && new[j].text ) {

	        DBUG_MED((stderr, "old[%d].text = %s\n", i, old[i].text));
                DBUG_MED((stderr, "new[%d].text = %s\n", j, new[j].text));
	      
		if (!strcmp(old[i].text, new[j].text)) {
		    DBUG_MED((stderr, "strcmp indicates match\n"));
		    match = TRUE;
		}
	    }

	DBUG_MED((stderr, "Match is %s\n", match == TRUE ? "true" : "false"));
	if (match == FALSE) {

	    if ( k == 0 )
		if ( (*diff  = (OTNote *)calloc(MAXNOTES, sizeof(OTNote)))
		    == NULL ) {
		    otPutPostedMessage( OT_MALLOC_LOCATION, "notescmp()" );
		    return OT_MALLOC_LOCATION;
		}

	    strcpy((*diff)[k].who,  old[i].who);
	    strcpy((*diff)[k].date, old[i].date);
	    strcpy((*diff)[k].sens, old[i].sens);

	    len = strlen(old[i].text) + 1;
	    if ( ((*diff)[k].text = (char *)malloc(len+3)) == NULL ) {
	        otPutPostedMessage( OT_MALLOC_LOCATION, "notescmp()" );
		return OT_MALLOC_LOCATION;
	    }
	    strcpy((*diff)[k].text, old[i].text);
	    k++;
	}
    }
    return OT_SUCCESS;

}

/*
 *                            o t N o t e d a t a C m p
 *
 * If there is any difference, return OT_TEMPLATE_CHANGED, otherwise
 * OT_SUCCESS.
 */
OTErr
otNotedataCmp(origndp, ndp)
OTNotedata *origndp;
OTNotedata *ndp;
{
    bool change = FALSE;

    for( ; !change ; origndp++, ndp++) {

        if ( origndp->nd_flags && !ndp->nd_flags )
	    change = TRUE;
        else if ( !origndp->nd_flags && ndp->nd_flags )
	    change = TRUE;
        else if ( !origndp->nd_flags && !ndp->nd_flags )
	    break;
	else if ( origndp->nd_flags != ndp->nd_flags )
	    change = TRUE;
	else if ( origndp->nd_weekday != ndp->nd_weekday )
	    change = TRUE;	  
	else if ( origndp->nd_mon != ndp->nd_mon )
	    change = TRUE;	  
	else if ( origndp->nd_day != ndp->nd_day )
	    change = TRUE;	  
	else if ( origndp->nd_year != ndp->nd_year )
	    change = TRUE;	  
	else if ( memcmp(origndp->nd_time, ndp->nd_time, MAXTIME ) )
	    change = TRUE;	  

	else if ( (!origndp->nd_kwd && ndp->nd_kwd) || 
	    (origndp->nd_kwd && !ndp->nd_kwd) )
	    change = TRUE;
	else if ( origndp->nd_kwd && ndp->nd_kwd && 
	    strcmp(origndp->nd_kwd, ndp->nd_kwd) )
	    change = TRUE;

	else if ( (!origndp->nd_agent && ndp->nd_agent) ||
	    (origndp->nd_agent && !ndp->nd_agent) )
	    change = TRUE;
	else if ( origndp->nd_agent && ndp->nd_agent && 
	    strcmp(origndp->nd_agent, ndp->nd_agent) )
	    change = TRUE;

	else if ( (!origndp->nd_field && ndp->nd_field) ||
	    (origndp->nd_field && !ndp->nd_field) )
	    change = TRUE;
	else if ( origndp->nd_field && ndp->nd_field && 
	    strcmp(origndp->nd_field, ndp->nd_field) )
	    change = TRUE;	  

	else if ( (!origndp->nd_previous && ndp->nd_previous) ||
	    (origndp->nd_previous && !ndp->nd_previous) )
	    change = TRUE;	  
	else if ( origndp->nd_previous && ndp->nd_previous && 
	    strcmp(origndp->nd_previous, ndp->nd_previous) )
	    change = TRUE;	  

	else if ( (!origndp->nd_current && ndp->nd_current) ||
	    (origndp->nd_current && !ndp->nd_current) )
	    change = TRUE;
	else if ( origndp->nd_current && ndp->nd_current && 
	    strcmp(origndp->nd_current, ndp->nd_current) )
	    change = TRUE;	  

    }
    if (change)
	return OT_SUCCESS;
    else
	return OT_TEMPLATE_UNCHANGED;

}



/*
 * This function initializes a blank OTTemplate template structure based on 
 * the contents of the OTMetaTemplate structure in the project definition
 * structure.  It also stores a pointer to this structure in the
 * ecb_blankStruct member of the OTEnterCB.  The OT template operation 
 * (ot -t) calls otInitTemplate() as a first step of getting a template for
 * the project.  The function returns a pointer to the initialized template
 * structure on success, a null pointer (0) on failure.
 *
 * Here is a table of strings and their substitutions in the blank template.
 *
 * Field name          marker             default      env variable
 * ================    =================  =========    ============
 * Project Name        <PROJECTNAME>                   OT_PROJECT
 * Status              <STATUS>           open         OT_STATUS
 * CR Type             <CRTYPE>           defect       OT_CRTYPE
 * Reported by         <USER>             <$OT_USER>   OT_USER, then USER,
 *							then LOGNAME  (assumed)
 * Date                <DATE>             <`today`>    TODAY  (generated)
 * Sensitivity         <SENSITIVITY>      public       OT_SENSITIVITY
 * (Note) Author       <NOTEAUTHOR>       <$OT_USER>   OT_USER, then USER,
 *							then LOGNAME  (assumed)
 * (Note) Date         <NOTEDATE>         <`today`>    TODAY  (generated)
 * (Note) Sensitivity  <NOTESENSITIVITY>  public       OT_NOTESENSITIVITY
 *
 * Where the <...> default item is calculated and the rest are literal text.
 */
OTErr
otInitTemplate()
{
    register int i, j;
    OTErr dupErr, mallocErr;
    OTTemplate *tp;
    OTMetaTemplate *mtp;
    OTProject *prjp;
    
    prjp = otCB->cb_pcb->pcb_project;
    if ( mallocErr = otMallocTemplate( &tp ) )
	return mallocErr;

    DBUG_MED((stderr, "otInitTemplate for proj %s\n", prjp->name));
    if ( dupErr = otDupMetaTemplate( prjp->otmp, &mtp ) )
        return dupErr;

    for (i=0; mtp[i].field[0]; i++) {
	if ( mtp[i].field[0] == '!') {
	    fixStr(mtp[i].line, prjp->name);
	    continue;
	}
	/* fix all fields */
	fixStr(mtp[i].field, prjp->name);
	fixStr(mtp[i].opts,  prjp->name);
	fixStr(mtp[i].defs,  prjp->name);
	fixStr(mtp[i].abbr,  prjp->name);
	fixStr(mtp[i].head,  prjp->name);
    }

    /*
     * Add field, type, etc, default value from metatemplate.
     */
    for (i=0, j=0;  mtp[i].field[0];  i++) {

      	DBUG_MAX((stderr, "field = |%s|, line = |%s|, defs = |%s|\n", mtp[i].field, mtp[i].line, mtp[i].defs));
        DBUG_MAX((stderr, "otInitTemplate: doing field %d\n", i));

	if (mtp[i].field[0] == '!') {
	    if (mtp[i].line[0] == '[') {
		(void)readByline( mtp[i].line, tp->notep[0].who,
		    tp->notep[0].date, tp->notep[0].sens );
		if ( !(tp->notep[0].text = (char *)calloc(1, LONGVALUE)) ) {
		    otPutPostedMessage(OT_MALLOC_LOCATION, "otInitTemplate()");
		    return OT_MALLOC_LOCATION;
		}
		sprintf( tp->notep[0].text, "[%s %s %s]", 
		    tp->notep[0].who, tp->notep[0].date, tp->notep[0].sens);
		break;
	    }
	    else
	        continue;
	}

	DBUG_MAX((stderr, "otInitTemplate: copy %s\n", mtp[i].field));
	strcpy(tp->tr[j].field, mtp[i].field);
	strcpy(tp->tr[j].list, mtp[i].list);
	tp->tr[j].type = mtp[i].type;
	tp->tr[j].optionality = mtp[i].optionality;
	tp->tr[j].degree = mtp[i].degree;

	if (mtp[i].defs[0]) {
	    if ( !(tp->tr[j].value = 
		(char *)malloc(LONGVALUE)) ) {
		otPutPostedMessage( OT_MALLOC_LOCATION, "otInitTemplate" );
		return OT_MALLOC_LOCATION;
	    }
	    strcpy(tp->tr[j].value, mtp[i].defs);
	}
	j++;
    }
    
    otCB->cb_ecb->ecb_blankStruct = tp;
    otFreeMetaTemplate( mtp );

    return OT_SUCCESS;

}


void
fixStr(string, projectName)
char *  string;			/* string to update */
char *  projectName;		/* name of the current project */
{
    register int i, j, k, l;
    register bool changed;
    char line[COMMAND];
    char field[NAMELEN];

/* now change all the fields */
    memset(field, 0, sizeof(field));
    memset(line, 0, sizeof(line));

    for (changed=FALSE, i=k=0;  string[i];  i++)    {

	if (string[i] != '<')   {
	    if (changed)
		line[k++] = string[i];
	    continue;
	}

	for (j=i+1, l=0;  string[j] && (string[j] != '>');  j++, l++)
	    field[l] = string[j];

	if (!string[j]) {
	    /* error? */
	}
	field[l] = 0;

	if (!changed)   {
	    string[i] = 0;
	    strcpy(line, string);
	    string[i] = '<';
	    changed = TRUE;
	} else
	    line[k] = 0;

	if (!strcmp(field, "PROJECTNAME"))
	    strcat(line, projectName);
	else if (!strcmp(field, "STATUS"))
	    strcat(line, otGetValue("OT_STATUS", "open"));
	else if (!strcmp(field, "CRTYPE"))
	    strcat(line, otGetValue("OT_CRTYPE", "def"));
	else if (!strcmp(field, "USER"))
	    strcat(line, otCB->cb_pcb->pcb_uName);
	else if (!strcmp(field, "DATE"))
	    strcat(line, otGetDate());
	else if (!strcmp(field, "SENSITIVITY"))
	    strcat(line, otGetValue("OT_SENSITIVITY", "public"));
	else if (!strcmp(field, "NOTEAUTHOR"))
	    strcat(line, otCB->cb_pcb->pcb_uName);
	else if (!strcmp(field, "NOTEDATE"))
	    strcat(line, otGetDate());
	else if (!strcmp(field, "NOTESENSITIVITY"))
	    strcat(line, otGetValue("OT_NOTESENSITIVITY", "public"));
	else if (field[0] == '$')
	    strcat(line, otGetValue(&field[1], ""));

	k = strlen(line);
	i = j;
    }

    if (changed)    {
	DBUG_MED((stderr, "fixStr: changed '%s' to '%s'\n", string, line));
	strcpy(string, line);
    }
}

/*
 *                     o t C o l l a t e T e m p l a t e
 *
 * Take the information in the template given as first argument and "collate"
 * it - reshuffle such that the fieldnames, types, etc are exactly as they
 * appear in the metatemplate associated with the current project.
 * Returns OT_TEMPLATE_UNCHANGED if it is already collated, or OT_SUCCESS
 * if it has been collated.
 */
OTErr
otCollateTemplate(tpp)
OTTemplate **tpp;
{
    register OTMetaTemplate *cmp;
    register OTHeaderField *hfp, *newhfp;
    OTProject *prjp;
    OTTemplate *tp;
    OTHeaderField *tmphfp;
    OTMetaTemplate *mtp;
    OTErr collErr;
    int i;
    char *cp, *kwd;
    bool chg, fnd;
    collErr = OT_SUCCESS;
    tp = *tpp;
    prjp = otCB->cb_pcb->pcb_project;
    mtp = prjp->otmp;

    for(chg = FALSE, cmp = mtp, hfp = tp->tr;
	!chg && cmp->field && *cmp->field && hfp->field && *hfp->field; ) {

#ifdef notdef
      	DBUG_MAX((stderr, "metatemplate field = |%s|, type = |%d|, optionality = |%d|, degree = |%d|\n", cmp->field, cmp->type, cmp->optionality, cmp->degree));
      	DBUG_MAX((stderr, "user field = |%s|, type = |%d|, optionality = |%d|, degree = |%d|\n", hfp->field, hfp->type, hfp->optionality, hfp->degree));
#endif

	if ( cmp->field && *cmp->field == '!' ) {
	    cmp++;
	    continue;
	}
	if ( (strcmp(cmp->field, hfp->field))
	     || (cmp->type != hfp->type) 
	     || (cmp->optionality != hfp->optionality) 
	     || (cmp->degree != hfp->degree) )

	    chg = TRUE;

	if ( cmp->list && hfp->list && strcmp(cmp->list, hfp->list) )
	    chg = TRUE;

	if ( (cmp->list && !hfp->list) || (!cmp->list && hfp->list) )
	    chg = TRUE;

	cmp++;
	hfp++;
    }

    if ( chg ) {

	if ( !(tmphfp =
	    (OTHeaderField *)calloc(MAXTEMPLATE, sizeof(OTHeaderField)) ) ) {
	    otPutPostedMessage( OT_MALLOC_LOCATION, "otCollateTemplate()" );
	    collErr = OT_MALLOC_LOCATION;
	}
	newhfp = tmphfp;
	for (cmp = mtp; 
	    cmp->field && *cmp->field && collErr != OT_MALLOC_LOCATION; cmp++){

	    DBUG_MAX((stderr, "field = |%s|, line = |%s|, defs = |%s|\n", cmp->field, cmp->line, cmp->defs));
	    
	    if ( cmp->field && *cmp->field == '!' ) {
	        continue;
	    }

	    COPY_STRING(cmp->field, &(newhfp->field), "otCollateTemplate");
	    COPY_STRING(cmp->list, &(newhfp->list), "otCollateTemplate");
	    newhfp->type = cmp->type;
	    newhfp->optionality = cmp->optionality;
	    newhfp->degree = cmp->degree;
	    newhfp->value = 0;

	    for( fnd = FALSE, hfp = tp->tr;
		!fnd && hfp->field && *hfp->field; hfp++ ) {

		if ( !strcmp(cmp->field, hfp->field) ) {
		    DBUG_MAX((stderr, "field = %s matches\n", hfp->field));
		    COPY_STRING(hfp->value, &(newhfp->value), "otCollateTemplate");
		    fnd = TRUE;
		}
	    }
	    newhfp++;

	}


	for (hfp = tp->tr; collErr != OT_MALLOC_LOCATION && hfp->field && *hfp->field; hfp++)

	    if ( !(kwd = otGetFieldNameAbbr(hfp->field, prjp)) ) {
	        DBUG_MAX((stderr, "hfp->field %s -> nonmatch\n", hfp->field));

		COPY_STRING(hfp->field, &(newhfp->field), "otCollateTemplate");
		COPY_STRING(hfp->list,  &(newhfp->list),  "otCollateTemplate");
		COPY_STRING(hfp->value, &(newhfp->value), "otCollateTemplate");

		newhfp->type = hfp->type;
		newhfp->optionality = hfp->optionality;
		newhfp->degree = hfp->degree;
		newhfp++;
	    }

	for (i = 0; i < MAXTEMPLATE; i++) {
	    if (tp->tr[i].value) {
		free( tp->tr[i].value );
		tp->tr[i].value = 0;
	    }
	    if (tp->tr[i].field) {
		free(tp->tr[i].field);
		tp->tr[i].field = 0;
	    }
	    if (tp->tr[i].list) {
		free(tp->tr[i].list);
		tp->tr[i].list = 0;
	    }
	}
	free( tp->tr );
	tp->tr = tmphfp;

	return collErr;

    } else
	return OT_TEMPLATE_UNCHANGED;

}

/*
 *                          r i n s e T e m p l a t e
 */
void rinseTemplate(tp)
OTTemplate *tp;
{
    register OTHeaderField *hfp;
    register OTNote *notep;

    for(hfp = tp->tr; hfp && hfp->field && *hfp->field; hfp++)
        if (hfp->value)
	    rinse(hfp->value);

    for(notep = tp->notep; notep && *notep->who; notep++)
        if (notep->text)
	    rinse(notep->text);

}


/*
 *                                  r i n s e
 */
void rinse(cp)
char *cp;
{
    char *pp, *start;

    pp = start = cp;

    DBUG_MED((stderr, "rinse begin |%s|\n", cp));
    if (cp) {
	while (*cp != 0) {
	    if ( !iscntrl(*cp) && isprint(*cp) )
		*pp++ = *cp;
	    else if ( (*cp == '\n') || (*cp == '\t') )
		*pp++ = *cp;
	    cp++;
	}
	*pp = 0;
    }
    DBUG_MED((stderr, "rinse result = |%s|\n", start));
}




/*
 * Get header field value.
 * This function finds the value of a header field given the header field 
 * label.
 * Returns a pointer to the value on success, a null pointer (0) on failure.
 */
char *
otGetHeaderFieldValue(headerFieldLabel, tp)
char *headerFieldLabel;
OTTemplate *tp;
{
    register int i;
    register OTHeaderField *hfp;

    DBUG_MED((stderr, "otGetHeaderField: reqd field %s\n", headerFieldLabel));

    /* Just linear search for the literal field name */
    if ( tp ) {
	for (hfp = tp->tr; hfp->field && *hfp->field; hfp++)
	    if ( (*headerFieldLabel == *hfp->field) && !strcmp(headerFieldLabel,
		hfp->field) )
		    return hfp->value;
    }

    return 0;

}



/*
 * Set header field value.
 * This function sets the value of a header field given the header field
 * label.
 * Returns zero on success, and -1 on failure.
 */
OTErr
otSetHeaderFieldValue(headerFieldLabel, headerFieldValue, tp)
char * headerFieldLabel;	/* header field label */
char * headerFieldValue;	/* header field value */
OTTemplate * tp;		/* template structure */
{
    register int i, len;
    register OTHeaderField *hfp;

    DBUG_MED((stderr, "setTField: reqd field %s\n", headerFieldLabel));
    for (hfp = tp->tr; hfp->field && *hfp->field; hfp++) {
        /*
	 * Note the value member of OTHeaderField can be NULL if there is
	 * no value (i.e., not just "") so we account for memory allocation
	 * here.
	 */
	if (!strcmp(headerFieldLabel, hfp->field)) {
	    DBUG_MED((stderr, "otSetHeaderFieldValue: got value %s\n", hfp->field));
	    len = headerFieldValue ? strlen(headerFieldValue) : 0;
	    if ( len ) {
	        if (hfp->value) {
		    /*
		     * We reallocate here because the original allocation to
		     * the value pointed in readTemplate() was based on the
		     * length of the original line in the input user tp.
		     */
		    if ( !( hfp->value = 
			(char *)realloc(hfp->value, len + 1) ) ) {
		        otPutPostedMessage( OT_MALLOC_LOCATION,
			    "otSetHeaderFieldValue" );
			return OT_MALLOC_LOCATION;
		    }
		}
		else {
		    if ( !(hfp->value=(char *)malloc(len + 1))) {
		        otPutPostedMessage( OT_MALLOC_LOCATION,
			    "otSetHeaderFieldValue" );
			return OT_MALLOC_LOCATION;
		    }
		}
		strcpy(hfp->value, headerFieldValue);
	    }
	    else {
		if (hfp->value)
		    free(hfp->value);
		hfp->value = 0;
	    }
	    return OT_SUCCESS;
	}
    }

    return OT_FIELDNAME_NOT_FOUND;
}


/*
 * Set IDnum value.
 * This function sets the value of the IDNUM field.
 * Returns an error status.
 */
OTErr
otSetIDNumValue(headerFieldValue, tStruct)
char * headerFieldValue;	/* header field value */
OTTemplate * tStruct;		/* template structure */
{
    register int i, len;

    DBUG_MED((stderr, "setTField\n"));
    for (i=0;  tStruct->tr[i].field[0];  i++) {
        /*
         * Note the value member of OTHeaderField can be NULL if there is
	 * no value (i.e., not just "") so we account for memory allocation
	 * here.
	 */
	if (tStruct->tr[i].type == TYPE_IDNUM) {
	    DBUG_MED((stderr, "otSetIDNumValue: got value %s\n", tStruct->tr[i].field));
	    len = headerFieldValue ? strlen(headerFieldValue) : 0;
	    if ( len ) {
	        if (tStruct->tr[i].value) {
		    /*
		     * We reallocate here because the original allocation to
		     * the value pointed in readTemplate() was based on the
		     * length of the original line in the input user tStruct.
		     */
		    if ( !( tStruct->tr[i].value = 
			(char *)realloc(tStruct->tr[i].value, len + 1) ) ) {
		        otPutPostedMessage( OT_MALLOC_LOCATION,
			    "otSetHeaderFieldValue" );
			return OT_MALLOC_LOCATION;
		    }
		}
		else {
		    if ( !(tStruct->tr[i].value=(char *)malloc(len + 1))) {
		        otPutPostedMessage( OT_MALLOC_LOCATION,
			    "otSetHeaderFieldValue" );
			return OT_MALLOC_LOCATION;
		    }
		}
		strcpy(tStruct->tr[i].value, headerFieldValue);
		otTrimWs(tStruct->tr[i].value);
	    }
	    else {
		if (tStruct->tr[i].value )
		    free(tStruct->tr[i].value);
		tStruct->tr[i].value = 0;
	    }
	    return OT_SUCCESS;
	}
    }

    return OT_IDNUMFIELD_NOT_FOUND;
}


/*
 * Get IDnum value.
 *
 * Returns integer value assigned to template field of type idnum,
 * 1000000 if the value is "nnnnnn" or 0 otherwise.
 *
 * Argument:
 *  template (IN)       = ptr to template
 */
long
otGetIDNumValue(tp)
OTTemplate * tp;
{
    int n = 0;
    register OTHeaderField *hfp;

    for (hfp = tp->tr; hfp->field && *hfp->field; hfp++) {

	DBUG_MED((stderr, "otGetIDNumValue %s %d\n", hfp->field, hfp->type));

	if ( hfp->type == TYPE_IDNUM ) {
	    DBUG_MIN((stderr, "otGetIDNumValue %s %d match IDNUM\n", hfp->field, hfp->type));
	    if (hfp->value && *(hfp->value)) {
		if (!strcmp(hfp->value, "nnnnnn"))
		    n = 1000000;
		else
		    n = atol(hfp->value);
	    }
	    break;
	}

    }
    DBUG_MED((stderr, "otGetIDNumValue --> %d\n", n));
    return n;
}


/*
 *                         g e t O p t S t r i n g 
 *
 * Take the option information in the OTHeaderField structure and turn it 
 * into a string.
 */
OTErr
getOptString(rec, cp)
OTHeaderField *rec;
char *cp;
{
    char *typename;
 
    if (!cp) {
	otPutPostedMessage( OT_MALLOC_LOCATION, "getOptString" );
	return OT_MALLOC_LOCATION;
    }

    typename = otGetTypeName(rec->type);
    if ( !strcmp(typename, "undefined") ) {
        *cp = 0;
	return OT_SUCCESS;
    }

    if ( (rec->type == TYPE_NULL) && (rec->optionality == OPT_NONE) ) {
        *cp = 0;
	return OT_SUCCESS;
    }

    if (rec->optionality == OPT_MAND)
	strcat(cp, "(");
    else
        strcat(cp, "[");

    if ( !strcmp(typename, "list") )
	strcat(cp, rec->list);
    else if ( !strcmp(typename, "enum") )
	strcat(cp, rec->list);
    else
	strcat(cp, typename);

    if (rec->optionality == OPT_MAND)
	strcat(cp, ")");
    else
        strcat(cp, "]");

    if (rec->degree == DEGREE_MANY)
	strcat(cp, "+");

    return OT_SUCCESS;

}



void 
getTypeOptDeg( fld, startPos )
OTHeaderField *fld;
int startPos;
{
    register int j, k;
    char delim = 0;

    j = startPos;
    for ( k = startPos - 1; isspace(fld->field[k]); k--);   /* Back over w/s */

    if (fld->field[k] == DEGREE_MANY_CHAR)
	k--;
    if (fld->field[k] == ')' ) {
	delim = '(';
	k--;
    }
    else if ( fld->field[k] == ']' ) {
	delim = '[';
	k--;
    }

    if (delim) {
	while ( fld->field[k] != delim && k )
	    k--;
	    
	if ( k && isspace(fld->field[k-1]) ) {

	    /*
	     * There is a delimited type field.  Back up over white space.
	     */
	    j = k;	        
	    for (k--; isspace(fld->field[k]) && k; k--);

	}
	/*
	 * If there is no white space before the delimiter, then the text
	 * in delimiters is part of the field name and not a delimited type 
	 * specifier (e.g., "Ticket number(used in support):").
	 */
    }	

    /*
     * The remainder of the "field" member is nulled out - the type information
     * (e.g., (mailname)) should not appear in the "field" string.
     */
    for (k++; k < j; k++)
	    fld->field[k] = 0;
        
    fld->optionality = OPT_NONE;
    fld->degree = DEGREE_ONE;
    fld->type = TYPE_NULL;

    if ( fld->list == 0 )
	fld->list = calloc(NAMELEN, sizeof(char));

    fld->list[0] = 0;

    if (fld->field[j] == '(') {
	for (k=0, j++; fld->field[j] && (fld->field[j] != ')');  j++, k++)
	    fld->list[k] = fld->field[j];
	fld->list[k] = 0;
	fld->optionality = OPT_MAND;
	if ( (fld->type = otGetType(fld->list)) != TYPE_ENUM )
	    fld->list[0] = 0;
    } else if (fld->field[j] == '[')  {
	for (k=0, j++;  fld->field[j] && (fld->field[j] != ']');  j++, k++)
	    fld->list[k] = fld->field[j];
	fld->list[k] = 0;
	fld->optionality = OPT_OPTL;
	if ( (fld->type = otGetType(fld->list)) != TYPE_ENUM)
	    fld->list[0] = 0;
    } else {
	fld->type = TYPE_NULL;
	fld->optionality = OPT_NONE;
    }

    if ( (fld->field[j] == ']') || (fld->field[j] == ')') ) {
	if (fld->field[++j] == DEGREE_MANY_CHAR) {
	    fld->degree = DEGREE_MANY;
	}
    }

    DBUG_MED((stderr, "getTypeOptDeg: field %s type %d list %s\n", fld->field, fld->type, fld->list));
    DBUG_MED((stderr, "\toptionality %d degree %d\n", fld->optionality, fld->degree));

}


/* otGetAbbrFieldName() - return the field name corresponding to the abbreviation
 */
char *otGetAbbrFieldName(abbr, proj)
char *abbr;
OTProject *proj;
{
    OTMetaTemplate *mTemplate;
    int i;

    if (!abbr || !*abbr || !proj || !proj->otmp)
        return 0;
    mTemplate = proj->otmp;
    for(i = 0; mTemplate[i].field[0]; i++) {
	if (mTemplate[i].abbr[0]) {
	    if ( !strcmp(mTemplate[i].abbr, abbr) )
	        return mTemplate[i].field;
	}
    }
    return 0;

}


/* getAbbrFieldType() - return type for field given the abbreviation.
 */

otType otGetAbbrFieldType(abbr, proj)
char *abbr;
OTProject *proj;
{
    int i;
    OTMetaTemplate *mTemplate;

    if (!abbr || !*abbr || !proj || !proj->otmp)
        return (otType)-1;

    mTemplate = proj->otmp;

    for(i = 0; mTemplate[i].field[0]; i++) {
	if (mTemplate[i].abbr[0]) {
	    if ( !strcmp(mTemplate[i].abbr, abbr) )
	        return mTemplate[i].type;
	}
    }
    return (otType)-1;

}


/* getFieldNameAbbr() - return the abbreviation corresponding to the fieldname
 */
char *otGetFieldNameAbbr(fldname, proj)
char *fldname;
OTProject *proj;
{
    OTMetaTemplate *mTemplate;
    int i;

    if (!fldname || !*fldname || !proj || !proj->otmp)
        return 0;
    mTemplate = proj->otmp;
    for(i = 0; i < MAXTEMPLATE && mTemplate[i].field[0]; i++) {
	if (mTemplate[i].abbr[0]) {
	    if ( !strcmp(mTemplate[i].field, fldname) )
	        return mTemplate[i].abbr;
	}
    }
    return 0;

}


 
/* 
 *                         o t M a l l o c T e m p l a t e
 *
 * Allocate space for a template.
 */
OTErr otMallocTemplate( tp )
OTTemplate **tp;
{
    int i;  

    if ( !(*tp = (OTTemplate *)calloc(1, sizeof(OTTemplate))) ) {
	otPutPostedMessage( OT_MALLOC_LOCATION, "otMallocTemplate" );
	return OT_MALLOC_LOCATION;
    }

    if ( !((*tp)->tr = (OTHeaderField *)calloc(MAXTEMPLATE, 
	sizeof(OTHeaderField)) ) ) {
	otPutPostedMessage( OT_MALLOC_LOCATION, "otMallocTemplate" );
	return OT_MALLOC_LOCATION;
    }

    for (i = 0; i < MAXTEMPLATE; i++) {
	if ( !((*tp)->tr[i].field = (char *)calloc(NAMELEN, sizeof(char))) ) {
	    otPutPostedMessage( OT_MALLOC_LOCATION, "otMallocTemplate" );
	    return OT_MALLOC_LOCATION;
	}
	if ( !((*tp)->tr[i].list = (char *)calloc(NAMELEN, sizeof(char))) ) {
	    otPutPostedMessage( OT_MALLOC_LOCATION, "otMallocTemplate" );
	    return OT_MALLOC_LOCATION;
	}
    }

    if ( !( (*tp)->notep = (OTNote *)calloc(MAXNOTES, sizeof(OTNote)) ) ) {
	otPutPostedMessage( OT_MALLOC_LOCATION, "otMallocTemplate" );
	return OT_MALLOC_LOCATION;
    }

    if ( !( (*tp)->ndp = (OTNotedata *)calloc(MAXNDATA, sizeof(OTNotedata)) ) ) {
	otPutPostedMessage( OT_MALLOC_LOCATION, "otMallocTemplate" );
	return OT_MALLOC_LOCATION;
    } 

    return OT_SUCCESS;
}


void
otFreeTemplate(tp)
OTTemplate *tp;
{
    register int i;

    if (tp) {
	if (tp->tr) {
	    for (i = 0; /* (tp->tr[i].field[0]) && */ (i < MAXTEMPLATE); i++) {
		if (tp->tr[i].value) {
		    free( tp->tr[i].value );
		    tp->tr[i].value = 0;
		}
		if (tp->tr[i].field) {
		    free(tp->tr[i].field);
		    tp->tr[i].field = 0;
		}
		if (tp->tr[i].list) {
		    free(tp->tr[i].list);
		    tp->tr[i].list = 0;
		}
	    }
	    free(tp->tr);
	}
	freeNotes(tp->notep);
	freeNotedata(tp->ndp);
	free(tp);
    }

    return;
}


/* free memory allocated to structures and substructures of ot_t_note. */

void freeNotes( np )
OTNote *np;
{
    register OTNote *tmp_np;

    if (np) {
 	for (tmp_np = np; *tmp_np->who; tmp_np++)
	    if (tmp_np->text) {
 		free( tmp_np->text);
		tmp_np->text = 0;
	    }

	free(np);
    }	

}

/* free memory allocated to structures and substructures of ot_t_history. */

void freeNotedata( ndp )
OTNotedata *ndp;
{
    register OTNotedata *tmp_ndp;

    if ( ndp ) {
	for(tmp_ndp = ndp ; tmp_ndp && tmp_ndp->nd_flags; tmp_ndp++ ) {
	    FREEANDZERO( tmp_ndp->nd_kwd );
	    FREEANDZERO( tmp_ndp->nd_agent );
	    FREEANDZERO( tmp_ndp->nd_field );
	    FREEANDZERO( tmp_ndp->nd_previous );
	    FREEANDZERO( tmp_ndp->nd_current );
	}
	free(ndp);
    }

}

OTErr
otDupTemplate(from, to)
OTTemplate *from;
OTTemplate **to;
{
    OTErr mallocErr;
    register OTHeaderField *fromHfp, *toHfp;
    register OTNote *fromNp, *toNp;
    register OTNotedata *fromNdp, *toNdp;

    if ( !from )
	return OT_SUCCESS;

    if ( mallocErr = otMallocTemplate( to ) )
        return mallocErr;

    toHfp = (*to)->tr;
    fromHfp = from->tr;
    
    for( ; fromHfp->field && *fromHfp->field; fromHfp++, toHfp++ ) {

	if (otCopyString( fromHfp->field, &(toHfp->field) )) {
	    otPutPostedMessage(OT_MALLOC_LOCATION, "otDupTemplate()");
	    return OT_MALLOC_LOCATION;
	}
	if (otCopyString( fromHfp->list, &(toHfp->list) )) {
	    otPutPostedMessage(OT_MALLOC_LOCATION, "otDupTemplate()");
	    return OT_MALLOC_LOCATION;
	}
	if (otCopyString( fromHfp->value, &(toHfp->value) )) {
	    otPutPostedMessage(OT_MALLOC_LOCATION, "otDupTemplate()");
	    return OT_MALLOC_LOCATION;
	}

	toHfp->type = fromHfp->type;
	toHfp->optionality = fromHfp->optionality;
	toHfp->degree = fromHfp->degree;
    }

    if ( from->notep ) {
	toNp = (*to)->notep;
	fromNp = from->notep;

	for( ; *fromNp->who; fromNp++, toNp++ ) {

	    strcpy(toNp->who, fromNp->who);
	    strcpy(toNp->date, fromNp->date);
	    strcpy(toNp->sens, fromNp->sens);

	    if ( fromNp->text ) {
		if (!(toNp->text = (char *)malloc(strlen(fromNp->text) + 1))) {
		    otFreeTemplate((*to));
		    otPutPostedMessage(OT_MALLOC_LOCATION, "otDupTemplate()");
		    return OT_MALLOC_LOCATION;
		}
		else
		    strcpy(toNp->text, fromNp->text);
	    }
	}
    }
    else
	(*to)->notep = 0;

    if ( from->ndp ) {
	toNdp = (*to)->ndp;
	fromNdp = from->ndp;

	for( ; fromNdp->nd_flags; fromNdp++, toNdp++ ) {
	    toNdp->nd_flags = fromNdp->nd_flags;
	    toNdp->nd_weekday = fromNdp->nd_weekday;
	    toNdp->nd_mon = fromNdp->nd_mon;
	    toNdp->nd_day = fromNdp->nd_day;
	    toNdp->nd_year = fromNdp->nd_year;
	    memcpy(toNdp->nd_time, fromNdp->nd_time, MAXTIME);

	    if (otCopyString( fromNdp->nd_kwd, &(toNdp->nd_kwd) )) {
		otPutPostedMessage(OT_MALLOC_LOCATION, "otDupTemplate()");
		return OT_MALLOC_LOCATION;
	    }
	    if (otCopyString( fromNdp->nd_agent, &(toNdp->nd_agent) )) {
		otPutPostedMessage(OT_MALLOC_LOCATION, "otDupTemplate()");
		return OT_MALLOC_LOCATION;
	    }
	    if (otCopyString( fromNdp->nd_field, &(toNdp->nd_field) )) {
		otPutPostedMessage(OT_MALLOC_LOCATION, "otDupTemplate()");
		return OT_MALLOC_LOCATION;
	    }
	    if (otCopyString( fromNdp->nd_previous, &(toNdp->nd_previous) )) {
		otPutPostedMessage(OT_MALLOC_LOCATION, "otDupTemplate()");
		return OT_MALLOC_LOCATION;
	    }
	    if (otCopyString( fromNdp->nd_current, &(toNdp->nd_current) )) {
		otPutPostedMessage(OT_MALLOC_LOCATION, "otDupTemplate()");
		return OT_MALLOC_LOCATION;
	    }
	}
    } else
	(*to)->ndp = 0;

    return OT_SUCCESS;

}
