static char rcsid[]="$Id: tkCanvLatex.c,v 1.2 94/02/14 14:53:00 mangin Exp $";
/* 
 * tkCanvLatex.c --
 *
 *	This file implements latex items for canvas widgets.
 *
 * Copyright (c) 1991-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdio.h>
#include <tkInt.h>
#include <tkConfig.h>
#include "tkCanvas.h"

/*
 * One of the following structures is kept for each line of text
 * in a latex item.  It contains geometry and display information
 * for that line.
 */

typedef struct TextLine {
    char *firstChar;		/* Pointer to the first character in this
				 * line (in the "latex" field of enclosing
				 * latex item). */
    int numChars;		/* Number of characters displayed in this
				 * line. */
    int totalChars;		/* Total number of characters included as
				 * part of this line (may include an extra
				 * space character at the end that isn't
				 * displayed). */
    int x, y;			/* Origin at which to draw line on screen
				 * (in integer pixel units, but in canvas
				 * coordinates, not screen coordinates). */
    int x1, y1;			/* Upper-left pixel that is part of text
				 * line on screen (again, in integer canvas
				 * pixel units). */
    int x2, y2;			/* Lower-left pixel that is part of text
				 * line on screen (again, in integer canvas
				 * pixel units). */
} TextLine;

/*
 * The structure below defines the record for each latex item.
 */

typedef struct LatexItem  {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    char *text;			/* Text for item (malloc-ed). */
    int numChars;		/* Number of non-NULL characters in text. */
    double bx, by;		/* Positioning point for the box. */
    double tx, ty;		/* Positioning point for the text */
    double ptWidth, ptHeight;   /* Point dims of box */
    Tk_Anchor anchor;		/* Where to anchor the box relative to (bx,by). */
    int lineWidth;		/* Width of lines for word-wrap, pixels.
				 * Zero means no word-wrap. */
    int rightEdge;		/* Pixel just to right of right edge of
				 * area of latex item.  Used for selecting
				 * up to end of line. */
    XFontStruct *fontPtr;	/* Font for drawing text. */
    int outlineWidth;		/* Width of box outline. */
    XColor *outlineColor;	/* Color for box outline and for text. */
    Pixmap stipple;		/* Stipple bitmap for box, or None. */
    GC outlineGC;		/* Graphics context for outline. */
    TextLine *linePtr;		/* Pointer to array of structures describing
				 * individual lines of text of item (malloc-ed). */
    int numLines;		/* Number of structs at *linePtr. */
} LatexItem;

/*
 * Information used for parsing configuration specs:
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_ANCHOR, "-anchor", (char *) NULL, (char *) NULL,
	"center", Tk_Offset(LatexItem, anchor),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_FONT, "-font", (char *) NULL, (char *) NULL,
	"-Adobe-Helvetica-Bold-R-Normal--*-120-*",
	Tk_Offset(LatexItem, fontPtr), 0},
    {TK_CONFIG_COLOR, "-outline", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(LatexItem, outlineColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_PIXELS, "-outlinewidth", (char *) NULL, (char *) NULL,
       "1", Tk_Offset(LatexItem, outlineWidth), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(LatexItem, stipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_STRING, "-text", (char *) NULL, (char *) NULL,
	"", Tk_Offset(LatexItem, text), 0},
    {TK_CONFIG_PIXELS, "-lineWidth", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(LatexItem, outlineWidth), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */

static void		ComputeLatexBbox _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    LatexItem *latexPtr));
static int		ConfigureLatex _ANSI_ARGS_((
			    Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateLatex _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    struct Tk_Item *itemPtr, int argc, char **argv));
static void		DeleteLatex _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr));
static void		DisplayLatex _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Drawable dst));
static void		ScaleLatex _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static int		LatexCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, int argc, char **argv));
static int		LatexToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *rectPtr));
static double		LatexToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *pointPtr));
static int		LatexToPostscript _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Tk_PostscriptInfo *psInfoPtr));
static void		TranslateLatex _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the latex item type
 * by means of procedures that can be invoked by generic item code.
 */

Tk_ItemType TkLatexType = {
    "latex",				/* name */
    sizeof(LatexItem),			/* itemSize */
    CreateLatex,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureLatex,			/* configureProc */
    LatexCoords,				/* coordProc */
    DeleteLatex,				/* deleteProc */
    DisplayLatex,			/* displayProc */
    0,					/* alwaysRedraw */
    LatexToPoint,			/* pointProc */
    LatexToArea,				/* areaProc */
    LatexToPostscript,			/* postscriptProc */
    ScaleLatex,				/* scaleProc */
    TranslateLatex,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,          /* indexProc */
    (Tk_ItemCursorProc *) NULL,         /* icursorProc */
    (Tk_ItemSelectionProc *) NULL,      /* selectionProc */
    (Tk_ItemInsertProc *) NULL,         /* insertProc */
    (Tk_ItemDCharsProc *) NULL,         /* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};

/***************************************************************
 ****          Canvas latex item type creation              ****
 ***************************************************************/

void LatexItemTypeCreate()
{
  Tk_CreateItemType(&TkLatexType);
}


/*
 *--------------------------------------------------------------
 *
 * CreateLatex --
 *
 *	This procedure is invoked to create a new latex item
 *	in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new latex item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateLatex(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing rectangle. */
{
    register LatexItem *latexPtr = (LatexItem *) itemPtr;

    if (argc < 2) {
	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		Tk_PathName(canvasPtr->tkwin), "\" create ",
		itemPtr->typePtr->name, " x y [options]", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    latexPtr->text = NULL;
    latexPtr->numChars = 0;
    latexPtr->anchor = TK_ANCHOR_CENTER;
    latexPtr->lineWidth = 0;
    latexPtr->ptWidth = latexPtr->ptHeight = 0.0;
    latexPtr->rightEdge = 0;
    latexPtr->fontPtr = NULL;
    latexPtr->outlineWidth = 0;
    latexPtr->outlineColor = NULL;
    latexPtr->stipple = None;
    latexPtr->outlineGC = None;
    latexPtr->linePtr = NULL;
    latexPtr->numLines = 0;

    /*
     * Process the arguments to fill in the item record.
     */

    if ((TkGetCanvasCoord(canvasPtr, argv[0], &latexPtr->bx) != TCL_OK)
	|| (TkGetCanvasCoord(canvasPtr, argv[1], &latexPtr->by) != TCL_OK)) {
      return TCL_ERROR;
    }

    if (ConfigureLatex(canvasPtr, itemPtr, argc-2, argv+2, 0) != TCL_OK) {
      DeleteLatex(canvasPtr, itemPtr);
      return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * LatexCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on latex items.  See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
LatexCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
  register LatexItem *latexPtr = (LatexItem *) itemPtr;
  char x[TCL_DOUBLE_SPACE], y[TCL_DOUBLE_SPACE];

  if (argc == 0) {
    Tcl_PrintDouble(canvasPtr->interp, latexPtr->bx, x);
    Tcl_PrintDouble(canvasPtr->interp, latexPtr->by, y);
    Tcl_AppendResult(canvasPtr->interp, x, " ", y, (char *) NULL);
  } else if (argc == 2) {
    if ((TkGetCanvasCoord(canvasPtr, argv[0], &latexPtr->bx) != TCL_OK) ||
	(TkGetCanvasCoord(canvasPtr, argv[1], &latexPtr->by) != TCL_OK)) {
      return TCL_ERROR;
    }
    ComputeLatexBbox(canvasPtr, latexPtr);
  } else {
    sprintf(canvasPtr->interp->result,
	    "wrong # coordinates:  expected 0 or 2, got %d",
	    argc);
    return TCL_ERROR;
  }
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureLatex --
 *
 *	This procedure is invoked to configure various aspects
 *	of a latex item.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureLatex(canvasPtr, itemPtr, argc, argv, flags)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Rectangle item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
  register LatexItem *latexPtr = (LatexItem *) itemPtr;
  XGCValues gcValues;
  GC newGC;
  unsigned long mask;
  int textFlag, i;

  textFlag = 0;
  for (i = 0; i < argc; i++)
    if ((*(1+argv[i]) == 't') &&
	!strcmp(argv[i], "-text")) {
      textFlag = 1;
      break;
    }
  
  if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
			 configSpecs, argc, argv,
			 (char *) latexPtr, flags) != TCL_OK) {
    return TCL_ERROR;
  }

  /*
   * A few of the options require additional processing, such as
   * graphics contexts.
   */

  latexPtr->numChars = strlen(latexPtr->text);
  newGC = None;
  if ((latexPtr->outlineColor != NULL) && (latexPtr->fontPtr != NULL)) {
    gcValues.foreground = latexPtr->outlineColor->pixel;
    gcValues.font = latexPtr->fontPtr->fid;
    mask = GCForeground|GCFont;
    if (latexPtr->stipple != None) {
      gcValues.stipple = latexPtr->stipple;
      gcValues.fill_style = FillStippled;
      mask |= GCForeground|GCStipple|GCFillStyle;
    }
    newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
  }
  if (latexPtr->outlineGC != None) {
    Tk_FreeGC(canvasPtr->display, latexPtr->outlineGC);
  }
  latexPtr->outlineGC = newGC;

  /**
   **    If the text was changed, recompile latex to get the point dims
   **/

  if (textFlag) {
    /*  recompute dims  */
    int largc;
    char **largv;
    
    if (Tcl_VarEval(canvasPtr->interp,
		    "computeLatexBbox {",
		    latexPtr->text, "}", (char *)0) != TCL_OK) {
      Tcl_AppendResult(canvasPtr->interp,
		       "computeLatexBbox error", (char *)0);
    }
    
    if (Tcl_SplitList(canvasPtr->interp,
                      canvasPtr->interp->result,
                      &largc, &largv) != TCL_OK) {
      Tcl_AppendResult(canvasPtr->interp,
		       "computeLatexBbox returned invalid list structure",
		       (char *)0);
      return TCL_ERROR;
    }

    if (largc != 2) {
      char buf[32];

      free((char *)largv);
      sprintf(buf, "%d", largc);
      Tcl_AppendResult(canvasPtr->interp,
		       "Expected two dimensions from computeLatexBbox, got ",
		       buf, (char *)0);
      return TCL_ERROR;
    }

    if ((Tcl_GetDouble(canvasPtr->interp, largv[0],
		       &latexPtr->ptWidth) != TCL_OK) ||
	(Tcl_GetDouble(canvasPtr->interp, largv[1],
		       &latexPtr->ptHeight) != TCL_OK)) {
      free((char *)largv);
      Tcl_AppendResult(canvasPtr->interp,
		       "Expected floating point dims from computeLatexBbox, got \"",
		       largv[0], largv[1], "\"", (char *)0);
      return(TCL_ERROR);
    }

    free((char *)largv);
  }
  
  ComputeLatexBbox(canvasPtr, latexPtr);
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteLatex --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a latex item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteLatex(canvasPtr, itemPtr)
    Tk_Canvas *canvasPtr;		/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
  register LatexItem *latexPtr = (LatexItem *) itemPtr;

  if (latexPtr->text != NULL) {
    ckfree(latexPtr->text);
  }
  if (latexPtr->fontPtr != NULL) {
    Tk_FreeFontStruct(latexPtr->fontPtr);
  }
  if (latexPtr->outlineColor != NULL) {
    Tk_FreeColor(latexPtr->outlineColor);
  }
  if (latexPtr->stipple != None) {
    Tk_FreeBitmap(canvasPtr->display, latexPtr->stipple);
  }
  if (latexPtr->outlineGC != None) {
    Tk_FreeGC(canvasPtr->display, latexPtr->outlineGC);
  }
  if (latexPtr->linePtr != NULL) {
    ckfree((char *) latexPtr->linePtr);
  }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeLatexBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a latex item.
 *	In addition, it recomputes all of the geometry information
 *	used to display a latex item or check for mouse hits.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr, and the linePtr structure is regenerated
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

#define PointToPixel(D) \
  ((D) * (25.4/72.0)*WidthOfScreen(Tk_Screen(canvasPtr->tkwin)) / \
   WidthMMOfScreen(Tk_Screen(canvasPtr->tkwin)))

static void
ComputeLatexBbox(canvasPtr, latexPtr)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    register LatexItem *latexPtr;		/* Item whose bbos is to be
					 * recomputed. */
{
    register TextLine *linePtr;
#define MAX_LINES 100
    char *lineStart[MAX_LINES];
    int lineChars[MAX_LINES];
    int linePixels[MAX_LINES];
    int numLines, wrapPixels, maxLinePixels, leftTX, topTY, y;
    double bxc, byc, bwidth, bheight;
    int bx1, by1, bx2, by2, bloat;
    int lineHeight, i;
    char *p;
    XCharStruct *maxBoundsPtr = &latexPtr->fontPtr->max_bounds;

    /**
     ** The bbox is the union of the text bbox
     **   and of the box bbox
     **/

    /*  First compute text positionning point  */
    /*  == center of box.  */

    bxc = latexPtr->bx;
    byc= latexPtr->by;
    bwidth = PointToPixel(latexPtr->ptWidth);
    bheight = PointToPixel(latexPtr->ptHeight);
    
    switch (latexPtr->anchor) {

    case TK_ANCHOR_NW:
    case TK_ANCHOR_N:
    case TK_ANCHOR_NE:
      byc += bheight/2.0;
      break;

    case TK_ANCHOR_W:
    case TK_ANCHOR_CENTER:
    case TK_ANCHOR_E:
      break;

    case TK_ANCHOR_SW:
    case TK_ANCHOR_S:
    case TK_ANCHOR_SE:
      byc -= bheight/2.0;
      break;
    }
    
    switch (latexPtr->anchor) {

    case TK_ANCHOR_NW:
    case TK_ANCHOR_W:
    case TK_ANCHOR_SW:
      bxc += bwidth/2;
      break;

    case TK_ANCHOR_N:
    case TK_ANCHOR_CENTER:
    case TK_ANCHOR_S:
      break;

    case TK_ANCHOR_NE:
    case TK_ANCHOR_E:
    case TK_ANCHOR_SE:
      bxc -= bwidth/2;
      break;
    }
    
    if (latexPtr->linePtr != NULL) {
      ckfree((char *) latexPtr->linePtr);
      latexPtr->linePtr = NULL;
    }

    /*
     * Work through the text computing the starting point, number of
     * characters, and number of pixels in each line.
     */

    p = latexPtr->text;
    if (latexPtr->lineWidth > 0) {
      wrapPixels = maxLinePixels = latexPtr->lineWidth;
    } else {
      wrapPixels = 10000000;
      maxLinePixels = 0;
    }
    for (numLines = 0; (numLines < MAX_LINES); numLines++) {
      int numChars, numPixels;
      numChars = TkMeasureChars(latexPtr->fontPtr, p,
				(latexPtr->text + latexPtr->numChars) - p, 0,
				wrapPixels, TK_WHOLE_WORDS|TK_AT_LEAST_ONE, &numPixels);
      if (numPixels > maxLinePixels) {
	maxLinePixels = numPixels;
      }
      lineStart[numLines] = p;
      lineChars[numLines] = numChars;
      linePixels[numLines] = numPixels;
      p += numChars;

      /*
       * Skip space character that terminates a line, if there is one.
       * In the case of multiple spaces, all but one will be displayed.
       */
      
      if (isspace(UCHAR(*p))) {
	p++;
      } else if (*p == 0) {
	/*
	 * The code below is tricky.  Putting the loop termination
	 * here guarantees that there's a TextLine for the last
	 * line of text, even if the line is empty (this can
	 * also happen if the entire text item is empty).
	 */
	
	numLines++;
	break;
      }
    }
    
    /*
     * Use overall geometry information to compute the top-left corner
     * of the bounding box for the latex item.
     */
    
    leftTX = 0.5 + bxc;
    topTY =  0.5 + byc;
    lineHeight = latexPtr->fontPtr->ascent + latexPtr->fontPtr->descent;

    topTY -= (lineHeight * numLines)/2;
    leftTX -= maxLinePixels/2;
    latexPtr->rightEdge = leftTX + maxLinePixels;

    /*
     * Create the new TextLine array and fill it in using the geometry
     * information gathered already.
     */

    if (numLines > 0) {
	latexPtr->linePtr = (TextLine *) ckalloc((unsigned)
		(numLines * sizeof(TextLine)));
    } else {
	latexPtr->linePtr = NULL;
    }
    latexPtr->numLines = numLines;
    for (i = 0, linePtr = latexPtr->linePtr, y = topTY;
	    i < numLines; i++, linePtr++, y += lineHeight) {
	linePtr->firstChar = lineStart[i];
	linePtr->numChars = lineChars[i];
	if (i == (numLines-1)) {
	    linePtr->totalChars = linePtr->numChars;
	} else {
	    linePtr->totalChars = lineStart[i+1] - lineStart[i];
	}
	linePtr->x = leftTX;
	linePtr->y = y + latexPtr->fontPtr->ascent;
	linePtr->x1 = linePtr->x + maxBoundsPtr->lbearing;
	linePtr->y1 = y;
	linePtr->x2 = linePtr->x + linePixels[i];
	linePtr->y2 = linePtr->y + latexPtr->fontPtr->descent - 1;
    }

    /*
     * Last of all, update the bounding box for the item.
     */

    linePtr = latexPtr->linePtr;
    latexPtr->header.x1 = latexPtr->header.x2 = leftTX;
    latexPtr->header.y1 = topTY;
    latexPtr->header.y2 = topTY + numLines*lineHeight;
    for (linePtr = latexPtr->linePtr, i = latexPtr->numLines; i > 0;
	    i--, linePtr++) {
	if (linePtr->x1 < latexPtr->header.x1) {
	    latexPtr->header.x1 = linePtr->x1;
	}
	if (linePtr->x2 >= latexPtr->header.x2) {
	    latexPtr->header.x2 = linePtr->x2 + 1;
	}
    }

    /**
     **    Compute the union of the computed
     **      text bbox with the box bbox
     **/
    bloat = (latexPtr->outlineWidth + 1)/2 + 1;
    bx1 = bxc - bwidth/2.0 - bloat;
    bx2 = bxc + bwidth/2.0 + bloat;
    by1 = byc - bheight/2.0 - bloat;
    by2 = byc + bheight/2.0 + bloat;

    if (bx1 < latexPtr->header.x1)
      latexPtr->header.x1 = bx1;
    if (bx2 > latexPtr->header.x2)
      latexPtr->header.x2 = bx2;
    if (by1 < latexPtr->header.y1)
      latexPtr->header.y1 = by1;
    if (by2 > latexPtr->header.y2)
      latexPtr->header.y2 = by2;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayLatex --
 *
 *	This procedure is invoked to draw a latex item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplayLatex(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
  register LatexItem *latexPtr = (LatexItem *) itemPtr;
  Display *display = Tk_Display(canvasPtr->tkwin);
  register TextLine *linePtr;
  int i;

  if (latexPtr->outlineGC == None) {
    return;
  }

  /*
   * If we're stippling, then modify the stipple offset in the GC.  Be
   * sure to reset the offset when done, since the GC is supposed to be
   * read-only.
   */

  if (latexPtr->stipple != None) {
    XSetTSOrigin(display, latexPtr->outlineGC,
		 -canvasPtr->drawableXOrigin, -canvasPtr->drawableYOrigin);
  }

  for (linePtr = latexPtr->linePtr, i = latexPtr->numLines;
       i > 0; linePtr++, i--) {

    TkDisplayChars(display, drawable, latexPtr->outlineGC, latexPtr->fontPtr,
		   linePtr->firstChar, linePtr->numChars,
		   linePtr->x - canvasPtr->drawableXOrigin,
		   linePtr->y - canvasPtr->drawableYOrigin, 0);
    }

  if (latexPtr->stipple != None) {
    XSetTSOrigin(display, latexPtr->outlineGC, 0, 0);
  }

  /*
   *  Display the box
   */
  {
    unsigned long vmask;
    XGCValues gcval, oldgcval;
    double bxc, byc, bwidth, bheight, bloat;
    
    bxc = latexPtr->bx;
    byc= latexPtr->by;
    bwidth = PointToPixel(latexPtr->ptWidth);
    bheight = PointToPixel(latexPtr->ptHeight);
    
    switch (latexPtr->anchor) {
      
    case TK_ANCHOR_NW:
    case TK_ANCHOR_N:
    case TK_ANCHOR_NE:
      byc += bheight/2.0;
      break;

    case TK_ANCHOR_W:
    case TK_ANCHOR_CENTER:
    case TK_ANCHOR_E:
      break;

    case TK_ANCHOR_SW:
    case TK_ANCHOR_S:
    case TK_ANCHOR_SE:
      byc -= bheight/2.0;
      break;
    }
    
    switch (latexPtr->anchor) {

    case TK_ANCHOR_NW:
    case TK_ANCHOR_W:
    case TK_ANCHOR_SW:
      bxc += bwidth/2;
      break;

    case TK_ANCHOR_N:
    case TK_ANCHOR_CENTER:
    case TK_ANCHOR_S:
      break;

    case TK_ANCHOR_NE:
    case TK_ANCHOR_E:
    case TK_ANCHOR_SE:
      bxc -= bwidth/2.0;
      break;
    }
    
    bloat = 0.5 * latexPtr->outlineWidth;

    vmask = GCFillStyle;
    XGetGCValues(display, latexPtr->outlineGC, vmask, &oldgcval);
    
    gcval.fill_style = FillSolid;
    XChangeGC(display, latexPtr->outlineGC, vmask, &gcval);

    XDrawRectangle(display, drawable, latexPtr->outlineGC,
		   (int)(0.5 + bxc - bwidth/2 + bloat),
		   (int)(0.5 + byc - bheight/2 + bloat),
		   (int)(0.5 + bwidth - latexPtr->outlineWidth),
		   (int)(0.5 + bheight - latexPtr->outlineWidth));

    XChangeGC(display, latexPtr->outlineGC, vmask, &oldgcval);
  }
}

/*
 *--------------------------------------------------------------
 *
 * LatexToPoint --
 *
 *	Computes the distance from a given point to a given
 *	latex item, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the item.  If the
 *	point isn't inside the item then the return value is the
 *	distance from the point to the item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
LatexToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    LatexItem *latexPtr = (LatexItem *) itemPtr;
    register TextLine *linePtr;
    int i;
    double xDiff, yDiff, dist, minDist;
    double x1, y1, x2, y2, bwidth, bheight, inc, tmp;
    
    /*
     * Treat each line in the latex item as a rectangle, compute the
     * distance to that rectangle, and take the minimum of these
     * distances.  Perform most of the calculations in integer pixel
     * units, since that's how the dimensions of the latex are defined.
     */

    minDist = -1.0;
    for (linePtr = latexPtr->linePtr, i = latexPtr->numLines;
	 i > 0; linePtr++, i--) {
      
      /*
       * If the point is inside the line's rectangle, then can
       * return immediately.
       */
      
      if ((pointPtr[0] >= linePtr->x1)
	  && (pointPtr[0] <= linePtr->x2)
	  && (pointPtr[1] >= linePtr->y1)
	  && (pointPtr[1] <= linePtr->y2)) {
	return 0.0;
      }
      
      /*
       * Point is outside line's rectangle; compute distance to nearest
       * side.
       */
      
      if (pointPtr[0] < linePtr->x1) {
	xDiff = linePtr->x1 - pointPtr[0];
      } else if (pointPtr[0] > linePtr->x2)  {
	xDiff = pointPtr[0] - linePtr->x2;
      } else {
	xDiff = 0;
      }
      
      if (pointPtr[1] < linePtr->y1) {
	yDiff = linePtr->y1 - pointPtr[1];
      } else if (pointPtr[1] > linePtr->y2)  {
	yDiff = pointPtr[1] - linePtr->y2;
      } else {
	yDiff = 0;
      }
      
      dist = hypot(xDiff, yDiff);
      if ((dist < minDist) || (minDist < 0.0)) {
	minDist = dist;
      }
    }

    /*  Consider the box now  */

    bwidth = PointToPixel(latexPtr->ptWidth);
    bheight = PointToPixel(latexPtr->ptHeight);
    x1 = latexPtr->bx;
    y1 = latexPtr->by;
      
    switch (latexPtr->anchor) {
      
    case TK_ANCHOR_NW:
    case TK_ANCHOR_N:
    case TK_ANCHOR_NE:
      break;

    case TK_ANCHOR_W:
    case TK_ANCHOR_CENTER:
    case TK_ANCHOR_E:
      y1 -= bheight/2.0;
      break;

    case TK_ANCHOR_SW:
    case TK_ANCHOR_S:
    case TK_ANCHOR_SE:
      y1 -= bheight;
      break;
    }
    
    switch (latexPtr->anchor) {

    case TK_ANCHOR_NW:
    case TK_ANCHOR_W:
    case TK_ANCHOR_SW:
      break;

    case TK_ANCHOR_N:
    case TK_ANCHOR_CENTER:
    case TK_ANCHOR_S:
      x1 -= bwidth/2.0;
      break;

    case TK_ANCHOR_NE:
    case TK_ANCHOR_E:
    case TK_ANCHOR_SE:
      x1 -= bwidth;
      break;
    }

    x2 = x1 + bwidth;
    y2 = y1 + bwidth;

    /*
     * If the point is inside the rectangle, handle specially:
     * distance is 0 if rectangle is filled, otherwise compute
     * distance to nearest edge of rectangle and subtract width
     * of edge.
     */

    if ((pointPtr[0] >= x1) && (pointPtr[0] < x2)
	&& (pointPtr[1] >= y1) && (pointPtr[1] < y2)) {
      xDiff = pointPtr[0] - x1;
      tmp = x2 - pointPtr[0];
      if (tmp < xDiff) {
	xDiff = tmp;
      }
      yDiff = pointPtr[1] - y1;
      tmp = y2 - pointPtr[1];
      if (tmp < yDiff) {
	yDiff = tmp;
      }
      if (yDiff < xDiff) {
	xDiff = yDiff;
      }
      xDiff -= latexPtr->outlineWidth;
      if (xDiff < 0.0) {
	return 0.0;
      }
      return xDiff;
    }
    
    /*
     * Point is outside rectangle.
     */
    
    if (pointPtr[0] < x1) {
      xDiff = x1 - pointPtr[0];
    } else if (pointPtr[0] > x2)  {
      xDiff = pointPtr[0] - x2;
    } else {
      xDiff = 0;
    }
    
    if (pointPtr[1] < y1) {
      yDiff = y1 - pointPtr[1];
    } else if (pointPtr[1] > y2)  {
      yDiff = pointPtr[1] - y2;
    } else {
      yDiff = 0;
    }
    
    dist = hypot(xDiff, yDiff);
    if (dist < minDist)
      return(dist);
    else
      return(minDist);
}

/*
 *--------------------------------------------------------------
 *
 * LatexToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangle.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
LatexToArea(canvasPtr, itemPtr, rectPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item to check against rectangle. */
    double *rectPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    LatexItem *latexPtr = (LatexItem *) itemPtr;
    register TextLine *linePtr;
    int i, textRes, boxRes;
    double x1, y1, x2, y2, bwidth, bheight;

    /*
     * Scan the lines one at a time, seeing whether each line is
     * entirely in, entirely out, or overlapping the rectangle.  If
     * an overlap is detected, return immediately;  otherwise wait
     * until all lines have been processed and see if they were all
     * inside or all outside.
     */

    textRes = 0;
    for (linePtr = latexPtr->linePtr, i = latexPtr->numLines;
	    i > 0; linePtr++, i--) {
	if ((rectPtr[2] < linePtr->x1) || (rectPtr[0] > linePtr->x2)
		|| (rectPtr[3] < linePtr->y1) || (rectPtr[1] > linePtr->y2)) {
	    if (textRes == 1) {
		return 0;
	    }
	    textRes = -1;
	    continue;
	}
	if ((linePtr->x1 < rectPtr[0]) || (linePtr->x2 > rectPtr[2])
		|| (linePtr->y1 < rectPtr[1]) || (linePtr->y2 > rectPtr[3])) {
	    return 0;
	}
	if (textRes == -1) {
	    return 0;
	}
	textRes = 1;
    }

    /*  Consider the box now  */

    bwidth = PointToPixel(latexPtr->ptWidth);
    bheight = PointToPixel(latexPtr->ptHeight);
    x1 = latexPtr->bx;
    y1 = latexPtr->by;
      
    switch (latexPtr->anchor) {
      
    case TK_ANCHOR_NW:
    case TK_ANCHOR_N:
    case TK_ANCHOR_NE:
      break;

    case TK_ANCHOR_W:
    case TK_ANCHOR_CENTER:
    case TK_ANCHOR_E:
      y1 -= bheight/2.0;
      break;

    case TK_ANCHOR_SW:
    case TK_ANCHOR_S:
    case TK_ANCHOR_SE:
      y1 -= bheight;
      break;
    }
    
    switch (latexPtr->anchor) {

    case TK_ANCHOR_NW:
    case TK_ANCHOR_W:
    case TK_ANCHOR_SW:
      break;

    case TK_ANCHOR_N:
    case TK_ANCHOR_CENTER:
    case TK_ANCHOR_S:
      x1 -= bwidth/2.0;
      break;

    case TK_ANCHOR_NE:
    case TK_ANCHOR_E:
    case TK_ANCHOR_SE:
      x1 -= bwidth;
      break;
    }

    x2 = x1 + bwidth;
    y2 = y1 + bwidth;

    if ((rectPtr[2] <= x1)
	|| (rectPtr[0] >= x2)
            || (rectPtr[3] <= y1)
            || (rectPtr[1] >= y2)) {
        boxRes = -1;
	goto boxDone;
    }
    if ((rectPtr[0] >= x1)
	&& (rectPtr[1] >= y1)
	&& (rectPtr[2] <= x2)
	&& (rectPtr[3] <= y2)) {
      boxRes = -1;
      goto boxDone;
    }
    if ((rectPtr[0] <= x1)
	&& (rectPtr[1] <= y1)
	&& (rectPtr[2] >= x2)
	&& (rectPtr[3] >= y2)) {
      boxRes = 1;
      goto boxDone;
    }

    boxRes = 0;

  boxDone:
    if ((textRes == -1) && (boxRes == -1))
      return -1;
    if ((textRes == 1) && (boxRes == 1))
      return 1;

    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleLatex --
 *
 *	This procedure is invoked to rescale a latex item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Scales the position of the latex, but not the size
 *	of the font for the latex.
 *
 *--------------------------------------------------------------
 */

static void
ScaleLatex(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing rectangle. */
    Tk_Item *itemPtr;			/* Rectangle to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    register LatexItem *latexPtr = (LatexItem *) itemPtr;

    latexPtr->bx = originX + scaleX*(latexPtr->bx - originX);
    latexPtr->by = originY + scaleY*(latexPtr->by - originY);
    ComputeLatexBbox(canvasPtr, latexPtr);
    return;
}

/*
 *--------------------------------------------------------------
 *
 * TranslateLatex --
 *
 *	This procedure is called to move a latex item by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the latex item is offset by (xDelta, yDelta),
 *	and the bounding box is updated in the generic part of the
 *	item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateLatex(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    register LatexItem *latexPtr = (LatexItem *) itemPtr;

    latexPtr->bx += deltaX;
    latexPtr->by += deltaY;
    ComputeLatexBbox(canvasPtr, latexPtr);
}

/*
 *--------------------------------------------------------------
 *
 * LatexToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	latex items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
LatexToPostscript(canvasPtr, itemPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
    register LatexItem *latexPtr = (LatexItem *) itemPtr;
  }
