
/*
 * RXtPlayer.c --
 *      This module contains the functions to replay a script
 *      of action proceedures
 *
 * Copyright 1994 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <ctype.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include "send/tclXtSend.h"

#include "funcs.h"
#include "ReplayXt.h"

#define TM_NUM_PARAMS 32

/* 
 * RXt_TopWidget
 *
 *	a list of widgets that can be used as
 *	the root of a tree for naming purposes.
 *	This has to be used for multiple toplevel
 *	shells, but can be used for convenience
 */
typedef struct RXt_TopWidget {
    Widget widget;
    String name;
} RXt_TopWidget;

/*
 * RXt_Modifier
 *
 *	A table of the modifier symbols that are allowed
 *	- basically, I can't figure out how Xt does it in
 *	e.g. _XtLookupModifier, so this is my version
 *	Contains the mod name as in the -modifiers list
 *	with method for calculating the actual modifier
 *	in the RXt_ModifierProc
 *
 */

typedef Boolean (* RXt_ModifierProc) _ANSI_ARGS_ ((Display *display,
		unsigned int value, Modifiers *modifier));

static Boolean
ModifierImmediate _ANSI_ARGS_ ((Display *display, 
			unsigned int value, Modifiers *modifier));

static Boolean
ModifierLookup _ANSI_ARGS_ ((Display *display, 
			unsigned int value, Modifiers *modifier));

typedef struct RXt_Modifier {
    char *mod_name;
    RXt_ModifierProc find_modifier;
    unsigned int value;
} RXt_Modifier;


static RXt_Modifier modifier_table[] = {
    {"Shift",	ModifierImmediate,	ShiftMask},
    {"Lock",	ModifierImmediate,	LockMask},
    {"Control",	ModifierImmediate,	ControlMask},
    {"Mod1",	ModifierImmediate,	Mod1Mask},
    {"Mod2",	ModifierImmediate,	Mod2Mask},
    {"Mod3",	ModifierImmediate,	Mod3Mask},
    {"Mod4",	ModifierImmediate,	Mod4Mask},
    {"Mod5",	ModifierImmediate,	Mod5Mask},
    {"Meta", 	ModifierLookup,		XK_Meta_L},
    {"Meta", 	ModifierLookup,		XK_Meta_R},
    {"Alt", 	ModifierLookup,		XK_Alt_L},
    {"Alt", 	ModifierLookup,		XK_Alt_R},
    {"Hyper", 	ModifierLookup,		XK_Hyper_L},
    {"Hyper", 	ModifierLookup,		XK_Hyper_R},
    {"Super", 	ModifierLookup,		XK_Super_L},
    {"Super", 	ModifierLookup,		XK_Super_R},

    /* Motif has additional modifiers Copy, Link, etc but because
       they are "virtual", and are not in <keysymdef.h.> we can 
       ignore them
     */
};


/* 
 * Global variables
 */

static Tcl_Interp *interp;
static Boolean sleepDone;

static char *lib_cmd[] = { 
	"if [file exists ", 
	"] {source ", 
	"}"
};

Boolean RXt_verbose;

/* This is a list of the apps top level shells */
RXt_TopWidget	*RXt_top_widgets = NULL;
int	RXt_num_top_widgets = 0;

/* This says which top level shell to currently use */
int RXt_which_shell = 0;

/* 
 *--------------------------------------------------------------
 * RXt_GetTopShell
 *
 *	This function returns the current top level shell widget
 *--------------------------------------------------------------
 */
Widget
RXt_GetTopShell()
{
    return RXt_top_widgets[RXt_which_shell].widget;
}

/*
 *--------------------------------------------------------------
 *
 * ExtractArg --
 *	find a name and its value in a list
 *
 * Results:
 *	returns a pointer to the value
 *
 * Side effects:
 *	none
 *--------------------------------------------------------------
 */

static char *
ExtractArg(name, argc, argv)
    char *name;
    int argc;
    char *argv[];
{
   int n;

    for (n = 0; n < argc; n++) {
	if (strcmp(name, argv[n]) == 0) {
	    if (++n < argc) {
		return argv[n];
	    } else {
		return NULL;
	    }
	}
    }
    return NULL;
}

/*
 *--------------------------------------------------------------
 *
 * ModifierImmediate
 *
 *	return as modifier the value passed in
 *	do this for Shift, Control, Lock, Mod1, ..., Mod5
 *
 *--------------------------------------------------------------
 */
static Boolean 
ModifierImmediate(display, value, modifier)
    Display *display;
    unsigned int value;
    Modifiers *modifier;
{
    *modifier = value;
    return True;
}

/*
 *--------------------------------------------------------------
 *
 * ModifierLookup
 *
 *	return as modifier the value looked up in ModifierKeymap
 *	do this for Meta, Alt, etc
 *
 *--------------------------------------------------------------
 */
static Boolean 
ModifierLookup(display, keysym, modifier)
    Display *display;
    unsigned int keysym;
    Modifiers *modifier;
{
    XModifierKeymap *keymap;
    int m, n;
    KeyCode *keycodes;
    Cardinal keycode_count;

    XtKeysymToKeycodeList(display, keysym, &keycodes, &keycode_count);

    keymap = XGetModifierMapping(display);

    for (m = 0; m < keycode_count; m++) {
        for (n = 0; n < (8 * keymap->max_keypermod); n++) {
    	    if (keymap->modifiermap[n] == keycodes[m]) {
	        *modifier = 1 << (n / keymap->max_keypermod);
	        return True;
	    }
        }
    }
    return False;
}

/*
 *--------------------------------------------------------------
 *
 * MakeModifier
 *
 *	construct a modifier field for Key or Button event
 *	from the tcl list in value
 *
 *--------------------------------------------------------------
 */
static int
MakeModifiers(interp, display, value, state)
    Tcl_Interp *interp;
    Display *display;
    char *value;
    Modifiers *state;
{
    int n, len;
    char **list, **start;
    Modifiers modifier;

    Tcl_SplitList(interp, value, &len, &list);
    start = list;
    while (len-- > 0) {
        for (n = 0; n < XtNumber(modifier_table); n++) {
    	    if (strcmp(*list, modifier_table[n].mod_name) == 0) {
    	        /* convert our modifier string to a real modifier,
    	           using the registered function
    	         */
    	        if ( (* modifier_table[n].find_modifier) 
    			(display,
    			 modifier_table[n].value, 
    			 &modifier) ) {
    	            *state |= modifier;
    	            break;
    	        }
            }
        }

        /* failed to find it? */
        if (n == XtNumber(modifier_table)) {
            sprintf(interp->result, "Bad state value %s", *list);
            return TCL_ERROR;
        }
        list++;
    }
    free((char *) start);
}

/*
 *--------------------------------------------------------------
 *
 * MakeButtonEvent
 *
 *	construct a button event from the args passed in
 *
 *--------------------------------------------------------------
 */
static int
MakeButtonEvent(interp, display, xev, argc, argv)
    Tcl_Interp *interp;
    Display *display;
    XButtonEvent *xev;
    int argc;
    char **argv;
{
    struct timeval  tp;
    unsigned int state = 0;
    KeySym keysym;
    KeyCode keycode;
    char *value;
    int n;

    /* set defaults */
    xev->x = 0;
    xev->y = 0;
    xev->x_root = 0;
    xev->y_root = 0;
    xev->state = 0;
    xev->button = Button1;
    xev->same_screen = True;

    /* can't use clock() because we don't know timer resolution,
       and lots of systems don't have CLK_TCK or CLOCKS_PER_SEC .
       So do what Xt does:
     */
        gettimeofday(&tp, 0);
        xev->time = (tp.tv_sec * 1000) + (tp.tv_usec / 1000);

    if ((value = ExtractArg("-x", argc, argv)) != NULL) {
        xev->x = atoi(value);
    }

    if ((value = ExtractArg("-y", argc, argv)) != NULL) {
        xev->y = atoi(value);
    }

    if ((value = ExtractArg("-key", argc, argv)) != NULL) {
        /* lots of possiblilites here */
        /* only allow some for now */
	fprintf(stderr, "Warning: -key is obsolete: use -modifiers\n");
        if (strcmp(value,"Ctrl") == 0)
    	    xev->state = ControlMask;
        else
        if (strcmp(value,"Shift") == 0)
    	    xev->state = ShiftMask;
        else
        if (strcmp(value,"CtrlShift") == 0)
    	    xev->state = ShiftMask | ControlMask;
    }

    if ((value = ExtractArg("-button", argc, argv)) != NULL) {
        /* some form of pattern required here */
        if (strcmp(value, "Button1") == 0)
    	    xev->button = Button1;
        else
        if (strcmp(value, "Button2") == 0)
    	    xev->button = Button2;
        else
        if (strcmp(value, "Button3") == 0)
    	    xev->button = Button3;
        else
        if (strcmp(value, "Button4") == 0)
    	    xev->button = Button4;
        else
        if (strcmp(value, "Button5") == 0)
    	    xev->button = Button5;
        else {
    	    sprintf(interp->result, "illegal button %s", value);
    	    return TCL_ERROR;
        }
    }

    if ((value = ExtractArg("-modifiers", argc, argv)) != NULL) {
        if (MakeModifiers(interp, display, value, &state) ==
                TCL_ERROR)
            return TCL_ERROR;
    }
    xev->state = state;

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * MakeKeyEvent
 *
 *	construct a key event from the args passed in
 *
 *--------------------------------------------------------------
 */
static int
MakeKeyEvent(interp, display, xev, argc, argv)
    Tcl_Interp *interp;
    Display *display;
    XKeyEvent *xev;
    int argc;
    char **argv;
{
    struct timeval  tp;
    unsigned int state;
    KeySym keysym;
    KeyCode keycode;
    char *value;
    int n;

    /* set modifier state now because funny Motif stuff may change it */
    state = 0;

    /* can't use clock() because we don't know timer resolution,
       and lots of systems don't have CLK_TCK or CLOCKS_PER_SEC .
       So do what Xt does:
     */
        gettimeofday(&tp, 0);
        xev->time = (tp.tv_sec * 1000) + (tp.tv_usec / 1000);

    if ((value = ExtractArg("-x", argc, argv)) != NULL) {
        xev->x = atoi(value);
    } else {
        xev->x = 0;
    }

    if ((value = ExtractArg("-y", argc, argv)) != NULL) {
        xev->y = atoi(value);
    } else {
        xev->y = 0;
    }

    if ((value = ExtractArg("-keysym", argc, argv)) != NULL) {
	Cardinal num;
	KeyCode *keycodes;
	KeySym real_keysym;

        if ((keysym = XStringToKeysym(value)) == NoSymbol) {
    	    sprintf(interp->result, "illegal keysym\"%s\"", value);
    	    return TCL_ERROR;
        }

#ifdef MOTIF
	/* Motif uses virtual keysym that begin with "osf...".
	   These are not on the server side, so don't appear in
	   in the keycode <-> keysym tables. We have to make them
	   back into "real" keysyms before translating them to
	   Keycodes. Regrettably, I have to use an _Xm function.
	 */
	_XmVirtualToActualKeysym(display, keysym, &real_keysym, &state);
	if (real_keysym != NoSymbol)
	    keysym = real_keysym;
#endif /* MOTIF */

	XtKeysymToKeycodeList(display, keysym, &keycodes, &num);
        if (num == 0) {
    	    sprintf(interp->result, "no keycode for keysym \"%s\"", value);
    	    return TCL_ERROR;
        }

        xev->keycode = keycodes[0];
	XtFree(keycodes);
    } else {
        sprintf(interp->result, "Key event must have a keysym");
        return TCL_ERROR;
    }

    if ((value = ExtractArg("-modifiers", argc, argv)) != NULL) {
	if (MakeModifiers(interp, display, value, &state) ==
		TCL_ERROR)
	    return TCL_ERROR;
#if 0
	int len;
	char **list, **start;
	Modifiers modifier;

	Tcl_SplitList(interp, value, &len, &list);
	start = list;
	while (len-- > 0) {
	    for (n = 0; n < XtNumber(modifier_table); n++) {
		if (strcmp(*list, modifier_table[n].mod_name) == 0) {
		    /* convert our modifier string to a real modifier,
		       using the registered function
		     */
		    if ( (* modifier_table[n].find_modifier) 
				(display,
				 modifier_table[n].value, 
				 &modifier) ) {
		        state |= modifier;
		        break;
		    }
		}
	    }

	    /* failed to find it? */
	    if (n == XtNumber(modifier_table)) {
	        sprintf(interp->result, "Bad state value %s", *list);
	        return TCL_ERROR;
	    }
	    list++;
	}
	free((char *) start);
#endif
    }
    xev->state = state;

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * MakeXEvent --
 *
 *	Create an X event that can be used to send to another window
 *
 * Results:
 *	a Tcl result value
 *
 * Side effects:
 *	returns a filled in event in xev
 *
 * Restrictions:
 *	only handles a small number of X types
 *--------------------------------------------------------------
 */
static int
MakeXEvent(w, interp, xev, argc, argv)
    Widget w;
    Tcl_Interp *interp;
    XEvent *xev;
    int argc;
    char **argv;
{
    char *type;
    char *value;
    struct timeval  tp;
    Display *display;


    /* let them know we made this thing up */
    xev->xany.send_event = True;

    /* fill in the rest of the xany values.
     */
    xev->xany.display = display = XtDisplay(w);
    xev->xany.window = XtWindow(w);

    if ((type = ExtractArg("-type", argc, argv)) == NULL) {
        type = "ClientMessage";
    }

    if (type[0] == 'B') {
        if (strcmp(type, "ButtonRelease") == 0) {
	    xev->type = ButtonRelease;
	} else
	if (strcmp(type, "ButtonPress") == 0) {
	    xev->type = ButtonPress;
	} else
	if (strcmp(type, "ButtonMotion") == 0) {
	    xev->type = MotionNotify;
	} else {
	    sprintf(interp->result, "illegal type %s\n", type);
	    return TCL_ERROR;
	}

	return MakeButtonEvent(interp, display, 
				(XButtonEvent *) xev, argc, argv);
    }

    if (type[0] == 'K') {
	unsigned int mask;
	KeySym keysym;
	KeyCode keycode;

        if (strcmp(type, "KeyRelease") == 0) {
	    xev->type = KeyRelease;
	} else
	if (strcmp(type, "KeyPress") == 0) {
	    xev->type = KeyPress;
	} else {
	    sprintf(interp->result, "illegal type %s\n", type);
	    return TCL_ERROR;;
	}

	return MakeKeyEvent(interp, display, 
			(XKeyEvent *) xev, argc, argv);
    }

    if (strcmp(type, "ButtonPress") == 0) {
	xev->type = ButtonPress;
	return TCL_OK;
    }

    if (strcmp(type, "MotionNotify") == 0) {
	xev->type = MotionNotify;
	return TCL_OK;
    }

    if (strcmp(type, "EnterNotify") == 0) {
	xev->type = EnterNotify;
	return TCL_OK;
    }

    if (strcmp(type, "LeaveNotify") == 0) {
	xev->type = LeaveNotify;
	return TCL_OK;
    }

    if (strcmp(type, "FocusIn") == 0) {
	xev->type = FocusIn;
	return TCL_OK;
    }

    if (strcmp(type, "FocusOut") == 0) {
	xev->type = FocusOut;
	return TCL_OK;
    }

    if (strcmp(type, "KeymapNotify") == 0) {
	xev->type = KeymapNotify;
	return TCL_OK;
    }

    if (strcmp(type, "Expose") == 0) {
	xev->type = Expose;
	return TCL_OK;
    }

    if (strcmp(type, "GraphicsExpose") == 0) {
	xev->type = GraphicsExpose;
	return TCL_OK;
    }

    if (strcmp(type, "NoExpose") == 0) {
	xev->type = NoExpose;
	return TCL_OK;
    }

    if (strcmp(type, "VisibilityNotify") == 0) {
	xev->type = VisibilityNotify;
	return TCL_OK;
    }

    if (strcmp(type, "CreateNotify") == 0) {
	xev->type = CreateNotify;
	return TCL_OK;
    }

    if (strcmp(type, "DestroyNotify") == 0) {
	xev->type = DestroyNotify;
	return TCL_OK;
    }

    if (strcmp(type, "UnmapNotify") == 0) {
	xev->type = UnmapNotify;
	return TCL_OK;
    }

    if (strcmp(type, "MapNotify") == 0) {
	xev->type = MapNotify;
	return TCL_OK;
    }

    if (strcmp(type, "MapRequest") == 0) {
	xev->type = MapRequest;
	return TCL_OK;
    }

    if (strcmp(type, "ReparentNotify") == 0) {
	xev->type = ReparentNotify;
	return TCL_OK;
    }

    if (strcmp(type, "ConfigureNotify") == 0) {
	xev->type = ConfigureNotify;
	return TCL_OK;
    }

    if (strcmp(type, "ConfigureRequest") == 0) {
	xev->type = ConfigureRequest;
	return TCL_OK;
    }

    if (strcmp(type, "GravityNotify") == 0) {
	xev->type = GravityNotify;
	return TCL_OK;
    }

    if (strcmp(type, "ResizeRequest") == 0) {
	xev->type = ResizeRequest;
	return TCL_OK;
    }

    if (strcmp(type, "CirculateNotify") == 0) {
	xev->type = CirculateNotify;
	return TCL_OK;
    }

    if (strcmp(type, "CirculateRequest") == 0) {
	xev->type = CirculateRequest;
	return TCL_OK;
    }

    if (strcmp(type, "PropertyNotify") == 0) {
	xev->type = PropertyNotify;
	return TCL_OK;
    }

    if (strcmp(type, "SelectionClear") == 0) {
	xev->type = SelectionClear;
	return TCL_OK;
    }

    if (strcmp(type, "SelectionRequest") == 0) {
	xev->type = SelectionRequest;
	return TCL_OK;
    }

    if (strcmp(type, "SelectionNotify") == 0) {
	xev->type = SelectionNotify;
	return TCL_OK;
    }

    if (strcmp(type, "ColormapNotify") == 0) {
	xev->type = ColormapNotify;
	return TCL_OK;
    }

    if (strcmp(type, "ClientMessage") == 0) {
	xev->type = ClientMessage;
	return TCL_OK;
    }

    if (strcmp(type, "MappingNotify") == 0) {
	xev->type = MappingNotify;
	return TCL_OK;
    }

    if (strcmp(type, "ClientMessage") == 0) {
	return TCL_OK;
    }

    sprintf(interp->result, "illegal type %s\n", type);
    return TCL_ERROR;;
}

/*
 *--------------------------------------------------------------
 *
 * EndToken --
 *
 *	return the next char past a string or quoted string
 *	using the actions parameter syntax (Asente & Swick appendix B)
 *
 * Results:
 *	return pointer to next char after string
 *
 *--------------------------------------------------------------
 */

static char *
EndToken(s)
    char *s;
{
    if (s == NULL || *s == '\0')
	return s;

    /* quoted string */
    if (*s == '"') {
	s++;
	while (*s != '"' && *s != '\0')
	    s++;
	if (*s == '"')
	    return (s+1);
	else
	    return s;
    }

    /* unquoted string */
    while (*s != ' ' && *s != '\t' && *s != ',' &&
	   *s != '\n' && *s != ')' && *s != '\0')
	s++;
    return s;
}

/*
 *--------------------------------------------------------------
 *
 * ParseAction --
 *
 *	Parse an action into action name + args
 *
 * Results:
 *	sets the action, options and num_options
 *
 * Side-effects:
 * 	none
 *
 * Caveats:
 *	Somehow, this is an ugly piece of code. Tidy it up!
 *--------------------------------------------------------------
 */

static int
ParseAction(orig, action, params, num_params)
    char *orig;
    char **action;
    char *params[];
    Cardinal *num_params;
{
    int n = 0;
    char *start, *end;
#   define skipblanks(p) while (isspace(*p)) p++;

    skipblanks(orig);
    /* make a new string, and put nulls in it at the end of tokens */
    start = XtNewString(orig);
    *action = end = start;

    /* delimit action name */
    while (isalnum(*end) || *end == '_' || *end == '-')
	end++;
    if (*end != '(') {
	fprintf(stderr, "no \"(\" after name\n");
	return TCL_ERROR;
    }
    *end = '\0';

    start = end + 1;
    for (n = 0; n < TM_NUM_PARAMS; True) {
	skipblanks(start);
	if (*start == ')')
	    break;

	/* new token */
	end = EndToken(start);

	/* lose quotes if it was a quoted string */
	if (*start == '"') {
	    start++;
	    *(end - 1) = '\0';
	}
	params[n++] = start;

	if (isspace(*end)) {
	    *end++ = '\0';
	}
	skipblanks(end);
	if (*end == '\0') {
	    fprintf(stderr, "unterminated action\n");
	    return TCL_ERROR;
	}
	if (*end == ')') {
	    *end = '\0';
	    break;
	}
	if (*end != ',') {
	    fprintf(stderr, "no \",\" delimiter\n");
	    return TCL_ERROR;
	}
	*end = '\0';
	start = end + 1;
    }

    *num_params = n;
    return TCL_OK;
}

static void
EventLoop(clientData, interp, level, command, cmdProc,
          cmdClientData, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int level;
    char *command;
    Tcl_CmdProc *cmdProc;
    ClientData cmdClientData;
    int argc;
    char *argv[];
{
    XtAppContext appContext = (XtAppContext) clientData;

    while (XtAppPending(appContext)) {
	XtAppProcessEvent(appContext, XtIMAll);
    }
}

static void
EndSleep(clientData, timer)
    XtPointer clientData;
    XtIntervalId *timer;
{
	sleepDone = True;
}

static int
SetTopWidget(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char *argv[];
{
    int n;

    for (n = 0; n < RXt_num_top_widgets; n++) {
	if (strcmp(argv[1], RXt_top_widgets[n].name) == 0) {
	    RXt_which_shell = n;
    	    return TCL_OK;
	}
    }
    sprintf(interp->result, "Can't set top widget %s\n", argv[1]);
    printf("Can't set top widget %s\n", argv[1]);
    return TCL_ERROR;
}

static int
Sleep(clientData, interp, argc, argv)
    ClientData *clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Widget w = (Widget) clientData;
    Display *display = XtDisplay(w);
    XtAppContext appContext = XtWidgetToApplicationContext(w);
    int timeOut;

    Tcl_GetInt(interp, argv[1], &timeOut);
    if (RXt_verbose)
	printf("sleeping for %d\n", timeOut);
    XtAppAddTimeOut(appContext, timeOut, EndSleep, NULL);
    sleepDone = False;

    /* we are guaranteed at least one event: when the timer goes off */
    while (1) {
	XtAppProcessEvent(appContext, XtIMAll);
	if (sleepDone)
	    return TCL_OK;
    }
}

static int
CallActionProc(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{

        XEvent xev;
        char *action;
        char *params[TM_NUM_PARAMS];
        Cardinal num_params;
	Widget w;
	Widget toplevel;

        if (argc < 3) {
            sprintf(interp->result, "wrong # args: should be callActionProc \
widget action");
            return TCL_ERROR;
        }

	toplevel = (Widget) clientData;

	if ((w = XtNameToWidget(RXt_GetTopShell(), argv[1])) == NULL) {
	    sprintf(interp->result, "no such widget: %s", argv[1]);
	    return TCL_ERROR;
	}

        if (MakeXEvent(w, interp, &xev, argc-3, argv+3) == TCL_ERROR) {
	    Tcl_AppendResult(interp, "In MakeXEvent", NULL);
	    return TCL_ERROR;
	}
        if (ParseAction(argv[2], &action, params, &num_params) != TCL_OK) {
            sprintf(interp->result, "parse error in action \"%.50s\"", argv[2]);
            return TCL_ERROR;
        }
	
	if (RXt_verbose) {
	    printf("action: %s on widget: ", action);
	    RXt_PrintFullName(w);
	    printf("\n");
	}
        XtCallActionProc(w, action, &xev, params, num_params);
        

    return TCL_OK;
}

static int
WarpPointer(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Widget toplevel = (Widget) clientData;
    Widget w;
    Window win;
    int x, y;

    if (argc < 4) {
        sprintf(interp->result, "wrong # args: should be warpPointer \
widget x y");
        return TCL_ERROR;
    }

    if ((w = XtNameToWidget(RXt_GetTopShell(), argv[1])) == NULL) {
	sprintf(interp->result, "no window for %s", argv[1]);
	return TCL_ERROR;
    }

    if ((Tcl_GetInt(interp, argv[2], &x) == TCL_ERROR) ||
	(Tcl_GetInt(interp, argv[3], &y) == TCL_ERROR))
	return TCL_ERROR;

    win = XtWindowOfObject(w);

    XWarpPointer(XtDisplayOfObject(w), None, win, 0, 0, 0, 0, x, y);
    return TCL_OK;
}


static int
IsManaged(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{

    Widget w;

    if (argc < 2) {
        sprintf(interp->result, "wrong # args: should be isManaged widget");
        return TCL_ERROR;
    }

    if ((w = XtNameToWidget(RXt_GetTopShell(), argv[1])) == NULL) {
        sprintf(interp->result, "no such widget: %s", argv[1]);
        return TCL_ERROR;
    }

    if (XtIsManaged(w)) {
        interp->result = "true";
    } else {
        interp->result = "false";
    }
    return TCL_OK;
}

static int
IsRealized(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{

    Widget w;

    if (argc < 2) {
        sprintf(interp->result, "wrong # args: should be isRealized widget");
        return TCL_ERROR;
    }

    if ((w = XtNameToWidget(RXt_GetTopShell(), argv[1])) == NULL) {
        sprintf(interp->result, "no such widget: %s", argv[1]);
        return TCL_ERROR;
    }

    if (XtIsRealized(w)) {
        interp->result = "true";
    } else {
        interp->result = "false";
    }
    return TCL_OK;
}

static int
IsSensitive(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{

    Widget w;

    if (argc < 2) {
        sprintf(interp->result, "wrong # args: should be isRealized widget");
        return TCL_ERROR;
    }

    if ((w = XtNameToWidget(RXt_GetTopShell(), argv[1])) == NULL) {
        sprintf(interp->result, "no such widget: %s", argv[1]);
        return TCL_ERROR;
    }

    if (XtIsSensitive(w)) {
        interp->result = "true";
    } else {
        interp->result = "false";
    }
    return TCL_OK;
}

static void
StartPlayer(clientData, timer)
    XtPointer clientData;
    XtIntervalId *timer;
{
    char *data_file = (char *) clientData;

/*
    if (Tcl_VarEval(interp, lib_cmd[0], data_file, lib_cmd[1],
		data_file, lib_cmd[2], NULL) == TCL_ERROR) {
*/
    if (Tcl_EvalFile(interp, data_file) == TCL_ERROR) {
	fprintf(stderr, "Fatal error: %s\n", interp->result);
	exit(1);
    }
}

static void
RemoveWidget(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    int i,j;
    Boolean foundit = FALSE;
    RXt_TopWidget *newlist;

    if (RXt_top_widgets == NULL)
	return;

    for (i=0; i < RXt_num_top_widgets; i++) {
	if (w == RXt_top_widgets[i].widget) {
	    foundit = TRUE;
	    break;
	}
    }
    if (!foundit)
	return;

    if (RXt_num_top_widgets == 1) {
	free(RXt_top_widgets);
	RXt_top_widgets = NULL;
	RXt_num_top_widgets = 0;
	return;
    }

    newlist = (RXt_TopWidget *) XtMalloc(sizeof(RXt_TopWidget) *
					(RXt_num_top_widgets-1));
    j = 0;
    for (i=0; i < RXt_num_top_widgets; i++) {
	if (w != RXt_top_widgets[i].widget) {
	    newlist[j] = RXt_top_widgets[i];
	    j++;
	}
    }
    free(RXt_top_widgets);

    RXt_top_widgets = newlist;
    RXt_num_top_widgets--;
    return;
}

static Boolean
AddNewWidgetTop(w)
    Widget	w;
{
    RXt_top_widgets = (RXt_TopWidget *) XtRealloc((char *) RXt_top_widgets,
       					sizeof(RXt_TopWidget) * 
					(RXt_num_top_widgets+1));

    RXt_top_widgets[RXt_num_top_widgets].widget = w;
    RXt_top_widgets[RXt_num_top_widgets].name = XtName(w);
    RXt_num_top_widgets++;

    XtAddCallback(w, XtNdestroyCallback, RemoveWidget, NULL);
    return TRUE;
}

Boolean
RXt_RegisterTopWidget(w)
    Widget w;
{
    return AddNewWidgetTop(w);
}

Boolean
RXt_RegisterPlayer(toplevel, rules_file, data_file, verbose)
    Widget toplevel;
    char *rules_file;
    char *data_file;
    Boolean verbose;
{
    char *name;
    int fp;
    off_t size;
    struct stat stat_buf;
    int argc;
    String *argv;
    
    RXt_verbose = verbose;

    /* register this shell as the first */
    AddNewWidgetTop(toplevel);


    interp = Tcl_CreateInterp();

    /* set the toplevel shell in argv0 */
    if (XtIsSubclass(toplevel, applicationShellWidgetClass)) {
	XtVaGetValues(toplevel, XtNargv, &argv, XtNargc, &argc, NULL);
	if (argc > 0) {
	    Tcl_SetVar(interp, "argv0", argv[0], TCL_GLOBAL_ONLY);
	}
    }
    RXt_RegisterConverters(interp, XtWidgetToApplicationContext(toplevel));

    /* register all the new tcl commands */
    Tcl_CreateCommand(interp, "callActionProc", CallActionProc,
		toplevel, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "getValues", RXt_GetValues,
		toplevel, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "sleep", Sleep,
		toplevel, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "warpPointer", WarpPointer,
		toplevel, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "setTopWidget", SetTopWidget,
		toplevel, (Tcl_CmdDeleteProc *) NULL);

    /* XtIs... functions */
    Tcl_CreateCommand(interp, "isManaged", IsManaged,
		NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "isRealized", IsRealized,
		NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "isSensitive", IsSensitive,
		NULL, (Tcl_CmdDeleteProc *) NULL);

    /* load the rules file */
    Tcl_EvalFile(interp, rules_file);

    /* call to EventLoop before each tcl command is executed */
    Tcl_CreateTrace(interp, 10000, EventLoop,
			XtWidgetToApplicationContext(toplevel));

    /* start dataFile evaluation */
    XtAppAddTimeOut(XtWidgetToApplicationContext(toplevel),
		0, StartPlayer, data_file);

    name = XtName(toplevel);
    if (TclXtSend_RegisterInterp(interp, name, toplevel) == TCL_ERROR) {
	fprintf(stderr, "couldn't register interpreter %s\n", name);
	return False;
    }

    return True;
}
