/*
 * news.c
 *
 * Utilities for getting articles and article listings.
 */

#include <stdio.h>
#include <limits.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>

#include "news.h"

/* group_to_path()
 *
 * convert group name to a path name.  Note that this is done by
 * simply replacing all indices of the character array which have
 * a period with a slash instead. (i.e. no allocation is done)
 */
int
group_to_path(group)
char *group;
{
    register int i;
    char         *s;

    s = group;
    for (i = 0; i < strlen(group); i++, s++) 
    {
	if (*s == '.') 
	    *s = '/';
    }
}

int 
  path_to_group(path)
char *path;
{
  register int i;
  char         *s;

  s = path;
  for (i = 0; i < strlen(path); i++, s++)
      if (*s == '/')
	*s= '.';
}

/*
 * get_article()
 *
 * group - string indicating desired newsgroup.
 * num - desired article number
 *
 */
int
    get_article(group, num, buff)
char *group;
int  num;
char **buff;  /* will contain article when returned */
{
    register int fd;
    char         filename[255];
    FILE         *file;
    char         *ret;
    struct stat  file_info;
    

    group_to_path(group);
    sprintf(filename, "%s/%s/%d", SPOOL_DIR, group, num);
    file = fopen(filename, "r");
    if (file == (FILE *) NULL) 
    {
	*buff = (char *) NULL;
	printf("No news access possible.\n");
	return NEWS_NOACCESS;
    }
    fd = fileno(file);

    if (fstat(fd, &file_info)) 
    {
	/* can't stat it, we can't get it */
	*buff = (char *) NULL;
	return NEWS_NOSTAT;
    }
    *buff = (char *) malloc(file_info.st_size+1);
    if (read(fd, *buff, file_info.st_size) <= 0) 
    {
	free(*buff);
	*buff = (char *) NULL;
	printf("Cannot load article %s of group %s\n", num, group);
	return NEWS_NOREAD;
    }
    close(file);
    return NEWS_SUCCESS;
}
    
/* get_header_field()
 *
 * This procudure will retrieve the field begun by start,
 * and the subsequent lines composing it from the file pointer
 * passed in.
 */
char *
  get_header_field(start, fp)
char *start;
FILE *fp;
{
  char ret[MAX_HEADER_LINES][MAX_LINE_LENGTH];
  char line[MAX_LINE_LENGTH];
  int  length, count, index;

  count = index = 0;
  strcpy(ret[index++], start);
  fgets(line, MAX_LINE_LENGTH, fp);
/*  while (count) { 
    while (isspace(line[count])) count++;
  } */
}

/* get_article_list()
 *
 * This procedure returns an article_list pointer containing
 * the header info on the article indicated by filename.  
 *
 */

struct article_list *
  get_article_list(filename)
char *filename;
{
  FILE                *file;
  struct article_list *node; 
  char                line[MAX_LINE_LENGTH];
  register int        i,done=0;
  
  /* figure out the article number through the filename */
  i = strlen(filename);
  while (filename[--i] != '/');

  /* try and open the file, return NULL if we cannot */
  file = fopen(filename, "r");
  if (file == (FILE *) NULL) 
    {
      return (struct article_list *) NULL;
    }
  /* allocate the new node */
  node = (struct article_list *) malloc(sizeof(struct article_list)+1);
  node->subject = node->from = node->date = (char *) NULL;
  node->id = node->refs = node->followup = (char *) NULL;
  node->lines = -1;
  node->next = (struct article_list *) NULL;
  node->number = atoi(&filename[i+1]);
  while (!done) {
    if (!fgets(line, MAX_LINE_LENGTH, file)) {
      DEFAULT_HEADERS(node);
      done = 1;
      continue;
    }
    if (line[0] == '\n') {
      DEFAULT_HEADERS(node);
      done = 1;
      continue;
    }
    if (!strncasecmp(line, "Subject:", 8)) {
      if (strlen(&line[9]) > (MAX_LINE_LENGTH-1)) {
	node->subject = (char *) malloc(MAX_LINE_LENGTH+1);
	strncpy(node->subject, &line[9], MAX_LINE_LENGTH-1);
	node->subject[MAX_LINE_LENGTH-1] = (char) NULL;
      } else {
	node->subject = (char *) malloc(strlen(&line[9])+2);
	strcpy(node->subject, &line[9]);
      }
      node->subject[strlen(node->subject)-1] = (char) NULL;
    }
    if (!strncasecmp(line, "Message-ID:", 11)) {
      if (strlen(&line[12]) > (MAX_LINE_LENGTH-1)) {
	node->id = (char *) malloc(MAX_LINE_LENGTH+1);
	strncpy(node->id, &line[12], MAX_LINE_LENGTH-1);
	node->id[MAX_LINE_LENGTH-1] = (char) NULL;
      } else {
	node->id = (char *) malloc(strlen(&line[12])+2);
	strcpy(node->id, &line[12]);
      }
      node->id[strlen(node->id)-1] = (char) NULL;
    }
    if (!strncasecmp(line, "Followup-To:", 12)) {
      if (strlen(&line[13]) > (MAX_LINE_LENGTH-1)) {
	node->followup = (char *) malloc(MAX_LINE_LENGTH-1);
	strncpy(node->followup, &line[13], MAX_LINE_LENGTH-1);
	node->followup[MAX_LINE_LENGTH-1] = (char) NULL;
      } else {
	node->followup = (char *) malloc(strlen(&line[13])+2);
	strcpy(node->followup, &line[13]);
      }
      node->followup[strlen(node->followup)-1] = (char) NULL;
    }
    if (!strncasecmp(line, "References:", 11)) {
      if (strlen(&line[12]) > (MAX_LINE_LENGTH-1)) {
	node->refs = (char *) malloc(MAX_LINE_LENGTH-1);
	strncpy(node->refs, &line[12], MAX_LINE_LENGTH-1);
	node->refs[MAX_LINE_LENGTH-1] = (char) NULL;
      } else {
	node->refs = (char *) malloc(strlen(&line[12])+2);
	strcpy(node->refs, &line[12]);
      }
      node->refs[strlen(node->refs)-1] = (char) NULL;
    }
    if (!strncasecmp(line, "From:", 5)) {
      if (strlen(&line[6]) > (MAX_LINE_LENGTH-1)) {
	node->from = (char *) malloc(MAX_LINE_LENGTH+1);
	strncpy(node->from, &line[6], MAX_LINE_LENGTH-1);
	node->from[MAX_LINE_LENGTH-1] = (char) NULL;
      } else {
	node->from = (char *) malloc(strlen(&line[6])+1);
	strcpy(node->from, &line[6]);
      }
      node->from[strlen(node->from)-1] = (char) NULL;
    }
    if (!strncasecmp(line, "Date:", 5)) {
      if (strlen(&line[6]) > (MAX_LINE_LENGTH-1)) {
	node->date = (char *) malloc(MAX_LINE_LENGTH+1);
	strncpy(node->date, &line[6], MAX_LINE_LENGTH-1);
	node->date[MAX_LINE_LENGTH-1] = (char) NULL;
      } else {
	node->date = (char *) malloc(strlen(&line[6])+1);
	strcpy(node->date, &line[6]);
      }
      node->date[strlen(node->date)-1] = (char) NULL;
    }
    if (!strncasecmp(line, "Lines:", 6)) 
      node->lines = atoi(&line[7]);
  }
  fclose(file);
  return node;
}
	

/* article_list2char()
 *
 * This procedure takes the article list indicated and converts it
 * into one string.  Each node in the article list is allocated a line
 * describing it, and each such line is separated by a newline character.
 */
char *
  article_list2char(node)
struct article_list *node;
{
  register int i;
  register unsigned int size;
  char         *ret;
  char         *r;
  char         field_sep = (char) 4;
  char         line[AL_CHUNK][MAX_LINE_LENGTH+255];

  i = size = 0;
  ret = (char *) NULL;
  while(node != (struct article_list *) NULL) {
    sprintf(line[i], "%d%c%s%c%s%c%s%c%d%c%s%c%s%c%s\n", node->number, 
	    field_sep, node->subject, field_sep, node->from, field_sep, 
	    node->date, field_sep, node->lines, field_sep, node->id, 
	    field_sep, node->refs, field_sep, node->followup);
    size += strlen(line[i])+1;
    i++;
    if (i == AL_CHUNK || node->next == (struct article_list *) NULL) {
      int j = i;
      /* copy the data into the return pointer and reallocate if 
       * necessary.
       */
      if (ret == (char *) NULL) {
	ret = (char *) malloc(size);
	ret[0] = (char) NULL;
      } else {
	r = (char *) realloc(ret, size);
	ret = r;
      }
      for (i = 0; i < j; i++) 
	strcat(ret, line[i]);
      i = 0;
    }
    node = node->next;
  }
  return ret;
}


/* list_articles()
 *
 * This procedure generates an article_list for the newsgroup
 * indicated, from the range first-last.   
 */

int
    list_articles(group, first, last, ret) 
char                *group;
int                 first, last;
struct article_list **ret;
{
  char                basename[255];
  char                filename[255];
  struct article_list *node;
  struct article_list *tail;
  int                 i;

  *ret = (struct article_list *) NULL;
  tail = *ret;
  sprintf(basename, "%s/%s", SPOOL_DIR, group);
  group_to_path(basename);
  for (i = first; i <= last; i++) {
    sprintf(filename, "%s/%d", basename, i);
    node = (struct article_list *) get_article_list(filename);
    if (node == (struct article_list *) NULL) 
      continue;
    if (tail != (struct article_list *) NULL) 
      tail->next = node;
    else
      *ret = node;
    tail = node;
  }
  return NEWS_SUCCESS;
}

/* free_articlelist()
 *
 * This procedure frees up the memory used by the indicated article
 * list.
 */
int
  free_articlelist(list)
struct article_list *list;
{
  struct article_list *node;
  
  while (list != (struct article_list *) NULL) {
    node = list;
    if (node->subject != (char *) NULL)
      free(node->subject);
    if (node->from != (char *) NULL)
      free(node->from);
    if (node->date != (char *) NULL)
      free(node->date);
    if (node->id != (char *) NULL)
      free(node->id);
    if (node->refs != (char *) NULL)
      free(node->refs);
    free(node);
    list = list->next;
  }
}

/* add_leaf()
 *
 * adds a leaf to the tree indicated by root.
 */
int
  add_leaf(root, num)
struct tree_node *root;
int              num;
{
  struct tree_node *node, *old;
  struct tree_node *new;

  node = old = root;
  while (node != (struct tree_node *) NULL) {
    old = node;
    if (num < node->val)
      node = node->left;
    else
      node = node->right;
  }
  if (old == root) {
    root->left = root->right = NULL;
    root->val = num;
    return;
  }
  new = (struct tree_node *) malloc(sizeof(struct tree_node));
  new->left = new->right = NULL;
  new->val = num;
  if (num < old->val)
    old->left = new;
  else
    old->right = new;
}

/* group_stat()
 *
 * This procedure obtains the relevant information about a directory.
 * 
 */
int
  group_stat(name, group)
char             *name;
struct newsgroup **group;
{
  char             basename[255];
  char             filename[255];
  DIR              *dirp;
  struct dirent    *dp;
  int              first, last, num;
  struct list_node *tail, *new;

  group_to_path(name);
  sprintf(basename, "%s/%s", SPOOL_DIR, name);
  dirp = opendir(basename);
  if (dirp == (DIR *) NULL) {
    return NEWS_NOACCESS;
  }
  *group = (struct newsgroup *) malloc(sizeof(struct newsgroup));
  (*group)->name = (char *) malloc(strlen(name)+1);
  path_to_group(name);
  strcpy((*group)->name, name);
  (*group)->head = (struct list_node *) NULL;
  (*group)->first = 0;
  (*group)->last = 0;
  (*group)->num_articles = 0;
  last = num = 0;
  first = INT_MAX;
  dp = readdir(dirp);
  tail = (*group)->head;
  while (dp != (struct dirent *) NULL) {
    register int i;
    
    i = atoi(dp->d_name);
    if (i) {
      new = (struct list_node *) malloc(sizeof(struct list_node));
      new->number = i;
      new->next = (struct list_node *) NULL;
      if (tail == (struct list_node *) NULL) 
	tail = (*group)->head = new;
      else {
	tail->next = new;
	tail = new;
      }
      if (i < first)
	first = i;
      if (i > last)
	last = i;
      num++;
    }
    dp = readdir(dirp);
  }
  closedir(dirp);
  if (first != INT_MAX)
    (*group)->first = first;
  (*group)->last = last;
  (*group)->num_articles = num;
  return NEWS_SUCCESS;
}
      
/* get_field()
 *
 * This function obtains the first field found in the string
 * passed to it.  The field's contents are determined by the
 * indicated separator character. At the first appearance of the
 * separator character, the program places the field value in
 * the address passed to it, and returns the index of the
 * character just after the separator character.   Fields
 * longer than 255 characters are truncated.
 *
 */
 
int
  get_field(string, sep, sret, iret, do_malloc)
char *string;   /** string to parse through **/
char sep;      
char **sret;    /** where to put char value, null if int desired **/
int  *iret;     /** where to put int value, null if string desired **/
int  do_malloc; /** whether or not to malloc character space **/
{
  int i = 0;
  char holder[255];
  
  while ((string[i] != sep) && (string[i] != '\n') && (string[i] != '\0')) {
    holder[i] = string[i];
    i++;
  }
  holder[i] = '\0';
  if (iret != (int *) NULL) {
    *iret = atoi(holder);
    return i+1;
  }
  if(do_malloc) {
    *sret = (char *) malloc(strlen(holder)+1);
  }
  strcpy(*sret, holder);
  return i+1;
}

/* get_range()
 *
 * This procedure converts a subrange into first and last components
 *
 */

int 
  get_range(first, last, range)
int  *first, *last;
char *range;
{
  int num1, num2;
  int i;

  
  i = get_field(range, '-', (char **) NULL, &num1, 0);
  if (i < strlen(range)) 
    num2 = atoi(&range[i]);
  else
    num2 = num1;
  *first = num1;
  *last = num2;
}


      
/* get_unread()
 *
 * This procedure takes a group name and a range of articles read
 * and returns the number of unread articles for that group,
 * or -1 on error.
 */

int 
  get_unread(group, range)
struct newsgroup *group;
char *range;
{
  char *subrange;
  int i = 0;
  int j = 0;
  int num = 0;
  struct bounds *ranges, *r;
  int first, last;

  /* first determine how many ranges we have for the newsgroup */
  for (i = 0; i < strlen(range); i++) 
    if (range[i] == ',')
      j++;
  i = 0;
  ranges = (struct bounds *) malloc(sizeof(struct bounds)*j);
  r = ranges;
  while (i < strlen(range)) {
    i += get_field(&range[i], ',', &subrange, (int *) NULL, 1);
    get_range(&(r->first), &(r->last), subrange);
    free(subrange);
    r++;
  }
  num = group_search_range(ranges, j, group); 
  free(ranges);
  return (group->num_articles - num);
}

int
  group_search_range(ranges, num_ranges, group)
struct bounds    *ranges;
int              num_ranges;
struct newsgroup *group;
{
  register int     ret=0;
  register int     i=0;
  struct bounds    *r;
  struct list_node *node;

  node = group->head;
  while (node != (struct list_node *) NULL) {
    r = ranges;
    for (i=0; i < num_ranges; i++, r++) {
	if (node->number >= r->first && node->number <= r->last) {
	    printf("%d -> (%d-%d) \n",node->number, r->first, r->last);
	    fflush(stdout);
	    ret++;
	    continue;
	}
    }
    node = node->next;
  }
  return ret;
}

