# MenuButton
# ----------------------------------------------------------------------
# A MenuButton is a frame containing two menubuttons - one showing a bitmap
# and the other a textual label - that together (in theory) behave in unison
# as if a single menubutton.
#
# It is up to the caller to ensure that both buttons display the same menu.
# Since Tk demands that a menubutton's menu be a descendant of the button
# then either two identical menus are required, or the contents are generated
# on the fly prior to the menu being popped up.
# 
#
#   PUBLIC ATTRIBUTES:
#
#     -bitmap ........ the bitmap to be displayed in the lefthand button.
#                      Although a filename a leading @ is not required.
#
#     -text .......... the text to be displayed in the righthand button.
#
#     -menu .......... the last component of the menus' pathnames, i.e.
#                      $this.bitmap.$menu and $this.label.$menu are the
#                      menu pathnames.
#
#     -state ......... the state of the menubuttons.
#
#     This section is incomplete. Something probably ought to be done to
#     pass on some of the remaining menubutton options, e.g. textvariable,
#     while others make more sense applied to the frame, e.g. borderwidth.
#
#   METHODS:
#
#     None
#
#   X11 OPTION DATABASE ATTRIBUTES
#
#     bitmapFilePath .. list of paths to search for bitmap file, i.e
#                       -bitmap value, if it is not an absolute pathname.
#                       The default is at the bottom of this file.
# 
# ----------------------------------------------------------------------
#   AUTHOR:  Jim Wight        <j.k.wight@newcastle.ac.uk>
# ----------------------------------------------------------------------
# Copyright 1994 The University of Newcastle upon Tyne (see COPYRIGHT)
# ======================================================================

itcl_class MenuButton {
    inherit itcl_frame

    constructor {args} {
        eval itcl_frame::constructor $args

	menubutton $this.bitmap -bitmap [bitmap] -menu $this.bitmap.$menu \
	                                                        -padx 0 -pady 0
        bind $this.bitmap <Any-Enter> "[bind Menubutton <Any-Enter>]; \
                                           $this.label configure -state active"
        bind $this.bitmap <Any-Leave> "[bind Menubutton <Any-Leave>]; \
			                   $this.label configure -state normal"
	menubutton $this.label -text $text -menu $this.label.$menu \
	                                                        -padx 0 -pady 0
        bind $this.label <Any-Enter> "[bind Menubutton <Any-Enter>]; \
                                          $this.bitmap configure -state active"
        bind $this.label <Any-Leave> "[bind Menubutton <Any-Leave>]; \
			                  $this.bitmap configure -state normal"

	pack $this.bitmap $this.label -in $this -side left \
	                                                -fill both -expand yes
    }

    public bitmap {menu10} {
	if {[winfo exists $this.bitmap]} {
	    $this.bitmap configure -bitmap [bitmap]
	}
    }

    public text {} {
	if {[winfo exists $this.label]} {
	    $this.label configure -text $text
	}
    }

    public menu {} {
	if {[winfo exists $this.label]} {
	    $this.bitmap configure -menu $this.bitmap.$menu
	    $this.label configure -menu $this.label.$menu
	}
    }

    public state {} {
	if {[winfo exists $this.label]} {
	    $this.bitmap configure -state $state
	    $this.label configure -state $state
	}
    }

    method bitmap {} {
	global _menubutton_priv

	if {$bitmap == ""} {
	    return $bitmap
	}

	if {[string range $bitmap 0 0] == "/"} {
	    if {[file readable $bitmap]} {
		return @$bitmap
	    } else {
		error "MenuButton $this: $bitmap is not readable"
	    }
	}

	if {![catch {set _menubutton_priv($bitmap)}]} {
	    return [set _menubutton_priv($bitmap)]
	}

	if {[set path [option get $this bitmapFilePath ""]] == ""} {
	    return ""
	}

	set dirs [split $path :]
	foreach dir $dirs {
	    if {[file readable [set xbm $dir/$bitmap]]} {
		set _menubutton_priv($bitmap) @$xbm
		return @$xbm
	    }
	}

	error "MenuButton $this: $bitmap not found in $dirs"
    }
}

option add *bitmapFilePath \
      /usr/include/X11/bitmaps:/usr/local/X11/include/X11/bitmaps widgetDefault
