# Paned
# ----------------------------------------------------------------------
# Implements a paned widget, similar to the Paned widget of the Athena 
# Widget Set. 
#
#   PUBLIC ATTRIBUTES:
#
#     -orient ........ orientation of widget
#     -relgrip ....... relative position of grips (between 0.0 and 1.0)
#
#   METHODS:
#
#     insert ......... add a child or children to the widget. If the option
#                      -noexpand is an argument then the child[ren] will not
#                      resize when the widget is resized.
#
#                      The user is responsible for destroying the children
#                      unless they are descendants of Paned, in which case
#                      destroying the Paned will automatically destroy them.
#
#   BUGS:
#
#     This widget does not behave particularly well after resizing in that
#     the panes do not adjust correctly until the grips have been used a
#     few times. If anyone has an explanation for the behaviour I would be
#     most interested in hearing it.
#
# ----------------------------------------------------------------------
#   AUTHOR:  Jim Wight <j.k.wight@newcastle.ac.uk>
# ----------------------------------------------------------------------
# Copyright 1994 The University of Newcastle upon Tyne (see COPYRIGHT)
# ======================================================================

itcl_class Paned {
    inherit itcl_frame

    constructor {args} {
	eval itcl_frame::constructor $args

	frame ${this}sash -$horw 2 -borderwidth 1 -relief raised

	bind $this <Map> "$this map"

	set configured 1
    }

    destructor {
	destroy ${this}sash
	for {set child 2} {$child <= $numChildren}  {incr child} {
	    destroy ${this}grip$child
	}
    }

    # ------------------------------------------------------------------
    #  METHOD insert - Add a child to the pane
    # ------------------------------------------------------------------
    method insert {args} {
	if {[set rigid [lsearch $args "-noexpand"]] != -1} {
	    set args [lreplace $args $rigid $rigid]
	    set expand false
	} else {
	    set expand true
	}

	foreach child $args {
	    incr numChildren  

	    frame $this.f$numChildren

	    place $child \
	        -in $this.f$numChildren \
		-x 0 -y 0 \
		-anchor nw \
		-relwidth 1 -relheight 1

	    pack $this.f$numChildren \
	        -in $this \
		-side $side \
		-fill both -expand $expand
	    lower $this.f$numChildren

	    if {$numChildren > 1} {
		frame ${this}grip$numChildren \
		    -geometry 10x10 \
		    -borderwidth 2 \
		    -relief raised -cursor crosshair

                bind ${this}grip$numChildren <Button-1> \
		    "$this grab-grip $numChildren %$yorx"
                bind ${this}grip$numChildren <B1-Motion> \
		    "$this drag-grip $numChildren %$yorx"
                bind ${this}grip$numChildren <B1-ButtonRelease-1> \
		    "$this drop-grip $numChildren %$yorx"
	    }

	    $this.f$numChildren configure \
	        -width [winfo reqwidth $child] \
		-height [winfo reqheight $child]
	}
    }

    # ------------------------------------------------------------------
    #  PUBLIC variables
    # ------------------------------------------------------------------
    public relgrip {} {
	set rel $relgrip
	if {$configured} {
	    re-grip
	}
    }

    public orient vertical {
	if {!$configured && $orient == "horizontal"} {
	    set side   left
	    set anchor n
	    set xory   y
	    set yorx   x
	    set horw   width
	    set worh   height
	    set rel    .1
	}
    }


    # ------------------------------------------------------------------
    #  Everything is for private use from here on down


    # ------------------------------------------------------------------
    #  METHOD grab-grip - Starts the sash drag and drop operation
    # ------------------------------------------------------------------
    method grab-grip {child where} {
	global mindrag maxdrag

	if {$child == 2} {
	    set mindrag 0
	} else {
	    set mindrag [winfo $yorx ${this}grip[expr $child - 1]]
	}

	if {$child == $numChildren} {
	    set maxdrag [winfo $horw $this]
	} else {
	    set maxdrag [winfo $yorx ${this}grip[expr $child + 1]]
	}

	set position [expr [winfo $yorx ${this}grip$child] + $where]
	${this}sash configure -$worh [winfo $worh $this.f$child]

	place ${this}sash -in $this -$xory 0 -$yorx $position -anchor $anchor
	raise ${this}sash
	raise ${this}grip$child
	grab ${this}grip$child
	${this}grip$child configure -relief sunken
    }

    # ------------------------------------------------------------------
    #  METHOD drag-grip - Motion action for sash
    # ------------------------------------------------------------------
    method drag-grip {child where} {
	global mindrag maxdrag

	set position [expr [winfo $yorx ${this}grip$child] + $where]
	if {$position > $mindrag && $position < $maxdrag} {
	    place ${this}sash \
	        -in $this \
		-$xory 0 \
		-$yorx $position \
		-anchor $anchor

	    place ${this}grip$child \
	        -in $this \
		-rel$xory $rel \
		-$yorx $position \
		-anchor center
	}
    }

    # ------------------------------------------------------------------
    #  METHOD drop-grip - Ends the sash drag and drop operation
    # ------------------------------------------------------------------
    method drop-grip {child where} {
	set position [expr [winfo $yorx ${this}grip$child] + $where]
	grab release ${this}grip$child
	${this}grip$child configure -relief raised

	set lower $this.f[expr $child - 1]
	set higher $this.f$child
	set bothh [expr [winfo $horw $lower] + [winfo $horw $higher]]
	set prevh [expr $position - [winfo $yorx $lower]]

	$lower configure -$horw $prevh
	$higher configure -$horw [expr $bothh - $prevh]
	lower ${this}sash

	place ${this}grip$child \
	    -in $this \
	    -rel$xory $rel \
	    -$yorx $position \
	    -anchor center
    }

    # ------------------------------------------------------------------
    #  METHOD re-grip - Reposition the grips after an adjustment
    # ------------------------------------------------------------------
    method re-grip {} {
	update
	for {set child 2}  {$child <= $numChildren}  {incr child} {
	    place ${this}grip$child \
	        -in $this \
		-$yorx [winfo $yorx $this.f$child] \
		-rel$xory $rel \
		-anchor center

	        place ${this}sash \
		   -in $this \
		   -$xory 0 \
		   -$yorx [winfo $yorx $this.f$child] \
		   -anchor $anchor
	}
    }

    # ------------------------------------------------------------------
    #  METHOD map - Initial positioning of the grips and setup for resizes
    # ------------------------------------------------------------------
    method map {} {
	re-grip
	bind $this <Map> {#}
	bind $this <Configure> "$this re-grip"
    }

    # ------------------------------------------------------------------
    #  PROTECTED variables
    # ------------------------------------------------------------------
    protected mindrag
    protected maxdrag
    protected side        top
    protected anchor      w
    protected xory        x
    protected yorx        y
    protected horw        height
    protected worh        width
    protected rel         .9 
    protected numChildren 0
    protected configured  0
}
