/* 
 * tkCanvIcon.c --
 *
 *	This file implements icon items for canvas widgets.
 *
 * Copyright (c) 1993 by Sven Delmas
 * All rights reserved.
 * See the file COPYRIGHT for the copyright notes.
 *
 *
 * This source is based upon the file tkCanvBmap.c from:
 *
 * John Ousterhout
 *
 * Copyright (c) 1992-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#ifndef lint
static char *AtFSid = "$Header: tkCanvIcon.c[1.17] Wed Oct 27 23:46:38 1993 garfield@garfield accessed $";
#endif

#include <stdio.h>
#include <math.h>
#include "tkInt.h"
#include "tkCanvas.h"
#include "tkConfig.h"

/*
 * The distance between bitmap and text.
 */
#define ICON_OFFSET 2

/*
 * The structure below defines the record for each icon item.
 */
typedef struct IconItem  {
  Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
  double x, y;		        /* Coordinates of positioning point
                                 * for icon. */
  unsigned int bitmapX;         /* Coordinates of positioning point */
  unsigned int bitmapY; 	/* for text. */
  unsigned int textX;           /* Coordinates of positioning point */
  unsigned int textY;	        /* for text. */
  unsigned int textWidth;       /* Width and height of the text. */
  unsigned int textHeight;
  char *command;		/* Command for item, usually evaluated
                                 * on double-click. */
  char *dropCommand;		/* Drop command for item, usually */
				/* evaluated on drop. */
  char *menu1;	                /* Standard menu for item, usually
                                 * activated with button-3. */
  char *menu2;		        /* Alternative menu for item, usually
                                 * activated with meta-button-3. */
  char *menu3;		        /* Alternative menu for item, usually
                                 * activated with control-button-3. */
  char *name;		        /* Name for item. */
  char *sortString;		/* The string that is used for
                                 * sorting, usually the same value as
                                 * stored in -name. */
  char *state;		        /* State of item, this value is used
                                 * to represent the selection status
                                 * (normal, selected). */
  char *type;		        /* Type for item. This information is
                                 * used to load the icon bitmap. */
  char *edges;	                /* Edges bound to this icon, this is a
                                 * list of thw form {{fromId toId
                                 * edgeId}...}. */
  char *layoutInfo;             /* This option holds permanent
				 * layouting information. */
  
  int borderwidth;		/* Width of outline (in pixels). */
  XColor *outlineColor;	        /* Color for outline.  NULL means don't
			         * draw outline. */
  int relief;			/* 3-d effect: TK_RELIEF_RAISED, etc. */
  
  Pixmap bitmap;		/* Bitmap to display in window. */
  Pixmap clipMask;		/* Bitmap to specify the clipping mask
				 * for transparent pixmaps, or None. */
  
  char *status;		        /* Status of item. */
  Pixmap statusBitmap;	        /* Bitmap to display status in window. */
  Pixmap statusClipMask;	/* Bitmap to specify the clipping mask
				 * for transparent pixmaps, or None. */
  
  Tk_Anchor textAnchor;	        /* Where to anchor text relative to
			           * bitmap. */
  char *text;			/* Text for item. */
  XFontStruct *fontPtr;	        /* Font for drawing text. */
  
  XColor *fgColor;		/* Foreground color to use for icon. */
  XColor *bgColor;		/* Background color to use for icon. */
  GC gc;			/* Graphics context to use for drawing
				 * icon on screen. */
  GC invertedGc;		/* Graphics context to use for drawing
				 * icon on screen. */
  GC outlineGc;		        /* Graphics context for outline. */
  unsigned int intWidth;        /* Internal width and height of icon. */
  unsigned int intHeight; 
  unsigned int intStatusWidth;  /* Internal width and height of statusicon. */
  unsigned int intStatusHeight;
  int width;
} IconItem;

/*
 * Information used for parsing configuration specs:
 */
static Tk_ConfigSpec configSpecs[] = {
  {TK_CONFIG_COLOR, "-background", (char *) NULL, (char *) NULL,
     (char *) NULL, Tk_Offset(IconItem, bgColor), TK_CONFIG_NULL_OK},
  {TK_CONFIG_PIXMAP, "-bitmap", (char *) NULL, (char *) NULL,
     (char *) NULL, Tk_Offset(IconItem, bitmap), TK_CONFIG_NULL_OK},
  {TK_CONFIG_INT, "-borderwidth", (char *) NULL, (char *) NULL,
     "0", Tk_Offset(IconItem, borderwidth), TK_CONFIG_DONT_SET_DEFAULT},
  {TK_CONFIG_BITMAP, "-mask", (char *) NULL, (char *) NULL,
     (char *) NULL, Tk_Offset(IconItem, clipMask), TK_CONFIG_NULL_OK},
  {TK_CONFIG_STRING, "-command", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, command), 0},
  {TK_CONFIG_STRING, "-dropcommand", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, dropCommand), 0},
  {TK_CONFIG_STRING, "-edges", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, edges), 0},
  {TK_CONFIG_FONT, "-font", (char *) NULL, (char *) NULL,
     "-Adobe-Helvetica-Bold-R-Normal--*-120-*",
     Tk_Offset(IconItem, fontPtr), 0},
  {TK_CONFIG_COLOR, "-foreground", (char *) NULL, (char *) NULL,
     "black", Tk_Offset(IconItem, fgColor), 0},
  {TK_CONFIG_STRING, "-layoutinfo", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, layoutInfo), 0},
  {TK_CONFIG_STRING, "-menu1", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, menu1), 0},
  {TK_CONFIG_STRING, "-menu2", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, menu2), 0},
  {TK_CONFIG_STRING, "-menu3", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, menu3), 0},
  {TK_CONFIG_STRING, "-name", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, name), 0},
  {TK_CONFIG_COLOR, "-outline", (char *) NULL, (char *) NULL,
     "black", Tk_Offset(IconItem, outlineColor), TK_CONFIG_NULL_OK},
  {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
     "flat", Tk_Offset(IconItem, relief), 0},
  {TK_CONFIG_STRING, "-sortstring", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, sortString), 0},
  {TK_CONFIG_STRING, "-state", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, state), 0},
  {TK_CONFIG_STRING, "-status", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, status), 0},
  {TK_CONFIG_PIXMAP, "-statusbitmap", (char *) NULL, (char *) NULL,
     (char *) NULL, Tk_Offset(IconItem, statusBitmap),
     TK_CONFIG_NULL_OK},
  {TK_CONFIG_BITMAP, "-statusmask", (char *) NULL, (char *) NULL,
     (char *) NULL, Tk_Offset(IconItem, statusClipMask),
     TK_CONFIG_NULL_OK},
  {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
  {TK_CONFIG_ANCHOR, "-textanchor", (char *) NULL, (char *) NULL,
     "center", Tk_Offset(IconItem, textAnchor),
     TK_CONFIG_DONT_SET_DEFAULT},
  {TK_CONFIG_STRING, "-text", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, text), 0},
  {TK_CONFIG_STRING, "-type", (char *) NULL, (char *) NULL,
     "", Tk_Offset(IconItem, type), 0},
  {TK_CONFIG_INT, "-width", (char *) NULL, (char *) NULL,
     "0", Tk_Offset(IconItem, width), TK_CONFIG_DONT_SET_DEFAULT},
  {TK_CONFIG_DOUBLE, "-x", (char *) NULL, (char *) NULL,
     "0", Tk_Offset(IconItem, x), TK_CONFIG_DONT_SET_DEFAULT},
  {TK_CONFIG_DOUBLE, "-y", (char *) NULL, (char *) NULL,
     "0", Tk_Offset(IconItem, y), TK_CONFIG_DONT_SET_DEFAULT},
  {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
     (char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */
static int		IconCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, int argc, char **argv));
static int		IconToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *rectPtr));
static double		IconToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *coordPtr));
static int		IconToPostscript _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Tk_PostscriptInfo *psInfoPtr));
static void		ComputeIconBbox _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    IconItem *bmapPtr));
static int		ConfigureIcon _ANSI_ARGS_((
			    Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateIcon _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    struct Tk_Item *itemPtr, int argc, char **argv));
static void		DeleteIcon _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr));
static void		DisplayIcon _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Drawable dst));
static void		ScaleIcon _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static void		TranslateIcon _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the rectangle and oval item types
 * by means of procedures that can be invoked by generic item code.
 */
Tk_ItemType TkIconType = {
  "icon",				/* name */
  sizeof(IconItem),			/* itemSize */
  CreateIcon,			        /* createProc */
  configSpecs,			        /* configSpecs */
  ConfigureIcon,			/* configureProc */
  IconCoords,			        /* coordProc */
  DeleteIcon,			        /* deleteProc */
  DisplayIcon,			        /* displayProc */
  0,					/* alwaysRedraw */
  IconToPoint,			        /* pointProc */
  IconToArea,			        /* areaProc */
  IconToPostscript,			/* postscriptProc */
  ScaleIcon,			        /* scaleProc */
  TranslateIcon,			/* translateProc */
  (Tk_ItemIndexProc *) NULL,		/* indexProc */
  (Tk_ItemCursorProc *) NULL,		/* icursorProc */
  (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
  (Tk_ItemInsertProc *) NULL,		/* insertProc */
  (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
  (Tk_ItemType *) NULL		        /* nextPtr */
};

/*
 *--------------------------------------------------------------
 *
 * CreateIcon --
 *
 *	This procedure is invoked to create a new icon
 *	item in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized, so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new icon item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateIcon(canvasPtr, itemPtr, argc, argv)
     register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
     Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
     int argc;				/* Number of arguments in argv. */
     char **argv;			/* Arguments describing rectangle. */
{
  register IconItem *iconPtr = (IconItem *) itemPtr;

  if (argc < 2 && argc != 0) {
    Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		     Tk_PathName(canvasPtr->tkwin), "\" create ",
		     itemPtr->typePtr->name, " x y ?options?",
		     (char *) NULL);
    return TCL_ERROR;
  }
  
  /*
   * Initialize item's record.
   */
  iconPtr->bgColor = NULL;
  iconPtr->bitmap = None;
  iconPtr->borderwidth = 0;
  iconPtr->clipMask = None;
  iconPtr->command = NULL;
  iconPtr->dropCommand = NULL;
  iconPtr->edges = NULL;
  iconPtr->fgColor = NULL;
  iconPtr->fontPtr = NULL;
  iconPtr->layoutInfo = NULL;
  iconPtr->menu1 = NULL;
  iconPtr->menu2 = NULL;
  iconPtr->menu3 = NULL;
  iconPtr->name = NULL;
  iconPtr->outlineColor = NULL;
  iconPtr->relief = TK_RELIEF_FLAT;
  iconPtr->sortString = NULL;
  iconPtr->state = NULL;
  iconPtr->status = NULL;
  iconPtr->statusBitmap = None;
  iconPtr->statusClipMask = None;
  iconPtr->text = NULL;
  iconPtr->textAnchor = TK_ANCHOR_S;
  iconPtr->type = NULL;
  iconPtr->width = 0;
  
  iconPtr->gc = None;
  iconPtr->invertedGc = None;
  iconPtr->outlineGc = None;
  iconPtr->intWidth = 0;
  iconPtr->intHeight = 0;

  /*
   * Process the arguments to fill in the item record.
   */
  if (argc >= 2) {
    /* did we get a default x and y position as first two parameters ? */
    if ((argv[0][0] == '-' && isdigit(argv[0][1])) || isdigit(argv[0][0])) {
      /* the first two parameters are numbers... so use them as x and */
      /* y coordinates. */
      if ((TkGetCanvasCoord(canvasPtr, argv[0], &iconPtr->x) != TCL_OK) ||
	  (TkGetCanvasCoord(canvasPtr, argv[1], &iconPtr->y) != TCL_OK)) {
	return TCL_ERROR;
      }
      
      if (ConfigureIcon(canvasPtr, itemPtr, argc-2, argv+2, 0) != TCL_OK) {
	DeleteIcon(canvasPtr, itemPtr);
	return TCL_ERROR;
      }
    } else {
      if (ConfigureIcon(canvasPtr, itemPtr, argc, argv, 0) != TCL_OK) {
	DeleteIcon(canvasPtr, itemPtr);
	return TCL_ERROR;
      }
    }
  }
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * IconCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on icon items.  See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
IconCoords(canvasPtr, itemPtr, argc, argv)
     register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
     Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
     int argc;				/* Number of coordinates supplied in
					 * argv. */
     char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
  register IconItem *iconPtr = (IconItem *) itemPtr;
  
  if (argc == 0) {
    sprintf(canvasPtr->interp->result, "%g %g", iconPtr->x, iconPtr->y);
  } else if (argc == 2) {
    if ((TkGetCanvasCoord(canvasPtr, argv[0], &iconPtr->x) != TCL_OK) ||
	(TkGetCanvasCoord(canvasPtr, argv[1], &iconPtr->y) != TCL_OK)) {
      return TCL_ERROR;
    }
    ComputeIconBbox(canvasPtr, iconPtr);
  } else {
    sprintf(canvasPtr->interp->result,
	    "wrong # coordinates:  expected 0 or 2, got %d", argc);
    return TCL_ERROR;
  }
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureIcon --
 *
 *	This procedure is invoked to configure various aspects
 *	of a icon item, such as its anchor position.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Configuration information may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureIcon(canvasPtr, itemPtr, argc, argv, flags)
     Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
     Tk_Item *itemPtr;		/* Icon item to reconfigure. */
     int argc;			/* Number of elements in argv.  */
     char **argv;		/* Arguments describing things to configure. */
     int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
  register IconItem *iconPtr = (IconItem *) itemPtr;
  XGCValues gcValues;
  GC newGC;
  unsigned long mask;
  char *value, *fullName, **list;
  int counter, listCounter = 0;
  Tcl_DString varName, fileName, getFileName, buffer, oldType, oldStatus;

  /* store the current (old) type/status of the icon. This value is */
  /* used later, when we update the bitmap and the status. */
  Tcl_DStringInit(&oldType);
  if (iconPtr->type != NULL && strlen(iconPtr->type) > (size_t) 0) {
    Tcl_DStringAppend(&oldType, iconPtr->type, -1);
  }
  Tcl_DStringInit(&oldStatus);
  if (iconPtr->status != NULL && strlen(iconPtr->status) > (size_t) 0) {
    Tcl_DStringAppend(&oldStatus, iconPtr->status, -1);
  }

  if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
			 configSpecs, argc, argv,
			 (char *) iconPtr, flags) != TCL_OK) {
    Tcl_DStringFree(&oldType);
    Tcl_DStringFree(&oldStatus);
    return TCL_ERROR;
  }

  /*
   * A few of the options require additional processing, such as those
   * that determine the graphics context.
   */
  /* no negative borders */
  if (iconPtr->borderwidth < 0) {
    iconPtr->borderwidth = 0;
  }

  /* the standard gc */
  gcValues.foreground = iconPtr->fgColor->pixel;
  if (iconPtr->bgColor != NULL) {
    gcValues.background = iconPtr->bgColor->pixel;
  } else {
    gcValues.background = Tk_3DBorderColor(canvasPtr->bgBorder)->pixel;
  }
  mask = GCForeground | GCBackground;
  if ((iconPtr->fgColor != NULL) && (iconPtr->fontPtr != NULL)) {
    gcValues.font = iconPtr->fontPtr->fid;
    mask |= GCFont;
  }
  newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
  if (iconPtr->gc != None) {
    Tk_FreeGC(canvasPtr->display, iconPtr->gc);
  }
  iconPtr->gc = newGC;

  /* the inverted display gc */
  gcValues.background = iconPtr->fgColor->pixel;
  if (iconPtr->bgColor != NULL) {
    gcValues.foreground = iconPtr->bgColor->pixel;
  } else {
    gcValues.foreground = Tk_3DBorderColor(canvasPtr->bgBorder)->pixel;
  }
  mask = GCForeground | GCBackground;
  if ((iconPtr->fgColor != NULL) && (iconPtr->fontPtr != NULL)) {
    gcValues.font = iconPtr->fontPtr->fid;
    mask |= GCFont;
  }
  newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
  if (iconPtr->invertedGc != None) {
    Tk_FreeGC(canvasPtr->display, iconPtr->invertedGc);
  }
  iconPtr->invertedGc = newGC;

  /* the outline color gc */
  if (iconPtr->outlineColor == NULL) {
    newGC = None;
  } else {
    gcValues.foreground = iconPtr->outlineColor->pixel;
    gcValues.cap_style = CapProjecting;
    gcValues.line_width = iconPtr->borderwidth;
    mask = GCForeground|GCCapStyle|GCLineWidth;
    newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
  }
  if (iconPtr->outlineGc != None) {
    Tk_FreeGC(canvasPtr->display, iconPtr->outlineGc);
  }
  iconPtr->outlineGc = newGC;

  /* do we have a new type ? We only automatically load a new bitmap */
  /* when the icontype was changed. This allows it to set the bitmap */
  /* directly with the -bitmap option even when the icon has a type. */
  /* When the type is changed to empty, the old bitmap will remain... */
  /* this is a decision I make, maybe a wrong decision (time will show). */
  if (iconPtr->type != NULL && strlen(iconPtr->type) > (size_t) 0 &&
      ((oldType.string != NULL && strcmp(iconPtr->type, oldType.string) != 0)
       || oldType.string == NULL)) {
    /* the following code retrieves the path list for the icons. This */
    /* is done because I don't want to attatch the pathname list to */
    /* each icon. */
    Tcl_DStringInit(&varName);
    Tcl_DStringAppend(&varName, "ip_priv(", -1);
    Tcl_DStringAppend(&varName, Tk_PathName(canvasPtr->tkwin), -1);
    Tcl_DStringAppend(&varName, ",iconpixmappath)", -1);
    if ((value = Tcl_GetVar(canvasPtr->interp, varName.string,
			    TCL_GLOBAL_ONLY)) != NULL) {
      if (Tcl_SplitList(canvasPtr->interp, value, &listCounter,
			&list) == TCL_OK) {
	/* walk through list of pathnames. */
	for (counter = 0; counter < listCounter; counter++) {
	  /* create the filename to load. */
	  Tcl_DStringInit(&fileName);
	  Tcl_DStringAppend(&fileName, list[counter], -1);
	  Tcl_DStringAppend(&fileName, "/", -1);
	  Tcl_DStringAppend(&fileName, iconPtr->type, -1);
	  Tcl_DStringAppend(&fileName, ".icon", -1);
	  Tcl_DStringInit(&buffer);
	  fullName = Tcl_TildeSubst(canvasPtr->interp,
				    fileName.string, &buffer);
	  /* does the file exist ? */
	  if (access(fullName, F_OK) != -1) {
	    Tcl_DStringInit(&getFileName);
	    Tcl_DStringAppend(&getFileName, "@", -1);
	    Tcl_DStringAppend(&getFileName, fullName, -1);
	    /* remove current bitmap. */
	    if (iconPtr->bitmap != None) {
	      Tk_FreePixmap(canvasPtr->display, iconPtr->bitmap);
	    }
	    if (iconPtr->clipMask != None) {
	      Tk_FreePixmap(canvasPtr->display, iconPtr->clipMask);
	    }
	    /* load the new bitmap. */
	    iconPtr->bitmap =
	      Tk_GetPixmap(canvasPtr->interp, canvasPtr->tkwin,
			   Tk_GetUid(getFileName.string));
	    Tcl_DStringFree(&getFileName);
	  }
	  Tcl_DStringFree(&fileName);
	  Tcl_DStringFree(&buffer);
	}
	ckfree((char *) list);
      }
    }
    Tcl_DStringFree(&varName);
  }
    
  /* do we have a new status ? We only automatically load a new bitmap */
  /* when the iconstatus was changed. This allows it to set the bitmap */
  /* directly with the -status option even when the icon has a status. */
  /* When the status is changed to empty, the old bitmap will remain... */
  /* this is a decision I make, maybe a wrong decision (time will show). */
  if (iconPtr->status != NULL && strlen(iconPtr->status) > (size_t) 0 &&
      ((oldStatus.string != NULL && strcmp(iconPtr->status, oldStatus.string) != 0)
       || oldStatus.string == NULL)) {
    /* the following code retrieves the path list for the icons */
    /* status. This is done because I don't want to attatch the */
    /* pathname list to each icon. */
    Tcl_DStringInit(&varName);
    Tcl_DStringAppend(&varName, "ip_priv(", -1);
    Tcl_DStringAppend(&varName, Tk_PathName(canvasPtr->tkwin), -1);
    Tcl_DStringAppend(&varName, ",iconstatuspath)", -1);
    if ((value = Tcl_GetVar(canvasPtr->interp, varName.string,
			    TCL_GLOBAL_ONLY)) != NULL) {
      if (Tcl_SplitList(canvasPtr->interp, value, &listCounter,
			&list) == TCL_OK) {
	/* walk through list of pathnames. */
	for (counter = 0; counter < listCounter; counter++) {
	  /* create the filename to load. */
	  Tcl_DStringInit(&fileName);
	  Tcl_DStringAppend(&fileName, list[counter], -1);
	  Tcl_DStringAppend(&fileName, "/", -1);
	  Tcl_DStringAppend(&fileName, iconPtr->status, -1);
	  Tcl_DStringAppend(&fileName, ".status", -1);
	  Tcl_DStringInit(&buffer);
	  fullName = Tcl_TildeSubst(canvasPtr->interp,
				    fileName.string, &buffer);
	  /* does the file exist ? */
	  if (access(fullName, F_OK) != -1) {
	    Tcl_DStringInit(&getFileName);
	    Tcl_DStringAppend(&getFileName, "@", -1);
	    Tcl_DStringAppend(&getFileName, fullName, -1);
	    /* remove current status bitmap. */
	    if (iconPtr->statusBitmap != None) {
	      Tk_FreePixmap(canvasPtr->display, iconPtr->statusBitmap);
	    }
	    if (iconPtr->statusClipMask != None) {
	      Tk_FreePixmap(canvasPtr->display, iconPtr->statusClipMask);
	    }
	    /* load the new status bitmap. */
	    iconPtr->statusBitmap =
	      Tk_GetPixmap(canvasPtr->interp, canvasPtr->tkwin,
			   Tk_GetUid(getFileName.string));
	    Tcl_DStringFree(&getFileName);
	  }
	  Tcl_DStringFree(&fileName);
	  Tcl_DStringFree(&buffer);
	}
	ckfree((char *) list);
      }
    }
    Tcl_DStringFree(&varName);
  }

  /* do we have a menu ? */
  if (iconPtr->menu1 != NULL && strlen(iconPtr->menu1) > (size_t) 0 &&
      iconPtr->menu1[0] != '.') {
    /* do we have to load the new menu definition ? */
    (void) Tcl_VarEval(canvasPtr->interp, "info commands .imenu-",
		       iconPtr->menu1, (char *) NULL);
    if (strlen(canvasPtr->interp->result) == 0) {
      /* the following code retrieves the path list for the menus. This */
      /* is done because I don't want to attatch the pathname list to */
      /* each icon. */
      Tcl_DStringInit(&varName);
      Tcl_DStringAppend(&varName, "ip_priv(", -1);
      Tcl_DStringAppend(&varName, Tk_PathName(canvasPtr->tkwin), -1);
      Tcl_DStringAppend(&varName, ",iconmenupath)", -1);
      if ((value = Tcl_GetVar(canvasPtr->interp, varName.string,
			      TCL_GLOBAL_ONLY)) != NULL) {
	if (Tcl_SplitList(canvasPtr->interp, value, &listCounter,
			  &list) == TCL_OK) {
	  /* walk through list of pathnames. */
	  for (counter = 0; counter < listCounter; counter++) {
	    /* create the filename to load. */
	    Tcl_DStringInit(&fileName);
	    Tcl_DStringAppend(&fileName, list[counter], -1);
	    Tcl_DStringAppend(&fileName, "/", -1);
	    Tcl_DStringAppend(&fileName, iconPtr->menu1, -1);
	    Tcl_DStringAppend(&fileName, ".imenu", -1);
	    Tcl_DStringInit(&buffer);
	    fullName = Tcl_TildeSubst(canvasPtr->interp,
				      fileName.string, &buffer);
	    if (access(fullName, F_OK) != -1) {
	      /* load new menu. */
	      Tcl_VarEval(canvasPtr->interp, "source ", fullName,
			  (char *) NULL);
	    }
	    Tcl_DStringFree(&fileName);
	    Tcl_DStringFree(&buffer);
	  }
	  ckfree((char *) list);
	}
      }
      Tcl_DStringFree(&varName);
    }
  }
    
  /* do we have a menu ? */
  if (iconPtr->menu2 != NULL && strlen(iconPtr->menu2) > (size_t) 0 &&
      iconPtr->menu2[0] != '.') {
    /* do we have to load the new menu definition ? */
    (void) Tcl_VarEval(canvasPtr->interp, "info commands .imenu-",
		       iconPtr->menu2, (char *) NULL);
    if (strlen(canvasPtr->interp->result) == 0) {
      /* the following code retrieves the path list for the menus. This */
      /* is done because I don't want to attatch the pathname list to */
      /* each icon. */
      Tcl_DStringInit(&varName);
      Tcl_DStringAppend(&varName, "ip_priv(", -1);
      Tcl_DStringAppend(&varName, Tk_PathName(canvasPtr->tkwin), -1);
      Tcl_DStringAppend(&varName, ",iconmenupath)", -1);
      if ((value = Tcl_GetVar(canvasPtr->interp, varName.string,
			      TCL_GLOBAL_ONLY)) != NULL) {
	if (Tcl_SplitList(canvasPtr->interp, value, &listCounter,
			  &list) == TCL_OK) {
	  /* walk through list of pathnames. */
	  for (counter = 0; counter < listCounter; counter++) {
	    /* create the filename to load. */
	    Tcl_DStringInit(&fileName);
	    Tcl_DStringAppend(&fileName, list[counter], -1);
	    Tcl_DStringAppend(&fileName, "/", -1);
	    Tcl_DStringAppend(&fileName, iconPtr->menu2, -1);
	    Tcl_DStringAppend(&fileName, ".imenu", -1);
	    Tcl_DStringInit(&buffer);
	    fullName = Tcl_TildeSubst(canvasPtr->interp,
				      fileName.string, &buffer);
	    if (access(fullName, F_OK) != -1) {
	      /* load new menu. */
	      Tcl_VarEval(canvasPtr->interp, "source ", fullName,
			  (char *) NULL);
	    }
	    Tcl_DStringFree(&fileName);
	    Tcl_DStringFree(&buffer);
	  }
	  ckfree((char *) list);
	}
      }
      Tcl_DStringFree(&varName);
    }
  }
  
  /* do we have a menu ? */
  if (iconPtr->menu3 != NULL && strlen(iconPtr->menu3) > (size_t) 0 &&
      iconPtr->menu3[0] != '.') {
    /* do we have to load the new menu definition ? */
    (void) Tcl_VarEval(canvasPtr->interp, "info commands .imenu-",
		       iconPtr->menu3, (char *) NULL);
    if (strlen(canvasPtr->interp->result) == 0) {
      /* the following code retrieves the path list for the menus. This */
      /* is done because I don't want to attatch the pathname list to */
      /* each icon. */
      Tcl_DStringInit(&varName);
      Tcl_DStringAppend(&varName, "ip_priv(", -1);
      Tcl_DStringAppend(&varName, Tk_PathName(canvasPtr->tkwin), -1);
      Tcl_DStringAppend(&varName, ",iconmenupath)", -1);
      if ((value = Tcl_GetVar(canvasPtr->interp, varName.string,
			      TCL_GLOBAL_ONLY)) != NULL) {
	if (Tcl_SplitList(canvasPtr->interp, value, &listCounter,
			  &list) == TCL_OK) {
	  /* walk through list of pathnames. */
	  for (counter = 0; counter < listCounter; counter++) {
	    /* create the filename to load. */
	    Tcl_DStringInit(&fileName);
	    Tcl_DStringAppend(&fileName, list[counter], -1);
	    Tcl_DStringAppend(&fileName, "/", -1);
	    Tcl_DStringAppend(&fileName, iconPtr->menu3, -1);
	    Tcl_DStringAppend(&fileName, ".imenu", -1);
	    Tcl_DStringInit(&buffer);
	    fullName = Tcl_TildeSubst(canvasPtr->interp,
				      fileName.string, &buffer);
	    if (access(fullName, F_OK) != -1) {
	      /* load new menu. */
	      Tcl_VarEval(canvasPtr->interp, "source ", fullName,
			  (char *) NULL);
	    }
	    Tcl_DStringFree(&fileName);
	    Tcl_DStringFree(&buffer);
	  }
	  ckfree((char *) list);
	}
      }
      Tcl_DStringFree(&varName);
    }
  }
    
  /* update the clipmask of the icon if we got a new bitmap. */
  if (iconPtr->bitmap != None) {
    if (Tk_GetPixmapClipMask(canvasPtr->display, iconPtr->bitmap) != None) {
      iconPtr->clipMask =
	Tk_GetPixmapClipMask(canvasPtr->display, iconPtr->bitmap);
    } else {
      iconPtr->clipMask = None;
    }
    Tk_SizeOfPixmap(canvasPtr->display, iconPtr->bitmap,
		    &iconPtr->intWidth, &iconPtr->intHeight);
  } else {
    iconPtr->clipMask = None;
  }
  
  /* update the status clipmask of the icon if we got a new */
  /* status bitmap. */
  if (iconPtr->statusBitmap != None) {
    if (Tk_GetPixmapClipMask(canvasPtr->display,
			     iconPtr->statusBitmap) != None) {
      iconPtr->statusClipMask =
	Tk_GetPixmapClipMask(canvasPtr->display,
			     iconPtr->statusBitmap);
    } else {
      iconPtr->statusClipMask = None;
    }
    Tk_SizeOfPixmap(canvasPtr->display, iconPtr->statusBitmap,
		    &iconPtr->intStatusWidth, &iconPtr->intStatusHeight);
  } else {
    iconPtr->statusClipMask = None;
  }

  ComputeIconBbox(canvasPtr, iconPtr);

  Tcl_DStringFree(&oldType);
  Tcl_DStringFree(&oldStatus);
  
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteIcon --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a icon item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteIcon(canvasPtr, itemPtr)
    Tk_Canvas *canvasPtr;		/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
    register IconItem *iconPtr = (IconItem *) itemPtr;

    if (iconPtr->bgColor != NULL) {
	Tk_FreeColor(iconPtr->bgColor);
    }
    if (iconPtr->bitmap != None) {
        Tk_FreePixmap(canvasPtr->display, iconPtr->bitmap);
    }
    if (iconPtr->clipMask != None) {
        Tk_FreePixmap(canvasPtr->display, iconPtr->clipMask);
    }
    if (iconPtr->command != NULL) {
	ckfree(iconPtr->command);
    }
    if (iconPtr->dropCommand != NULL) {
	ckfree(iconPtr->dropCommand);
    }
    if (iconPtr->edges != NULL) {
	ckfree(iconPtr->edges);
    }
    if (iconPtr->fgColor != NULL) {
	Tk_FreeColor(iconPtr->fgColor);
    }
    if (iconPtr->fontPtr != NULL) {
	Tk_FreeFontStruct(iconPtr->fontPtr);
    }
    if (iconPtr->layoutInfo != NULL) {
	ckfree(iconPtr->layoutInfo);
    }
    if (iconPtr->menu1 != NULL) {
	ckfree(iconPtr->menu1);
    }
    if (iconPtr->menu2 != NULL) {
	ckfree(iconPtr->menu2);
    }
    if (iconPtr->menu3 != NULL) {
	ckfree(iconPtr->menu3);
    }
    if (iconPtr->name != NULL) {
	ckfree(iconPtr->name);
    }
    if (iconPtr->outlineColor != NULL) {
	Tk_FreeColor(iconPtr->outlineColor);
    }
    if (iconPtr->sortString != NULL) {
	ckfree(iconPtr->sortString);
    }
    if (iconPtr->state != NULL) {
	ckfree(iconPtr->state);
    }
    if (iconPtr->status != NULL) {
	ckfree(iconPtr->status);
    }
    if (iconPtr->statusBitmap != None) {
	Tk_FreePixmap(canvasPtr->display, iconPtr->statusBitmap);
    }
    if (iconPtr->statusClipMask != None) {
	Tk_FreePixmap(canvasPtr->display, iconPtr->statusClipMask);
    }
    if (iconPtr->text != NULL) {
	ckfree(iconPtr->text);
    }
    if (iconPtr->type != NULL) {
	ckfree(iconPtr->type);
    }

    if (iconPtr->gc != NULL) {
	Tk_FreeGC(canvasPtr->display, iconPtr->gc);
    }
    if (iconPtr->invertedGc != NULL) {
	Tk_FreeGC(canvasPtr->display, iconPtr->invertedGc);
    }
    if (iconPtr->outlineGc != NULL) {
	Tk_FreeGC(canvasPtr->display, iconPtr->outlineGc);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeIconBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a icon item.
 *	This procedure is where the child icon's placement is
 *	computed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ComputeIconBbox(canvasPtr, iconPtr)
    Tk_Canvas *canvasPtr;		/* Canvas that contains item. */
    register IconItem *iconPtr;	        /* Item whose bbox is to be
					 * recomputed. */
{
  int lineHeight, lineWidth, numChars;
  unsigned int width = 0, height = 0, textWidth = 0;
  int x, y, ascent, descent;
  
  x = iconPtr->x;
  y = iconPtr->y;
  
  if (iconPtr->bitmap == None && iconPtr->text == NULL) {
    iconPtr->header.x1 = iconPtr->header.x2 = x;
    iconPtr->header.y1 = iconPtr->header.y2 = y;
    return;
  }
  
  /*
   * Compute location and size of icon, using anchor information.
   */
  
  if (iconPtr->bitmap != None) {
    Tk_SizeOfPixmap(canvasPtr->display, iconPtr->bitmap, &width,
		    &height);
  }
  ascent = iconPtr->fontPtr->ascent;
  descent = iconPtr->fontPtr->descent;
  lineHeight = iconPtr->fontPtr->ascent + iconPtr->fontPtr->descent;
  if (iconPtr->width != 0 &&
      iconPtr->width < (unsigned int) strlen(iconPtr->text)) {
    textWidth = iconPtr->width;
  } else {
    textWidth = strlen(iconPtr->text);
  }
  numChars = TkMeasureChars(iconPtr->fontPtr, iconPtr->text,
			    textWidth, 0, 10000000,
			    TK_AT_LEAST_ONE, &lineWidth);
  switch (iconPtr->textAnchor) {
  case TK_ANCHOR_N:
    iconPtr->textY = y + lineHeight + iconPtr->borderwidth;
    iconPtr->bitmapY = y + lineHeight + ICON_OFFSET +
      iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      iconPtr->textX = x + (iconPtr->intWidth/2) -
	(lineWidth/2) + iconPtr->borderwidth;
      iconPtr->bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      iconPtr->textX = x + iconPtr->borderwidth;
      iconPtr->bitmapX = x + (lineWidth/2) -
	(iconPtr->intWidth/2) + iconPtr->borderwidth;
    }
    height += lineHeight + ICON_OFFSET + ICON_OFFSET;
    break;
  case TK_ANCHOR_NW:
    iconPtr->textX = x + iconPtr->borderwidth;
    iconPtr->bitmapX = x + iconPtr->borderwidth;
    iconPtr->textY = y + lineHeight +
      iconPtr->borderwidth;
    iconPtr->bitmapY = y + lineHeight + ICON_OFFSET +
      iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
    } else {
      width = lineWidth;
    }
    height += lineHeight + ICON_OFFSET + ICON_OFFSET;
    break;
  case TK_ANCHOR_NE:
    iconPtr->textY = y + lineHeight +
      iconPtr->borderwidth;
    iconPtr->bitmapY = y + lineHeight + ICON_OFFSET +
      iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      iconPtr->textX = x + (iconPtr->intWidth - lineWidth) +
	iconPtr->borderwidth;
      iconPtr->bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      iconPtr->textX = x + iconPtr->borderwidth;
      iconPtr->bitmapX = x + (lineWidth - iconPtr->intWidth) +
	iconPtr->borderwidth;
    }
    height += lineHeight + ICON_OFFSET + ICON_OFFSET;
    break;
  case TK_ANCHOR_S:
    iconPtr->textY = y + height + ascent + ICON_OFFSET +
      iconPtr->borderwidth;
    iconPtr->bitmapY = y + iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      iconPtr->textX = x + (iconPtr->intWidth/2) -
	(lineWidth/2) + iconPtr->borderwidth;
      iconPtr->bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      iconPtr->textX = x + iconPtr->borderwidth;
      iconPtr->bitmapX = x + (lineWidth/2) -
	(iconPtr->intWidth/2) + iconPtr->borderwidth;
    }
    height += lineHeight + ICON_OFFSET + ICON_OFFSET;
    break;
  case TK_ANCHOR_SW:
    iconPtr->textX = x + iconPtr->borderwidth;
    iconPtr->bitmapX = x + iconPtr->borderwidth;
    iconPtr->textY = y + height + ascent + ICON_OFFSET +
      iconPtr->borderwidth;
    iconPtr->bitmapY = y + iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
    } else {
      width = lineWidth;
    }
    height += lineHeight + ICON_OFFSET + ICON_OFFSET;
    break;
  case TK_ANCHOR_SE:
    iconPtr->textY = y + height + ascent + ICON_OFFSET +
      iconPtr->borderwidth;
    iconPtr->bitmapY = y + iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      iconPtr->textX = x + (iconPtr->intWidth - lineWidth) +
	iconPtr->borderwidth;
      iconPtr->bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      iconPtr->textX = x + iconPtr->borderwidth;
      iconPtr->bitmapX = x + (lineWidth - iconPtr->intWidth) +
	iconPtr->borderwidth;
    }
    height += lineHeight + ICON_OFFSET + ICON_OFFSET;
    break;
  case TK_ANCHOR_W:
    iconPtr->textY = y + (height/2) + descent +
      iconPtr->borderwidth;
    iconPtr->bitmapY = y + iconPtr->borderwidth;
    iconPtr->textX = x + iconPtr->borderwidth;
    iconPtr->bitmapX = x + lineWidth + ICON_OFFSET +
      iconPtr->borderwidth;
    width += lineWidth + ICON_OFFSET;
    break;
  case TK_ANCHOR_E:
    iconPtr->textY = y + (height/2) + descent +
      iconPtr->borderwidth;
    iconPtr->bitmapY = y + iconPtr->borderwidth;
    iconPtr->textX = x + width + ICON_OFFSET +
      iconPtr->borderwidth;
    iconPtr->bitmapX = x + iconPtr->borderwidth;
    width += lineWidth + ICON_OFFSET;
    break;
  case TK_ANCHOR_CENTER:
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      iconPtr->textX = x + (iconPtr->intWidth/2) -
	(lineWidth/2) + iconPtr->borderwidth;
      iconPtr->bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      iconPtr->textX = x + iconPtr->borderwidth;
      iconPtr->bitmapX = x + (lineWidth/2) -
	(iconPtr->intWidth/2) + iconPtr->borderwidth;
    }
    if (iconPtr->intHeight > lineHeight) {
      height = iconPtr->intHeight;
      iconPtr->textY = y + (iconPtr->intHeight/2) +
	descent + iconPtr->borderwidth;
      iconPtr->bitmapY = y + iconPtr->borderwidth;
    } else {
      height = lineHeight;
      iconPtr->textY = y + descent + iconPtr->borderwidth;
      iconPtr->bitmapY = y + (lineHeight/2) -
	(iconPtr->intHeight/2) + iconPtr->borderwidth;
    }
    break;
  }
  
  /*
   * Store the information in the item header.
   */
  iconPtr->header.x1 = x;
  iconPtr->header.y1 = y;
  iconPtr->header.x2 = x + width + iconPtr->borderwidth +
    iconPtr->borderwidth;
  iconPtr->header.y2 = y + height + iconPtr->borderwidth +
    iconPtr->borderwidth;
  iconPtr->textWidth = lineWidth;
  iconPtr->textHeight = lineHeight;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayIcon --
 *
 *	This procedure is invoked to draw a icon item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplayIcon(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
  register IconItem *iconPtr = (IconItem *) itemPtr;
  int textWidth;

  /* display icon bitmap */
  if (iconPtr->bitmap != None) {
    if (iconPtr->clipMask != None) {
      XSetClipOrigin(Tk_Display(canvasPtr->tkwin), iconPtr->gc,
		     iconPtr->bitmapX - canvasPtr->drawableXOrigin,
		     iconPtr->bitmapY - canvasPtr->drawableYOrigin);
      XSetClipMask(Tk_Display(canvasPtr->tkwin), iconPtr->gc,
		   iconPtr->clipMask);
    }
    if (Tk_DepthOfPixmap(Tk_Display(canvasPtr->tkwin),
			 iconPtr->bitmap) == 1) {
      XCopyPlane(Tk_Display(canvasPtr->tkwin), iconPtr->bitmap,
		 drawable, iconPtr->gc, 0, 0,
		 (unsigned int) iconPtr->intWidth,
		 (unsigned int) iconPtr->intHeight,
		 iconPtr->bitmapX - canvasPtr->drawableXOrigin,
		 iconPtr->bitmapY - canvasPtr->drawableYOrigin, 1);
    } else {
      XCopyArea(Tk_Display(canvasPtr->tkwin), iconPtr->bitmap,
                drawable, iconPtr->gc, 0, 0,
		(unsigned int) iconPtr->intWidth,
                (unsigned int) iconPtr->intHeight,
                iconPtr->bitmapX - canvasPtr->drawableXOrigin,
                iconPtr->bitmapY - canvasPtr->drawableYOrigin);
    }
    if (iconPtr->clipMask != None) {
      XSetClipMask(Tk_Display(canvasPtr->tkwin), iconPtr->gc,
		   None);
    }
  }

  /* display status bitmap */
  if (iconPtr->statusBitmap != None) {
    if (iconPtr->statusClipMask != None) {
      XSetClipOrigin(Tk_Display(canvasPtr->tkwin), iconPtr->gc,
		     iconPtr->bitmapX - canvasPtr->drawableXOrigin,
		     iconPtr->bitmapY - canvasPtr->drawableYOrigin);
      XSetClipMask(Tk_Display(canvasPtr->tkwin), iconPtr->gc,
		   iconPtr->statusClipMask);
    }
    if (Tk_DepthOfPixmap(Tk_Display(canvasPtr->tkwin),
			 iconPtr->statusBitmap) == 1) {
      XCopyPlane(Tk_Display(canvasPtr->tkwin), iconPtr->statusBitmap,
		 drawable, iconPtr->gc, 0, 0,
		 (unsigned int) iconPtr->intStatusWidth,
		 (unsigned int) iconPtr->intStatusHeight,
		 iconPtr->bitmapX - canvasPtr->drawableXOrigin,
		 iconPtr->bitmapY - canvasPtr->drawableYOrigin, 1);
    } else {
      XCopyArea(Tk_Display(canvasPtr->tkwin), iconPtr->statusBitmap,
                drawable, iconPtr->gc, 0, 0,
	        (unsigned int) iconPtr->intStatusWidth,
                (unsigned int) iconPtr->intStatusHeight,
                iconPtr->bitmapX - canvasPtr->drawableXOrigin,
                iconPtr->bitmapY - canvasPtr->drawableYOrigin);
    }
    if (iconPtr->statusClipMask != None) {
      XSetClipMask(Tk_Display(canvasPtr->tkwin), iconPtr->gc,
		   None);
    }
  }

  /* display text */
  if (iconPtr->text != NULL) {
    if (iconPtr->width != 0 &&
	iconPtr->width < (unsigned int) strlen(iconPtr->text)) {
      textWidth = iconPtr->width;
    } else {
      textWidth = strlen(iconPtr->text);
    }
    if (strcmp(iconPtr->state, "selected") == 0) {
      XFillRectangle(canvasPtr->display, drawable,
		     iconPtr->gc,
		     iconPtr->textX - canvasPtr->drawableXOrigin,
		     iconPtr->textY - iconPtr->fontPtr->ascent -
		     canvasPtr->drawableYOrigin - ICON_OFFSET,
		     iconPtr->textWidth,
		     iconPtr->textHeight + ICON_OFFSET + ICON_OFFSET);
      TkDisplayChars(canvasPtr->display, drawable, iconPtr->invertedGc,
		     iconPtr->fontPtr, iconPtr->text, textWidth,
		     iconPtr->textX - canvasPtr->drawableXOrigin,
		     iconPtr->textY - canvasPtr->drawableYOrigin,
		     0);
    } else {
      XFillRectangle(canvasPtr->display, drawable,
		     iconPtr->invertedGc,
		     iconPtr->textX - canvasPtr->drawableXOrigin,
		     iconPtr->textY - iconPtr->fontPtr->ascent -
		     canvasPtr->drawableYOrigin - ICON_OFFSET,
		     iconPtr->textWidth,
		     iconPtr->textHeight + ICON_OFFSET + ICON_OFFSET);
      TkDisplayChars(canvasPtr->display, drawable, iconPtr->gc,
		     iconPtr->fontPtr, iconPtr->text, textWidth,
		     iconPtr->textX - canvasPtr->drawableXOrigin,
		     iconPtr->textY - canvasPtr->drawableYOrigin,
		     0);
    }
  }

  /* draw border */
  if (iconPtr->borderwidth > 0) {
    if (iconPtr->relief == TK_RELIEF_FLAT) {
      if (iconPtr->outlineGc != None) {
	XDrawRectangle(canvasPtr->display, drawable, iconPtr->outlineGc,
		       (iconPtr->header.x1 - canvasPtr->drawableXOrigin) +
		       (iconPtr->borderwidth / 2),
		       (iconPtr->header.y1 - canvasPtr->drawableYOrigin) +
		       (iconPtr->borderwidth / 2),
		       (iconPtr->header.x2 - iconPtr->header.x1) -
		       iconPtr->borderwidth,
		       (iconPtr->header.y2 - iconPtr->header.y1) -
		       iconPtr->borderwidth);
      }
    } else {
      Tk_Draw3DRectangle(canvasPtr->display, drawable, canvasPtr->bgBorder,
			 (iconPtr->header.x1 - canvasPtr->drawableXOrigin),
			 (iconPtr->header.y1 - canvasPtr->drawableYOrigin),
			 (iconPtr->header.x2 - iconPtr->header.x1),
			 (iconPtr->header.y2 - iconPtr->header.y1),
			 iconPtr->borderwidth, iconPtr->relief);
    }
  }
}

/*
 *--------------------------------------------------------------
 *
 * IconToPoint --
 *
 *	Computes the distance from a given point to a given
 *	rectangle, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are coordPtr[0] and coordPtr[1] is inside the icon.  If the
 *	point isn't inside the icon then the return value is the
 *	distance from the point to the icon.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
IconToPoint(canvasPtr, itemPtr, coordPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *coordPtr;		/* Pointer to x and y coordinates. */
{
  register IconItem *iconPtr = (IconItem *) itemPtr;
  double x1, x2, y1, y2, xDiff, yDiff;
  
  x1 = iconPtr->header.x1;
  y1 = iconPtr->header.y1;
  x2 = iconPtr->header.x2;
  y2 = iconPtr->header.y2;
  
  /*
   * Point is outside rectangle.
   */
  
  if (coordPtr[0] < x1) {
    xDiff = x1 - coordPtr[0];
  } else if (coordPtr[0] > x2)  {
    xDiff = coordPtr[0] - x2;
  } else {
    xDiff = 0;
  }
  
  if (coordPtr[1] < y1) {
    yDiff = y1 - coordPtr[1];
  } else if (coordPtr[1] > y2)  {
    yDiff = coordPtr[1] - y2;
  } else {
    yDiff = 0;
  }
  
  return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * IconToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangle.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
IconToArea(canvasPtr, itemPtr, rectPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against rectangle. */
    double *rectPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
  register IconItem *iconPtr = (IconItem *) itemPtr;
  
  if ((rectPtr[2] <= iconPtr->header.x1)
      || (rectPtr[0] >= iconPtr->header.x2)
      || (rectPtr[3] <= iconPtr->header.y1)
      || (rectPtr[1] >= iconPtr->header.y2)) {
    return -1;
  }
  if ((rectPtr[0] <= iconPtr->header.x1)
      && (rectPtr[1] <= iconPtr->header.y1)
      && (rectPtr[2] >= iconPtr->header.x2)
      && (rectPtr[3] >= iconPtr->header.y2)) {
    return 1;
  }
  return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleIcon --
 *
 *	This procedure is invoked to rescale a rectangle or oval
 *	item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The rectangle or oval referred to by itemPtr is rescaled
 *	so that the following transformation is applied to all
 *	point coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

static void
ScaleIcon(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing rectangle. */
    Tk_Item *itemPtr;			/* Rectangle to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
  register IconItem *iconPtr = (IconItem *) itemPtr;
  
  iconPtr->x = originX + scaleX*(iconPtr->x - originX);
  iconPtr->y = originY + scaleY*(iconPtr->y - originY);
  ComputeIconBbox(canvasPtr, iconPtr);
}

/*
 *--------------------------------------------------------------
 *
 * TranslateIcon --
 *
 *	This procedure is called to move a rectangle or oval by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the rectangle or oval is offset by
 *	(xDelta, yDelta), and the bounding box is updated in the
 *	generic part of the item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateIcon(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
  register IconItem *iconPtr = (IconItem *) itemPtr;
  
  iconPtr->x += deltaX;
  iconPtr->y += deltaY;
  ComputeIconBbox(canvasPtr, iconPtr);
}

/*
 *--------------------------------------------------------------
 *
 * IconToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	icon items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
IconToPostscript(canvasPtr, itemPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
  register IconItem *iconPtr = (IconItem *) itemPtr;
  double x, y, bitmapX = 0, bitmapY = 0, textX = 0, textY = 0;
  int lineHeight, lineWidth, numChars;
  unsigned int width, height, ascent, descent;
  char buffer[200];

  /*
   * Compute the coordinates of the lower-left corner of the icon,
   * taking into account the anchor position for the bitmp.
   */
  
  if (iconPtr->bitmap == None) {
    return TCL_OK;
  }
  
  x = iconPtr->x;
  y = TkCanvPsY(psInfoPtr, iconPtr->y);
  Tk_SizeOfPixmap(canvasPtr->display, iconPtr->bitmap, &width, &height);
  ascent = iconPtr->fontPtr->ascent;
  descent = iconPtr->fontPtr->descent;
  lineHeight = iconPtr->fontPtr->ascent + iconPtr->fontPtr->descent;
  numChars = TkMeasureChars(iconPtr->fontPtr, iconPtr->text,
			    strlen(iconPtr->text), 0, 10000000,
			    TK_AT_LEAST_ONE, &lineWidth);
  switch (iconPtr->textAnchor) {
  case TK_ANCHOR_N:
    textY = y + lineHeight + iconPtr->borderwidth;
    bitmapY = y + lineHeight + ICON_OFFSET + iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      textX = x + (iconPtr->intWidth/2) -
	(lineWidth/2) + iconPtr->borderwidth;
      bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      textX = x + iconPtr->borderwidth;
      bitmapX = x + (lineWidth/2) -
	(iconPtr->intWidth/2) + iconPtr->borderwidth;
    }
    height += lineHeight + ICON_OFFSET;
    break;
  case TK_ANCHOR_NW:
    textX = x + iconPtr->borderwidth;
    bitmapX = x + iconPtr->borderwidth;
    textY = y + lineHeight + iconPtr->borderwidth;
    bitmapY = y + lineHeight + ICON_OFFSET +
      iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
    } else {
      width = lineWidth;
    }
    height += lineHeight + ICON_OFFSET;
    break;
  case TK_ANCHOR_NE:
    textY = y + lineHeight + iconPtr->borderwidth;
    bitmapY = y + lineHeight + ICON_OFFSET +
      iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      textX = x + (iconPtr->intWidth - lineWidth) +
	iconPtr->borderwidth;
      bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      textX = x + iconPtr->borderwidth;
      bitmapX = x + (lineWidth - iconPtr->intWidth) +
	iconPtr->borderwidth;
    }
    height += lineHeight + ICON_OFFSET;
    break;
  case TK_ANCHOR_S:
    textY = y - height - ascent - ICON_OFFSET +
      iconPtr->borderwidth;
    bitmapY = y - height + iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      textX = x + (iconPtr->intWidth/2.0) -
	(lineWidth/2.0) + iconPtr->borderwidth;
      bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      textX = x + iconPtr->borderwidth;
      bitmapX = x + (lineWidth/2.0) -
	(iconPtr->intWidth/2.0) + iconPtr->borderwidth;
    }
    height += lineHeight + ICON_OFFSET;
    break;
  case TK_ANCHOR_SW:
    textX = x + iconPtr->borderwidth;
    bitmapX = x + iconPtr->borderwidth;
    textY = y + height + ascent + ICON_OFFSET +
      iconPtr->borderwidth;
    bitmapY = y - height + iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
    } else {
      width = lineWidth;
    }
    height += lineHeight + ICON_OFFSET;
    break;
  case TK_ANCHOR_SE:
    textY = y + height + ascent + ICON_OFFSET +
      iconPtr->borderwidth;
    bitmapY = y - height + iconPtr->borderwidth +
      iconPtr->borderwidth;
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      textX = x + (iconPtr->intWidth - lineWidth) +
	iconPtr->borderwidth;
      bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      textX = x + iconPtr->borderwidth;
      bitmapX = x + (lineWidth - iconPtr->intWidth) +
	iconPtr->borderwidth;
    }
    height += lineHeight + ICON_OFFSET;
    break;
  case TK_ANCHOR_W:
    textY = y + (height/2) + descent + iconPtr->borderwidth;
    bitmapY = y + iconPtr->borderwidth;
    textX = x + iconPtr->borderwidth;
    bitmapX = x + lineWidth + ICON_OFFSET + iconPtr->borderwidth;
    width += lineWidth + ICON_OFFSET;
    break;
  case TK_ANCHOR_E:
    textY = y + (height/2) + descent + iconPtr->borderwidth;
    bitmapY = y + iconPtr->borderwidth;
    textX = x + width + ICON_OFFSET + iconPtr->borderwidth;
    bitmapX = x + iconPtr->borderwidth;
    width += lineWidth + ICON_OFFSET;
    break;
  case TK_ANCHOR_CENTER:
    if (iconPtr->intWidth > lineWidth) {
      width = iconPtr->intWidth;
      textX = x + (iconPtr->intWidth/2) -
	(lineWidth/2) + iconPtr->borderwidth;
      bitmapX = x + iconPtr->borderwidth;
    } else {
      width = lineWidth;
      textX = x + iconPtr->borderwidth;
      bitmapX = x + (lineWidth/2) -
	(iconPtr->intWidth/2) + iconPtr->borderwidth;
    }
    if (iconPtr->intHeight > lineHeight) {
      height = iconPtr->intHeight;
      textY = y + (iconPtr->intHeight/2) + descent +
	iconPtr->borderwidth;
      bitmapY = y + iconPtr->borderwidth;
    } else {
      height = lineHeight;
      textY = y + descent + iconPtr->borderwidth;
      bitmapY = y + (lineHeight/2) -
	(iconPtr->intHeight/2) + iconPtr->borderwidth;
    }
    break;
  }
  
  /*
   * Color the background, if there is one.
   */
  if (iconPtr->bgColor != NULL) {
    sprintf(buffer,
	    "%g %g moveto %d 0 rlineto 0 %d rlineto %d %s\n",
	    bitmapX, bitmapY, width, height, -width,
	    "0 rlineto closepath");
    Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
    if (TkCanvPsColor(canvasPtr, psInfoPtr, iconPtr->bgColor) != TCL_OK) {
      return TCL_ERROR;
    }
    Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
  }
  
  /*
   * Draw the bitmap, if there is a foreground color.
   */
  if (iconPtr->fgColor != NULL ||
      Tk_DepthOfPixmap(canvasPtr->display, iconPtr->bitmap) > 1) {
    if (iconPtr->fgColor != NULL) {
      if (TkCanvPsColor(canvasPtr, psInfoPtr, iconPtr->fgColor) != TCL_OK) {
	return TCL_ERROR;
      }
    }
    if (Tk_DepthOfPixmap(canvasPtr->display, iconPtr->bitmap) == 1) {
      sprintf(buffer, "%.15g %.15g translate\n %d %d true matrix {\n",
	      bitmapX, bitmapY, width, height);
      Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
      if (TkCanvPsBitmap(canvasPtr, psInfoPtr,
			 iconPtr->bitmap) != TCL_OK) {
	return TCL_ERROR;
      }
      Tcl_AppendResult(canvasPtr->interp, "\n} imagemask\n",
		       (char *) NULL);
    } else {
      if (TkCanvPsPixmap(canvasPtr, psInfoPtr,
			 iconPtr->bitmap, bitmapX, bitmapY) != TCL_OK) {
	return TCL_ERROR;
      }
    }
  }
  
  /*
   * Now draw the outline, if there is one.
   */
  if (iconPtr->outlineColor != NULL) {
    sprintf(buffer,
	    "%g %g moveto %d 0 rlineto 0 %d rlineto %d %s\n",
	    bitmapX, bitmapY, width, height, -width,
	    "0 rlineto closepath");
    Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
    sprintf(buffer, "%d setlinewidth", iconPtr->borderwidth);
    Tcl_AppendResult(canvasPtr->interp, buffer,
		     " 0 setlinejoin 2 setlinecap\n", (char *) NULL);
    if (TkCanvPsColor(canvasPtr, psInfoPtr, iconPtr->outlineColor)
	!= TCL_OK) {
      return TCL_ERROR;
    }
    Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
  }
  return TCL_OK;
}
