/* Bos_GDBM.c - GDBM-based object storage routines
 *
 * Copyright (C) 1992,1993 Engineering Design Research Center
 *
 * Author: Sean Levy (snl+@cmu.edu)
 *         n-dim Group
 *         Engineering Design Research Center
 *         Carnegie Mellon University
 *         5000 Forbes Ave / PGH, PA / 51221
 *
 *         Fax: (412) 268-5229
 *         Voice: (412) 268-5226
 */

/*
 * Bos_GDBM.c,v 1.6 1992/08/03 20:50:01 snl Exp
 *
 * Bos_GDBM.c,v
 * Revision 1.6  1992/08/03  20:50:01  snl
 * Debugged new storage implementation
 *
 * Revision 1.5  1992/08/03  13:04:48  snl
 * added debugging code
 *
 * Revision 1.4  1992/08/03  12:35:26  snl
 * bug fixes
 *
 * Revision 1.3  1992/07/31  14:26:11  snl
 * more big fixes
 *
 * Revision 1.2  1992/07/30  20:02:43  snl
 * bug fixes to new storage code
 *
 * Revision 1.1  1992/07/30  17:53:39  snl
 * Re-implemented GDBM-based storage scheme
 *
 */

static char rcsID[] = "Bos_GDBM.c,v 1.6 1992/08/03 20:50:01 snl Exp";

#include <stdio.h>
#include "bosInt.h"

#define STORAGE_BLOCK_SIZE	512
#define STORAGE_OPEN_MODE	0x1B4
#define STORAGE_FILE_NAME	"bosobjects.gdbm"

typedef struct _StorageInfoRec {
  GDBM_FILE gdbmf;
  char *directory;
  int mode;
  int lock_file;
  struct _StorageInfoRec *next;
} StorageInfo;

static StorageInfo *last_used = (StorageInfo *)0;
static StorageInfo *head = (StorageInfo *)0;
static int initialized = 0;

extern gdbm_error gdbm_errno;
char *_gdbmErrMsg();

GDBM_FILE
Bos_InitializeStorage(dir, lock, mode)
     char *dir;
     int lock;
     int mode;
{
  void fatal_func();
  char fname[1024];
  
  if (!initialized) {
    void Bos_StorageCatastropheHandler();
    
#ifdef sunos41
    on_exit(Bos_StorageCatastropheHandler, 0);
#else
    atexit(Bos_StorageCatastropheHandler);
#endif
    initialized = 1;
  }
  if(last_used == (StorageInfo *)0) {
    /* not opened this directory previously.  Open now */
    last_used = (StorageInfo *)ckalloc(sizeof(StorageInfo));
    last_used->directory = (char *)ckalloc(strlen(dir)+1);
    strcpy(last_used->directory, dir);
    last_used->next = (StorageInfo *)0;
    last_used->mode = mode;
    last_used->lock_file = lock;
    sprintf(fname, "%s/%s", dir, STORAGE_FILE_NAME);
    last_used->gdbmf = gdbm_open(fname, STORAGE_BLOCK_SIZE,
				 mode, STORAGE_OPEN_MODE,
				 fatal_func);
    if(last_used->gdbmf == (GDBM_FILE)0) {
      fprintf(stderr, "major fucking problem #1 in GDBM (%d: %s)\n",
	      gdbm_errno, _gdbmErrMsg(gdbm_errno));
      return((GDBM_FILE)0);
    }
    if (head == (StorageInfo *)0)
      head = last_used;
    else {
      last_used->next = head;
      head = last_used;
    }
#ifdef DEBUGGING
    fprintf(stderr, "Bos_InitializeStorage(%s,%d,%x) => %x [1]\n",
	    dir, lock, mode, last_used->gdbmf);
    Bos_DumpStorageStatus(stderr);
#endif
    return(last_used->gdbmf);
  }
  
  /*
    check if the last_used was a locked file opened in write mode
    and is now being used in read mode.  If so, reopen in read mode to let
    others get a chance to write to it
    */
  if(last_used->lock_file && last_used->mode == GDBM_WRCREAT &&
     mode != GDBM_WRCREAT) {
    /* close and reopen in read mode */
    gdbm_close(last_used->gdbmf);
    last_used->mode = GDBM_READER;
    sprintf(fname, "%s/%s", last_used->directory, STORAGE_FILE_NAME);
    last_used->gdbmf = gdbm_open(fname, STORAGE_BLOCK_SIZE,
				 mode, STORAGE_OPEN_MODE,
				 fatal_func);
    if(last_used->gdbmf == (GDBM_FILE)0)
      {
	fprintf(stderr, "major fucking problem #2 in GDBM (%d: %s)\n",
		gdbm_errno, _gdbmErrMsg(gdbm_errno));
	return((GDBM_FILE)0);
      }
  }
  
  /* see if we have already opened the desired file */
  if(strcmp(last_used->directory, dir)) {
    /* Not the last one used. Find one in the chain if it exists */
    for(last_used = head;
	last_used != (StorageInfo *)0;
	last_used = last_used->next)
      if(!strcmp(last_used->directory, dir))
	break;
    if(!last_used) {
      /* does not exist in the chain.  recurse */
#ifdef DEBUGGING
      fprintf(stderr, "Bos_InitializeStorage(%s,%d,%x) recursing [2]\n",
	      dir, lock, mode);
      Bos_DumpStorageStatus(stderr);
#endif
      return Bos_InitializeStorage(dir, lock, mode);
    }
  }
  /* use the one in the last_used */
  if(last_used->mode == mode) {
#ifdef DEBUGGING
    fprintf(stderr, "Bos_InitializeStorage(%s,%d,%x) => %x [3]\n",
	    dir, lock, mode, last_used->gdbmf);
    Bos_DumpStorageStatus(stderr);
#endif
    return(last_used->gdbmf);
  }
  /* modes differ.  close and reopen only if mode is to write */
  if(mode == GDBM_WRCREAT) {
    gdbm_close(last_used->gdbmf);
    last_used->mode = mode;
    sprintf(fname, "%s/%s", last_used->directory, STORAGE_FILE_NAME);
    last_used->gdbmf = gdbm_open(fname, STORAGE_BLOCK_SIZE, mode,
				 STORAGE_OPEN_MODE, fatal_func);
    if(last_used->gdbmf == (GDBM_FILE)0) {
      fprintf(stderr, "major danged problem #3 in GDBM (%d: %s)\n",
	      gdbm_errno, _gdbmErrMsg(gdbm_errno));
      return((GDBM_FILE)0);
    }
  }
#ifdef DEBUGGING
  fprintf(stderr, "Bos_InitializeStorage(%s,%d,%x) => %x [4]\n",
	  dir, lock, mode, last_used->gdbmf);
  Bos_DumpStorageStatus(stderr);
#endif
  return(last_used->gdbmf);
}

void
Bos_StorageCatastropheHandler()
{
  (void) Bos_ShutdownStorage(0);
}

int
Bos_ShutdownStorage(dir)
     char *dir;
{
  StorageInfo *f, *l;
  int n;

  n = 0;
  l = (StorageInfo *)0;
  for (f = head; f != (StorageInfo *)0; f = f->next)
    {
      if (dir == (char *)0) {
	gdbm_close(f->gdbmf);
	ckfree(f->directory);
	ckfree(f);
	n++;
      } else if (strcmp(dir, f->directory))
	l = f;
      else {
	gdbm_close(f->gdbmf);
	ckfree(f->directory);
	n++;
	if (l == (StorageInfo *)0)
	  head = f->next;
	else
	  l->next = f->next;
	ckfree(f);
	break;
      }
    }
  if (dir == (char *)0)
    head = (StorageInfo *)0;
  return n;
}

int
Bos_CompactStorage(dir)
     char *dir;
{
  GDBM_FILE f;
  int err, saved_error;

  f = Bos_InitializeStorage(dir, 1, GDBM_WRCREAT);
  if (f == (GDBM_FILE)0)
    return -1;
  err = gdbm_reorganize(f);
  if (err)
    saved_error = (int)gdbm_errno;
  (void) Bos_ShutdownStorage(dir);
  return err? saved_error: 0;
}

int
Bos_StoreObject(world, obj_name, dir, lock)
     Bos_World *world;
     char *obj_name;
     char *dir;
     int lock;
{
  Bos_Object *obj;
  GDBM_FILE f;
  datum key;
  datum bytes;
  int result;
  char errmsg[300];

  obj = Bos_Find(world, obj_name);
  if (obj == (Bos_Object *)0)
    return BOS_NOT_FOUND;
  f = Bos_InitializeStorage(dir, lock, GDBM_WRCREAT);
  if (f == (GDBM_FILE)0)
    return BOS_ERROR;
  bytes.dsize = 0;
  bytes.dptr = Bos_ObjectSlots2ByteStream(obj, 0, &bytes.dsize, 0, errmsg);
  if (bytes.dptr == (char *)0) {
    fprintf(stderr, "BOS ERROR: StoreObject(%s,%s,%d): %s\n",
	    obj_name, dir, lock, errmsg);
    return BOS_ERROR;
  }
  key.dptr = obj_name;
  key.dsize = strlen(obj_name);
  if (gdbm_store(f, key, bytes, GDBM_REPLACE) < 0) {
    fprintf(stderr, "BOS ERROR: StoreObject(%s,%s,%d): GDBM error %s\n",
	    obj_name, dir, lock, _gdbmErrMsg(gdbm_errno));
    result = BOS_ERROR;
  } else
    result = BOS_OK;
  ckfree(bytes.dptr);
  return result;
}

Bos_Object *
Bos_FetchObject(world, obj_name, dir, lock)
     Bos_World *world;
     char *obj_name;
     char *dir;
     int lock;
{
  Bos_Object *obj;
  GDBM_FILE f;
  datum key;
  datum bytes;
  int result;
  char errmsg[300];

  f = Bos_InitializeStorage(dir, lock, GDBM_READER);
  if (f == (GDBM_FILE)0)
    return BOS_ERROR;
  key.dptr = obj_name;
  key.dsize = strlen(obj_name);
  bytes = gdbm_fetch(f, key);
  if (bytes.dptr == NULL)
    return BOS_NOT_FOUND;
  obj = Bos_CreateNewObject(world, obj_name);
  if (Bos_ByteStream2ObjectSlots(obj, bytes.dptr, 0, errmsg) < 0) {
    fprintf(stderr, "BOS ERROR: FetchObject(%s,%s,%d): %s\n",
	    obj_name, dir, lock, errmsg);
    result = BOS_ERROR;
    Bos_Destroy(world, obj_name);
  } else {
    Tcl_CreateCommand(Bos_Methods(world), obj_name, Bos_ObjectCmd, world, 0);
    result = BOS_OK;
  }
  free(bytes.dptr);
  return result;
}

int
Bos_RemoveObject(obj_name, dir, lock)
     char *obj_name;
     char *dir;
     int lock;
{
  GDBM_FILE f;
  datum key;

  f = Bos_InitializeStorage(dir, lock, GDBM_WRCREAT);
  if (f == (GDBM_FILE)0)
    return BOS_ERROR;
  key.dptr = obj_name;
  key.dsize = strlen(obj_name);
  if (gdbm_delete(f, key) < 0)
    return BOS_NOT_FOUND;
  return BOS_OK;
}

int
Bos_IsObjectStored(obj_name, dir, lock)
     char *obj_name;
     char *dir;
     int lock;
{
  GDBM_FILE f;
  datum key;
  datum bytes;
  int result;

  f = Bos_InitializeStorage(dir, lock, GDBM_READER);
  if (f == (GDBM_FILE)0)
    return BOS_ERROR;
  key.dptr = obj_name;
  key.dsize = strlen(obj_name);
  bytes = gdbm_fetch(f, key);
  result = (bytes.dptr != NULL);
  if (result)
    free(bytes.dptr);
  return result;
}

#define USAGE(_e,_u) \
Tcl_AppendResult(interp,argv[0],": ",_e," -- usage: \"",argv[0]," ",_u,"\"",0);\
return TCL_ERROR

int
Bos_ListStoredObjectsCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  GDBM_FILE f;
  datum k;
  char *dir;

  if (argc != 2) {
    USAGE("wrong # args","dir");
  }
  dir = argv[1];
  f = Bos_InitializeStorage(dir, 0, GDBM_READER);
  k = gdbm_firstkey(f);
  while (k.dptr != NULL) {
    Tcl_AppendResult(interp, k.dptr, " ", 0);
    k = gdbm_nextkey(f, k);
  }
  return TCL_OK;
}

static char *
_gdbmErrMsg(code)
     gdbm_error code;
{
  static struct {gdbm_error code; char *text;} _msgs[] = {
    { GDBM_NO_ERROR, "No error" },
    { GDBM_MALLOC_ERROR, "Malloc error" },
    { GDBM_BLOCK_SIZE_ERROR, "Block size error" },
    { GDBM_FILE_OPEN_ERROR, "File open error" },
    { GDBM_FILE_WRITE_ERROR, "File write error" },
    { GDBM_FILE_SEEK_ERROR, "File seek error" },
    { GDBM_FILE_READ_ERROR, "File read error" },
    { GDBM_BAD_MAGIC_NUMBER, "Bad magic number" },
    { GDBM_EMPTY_DATABASE, "Empty database" },
    { GDBM_CANT_BE_READER, "Can't be reader" },
    { GDBM_CANT_BE_WRITER, "Can't be writer" },
    { GDBM_READER_CANT_DELETE, "Reader can't delete" },
    { GDBM_READER_CANT_STORE, "Reader can't store" },
    { GDBM_READER_CANT_REORGANIZE, "Reader can't reorganize" },
    { GDBM_UNKNOWN_UPDATE, "Unknown update" },
    { GDBM_ITEM_NOT_FOUND, "Item not found" },
    { GDBM_REORGANIZE_FAILED, "Reorganize failed" },
    { GDBM_CANNOT_REPLACE, "Cannot replace" },
    { GDBM_NO_ERROR, (char *)0 }
  };
  static char _unknMsg[200];
  int i;

  for (i = 0; _msgs[i].text != (char *)0; i++)
    if (_msgs[i].code == code)
      return _msgs[i].text;
  sprintf(_unknMsg, "Unknown GDBM error code: %d", code);
  return _unknMsg;
}

static void
fatal_func()
{
  extern gdbm_error gdbm_errno;

  fprintf(stderr, "major bloody problem #4 in GDBM (%d: %s)\n",
	  gdbm_errno, _gdbmErrMsg(gdbm_errno));
  Bos_DumpStorageStatus(stderr);
  abort();
}

void
Bos_DumpStorageStatus(f)
     FILE *f;
{
  fprintf(f, "Bos_GDBM storage file chain:\n");
  if (head == (StorageInfo *)0)
    fprintf(f, "  Empty.\n");
  else {
    StorageInfo *h;

    for (h = head; h != (StorageInfo *)0; h = h->next)
      fprintf(f, "  %s%x: gdbmf=%x dir=%s mode=%x lock=%d next=%x\n",
	      (h == last_used)? "*": " ", h,
	      h->gdbmf, h->directory, h->mode, h->lock_file, h->next);
  }
  fflush(f);
}
