#
# Generic Conference
# ------------------

#
# This module contains routines which are common to all groupware conference
# applications.  Typically it is sourced by the specific application (e.g.
# brainstorm.tcl).
#
# The generic conference essentially handles all the details of maintaining
# connections with other conference users; the main data structure for this
# is the "others" list, which holds the list of all other users, where each
# user is represented as a keyed list.  Interactions between the generic
# conference and the Registrar Clients (see rc.tcl) result in users being
# added to or deleted from this others list.
#
# Each keyed list in the "others" list maintains at least the following
# useful information on each user:
#
#    usernum  - the unique user number for that user
#    userid   - the login id of the user
#    username - the user's proper name (if available) or the userid
#    filedesc - the file descriptor used by tcl-dp to send messages on
#
# Additionally, the variable "ouropts" is a keyed list containing information
# about ourself, including at least the following:
#
#    confname   - the name of our conference
#    confType   - the type of our conference
#    originator - the host and port of the Registrar Client who initiated
#                 the conference
#    confnum    - the unique id number of the conference
#    reghost    - the host of the Registrar Client that started the conference
#    regport    - the port of the Registar Client that started the conference
#    usernum    - our unique user number 
#    username   - our proper name (if available) or the userid
#    userid     - our login id
#

initGkErrorHandling

catch {source ~/.tclgkrc}
if {[info exists internetdomain] == 0} {
    set internetdomain ""
}
set hostprefix [exec hostname]
set host $hostprefix$internetdomain


#
# This routine initializes the conference application, and should be the
# first thing called by the actual application conference.  It sets up
# several things, and the connects back to the Registar Client that spawned
# us, then telling the Registar Client who we are so that messages from
# the registration system can be sent to us (see rc.tcl).
#

proc initConf {argv} { global others ouropts host
    set myport [MakeRPCServer 0]
    set others {}
    set ouropts $argv
    set regclient [MakeRPCClient [keylget ouropts reghost] \
		   [keylget ouropts regport]]
    keylset msg confnum [keylget ouropts confnum] host $host \
	port $myport usernum [keylget ouropts usernum]
    RDO $regclient remoteInfo $msg
}


#
# This routine is used to exchange information with other conferences
# that establish connections to us (usually as a result of connectTo
# calls initiated by the registration system).  We receive the remote user's
# user number and other information, and if requested, send the remote user
# our information.
#
# When we receive this information, the remote user is officially "in" our
# conference, and is added to the others list automatically.  If you wish
# your program to be notified when new users enter, put a write trace on
# the global variable "new_user" which will be set to the keyed list of
# information on the new user.
#

proc remoteInfo {whom echo} {  global rpcFile others new_user ouropts
    if {[finduser [keylget whom usernum]] == ""} {
	keylset whom filedesc $rpcFile
	lappend others $whom
	set new_user $whom
    }
    if {$echo} {
	RDO $rpcFile remoteInfo $ouropts 0
    }
}


#
# The registration system calls this to tell us to hook up to a particular
# user, whose host and port number are specified in the parameter whom.  
# After initiating a connection, we send the remote conference information
# on ourselves, and request that they send us their information.  Note that
# the new user is not in the conference until we receive this information
# from them (see proc remoteInfo, above).
#

proc connectTo whom { global ouropts
    set file [MakeRPCClient [keylget whom host] [keylget whom port]]
    RDO $file remoteInfo $ouropts 1
}


#
# The registration system calls this to tell us to remove the indicated user
# from the conference.  The others list is updated.  If the user is ourself
# the deleteconf routine (below) is called to delete the conference application
#
# If you wish your program to be notified when users leave a conference,
# put a write trace on the global variable "deleted_user" which will be set
# to a keyed list containing at least the user number of the deleted user.
#

proc removeUser whom { global others deleted_user ouropts
    set i 0; while { $i < [llength $others] } {
	set tmp [lindex $others $i]
	if {[keylget tmp usernum] == [keylget whom usernum] } {
	    set others [lreplace $others $i $i]
	    set deleted_user $whom
	}
	incr i
    }
    if {[keylget ouropts usernum] == [keylget whom usernum]} {
	deleteconf
    }
}


#
# The procedure called when this conference is to be deleted, either 
# because the whole conference (ie. all users) are deleted, or because
# the local user has left the conference.  You may wish to override this
# to do some sort of cleanup.
#

proc deleteconf {} {
    exit
}


#
# This routine returns the keyed list representing the particular user
# given their user number.
#

proc finduser idnum {  global others;
    foreach i $others {
	set user [keylget i usernum]
	if {$user == $idnum} { return $i }
    }
    return ""
}


#
# This convenience routine just returns the others list
# 

proc others {} { global others;
    return $others
}


#
# This routine sets an attribute in the keyed list for the particular user;
# works for both the others list and ourself
#  NOT TESTED YET
#

proc setuserattr {usernum key value} {   
    global others ouropts
    if {$usernum == [keylget ouropts usernum]} {
	keylset ouropts $key $value
    } else {
	set idx 0; while {$idx < [llength $others]} {
	    set i [lindex $others $idx]
	    if {$usernum == [keylget i usernum]} {
		set tmp $i
		keylset tmp $key $value
		set others [lreplace $others $idx $idx $tmp]
	    }
	    incr idx
	}
    }
}


#
# This routine gets an attribute in the keyed list for the particular user;
# works for both the others list and ourself
#  NOT TESTED YET
#

proc getuserattr {usernum key} { 
    global ouropts others
    if {$usernum == [keylget ouropts usernum]} {
	return [keylget ouropts $key]
    } else {
	foreach i $others {
	    if {$usernum == [keylget i usernum]} {
		return [keylget i $key]
	    }
	}
    }
    return ""
}

