/*
 *	hpux.c - dynamic loading support for TCL under HP-UX 8.x/9.x
 *
 *	Derived from code contributed by Michael Moore (mdm@cis.ohio-state.edu)
 *
 *  -----------------------------------------------------------------
 *    Copyright 1993 D.I.S. - Universita` di Pavia - Italy
 *  -----------------------------------------------------------------
 *
 *  Permission to  use,  copy,   modify,   distribute  this  software
 *  and  its  documentation for any purpose is hereby granted without
 *  fee, provided that the above copyright  notice   appear   in  all
 *  copies   and  that both that copyright notice and this permission
 *  notice appear in supporting documentation, and that the  name  of
 *  D.I.S.   not  be  used  in advertising or publicity pertaining to
 *  distribution of the software without specific, written prior per-
 *  mission.   D.I.S.  makes no representations about the suitability
 *  of this software for any purpose.  It is provided "as is" without
 *  express or implied warranty.
 *
 *  D.I.S. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, IN-
 *  CLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 *  NO EVENT SHALL D.I.S.  BE LIABLE FOR  ANY  SPECIAL,  INDIRECT  OR
 *  CONSEQUENTIAL  DAMAGES  OR  ANY DAMAGES WHATSOEVER RESULTING FROM
 *  LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION  OF  CONTRACT,
 *  NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNEC-
 *  TION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dl.h>

#include <tcl.h>

extern int errno;


/*
 *----------------------------------------------------------------------
 *
 * Tcl_LoadCmd --
 *
 *  This procedure is invoked to load an extension into a running TCL
 *  interpreter.
 *  See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */

    /* ARGSUSED */
int
Tcl_LoadCmd(dummy, interp, argc, argv)
    ClientData dummy;          /* Not used. */
    Tcl_Interp *interp;        /* Current interpreter. */
    int argc;                  /* Number of arguments. */
    char **argv;               /* Argument strings. */
{
    char *s, *fileName, initName[256];

    shl_t xlib;
    int (*init)_ANSI_ARGS_((Tcl_Interp *));

    /*
     *    Syntax check
     */
    if ( argc != 2 ) {
        sprintf(interp->result, "wrong # args: should be \"%s file\"", argv[0]);
        return TCL_ERROR;
    }
    
    /*
     *    Extract the name of the initialization function from a library 
     *    path name in the form:
     *        [<path>/]lib<exName>.sl
     */
    fileName=( (s=strrchr(argv[1], '/')) != NULL )? s+1 : argv[1];

    if ( sscanf(fileName, "lib%[^.].sl", initName) != 1 ) {
        interp->result="bad library name";
        return TCL_ERROR;
    }

    initName[0]=toupper(initName[0]);
    strcat(initName, "_Init");


    /*
     *    If the named library does not exist, if it's not loadable or if
     *    it does not define the init proc, report an error
     */
    if ( (xlib=shl_load(argv[1], BIND_IMMEDIATE, 0L)) == NULL ) {
        Tcl_SetResult(interp, strerror(errno), TCL_VOLATILE);
        return TCL_ERROR;
    }
    
    if ( shl_findsym(&xlib, initName, TYPE_PROCEDURE, (long *)&init) == -1 
            || init == NULL ) {
        Tcl_SetResult(interp, strerror(errno), TCL_VOLATILE);
        shl_unload(xlib);
        return TCL_ERROR;
    }

    /*    
     *    Try to init the extension and report success or failure
     */
    return (*init)(interp);
}

