/*
 * Copyright 1992 by the National Optical Astronomy Observatories(*)
 *
 * Permission to use, copy, and distribute
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * This software is provided "as is" without any express or implied warranty.
 *
 * (*) Operated by the Association of Universities for Research in
 *     Astronomy, Inc. (AURA) under cooperative agreement with the
 *     National Science Foundation.
 */
/* Program: ipcHandles.c
 *      This file contains an interface to the extended TCL handles
 *      routines for tclIPC.
 *
 * Created: K. Gillies 26 June 1992
 * SCCS INFO
 *      @(#)ipcHandles.c	1.1 9/11/92
 *
 */
#include <stdio.h>
#include <assert.h>
#include <sys/types.h>
#include <string.h>
#include "tclExtend.h"     /* Included with distribution */
#include "tclIPC.h"
#include "tclipcP.h"
#include "tclipc.h"            /* Public interface */
#include "ipcHandles.h"

/* Compare strings */
#define STREQU(str1, str2) \
        ((str1[0] == str2[0]) && (strcmp (str1, str2) == 0))

#define HANDLEROOT   "ipc" /* All handles will be ipcxx */

/* IPCHLENGTH is defined in tclipc.h */
/* Table structure is private */
typedef struct _sendTableRecord {
  int useCount;            /* Commands that currently share globals */
  void *tblHdrPtr;         /* Send handle table */
  char curName[IPCHLENGTH]; /* Current name */
} sendTableRecord, *sendTable;

/* In file prototypes */
static void ipcHandleDumpOneEntry _ANSI_ARGS_((Tcl_Interp *interp, 
               char *handleName, sendTableEntry entry));

/* The send/receive table.  Currently only one GLOBAL table. */
static sendTable srtable;

/*
 *--------------------------------------------------------------
 *
 * ipcHandleTblInit
 *
 *      Initialize the handle table and set the number of commands
 *      that use the table.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
void ipcHandleTableInit(commandcount)
int commandcount;
{
  srtable = (sendTable)ckalloc(sizeof (sendTableRecord));
  assert(srtable != NULL);
  srtable->tblHdrPtr =
    Tcl_HandleTblInit(HANDLEROOT, sizeof(sendTableEntryRecord), 5);

  srtable->useCount = commandcount;         /* Number of commands */
}

/*
 *--------------------------------------------------------------
 *
 * ipcHandleAddEntry
 *
 *      Create a new handle and add program, hostname, and
 *      whether it's a receiver or sender.  Called by ipcNewSender
 *      and ipcNewReceiver
 *
 * Results:
 *      Returns the new handle name.
 *--------------------------------------------------------------
 */
char *ipcHandleAddEntry(sender, receiver, hostname, program)
Sender sender;
Receiver receiver;
char *hostname;
char *program;
{
  sendTable tbl;
  sendTableEntry entry;
  char *nameptr;

  /* Currently use global */
  tbl = srtable;
  nameptr = (sender == NULL) ? receiver->handleName : sender->handleName;
  strcpy(nameptr, "");
  /* We are all set so now set up the table entry */
  entry = (sendTableEntry)Tcl_HandleAlloc(tbl->tblHdrPtr, nameptr);
  assert(entry != NULL);

  /* Initialize the values */
  entry->s_sender = sender;
  entry->s_receiver = receiver;
  strncpy(entry->s_hostname,hostname, ENTRYNAMESIZE);
  strncpy(entry->s_program, program, ENTRYNAMESIZE);
  return nameptr;
}

/*
 *--------------------------------------------------------------
 *
 * ipcHandleDestroy
 *
 *      Remove a handle with handleName from the table.  Called
 *      by ipcDestroySender and ipcDestroyReceiver.
 *
 * Results:
 *      Returns TCL_OK or TCL_ERROR.
 *--------------------------------------------------------------
 */
int ipcHandleDestroy(interp, handleName)
Tcl_Interp *interp;
char *handleName;
{
  sendTable tbl;
  sendTableEntry entry;
    
  /* Currently use global */
  tbl = srtable;
  if ((entry = Tcl_HandleXlate(interp, tbl->tblHdrPtr, handleName)) == NULL) {
    return TCL_ERROR;
  }

  /* Free the entry */
  Tcl_HandleFree(tbl->tblHdrPtr, entry); 

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ipcHandleGet
 *
 *      Go to the Handle table and return the entry for handleName.
 *
 * Results:
 *      NULL or the entry.
 *--------------------------------------------------------------
 */
sendTableEntry ipcHandleGet(interp, handleName)
Tcl_Interp *interp;
char *handleName;
{
  sendTable tbl;

  /* Currently use global */
  tbl = srtable;
  
  /* Returns NULL if not found */  
  return Tcl_HandleXlate(interp, tbl->tblHdrPtr, handleName);
}

/*
 *--------------------------------------------------------------
 *
 * ipcHandleDump
 *
 *      Dump the state of the handle table.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
int ipcHandleDump(interp)
Tcl_Interp *interp;
{
  sendTable tbl;
  sendTableEntry entry;
  int walkKey;
  char handleName[20];

  /* Currently use global */
  tbl = srtable;
  
  walkKey = -1;
  while ((entry = (sendTableEntry)Tcl_HandleWalk((void_pt)tbl->tblHdrPtr, 
						 &walkKey)) != NULL) {
    Tcl_WalkKeyToHandle((void_pt)tbl->tblHdrPtr, walkKey, handleName);
    ipcHandleDumpOneEntry(interp, handleName, entry);
  }
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ipcHandleDumpOneEntry
 *
 *      Append the state of one handle.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
static void ipcHandleDumpOneEntry(interp, handleName, entry)
Tcl_Interp *interp;
char *handleName;
sendTableEntry entry;
{
  if (entry == NULL) {
    return;
  }
  if (entry->s_receiver != NULL != NULL) {
    Tcl_AppendResult(interp,
		     "Entry: ", handleName, " is a: Receiver.\n", 
		     "\tProgram: ", entry->s_program,
		     " PortManHost: ", entry->s_hostname, "\n",
		     (char *)NULL);

  } else if (entry->s_sender != NULL) {
    /* Show sender specific information */
    Tcl_AppendResult(interp,
		     "Entry: ", handleName, " is a: Sender.\n", 
		     "\tProgram: ", entry->s_program,
		     " Destination host: ", entry->s_hostname, "\n",
		     (char *)NULL);
  }
}

/*
 *--------------------------------------------------------------
 *
 * ipcDestroyAll
 *
 *      Walks handle list and deletes all Senders and
 *      receivers. 
 *
 * Results:
 *      Returns TCL_OK or TCL_ERROR
 *--------------------------------------------------------------
 */
int ipcDestroyAll(interp)
Tcl_Interp *interp;
{
  sendTable tbl = srtable;
  sendTableEntry entry;
  int walkKey;
  char handleName[20];

  /* This must be done twice.  The first time, just delete Senders
   * in case a Sender is talking to a local Receiver.  A local
   * receiver has a special Destroy.
   */
  walkKey = -1;
  while ((entry = (sendTableEntry)Tcl_HandleWalk((void_pt)tbl->tblHdrPtr,
                                                 &walkKey)) != NULL) {
    Tcl_WalkKeyToHandle((void_pt)tbl->tblHdrPtr, walkKey, handleName);
#ifdef DEBUG
    fprintf(stderr, "Destroy handleName: %s\n", handleName);
#endif
    if(entry->s_sender != NULL) {
#ifdef DEBUG
      fprintf(stderr, "Destroy sender on: %s\n", entry->s_hostname);
#endif
      ipcDestroySender(interp, entry->s_sender); 
    }
  }
  /* Now remove the receivers */
  walkKey = -1;
  while ((entry = (sendTableEntry)Tcl_HandleWalk((void_pt)tbl->tblHdrPtr,
                                                 &walkKey)) != NULL) {
    Tcl_WalkKeyToHandle((void_pt)tbl->tblHdrPtr, walkKey, handleName);
#ifdef DEBUG
    fprintf(stderr, "Destroy receiver on portman: %s\n", entry->s_hostname);
#endif
    ipcDestroyReceiver(interp, entry->s_receiver); 
  }
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ipcGetInterp
 *
 *      Searches the handle table for the receiver that matches
 *      the port of the incoming messages and sets the
 *      correct interpreter.  
 *
 * Results:
 *      Returns a interpreter as an argument and TCL_OK or TCL_ERROR
 *--------------------------------------------------------------
 */
int ipcGetInterp(portIn, interp)
int portIn;
Tcl_Interp **interp;
{
  sendTable tbl = srtable;
  sendTableEntry entry;
  int walkKey;
  
  walkKey = -1;
  while ((entry = (sendTableEntry)Tcl_HandleWalk((sendTable)tbl->tblHdrPtr, 
						 &walkKey)) != NULL) {
    if (entry->s_receiver != NULL) {
      if (entry->s_receiver->receivePort.portNumber == portIn) {
#ifdef DEBUG
      fprintf(stderr, "Found: %s %d\n", entry->s_program,
	      entry->s_receiver->receivePort.portNumber);
#endif
	*interp = entry->s_receiver->interp;
        return TCL_OK;
      }
    }
  }
  *interp = (Tcl_Interp *)NULL;
  return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ipcCheckForLocal
 *
 *      Searches the handle table for the receiver that matches
 *      the port of the incoming messages and sets the
 *      correct interpreter.  
 *
 * Results:
 *      Returns a interpreter as an argument and TCL_OK or TCL_ERROR
 *--------------------------------------------------------------
 */
Receiver ipcCheckForLocal(port)
Port *port;
{
  sendTable tbl = srtable;
  sendTableEntry entry;
  int walkKey;
  
  walkKey = -1;
  while ((entry = (sendTableEntry)Tcl_HandleWalk((sendTable)tbl->tblHdrPtr, 
						 &walkKey)) != NULL) {
    if (entry->s_receiver != NULL) {
      if (entry->s_receiver->receivePort.portNumber == port->portNumber &&
          STREQU(port->hostName, ipcThisHostName)) {
#ifdef DEBUG
        fprintf(stderr, "Local: %d %s\n", port->portNumber, ipcThisHostName);
#endif
        return entry->s_receiver;
      }
    }
  }
  return NULL;
}
