/*
 * Copyright 1992 by the National Optical Astronomy Observatories(*)
 *
 * Permission to use, copy, and distribute
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * This software is provided "as is" without any express or implied warranty.
 *
 * (*) Operated by the Association of Universities for Research in
 *     Astronomy, Inc. (AURA) under cooperative agreement with the
 *     National Science Foundation.
 */
/* Program: tclpm.c
 *     This file contains the PortManager specific commands. 
 *
 * Created: K. Gillies 26 June 1992
 * SCCS INFO
 *      @(#)tclpm.c	1.1 9/11/92
 *
 */
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <sys/types.h>
#include "tcl.h"
#include "tclIPC.h"
#include "tclipcP.h"
#include "tclipc.h"
#include "ipcHandles.h"
#include "plist.h"
#include "tclpm.h" 

#define STREQU(str1, str2) \
        ((str1[0] == str2[0]) && (strcmp (str1, str2) == 0))
#define STRNEQU(str1, str2, cnt) \
        ((str1[0] == str2[0]) && (strncmp (str1, str2, cnt) == 0))

#define ANYHOST       "AnyHost"
#define PORTMGRNAME   "portmanager"

static PortManList ports_inuse;       /* The list of connections */
static char portMgrHostName[256];     /* The host name */
static Receiver portMgrReceiver;      /* The only PortManager receiver */
static int descriptorTableSize;       /* Cache table size value */

/* Holds last time the Port Manager scanned its list of... */
/* ...applications to see if any were invalid ("dead") */
static struct timeval lastTestTime;

/* In file prototypes */
static int cmdConnect _ANSI_ARGS_((ClientData *clientData, 
				   Tcl_Interp *interp, 
				   int argc, char *argv[]));
static int cmdIsPortUnique _ANSI_ARGS_((ClientData *clientData, 
					Tcl_Interp *interp, 
					int argc, char *argv[]));
static int cmdDisconnect _ANSI_ARGS_((ClientData *clientData, 
				      Tcl_Interp *interp, 
				      int argc, char *argv[]));
static int cmdPortFromName _ANSI_ARGS_((ClientData *clientData, 
					Tcl_Interp *interp, 
					int argc, char *argv[]));
static int cmdOpenApps _ANSI_ARGS_((ClientData *clientData, 
				    Tcl_Interp *interp, 
				    int argc, char *argv[]));
static int pmDump _ANSI_ARGS_((ClientData *clientData, 
			       Tcl_Interp *interp, 
			       int argc, char *argv[]));
static int pmCheckUniqueness _ANSI_ARGS_((char *appName, char *hostName));

/*
 *--------------------------------------------------------------
 *
 * tclPmInit
 *
 *      Initialize an interpreter for use as a PortManager.
 *      Add the Portmanager commands, initialize the PortManager
 *      list and make a receiver entry for this PortManager.
 *      commands.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
void tclPmInit(interp)
Tcl_Interp *interp;
{
  Port portMgrSender;
  Sender sender = NULL;             /* Used to create receiver */
  struct timeval now;		    /* Holds the current time */
  int tdayresult;
  char tmpbuff[256];                /* Used to register portmanager */

  /* Initialize the PortManager list */
  pListInit(&ports_inuse);
  assert(ports_inuse != NULL);

  /* Create PortManager commands */
  Tcl_CreateCommand(interp, "pmdump", (Tcl_CmdProc *)cmdOpenApps,
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "connect", (Tcl_CmdProc *)cmdConnect,
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "isunique", (Tcl_CmdProc *)cmdIsPortUnique,
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "disconnect", (Tcl_CmdProc *)cmdDisconnect,
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "portfromname", (Tcl_CmdProc *)cmdPortFromName,
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "openapps", (Tcl_CmdProc *)cmdOpenApps,
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);

  /* Register the portmanager */
  (void)gethostname(portMgrHostName, sizeof(portMgrHostName));
  portMgrSender.hostName = strdup(portMgrHostName);

  /* Special mark to do little in NewSender and NewReceiver */
  /* Create a "dummy" sender */
  portMgrSender.portNumber = 0;
  sender = NewSender(interp, &portMgrSender);

  /* Make a receiver for the PortManager */
  portMgrReceiver = NewReceiver(sender, 
                                "PortManager", PortMgrPortNumber); 
  /* Leave if the receiver could not be created */
  if (portMgrReceiver == (Receiver)NULL) {
    fprintf(stderr, 
  "Port Manager could not reserve port %d to listen for messages. Exiting.\n", 
	    PortMgrPortNumber);
    exit(-1);
  }

  /* We are all set so now set up the sender/receiver table entry */
  Tcl_SetResult(interp,
		ipcHandleAddEntry(NULL, portMgrReceiver, 
				  portMgrHostName, PORTMGRNAME), TCL_STATIC);
  /* Execute the connect command for ourselves */
  sprintf(tmpbuff, "connect %s %s %d", 
	  PORTMGRNAME, portMgrHostName, PortMgrPortNumber);
  if (Tcl_Eval(interp, tmpbuff, 0, NULL) == TCL_ERROR) {
    fprintf(stderr, "Could not register \"PortManager\".  Exiting.\n"); 
    exit(-1);
  }
#ifdef DEBUG
  printf("Listening on host %s on port %d.\n",
	 portMgrHostName, PortMgrPortNumber);
#endif
  /* Set the global table size once */
  descriptorTableSize = getdtablesize();

  /* Get current time to setup listen */
  tdayresult = gettimeofday(&now,(struct timezone*)NULL);
  if (tdayresult != 0) {
    /* Error, print an error and exit the program */
    fprintf(stderr, "Could not get the time of day.  Exiting.\n");
    exit(tdayresult);
  }
  /* Initialize the last time scanned to now */
  lastTestTime.tv_sec = now.tv_sec;
  lastTestTime.tv_usec = now.tv_usec;
}

/*
 *--------------------------------------------------------------
 *
 * pmDump
 *
 *      TCL command to list the connections list.
 *
 * Results:
 *      TCL_OK always!
 *--------------------------------------------------------------
 */
static int pmDump(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  if (argc != 1) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], (char *)NULL);
    return TCL_ERROR;
  }
  pListPrint(ports_inuse);
  
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * cmdConnect
 *
 *      TCL command to register a receiver with the PortManager.
 *
 * Results:
 *      TCL_OK or TCL_ERROR
 *--------------------------------------------------------------
 */

static int cmdConnect(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  char *appName = NULL;
  char *hostName = NULL;
  int portNumber;
  Port newPort;
  
  if (argc != 4) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], " appName hostName portNumber\"", 
      (char *) NULL);
    return TCL_ERROR;
  }
  /* Use argv[1] as host and argv[2] as the program name */
  appName = argv[1];
  hostName = argv[2];
  /* Get the port number value */
  if (Tcl_GetInt(interp, argv[3], &portNumber) != TCL_OK) {
    Tcl_AppendResult(interp, ", the port arg is not an integer.", 
                     (char *)NULL);
    return TCL_ERROR;
  }
#ifdef DEBUG
  fprintf(stderr, "CONNECT: %s %s %d\n", appName, hostName, portNumber);
#endif
  /* First check uniqueness.  Host/label pairs must be unique. */
  if (!pmCheckUniqueness(appName, hostName)) {
    Tcl_AppendResult(interp, "The app-host combination: ",
		     appName, "-",hostName, " is not unique.\n", 
                     (char *)NULL);
    /* Return an error if it's not unique */
    return TCL_ERROR;
  }
  /* It's ok so register it */
  newPort.appName = appName;
  newPort.hostName = hostName;
  newPort.portNumber = portNumber;
  /* Adding never fails? */
  pListAddPortToList(ports_inuse, &newPort);

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * cmdDisconnect
 *
 *      The TCL command that removes a host/label receiver from
 *      the PortManager receiver list.
 *
 * Results:
 *      TCL_ERROR or TCL_OK
 *--------------------------------------------------------------
 */
static int cmdDisconnect(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  char *appName = NULL;
  char *hostName = NULL;
  int portNumber;
  Port thePort;
  
  if (argc != 4) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], " appName hostName portNumber\"", 
      (char *) NULL);
    return TCL_ERROR;
  }
  /* Use argv[1] as host and argv[2] as the program name */
  appName = argv[1];
  hostName = argv[2];
  /* Get the port number */
  if (Tcl_GetInt(interp, argv[3], &portNumber) != TCL_OK) {
    return TCL_ERROR;
  }
#ifdef DEBUG
  fprintf(stderr, "DISCONNECT: %s %s %d\n", appName, hostName, portNumber);
#endif 
  /* Remove the connection */
  thePort.appName = appName;
  thePort.hostName = hostName;
  thePort.portNumber = portNumber;
  pListRemovePortFromList(ports_inuse, &thePort);
  /* This always succeeds or it doesn't matter. */
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * cmdPortFromName
 *
 *      This command is sent when a client wishes to locate a
 *      label/host pair.  Clients can look up just labels or require
 *      a perfect match of label and host name.
 *
 * Results:
 *      TCL_ERROR or TCL_OK
 *--------------------------------------------------------------
 */
static int cmdPortFromName(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  char *appName = NULL;
  char *hostName = NULL;
  Port *port;
  generic_ptr member;
  char result[512];
  
  if (argc > 3) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], " appName [hostName]\"", 
      (char *) NULL);
    return TCL_ERROR;
  }
  /* Use localhost, argv[1] is the program name */
  appName = argv[1];
  if (argc == 2) {
    /* Just match the program name */
    hostName = ANYHOST;
    CollectionForEach(ports_inuse, member) {
      port = (Port *)member;
      if (strcmp(appName, port->appName) == 0) {
#ifdef DEBUG
        fprintf(stderr, "App Name: %s, Host Name: %s, Port Number: %d.\n",
	       port->appName,
	       port->hostName,
	       port->portNumber);
#endif
        /* Add the found pair to the result */
        sprintf(result, "%s %d", port->hostName, port->portNumber);
        Tcl_AppendElement(interp, result, 0);
      }
    } CollectionNext();
  } else if (argc == 3) {
    /* Match the program name and host */
    hostName = argv[2];
    CollectionForEach(ports_inuse, member) {
      port = (Port *)member;
      if ((strcmp(appName, port->appName) == 0) &&
          (strcmp(hostName, port->hostName) == 0)) {
#ifdef DEBUG
        printf("App Name: %s, Host Name: %s, Port Number: %d.\n",
	       port->appName,
	       port->hostName,
	       port->portNumber);
#endif
        /* Add the found pair to the result */
        sprintf(result, "%s %d", port->hostName, port->portNumber);
        Tcl_AppendElement(interp, result, 0);
      }
    } CollectionNext();
  }
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * pmCheckUniqueness
 *
 *      Check the PortManager receiver list for an already
 *      present appName/hostName pair.
 *
 * Results:
 *      TRUE if unique, FALSE if not.
 *--------------------------------------------------------------
 */
static int pmCheckUniqueness(appName, hostName)
char *appName;
char *hostName;
{
  Port *port;
  generic_ptr member;
  int isunique = TRUE;          /* Assume it's unique */

  /* If hostName is Null, just check appName */
  if (hostName == NULL) {
    /* Just match the program name */
    CollectionForEach(ports_inuse, member) {
      port = (Port *)member;
      if (strcmp(appName, port->appName) == 0) {
#ifdef DEBUG
        fprintf(stderr, "App Name: %s, Host Name: %s, Port Number: %d.\n",
	       port->appName, port->hostName, port->portNumber);
#endif
        isunique = FALSE;
      }
    } CollectionNext();
  } else {
    /* Match the program name and host */
    CollectionForEach(ports_inuse, member) {
      port = (Port *)member;
      if ((strcmp(appName, port->appName) == 0) &&
          (strcmp(hostName, port->hostName) == 0)) {
#ifdef DEBUG
        fprintf(stderr, "App Name: %s, Host Name: %s, Port Number: %d.\n",
	       port->appName, port->hostName, port->portNumber);
#endif
        isunique = FALSE;
      }
    } CollectionNext();
  }
  return isunique;
}


/*
 *--------------------------------------------------------------
 *
 * cmdIsPortUnique
 *
 *      A TCL command to check for a unique label/host pair.
 *      Currently, unused.
 *
 * Results:
 *      TCL_OK and Yes if unque or No if not.
 *--------------------------------------------------------------
 */
static int cmdIsPortUnique(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  int isunique;

  if (argc > 3 || argc < 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], " appName [hostName]\"", (char *) NULL);
    return TCL_ERROR;
  }
  /* Call the uniqueness function and return Yes or No as the result */
  isunique = pmCheckUniqueness(argv[1], argv[2]);
  Tcl_AppendResult(interp, ((isunique) ? "Yes" : "No"), (char *)NULL);
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * cmdOpenApps
 *
 *      A TCL command that dumps the current receiver list in
 *      TCL list format.
 *
 * Results:
 *      TCL_ERROR or TCL_OK
 *--------------------------------------------------------------
 */
static int cmdOpenApps(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  Port *port;
  generic_ptr member;
  char result[512];
  
  if (argc != 1) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], (char *) NULL);
    return TCL_ERROR;
  }
  /* Go through the list and dump the contents */
  CollectionForEach(ports_inuse, member) {
    port = (Port *)member;
#ifdef DEBUG
    printf("App Name: %s, Host Name: %s, Port Number: %d.\n",
            port->appName,
	    port->hostName,
	    port->portNumber);
#endif
    sprintf(result, "%s %s %d", port->appName, 
	                        port->hostName, port->portNumber);
    Tcl_AppendElement(interp, result, 0);
  } CollectionNext();
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * pmListenForMessages
 *
 * This function is responsible for listening for RPC requests 
 * and dispatching them in the PortManager.  It also periodically
 * checks the list of apps that have registered with the Port Manager 
 * to see that they're still running.
 * If an application on the Port Manager's list is no longer running, 
 * this function removes that application from the list.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
void pmListenForMessages()
{
  /* Variable holding list of RPC sockets to listen... */
  fd_set rpcSockets;
  int numSockets = 0;
  /* Used to tell select() how long to wait until it returns */
  struct timeval waitTime;
  struct timeval now;		    /* Holds the current time */
  /* Determines how often to check for "dead" apps */
  int updateInterval = 10;           /* Every x seconds */
  int tdayresult;
  
  /* Get size of descriptor table for later use by select() */
  /* Set up a timeval to wait for 0.05 seconds */
  waitTime.tv_sec = 0;
  waitTime.tv_usec = 50000;
  /* Get current time */
  tdayresult = gettimeofday(&now,(struct timezone*)NULL);
  if (tdayresult != 0) {
    /* Error, print an error and exit the program */
    fprintf(stderr, "Could not get the time of day.  Exiting.\n");
    exit(tdayresult);
  }

  /* Get the list of RPC sockets accepting requests */
  /* Check for input on incoming RPC sockets */
  rpcSockets = svc_fdset;
  numSockets = select(descriptorTableSize, &rpcSockets, 
		      (int*)0, (int*)0, &waitTime);
  /* Based on how many sockets had input, determine what to do */
  switch (numSockets) {
  case -1:
    /* An error occurred */
    break;
  case 0:
    /* There was no input to process, get current time and test
     * to see if it's time to look for _non-responders_.
     */
    tdayresult = gettimeofday(&now,(struct timezone*)NULL);
    if (tdayresult != 0) {
      /* Yes, print an error and exit the program */
      fprintf(stderr, "Could not get the time of day.  Exiting.\n");
      exit(tdayresult);
    }
    /* Did "updateInterval" seconds pass since last scan? */
    if ((now.tv_sec - lastTestTime.tv_sec) >= updateInterval) {
#ifdef DEBUG
      pListPrint(ports_inuse);
#endif
      pListScanForDeadApps(ports_inuse);
      /* Reset the last time scanned to now */
      lastTestTime.tv_sec = now.tv_sec;
      lastTestTime.tv_usec = now.tv_usec;
    }
    break;
  default:
    /* Handle the incoming RPC requests */
    svc_getreqset(&rpcSockets);
    break;
  }
} /* end function ListenForMessages */
