/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifdef USE_MPU401COPY
#include <assert.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strstream.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#ifdef SVR4
#include <sys/filio.h>
#include <sys/midi.h>
#endif

#ifdef LINUX
#include <linux/termios.h>
#include <linux/midi.h>
#endif

#ifdef BSD
#include <i386/isa/midiioctl.h>
#endif

#include "MPU401Copy.h"
#include "EventUtil.h"
#include "Note.h"

typedef void Sigfunc(int);

static MPU401Copy *CurrentMPU401 = 0;
static Sigfunc *posixsignal(int signo, Sigfunc *func);

MPU401Copy::MPU401Copy() : fd(-1), last_rec_time(0), finished(0),
    last_record_rs(0), playbuf(0), playbufptr(0), playbufend(0)
{

	assert(CurrentMPU401 == 0);
	CurrentMPU401 = this;
	posixsignal(SIGIO, SIG_IGN);
}

MPU401Copy::MPU401Copy(const char *dev) : MidiDevice(dev), fd(-1),
    last_rec_time(0), finished(0), last_record_rs(0), playbuf(0),
    playbufptr(0), playbufend(0)
{

	assert(CurrentMPU401 == 0);
	CurrentMPU401 = this;
	posixsignal(SIGIO, SIG_IGN);
}

MPU401Copy::~MPU401Copy()
{
	
	if (fd != -1)
		Stop();
	play_track.Empty();
	CurrentMPU401 = 0;
}

int
MPU401Copy::Play(Song *s, int r)
{
	ostrstream err;
	const char *name;
	char *str;
	int arg;

	// open the device
	if (fd != -1) {
		SetError("Device already open");
		return (0);
	}
	if ((name = GetName()) == 0) {
		SetError("No device set");
		return (0);
	}

	// set repeat
	SetRepeat(r);

	// initialize track/event info
	if (!BuildSMF(play_track, s)) {
		SetError("Out of memory");
		return (0);
	}
	playbufptr = playbuf = play_track.PeekByte();
	playbufend = &playbuf[play_track.GetLength()];

	should_stop = 0;
	finished = 0;

	// set up async stuff
	posixsignal(SIGIO, MPU401CopySigIO);

	// open device
	if ((fd = open(name, O_WRONLY | O_NONBLOCK)) == -1) {
		err << "open failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

#ifndef LINUX
	/* Linux doesn't allow F_SETOWN except for sockets - bug */
	if (fcntl(fd, F_SETOWN, getpid()) == -1) {
		err << "fcntl F_SETOWN failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
#else
	arg = getpid();
	if (ioctl(fd, TIOCSPGRP, &arg) == -1) {
		err << "ioctl TIOCSPGRP failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
#endif

	// set division
	arg = s->GetDivision();
	if (ioctl(fd, MSDIVISION, &arg) == -1) {
		err << "ioctl MSDIVISION failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	arg = GetMidiThru();
	if (ioctl(fd, MTHRU, &arg) == -1) {
		err << "ioctl MTHRU failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	// the following should be a fcntl FIOASYNC call, but the
	// FIOASYNC ioctl doesn't always get passed down to the driver
	// properly under all OSes (read Linux), thus we use the MASYNC
	// entry point.
	arg = 1;
	if (ioctl(fd, MASYNC, &arg) == -1) {
		err << "ioctl MASYNC failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	return (1);
}

int
MPU401Copy::Record(Song *rs, Song *ps, int r)
{
	ostrstream err;
	const char *name;
	char *str;
	int arg;

	if (fd != -1)
		return (0);
	if ((name = GetName()) == 0)
		return (0);

	rec_song = rs;
	last_rec_time = 0;
	last_record_rs = 0;
	if (ps == 0) {
		play_track.Empty();
		playbuf = 0;
		playbufptr = 0;
		playbufend = 0;
		if ((fd = open(name, O_RDONLY)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
		arg = rec_song->GetDivision();
		if (ioctl(fd, MSDIVISION, &arg) == -1) {
			err << "ioctl MSDIVISION failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
	} else {

		// initialize track/event info
		if (!BuildSMF(play_track, ps)) {
			SetError("Out of memory");
			return (0);
		}

		playbufptr = playbuf = play_track.PeekByte();
		playbufend = &playbuf[play_track.GetLength()];

		if ((fd = open(name, O_RDWR | O_NONBLOCK)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

		// set division
		arg = ps->GetDivision();
		if (ioctl(fd, MSDIVISION, &arg) == -1) {
			err << "ioctl MSDIVISION failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

		// start record timer when first event is scheduled to play
		arg = 1;
		if (ioctl(fd, MRECONPLAY, &arg) == -1) {
			err << "ioctl MRECONPLAY failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

		should_stop = 0;
		finished = 0;
	}

	// set repeat
	SetRepeat(r);

	// set up async stuff
	posixsignal(SIGIO, MPU401CopySigIO);
#ifndef LINUX
	/* Linux doesn't allow F_SETOWN except for sockets - bug */
	if (fcntl(fd, F_SETOWN, getpid()) == -1) {
		err << "fcntl F_SETOWN failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
#else
	arg = getpid();
	if (ioctl(fd, TIOCSPGRP, &arg) == -1) {
		err << "ioctl TIOCSPGRP failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
#endif
	arg = GetMidiThru();
	if (ioctl(fd, MTHRU, &arg) == -1) {
		err << "ioctl MTHRU failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	// the following should be a fcntl FIOASYNC call, but the
	// FIOASYNC ioctl doesn't always get passed down to the driver
	// properly under all OSes (read Linux), thus we use the MASYNC
	// entry point.
	arg = 1;
	if (ioctl(fd, MASYNC, &arg) == -1) {
		err << "ioctl MASYNC failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	return (1);
}

int
MPU401Copy::Stop(void)
{
	ostrstream err;
	char *str;
	int arg;

	finished = 1;
	should_stop = 0;
	if (fd == -1)
		return (1);

	posixsignal(SIGIO, SIG_IGN);

	arg = 0;
	if (ioctl(fd, MASYNC, &arg) == -1) {
		err << "ioctl MASYNC failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (ioctl(fd, MGPLAYQ, &arg) == -1) {
		err << "ioctl MGPLAYQ failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	// flush queues
	if (arg != 0)
		if (ioctl(fd, MDRAIN, NULL) == -1) {
			err << "ioctl MDRAIN failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

	close (fd);
	fd = -1;

	play_track.Empty();
	playbuf = 0;
	playbufptr = 0;
	playbufend = 0;
	rec_song = 0;
	return (1);
}

int
MPU401Copy::Wait(void)
{
	sigset_t sigset;

	if (play_track.GetLength() == 0)
		return (1);
	if (finished)
		return (1);
	finished = 0;
	sigemptyset(&sigset);
	do
		sigsuspend(&sigset);
	while (!finished);
	return (1);
}

/*
 * Since this routine is called ASYNC, we just print out errors
 * instead of putting them in SetError
 */
void
MPU401CopySigIO(int notused)
{
	struct timeval t;
	fd_set rset, wset;
	MPU401Copy *mpu;
	MidiDeviceCallback callback;
	const char *errstr;

	mpu = CurrentMPU401;
	if (mpu == 0 || mpu->fd == -1)
		return;

	FD_ZERO(&rset);
	FD_ZERO(&wset);
	FD_SET(mpu->fd, &rset);
	FD_SET(mpu->fd, &wset);
	t.tv_sec = 0;
	t.tv_usec = 0;

	// poll to find out what caused the SIGIO
	if (select(mpu->fd + 1, &rset, &wset, 0, &t) == -1) {
		cerr << "error selecting MPU401Copy: " << strerror(errno)
		    << "\n";
		return;
	}

	// check for record event
	if (FD_ISSET(mpu->fd, &rset)) {
		if (mpu->rec_song != 0) {
			int num_read;
			EventType etype;
			unsigned char event[MaxEventSize];
			SMFTrack track;
			Event *e, *first_e;

			// read event
			// This involves dynamic allocation, but
			// what can we do?  Events might be larger
			// than can fit in the event buffer.
			do {
				if ((num_read = read(mpu->fd, event,
				    MaxEventSize)) == -1) {
					cerr << "error reading MPU401Copy "
					    "event: " << strerror(errno)
					    << "\n";
					return;
				}
				track.PutData(event, num_read);
			} while (num_read == MaxEventSize);

			// process events
			track.SetRunningState(mpu->last_record_rs);
			first_e = 0;
			/*
			 * XXX This is a problem.  ReadEventFromSMFTrack
			 * calls new to create the event.  New uses
			 * malloc which is not reentrant.  I don't know
			 * how to get around this.
			 */
			while ((e = ReadEventFromSMFTrack(track,
			    mpu->last_rec_time, errstr)) != 0) {
				Event *new_e;

				if (first_e == 0)
					first_e = e;
				new_e = mpu->rec_song->PutEvent(0, *e);
				delete e;
				if (new_e == 0) {
					continue;
				}
				etype = new_e->GetType();
				// put links on noteoffs
				if ((etype == NOTEON &&
				    ((NoteEvent *)new_e)->GetVelocity() == 0)
				    || etype == NOTEOFF)
					mpu->rec_song->SetNotePair(0, new_e);
			}
			if (errstr != 0)
				cerr << "Error while recording: " << errstr
				    << "\n";
			mpu->last_record_rs = track.GetRunningState();
			if ((callback = mpu->GetRecordCallback()) != 0)
				callback(first_e);
		}
	}

	if (FD_ISSET(mpu->fd, &wset)) {
		if (mpu->play_track.GetLength() != 0 && !mpu->finished) {
			if (!mpu->Send()) {
				cerr << "Couldn't write play event\n";
				exit(1);
			}
		}
	}
}

int
MPU401Copy::Send()
{
	int numleft, numwritten;
	int arg;

	if (should_stop) {
		if (ioctl(fd, MGPLAYQ, &arg) == -1) {
			cerr << "MPU401Copy: ioctl MGPLAY failed: " <<
			    strerror(errno) << "\n";
			return (0);
		}
		if (arg == 0) {
			// the sigio we just caught signifies all events
			// have been played 
			finished = 1;
			return (1);
		}
	}
	if (playbuf == 0)
		return (0);
	numleft = playbufend - playbufptr;
	if ((numwritten = write(fd, playbufptr, numleft)) == -1) {
		cerr << "Couldn't write: " << strerror(errno) << "\n";
		return (0);
	}
	playbufptr += numwritten;
	numleft -= numwritten;
	if (numleft == 0) {
		if (!GetRepeat()) {
			should_stop = 1;
			return (1);
		} else {
			playbufptr = playbuf;
			numleft = playbufend - playbufptr;
		}
	}
	return (1);
}

int
MPU401Copy::BuildSMF(SMFTrack &smftrack, Song *s)
{
	unsigned long firsttime, lasttime, t;
	int empty, i, lowest, numtracks;
	Event *e, **trackevents;
	const char *errstr;

	numtracks = s->GetNumTracks();
	trackevents = new Event *[numtracks];
	if (trackevents == 0)
		return (0);
	empty = 1;
	// get first events in each track
	for (i = 0; i < numtracks; i++) {
		trackevents[i] = s->GetTrack(i).GetFirstEvent();
		if (trackevents[i] != 0)
			empty = 0;
	}
	lasttime = 0;
	while (!empty) {
		/* find track with lowest numbered event */
		lowest = -1;
		/* quiet warning */
		firsttime = 0;
		for (i = 0; i < numtracks; i++) {
			if (trackevents[i] == 0)
				continue;
			if (lowest == -1) {
				firsttime = trackevents[i]->GetTime();
				lowest = i;
			} else {
				t = trackevents[i]->GetTime();
				if (t < firsttime) {
					firsttime = t;
					lowest = i;
				}
			}
		}
		if (lowest == -1) {
			empty = 1;
			break;
		}
		/* stick events into SMF */
		for (e = trackevents[lowest]; e != 0; e = e->GetNextEvent()) {
			if (!WriteEventToSMFTrack(smftrack, lasttime, e,
			    errstr))
				return (0);
		}
		/* update trackevents pointer */
		trackevents[lowest] = s->GetTrack(lowest).NextEvents(
		    trackevents[lowest]);
	}
	delete trackevents;
	return (1);
}

ostream &
operator<<(ostream &os, const MPU401Copy &mpu)
{

	os << mpu.GetName();
	if (mpu.playbuf != 0)
		os << " Playing";
	if (mpu.rec_song != 0)
		os << " Recording";
	return (os);
}

Sigfunc *
posixsignal(int signo, Sigfunc *func)
{
	struct sigaction act, oact;

	sigemptyset(&oact.sa_mask);
	sigemptyset(&act.sa_mask);
	act.sa_handler = func;
	act.sa_flags = 0;

	if (signo == SIGALRM) {
#ifdef SA_INTERRUPT
		act.sa_flags |= SA_INTERRUPT;	/* SunOS */
#endif
	} else {
#ifdef SA_RESTART
		act.sa_flags |= SA_RESTART;	/* SVR4, 4.4BSD */
#endif
	}

	if (sigaction(signo, &act, &oact) < 0)
		return (SIG_ERR);
	return (oact.sa_handler);
}
#endif
