#---------------------------------------------------------------------------
#
#	Generic window objects
#
#---------------------------------------------------------------------------

defobject Window

defmethod Window DEFAULT {method args} {
  eval [concat [list $self! $method] $args]
}

defmethod Window instantiate {name layout {slots {}}} {

  rename $name $name!
  $self clone $name $slots
  $name layout $layout
}

defmethod Window Dismiss0 {{action {}}} {

  if { $action != {} } {
    uplevel #0 $action
  }

  $self reclaimall
  catch {destroy $self}
  return
}

defmethod Window DismissChildren {{action {}}} {

  if { $action != {} } {
    uplevel #0 $action
  }

  $self reclaimall true
  return
}

defmethod Window Dismiss {{action {}}} {

  $self Dismiss0
  if { $self != "." } {
    update idletasks
    update
  }
  return
}

defmethod Window demos {} {

  Window slot _demos
}

defmethod Window addDemo {name} {

  set demos [Window slot _demos]
  if { [position $name $demos] < 0 } {
    Window slot _demos [concat $demos [list $name]]
  }
  return $name
}

defmethod Window Help {topic} {

  Help new * -help $topic
}

proc defwidget {name {super {}} {slots {}}} {

  if { $super == {} } {
    set super Window
  }
  defobject $name $super $slots
  return $name
}

#---------------------------------------------------------------------------
#
#	Place a window on the screen
#
#---------------------------------------------------------------------------

# window layout {name layout}
#	Place a window on the screen. For toplevel windows, the argument
#	LAYOUT may be one of "" (no placement), "center" (center window
#	on screen), "choose" (let the use choose the position), a window
#	name (center relative to that window), or a geometry specification.
#	For subwindows, LAYOUT specifies packing instructions.
#	Note: For toplevel windows, this defines the position. For all
#	others, this defines packing, i.e. layout.
#

defmethod Window layout {layout} {

  # don't do anything
  if { $layout == {} } then {
    return $self
  }

  # simple packing
  if { [winfo toplevel $self] != $self } then {

    pack append [winfo parent $self] $self $layout
    return $self

  }

  # handle toplevel windows

  wm withdraw $self
  update

  if { $layout == "center" } then {

    set sx [winfo screenwidth .]
    set sy [winfo screenheight .]

    set wx [winfo reqwidth $self]
    set wy [winfo reqheight $self]

    set cx [expr ($sx-$wx)/2]
    set cy [expr ($sy-$wy)/2]

    set layout "+$cx+$cy"

  } elseif { $layout == "choose" } {

    set layout {}

  } elseif { [string index $layout 0] == "." } {

    set sx [winfo rootx $layout]
    set sy [winfo rooty $layout]
    set sw [winfo width $layout]
    set sh [winfo height $layout]

    set wx [winfo reqwidth $self]
    set wy [winfo reqheight $self]

    set layout "+[expr $sx+($sw-$wx)/2]+[expr $sy+($sh-$wy)/2]"
  }

  wm geometry $self $layout
  wm deiconify $self
  update
  wm positionfrom $self {}

  return $self
}

#---------------------------------------------------------------------------

Window slot grabcount 0

defmethod Window grab {args} {

  args	

  catch "grab set -global $self"
}

defmethod Window timeoutGrab {args} {

  args	{time 20}

  update
  catch "grab set -global $self"
  if { $time != {} } {
    set x [uplevel #0 [list incr [slot-variable Window grabcount]]]
    Window slot grabwin $self
    after [expr $time*1000] "if { \[winfo exists $self\] } {$self _ungrab $x}"
  }
  return $self
}

defmethod Window _ungrab {id} {

  if { [Window slot grabwin] == $self && [Window slot grabcount] == $id } {
    $self ungrab
    $self Dismiss
  }
}

defmethod Window ungrab {} {

  foreach win [grab current] {
    grab release $win
  }
}

#---------------------------------------------------------------------------

proc . {args} {
  error ". does not handle $args"
}

Window instantiate . {}

defmethod . Dismiss {} {
  destroy .
}
