 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File:	buttonAux.tcl
#
# Description:
#	Auxiliary procedures needed to support keyboard traversal
#	of buttons using Tab and Shift Tab.
#
# Global variables used:
#c	button_priv(current,$d)
#		Current button under the mouse for display $d.
#c	button_priv(default,$w)
#		Default button for toplevel window $w.
#c	button_priv(relief,$w)
#		Saved relief of button $w while the button is pressed.

 # $Id: buttonAux.tcl,v 1.13 1993/11/01 18:20:46 kennykb Exp $
 # $Source: /homedisk/julius/u0/kennykb/src/tkauxlib_ship/RCS/buttonAux.tcl,v $
 # $Log: buttonAux.tcl,v $
 # Revision 1.13  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.12  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.11  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.10  1993/10/20  18:36:58  kennykb
 # Made changes so that keyboard traversal works in stacking order in tk 3.3,
 # and not in reverse stacking order.  Tk 3.2 still works in order of creation.
 #
 # Repaired copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.9  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.8  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.7  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.6  1993/10/14  13:19:17  kennykb
 # Added the `-default' option for better Motif-style control of the `default'
 # button.
 #
 # Cleaned up geometry management of `focusable' buttons, just a little bit.
 #
 # Revision 1.5  1993/07/20  19:17:12  kennykb
 # Improved structured comments.
 # Changed modules through `g' in the alphabet to follow `:' and `_' naming
 # conventions.
 #
 # Revision 1.4  1993/07/20  13:12:03  kennykb
 # Made `choicebox', `collapsible', and `debug' conform with naming and
 # commentary conventions
 #
 # Revision 1.3  1993/07/19  18:49:24  kennykb
 # Renamed all button_ commands to be either button. or button:, in
 # conformance with new module naming conventions.
 #
 # Cleaned up structured comments in buttonAux.tcl
 #
 # Revision 1.2  1993/07/16  15:58:00  kennykb
 # Renamed all commands that start with `wiget.' to either `widget_' or
 # `widget:'.  Added the code for creating composite widgets.
 #
 # Revision 1.1  1993/06/03  15:24:29  kennykb
 # Initial revision

# Procedure:	button_bindForTraversal
#
# Synopsis:
#	Set up a widget or class for the Button feel.
#
# Usage:
#c	button_bindForTraversal arg arg...
#
# Parameters:
#	Any number of window or class names that are to be bound with the
#	Button feel and traversal via <Tab> and <Shift-Tab>
#
# Return value:
#	None.
#
# Description:
#	button_bindForTraversal causes a set of widgets or classes to
#	acquire bindings for keyboard traversal as buttons.
#
#	The bindings are as follows.
#
#	- The <space>, <Return>, and <Enter> keys are bound so that
#	pressing any of them when the button has the keyboard focus
#	has the same effect as pressing the button.
#
#	- The <Tab> and <Shift-Tab> keys are bound to transfer keyboard
#	focus to the next and previous application, respectively.
#
#	- The <F10> and <Alt> keys are bound as for tk_bindForTraversal
#
#	- <Any-Enter>, <Any-Leave>, <1> and <ButtonRelease-1> are
#	rebound to versions that are compatible with keyboard
#	traversal.
#
#	- <FocusIn> and <FocusOut> events are bound to procedures that
#	highlight the focused button.

proc button_bindForTraversal args {
	foreach w $args {
		focus_bindForTraversal $w
		tk_bindForTraversal $w
		bind $w <Any-Enter> "button:enter %W"
		bind $w <Any-Leave> "button:leave %W"
		bind $w <1> "button_press %W"
		bind $w <ButtonRelease-1> "button_release %W"
		bind $w <Key-space> "button_invoke %W -default"
		catch { bind $w <Control-Key-Linefeed> {
				button_invoke %W -default
			}}
		catch { bind $w <Control-Key-Return> {
				button_invoke %W -default
			}}
		catch { bind $w <Control-Key-KP_Enter> {
				button_invoke %W -default
			}}
		catch { bind $w <Control-Key-Enter> {
				button_invoke %W -default
			}}
		catch { bind $w <Key-Linefeed> {
				button_invoke %W -default
			}}
		catch { bind $w <Key-Return> {
				button_invoke %W -default
			}}
		catch { bind $w <Key-KP_Enter> {
				button_invoke %W -default
			}}
		catch { bind $w <Key-Enter> {
				button_invoke %W -default
			}}
		widget_bind $w GainFocus button:gainFocus
		widget_bind $w LoseFocus button:loseFocus
	}
}

# Procedure:	button:enter
#
# Synopsis:
#	Process mouse entry into a button.
#
# Usage:
#c	button:enter pathName
#
# Parameters:
#c	pathName
#		Path name of a button.
#
# Return value:
#	None.
#
# Description:
#	button:enter handles having the mouse enter a button.  It makes the
#	button's colors `active', providing that the button is not disabled.

proc button:enter w {
	global button_priv
	if {[lindex [$w config -state] 4] != "disabled"} {
		set screen [winfo screen $w]
		set button_priv(current,$screen) $w
		$w config -state active
	}
}

# Procedure:	button:leave
#
# Synopsis:
#	Process mouse exit from a button
#
# Usage:
#c	button:leave pathName
#
# Parameters:
#c	pathName
#		Path name of a button
#
# Return value:
#	None.
#
# Description:
#	button:leave handles having the mouse leave a button.  It makes
#	the button's state `normal' if it wasn't `disabled'.

proc button:leave w {
	global button_priv
	if {[lindex [$w config -state] 4] != "disabled"} {
		$w config -state normal
	}
	set screen [winfo screen $w]
	catch {unset button_priv(current,$screen)}
}

# Procedure:	button:gainFocus
#
# Synopsis:
#	Process keyboard focus into a button.
#
# Usage:
#c	button:gainFocus pathName
#
# Parameters:
#c	pathName
#		Path name of a button
#
# Return value:
#	None.
#
# Description:
#	button:gainFocus is called when a button receives the keyboard focus.
#	If the button is `focusable', it is marked to have the focus by
#	changing its frame's background to equal its own active background.

proc button:gainFocus w {
	set frame [winfo parent $w]
	if {[winfo class $frame] == "Focusable"} {
		set abg [lindex [$w configure -activebackground] 4]
		if {$abg != [lindex [$frame configure -background] 4]} {
			$frame configure -background $abg
		}
	}
}

# Procedure:	button:loseFocus
#
# Synopsis:
#	Process keyboard focus out of a button.
#
# Usage:
#c	button:loseFocus pathName
#
# Parameters:
#c	pathName
#		Path name of a button
#
# Return value:
#	None.
#
# Description:
#	button:loseFocus is called when a button loses the keyboard focus.
#	If the button has the `focusable' look, its frame changes to have
#	the `inactive' colours.

proc button:loseFocus w {
	if {[winfo exists $w] && [info commands $w] == $w} {
		set frame [winfo parent $w]
		if {[winfo class $frame] == "Focusable"} {
			set ibg [lindex [$w configure -background] 4]
			if {$ibg != [lindex \
					[$frame configure -background] \
					4]} {
				$frame configure -background $ibg
			}
		}
	}
}

# Procedure:	button_invoke
#
# Synopsis:
#	Process keyboard event to invoke a button.
#
# Usage:
#c	button_invoke pathName ?-default?
#
# Parameters:
#c	pathName
#		Path name of a button
#
# Options:
#	-default
#		Specifies that the button should become the default for its
#		top-level window.
#
# Return value:
#	None.
#
# Description:
#	button_invoke handles the case where the user invokes a button
#	by means of a key press.  It flashes the button if it's a push button,
#	then invokes the button as if the mouse had been pressed and released
#	over the button.  It also makes the button the default for its
#	toplevel window, if requested

proc button_invoke {w {default {-nodefault}}} {
	global button_priv
	if {$default == [string range "-default" 0 \
				[expr [string length $default]-1]]} {
		button_makeDefault $w
	}
	if {[winfo class $w] == "Button"} {
		set screen [winfo screen $w]
		if {![info exists button_priv(current,$screen)] 
		    || $button_priv(current,$screen) != $w} {
			uplevel #0 [list $w flash]
		}
	}
	uplevel #0 [list $w invoke]
}

# Procedure:	button_press
#
# Synopsis:
#	Process mouse press within a button.
#
# Usage:
#c	button_press pathName
#
# Parameters:
#c	pathName
#		Path name of a button
#
# Return value:
#	None
#
# Description:
#	button_press handles having the user press a button.  It checks
#	it the button is disabled.  If it isn't, it makes the button's relief
#	`sunken' (to simulate pressing it), and transfers the focus to it.

proc button_press w {
	global button_priv
	global button_strictMotif
	if {[lindex [$w config -state] 4] != "disabled"} {
		set button_priv(relief,$w) [lindex [$w config -relief] 4]
		$w config -relief sunken
		if {[info exists button_strictMotif]} {
			focus_goTo $w
		}
	}
}

# Procedure:	button_release
#
# Synopsis:
#	Process mouse release within a button.
#
# Usage:
#c	button_release pathName
#
# Parameters:
#c	pathName
#		Path name of a button.
#
# Return value:
#	None.
#
# Description:
#	button_release handles having the user release the mouse over a
#	button.  It checks that the button is still the `current window'
#	(defined by button_priv(current)), and invokes it if it is.

proc button_release w {
	global button_priv
	if {[info exists button_priv(relief,$w)]} {
		$w config -relief $button_priv(relief,$w)
		unset button_priv(relief,$w)
	}
	set screen [winfo screen $w]
	if {[info exists button_priv(current,$screen)]} {
		if {$w == $button_priv(current,$screen)} {
			button_invoke $w
		}
	}
}

# Procedure:	button_makeDefault
#
# Synopsis:
#	Designate the `default button' for a top-level window.
#
# Usage:
#c	button_makeDefault pathName
#
# Parameters:
#c	pathName
#		Path name of a button.
#
# Return value:
#	None.
#
# Description:
#	button_makeDefault makes a button the `default button' for its
#	top-level window.

proc button_makeDefault w {
	global button_priv
	if {[lindex [$w configure -state] 4] == "disabled"} return
	set t [winfo toplevel $w]
	if [info exists button_priv(default,$t)] {
		set d $button_priv(default,$t)
		if {$d != $w} {
			if {[winfo exists $d] \
			    && [info commands $d] == $d} {
				set frame [winfo parent $d]
				if {[winfo class $frame] == "Focusable"} {
					$frame config -relief flat
				}
			}
		}
	}
	set button_priv(default,$t) $w
	widget_addBinding $t Destroy "catch \"unset button_priv(default,$t)\""
	set frame [winfo parent $w]
	if {[winfo class $frame] == "Focusable"} {
		$frame config -relief sunken
	}
}

# Procedure:	button_invokeDefault
#
# Synopsis:
#	Invoke the default button for a widget's top-level window
#
# Usage:
#c	button_invokeDefault pathName
#
# Parameters:
#c	pathName
#		Path name of any window
#
# Return value:
#	None.
#
# Description:
#	button_invokeDefault invokes the `default button' for any widget
#	in an application.

proc button_invokeDefault w {
	global button_priv
	set t [winfo toplevel $w]
	if {[info exists button_priv(default,$t)]} {
		set b $button_priv(default,$t)
		if {![winfo exists $b] \
		    || [info commands $b] != $b \
		    || [lindex [$b config -state] 4] == "disabled"} {
			unset b
		}
	}
	if {![info exists b]} {
		set b [button:findDefault $t]
	}
	if {$b == ""} {
		error "No command to invoke"
	}
	button_invoke $b
}

# Procedure:	button:findDefault
#
# Synopsis:
#	Establish a default button for a top-level window where none
#	has been specified.
#
# Usage:
#c	button:findDefault pathName
#
# Parameters:
#c	pathName
#		Path name of a window
#
# Return value:
#	Name of the first non-disabled button among the window's children
#
# Description:
#	button:findDefault is used to locate a default button where none
#	has been defined.

proc button:findDefault w {
	global tk_version
	case [winfo class $w] in {
		{Button Checkbutton Radiobutton} {
			if {[lindex [$w config -state] 4] != "disabled"} {
				return $w
			}
		}
	}
	set kids [winfo children $w]
	if {$tk_version >= 3.3} {
		set init {set i 0}
		set cond {$i < [llength $kids]}
		set reinit {incr i}
	} else {
		set init {set i [expr [llength $kids]-1]}
		set cond {$i >= 0}
		set reinit {incr i -1}
	}
	for $init $cond $reinit {
		set kid [lindex $kids $i]
		if {$kid != [winfo toplevel $kid]} {
			set d [button:findDefault $kid]
			if {$d != ""} {
				return $d
			}
		}
	}
	return ""
}

# Procedure: button_setDefault
#
# Synopsis:
#	Find the default button of a top-level window, and return its
#	identity.  Establish a default if none has been specified.
#
# Usage:
#c	button_setDefault pathName
#
# Parameters:
#c	pathName
#		Path name of a window, generally a toplevel.
#
# Return value:
#	Name of the default button of the window
#
# Description:
#	button_setDefault finds the default button of a window, and
#	returns its identity.  If the window has no default defined,
#	a suitable button is located and made the default.

proc button_setDefault w {
	global button_priv
	if [info exists button_priv(default,[winfo toplevel $w])] {
		return $button_priv(default,[winfo toplevel $w])
	}
	set db [button:findDefault $w]
	if {$db != ""} {
		button_makeDefault $db
	}
	return $db
}

# Procedure: focusable
#
# Synopsis:
#	Display the Motif `focus frame' around a button, checkbutton,
#	or radiobutton.  The `focus frame' indicates the current position
#	of the keyboard focus and the current default button for the
#	application.
#
# Usage:
#c	focusable buttonCommand pathName ?args?
#
# Parameters:
#c	buttonCommand
#		one of `button,' `checkbutton,' or `radiobutton.'
#c	pathName
#		Path name of the `focus frame' widget to be created
#c	args
#		Arguments for the button command
#
# Return value:
#	Path name of the `focus frame' widget
#
# Description:
#	Motif-style button actions require a frame that decorates each
#	button to indicate the default action and the position of the
#	keyboard focus.  Preceding the command that creates the button
#	with the `focusable' command greates this indication.

option add *Focusable.borderWidth 2 widgetDefault

proc focusable {type w args} {
	global button_priv
	frame $w -class Focusable
	pack append $w \
		[frame $w.strut1 -geom 0x2] {top} \
		[frame $w.strut2 -geom 0x2] {bottom} \
		[frame $w.strut3 -geom 2x0] {left} \
		[frame $w.strut4 -geom 2x0] {right} \
		[uplevel 1 [list $type $w.b] $args] \
			{expand fill}
	widget_addBinding $w.b Destroy "
		after 1 catch \"widget_checkAndDestroy $w\"
	"
	$w config -background [lindex [$w.b config -background] 4]
	set t [winfo toplevel $w]
	if {[info exists button_priv(default,$t)] \
	    && "$w.b" == $button_priv(default,$t)} {
		button_makeDefault $w.b
	}
	return $w
}

