 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File: choicebox.tcl
#
# Description:
#	Tcl procedures to implement a `choice box' widget, that is,
#	a widget with a possible icon, a possible message, and a set of
#	buttons offering various choices.

 # $Id: choicebox.tcl,v 1.10 1993/11/01 18:20:46 kennykb Exp $
 # $Source: /homedisk/julius/u0/kennykb/src/tkauxlib_ship/RCS/choicebox.tcl,v $
 # $Log: choicebox.tcl,v $
 # Revision 1.10  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.9  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.8  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.7  1993/10/20  18:40:07  kennykb
 # Repaired copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.6  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.5  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.4  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.3  1993/10/14  13:28:46  kennykb
 # Changed message aspect to message width, which gives a better appearance.
 #
 # Revision 1.2  1993/07/20  13:12:03  kennykb
 # Made `choicebox', `collapsible', and `debug' conform with naming and
 # commentary conventions
 #
 # Revision 1.1  1993/06/03  15:25:52  kennykb
 # Initial revision
 #

# Procedure:	choicebox
#
# Synopsis:
#	Widget that constructs a dialog box offering the user a multiple
#	choice.
#
# Usage:
#c	choicebox pathName ?-option value...?
#
# Parameters:
#c	pathName
#		Path name of the choice box to create
#
# Options:
#c	-text TEXT
#		Uses TEXT as the message for the choice box.
#
#c	-icon ICON
#		Uses the bitmap in the file `ICON.xbm' on the current
#		auto_path as the icon for the choice box.
#
#c	-buttons {b1 b2 b3...}
#		Creates buttons with text `b1', `b2', `b3', ...
#		If any of the buttons is a two-element list, the
#		first element is used as the text, and the second as
#		a variable value
#
#c	-textvariable NAME
#		Name of a global variable that receives the name of the
#		button that was pressed.
#
#	Other options are as for `frame.'  The `-class' option is illegal;
#	the class name is always `choicebox'.
#
# Return value:
#	Name of the widget created.
#
# Example:
#c	choicebox $w 	-text "Are you sure?" \

#c			-icon think \

#c			-buttons {{Yes 1} {No 0} {Help ""}} \

#c			-textvariable isSure$w

# Bugs:
#	- The choicebox widget does not use the widget construction
#	primitives to make itself a first-class widget.  It does not
#	respond to the `configure' widget command properly.
#
#	- The options cannot be specified as X defaults.
#
#	- foreground, background, etc. are not supplied as options.

	# Default look:

option add *Choicebox*background gray90 widgetDefault
option add *Choicebox.m.foreground red widgetDefault
option add *Choicebox*activeBackground gray80 widgetDefault

proc choicebox {w args} {
	set fargs {}
	set text {}
	set icon {}
	set buttons {}
	set textvariable $w
	while {[llength $args] >= 2} {
		set option [lindex $args 0]
		set value [lindex $args 1]
		set args [lrange $args 2 end]
		case $option in {
			-text { set text $value }
			-icon { set icon $value }
			-buttons { set buttons $value }
			-textvariable { set textvariable $value }
			default {
				lappend fargs $option $value
			}
		}
	}
	if {$args != ""} {
		error "choicebox: extraneous argument $args"
	}
	eval frame $w -class Choicebox $fargs
	pack append $w \
		[choicebox:buttons $w.b \
			-buttons $buttons \
			-textvariable $textvariable] \
				{bottom expand fill}
	if {$text != ""} {
		pack append $w \
			[message $w.m -text $text -width 400] \
				{right expand fill}
	}
	if {$icon != ""} {
		pack append $w \
			[icon $w.i -icon $icon] \
				{fill}
	}
	return $w
}

# Procedure: choicebox:buttons
#
# Synopsis:
#	Make the push-button panel for a choicebox.
#
# Usage:
#c	choicebox:buttons pathName ?-option value...?
#
# Parameters:
#c	pathName
#		Path name of the button box widget
#
# Options:
#c	-buttons {b1 b2 b3}
#		List of button text, or button text - variable value pairs
#
#c	-textvariable v
#		Variable name in which to store the button value.
#
#	Other options as for frame.
#
# Return value:
#	Name of the button box widget.
#
# Description:
#	choicebox:buttons is an internal procedure used by the choicebox
#	procedure to fabricate the array of push buttons that appear at
#	the bottom of a choicebox.

proc choicebox:buttons {w args} {
	set fargs {}
	set textvariable $w
	set buttons {}
	while {[llength $args] >= 2} {
		set option [lindex $args 0]
		set value [lindex $args 1]
		set args [lrange $args 2 end]
		case $option in {
			-buttons { set buttons $value }
			-textvariable { set textvariable $value }
			default {
				lappend fargs $option $value
			}
		}
	}
	if {$args != ""} {
		error "choicebox:buttons: extraneous argument $args"
	}
	eval frame $w $fargs
	set bnum 0
	foreach b $buttons {
		incr bnum
		if {[llength $b] > 1} {
			set bval [lindex $b 1]
		} else {
			set bval [lindex $b 0]
		}
		pack append $w \
			[focusable button $w.b$bnum \
				-text [lindex $b 0] \
				-command "set $textvariable $bval"] \
					{left expand padx 10 pady 10}
	}
	return $w
}
