 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File: entryAux.tcl
#
# Description:
#	Auxiliary procedures that enhance the bindings of Tk entry
#	widgets.
#
# Global variables:
#c	entry_priv(badText,$w)
#		Text that was in the entry box $w when a validation check
#		failed.  The text is restored when the entry regains the
#		focus (generally, when the error message is dismissed)
#		to allow the user to edit it.
#
#c	entry_priv(replmode,$w)
#		If an entry box is in `replace mode' as opposed to `insert
#		mode', its behavior for `plain' keystrokes changes, and the
#		appearance of its cursor changes.  This variable contains
#		the `configure' options needed to restore the appearance
#		of the cursor to the default `insert mode'.
#
#c	entry_priv(textvariable,$w)
#		Name of the text variable associated with the entry widget $w,
#		saved while $w has the focus (the association is broken while
#		the entry has the focus, to avoid needless trace invocations).
#
# Notes:
#
#	Loading this procedure allows `entry' widgets to recognize,
#	in addition to the standard X resources, the options from the
#	following set.
#
#	Name:		replaceBorderWidth
#	Class:		BorderWidth
#	Default:	2
#	Description:
#		Border width of the insertion cursor of an entry box in
#		replace mode.
#
#	Name:		replaceWidth
#	Class: 		ReplaceWidth
#	Default: 	8
#	Description:
#		Cursor width of an entry box when it is in replace mode.
#
#	Name: 		replaceBackground
#	Class: 		Foreground
#	Default: 	#b2dfee
#	Description:
#		Background color of the insertion cursor of an entry
#		box in replace mode.
#
# Bugs:
#	- There are no command-line flags corresponding to the new `replace'
#	options.
#	- The new `replace' options cannot be configured using the `config'
#	widget command.
#	- The `config' widget command doesn't necessarily work for
#	the `-textvariable' flag.
#
#		These problems stem from a limitation in Tk.  In order
#		to have them work, a wrapper would have to be put around
#		the `entry' procedure.  Unfortunately, Tk's widget-creation
#		commands cannot usually be renamed, since they inspect the
#		name of the command.  (This may not apply to `entry', and
#		the author hasn't examined the  code.  It is known to apply
#		to the label, button, checkbutton, menubutton, and
#		radiobutton widgets.)
#
#	- There is no means of warning the user when an invalid keyboard
#	action is attempted.
#
#		What do users think of flashing the entry box for 
#		a warning (reversing foreground and background a few times)?
#
#	- There are no word-level operations.
#
#	- There is no `transpose characters' operation.
#
#	- There is no access to the cutbuffer.

 # $Id: entryAux.tcl,v 1.15 1993/11/01 18:20:46 kennykb Exp $
 # $Source: /homedisk/julius/u0/kennykb/src/tkauxlib_ship/RCS/entryAux.tcl,v $
 # $Log: entryAux.tcl,v $
 # Revision 1.15  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.14  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.13  1993/10/25  16:09:46  kennykb
 # Fixed an off-by-two (sign error on a +1) bug that caused the
 # selection not to be overtyped if the cursor was within two
 # places of its end.
 #
 # Fixed the `entry:unTrace' procedure to act safely even if it is
 # attempting to untrace the focus window.  This is needed to handle
 # the case where the focus window is destroyed.
 #
 # Revision 1.12  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.11  1993/10/20  18:42:37  kennykb
 # Repaired copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.10  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.9  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.8  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.7  1993/10/14  13:31:49  kennykb
 # Revised controller to work correctly when no `-textvariable' was supplied.
 #
 # Revision 1.6  1993/07/22  22:00:57  kennykb
 # Fixed typo that meant that updating an entry's text variable failed
 # if the entry had lost the focus.
 #
 # Revision 1.5  1993/07/21  19:44:36  kennykb
 # Finished cleaning up structured commentary.
 #
 # Revision 1.4  1993/07/20  19:17:12  kennykb
 # Improved structured comments.
 # Changed modules through `g' in the alphabet to follow `:' and `_' naming
 # conventions.
 #
 # Revision 1.3  1993/07/19  18:49:24  kennykb
 # Renamed all button_ commands to either button. or button:, in
 # conformance with new module naming conventions.
 #
 # Revision 1.2  1993/07/16  15:58:00  kennykb
 # Renamed all commands that start with `wiget.' to either `widget_' or
 # `widget:'.  Added the code for creating composite widgets.
 #
 # Revision 1.1  1993/06/03  15:26:45  kennykb
 # Initial revision


# Procedure:	entry_begin
#
# Synopsis:
#	Handle the <Home> key in an entry widget
#
# Usage:
#c	entry_begin pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	`entry_begin' is called when the user presses the HOME key in an
#	entry widget.  It positions the cursor to the beginning of the widget,
#	and makes the cursor visible.

proc entry_begin {w} {
	$w icursor 0
	$w view 0
}

# Procedure:	entry_bindForTraversal
#
# Synopsis:
#	Establish keyboard bindings for Entry widgets
#
# Usage:
#c	entry_bindForTraversal name name...
#
# Parameters:
#	One or more widget or class names
#
# Return value:
#	None
#
# Description:
#	entry_bindForTraversal establishes all the default key and mouse
#	bindings for entry boxes, including everything needed for keyboard
#	traversals.  It is normally called from init.tcl at startup time,
#	giving the Entry class, but a user is welcome to recode init.tcl.
#
# Default bindings for entry widgets:
#
#	- All of Tk's default bindings remain, except as shown below.
#
#	- The left mouse button calls `entry_button1' instead of the 
#	corresponding action in the Tk library.  The only difference
#	between the two is that `focus_goTo' is used instead of `focus'
#	to direct the focus, so that the focus management works.
#
#		The following keyboard bindings are applied.
#
#	<Any-Key>:
#		 If the cursor is in the selection, delete the selection.
#		If the widget is in insert mode, insert the character at the
#		cursor, otherwise replace the character under the cursor with
#		the key's character.
#	<Home>, <Begin>, <Control-A>:
#		Position the cursor to the left edge of the entry box
#	<Left>, <Control-B>:
#		Back up the cursor one position.
#	<Control-C>:
#		Interrupt the process.
#	<DeleteRight>, <Control-D>:
#		Delete the character to the right of the cursor.
#	<End>, <Control-E>:
#		Position the cursor at the end of the field.
#	<Right>, <Control-F>:
#		Position the cursor one position to the right.
#	<BackSpace>, <Control-H>:
#		Delete the character to the left of the cursor.
#	<Control-K>:
#		Erase from the cursor to the end of the field.
#	<Control-L>:
#		Center the cursor in the field.
#	<Return>, <KP_Enter>, <LineFeed>, <Control-J>, <Control-M>:
#		Invoke the default button associated with the entry's
#		top-level window.
#	<Control-U>:
#		Clear the entry.
#	<Control-W>:
#		Clear the selection.
#	<Control-Y>:
#		Paste the selection into the entry.
#	<Insert>:
#		Enter or leave `replace mode'.  All entries start in
#		`insert mode' originally.
#	<Delete>:
#		Clear the selection if there is one.  Otherwise, clear the
#		character to the left of the cursor.
#	<Escape>, <Break>:
#		Undo the effect of typing in the entry and revert it to the
#		content of its text variable.
#	<Tab>, <Control-Tab>:
#		Position the focus to the next focusable item
#	<Shift-Tab>, <Control-Shift-Tab>:
#		Position the focus to the previous focusable item.

proc entry_bindForTraversal args {
	foreach item $args {
		bind $item <1> {entry_button1 %W %x}
		bind $item <Any-Key> {entry_keyPress %W %A}
		catch { bind $item <Any-Key-Home> {entry_begin %W} }
		catch { bind $item <Any-Key-Begin> {entry_begin %W} }
		bind $item <Control-Key-a> {entry_begin %W}
		catch { bind $item <Key-Left> {entry_cursorleft %W} }
		bind $item <Control-Key-b> {entry_cursorleft %W}
		bind $item <Control-Key-c> {error Interrupt}
		bind $item <Control-Key-d> {entry_deleteRight %W}
		catch { bind $item <Key-End> {entry_end %W} }
		bind $item <Control-Key-e> {entry_end %W}
		catch { bind $item <Key-Right> {entry_cursorright %W} }
		bind $item <Control-Key-f> {entry_cursorright %W}
		bind $item <Control-Key-h> {entry_deleteLeft %W}
		catch { bind $item <Key-BackSpace> {entry_deleteLeft %W} }
		bind $item <Control-Key-k> {
			%W delete insert end
			entry_cursor %W
		}
		bind $item <Control-Key-l> {entry_center %W}
		# Anyone volunteer to write the next one?
		#bind $item <Control-Key-t> {entry.twiddle %W}
		bind $item <Control-Key-u> {entry_clear %W}
		bind $item <Control-Key-w> {entry_wipe %W}
		bind $item <Control-Key-y> {entry_yank %W}
		bind $item <Key-Insert> {entry_insertKey %W}
		bind $item <Key-Delete> {entry_deleteKey %W}
		if {[winfo exists $item] && [info commands $item] == $item} {
			widget_addBinding $item GainFocus \
				{entry:gainFocus $item}
			widget_addBinding $item LoseFocus \
				{entry:loseFocus $item}
			widget_addBinding $item Destroy {entry:destroy $item}
			widget_addBinding $item UpdateContent {
				entry_update $item
			}
		} else {
			widget_bind $item GainFocus {entry:gainFocus}
			widget_bind $item LoseFocus {entry:loseFocus}
			widget_bind $item Destroy {entry:destroy}
			widget_bind $item UpdateContent {entry_update}
		}
		bind $item <Any-Key-Escape> {entry_cancel %W}
		bind $item <Any-Control-Key-bracketleft> {entry_cancel %W}
		catch { bind $item <Any-Key-Break> {entry_cancel %W} }
		focus_bindForTraversal $item
		bind $item <Control-Key-j> {entry_invokeDefaultButton %W}
		bind $item <Control-Key-m> {entry_invokeDefaultButton %W}
		catch { 
			bind $item <Any-Key-Linefeed> {
				entry_invokeDefaultButton %W
			}
		}
		catch {
			bind $item <Any-Key-Return> {
				entry_invokeDefaultButton %W
			}
		}
		catch {
			bind $item <Any-Key-KP_Enter> {
				entry_invokeDefaultButton %W
			}
		}
		catch {
			bind $item <Any-Key-Enter> {
				entry_invokeDefaultButton %W
			}
		}
	}
}

# Procedure:	entry_cancel
#
# Synopsis:
#	Handle the <Escape> key in an entry widget.
#
# Usage:
#c	entry_cancel pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget that has the focus.
#
# Return value:
#	None.
#
# Description:
#	entry_cancel is invoked when the user presses the <Esacape> key in an
#	entry widget.  It determines the value of the widget's text variable
#	(assuming that it has one), and resets the widget's content to that
#	value.

proc entry_cancel {w} {
	global entry_priv
	if {[info exists entry_priv(textvariable,$w)]
	    && $entry_priv(textvariable,$w) != ""} {
		set tvar $entry_priv(textvariable,$w)
		upvar #0 $tvar value
		if {[string compare $value [$w get]] != 0} {
			$w delete 0 end
			$w insert 0 [uplevel #0 set $tvar]
			entry_cursor $w
		}
	} else {
		$w delete 0 end
	}
}

# Procedure:	entry_center
#
# Synopsis:
#	Center the insertion cursor in an entry widget's window.
#
# Usage:
#c	entry_center pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	entry_center is used to place the cursor of an entry box at the
#	center of the visible area.  

proc entry_center {w} {
	set cursor_position [$w index insert]
	$w view 0
	set left_extent [$w index @0]
	set right_extent [$w index @[winfo width $w]]
	set entry_length [expr {$right_extent - $left_extent}]
	set text_length [expr [$w index end]]
	if {$text_length > $entry_length} {
		$w view [expr {$cursor_position - $entry_length/2 + 1}]
	}
}

# Procedure:	entry_clear
#
# Synopsis:
#	Clear an entry widget.
#
# Usage:
#c	entry_clear pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None
#
# Description:
#	`entry_clear' clears the content of an entry.

proc entry_clear {w} {
	$w delete 0 end
	$w view 0
}

# Procedure:	entry_cursor
#
# Synopsis:
#	Make sure that the insertion cursor of an entry widget is visible.
#
# Usage:
#c	entry_cursor pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	entry_cursor is used to correct the display if the cursor has left
#	the visible area.  If the cursor is on screen, nothing is done.
#	Otherwise, the view is changed to place the cursor as near to
#	the center of the screen as possible.

proc entry_cursor {w} {
	global tk_version
	set left_extent [$w index @0]
	set right_extent [$w index @[winfo width $w]]
	set cursor_position [$w index insert]
	set entry_length [expr {$right_extent - $left_extent}]
	if {$cursor_position >= $right_extent \
	 || $cursor_position <= $left_extent} {
		entry_center $w
	}
}

# Procedure:	entry_cursorleft
#
# Synopsis:
#	Move the insertion cursor in an entry widget to the left.
#
# Usage:
#c	entry_cursorleft pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	`entry_cursorleft' is invoked when the user presses the <Left>
#	key in an entry widget.  It positions the cursor one position to the
#	left, and makes the cursor visible.

proc entry_cursorleft {w} {
	set x [expr {[$w index insert] - 1}]
	if {$x >= 0} {$w icursor $x}
	entry_cursor $w
}

# Procedure:	entry_cursorright
#
# Synopsis:
#	Position the cursor in an entry widget one space to the right.
#
# Usage:
#c	entry_cursorright pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	`entry_cursorright' is invoked when the user presses the <Right>
#	key in an entry widget.  If the cursor is not at the end of the
#	entry's content, it is moved one position to the right.  In any
#	case, the cursor is made visible.

proc entry_cursorright {w} {
	set x [expr {[$w index insert] + 1}]
	set xm [$w index end]
	if {$x <= $xm} {$w icursor $x}
	entry_cursor $w
}

# Procedure:	entry_deleteKey
#
# Synopsis:
#	Handle the <Delete> key in an entry widget.
#
# Usage:
#c	entry_deleteKey pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	`entry_deleteKey' is called when the user presses the <Delete> key in
#	an entry.  If the cursor is in a selection, the selection is deleted.
#	Otherwise, one character is deleted to the left of the cursor.
#
# Bugs:
#	If the cursor is at the start of the entry and outside of any
#	selection, nothing is done.  The user should be warned with a visual
#	or audible alarm, but Tk provides at present no mechanism for doing so.

proc entry_deleteKey {w} {
	set ins [$w index insert]
	set haveselection [expr ![catch {
		set sfirst [$w index sel.first]
		set slast [$w index sel.last]
	}]]
	if {$haveselection && $ins >= $sfirst && $ins <= [expr $slast+1]} {
			$w delete $sfirst $slast
			$w icursor $sfirst
	} else {
		if {$ins > 0 && $ins <= [$w index end]} {
			$w delete [expr $ins-1]
		}
	}
	entry_cursor $w
} 

# Procedure:	entry_deleteLeft
#
# Synopsis:
#	Delete one character leftward in an entry widget (back space).
#
# Usage:
#c	entry_deleteLeft pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	`entry_deleteLeft' is called when the user presses the <BackSpace>
#	key in an entry widget.  If the cursor is not at the beginning of the
#	field, it is moved one position to the left, and then the character
#	under the cursor is deleted.  Otherwise nothing happens.
#
# Bugs:
#	If the cursor is at the left edge of the field, nothing happens to
#	warn the user.  The audible or visual alarm should probably be
#	activated, but Tk at present offers no means of doing so.

proc entry_deleteLeft {w} {
	set x [expr {[$w index insert] - 1}]
	if {$x >= 0} {$w delete $x}
	entry_cursor $w
}

# Procedure:	entry_deleteRight
#
# Synopsis:
#	Delete a character to the right in an entry widget.
#
# Usage:
#c	entry_deleteRight pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	`entry_deleteRight' is called when the user presses the <Control-D>
#	or <DeleteRight> key in an entry widget.  If the cursor is not at
#	the end of the field, the character under the cursor is deleted.
#	Otherwise nothing happens.
#
# Bugs:
#	If the cursor is at the end of the field, nothing happens to
#	warn the user.  The audible or visual alarm should probably be
#	activated, but Tk at present offers no means of doing so.

proc entry_deleteRight {w} {
	set x [$w index insert]
	set xm [$w index end]
	if {$x < $xm} {$w delete $x}
	entry_cursor $w
}

# Procedure:	entry:destroy
#
# Synopsis:
#	Internal procedure to clean up when an entry widget is destroyed.
#
# Usage:
#c	entry:destroy pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None
#
# Description:
#	`entry:destroy' is called by `widget:destroy' when an entry widget
#	is destroyed.  It removes a text variable trace, if any, and unsets
#	all variables relating to the widget.

proc entry:destroy {w} {
	global entry_priv
	entry:unTrace $w
	if {[info exists entry_priv]} {
		foreach item [array names entry_priv] {
			if [string match *,$w $item] {
				unset entry_priv($item)
			}
		}
	}
}

# Procedure:	entry_end
#
# Synopsis:
#	Position the insertion cursor at the end of an entry widget.
#
# Usage:
#c	entry_end pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget.
#
# Return value:
#	None.
#
# Description:
#	`entry_end' is invoked when the user presses the <End> key in an entry
#	widget.  It positions the cursor at the end of the widget's content,
#	and makes the cursor visible.

proc entry_end {w} {
	$w icursor end
	entry_cursor $w
}

# Procedure:	entry:gainFocus
#
# Synopsis:
#	Direct the keyboard focus into an entry widget.
#
# Usage:
#c	entry:gainFocus pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget.
#
# Return value:
#	None.
#
# Description:
#	entry:gainFocus is called when an entry receives the keyboard focus
#	(via `focus_goTo', not just the X events, since the focus may be
#	gained and lost for many spurious reasons in the X event cycle).
#
#	If the entry is associated with a text variable, the association is
#	broken; this avoids needless updates to the text variable and
#	possible trace invocations.  Because it is possible for the text
#	variable's value to change while the entry is focused, a trace
#	is established on the text variable, which updates the text of the
#	entry when the text variable changes.
#
#	If there was a `bad value' saved when the entry lost the focus, it is
#	restored, to allow the user to edit it.
#
#	Finally, the cursor is made visible within the entry; there are
#	various pathological cases where it might conceivably disappear.

proc entry:gainFocus {w} {
	global entry_priv
	set v [lindex [$w config -textvariable] 4]
	if {![info exists entry_priv(textvariable,$w)]} {
		set entry_priv(textvariable,$w) $v
	}
	if {$v != ""} {
		$w config -textvariable {}
		proc entry:updateText$w {v1 v2 op} [format {
		    trace_action $v1 $v2 $op {
			set w %s
			if {[string compare $value [$w get]] != 0} {
				$w delete 0 end
				$w insert 0 $value
				after 1 entry_cursor $w
			}
		    }
		} $w]
		upvar #0 $v var
		if {![info exists var]} {
			set var ""
		}
		uplevel #0 trace variable $v w entry:updateText$w
	}
	if [info exists entry_priv(badText,$w)] {
		if {[string compare $entry_priv(badText,$w) [$w get]] != 0} {
			$w delete 0 end
			$w insert 0 $entry_priv(badText,$w)
			unset entry_priv(badText,$w)
		}
	}
	entry_cursor $w
}

# Procedure:	entry_insertKey
#
# Synopsis:
#	Toggle `replace mode' in an entry widget.
#
# Usage:
#c	entry_insertKey pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	`entry_insertKey' is called when the user presses the <Insert> key in
#	an entry.  If the entry is in insert mode, it goes into replace mode,
#	and vice versa.  The cursor's appearance is changed appropriately.

proc entry_insertKey {w} {
	global entry_priv
	if [info exists entry_priv(replmode,$w)] {
		eval $w config $entry_priv(replmode,$w)
		unset entry_priv(replmode,$w)
	} else {
		set entry_priv(replmode,$w) -insertwidth
		lappend entry_priv(replmode,$w) \
			[lindex [$w config -insertwidth] 4]
		lappend entry_priv(replmode,$w) -insertborderwidth
		lappend entry_priv(replmode,$w) \
			[lindex [$w config -insertborderwidth] 4]
		lappend entry_priv(replmode,$w) -insertbackground
		lappend entry_priv(replmode,$w) \
			[lindex [$w config -insertbackground] 4]
		set cwidth \
			[option get $w replaceWidth ReplaceWidth]
		if {$cwidth == ""} {
			set cwidth 8
		}
		$w config -insertwidth $cwidth
		set cbwidth [option get $w replaceBorderWidth BorderWidth]
		if {$cbwidth == ""} {
			set cbwidth 2
		}
		$w config -insertborderwidth $cbwidth
		set cbbg [option get $w replaceBackground Foreground]
		if {$cbbg == ""} {
			set cbbg #b2dfee
		}
		$w config -insertbackground $cbbg
	}
}

# Procedure:	entry_invokeDefaultButton
#
# Synopsis:
#	Invoke the default button associated with an entry widget's
#	top-level window.
#
# Usage:
#c	entry_invokeDefaultButton pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget that has the keyboard focus
#
# Return value:
#	None
#
# Description:
#	`entry_invokeDefaultButton' is called when the user presses <Return>
#	or <KP_Enter> in an entry widget.  It updates the entry's text
#	variable, and if that succeeds, invokes the default button associated
#	with the entry's top-level window.

proc entry_invokeDefaultButton w {
	if {[entry_update $w] == 0} {
		button_invokeDefault $w
	}
}

# Procedure:	entry:loseFocus
#
# Synopsis:
#	Direct keyboard focus away from an entry widget
#
# Usage:
#c	entry:loseFocus pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget.
#
# Return value:
#	None.
#
# Description:
#	entry:loseFocus is called when an entry widget loses the keyboard
#	focus (via focus_goTo, not the X events, since the focus may be gained
#	and lost for many spurious reasons as far as the X server is
#	concerned).
#
#	entry:loseFocus updates the content of the entry box, and restores
#	the association between the entry box and its text variable.  It
#	removes any trace that may have been applied when the entry first
#	received the focus.

proc entry:loseFocus {w} {
	global entry_priv
	if {[winfo exists $w] && [info commands $w] == $w} {
		entry_update $w
		$w config -textvariable $entry_priv(textvariable,$w)
	}
	entry:unTrace $w
}

# Procedure:	entry_keyPress
#
# Synopsis:
#	Handle a keypress in an entry widget.
#
# Usage:
#c	entry_keyPress pathName character
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#c	character
#		Any single character
#
# Return value:
#	None.
#
# Description:
#	`entry_keyPress' is invoked when the user presses a key in an entry
#	widget, and that key does not have any other binding.
#
#	If the box owns the X selection, and the cursor is within the
#	selection, the user is overwriting the selection with the key.  The
#	selected region is deleted, and the character is inserted in its place.
#
#	Otherwise, if the entry box is in replace mode and the cursor is not
#	at the end of the box, the character under the cursor is replaced
#	with the supplied character.
#
#	Otherwise, the character is inserted at the cursor.
#
#	In any case, the cursor is made visible.

proc entry_keyPress {w c} {
	global entry_priv
	if {$c == ""} return
	set ins [$w index insert]
	set haveselection [expr ![catch {
		set sfirst [$w index sel.first]
		set slast [$w index sel.last]
	}]]
	if {$haveselection && $ins >= $sfirst && $ins <= [expr $slast+1]} {
			$w delete $sfirst $slast
			$w insert $sfirst $c
	} else {
		if {[info exists entry_priv(replmode,$w)] \
		    && $ins < [$w index end]} {
			$w delete $ins
		}
		$w insert $ins $c
	}
	entry_cursor $w
}

# Procedure:	entry_button1
#
# Synopsis:
#	Handle mouse button 1 in a widget
#
# Usage:
#c	entry_button1 pathName x
#
# Parameters:
#c	pathName
#		Path name of the entry widget
#c	x
#		X co-ordinate of the mouse cursor within the widget.
#
# Return value:
#	None.
#
# Description:
#	entry_button1 handles a left-mouse-click in a widget.  It sets the
#	insertion cursor to the nearest character to the mouse, sets the
#	selection to a zero-length string originating at the cursor, and
#	directs focus to the widget if the widget is not disabled.

proc entry_button1 {w x} {
	$w icursor @$x
	$w select from @$x
	if {[lindex [$w config -state] 4] != "disabled"} {
		focus_goTo $w
	}
}

# Procedure:	entry:unTrace
#
# Synopsis:
#	Internal procedure to remove trace from an entry's text variable.
#
# Usage:
#c	entry:unTrace pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	entry:unTrace removes the trace on an entry's text variable, if one
#	is applied.  It then unsets the indication that the text variable was
#	on the entry (the assumption is that someone else already reconfigured
#	the window's -textvariable, or destroyed the window, or something else
#	equally violent, so the window's configuration isn't touched).

proc entry:unTrace {w} {
	global entry_priv
	if {[info exists entry_priv(textvariable,$w)]} {
		if {$entry_priv(textvariable,$w) != ""} {
			uplevel #0 trace vdelete \
				$entry_priv(textvariable,$w) \
				w entry:updateText$w
		}
		catch {rename entry:updateText$w {}}
		catch {unset entry_priv(textvariable,$w)}
	}
}	

# Procedure:	entry_update
#
# Synopsis:
#	Reconcile an entry's text variable with the contents of its window.
#
# Usage:
#c	entry_update pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	0 if the validation check on the entry widget succeeded, and a
#	non-zero value otherwise.
#
# Description:
#	`entry_update' is called whenever it is necessary to reconcile the
#	text variable of an entry with the entry contents.  These include
#
#	- losing the focus
#	- invoking a command via <Return> or <LineFeed> while focus is
#	  in the entry.
#	- executing a command (via button or menu) that uses the value
#	  of the text variable.
#
#	First, the validation procedure for the entry is called via
#	`widget:check.'  If the validation fails, entry_update returns a
#	non-zero status, the entry content reverts to the previous value of the
#	text variable, and a nonzero status is returned.  Otherwise, the
#	text variable is updated and zero is returned.
#
#	In the event of validation failure, the bad text is saved, and is
#	restored to the entry box the next time it gains the focus; this
#	process allows the user to correct it without messing with the text
#	variable.
#
# Notes:
#	The user should seldom need to call this procedure.  Instead, the
#	`focus_update' procedure should be called routinely at the beginning
#	of any command that depends on having entries' text variables
#	validated and up to date.
#
#	Note that button and menu bindings will not call `focus_update'
#	on the command's behalf.  This behavior is intentional.  Certain
#	button and menu commands may want to execute without checking
#	entries for validity.  For instance, a `load data' command may want
#	to overwrite the contents of entries wholesale, without regard
#	to their current content.  An `undo' command may want to undo
#	typing in an entry box.  These behaviors would be impossible if
#	entry boxes were validated on all button and menu actions.
#
# See also:
#	focus_update

proc entry_update {w} {
	global entry_priv
	set status [catch {widget:check $w}]
	if {$status == 0} {
		if {[info exists entry_priv(textvariable,$w)]
		    && $entry_priv(textvariable,$w) != ""} {
			set tvar $entry_priv(textvariable,$w)
			if {$tvar != ""} {
				uplevel #0 set? [list $tvar] [list [$w get]]
			}
		}
	} else {
		set entry_priv(badText,$w) [$w get]
		entry_cancel $w
	}
	return $status
}

# Procedure:	entry_variable
#
# Synopsis:
#	Find the name of an entry's text variable
#
# Usage:
#c	entry_variable pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	Name of the entry's text variable
#
# Description:
#	`entry_variable' is used to find the name of an entry's text variable.
#	`$w config -textvariable' cannot be used when the entry has the
#	focus, because the text variable association is temporarily broken at
#	that time.
#
# Bugs:
#	The entry auxiliary procedures should work with the `config' widget
#	command.

proc entry_variable w {
	global entry_priv
	if [info exists entry_priv(textvariable,$w)] {
		return $entry_priv(textvariable,$w)
	} else {
		return [lindex [$w config -textvariable] 4]
	}
}

# Procedure:	entry_wipe
#
# Synopsis:
#	Remove selected text from an entry widget.
#
# Usage:
#c	entry_wipe pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	`entry_wipe' deletes the current selection from an entry widget.
#
# Bugs:
#	There really should be some way of putting the selected text into the
#	cutbuffer.  Unfortunately, Tk appears to provide no access to the
#	cutbuffer.


proc entry_wipe {w} {
	catch {
		$w delete sel.first sel.last
	}
	entry_cursor $w
}

# Procedure:	entry_yank
#
# Synopsis:
#	Yank the current X selection into a widget.
#
# Usage:
#c	entry_yank pathName
#
# Parameters:
#c	pathName
#		Path name of an entry widget
#
# Return value:
#	None.
#
# Description:
#	`entry_yank' yanks the current X selection into a widget at the
#	current insertion point.  It is used to handle, among other things,
#	the functionality of the `control-Y' operation.
#
# Bugs:
#	It should also be possible to yank the cutbuffer.  Tk, unfortunately,
#	provides no way to access the cutbuffer.

proc entry_yank {w} {
	catch {
		$w insert insert [selection get]
	}
	entry_cursor $w
}
