/*
 * Dir.cc - implementation of tcl "dir" command
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1993 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  Allan
 * Brighton make no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 * -----------------------------------------------------------------------------
 */


extern  "C" {
#include <tcl.h>
}
#include <stdlib.h>
#include <string.h>
#include <OS/string.h>
#include <OS/directory.h>

/*
 * Local class to implement the dir subcommands
 * 
 * - uses the interviews Directory class
 */
class Dir {
private:
    Tcl_Interp* interp_;
    Directory* dir_;
    int count_; 	// count of files in dir
    int status_;
public:
    Dir(Tcl_Interp*, const char* name);
    ~Dir();
    int status() const { return status_; }
    
    // sub commands
    void ls();
    void lsFiles();
    void lsDirs();
    void hasSubdirs();
};



/*
 * Implementation of the tcl extended dir command -
 * Efficient operations on directories
 * 
 * usage: dir subcmd $dir
 * 
 * sub commands:
 * 
 * dir ls $dir             list contents of $dir
 * dir lsfiles $dir        list files in $dir
 * dir lsdirs $dir         list dirs in dirs
 * dir hassubdirs $dir     return true if $dir has subdirs
 */
extern "C"
int dirCmd(ClientData, Tcl_Interp* interp, int argc, char *argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"dir subCmd <dir>\"", 0);
	return TCL_ERROR;
    }
    
    char* cmd = argv[1];
    int length = strlen(cmd);
    
    Dir dir(interp, argv[2]);
    if (dir.status() != TCL_OK) {
	Tcl_AppendResult(interp, argv[2], " is not a directory", 0);
	return TCL_ERROR;
    }
    
    switch (*cmd) {
    case 'l':
	if (strncmp(cmd, "ls", length) == 0 && length == 2)
	    dir.ls();
	else if (strncmp(cmd, "lsfiles", length) == 0 && length > 2)
	    dir.lsFiles();
	else if (strncmp(cmd, "lsdirs", length) == 0 && length > 2)
	    dir.lsDirs();
	break;
    case 'h':
	if (strncmp(cmd, "hassubdirs", length) == 0)
	    dir.hasSubdirs();
	break;
    default:
        Tcl_AppendResult(interp, "unknown dir subcommand: \"", cmd, "\"", 0);
        return TCL_ERROR;
    }
    return TCL_OK;
}


/*
 * Constructor -
 * 
 * opens the directory and sets status to TCL_OK or TCL_ERROR indicate
 * success or failure
 */
Dir::Dir(Tcl_Interp* interp, const char* name)
: interp_(interp),
  dir_(Directory::open(name)),
  count_(0),
  status_(TCL_OK)
{
    if (!dir_) {
	status_ =  TCL_ERROR;
	return;
    }
    count_ = dir_->count();
}


/*
 * Destructor - close the dir
 */
Dir::~Dir()
{
    if (dir_)
        dir_->close();
}


/*
 * Return a list in tcl with the contents of this directory
 */
void Dir::ls()
{
    for (int i = 0; i < count_; i++) {
        const String& name = *dir_->name(i);
        if (name != "." &&  name != "..")
            Tcl_AppendResult(interp_, name.string(), " ", 0);
    }
}


/*
 * Return a list in tcl with the files in this directory
 */
void Dir::lsFiles()
{
    for (int i = 0; i < count_; i++) {
        const String& name = *dir_->name(i);
        if (name != "." &&  name != ".." && !dir_->is_directory(i))
            Tcl_AppendResult(interp_, name.string(), " ", 0);
    }
}


/*
 * Return a list in tcl with the dirs in this directory
 */
void Dir::lsDirs()
{
    for (int i = 0; i < count_; i++) {
        const String& name = *dir_->name(i);
        if (name != "." &&  name != ".." && dir_->is_directory(i))
            Tcl_AppendResult(interp_, name.string(), " ", 0);
    }
}


/*
 * Return true if this dir has subdirs
 */
void Dir::hasSubdirs()
{
    for (int i = 0; i < count_; i++) {
	const String& name = *dir_->name(i);
	if (name != "." &&  name != ".." && dir_->is_directory(i)) {
	    Tcl_AppendResult(interp_, "1", 0);
	    return;
	}
    }

    Tcl_AppendResult(interp_, "0", 0);
}


