/*
 *=============================================================================
 *                              tSippShadows.c
 *-----------------------------------------------------------------------------
 * Shadow manipulation commands.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippShadows.c,v 5.0 1994/09/05 01:23:43 markd Rel $
 *============================================================================
 */

#include "tSippInt.h"

/*=============================================================================
 * SippShadows --
 *   Implements the command:
 *     SippShadows flag [size]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShadows (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    int           flag;
    unsigned      size;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " flag [size]", (char *) NULL);
        return TCL_ERROR;
    }                     
    if (Tcl_GetBoolean (interp, argv [1], &flag) != TCL_OK)
        return TCL_ERROR;
    if (flag) {
        if (argc != 3) {
            Tcl_AppendResult (interp, "size argument must be specified when ",
                              "enabling shadows", (char *) NULL);
            return TCL_ERROR;
        }
        if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [2], &size))
            return TCL_ERROR;
    } else {
        if (argc != 2) {
            Tcl_AppendResult (interp, "size argument must be omitted when ",
                              "disabling shadows", (char *) NULL);
            return TCL_ERROR;
        }
        size = 0;
    }
    sipp_shadows (flag, size);
    return TCL_OK;
}

/*=============================================================================
 * SippShadowMapsCreate --
 *   Implements the command:
 *     SippShadowMapsCreate size
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShadowMapsCreate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    unsigned      size;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " size", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [1], &size))
        return TCL_ERROR;

    shadowmaps_create (size);
    return TCL_OK;
}

/*=============================================================================
 * SippShadowMapsDestruct --
 *   Implements the command:
 *     SippShadowMapsDestruct
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShadowMapsDestruct (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    unsigned      size;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], (char *) NULL);
        return TCL_ERROR;
    }

    shadowmaps_destruct ();
    return TCL_OK;
}

/*=============================================================================
 * TSippShadowsInit --
 *   Initialized the shadows-releated commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the top level global data structure.
 *-----------------------------------------------------------------------------
 */
void
TSippShadowsInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippShadows",             (Tcl_CmdProc *) SippShadows},
        {"SippShadowMapsCreate",    (Tcl_CmdProc *) SippShadowMapsCreate},
        {"SippShadowMapsDestruct",  (Tcl_CmdProc *) SippShadowMapsDestruct},
        {NULL,                      NULL}
    };

    TSippInitCmds (tSippGlobPtr, cmdTable);
}


