#
# Sterno classes.
#
# $Id: class.tcl,v 1.10 1997/12/27 06:25:56 markd Exp $
#

#
# Class objects:
#   o Standard methods:
#     o method - Define a new method.
#     o field - Define a new field that is automatically imported into
#       methods.
#     o new - Create a new instance of this class.
#     o localNew - Create a new instance of this class that is in the
#       caller's namespace.
#  o Standard fields:
#     o self - Class command.
#  o Internal fields:
#    o ___nextObjectNum - Id number of next object in class.
#    o ___instFields - Hash table of instance field names and initial value.
#    o ___instMethods - Hash table of instance method names and proc definition.
#    o ___localObjs - Hash table of local objects.
#

#
# Initialize class stuff in ::Sterno
#
namespace eval ::Sterno {
    namespace export defClass defLocalClass
    variable nextClassNum 0
}

##
# Define a new class.
#
proc ::Sterno::defClass {name {body {}}} {
    variable nextClassNum

    set classNS ::Sterno::Classes::class$nextClassNum
    incr nextClassNum

    if [catch {
        # Get alias info if named
        set alias {}
        if {[llength $name] != 0} {
            set alias [list [expr [info level]-1] $name]
        }
        set classCmd [::Sterno::_doDefClass $classNS $alias $body]
    } errorResult] {
        global errorInfo errorCode
        incr nextClassNum -1
        error $errorResult $errorInfo $errorCode
    }
    return $classCmd
}

##
# Define a new class local to the current namespace of object.
#
proc ::Sterno::defLocalClass {name {body {}}} {
    variable nextClassNum

    set classNS [uplevel 1 namespace current]::Sterno::class$nextClassNum
    incr nextClassNum

    if [catch {
        # Get alias info if named
        set alias {}
        if {[llength $name] != 0} {
            set alias [list [expr [info level]-1] $name]
        }
        set classCmd [::Sterno::_doDefClass $classNS $alias $body]
    } errorResult] {
        global errorInfo errorCode
        incr nextClassNum -1
        error $errorResult $errorInfo $errorCode
    }
    return $classCmd
}

#
# Do the work of defining a new class.
#
proc ::Sterno::_doDefClass {classNS alias body} {
    # Set up the class object and eval the body
    if [catch {
        set classCmd [_defineObjectNS $classNS $alias]
        _initStdClassMembers $classNS
        namespace eval $classNS $body
        _addObjDeleteCmd $classNS [list ::Sterno::_doDelClass $classNS]
    } errorResult] {
        global errorInfo errorCode
        set err [list error $errorResult $errorInfo $errorCode]
        catch {namespace delete $classNS}
        eval $err
    }
    return $classCmd
}

#
# Initialize the standard methods in the object.
#
proc ::Sterno::_initStdClassMembers {classNS} {
    namespace eval $classNS {
        variable ___nextObjectNum 0
        variable ___instFields
        variable ___instMethods
        variable ___localObjs

        # Define an instance method
        proc method {name args body} {
            variable ___instMethods
            set ___instMethods($name) [list proc $name $args $body]

        }

        # Define an instance field
        proc field {field args} {
            variable ___instFields
            set ___instFields($field) $args
        }

        # Create a new object
        proc new args {
            variable selfns
            return [::Sterno::_doNew $selfns {} $args]
        }

        # Create a new object in the callers namespaceobject
        proc localNew args {
            variable selfns
            return [::Sterno::_doNew $selfns \
                    [uplevel 2 namespace current]::Sterno::[namespace tail $selfns] $args]
        }
    }
}

#
# Proc called when class object is deleted.
# Cleans up any local objects.
#
proc ::Sterno::_doDelClass classNS {
    foreach obj [set ${classNS}::___localObjs] {
        _deleteObjIfExists $obj
    }
}

