/*
 *=============================================================================
 *                               tSippError.c
 *-----------------------------------------------------------------------------
 * Error trap handling for tsipp.
 *-----------------------------------------------------------------------------
 * Copyright 1994-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippError.c,v 1.2 1995/01/11 04:08:02 markd Exp $
 *============================================================================
 */

#include "tSippInt.h"
#include <math.h>
#include <errno.h>
#include <signal.h>

extern int errno;

/*
 * Pointer to tsipp globals use by the matherr handler.  Since matherrs are
 * global, we can do this (wish they had client-data).
 */ 
static tSippGlob_t  *mathErrGlobPtr = NULL;

/*
 * The following variable is secretly shared with Tcl so we can
 * tell if expression evaluation is in progress.
 */

extern int tcl_MathInProgress;

/*
 * The following definitions allow matherr to compile on systems
 * that don't really support it.  The compiled procedure is bogus,
 * but it will never be executed on these systems anyway.
 */

#ifndef NEED_MATHERR
struct exception {
    int type;
};
#define DOMAIN 0
#define SING 0
#endif

/*
 * Internal prototypes.
 */
static int
EnableFPESignal _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

static int
IgnoreFPESignal _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

static int
MathErrAsyncHandler _ANSI_ARGS_((ClientData  clientData,
                                 Tcl_Interp *interp,
                                 int         cmdResultCode));

static RETSIGTYPE
SignalHandler _ANSI_ARGS_((int signalNum));

/*=============================================================================
 * EnableFPESignal --
 *   Enable catching SIGFPE.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *
 * Returns:
 *   TCL_OK or TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
static int
EnableFPESignal (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
#ifdef HAVE_SIGACTION
    struct sigaction newState;
    
    newState.sa_handler = SignalHandler;
    sigfillset (&newState.sa_mask);
    newState.sa_flags = 0;

    if (sigaction (SIGFPE, &newState, NULL) < 0)
        goto errorExit;
#else
    if (signal (SIGFPE, SignalHandler) == SIG_ERR)
        goto errorExit;
#endif
    return TCL_OK;

  errorExit:
    Tcl_AppendResult (tSippGlobPtr->interp, "enabling SIGFPE: ",
                      Tcl_PosixError (tSippGlobPtr->interp));
    return TCL_ERROR;
}

/*=============================================================================
 * IgnoreFPESignal --
 *   Set SIGFPE to be ignored.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *
 * Returns:
 *   TCL_OK or TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
static int
IgnoreFPESignal (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
#ifdef HAVE_SIGACTION
    struct sigaction newState;
    
    newState.sa_handler = SIG_IGN;
    sigfillset (&newState.sa_mask);
    newState.sa_flags = 0;

    if (sigaction (SIGFPE, &newState, NULL) < 0)
      goto errorExit;
#else
    if (signal (SIGFPE, SIG_IGN) == SIG_ERR)
      goto errorExit;
#endif
    return TCL_OK;

  errorExit:
    Tcl_AppendResult (tSippGlobPtr->interp, "disabling SIGFPE: ",
                      Tcl_PosixError (tSippGlobPtr->interp));
    return TCL_ERROR;
}

/*=============================================================================
 * MathErrAsyncHandler --
 *   Tcl asynchronous handler that is called to return math errors when the
 * Tcl interpreter is in a safe state.
 *
 * Parameters:
 *   o clientData (I) - Actually a pointer to the tsipp globals.  All error
 *     information is stored in them.
 *   o interp (I) - Interpreter to return the error in.  If this is NULL, the
 *     no interpreter is directly available (i.e. Tk event loop).  In this
 *     case, the tSippGlob interpreter is used.
 *   o cmdResultCode (I) - Not used.
 * Returns:
 *   TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
static int
MathErrAsyncHandler (clientData, interp, cmdResultCode)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         cmdResultCode;
{
    tSippGlob_t  *tSippGlobPtr = (tSippGlob_t *) clientData;
    char         *signalName;

    if (interp == NULL)
        interp = tSippGlobPtr->interp;

    Tcl_ResetResult (interp);

    /*
     * If we got a signal, as indicated by EINTR, return a signal error
     * in the same manner as the TclX error handler.  Otherwise, return
     * the floating pointer error.
     */

    if (tSippGlobPtr->mathSignal != 0) {
        signalName = Tcl_SignalId (tSippGlobPtr->mathSignal);
        Tcl_SetErrorCode (interp, "POSIX", "SIG", signalName, (char*) NULL);
        Tcl_AppendResult (interp, signalName, " signal received", 
                          (char *)NULL);
        Tcl_SetVar (interp, "errorInfo", "", TCL_GLOBAL_ONLY);
        EnableFPESignal (tSippGlobPtr);
    } else {
        errno = tSippGlobPtr->mathErrorNum;
        Tcl_ResetResult (interp);
        Tcl_AppendResult (interp, "floating point math error: ",
                          Tcl_PosixError (interp), (char *) NULL);
    }
    
    /*
     * Clear the error flags.
     */
    tSippGlobPtr->mathErrorNum = 0;
    tSippGlobPtr->mathSignal = 0;

    return TCL_ERROR;
}

/*=============================================================================
 * matherr --
 *   Math error trap handler.  If the error occurs while in Tcl expression
 * evaluation, handle in the way Tcl expects it to be handled.  Otherwise
 * schedule an asynchronous event to return the error to the Tcl interpreter
 * when its safe.  Also abort rendering if its in progress.
 *
 * Parameters:
 *   o expectInfo (I) - Infomation describing the exception.
 * Returns:
 *   Always returns 1 to continue execution until the error can be handled.
 *-----------------------------------------------------------------------------
 */
int
matherr (exceptInfo)
    struct exception *exceptInfo;
{
    int errorNum;

    if ((exceptInfo->type == DOMAIN) || (exceptInfo->type == SING)) {
        errno = EDOM;
    } else {
        errno = ERANGE;
    }

    /*
     * Handle Tcl expressions errors.
     */
    if (tcl_MathInProgress) {
        errno = errorNum;
        return 1;
    }

    /*
     * If another error is not already reported, schedule an asynchronous
     * event.
     */
    if ((mathErrGlobPtr->mathErrorNum == 0) &&
        (mathErrGlobPtr->mathSignal == 0)) {

        mathErrGlobPtr->mathErrorNum = errorNum;
        Tcl_AsyncMark (mathErrGlobPtr->mathErrorHandler);
        sipp_render_terminate ();
    }

    return 1;
}

/*=============================================================================
 * SignalHandler --
 *   Signal handler for math errors.  Save the signal number and schedule
 * the asynchronous error handler, but only if a error is not already pending.
 * Abort rendering if in progress.
 *
 * Parameters:
 *   o signalNum (I) - Signal number.
 * Globals:
 *   o mathErrGlobPtr (I) - Pointer to the tsipp globals.
 *-----------------------------------------------------------------------------
 */
static RETSIGTYPE
SignalHandler (signalNum)
    int signalNum;
{
    if ((mathErrGlobPtr->mathErrorNum == 0) &&
        (mathErrGlobPtr->mathSignal == 0)) {

        mathErrGlobPtr->mathSignal = signalNum;
        Tcl_AsyncMark (mathErrGlobPtr->mathErrorHandler);
        sipp_render_terminate ();
        IgnoreFPESignal (mathErrGlobPtr);
    }
}

/*=============================================================================
 * TSippErrorHandlingInit -- 
 *   Initialize the Tcl/Sipp error handling.  This arms a signal handler that
 * traps errors and aborts the rendering if it is in progress.  It also 
 * saves a global pointer to the interpreter to be called by our matherr
 * routine.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *
 * Returns:
 *   TCL_OK or TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
int
TSippErrorHandlingInit (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
    mathErrGlobPtr = tSippGlobPtr;
    
    tSippGlobPtr->mathErrorNum = 0;
    tSippGlobPtr->mathSignal = 0;

    tSippGlobPtr->mathErrorHandler = Tcl_AsyncCreate (MathErrAsyncHandler,
                                                      tSippGlobPtr);
    if (EnableFPESignal (tSippGlobPtr) == TCL_ERROR)
        return TCL_ERROR;
    return TCL_OK;
}

