/*
 *=============================================================================
 *                                  tSippObj.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manage SIPP objects.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippObj.c,v 5.5 1995/09/19 06:52:18 markd Exp $
 *=============================================================================
 */

#include "tSippInt.h"

/* 
 * Entry in the object handle table. 
 */
typedef struct {
    Object *ptr;
    int     handleNum;
} objectEntry_t;

/*
 * Internal function prototypes.
 */
typedef Object *
dupFunc_t _ANSI_ARGS_((Object  *objectPtr));

typedef void
rotateFunc_t _ANSI_ARGS_((Object  *objectPtr,
                          double   angle));

void
ObjectHandleFree _ANSI_ARGS_((tSippGlob_t    *tSippGlobPtr,
                              objectEntry_t **handleEntryPtr));

static objectEntry_t *
ObjectHandleToEntry _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                 char         *handle));

static bool
ObjectListConvert _ANSI_ARGS_((tSippGlob_t     *tSippGlobPtr,
                               char            *listPtr,
                               handleList_t    *handleListPtr,
                               char          ***handleListArgvPtr));

static objectEntry_t *
ObjectPtrToObjectEntry _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                                    Object      *objectPtr));

static void
ObjectEntryToHandle _ANSI_ARGS_((tSippGlob_t   *tSippGlobPtr,
                                 objectEntry_t *objectEntryPtr));

static void
ObjectHandleCleanup _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                                 bool         deleteWorld));

static void
UnrefAllSubObjects _ANSI_ARGS_((Object *objectPtr));

static Object *
ObjectHandleCmdSetup _ANSI_ARGS_((tSippGlob_t   *tSippGlobPtr,
                                  int            argc,
                                  char         **argv));

static int
ObjectDup _ANSI_ARGS_((tSippGlob_t   *tSippGlobPtr,
                       int            argc,
                       char         **argv,
                       dupFunc_t      dupFunc));

static bool
CheckSubObjAdd _ANSI_ARGS_((Object  *objectPtr,
                            Object  *subObjectPtr));

static int
ObjectAxisRotate _ANSI_ARGS_((tSippGlob_t   *tSippGlobPtr,
                              int            argc,
                              char         **argv,
                              rotateFunc_t  *rotateFunc));

/*
 * Global constants.
 */
static char *worldHandle = "WORLD";
static char *intWorldHandle = "object0";

static char *handlePrefix = "object";
#define HANDLE_PREFIX_LEN 6

/*=============================================================================
 * TSippBindObjectToHandle --
 *   Assigns a handle to the specified object.  Make an entry in the object
 * to object entry hash table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals. The handle is
 *     returned in interp->result.
 *   o objectPtr (I) - A pointer to the object.
 *-----------------------------------------------------------------------------
 */
void
TSippBindObjectToHandle (tSippGlobPtr, objectPtr)
    tSippGlob_t    *tSippGlobPtr;
    Object         *objectPtr;
{
    objectEntry_t  *objectEntryPtr, **handleEntryPtr;
    int             newFlag;
    Tcl_HashEntry  *xrefEntryPtr;
    
    /*
     * Create an object handle table entry for the object.
     */
    objectEntryPtr = (objectEntry_t *) smalloc (sizeof (objectEntry_t));
    objectEntryPtr->ptr = objectPtr;

    /*
     * Add the object to the handle table and save the handle number.
     */
    handleEntryPtr = (objectEntry_t **)
        Tcl_HandleAlloc (tSippGlobPtr->objectTblPtr,
                         tSippGlobPtr->interp->result);
    *handleEntryPtr = objectEntryPtr;
    objectEntryPtr->handleNum = 
        strtod (tSippGlobPtr->interp->result + HANDLE_PREFIX_LEN,
                NULL);

    /*
     * Add object to the xref table.
     */
    xrefEntryPtr = Tcl_CreateHashEntry (&tSippGlobPtr->objectXRefTbl,
                                        (char *) objectPtr,
                                        &newFlag);
    if (!newFlag)
        panic ("TSIPP: duplicate object hash entry\n");
    Tcl_SetHashValue (xrefEntryPtr,
                      objectEntryPtr);
}

/*=============================================================================
 * ObjectHandleFree --
 *   Disassociate a handle from an object.  Unreference the object.  If the
 * count of objects associated with it goes to zero, the object is freed.
 * The object is removed from the xref table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals.
 *   o handleEntryPtr (I) - A pointer to the object entry in the handle table.
 *-----------------------------------------------------------------------------
 */
void
ObjectHandleFree (tSippGlobPtr, handleEntryPtr)
    tSippGlob_t    *tSippGlobPtr;
    objectEntry_t **handleEntryPtr;
{
    Tcl_HashEntry  *xrefEntryPtr;

    object_unref ((*handleEntryPtr)->ptr);

    xrefEntryPtr = Tcl_FindHashEntry (&tSippGlobPtr->objectXRefTbl,
                                      (char *) (*handleEntryPtr)->ptr);
    if (xrefEntryPtr == NULL)
        panic ("TSIPP: missing object hash entry\n");
    Tcl_DeleteHashEntry (xrefEntryPtr);
    
    sfree (*handleEntryPtr);

    Tcl_HandleFree (tSippGlobPtr->objectTblPtr,
                    handleEntryPtr);
}

/*=============================================================================
 * ObjectHandleToEntry --
 *   Utility procedure to convert an object handle to an object entry.
 * Checks for magic handle "WORLD".
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A object handle.
 * Returns:
 *   A pointer to the object entru, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static objectEntry_t *
ObjectHandleToEntry (tSippGlobPtr, handle)
    tSippGlob_t  *tSippGlobPtr;
    char         *handle;
{
    objectEntry_t **handleEntryPtr;

    if (STREQU (handle, worldHandle))
        handle = intWorldHandle;

    handleEntryPtr = (objectEntry_t **)
        Tcl_HandleXlate (tSippGlobPtr->interp, 
                         tSippGlobPtr->objectTblPtr,
                         handle);
    if (handleEntryPtr == NULL)
        return NULL;
    return *handleEntryPtr;
}

/*=============================================================================
 * TSippObjectHandleToPtr --
 *   Utility procedure to convert an object handle to an object pointer.
 * For use of by functions outside of this module. Checks for magic handle
 * "WORLD".
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A object handle.
 * Returns:
 *   A pointer to the object, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
Object *
TSippObjectHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_t    *tSippGlobPtr;
    char           *handle;
{
    objectEntry_t *objectEntryPtr;

    objectEntryPtr = ObjectHandleToEntry (tSippGlobPtr,
                                          handle);
    if (objectEntryPtr == NULL)
        return NULL;
    return objectEntryPtr->ptr;
}

/*=============================================================================
 * ObjectListConvert --
 *   Convert a list of object handles into an array of generic pointers
 * to SIPP objects.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o listPtr (I) - The list to convert.
 *   o handleListPtr (O) - A handle list structure, which be contain the
 *     array of generic pointers. HandleListFree must be called to clean up
 *     this structure. The pointers are really of type Object.
 *   o handleArgvPtr (O) - A pointer to the argv returned from spliting the
 *     handle list is returned here.  Must be freed after use.  If NULL, then
 *     the list argv is not returned.
 * Returns:
 *   TRUE if the conversion succeeds, FALSE and an error in
 * tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
ObjectListConvert (tSippGlobPtr, listPtr, handleListPtr, handleListArgvPtr)
    tSippGlob_t     *tSippGlobPtr;
    char            *listPtr;
    handleList_t    *handleListPtr;
    char          ***handleListArgvPtr;
{
    int  idx;

    if (!TSippHandleListConvert (tSippGlobPtr,
                                 tSippGlobPtr->objectTblPtr,
                                 listPtr,
                                 handleListPtr,
                                 NULL,
                                 handleListArgvPtr))
        return FALSE;

    /*
     * Convert the handle table entries to objects.
     */
    for (idx = 0; idx < handleListPtr->len; idx++) {
        handleListPtr->ptr [idx] = 
            ((objectEntry_t *) (handleListPtr->ptr [idx]))->ptr;
    }

    return TRUE;
}

/*=============================================================================
 * ObjectPtrToObjectEntry --
 *   Utility procedure to convert an object pointer to an object entry pointer.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o objectPtr (I) - The object to lookup.
 * Returns:
 *   A pointer to the object entry, or NULL if the object is not bound to a
 * handle.
 *-----------------------------------------------------------------------------
 */
static objectEntry_t *
ObjectPtrToObjectEntry (tSippGlobPtr, objectPtr)
    tSippGlob_t *tSippGlobPtr;
    Object      *objectPtr;
{
    Tcl_HashEntry *xrefEntryPtr;
    objectEntry_t *objectEntryPtr;

    xrefEntryPtr = Tcl_FindHashEntry (&tSippGlobPtr->objectXRefTbl,
                                      (char *) objectPtr);
    if (xrefEntryPtr == NULL)
        return NULL;
    objectEntryPtr = (objectEntry_t *) Tcl_GetHashValue (xrefEntryPtr);

    return objectEntryPtr;
}

/*=============================================================================
 * ObjectEntryToHandle --
 *   Utility procedure to convert an object entry pointer to its handle.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.  The
 *     handle is return in interp->result.
 *   o objectEntryPtr (I) - The object to lookup.
 *-----------------------------------------------------------------------------
 */
static void
ObjectEntryToHandle (tSippGlobPtr, objectEntryPtr)
    tSippGlob_t   *tSippGlobPtr;
    objectEntry_t *objectEntryPtr;
{
    if (objectEntryPtr->ptr == sipp_world) {
        tSippGlobPtr->interp->result = worldHandle;
    } else {
        sprintf (tSippGlobPtr->interp->result, "%s%d",
                 handlePrefix, objectEntryPtr->handleNum);
    }
}

/*=============================================================================
 * TSippObjectToFirstHandle --
 *   Utility procedure to convert an object to the either its handle, if it
 * has one or the handle of the first ancestor object to have a handle.
 * WORLD is not returned.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.  The
 *     handle is return in interp->result.
 *   o objectPtr (I) - The object to lookup.  Can be NULL.
 * Returns:
 *   TRUE if an object with a handle is found, FALSE if not.
 *-----------------------------------------------------------------------------
 */
bool
TSippObjectToFirstHandle (tSippGlobPtr, objectPtr)
    tSippGlob_t *tSippGlobPtr;
    Object      *objectPtr;
{
    objectEntry_t *objectEntryPtr;

    while ((objectPtr != NULL) && (objectPtr != sipp_world)) {
        objectEntryPtr = ObjectPtrToObjectEntry (tSippGlobPtr,
                                                 objectPtr);
        if (objectEntryPtr != NULL) {
            ObjectEntryToHandle (tSippGlobPtr,
                                 objectEntryPtr);
            return TRUE;
        }
        objectPtr = objectPtr->parent1;
    }
    return FALSE;
}

/*=============================================================================
 * ObjectHandleCleanup --
 *    Unreference all object handles that are defined.  This deletes all
 * objects except those that are sub-objects of the WORLD object.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o deleteWorld (I) - If TRUE, delete handle entry associated with world.
 *     Does not actually delete world object.
 *-----------------------------------------------------------------------------
 */
static void
ObjectHandleCleanup (tSippGlobPtr, deleteWorld)
    tSippGlob_t *tSippGlobPtr;
    bool         deleteWorld;
{
    int             walkKey = -1;
    objectEntry_t **handleEntryPtr;

    while (TRUE) {
        handleEntryPtr =
            Tcl_HandleWalk (tSippGlobPtr->objectTblPtr,
                            &walkKey);
        if (handleEntryPtr == NULL)
            break;
        if ((!deleteWorld) && ((*handleEntryPtr)->ptr == sipp_world))
            continue;

        ObjectHandleFree (tSippGlobPtr,
                          handleEntryPtr);
    }
}

/*=============================================================================
 * UnrefAllSubObjects --
 *    Recursively subtract and unreference all sub-objects of the specified
 * object.
 *
 * Parameters:
 *   o objectPtr (I) - A pointer to the object whose sub-objects are to be
 *     unreferenced.  This object is not unreferenced.
 *-----------------------------------------------------------------------------
 */
static void
UnrefAllSubObjects (objectPtr)
    Object *objectPtr;
{
    Object *subObjectPtr;

    while (objectPtr->num_sub_objs > 0) {
        subObjectPtr = objectPtr->sub_objs [0];
        object_sub_subobj (objectPtr, subObjectPtr);
    }
}

/*=============================================================================
 * ObjectHandleCmdSetup --
 *    Utility procedure for the commands that take a single argument of an
 * object handle.  Validates argv and retrieves the handle table entry.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o argc, argv (I) - Command argument vector.
 * Returns:
 *   A pointer to the object associated with the handle or NULL and an error
 *   in tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static Object *
ObjectHandleCmdSetup (tSippGlobPtr, argc, argv)
    tSippGlob_t   *tSippGlobPtr;
    int            argc;
    char         **argv;
{
    if (tSippGlobPtr->rendering) {
        TSippNotWhileRendering (tSippGlobPtr->interp);
        return NULL;
    }

    if (argc != 2) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " objecthandle", (char *) NULL);
        return NULL;
    }
    return TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
}

/*=============================================================================
 * SippObjectCreate --
 *   Implements the command:
 *     SippObjectCreate
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectCreate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (((tSippGlob_t *) clientData)->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv[0], (char *) NULL);
        return TCL_ERROR;
    }                     

    TSippBindObjectToHandle ((tSippGlob_t *) clientData, object_create ());
    return TCL_OK;
}

/*=============================================================================
 * SippObjectUnref --
 *   Implements the command:
 *     SippObjectUnref objectlist|ALL
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectUnref (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t    *tSippGlobPtr = (tSippGlob_t *) clientData;
    int             idx;
    handleList_t    objectEntryList;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objectlist|ALL", (char *) NULL);
        return TCL_ERROR;
    }
    
    if (STREQU (argv [1], "ALL")) {
        ObjectHandleCleanup (tSippGlobPtr,
                             FALSE);
        return TCL_OK;
    }

    if (!TSippHandleListConvert (tSippGlobPtr, 
                                 tSippGlobPtr->objectTblPtr,
                                 argv [1],
                                 NULL,
                                 &objectEntryList,
                                 NULL))
        return TCL_ERROR;

    for (idx = 0; idx < objectEntryList.len; idx++) {
        ObjectHandleFree (tSippGlobPtr,
                          (objectEntry_t **) objectEntryList.ptr [idx]);
    }

    TSippHandleListFree (&objectEntryList);
    return TCL_OK;
}

/*=============================================================================
 * ObjectDup --
 *   Parse the arguments for one of the dup commands and perform the dup
 * operation using the specified function.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals. 
 *   o argc, argv (I) - Command argument vector.
 *   o dupFunc (O) - SIPP duplication function to call.
 * Returns:
 *   TCL_OK or TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
static int
ObjectDup (tSippGlobPtr, argc, argv, dupFunc)
    tSippGlob_t    *tSippGlobPtr;
    int            argc;
    char         **argv;
    dupFunc_t      dupFunc;
{
    Object    *srcObjPtr, *newObjPtr;
    bool       keepTrans;
    int        idx;

    if (tSippGlobPtr->rendering) {
        TSippNotWhileRendering (tSippGlobPtr->interp);
        return NULL;
    }

    keepTrans = FALSE;
    for (idx = 1; (idx < argc) && (argv [idx][0] == '-'); idx++) {
        if (STREQU (argv [idx], "-cleartransf")) {
            keepTrans = FALSE;
            continue;
        }
        if (STREQU (argv [idx], "-keeptransf")) {
            keepTrans = TRUE;
            continue;
        }
        Tcl_AppendResult (tSippGlobPtr->interp, "expected option of ",
                          "\"-cleartransf\" or \"-keeptransf\", got \"",
                          argv [idx], "\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (idx != argc - 1) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " [flags] objecthandle", (char *) NULL);
        return TCL_ERROR;
    }
    srcObjPtr = TSippObjectHandleToPtr (tSippGlobPtr, argv [idx]);
    if (srcObjPtr == NULL)
        return TCL_ERROR;    

    newObjPtr = dupFunc (srcObjPtr);
    if (keepTrans)
        newObjPtr->transf = srcObjPtr->transf;

    TSippBindObjectToHandle (tSippGlobPtr,
                             newObjPtr);
    return TCL_OK;
}

/*=============================================================================
 * SippObjectInstance --
 *   Implements the command:
 *     SippObjectInstance objecthandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectInstance (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return ObjectDup ((tSippGlob_t *) clientData,
                      argc, argv,
                      object_instance);
}

/*=============================================================================
 * SippObjectDup --
 *   Implements the command:
 *     SippObjectDup objecthandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectDup (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return ObjectDup ((tSippGlob_t *) clientData,
                      argc, argv,
                      object_dup);
}

/*=============================================================================
 * SippObjectDeepDup --
 *   Implements the command:
 *     SippObjectDeepDup objecthandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectDeepDup (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return ObjectDup ((tSippGlob_t *) clientData,
                      argc, argv,
                      object_deep_dup);
}

/*=============================================================================
 * SippObjectGetTransf --
 *   Implements the command:
 *     SippObjectGetTransf objecthandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectGetTransf (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t    *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object         *objectPtr;
    Transf_mat      matrix;

    objectPtr = ObjectHandleCmdSetup ((tSippGlob_t *) clientData,
                                      argc, argv);
    if (objectPtr == NULL)
        return TCL_ERROR;    

    object_get_transf (objectPtr, &matrix);
    Tcl_SetResult (interp,
                   TSippFormatMatrix (tSippGlobPtr,
                                      &matrix),
                   TCL_DYNAMIC);

    return TCL_OK;
}

/*=============================================================================
 * SippObjectSetTransf --
 *   Implements the command:
 *     SippObjectSetTransf objectHandle matrix
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectSetTransf (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t    *tSippGlobPtr = (tSippGlob_t *) clientData;
    Transf_mat      matrix;
    Object         *objectPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " objectHandle matrix", (char *) NULL);
        return TCL_ERROR;
    }
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_t *) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippConvertMatrix (tSippGlobPtr, argv [2], &matrix))
        return TCL_ERROR;

    object_set_transf (objectPtr, &matrix);

    return TCL_OK;
}

/*=============================================================================
 * SippObjectClearTransf --
 *   Implements the command:
 *     SippObjectClearTransf objectHandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectClearTransf (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t    *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object        *objectPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objectHandle", (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_t *) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    object_clear_transf (objectPtr);

    return TCL_OK;
}

/*=============================================================================
 * SippObjectTransform --
 *   Implements the command:
 *     SippObjectTransform objectHandle matrix
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectTransform (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t    *tSippGlobPtr = (tSippGlob_t *) clientData;
    Transf_mat      matrix;
    Object        *objectPtr;
    
    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " objectHandle matrix", (char *) NULL);
        return TCL_ERROR;
    }
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_t *) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippConvertMatrix (tSippGlobPtr, argv [2], &matrix))
        return TCL_ERROR;

    object_transform (objectPtr, &matrix);

    return TCL_OK;
}

/*=============================================================================
 * SippObjectAddSurface --
 *   Implements the command:
 *     SippObjectAddSurface objectHandle surfacelist
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectAddSurface (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object         *objectPtr;
    int             idx, surfIdx;
    handleList_t    surfaceList;
    char          **handleArgv;
    
    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objectHandle surfacelist", (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_t *) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippSurfaceListConvert (tSippGlobPtr,
                                  argv [2],
                                  &surfaceList,
                                  &handleArgv))
        return TCL_ERROR;

    /*
     * Check to make sure they are not already surfaces of the object.
     */
    for (idx = 0; idx < surfaceList.len; idx++) {
        for (surfIdx = 0; surfIdx < objectPtr->num_surfaces; surfIdx++) {
            if (surfaceList.ptr [idx] == objectPtr->surfaces [surfIdx]) {
                Tcl_AppendResult (interp, handleArgv [idx],
                                  " is already a surface of ", argv [1],
                                  (char *) NULL);
                goto errorExit;
            }
        }
    }

    /*
     * Add the surfaces.
     */
    for (idx = 0; idx < surfaceList.len; idx++)
        object_add_surface (objectPtr, (Surface *) (surfaceList.ptr [idx]));

    TSippHandleListFree (&surfaceList);
    sfree (handleArgv);
    return TCL_OK;

  errorExit:
    TSippHandleListFree (&surfaceList);
    sfree (handleArgv);
    return TCL_ERROR;
}

/*=============================================================================
 * SippObjectSurfaceCreate --
 *   Implements the command:
 *     SippObjectSurfaceCreate objecthandle shaderhandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectSurfaceCreate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t    *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object         *objectPtr;
    Surface        *surfacePtr;
    Shader         *shaderPtr;
    void           *surfDescPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objecthandle shaderhandle", (char *) NULL);
        return TCL_ERROR;
    }                     

    objectPtr = TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [2],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    surfacePtr = surface_create (surfDescPtr, shaderPtr);
    if (surfacePtr == NULL) {
        Tcl_AppendResult (interp, "the polygon stack is empty",
                         (char *) NULL);
        return TCL_ERROR;
    }

    object_add_surface (objectPtr, surfacePtr);
    surface_unref (surfacePtr);

    return TCL_OK;
}

/*=============================================================================
 * SippObjectSubSurface --
 *   Implements the command:
 *     SippObjectSubSurface objecthandle surfacelist
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectSubSurface (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object        *objectPtr;
    int            idx, surfIdx;
    handleList_t   surfaceList;
    char          **handleArgv;
    
    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objectHandle surfacelist", (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_t *) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippSurfaceListConvert (tSippGlobPtr,
                                  argv [2],
                                  &surfaceList,
                                  &handleArgv))
        return TCL_ERROR;

    /*
     * Check to make sure they are surfaces of the object.
     */
    for (idx = 0; idx < surfaceList.len; idx++) {
        for (surfIdx = 0; surfIdx < objectPtr->num_surfaces; surfIdx++) {
            if (surfaceList.ptr [idx] == objectPtr->surfaces [surfIdx])
                break;
        }
        if (surfIdx == objectPtr->num_surfaces) {
            Tcl_AppendResult (interp, handleArgv [idx],
                              " is not a surface of ", argv [1],
                              (char *) NULL);
            goto errorExit;
        }
    }

    /*
     * Subtract the surfaces.
     */
    for (idx = 0; idx < surfaceList.len; idx++)
       object_sub_surface (objectPtr, (Surface *) (surfaceList.ptr [idx]));

    TSippHandleListFree (&surfaceList);
    sfree (handleArgv);
    return TCL_OK;

  errorExit:
    TSippHandleListFree (&surfaceList);
    sfree (handleArgv);
    return TCL_ERROR;
}

/*=============================================================================
 * CheckSubObjAdd --
 *    Recursively that a object's new parent is not one of it's decendents.
 * This would lead to a circular object list.
 *
 * Parameters:
 *   o objectPtr (I) - Pointer to the object to add the object to.
 *   o subObjectPtr (I) - Pointer to the subobject.
 * Returns:
 *   TRUE if all is ok, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
static bool
CheckSubObjAdd (objectPtr, subObjectPtr)
    Object  *objectPtr;
    Object  *subObjectPtr;
{
    register int     idx, numSubObjs = subObjectPtr->num_sub_objs; 
    register Object  **subObjTblPtr  = subObjectPtr->sub_objs;

    if (subObjectPtr == objectPtr)
        return FALSE;

    for (idx = 0; idx < numSubObjs; idx++) {
        if (!CheckSubObjAdd (objectPtr, subObjTblPtr [idx]))
            return FALSE;
    }
    return TRUE;
}

/*=============================================================================
 * SippObjectAddSubobj --
 *   Implements the command:
 *     SippObjectAddSubobj [-flag] objectHandle subobjlist
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectAddSubobj (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object        *objectPtr, *subObjectPtr;
    int            idx, objIdx;
    handleList_t   subObjList;
    char         **handleArgv;
    int            nextArg = 1;
    bool           check = TRUE;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc < 3 || argc > 4) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " [-flag] objectHandle subobjlist", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (argc == 4) {
        if (STREQU (argv [1], "-check"))
            check = TRUE;
        else if (STREQU (argv [1], "-nocheck"))
            check = FALSE;
        else {
            Tcl_AppendResult (interp, "expected one of \"-check\" or ",
                              "\"-nocheck\", got \"", argv [1], "\"",
                              (char *) NULL);
            return TCL_ERROR;
        }
        nextArg = 2;
    }

    objectPtr = TSippObjectHandleToPtr ((tSippGlob_t *) clientData,
                                        argv [nextArg]);
    if (objectPtr == NULL)
        return TCL_ERROR;

    if (!ObjectListConvert (tSippGlobPtr,
                            argv [nextArg + 1],
                            &subObjList,
                            &handleArgv))
        return TCL_ERROR;

    /*
     * Check to make sure none are already subobjects of the object.  If
     * checking is enabled, recursively check for a circular list (if we are
     * not adding to WORLD.
     */
    if (objectPtr == sipp_world)
        check = FALSE;

    for (idx = 0; idx < subObjList.len; idx++) {
        subObjectPtr = (Object *) subObjList.ptr [idx];

        for (objIdx = 0; objIdx < objectPtr->num_sub_objs; objIdx++) {
            if (subObjectPtr == objectPtr->sub_objs [objIdx]) {
                Tcl_AppendResult (interp, handleArgv [idx],
                                  " is already a subobject of ", argv [1],
                                  (char *) NULL);
                goto errorExit;
            }
        }
        if (check && !CheckSubObjAdd (objectPtr, subObjectPtr)) {
            Tcl_AppendResult (interp, argv [1], " is the same as or a ",
                              "decendent of ", handleArgv [idx],
                              " - would cause a circular object tree",
                              (char *) NULL);
            goto errorExit;
        }
    }

    /*
     * Add the subobjects.
     */
    for (idx = 0; idx < subObjList.len; idx++)
        object_add_subobj (objectPtr,
                           (Object *) (subObjList.ptr [idx]));

    TSippHandleListFree (&subObjList);
    sfree (handleArgv);
    return TCL_OK;

  errorExit:
    TSippHandleListFree (&subObjList);
    sfree (handleArgv);
    return TCL_ERROR;
}

/*=============================================================================
 * SippObjectSubSubobj --
 *   Implements the command:
 *     SippObjectSubSubobj objectHandle subobjlist|ALL
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectSubSubobj (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t    *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object         *objectPtr;
    int             idx, objIdx;
    handleList_t    subObjList;
    char          **handleArgv;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " objectHandle subobjlist|ALL", (char *) NULL);
        return TCL_ERROR;
    }                     

    objectPtr = TSippObjectHandleToPtr ((tSippGlob_t *) clientData, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;

    if (STREQU (argv [2], "ALL")) {
        UnrefAllSubObjects (objectPtr);
        return TCL_OK;
    }

    if (!ObjectListConvert (tSippGlobPtr,
                            argv [2],
                            &subObjList,
                            &handleArgv))
        return TCL_ERROR;

    /*
     * Check to make sure they are subobjects of the object.
     */
    for (idx = 0; idx < subObjList.len; idx++) {
        for (objIdx = 0; objIdx < objectPtr->num_sub_objs; objIdx++) {
            if (subObjList.ptr [idx] == objectPtr->sub_objs [objIdx])
                break;
        }
        if (objIdx == objectPtr->num_sub_objs) {
            Tcl_AppendResult (interp, handleArgv [idx],
                              " is not a subobject of ", argv [1],
                              (char *) NULL);
            goto errorExit;
        }
    }

    /*
     * Subtract the subobjects.
     */
    for (idx = 0; idx < subObjList.len; idx++)
        object_sub_subobj (objectPtr,
                           (Object *) (subObjList.ptr [idx]));

    TSippHandleListFree (&subObjList);
    sfree (handleArgv);
    return TCL_OK;

  errorExit:
    TSippHandleListFree (&subObjList);
    sfree (handleArgv);
    return TCL_ERROR;
}

/*=============================================================================
 * ObjectAxisRotate --
 *   Process parameters for the commands to rotate an object around an axis
 * and call the function to do the rotation.  These commands have the
 * arguments: objecthandle angle
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals.
 *   o argc, argv (I) - The arguments to the command.
 *   o rotateFunc (I) - The rotate function that is to be called.
 * Returns:
 *   TCL_OK or TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
static int
ObjectAxisRotate (tSippGlobPtr, argc, argv, rotateFunc)
    tSippGlob_t   *tSippGlobPtr;
    int            argc;
    char         **argv;
    rotateFunc_t  *rotateFunc;
{
    Object  *objectPtr;
    double  angle;

    if (tSippGlobPtr->rendering) {
        TSippNotWhileRendering (tSippGlobPtr->interp);
        return TCL_ERROR;
    }

    if (argc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " objectHandle angle", (char *) NULL);
        return TCL_ERROR;
    }
    objectPtr = TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;

    if (!TSippConvertAngleRad (tSippGlobPtr, argv [2], &angle))
        return TCL_ERROR;
    
    (*rotateFunc) (objectPtr, angle);
    return TCL_OK;
}

/*=============================================================================
 * SippObjectRotateX --
 *   Implements the command:
 *     SippObjectRotateX objectHandle angle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectRotateX (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return ObjectAxisRotate ((tSippGlob_t *) clientData,
                             argc, argv,
                             object_rot_x);
}

/*=============================================================================
 * SippObjectRotateY --
 *   Implements the command:
 *     SippObjectRotateY objectHandle angle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectRotateY (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return ObjectAxisRotate ((tSippGlob_t *) clientData,
                             argc, argv,
                             object_rot_y);
}

/*=============================================================================
 * SippObjectRotateZ --
 *   Implements the command:
 *     SippObjectRotateZ objectHandle angle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectRotateZ (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return ObjectAxisRotate ((tSippGlob_t *) clientData,
                             argc, argv,
                             object_rot_z);
}

/*=============================================================================
 * SippObjectRotate --
 *   Implements the command:
 *     SippObjectRotate objectHandle point vector angle
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectRotate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object        *objectPtr;
    Vector         point, vector;
    double         angle;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 5) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " objectHandle point vector angle", (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr = TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &point))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [3], &vector))
        return TCL_ERROR;
    if (!TSippConvertAngleRad (tSippGlobPtr, argv [4], &angle))
        return TCL_ERROR;

    object_rot (objectPtr, &point, &vector, angle);

    return TCL_OK;
}

/*=============================================================================
 * SippObjectScale --
 *   Implements the command:
 *     SippObjectScale objectHandle factor|{xfactor yfactor zfactor}
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectScale (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object        *objectPtr;
    Vector         scale;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " objectHandle factor|{xfactor yfactor zfactor}",
                          (char *) NULL);
        return TCL_ERROR;
    }                     

    objectPtr = TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    /*
     * Scale can be a list or a single factor.  If it contains any white space
     * assume its a list.
     */
    if (strpbrk (argv [2], " \f\t\n\r\v") == NULL) {
        if (Tcl_GetDouble (interp, argv [2], &scale.x) != TCL_OK)
            return TCL_ERROR;
        scale.y = scale.z = scale.x;
    } else {
        if (!TSippConvertVertex (tSippGlobPtr, argv [2], &scale))
            return TCL_ERROR;
    } 

    object_scale (objectPtr, scale.x, scale.y, scale.z);

    return TCL_OK;
}

/*=============================================================================
 * SippObjectMove --
 *   Implements the command:
 *     SippObjectMove objectHandle {xdist ydist zdist}
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectMove (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object        *objectPtr;
    Vector         translation;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " objectHandle {xdist ydist zdist}",
                          (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr = TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &translation))
        return TCL_ERROR;

    object_move (objectPtr, translation.x, translation.y, translation.z);

    return TCL_OK;
}

/*=============================================================================
 * SippObjectAncestor --
 *   Implements the command:
 *     SippObjectAncestor objectHandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectAncestor (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Object        *objectPtr;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " objectHandle",
                          (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr = TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;

    TSippObjectToFirstHandle (tSippGlobPtr,
                              objectPtr->parent1);
    return TCL_OK;
}

/*=============================================================================
 * TSippObjectInit --
 *   Initialized the object commands, including creating the object table.
 *
 * Parameters:
 *   o tSippGlobP (I) - Pointer to the top level global data structure.
 *-----------------------------------------------------------------------------
 */
void
TSippObjectInit (tSippGlobPtr)
    tSippGlob_t    *tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippObjectCreate",         (Tcl_CmdProc *) SippObjectCreate},
        {"SippObjectUnref",          (Tcl_CmdProc *) SippObjectUnref},
        {"SippObjectInstance",       (Tcl_CmdProc *) SippObjectInstance},
        {"SippObjectDup",            (Tcl_CmdProc *) SippObjectDup},
        {"SippObjectDeepDup",        (Tcl_CmdProc *) SippObjectDeepDup},
        {"SippObjectGetTransf",      (Tcl_CmdProc *) SippObjectGetTransf},
        {"SippObjectSetTransf",      (Tcl_CmdProc *) SippObjectSetTransf},
        {"SippObjectClearTransf",    (Tcl_CmdProc *) SippObjectClearTransf},
        {"SippObjectTransform",      (Tcl_CmdProc *) SippObjectTransform},
        {"SippObjectUnref",          (Tcl_CmdProc *) SippObjectUnref},
        {"SippObjectAddSurface",     (Tcl_CmdProc *) SippObjectAddSurface},
        {"SippObjectSurfaceCreate",  (Tcl_CmdProc *) SippObjectSurfaceCreate},
        {"SippObjectSubSurface",     (Tcl_CmdProc *) SippObjectSubSurface},
        {"SippObjectAddSubobj",      (Tcl_CmdProc *) SippObjectAddSubobj},
        {"SippObjectSubSubobj",      (Tcl_CmdProc *) SippObjectSubSubobj},
        {"SippObjectRotateX",        (Tcl_CmdProc *) SippObjectRotateX},
        {"SippObjectRotateY",        (Tcl_CmdProc *) SippObjectRotateY},
        {"SippObjectRotateZ",        (Tcl_CmdProc *) SippObjectRotateZ},
        {"SippObjectRotate",         (Tcl_CmdProc *) SippObjectRotate},
        {"SippObjectScale",          (Tcl_CmdProc *) SippObjectScale},
        {"SippObjectMove",           (Tcl_CmdProc *) SippObjectMove},
        {"SippObjectAncestor",       (Tcl_CmdProc *) SippObjectAncestor},
        {NULL,                       NULL}
    };

    /*
     * Set up the table of objects.
     */
    tSippGlobPtr->objectTblPtr = Tcl_HandleTblInit (handlePrefix,
                                                    sizeof (objectEntry_t *),
                                                    64);

    /*
     * Determine the number of ints required to represent a pointer to
     * an object.  This is the key type of the object to handle hash table.
     * The code assumes that a pointer is an even multiple of ints and we
     * verify that here.
     */
    if ((sizeof (Object *) % sizeof (int)) != 0)
        panic ("TSIPP: sizeof a pointer (%d) not a multiple of int (%d)\n",
               sizeof (Object *), sizeof (int));
    
    Tcl_InitHashTable (&tSippGlobPtr->objectXRefTbl,
                       (sizeof (Object *) / sizeof (int)));

    /*
     * Add sipp_world to the handle table.
     */
    TSippBindObjectToHandle (tSippGlobPtr,
                             sipp_world);
    if (!STREQU (tSippGlobPtr->interp->result, intWorldHandle))
        panic ("TSIPP: wrong world handle returned: %s\n", 
               tSippGlobPtr->interp->result);
    Tcl_ResetResult (tSippGlobPtr->interp);

    TSippInitCmds (tSippGlobPtr,
                   cmdTable);
}

/*=============================================================================
 * TSippObjectCleanUp --
 *   Cleanup the object table and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippObjectCleanUp (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
    UnrefAllSubObjects (sipp_world);

    ObjectHandleCleanup (tSippGlobPtr,
                         TRUE);

    Tcl_HandleTblRelease (tSippGlobPtr->objectTblPtr);
    tSippGlobPtr->objectTblPtr = NULL;
    Tcl_DeleteHashTable (&tSippGlobPtr->objectXRefTbl);
}


