# Bindings for incremental and regexp search


# Find_forward and find_reverse can be used by other procedures; they have
# no visible effects.

# Finds first occurrence of string after point in text widget t
# Returns start and end of occurance or "" if unsuccessful.
proc find_forward {t point string} {
	set answer [string first $string [$t get $point end]]
	if {($answer < 0)} {return ""
	} else {return [list "$point +$answer chars" "$point +$answer chars +[string length $string] chars"]
}}

# Finds first occurrence of string before point in text widget t
# Returns start and end of occurance or "" if unsuccessful.
proc find_reverse {t point string} {
	set answer [string last $string [$t get 1.0 $point]]
	if {($answer < 0)} {return ""
	} else {return [list "1.0 +$answer chars" "1.0 +$answer chars +[string length $string] chars"]
}}

# A regular expression searcher
# Finds first occurrence of regular expression after point in text widget t
# Returns start and end of first occurance or "" if unsuccessful.
proc regexp_find_forward {t point exp} {
	if {[catch {set success [regexp -indices $exp [$t get $point end] where]}]} {set success 0}
	if {($success == 0)} {return ""
	} else {return [list "$point +[lindex $where 0] chars" \
				"$point +[lindex $where 1] chars +1 chars"]
}}

# Informs user of success or failure of search and resets search tag.
proc search_any {t f success_msg failure_msg search_fn search_string direction} {
	catch "$t tag remove search 1.0 end"
	set length [string length $search_string]
	if {($length == 0)} {return 0}

	set slist [$search_fn $t insert $search_string]
	if {($slist != "")} {
		$t tag add search [lindex $slist 0] [lindex $slist 1]
		$f.s configure -text $success_msg
		if {($direction == "s")} {	$t mark set insert search.first
		} else {			$t mark set insert search.last}
		$t yview -pickplace insert
		return 1
	} else {$f.s configure -text $failure_msg ; beep
		return 0
}}

# Direction we're searching. "s" == forward, "r" == reverse
set direction "s"

proc re_search {t f success_msg failure_msg search_fn search_string d} {
	global direction
	set direction $d
	set new_index [$t index insert]

	if {($d == "s")} {
		if {[lindex [$f.s configure -text] 4] == $failure_msg} {
			$t mark set insert 1.0
		} else {$t mark set insert {insert +1 chars}
			if {[catch {set new_index [$t index search.last]}]} {
				set new_index [$t index "insert +1 chars"]}}
	} else {
		if {[lindex [$f.s configure -text] 4] == $failure_msg} {
			$t mark set insert end
		} else {$t mark set insert {insert -1 chars}
			if {[catch {set new_index [$t index search.first]}]} {
				set new_index [$t index "insert -1 chars"]}}
	}

	search_any $t $f $success_msg $failure_msg $search_fn \
				$search_string $direction
	if {[lindex [$f.s configure -text] 4] == $failure_msg} {
		$t mark set insert $new_index}
}

proc exit_search {t f c} {
	if {![regexp . $c]} {return}
	$t tag remove search 1.0 end
	destroy_f_entry $t $f.s $f.ss
	foreach binding [bind $t] {	bind $t $binding ""
}}

# Messages
set search_msg "Search: "
set reverse_search_msg "Reverse Search: "
set search_failed_msg "Search failed: "
set reverse_search_failed_msg "Reverse Search failed: "
set regexp_msg "Regexp Search: "
set regexp_failed_msg "Regexp Search Failed: "

# The string currently being searched for (it's bound to the search entry)
set search_string ""

# Called whenever key is hit in entry during incremental search.
proc revise_search {t f c} {
	global search_msg reverse_search_msg direction
	global search_failed_msg reverse_search_failed_msg search_string
	if {![regexp . $c]} {return}
	if {($direction == "s")} {set result [search_any $t $f $search_msg $search_failed_msg find_forward $search_string $direction]
	}  else {$t mark set insert {insert +1 chars}
		set result [search_any $t $f $reverse_search_msg $reverse_search_failed_msg find_reverse $search_string $direction]}
}

proc make_search_incremental {t f} {
	foreach binding [bind Entry] {
		if {([bind $f.ss $binding] == "")} {
			bind $f.ss $binding "[bind Entry $binding]
			revise_search $t $f %A"}}
}

# For any general type of search
proc search_setup {t f success_msg failure_msg search_function} {
	create_f_entry $t $f.s $f.ss
	$f.ss configure -textvariable search_string
	bind $f.ss <Control-g> "beep ; exit_search $t $f x"
	bind $f.ss <Return> "exit_search $t $f x"
	bind $f.ss <Control-s> "re_search $t $f \"$success_msg\" \"$failure_msg\" $search_function \[$f.ss get\] s"
	bind $f.ss <Find> [bind $f.ss <Control-s>]
}

# For regular expression searches
proc regexp_search_setup {t f} {
	global regexp_msg regexp_failed_msg
	search_setup $t $f $regexp_msg $regexp_failed_msg regexp_find_forward
	$f.s configure -text $regexp_msg
}

# For forward and reverse searching
proc bidirectional_search_setup {t f d} {
	global search_msg reverse_search_msg direction
	global search_failed_msg reverse_search_failed_msg search_string

	search_setup $t $f $search_msg $search_failed_msg find_forward

	if {($d == "s")} {$f.s configure -text $search_msg
	} else {	$f.s configure -text $reverse_search_msg}
	set direction $d

	bind $f.ss <Control-r> "re_search $t $f \"$reverse_search_msg\" \"$reverse_search_failed_msg\" find_reverse \[$f.ss get\] r"
	bind $f.ss <Shift-Find> [bind $f.ss <Control-r>]
}


# Search bindings. f is a frame widget to put messages in.
proc searchbind {f} {
	bind Text <Control-g> "+catch \{exit_search %W $f x\}"
	bind Text <Control-r> "set search_string {} ; bidirectional_search_setup %W $f r ; make_search_incremental %W $f"
	bind Text <Control-s> "set search_string {} ; bidirectional_search_setup %W $f s ; make_search_incremental %W $f"
	bind Text <Control-R> "bidirectional_search_setup %W $f r ; make_search_incremental %W $f"
	bind Text <Control-S> "bidirectional_search_setup %W $f s ; make_search_incremental %W $f"
	bind Text <Control-Meta-s> "set search_string {} ; regexp_search_setup %W $f"

# Duplicate bindings
	bind Text <Shift-Find> [bind Text <Control-r>]
	bind Text <Find> [bind Text <Control-s>]
}

searchbind $frame
