# Bindings for textual documents


proc beginning_of_paragraph {t index} {
	set beginning [string last "\n\n" [$t get 1.0 $index]]
	if {($beginning == -1)} {return "1.0"
	} else {return "1.0 +$beginning chars +2 chars"}
}

proc end_of_paragraph {t index} {
	set end [string first "\n\n" [$t get $index end]]
	if {($end == -1)} {return end
	} else {return "insert +$end chars"}
}

proc select_paragraph {t index} {
	catch {$t tag remove sel sel.first sel.last}
	$t tag add sel [beginning_of_paragraph $t $index] \
			[end_of_paragraph $t $index]
}


# Splits line if it is longer than length. Returns number of extra lines
# produced (0 is if line was not broken). Index is on line to break, length
# is desired length, string is contents of line.
proc split_line {t {index insert} {length ""} {string ""}} {
	if {($string == "")} {
		set string [$t get "$index linestart" "$index lineend"]
	}
	if {($length == "")} {set length [lindex [$t configure -width] 4]}
	if {([string length $string] < $length)} {return 0}

	set offset [string last " " [string range $string 0 $length]]
	if {($offset < 0)} {return 0}
	set break [$t index "$index linestart +$offset chars"]
	$t delete $break
	$t insert $break \n
	set breaks [split_line $t "$break +1 chars" $length]
	global modified ; set modified 1
	return [incr breaks]
}

# Adjusts selected region to fit in length columns, so that no lines wrap
# If unspecified, length defaults to window width.
proc adjust_region {t {length ""}} {
	set chars [$t get sel.first sel.last]
	set m1 [gensym] ; set m2 [gensym]
	$t mark set $m1 sel.first ; $t mark set $m2 sel.last
	register_undoable_cmd $t [list undo_filter $t $m1 $m2 $chars] "Adjust $chars" "$m1 $m2"
	do_adjust_region $t $length
}

proc do_adjust_region {t length} {
	set trace sel.first
	while {([set offset [string first \n [$t get $trace sel.last]]] >= 0)} {
		set trace [$t index "$trace +$offset chars"]
		global modified ; set modified 1
		$t delete $trace ;	$t insert $trace " "
	}
	split_line $t sel.first $length
}

# A re-definition of indent_width, to ensure paragraph wrap-around.
proc indent_with {t l e} {
	set prefix [$e get]
	destroy_f_entry $t $l $e

	if {([catch {$t index sel.first}])} {beep ; return}
	if {($prefix == "")} {beep ; return}
	set mark1 [$t index sel.first]
	set mark2 [$t index sel.last]

	set chars [$t get sel.first sel.last]
	set m1 [gensym] ; set m2 [gensym]
	register_undoable_cmd $t [list undo_filter $t $m1 $m2 $chars] "Indent $chars" "$m1 $m2"

	set length [lindex [$t configure -width] 4]
	set length [expr "$length - [string length $prefix]"]
	do_adjust_region $t $length

	global modified
	set modified 1
	set sellast [$t index "sel.last linestart"]
	for {set mark [$t index "sel.first linestart"]} \
			{[$t compare $mark < $sellast]} \
			{set mark [$t index "$mark +1 lines linestart"]} {
		$t insert $mark $prefix
		}
	$t insert $mark $prefix
	$t mark set $m1 "sel.first linestart"; $t mark set $m2 sel.last
	$t tag remove sel 1.0 end
	$t tag add sel $m1 $m2
}

proc do_pararegion {t fn args} {
	if {([catch {$t get sel.first}])} {
		select_paragraph $t insert
		eval $fn $args
		$t tag remove sel 1.0 end
	} else {eval $fn $args
}}


# Textual bindings.
proc textbind {f} {
	bind Text <Meta-a> {move_insert %W [beginning_of_paragraph %W insert]}
	bind Text <Meta-e> {move_insert %W [end_of_paragraph %W insert]}
	bind Text <Control-i> "do_pararegion %W indent_region %W $f"
	bind Text <Control-j> "select_paragraph %W insert"
	bind Text <Meta-j> "do_pararegion %W adjust_region %W"

	bind Text <Return> {+split_line %W {insert -1 chars}}
	bind Text <KP_Enter> "[bind Text <Return>]"
	bind Text <Shift-Return> "[bind Text <Return>] ; [bind Text <Return>]"
}

textbind $frame

$text configure -wrap word

flash_label $frame -text "Loaded text bindings"
