
#include <tkConfig.h>
#include <tkInt.h>

/*
 * The data structure below is used by the "after" command to remember
 * the command to be executed later.
 */

typedef struct a AfterInfo;

struct a {
	Tcl_Interp *interp;		/* Interpreter in which to execute command. */
	Tcl_DString command;		/* the command text */
	Tk_TimerToken ttoken;   /* handle for the scheduled timer events list */
	AfterInfo *next,*prev;  /* links to other commands in the pending list */
	struct timeval time;    /* time when the function will execute */
};

/*
 * A list of the pending 'after' commands. Used to search for commands
 * with the same command or command+parameter set 
 */
static AfterInfo *after_list=NULL;

/* get the absolute time ms milliseconds in the future */
static struct timeval future(ms)
int ms;
{
	struct timeval t;
	(void) gettimeofday(&t,(struct timezone *)NULL);
	t.tv_sec += ms/1000;
	t.tv_usec += (ms%1000)*1000;
	if (t.tv_usec>1000000) {
		t.tv_usec -= 1000000;
		t.tv_sec++;
	}
	return t;
}

/* get the difference in ms between a registered time value and
	the current time */
static long timediff(tv)
struct timeval *tv;
{
	struct timeval tnow;
	long usdiff,sdiff;
	(void) gettimeofday(&tnow,(struct timezone *)NULL);
	usdiff = tv->tv_usec - tnow.tv_usec;
	sdiff = tv->tv_sec - tnow.tv_sec;
	return sdiff*1000+usdiff/1000;
}

/*
 * find a pending 'after' command in the list, using full text search
 */
static AfterInfo
*find_full(cmd,start)
char *cmd;
AfterInfo *start;
{
	AfterInfo *a;
	for (a=start;a!=NULL;a=a->next) {
		if (*Tcl_DStringValue(&a->command) == *cmd &&
		    !strcmp(Tcl_DStringValue(&a->command),cmd)) return a;
	}
	return NULL;
}

/*
 * find a pending 'after' command in the list, comparing only the
 * first word, i.e. the command name
 */
static AfterInfo
*find_cmd(word,start)
char *word;
AfterInfo *start;
{
	AfterInfo *a;
	int l = strlen(word);
	for (a=start;a!=NULL;a=a->next)  {
		if (*Tcl_DStringValue(&a->command) == *word &&
		    !strncmp(Tcl_DStringValue(&a->command),word,l) &&
		    (!Tcl_DStringValue(&a->command)[l] || isspace(UCHAR(Tcl_DStringValue(&a->command)[l])))) return a;
	}
	return NULL;
}

/*
 * execute a command scheduled with the 'after' command
 * afterwards the command ist removed from the pending list
 * and the procedure storage is reclaimed
 * note that we have to remove the after info ptr from the cmd list before
 * we enter the user procedure, otherwise the calling proc will
 * be still registered in the user functions and the pending 'free'
 * function call might wreck havoc
 */
static void
AfterProc(clientData)
ClientData clientData;	/* Describes command to execute. */
{
	AfterInfo *aPtr = (AfterInfo *) clientData;
	int result;
	char *cmdtxt;

	/* this should not happen, but was part of the original Tk3.6 code */
	if (aPtr==NULL) return;
	/* unlink from the pending command list */
	if (aPtr->next != NULL) aPtr->next->prev = aPtr->prev;
	if (aPtr->prev==NULL) after_list = aPtr->next;
	else aPtr->prev->next = aPtr->next;
	cmdtxt = Tcl_DStringValue(&aPtr->command);
	if (*cmdtxt) {
		result = Tcl_GlobalEval(aPtr->interp,cmdtxt);
		if (result != TCL_OK) Tk_BackgroundError(aPtr->interp);
	}
	Tcl_DStringFree(&aPtr->command);
	ckfree((char*)aPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Tk_AfterCmd --
 *
 *	This procedure is invoked to process the "after" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

/* ARGSUSED */
int
Tk_NewAfterCmd(clientData, interp, argc, argv)
ClientData clientData;	/* Main window associated with
				 * interpreter.  Not used.*/
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;		/* Argument strings. */
{
	int ms,i,offset=0,cnt;
	AfterInfo *aPtr,*aNext;
	char *fulltxt=NULL,buf[30];
	int exact=0,reschedule=0,nonew=0,needed;
	enum { AFTER_STD,AFTER_UNIQ,AFTER_OVERRIDE,AFTER_ABORT,AFTER_PENDING } mode=AFTER_STD;

	if (argc < 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " ?unique|override|abort|pending? ?-exact? ?-reschedule? milliseconds ?command? ?arg arg ...?\"",
		    (char *) NULL);
		return TCL_ERROR;
	}

	if (!strcmp(argv[1],"unique")) mode=AFTER_UNIQ,offset++;
	else if (!strcmp(argv[1],"override")) mode=AFTER_OVERRIDE,offset++;
	else if (!strcmp(argv[1],"abort")) mode=AFTER_ABORT,offset++;
	else if (!strcmp(argv[1],"pending")) mode=AFTER_PENDING,offset++;

	while(argc>1+offset && *argv[1+offset]=='-' && !isdigit(argv[1+offset][1])) {
		if (!strcmp(argv[1+offset]+1,"exact")) exact=1,offset++;
		else if (!strcmp(argv[1+offset]+1,"reschedule")) reschedule=1,offset++;
		else if (!strcmp(argv[1+offset]+1,"nonew")) nonew=1,offset++;
		else {
			Tcl_AppendResult(interp,"illegal option \"",argv[1+offset],"\": should be one of -exact, -reschedule",(char *)NULL);
			return TCL_ERROR;
		}
	}

	switch(mode) {
		case AFTER_ABORT:
		case AFTER_PENDING:
			needed = 1+offset;
			break;
		case AFTER_UNIQ:
		case AFTER_OVERRIDE:
			needed = 3+offset;
		default:
			needed = 2+offset;
	}

	if (argc<needed) {
		Tcl_AppendResult(interp,"missing command or timer value after options",(char *)NULL);
		return TCL_ERROR;
	}

	if (mode!=AFTER_ABORT && mode!=AFTER_PENDING) {
		if (Tcl_GetInt(interp, argv[1+offset], &ms) != TCL_OK) {
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp, "bad milliseconds value \"",
			    argv[1+offset], "\"", (char *) NULL);
			return TCL_ERROR;
		}
		if (ms < 0) ms = 0;
		if (argc == 2+offset) {
			Tk_Sleep(ms);
			return TCL_OK;
		}
	}

	switch(mode) {
	case AFTER_PENDING:
		if (exact && argc!=1+offset)
			fulltxt = Tcl_Concat(argc-1-offset,argv+1+offset);
		for(aPtr=after_list;aPtr!=NULL;aPtr=aPtr->next) {
			if (argc!=1+offset) {
				if (exact) {
					aPtr = find_full(fulltxt,aPtr);
				}
				else {
					aPtr = find_cmd(argv[1+offset],aPtr);
				}
			}
			if (aPtr!=NULL) {
				(void) sprintf(buf,"{%ld ",timediff(&aPtr->time));
				Tcl_AppendResult(interp,buf,NULL);
				Tcl_AppendElement(interp,Tcl_DStringValue(&aPtr->command));
				Tcl_AppendResult(interp,"}",NULL);
			}
		};
		if (exact && argc!=1+offset) ckfree(fulltxt);
		return TCL_OK;
	case AFTER_UNIQ:
		if (exact) {
			fulltxt = Tcl_Concat(argc-2-offset,argv+2+offset);
			aPtr = find_full(fulltxt,after_list);
			ckfree(fulltxt);
		} else {
			aPtr = find_cmd(argv[2+offset],after_list);
		}
		if (aPtr!=NULL) { /* already in list */
			if (reschedule) {
				Tk_DeleteTimerHandler(aPtr->ttoken);
				aPtr->ttoken = Tk_CreateTimerHandler(ms,AfterProc,(ClientData)aPtr);
				aPtr->time = future(ms);
			}
			(void) strcpy(interp->result,"0");
			return TCL_OK;
		}
		(void) strcpy(interp->result,"1");
		break;
	case AFTER_ABORT:
		cnt = 0;
		if (exact && argc!=1+offset)
			fulltxt = Tcl_Concat(argc-1-offset,argv+1+offset);
		for(aPtr=after_list;aPtr!=NULL;aPtr=aNext) {
			if (argc!=1+offset) {
				if (exact) {
					aPtr = find_full(fulltxt,aPtr);
				}
				else {
					aPtr = find_cmd(argv[1+offset],aPtr);
				}
				if (aPtr==NULL) break;
			}
			cnt++;
			Tk_DeleteTimerHandler(aPtr->ttoken);
			if (aPtr->prev==NULL) {
				after_list = aPtr->next;
			}
			else {
				aPtr->prev->next = aPtr->next;
			}
			aNext = aPtr->next;
			if (aPtr->next != NULL) aNext->prev = aPtr->prev;
			Tcl_DStringFree(&aPtr->command);
			ckfree((char *)aPtr);
		};
		if (exact && argc!=1+offset) ckfree(fulltxt);
		(void) sprintf(interp->result,"%d",cnt);
		return TCL_OK;
	case AFTER_OVERRIDE:
		if (exact) {
			fulltxt = Tcl_Concat(argc-2-offset,argv+2+offset);
			aPtr = find_full(fulltxt,after_list);
			ckfree(fulltxt);
		} else {
			aPtr = find_cmd(argv[2+offset],after_list);
		}
		if (aPtr!=NULL) {
			if (reschedule) {
				Tk_DeleteTimerHandler(aPtr->ttoken);
				aPtr->ttoken = Tk_CreateTimerHandler(ms,AfterProc,(ClientData)aPtr);
				aPtr->time = future(ms);
			}
			Tcl_DStringTrunc(&aPtr->command,0);
			for (i=2+offset;i<argc;i++) {
				if (i>2+offset) Tcl_DStringAppend(&aPtr->command," ",-1);
				Tcl_DStringAppend(&aPtr->command,argv[i],-1);
			}
			(void) strcpy(interp->result,"1");
			return TCL_OK;
		}
		(void) strcpy(interp->result,"0");
		if (nonew) return TCL_OK;
	}

	aPtr = (AfterInfo *) ckalloc((unsigned) (sizeof(AfterInfo)));
	Tcl_DStringInit(&aPtr->command);
	aPtr->interp = interp;
	if (after_list != NULL) after_list->prev = aPtr;
	aPtr->prev = NULL;
	aPtr->next = after_list;
	after_list = aPtr;
	for (i=2+offset;i<argc;i++) {
		if (i>2+offset) Tcl_DStringAppend(&aPtr->command," ",-1);
		Tcl_DStringAppend(&aPtr->command,argv[i],-1);
	}
	aPtr->ttoken = Tk_CreateTimerHandler(ms, AfterProc, (ClientData) aPtr);
	aPtr->time = future(ms);
	return TCL_OK;
}

/*
 * Initialization routine. Will replace old after cmd with the new one found
 * in this source
 */
int NewAfter_Init(interp)
Tcl_Interp *interp;
{
	Tcl_CreateCommand(interp,"after",Tk_NewAfterCmd,(ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
	return TCL_OK;
}
