/* Bos_ByteStream.c - Object to/from byte streams
 *
 * Copyright (C) 1992,1993 Engineering Design Research Center
 *
 * Author: Sean Levy (snl+@cmu.edu)
 *         n-dim Group
 *         Engineering Design Research Center
 *         Carnegie Mellon University
 *         5000 Forbes Ave / PGH, PA / 51221
 *
 *         Fax: (412) 268-5229
 *         Voice: (412) 268-5226
 */

/*
 * Bos_ByteStream.c,v 1.5 1992/08/03 20:49:59 snl Exp
 *
 * Bos_ByteStream.c,v
 * Revision 1.5  1992/08/03  20:49:59  snl
 * Debugged new storage implementation
 *
 * Revision 1.4  1992/07/31  14:26:05  snl
 * more big fixes
 *
 * Revision 1.3  1992/07/30  20:02:40  snl
 * bug fixes to new storage code
 *
 * Revision 1.2  1992/07/30  17:56:48  snl
 * dumb typo
 *
 * Revision 1.1  1992/07/30  17:53:35  snl
 * Re-implemented GDBM-based storage scheme
 *
 */

static char rcsID[] = "Bos_ByteStream.c,v 1.5 1992/08/03 20:49:59 snl Exp";

#include "bosInt.h"
#ifndef NO_BORDER_MACROS	/* for systems without netinet/in.h */
#include <sys/types.h>
#include <netinet/in.h>
/* I should write an else here that defines [nh]to[nh][sl] as macros
 * depending on the definition of the LITTLE_ENDIAN macro, but I'm
 * lazy right now and it works on mips and suns for the moment... --S
 */
#endif /* NO_BORDER_MACROS */

/*
 * Macros to scribble in and read from buffers
 */

#define PUTSTRP(s,b,i) memcpy(b+i,s,strlen(s)+1);i+=strlen(s)+1
#define PUTINT4(x,b,i) {\
 int _x = (x);\
 _x=htonl(_x);memcpy(b+i,(char *)&_x,sizeof(_x));i+=sizeof(_x);\
 }
#define PUTINT2(x,b,i) {\
 short _x = (x);\
 _x=htons(_x);memcpy(b+i,(char *)&_x,sizeof(_x));i+=sizeof(_x);\
 }
#define GETINT4(i,b,bi) memcpy(&i,&b[bi],sizeof(i));i=ntohl(i);bi+=sizeof(i)
#define GETINT2(i,b,bi) memcpy(&i,&b[bi],sizeof(i));i=ntohs(i);bi+=sizeof(i)
#define GETSTRP(s,b,bi) s=b+bi;bi+=strlen(s)+1

#ifndef NO_VERSIONS
#define BYTE_STREAM_VERSION 1
#endif /* NO_VERSIONS */

/*
 * Exported interface
 */

char *
Bos_Object2ByteStream(obj, buf, idx_p, size_p, errmsg)
     Bos_Object *obj;
     char *buf;
     int *idx_p;
     int *size_p;
     char *errmsg;
{
  int nbytes, idx, size;

  nbytes = strlen(obj->name) + 1 + sizeof(int) + /* = \0 + slot count */
           Bos_ObjectSlotsSize(obj, (int *)0);
  if (buf != (char *)0) {
    if (idx_p != (int *)0)
      idx = *idx_p;
    else
      idx = 0;
    if (size_p != (int *)0)
      size = *size_p;
    else
      size = 0;
  } else {
    buf = (char *)ckalloc(nbytes);
    idx = 0;
    size = nbytes;
  }
  if (size < idx + nbytes) {
    char *enlarge_buffer();

    buf = enlarge_buffer(buf, idx, &size, nbytes);
    if (buf == (char *)0) {
      sprintf(errmsg, "Object2ByteStream: no buffer space (%d < %d)",
	      size, idx + nbytes);
      return (char *)0;
    }
  }
  PUTSTRP(obj->name, buf, idx);
  if (idx_p != (int *)0)
    *idx_p = idx;
  if (size_p != (int *)0)
    *size_p = size;
  return Bos_ObjectSlots2ByteStream(obj, buf, idx_p, size_p, errmsg);
}

Bos_Object *
Bos_ByteStream2Object(world, buf, idx_p, errmsg)
     Bos_World *world;
     char *buf;
     int *idx_p;
     char *errmsg;
{
  int idx;
  char *obj_name;
  Bos_Object *obj;

  if (idx_p != (int *)0)
    idx = *idx_p;
  else
    idx = 0;
  GETSTRP(obj_name, buf, idx);
  obj = Bos_CreateNewObject(world, obj_name);
  if (idx_p != (int *)0)
    *idx_p = idx;
  if (Bos_ByteStream2ObjectSlots(obj, buf, idx_p, errmsg) < 0) {
    Bos_Destroy(world, obj_name);
    obj = (Bos_Object *)0;
  }
  return obj;
}

char *
Bos_ObjectSlots2ByteStream(obj, buf, idx_p, size_p, errmsg)
     Bos_Object *obj;
     char *buf;
     int *idx_p;
     int *size_p;
     char *errmsg;
{
  int nbytes, nslots, idx, size;
  Tcl_HashEntry *e;
  Tcl_HashSearch search;
  char *slot_Value2String();

#ifdef BYTE_STREAM_VERSION
  nbytes = 1;
#else
  nbytes = 0;
#endif
  nbytes += sizeof(int) + Bos_ObjectSlotsSize(obj, &nslots);
  if (buf != (char *)0) {
    if (idx_p != (int *)0)
      idx = *idx_p;
    else
      idx = 0;
    if (size_p != (int *)0)
      size = *size_p;
    else
      size = 0;
  } else {
    buf = (char *)ckalloc(nbytes);
    idx = 0;
    size = nbytes;
  }
  if (size < idx + nbytes) {
    char *enlarge_buffer();

    buf = enlarge_buffer(buf, idx, &size, nbytes);
    if (buf == (char *)0) {
      sprintf(errmsg, "ObjectSlots2ByteStream: no buffer space (%d < %d)",
	      size, idx + nbytes);
      return (char *)0;
    }
  }
#ifdef BYTE_STREAM_VERSION
  buf[idx++] = (char)BYTE_STREAM_VERSION;
#endif
  PUTINT4(nslots, buf, idx);
  for (e = Tcl_FirstHashEntry(obj->slots, &search);
       e != (Tcl_HashEntry *)0;
       e = Tcl_NextHashEntry(&search)) {
    Bos_Slot *s;
    char *vs;
    int f;

    f = 0;
    s = (Bos_Slot *)Tcl_GetHashValue(e);
    PUTSTRP(s->name, buf, idx);
    PUTINT2(s->type, buf, idx);
    PUTINT2(s->pri, buf, idx);
    vs = slot_Value2String(s, &f);
    if (vs == (char *)0) {
      sprintf(errmsg,
	"ObjectSlots2ByteStream: slot %s<%u%s,%u,%x> value conversion error",
	      s->name, Bos_PlainSlotType(s->type),
	      ((s->type)&Bos_SLOT_EVANESCENT_MASK)? "E": "",
	      s->pri, s->value);
      ckfree(buf);
      return (char *)0;
    }
    PUTSTRP(vs, buf, idx);
    if (f)
      ckfree(vs);
  }
  if (idx_p != (int *)0)
    *idx_p = idx;
  if (size_p != (int *)0)
    *size_p = size;
  return buf;
}

int
Bos_ByteStream2ObjectSlots(obj, buf, idx_p, errmsg)
     Bos_Object *obj;
     char *buf;
     int *idx_p;
     char *errmsg;
{
  int idx, ns, nslots;
#ifdef BYTE_STREAM_VERSION
  int version;
#endif /* BYTE_STREAM_VERSION */

  if (idx_p != (int *)0)
    idx = *idx_p;
  else
    idx = 0;
#ifdef BYTE_STREAM_VERSION
  version = (int)buf[idx++];
  if (version != BYTE_STREAM_VERSION) {
    if (idx_p != (int *)0)
      *idx_p = idx;
    if (errmsg != (char *)0)
      sprintf(errmsg, "version mismatch (stream=%d current=%d)",
	      version, BYTE_STREAM_VERSION);
    return (Bos_Object *)0;
  }
#endif /* BYTE_STREAM_VERSION */
  GETINT4(nslots, buf, idx);
  ns = nslots;
  while (ns-- > 0) {
    char *slot_name, *value;
    Bos_Slot_Type type, plain_type;
    Bos_Slot_Pri pri;
    int E, add_stat, op_mask;
    void *val, *slot_String2Value();

    GETSTRP(slot_name, buf, idx);
    GETINT2(type, buf, idx);
    GETINT2(pri, buf, idx);
    GETSTRP(value, buf, idx);
    plain_type = Bos_PlainSlotType(type);
    E = (type & Bos_SLOT_EVANESCENT_MASK);
    val = slot_String2Value(value, type, pri);
    if (val == (void *)0) {
      sprintf(errmsg,
	"ByteStream2ObjectSlots: slot %s<%u%s,%u,%s> conversion error",
	      slot_name, plain_type, E? "E": "", pri, value);
      nslots = -1;
      break;
    }
    if (E && (_BosFindSlot(obj, slot_name) != (Bos_Slot *)0))
      continue;
    add_stat = Bos_AddSlot(obj, slot_name, plain_type, pri, val);
    if (add_stat == BOS_ALREADY) {
      op_mask = Bos_SET_TYPE | Bos_SET_PRI | Bos_SET_VALUE;
      add_stat = Bos_SetSlot(obj, op_mask, slot_name, plain_type, pri, val);
    }
    if (add_stat == BOS_OK && E) {
      op_mask = Bos_SET_EVANESCENT_ON;
      add_stat = Bos_SetSlot(obj, op_mask, slot_name, plain_type, pri, val);
    }
    if (add_stat != BOS_OK) {
      sprintf(errmsg, "could not add slot %s<%u%s,%u,%s> to %s",
	      slot_name, plain_type, E? "E": "", pri, value, obj->name);
      nslots = -1;
      break;
    }
  }
  if (idx_p != (int *)0)
    *idx_p = idx;
  return nslots;
}

int
Bos_ObjectSlotsSize(obj, nslots_p)
     Bos_Object *obj;
     int *nslots_p;
{
  int nslots, nbytes;
  Tcl_HashEntry *e;
  Tcl_HashSearch search;

  nslots = nbytes = 0;
  for (e = Tcl_FirstHashEntry(obj->slots, &search);
       e != (Tcl_HashEntry *)0;
       e = Tcl_NextHashEntry(&search)) {
    Bos_Slot *s;
    char *vs;
    int f;

    f = 0;
    s = (Bos_Slot *)Tcl_GetHashValue(e);
    nbytes += strlen(s->name) + sizeof(s->type) + sizeof(s->pri) + 1;
    vs = slot_Value2String(s, &f);
    nbytes += strlen(vs) + 1;
    if (f)
      ckfree(vs);
    nslots++;
  }
  if (nslots_p != (int *)0)
    *nslots_p = nslots;
  return nbytes;
}

/*
 * Local utility routines
 */

static char *
slot_Value2String(s, free_after_p)
     Bos_Slot *s;
     int *free_after_p;
{
  Bos_Slot_Type plain_type;
  int E, f;
  char *vs;

  plain_type = Bos_PlainSlotType(s->type);
  E = (s->type & Bos_SLOT_EVANESCENT_MASK);
  f = 0;
  switch (plain_type) {
  case Bos_SLOT_CMETHOD:
    vs = Bos_GetCMethodName(s->value);
    break;
  case Bos_SLOT_FOREIGN:
    vs = Bos_GetCSlotString(s->value, s->pri);
    f = 1;
    break;
  case Bos_SLOT_NORMAL:
  case Bos_SLOT_OBJECT:
  case Bos_SLOT_REFERENCE:
    vs = (char *)s->value;
    break;
  case Bos_SLOT_METHOD:
    vs = ((Bos_Method *)(s->value))->body;
    break;
  default:
    vs = (char *)0;
    break;
  }
  /* XXX
   *
   * For evanescent slots, we play the following trick: if the value
   * scans as an integer, then return the string "0", otherwise return
   * an empty string for its value. This is because E slots can't have
   * their values saved across executions -- it doesn't make any sense.
   * However, we want to preserve whether or not the slot was a number
   * or not, and we don't have any way to do that vis a vis slot types,
   * so we play this trick. This will be fixed for good when we get
   * real Int,Real,String slot types in the next release. --S
   */
  if (E) {
    char *new_vs;
    int xxx;

    if (sscanf(vs, "%d", &xxx) == 1)
      new_vs = "0";
    else
      new_vs = "";
    if (f)
      ckfree(vs);
    vs = new_vs;
  }
  *free_after_p = f;
  return vs;
}

static void *
slot_String2Value(v, t, p)
     char *v;
     Bos_Slot_Type t;
     Bos_Slot_Pri p;
{
  Bos_Slot_Type plain_type;
  void *val;

  plain_type = Bos_PlainSlotType(t);
  switch (plain_type) {
  case Bos_SLOT_CMETHOD:
    val = Bos_GetCMethodPointer(v);
    break;
  case Bos_SLOT_FOREIGN:
    val = Bos_ParseCSlotString(v, p);
    break;
  case Bos_SLOT_NORMAL:
  case Bos_SLOT_OBJECT:
  case Bos_SLOT_REFERENCE:
  case Bos_SLOT_METHOD:
    val = (void *)v;
    break;
  default:
    val = (void *)0;
    break;
  }
  return val;
}

static char *
enlarge_buffer(buf, len, size_p, need)
     char *buf;
     int len;
     int *size_p;
     int need;
{
  int new_size, size;
  char *temp;

  size = *size_p;
  new_size = 1024 * (1 + (size + need)/1024);
  temp = (char *)ckalloc(new_size);
  if (temp == (char *)0) {
    ckfree(buf);
    return (char *)0;
  }
  memcpy(temp, buf, len);
  ckfree(buf);
  buf = temp;
  *size_p = new_size;
  return buf;
}
