/* postgres-tcl.c - POSTGRES libpq/Tcl Interface
 *
 * Copyright (C) 1992,1993 Engineering Design Research Center
 *
 * Author: Sean Levy (snl+@cmu.edu)
 *         n-dim Group
 *         Engineering Design Research Center
 *         Carnegie Mellon University
 *         5000 Forbes Ave / PGH, PA / 51221
 *
 *         Fax: (412) 268-5229
 *         Voice: (412) 268-5226
 */

/*
 * postgres-tcl.c,v 1.1 1992/07/31 20:12:11 snl Exp
 *
 * postgres-tcl.c,v
 * Revision 1.1  1992/07/31  20:12:11  snl
 * Massive checkin
 *
 */

static char rcsID[] = "postgres-tcl.c,v 1.1 1992/07/31 20:12:11 snl Exp";

#include <stdio.h>
#include <tmp/libpq.h>
#include <tcl/tclHash.h>

#define TclProc(name) \
int name/**/Cmd (clientData, interp, argc, argv) \
ClientData clientData; \
Tcl_Interp *interp; \
int argc; \
char **argv;

#define UNIMPLEMENTED()\
Tcl_AppendResult(interp, argv[0], ": Tcl interface not yet implemented", 0);\
return TCL_ERROR

#define USAGE(_e,_u)\
Tcl_AppendResult(interp, argv[0], ": ", _e, " -- usage: \"", argv[0], _u, "\"", 0);\
return TCL_ERROR

#define lookup_PortalBuffer(pname)\
Tcl_FindHashEntry((Tcl_HashTable *)clientData, pname)

void
PQ_InitInterp(interp)
  Tcl_Interp *interp;
{
  int PQsetdbCmd(), PQdbCmd(), PQresetCmd(), PQfinishCmd(),
      PQexecCmd(), PQnportalsCmd(), PQnamesCmd(), pqInfoCmd(), PQparrayCmd(),
      PQclearCmd(), PQrulepCmd(), PQntuplesCmd(), PQngroupsCmd(),
      PQntuplesGroupCmd(), PQnfieldsGroupCmd(), PQfnameGroupCmd(),
      PQfnumberGroupCmd(), PQgetgroupCmd(), PQnfieldsCmd(),
      PQfnumberCmd(), PQfnameCmd(), PQftypeCmd(), PQsametypeCmd(),
      PQgetvalueCmd(), PQgetlineCmd(), PQputlineCmd(), PQtraceCmd(),
      PQuntraceCmd(), PQendcopyCmd();
  Tcl_HashTable *portals;

  portals = (Tcl_HashTable *)malloc(sizeof(Tcl_HashTable));
  Tcl_InitHashTable(portals, TCL_STRING_KEYS);

  Tcl_CreateCommand(interp, "PQinfo", pqInfoCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQsetdb", PQsetdbCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQdb", PQdbCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQreset", PQresetCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQfinish", PQfinishCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQexec", PQexecCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQnportals", PQnportalsCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQnames", PQnamesCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQparray", PQparrayCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQclear", PQclearCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQrulep", PQrulepCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQntuples", PQntuplesCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQngroups", PQngroupsCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQntuplesGroup", PQntuplesGroupCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQnfieldsGroup", PQnfieldsGroupCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQfnameGroup", PQfnameGroupCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQfnumberGroup", PQfnumberGroupCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQgetgroup", PQgetgroupCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQnfields", PQnfieldsCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQfnumber", PQfnumberCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQfname", PQfnameCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQftype", PQftypeCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQsametype", PQsametypeCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQgetvalue", PQgetvalueCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQgetline", PQgetlineCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQputline", PQputlineCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQendcopy", PQendcopyCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQtrace", PQtraceCmd, portals, 0);
  Tcl_CreateCommand(interp, "PQuntrace", PQuntraceCmd, portals, 0);
}

static
TclProc(pqInfo)
{
  extern char *PQhost, *PQport, *PQtty, *PQoption, *PQdatabase;
  extern int PQportset, PQxactid, PQtracep;

#define pretty_string(s) s? "\"": "", s? s: "(null)", s? "\"": ""

  printf("libpq globals:\n  PQhost=%s%s%s\n  PQport=%s%s%s\n  PQtty=%s%s%s\n  PQoption=%s%s%s\n  PQdatabase=%s%s%s\n  PQportset=%d\n  PQxactid=%d\n  PQtracep=%d\n",
	 pretty_string(PQhost),
	 pretty_string(PQport),
	 pretty_string(PQtty),
	 pretty_string(PQoption),
	 pretty_string(PQdatabase),
	 PQportset, PQxactid, PQtracep);
  return TCL_OK;
}

static TclProc(PQsetdb)
{
  char *db_name;

  if (argc < 2) {
    USAGE("wrong # args","dbname");
  }
  db_name = (char *)malloc(strlen(argv[1]) + 1);
  strcpy(db_name, argv[1]);
  PQsetdb(db_name);
  return TCL_OK;
}

static TclProc(PQdb)
{
  Tcl_SetResult(interp, PQdb(), TCL_STATIC);
  return TCL_OK;
}

static TclProc(PQreset)
{
  PQreset();
  return TCL_OK;
}

static TclProc(PQfinish)
{
  PQfinish();
  return TCL_OK;
}

static TclProc(PQexec)
{
  int a, e_len, result;
  char *e_buf, *pg_return, *PQexec();

  for (a = 1, e_len = 0; a < argc; a++)
    e_len += strlen(argv[a]);
  if (!e_len) {
    USAGE("need some sort of command","cmd ?args?");
  }
  e_buf = (char *)malloc(e_len + argc + 1);
  if (e_buf == (char *)0) {
    char n[20];

    sprintf(n, "%d", e_len + argc + 1);
    Tcl_AppendResult(interp, argv[0],
		     ": ran out of memory trying to malloc ", n,
		     " bytes for command", (char *)0);
    return TCL_ERROR;
  }
  *e_buf = '\0';
  for (a = 1; a < argc; a++) {
    strcat(e_buf, argv[a]);
    if ((a+1) < argc)
      strcat(e_buf, " ");
  }
  pg_return = PQexec(e_buf);
  switch (*pg_return) {
  case 'P':			/* command was retrieve: this is a portal */
    {
      Tcl_HashTable *portals = (Tcl_HashTable *)clientData;
      Tcl_HashEntry *portal;
      PortalBuffer *pb;
      int new_p;
      char *pname = pg_return+1;

      pb = PQparray(pname);
      if (pb == (PortalBuffer *)0) {
	Tcl_AppendResult(interp, argv[0], ": invalid portal name \"", argv[1],
			 "\"", (char *)0);
	result = TCL_ERROR;
	break;
      }
      new_p = 0;
      portal = Tcl_CreateHashEntry(portals, pname, &new_p);
      Tcl_SetHashValue(portal, (ClientData)pb);
    }
    /* Fall through...
     */
  case 'C':			/* command was not retrieve and won */
  case 'D':			/* FOO! Undocumented return from COPY */
    Tcl_SetResult(interp, pg_return+1, TCL_VOLATILE);
    result = TCL_OK;
    break;
  case 'R':			/* command lost */
    Tcl_AppendResult(interp, argv[0], ": POSTGRES error on command \"",
		     e_buf, "\"", (char *)0);
    result = TCL_ERROR;
    break;
  default:			/* huh? */
    Tcl_AppendResult(interp, argv[0], ": unknown POSTGRES return code \"",
		     pg_return, "\" while executing \"", e_buf, "\"",
		     (char *)0);
    result = TCL_ERROR;
    break;
  }
  free(e_buf);
  return result;
}

static
TclProc(PQparray)
{
  Tcl_HashTable *portals = (Tcl_HashTable *)clientData;
  Tcl_HashEntry *portal;
  PortalBuffer *pb;
  int new_p;
  
  if (argc < 2) {
    USAGE("wrong # args", "pname");
  }
  pb = PQparray(argv[1]);
  if (pb == (PortalBuffer *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal name \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  new_p = 0;
  portal = Tcl_CreateHashEntry(portals, argv[1], &new_p);
  Tcl_SetHashValue(portal, (ClientData)pb);
  Tcl_SetResult(interp, argv[1], TCL_VOLATILE);
}

static TclProc(PQnportals)
{
  char n[20];
  int rule_p;

  if (argc < 2) {
    USAGE("wrong # args","rule_p");
  }
  if (sscanf(argv[1], "%d", &rule_p) != 1) {
    USAGE("rule_p must be an integer", "rule_p");
  }
  sprintf(n, "%d", PQnportals(rule_p));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQnames)
{
  UNIMPLEMENTED();
}

static TclProc(PQclear)
{
  Tcl_HashEntry *pbe;

  if (argc < 2) {
    USAGE("wrong # args","pname");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  Tcl_DeleteHashEntry(pbe);
  PQclear(argv[1]);
  return TCL_OK;
}

static TclProc(PQrulep)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];

  if (argc < 2) {
    USAGE("wrong # args", "pname");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQrulep(pb));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQntuples)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];

  if (argc < 2) {
    USAGE("wrong # args", "pname");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQntuples(pb));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQngroups)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];

  if (argc < 2) {
    USAGE("wrong # args", "pname");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQngroups(pb));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQntuplesGroup)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int group_index;

  if (argc < 3) {
    USAGE("wrong # args", "pname group_index");
  } else if (sscanf(argv[2], "%d", &group_index) != 1) {
    USAGE("group_index did not scan as integer", "pname group_index");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQntuplesGroup(pb, group_index));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQnfieldsGroup)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int group_index;

  if (argc < 3) {
    USAGE("wrong # args", "pname group_index");
  } else if (sscanf(argv[2], "%d", &group_index) != 1) {
    USAGE("group_index did not scan as integer", "pname group_index");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQnfieldsGroup(pb, group_index));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQfnameGroup)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int group_index, field_number;

  if (argc < 4) {
    USAGE("wrong # args", "pname group_index field_number");
  } else if (sscanf(argv[2], "%d", &group_index) != 1) {
    USAGE("group_index did not scan as integer", "pname group_index field_number");
  } else if (sscanf(argv[3], "%d", &field_number) != 1) {
    USAGE("field_number did not scan as integer", "pname group_index field_number");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  Tcl_SetResult(interp, PQfnameGroup(pb, group_index, field_number),
		TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQfnumberGroup)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int group_index;

  if (argc < 4) {
    USAGE("wrong # args", "pname group_index field_name");
  } else if (sscanf(argv[2], "%d", &group_index) != 1) {
    USAGE("group_index did not scan as integer", "pname group_index field_name");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQfnumberGroup(pb, group_index, argv[3]));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQgetgroup)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int tuple_index;

  if (argc < 3) {
    USAGE("wrong # args", "pname tuple_index");
  } else if (sscanf(argv[2], "%d", &tuple_index) != 1) {
    USAGE("tuple_index did not scan as integer", "pname tuple_index");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQgetgroup(pb, tuple_index));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQnfields)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int tuple_index;

  if (argc < 3) {
    USAGE("wrong # args", "pname tuple_index");
  } else if (sscanf(argv[2], "%d", &tuple_index) != 1) {
    USAGE("tuple_index did not scan as integer", "pname tuple_index");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQnfields(pb, tuple_index));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQfnumber)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int tuple_index;

  if (argc < 4) {
    USAGE("wrong # args", "pname tuple_index field_name");
  } else if (sscanf(argv[2], "%d", &tuple_index) != 1) {
    USAGE("tuple_index did not scan as integer", "pname tuple_index field_name");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQfnumber(pb, tuple_index, argv[3]));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQfname)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int tuple_index, field_number;

  if (argc < 4) {
    USAGE("wrong # args", "pname tuple_index field_number");
  } else if (sscanf(argv[2], "%d", &tuple_index) != 1) {
    USAGE("tuple_index did not scan as integer", "pname tuple_index field_number");
  } else if (sscanf(argv[3], "%d", &field_number) != 1) {
    USAGE("field_number did not scan as integer", "pname tuple_index field_number");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  Tcl_SetResult(interp, PQfname(pb, tuple_index, field_number), TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQftype)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int tuple_index, field_number;

  if (argc < 4) {
    USAGE("wrong # args", "pname tuple_index field_number");
  } else if (sscanf(argv[2], "%d", &tuple_index) != 1) {
    USAGE("tuple_index did not scan as integer", "pname tuple_index field_number");
  } else if (sscanf(argv[3], "%d", &field_number) != 1) {
    USAGE("field_number did not scan as integer", "pname tuple_index field_number");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQftype(pb, tuple_index, field_number));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQsametype)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int tuple_index1, tuple_index2;

  if (argc < 4) {
    USAGE("wrong # args", "pname tuple_index1 tuple_index2");
  } else if (sscanf(argv[2], "%d", &tuple_index1) != 1) {
    USAGE("tuple_index1 did not scan as integer", "pname tuple_index1 tuple_index2");
  } else if (sscanf(argv[3], "%d", &tuple_index2) != 1) {
    USAGE("tuple_index2 did not scan as integer", "pname tuple_index1 tuple_index2");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  sprintf(n, "%d", PQsametype(pb, tuple_index1, tuple_index2));
  Tcl_SetResult(interp, n, TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQgetvalue)
{
  Tcl_HashEntry *pbe;
  PortalBuffer *pb;
  char n[20];
  int tuple_index, field_number;

  if (argc < 4) {
    USAGE("wrong # args", "pname tuple_index field_number");
  } else if (sscanf(argv[2], "%d", &tuple_index) != 1) {
    USAGE("tuple_index did not scan as integer", "pname tuple_index field_number");
  } else if (sscanf(argv[3], "%d", &field_number) != 1) {
    USAGE("field_number did not scan as integer", "pname tuple_index field_number");
  }
  pbe = lookup_PortalBuffer(argv[1]);
  if (pbe == (Tcl_HashEntry *)0) {
    Tcl_AppendResult(interp, argv[0], ": invalid portal \"", argv[1],
		     "\"", (char *)0);
    return TCL_ERROR;
  }
  pb = (PortalBuffer *)Tcl_GetHashValue(pbe);
  Tcl_SetResult(interp, PQgetvalue(pb, tuple_index, field_number),
		TCL_VOLATILE);
  return TCL_OK;
}

static TclProc(PQgetline)
{
  UNIMPLEMENTED();
}

static TclProc(PQputline)
{
  if (argc < 2) {
    USAGE("wrong # args", "line");
  }
  PQputline(argv[1]);
  return TCL_OK;
}

static TclProc(PQendcopy)
{
  PQendcopy();
  return TCL_OK;
}

static TclProc(PQtrace)
{
  PQtrace();
  return TCL_OK;
}

static TclProc(PQuntrace)
{
  PQuntrace();
  return TCL_OK;
}
