#
#------------------------------------------------------------------------------
#  Copyright (c) 1993-1995 European Synchrotron Radiation Facility
#
#  Permission to use, copy, modify, and distribute this software and its
#  documentation for any purpose and without fee is hereby granted, provided
#  that the "Terms and Conditions of Distribution", given in the documentation
#  to this software, are applicable and the above copyright message appears in
#  each copy.
#
#------------------------------------------------------------------------------
#
#
#  13.12.1993   hp      1st version
#
#  Simple find written in Tcl. This script searches the directory tree for
#  given filenames, starting at certain points in the tree and processing
#  them in a depth-first manner. As soon as it finds a file, it stops looking
#  for it any further. The script produces an output on stdout like:
#  			   [argv0=File0;argv1=File1;....] 
#  and returns an exit code of zero. File? is the path leading to the file 
#  (excluding the filename itself). If a file isn't found, its variable is left
#  empty (i.e.: ....argvX=FileX;argvY=;argvZ=FileZ.....). If no files are
#  found, all the variables are left empty. If the script should produce
#  absolute pathnames, it should be called with absolute pathnames, where to
#  start the search in (i.e. no "." ".." "xyz/" "~xyz" etc.). If the paths to
#  start the search in, aren't disjoint, they should be ordered top to bottom
#  (in terms of the directory tree), so that a directory isn't scanned twice.
#
#  The script produces an output, that can be used within a shell script to set
#  the variables as shell variables. A shell script might look like:
#
#	eval `wish -f ./find.tcl A fsck B login -dir /bin /etc`
#	echo $A
#	echo $B
#
#
#  Examples: 
#
#  1)  wish -f find.tcl A xyz B abc -dir /usr /users /
#
#	If one of the files (xyz, abc) isn't found in /usr and /users, the
#	script will scan / excluding /usr and /users and all their subdirs.
#
#  2)  wish -f find.tcl A fsck B login C notFound -dir /bin /etc /usr
#
#	On a "normal" Unix system this will produce the output:
#
#	B=/bin;A=/etc;C=
#
#




#  check the arguments  #

if {$argc < 3} then {
  puts stdout "Syntax is: $argv0 Var File ?Var File ....? -dir Directory \
	       ?Directory ...?"
  exit 1
}


#  set variables according to the arguments  #

set listOfFiles $argv
set listOfVars  $argv
set listOfDirs  ""
set index 0
set vars 0
set files 0
foreach arg $argv {
  if {$arg == "-dir"} {
    set listOfDirs [lreplace $argv 0 $index]
    set listOfVars [lreplace $listOfVars $vars [llength $listOfVars]]
    set listOfFiles [lreplace $listOfFiles $files [llength $listOfFiles]]
    break
  }
  if {($index % 2) == 0} then {
    set listOfFiles [lreplace $listOfFiles $files $files]
    incr vars
  } else {
      set listOfVars [lreplace $listOfVars $vars $vars]
      incr files
    }
  incr index
}
set index 0
foreach dir $listOfDirs {
  set dir [string trimright $dir /]
  if {$dir == ""} {
    set dir /
  }
  set listOfDirs [lreplace $listOfDirs $index $index $dir]
  if {([file readable $dir] != 1) || ([file isdirectory $dir] != 1)} then {
    set listOfDirs [lreplace $listOfDirs $index $index]
  } else {
      incr index
    }
}
if {($listOfFiles == "") || ($listOfVars == "") || ($listOfDirs == "") ||
    ([llength $listOfFiles] != [llength $listOfVars])} {
  puts stdout "Syntax is: $argv0 Var File ?Var File ....? -dir Directory \
	       ?Directory ...?"
  exit 1
}
set dirList ""
set curDir ""




#  prepare the window  #

wm geometry . 600x496
wm title . "Find Files"
frame .t1
frame .t2
frame .t1.f1 -bd 2
frame .t1.f2 -bd 2
frame .t2.f3 -bd 2
frame .t2.f4 -bd 2
frame .t2.f5 -bd 2
frame .t2.f6 -bd 2
label .t1.f1.l -text "List of Variables" -relief raised -bd 2
label .t1.f2.l -text "List of Files" -relief raised -bd 2
label .t2.f3.l -text "List of Directories" -relief raised -bd 2
label .t2.f4.l -text "Current Directory" -relief raised -bd 2
label .t2.f5.l -text "Found Files" -relief raised -bd 2
listbox .t1.f1.m -relief sunken -geometry 35x4
listbox .t1.f2.m -relief sunken -geometry 35x4
listbox .t2.f3.m -relief sunken -geometry 80x10
listbox .t2.f5.m -relief sunken -geometry 80x4
entry .t2.f4.e -textvar curDir -relief sunken -bd 2 -width 50 -state disabled
button .t2.f6.cancel -text "Cancel" \
		     -command "foreach var \$listOfVars \{
				 if \{\$output == \"\"\} then \{
				   set output \"\$var=\"
				 \} else \{
				      set output \"\$output;\$var=\"
    				    \}
			       \};
			       puts stdout \$output;
			       exit 0
			      "
pack .t1 .t2 -side top
pack .t1.f1 -side left
pack .t1.f2 -side right
pack .t2.f3 .t2.f4 .t2.f5 .t2.f6 -side top -fill x
pack .t1.f1.l .t1.f1.m -side top -fill x
pack .t1.f2.l .t1.f2.m -side top -fill x
pack .t2.f3.l .t2.f3.m -side top -fill x
pack .t2.f4.l -side left -fill y
pack .t2.f4.e -side right -fill y
pack .t2.f5.l .t2.f5.m -side top -fill x
pack .t2.f6.cancel -side top -fill x
set dirList $listOfDirs
foreach var $listOfVars {
  .t1.f1.m insert end $var
}
foreach file $listOfFiles {
  .t1.f2.m insert end $file
}
foreach dir $dirList {
  .t2.f3.m insert end $dir
}
update




#  start searching  #

set output ""
while {$dirList != {}} {

  update

  #  take the first argument of the directory list and check it  #

  set curDir [lindex $dirList 0]
  set dirList [lreplace $dirList 0 0]
  .t2.f3.m delete 0 0

  #  check if the file is in there  #

  set fileIndex 0
  set delList ""
  foreach file $listOfFiles {
    if {$curDir == "/"} then {
      set absFile [glob -nocomplain /$file]
    } else {
        set absFile [glob -nocomplain $curDir/$file]
      }
    if {([llength $absFile] != 0) && ([file isfile $absFile] == 1)} {
      set path [split $file /]
      set curPath [split $absFile /]
      set index1 [expr {[llength $curPath]-[llength $path]}]
      set index2 [expr {[llength $curPath]-1}]
      .t2.f5.m insert end [format "%s=%s" \
			     [lindex $listOfVars $fileIndex] \
			     [join [lreplace $curPath $index1 $index2] /]]
      if {$output == ""} then {
	set output [format "%s=%s" \
		      [lindex $listOfVars $fileIndex] \
		      [join [lreplace $curPath $index1 $index2] /]]
      } else {
	  set output [format "%s;%s=%s" \
			$output \
			[lindex $listOfVars $fileIndex] \
			[join [lreplace $curPath $index1 $index2] /]]
	}
      set delList "$fileIndex $delList"
    }
    incr fileIndex
  }
  foreach index $delList {
    .t1.f1.m delete $index $index
    .t1.f2.m delete $index $index
    set listOfVars [lreplace $listOfVars $index $index]
    set listOfFiles [lreplace $listOfFiles $index $index]
  }
  if {[llength $listOfFiles] == 0} {
    puts stdout $output
    exit 0
  }

  #  scan the directory for subdirectories  #

  if {[catch {set dir [exec ls -1F $curDir | grep /\$]}] == 0} {
    if {$curDir == "/"} then {
      foreach file $dir {
	set file "/[string trimright $file /]"    
	if {([file readable $file] == 1) && 
	    ([lsearch $listOfDirs $file] == -1)} {
	  set dirList "$file $dirList"
	  .t2.f3.m insert 0 $file
	}
      }
    } else {
	foreach file $dir {
	  set file "$curDir/[string trimright $file /]"    
	  if {([file readable $file] == 1) && 
	      ([lsearch $listOfDirs $file] == -1)} {
	    set dirList "$file $dirList"
	    .t2.f3.m insert 0 $file
	  }
	}
      }
  }
}
		    


#  if files aren't found, set their corresponding variables to the empty string

foreach var $listOfVars {
  if {$output == ""} then {
    set output "$var="
  } else {
      set output "$output;$var="
    }
}
puts stdout $output
exit 0
