# $Id: interpBaseC.tcl,v 1.3 1993/06/08 06:15:41 david Exp $
# AUTHOR:	David Herron <david@davids.mmdf.com> (home)
#			     <david@twg.com>	     (work)
#
# Copyright 1993 David Herron.
# Permission to use, copy, modify and distribute this software
# and its documentation for any purpose and without fee is
# hereby granted, provided that the above copyright notice
# appear in all copies.  David Herron makes no representations
# about the suitability of this software for any purpose save
# printing it out and using the paper as bird cage lining.
#
#
#
#
#
# $Log: interpBaseC.tcl,v $
# Revision 1.3  1993/06/08  06:15:41  david
# Add gen_interp_name.
#
# Revision 1.2  1993/06/06  06:35:33  david
# Add `add_module_directory' and `load_module' definition.  Tweaks for
# new module format.
#
#

Define_Module InterpBase {

# Method - Create a proc in an interp which is executable from outside the interp.
#
# This procedure itself cannot be executable from outside the interp.  If it
# were then anybody could add commands to the interp, negating all our
# security procedures.
#
proc Method {name args body} {
	if ![-IsMethod $name] {	-AddMethod $name }
	proc $name $args $body
}

# -export - Record that a particular variable is allowed to
#	be accessed (set/unset/get) from outside.
Method -export args {
	global ExportList
	if ![info exists ExportList] {
		set ExportList ""
	}
	foreach var $args {
		lappend ExportList "$var"
	}
}

Method -unexport args {
	global ExportList
	if [info exists ExportList] {
		foreach var $args {
			set o [lsearch $ExportList $var]
			if {$o >= 0} {
				set ExportList [lreplace $ExportList $o $o]
			}
		}
	}
}

Method -exported {} { global ExportList ; return $ExportList }

#
# -set		Set the named variable.
# -get		Get the value of the variable.
# -unset	Unset the named variable.
#
Method -set {varName value} {
	global ExportList
	if {![info exists ExportList] || [lsearch $ExportList $varName] < 0} {
		error "var: Not allowed to access variable $varName"
	}
	upvar #0 $varName var
	set var $value
	return $var
}

Method -get {varName} {
	global ExportList
	if {![info exists ExportList] || [lsearch $ExportList $varName] < 0} {
		error "var: Not allowed to access variable $varName"
	}
	upvar #0 $varName var
	return $var
}

Method -unset {varName} {
	global ExportList
	if {![info exists ExportList] || [lsearch $ExportList $varName] < 0} {
		error "var: Not allowed to access variable $varName"
	}
	global $varName
	unset $varName
}

#
# -import	Create command in interp `dest' which exec's the named
#		command that is stored in interp `src'.  But the execution
#		is within dest's context.
#
# ???? What about the functions which are *not* Method's ????
#
Method -import {dest src methods} {
	if {[lsearch [$dest -exec { info commands }] Method] < 0} {
		$dest -exec [list proc Method [info args Method] \
					 [info body Method]]
	}
	foreach method $methods {
		$dest -exec [list Method $method args \
			"eval \"-execHere $src $method \$args\"" ]
	}
}

Method -importAll {dest src} { -import $dest $src [$src -Methods] }

#
# -chain	Create a command in the local interp which passes itself
#		to interp `dest' for evaluation.
#
# There is a significant difference between -chain and -import.  It is
# in *where* the execution and side effects happen.  In -import, these happen
# in the local interp, while in -chain it is in the remote one.
#
# ???? WHAT IF $dest DOES NOT ALLOW -exec ????
#
Method -chain {dest cmds} {
	foreach cmd $cmds {
		proc $cmd args "$dest -exec \"$cmd \$args\""
	}
}

# TODO:
#
# Set up traces so that accesses end up over in the other interpretor.
# Should -export allow specifying ACL's?  Or simple read/write permissions?
# Should -chainVar allow specifying what kind of `chain'? (read-only, etc)
#
# Method -chainVar {varName} {
# }

#
# gen_interp_name basename
#
# Generate a new interpreter name from the base name.  Keeps a counter
# in an array for each base name.  As new ones are requested the counter
# is incremented until it finds an interpretor not already named.
#
Method gen_interp_name basename {
	global icount
	if ![info exists icount($basename)] { set icount($basename) 0 }
	incr icount($basename)
	set name [format "%s%d" $basename $icount($basename)]
	while {[interp exists "$name"]} {
		incr icount($basename)
		set name [format "%s%d" $basename $icount($basename)]
	}
	return $name
}

#
proc read_module_index dir {
	global minterp_index

	if ![file exists $dir/minterpIndex] {
		error "Index file for module directory $dir does not exist.  Try running `auto_index_modules $dir *'."
	}

	set f [open "$dir/minterpIndex" "r"]
	gets $f line
	if {[regexp {^# MINTERP autoload index file} $line] != 1} {
		error "Bad index file in module directory $dir.  Try running `auto_index_modules $dir *'."
	}
	while {[gets $f line] >= 0} {
		if {([string index $line 0] == "#") ||
		    ([llength $line] != 2)} {
			continue
		}
		set modulename [lindex $line 0]
		set minterp_index($modulename) $dir/[lindex $line 1]
	}
	close $f
}


Method add_module_directory dirs {
	global mod_dirs
	if ![info exists mod_dirs]         { set mod_dirs "" }

	foreach dir $dirs {
		if {[lsearch $mod_dirs $dir] >= 0} { continue }
		read_module_index $dir
		lappend mod_dirs $dir
	}
}

Method load_module module {
	if [interp exists $module] { return $module }

	global minterp_index
	if [info exists minterp_index($module)] {
		MainInterp -exec "source $minterp_index($module)"
	} else {
		error "Module $module is not installed in any of the module directories."
	}

	if ![interp exists $module] {
		error "Could not find module $module."
	}

	return $module
}

# Allow for calling interp's to do:
#
#	-chain MainInterp [InterpBase -get $TkChain]
#
set TkChain {
	bind button canvas checkbutton destroy entry grab frame
	label listbox options
	pack scrollbar
	text toplevel wm winfo
	}
-export TkChain

set EXPORTS {
	-export -unexport -exported -set -get -unset -import -importAll -chain
	gen_interp_name
	}
-export EXPORTS

-AllowExec false
}
