#
#	Implementation of a simple bar chart
#
#	A bar chart is given by
#	o  a title
#	o  a range specification
#	o  two axis labels
#	o  a list of bar labels
#	o  a list of data elements
#
#	Each data element is a list of a label and values, where the
#	number of values must be the same as the number of bar labels
#	given.
#
#	Baractions are activated on double clicks with the left button.
#	They are invoked with four arguments:
#	index of x-axis label (0..), index of bar for that label (0..),
#	label of that bar (legend), value associated with that bar.
#
#	Legendtext is a method for generating the additional legend text.
#	The action handler is called with two arguments: the sum of the
#	values for this legend item, and the number of items.
#	Standard methods are LegendTotal, LegendTotalFloat, LegendAverage,
#	LegendAverageFloat.
#
#	Stacked bars are available by setting "-stacked true".
#

defwidget Barchart Window {
	{sizex 900}
	{sizey 600}
	{sep 4}
	{marg 10}
}

defmethod Barchart new {name args} {

  args	title text {step 10} {max 100} xlabel ylabel buttons actions \
	baraction legend data layout colors {factor 1} {embedded true} \
	legendtext stacked bartext nolabels

  if { [streq $embedded "true"] } {
    Frame new $name
  } {
    Toplevel new $name \
	-title $title \
	-handler [list $name _button] \
	-buttons $buttons \
	-actions $actions
  }
  defsuper $name Barchart

  if { [streq $legendtext {}] } {
    set legendtext [list $name LegendTotal Total]
  } {
    set legendtext [$self buildAction $legendtext $name]
  }

  if { [streq $bartext {}] } {
    set bartext [list $name BarTotal]
  } {
    set bartext [$self buildAction $bartext $name]
  }

  set stacked [streq $stacked "true"]

  set sizex [$self slot sizex]
  set sizey [$self slot sizey]
  set sep [$self slot sep]
  set marg [$self slot marg]
  set marg2 [expr $marg/2]

  set margl [expr 6*[Font slot text,w]+$marg]
  set margr $marg
  set margb [expr [Font slot textbold,h]+[Font slot text,h]+$marg]
  set margt [expr [Font slot largebold,h]+[Font slot textbold,h]+$marg]

  # n: number of x-axis items
  # m: number of legend items (subitems per x-axis item)

  set n [llength $data]
  set m [llength $legend]

  set smallboldfont [Font slot smallbold]
  set smallboldh [Font slot smallbold,h]
  set textfont [Font slot text]
  set textboldfont [Font slot textbold]

  # Determine the colors for the bars

  if { [streq [Color pick mono color] "mono"] } {
    set colors {}
    foreach c $legend {
      lappend colors black
    }
  }
  if { [streq $colors {}] } {
    set level 216
    if { $m == 1 } {
      set levelinc 0
    } {
      set levelinc [expr -1*($level-60)/$m]
      incr level [expr $levelinc/2]
    }
    foreach c $legend {
      lappend colors [format "#%02x%02x%02x" $level $level $level]
      incr level $levelinc
    }
  }

  foreach label $legend {
    set sum($label) 0
  }

  Canvas new $name.c -width $sizex -height $sizey -layout {top}
  set canvas [$name.c canvas]

  # Create the title and the axes

  $canvas! create text [expr $sizex/2] $marg \
	-text $text -anchor n -font [Font slot hugebold] -tags title
  set bbox [$canvas! bbox title]
  set bbox [list [expr [lindex $bbox 0]-$marg2] \
		 [expr [lindex $bbox 1]-$marg2] \
		 [expr [lindex $bbox 2]] \
		 [expr [lindex $bbox 3]+$marg2]]
  eval [concat	[list $canvas! create rectangle] $bbox \
		[list -fill white -tags titlebackground]]
  $canvas! lower titlebackground title
  set bbox [list [expr [lindex $bbox 0]+4] \
		 [expr [lindex $bbox 1]+4] \
		 [expr [lindex $bbox 2]+4] \
		 [expr [lindex $bbox 3]+4]]
  eval [concat	[list $canvas! create rectangle] $bbox \
		[list -fill black -tags titlebackground2]]
  $canvas! lower titlebackground2 titlebackground

  $canvas! create text 10 [expr 10+[Font slot largebold,h]] \
	-text $xlabel -anchor nw -font $textboldfont
  $canvas! create text [expr $sizex-10] [expr $sizey-10] \
	-text $ylabel -anchor se -font $textboldfont

  $canvas! create line $margl $margt $margl [expr $sizey-$margb] \
	[expr $sizex-$margr] [expr $sizey-$margb] \
	-width 2 -fill black

  set stepy [expr ($sizey-$margt-$margb)*$step/$max]
  set maxx [expr $sizex-$margr]

  # Draw the y axis labels

  for {set c 0; set y [expr $sizey-$margb]; set i 0} \
	{$c <= $max} \
	{incr c $step; incr y -$stepy; incr i} {
    $canvas! create text [expr $margl-$marg2] $y \
	-text $c -anchor e -font $textfont
    $canvas! create line $margl $y $maxx $y \
	-width 1 -fill black
  }

  set stepx [expr ($sizex-$margl-$margr)/$n]
  set maxy [expr $sizey-$margb]
  set texty [expr $maxy+3]
  set barx [expr $stepx-2*$sep]
  if { ! $stacked } {
    set barx [expr $barx/$m]
  }
  set height [expr $max/$step*$stepy]

  if { ![streq $baraction ""] } {
    $canvas bind bar(rect) <Double-Button-1> \
	[list $name _barAction [$name buildAction $baraction]]
  }

  # Draw the barchart itself

  for {set c 0; set x $margl} {$c < $n} {incr c; incr x $stepx} {

    set values [lindex $data $c]
    $canvas! create text [expr $x+$stepx/2] $texty \
	-text [lindex $values 0] -anchor n -font $textfont

    set pos [expr $x+$sep]
    set i 1
    if { $stacked } {
      # If stacked, reset position of first bar to bottom line
      set maxy [expr $sizey-$margb]
    }

    foreach label $legend {
      set val [lindex $values $i]
      if { [streq $val {}] } {
        set val 0
      } {
	set val [expr [lindex $values $i]*$factor]
	set sum($label) [expr $sum($label)+$val]
      }

      if { $val > 0 } {

	set valuey [expr $maxy-$val*$height/$max]
	set tags [list bar(rect) \
		[list bar $c [expr $i-1] $label [lindex $values $i]]]

	$canvas newRect3D $pos $valuey [expr $pos+$barx+1] $maxy \
	  -fill [lindex $colors [expr $i-1]] \
	  -tags $tags -tags2 $tags

	set xpos [expr $pos+$barx/2+2]
	if { ![streq $nolabels "true"] } {
	  set valtext [uplevel #0 [concat $bartext $val]]
	  set bbox [textsize $smallboldfont $valtext]
	  set tx [lindex $bbox 0]
	  set ty [lindex $bbox 1]

	  if { $stacked } {
	    set ypos1 [expr $valuey-2]
	    set ypos2 [expr $valuey+$ty]
	    set ypos3 [expr $valuey-1+$ty]
	  } {
	    set ypos1 [expr $valuey-7]
	    set ypos2 [expr $valuey-9-$ty]
	    set ypos3 [expr $valuey-8]
	  }

	  $canvas! create rectangle \
		[expr $xpos-$tx/2-1] $ypos1 \
		[expr $xpos+$tx/2+1] $ypos2 \
		-fill white -tags bar(label)
	  $canvas! create text $xpos $ypos3 \
		-text $valtext \
		-anchor s -font $smallboldfont \
		-tags bar(label)
	}

	if { $stacked } {
	  # If stacked: increment position only if val>0.
	  set maxy $valuey
	}
      }

      if { ! $stacked } {
	# If not stacked: increment position to new bar anyway.
	incr pos $barx
      }

      incr i
    }
  }

  catch {$canvas! lower bar(rect) bar(label)}

  set xpos [expr $sizex-$margr-$marg]
  set ypos $margt
  set x2pos [expr $xpos-$smallboldh*3]
  set y2pos [expr $ypos+$smallboldh*6/5]
  set steppos [expr $smallboldh*13/5]

  # Draw the legend

  set i 0

  foreach label $legend {

    $canvas! create rectangle $x2pos [expr $y2pos+$smallboldh] $xpos $ypos \
	-fill [lindex $colors $i] -tags legend

    $canvas! create text [expr $x2pos-6] [expr ($ypos+$y2pos)/2] \
	-text $label -font $smallboldfont -anchor e -tags legend

    set legendlabel [uplevel #0 [concat $legendtext [list $sum($label) $n]]]
    $canvas! create text [expr $x2pos-6] [expr ($ypos+$y2pos)/2+$smallboldh] \
	-text $legendlabel -font [Font slot small] \
	-anchor e -tags legend

    $name slot sum$i $sum($label)
    incr ypos $steppos
    incr y2pos $steppos
    incr i
  }

  set bbox [$canvas! bbox legend]
  set bbox [list [expr [lindex $bbox 0]-$marg2] \
		 [expr [lindex $bbox 1]-$marg2] \
		 [expr [lindex $bbox 2]+$marg2] \
		 [expr [lindex $bbox 3]+$marg2]]
  eval [concat	[list $canvas! create rectangle] $bbox \
		[list -fill white -tags legendbackground]]
  $canvas! lower legendbackground legend

  $name layout $layout
}

defmethod Barchart postscript {args} {

  eval [concat [list $self.c postscript] $args]
}

defmethod Barchart _button {button action} {

  case $button {
  {Dismiss} {
	$self Dismiss
    }
  {Help} {
	$self Help $action
    }
  default {
	if { ![streq $action {}] } {
	  uplevel #0 $action
	}
    }
  }
}

defmethod Barchart _barAction {action} {

  set canvas [$self.c canvas]
  set id [$canvas find withtag current]
  set tags [$canvas gettags $id]

  set data [assoc bar $tags]
  if { ![streq $data {}] && ![streq $action {}] } {
    uplevel #0 [concat $action $data]
  }
}

#---------------------------------------------------------------------------

defmethod Barchart BarTotal {n} {

  format %0.1f $n
}

#---------------------------------------------------------------------------

defmethod Barchart LegendAverage {label sum n} {

  return "${label}: [expr $sum/$n]"
}

defmethod Barchart LegendAverageFloat {label sum n} {

  return "${label}: [format %0.1f [expr $sum/$n]]"
}

defmethod Barchart LegendTotal {label sum n} {

  return "${label}: $sum"
}

defmethod Barchart LegendTotalFloat {label sum n} {

  return "${label}: [format %0.1f $sum]"
}

#---------------------------------------------------------------------------

Window addDemo Barchart

defmethod Barchart demo {} {

  set items {
	{Abt1 0 30} {Abt2 100 100} {Abt3 150 300} {Abt4 100 550}
	{Abt5 20 610} {Abt6 250 600} {Abt7 350 750} {Abt8 100 1000}
	{Abt9 100 420} {Abt10 10 30}
	}

  set b [anon]

  Barchart new $b \
	-colors {#e74f4f #ac1c1c} \
	-text "3.5\" Diskettenverbrauch in 1993" \
	-legend [list "3.5 DD" "3.5 HD"] \
	-embedded false \
	-max 1500 \
	-step 100 \
	-xlabel "Verbrauch" \
	-ylabel "Abteilungen" \
	-data $items \
	-baraction {- showAction}

  Button new *$b \
	-text "Dismiss" \
	-textfont large \
	-action "$b Dismiss" \
	-layout {bottom padx 30 pady 30}

  Button new *$b \
	-text "Print" \
	-textfont large \
	-action [list $b postscript -colormode gray -file Stats.ps -rotate 1] \
	-layout {bottom padx 30 pady 30}

  $b layout +10+10

  set b [anon]

  Barchart new $b \
	-colors {#e74f4f #ac1c1c} \
	-text "3.5\" Diskettenverbrauch in 1993" \
	-legend [list "3.5 DD" "3.5 HD"] \
	-embedded false \
	-max 1500 \
	-step 100 \
	-xlabel "Verbrauch" \
	-ylabel "Abteilungen" \
	-data $items \
	-baraction {- showAction} \
	-legendtext {- LegendAverage Durchschnitt} \
	-stacked true

  Button new *$b \
	-text "Dismiss" \
	-textfont large \
	-action "$b Dismiss" \
	-layout {bottom padx 30 pady 30}

  Button new *$b \
	-text "Print" \
	-textfont large \
	-action [list $b postscript -colormode gray -file Stats.ps -rotate 1] \
	-layout {bottom padx 30 pady 30}

  $b layout -10-10
}
