#---------------------------------------------------------------------------
#
#	File:	the-matrix.tcl
#	Rectangular matrix widgets of various text items
#
#	I wrote this widget within 3 hours as I tried to demonstrate how
#	easy it is to create a tabular (matrix) widget with deletable
#	and insertable rows and columns with Tcl/Tk (in contrast to the
#	amount of work necessary with a particular different GUI builder
#	based on C that we use here at FhG-IAO).
#							--Juergen
#
#	The elements (actually, the creator objects) arranged in a matrix
#	must understand following messages:
#
#	new NAME -width WIDTH -text TEXT -textfont FONT -action ACTION
#	get
#	set VALUE
#
#	Actions are invoked with the entire widget path name, the 0-origin
#	coordinates of the element causing the action, and additional,
#	widget-specific arguments.
#
#---------------------------------------------------------------------------

defwidget Matrix Window {
	{cw 600}
	{ch 400}
	{margx 6}
	{margy 6}
	{sepx 6}
	{sepy 6}
	{border 4}
}

defmethod Matrix new {name args} {

  args	layout xlabels ylabels values textfont labelfont width height \
	action creators grid widths

  lappend creators {*,* Entry}
  lappend width {* 15}

  if { $height == {} } {
    set height 1
  }

  if { $textfont == {} } {
    set textfont text
  }
  if { $labelfont == {} } {
    set labelfont bold
  }

  Canvas new $name -relief flat \
	-width [$self slot cw] -height [$self slot ch] \
	-layout $layout \
	-scroll [list [$self slot cw] [$self slot ch]]
  defsuper $name Matrix

  set action [$name buildAction $action]
  foreach slot {xlabels ylabels values textfont labelfont width height
		action creators grid} {
    $name slot $slot [set $slot]
  }

  $name Redisplay 1
}

defmethod Matrix Redisplay {{always 0}} {

  if { $always == 0 && [$self slot _ok] } {
    return
  }
  $self slot _ok 1

  foreach slot {xlabels ylabels values textfont labelfont width height
		action creators grid} {
    set $slot [$self slot $slot]
  }
  set canvas [$self canvas]

  $canvas DismissChildren
  $canvas delete matrix

  set nx [llength $xlabels]
  set ny [llength $ylabels]
  set fonttext [Font slot $textfont]
  set fontlabel [Font slot $labelfont]

  set margx [Matrix slot margx]
  set margy [Matrix slot margy]
  set sepx [Matrix slot sepx]
  set sepy [Matrix slot sepy]

  set bd [Matrix slot border]
  set bbox [textsize $fontlabel "x"]
  set width1 [lindex $bbox 0]
  set sizey [expr [lindex $bbox 1]*$height+2*$bd]
  set incy [expr $sizey+$sepy]

  # create x labels
  set posx [expr $margx+$width1*[assocp label $width]+2*$bd+$sepx]
  set posy $margy
  for {set i 0} {$i < $nx} {incr i} {
    set wid [assocp $i $width]
    set sizex [expr $width1*$wid+2*$bd]
    set win [$self newTextElement $posx $posy $sizex $sizey \
	-type {Label -anchor c} -tags {matrix matrix(xlabel)} \
	-text [lindex $xlabels $i] -textfont $labelfont \
	-width $wid]
    $self slot x$i $win
    incr posx [expr $sizex+$sepx]
  }
  set maxx [expr $posx+$margx-$sepx]

  # create y labels
  set wid [assocp label $width]
  set sizex [expr $width1*$wid+2*$bd]
  set posx $margx
  set posy $margy
  for {set j 0} {$j < $ny} {incr j} {
    incr posy $incy
    set win [$self newTextElement $posx $posy $sizex $sizey \
	-type {Label -anchor w} -tags {matrix matrix(ylabel)} \
	-text [lindex $ylabels $j] -textfont $labelfont \
	-width $wid]
    $self slot y$i $win
  }
  set maxy [expr $posy+$sizey+$margy]

  # Adjust the canvas scroll region
  $self adjust [list $maxx $maxy]

  set posx0 [expr $margx+$width1*[assocp label $width]+2*$bd+$sepx]
  set posy $margy
  for {set j 0} {$j < $ny} {incr j} {
    incr posy $incy
    set posx $posx0
    set list [lindex $values $j]
    for {set i 0} {$i < $nx} {incr i} {
      # Create and set element <i,j>
      set wid [assocp $i $width]
      set sizex [expr $width1*$wid+2*$bd]
      set creator [assocp $i,$j $creators]
      if { [lindex $creator 0] == {} } {
	set id [eval [concat [list $canvas create text \
			[expr $posx+$sizex/2] [expr $posy+$sizey/2] \
			-text [lindex $list $i] \
			-font [Font slot $textfont] -width $sizex \
			-tags {matrix matrix(value)} -anchor c] \
		     [lrange $creator 1 end]]]
	set win [TextItem clone [anon $canvas] \
			[list [list canvas $canvas] [list id $id]]]
      } {
        set extra [list -action [list $self _action $i $j $action]]
        set win [$self newTextElement $posx $posy $sizex $sizey \
		-type [concat $creator $extra] \
		-tags {matrix matrix(value)} \
		-text [lindex $list $i] -textfont $textfont -width $wid]
      }
      $self slot $i,$j $win
      incr posx [expr $sizex+$sepx]
    }
  }

  if { $grid == "true" } {
    set posx [expr $posx0-$sepx/2]
    for {set i 0} {$i < $nx} {incr i} {
      $canvas create line $posx 0 $posx $maxy -tags {matrix matrix(line)}
      incr posx [expr $width1*[assocp $i $width]+2*$bd+$sepx]
    }

    set posy [expr $margy+$sizey+$sepy/2]
    for {set j 0} {$j < $ny} {incr j} {
      $canvas create line 0 $posy $maxx $posy -tags {matrix matrix(line)}
      incr posy $incy
    }
  }

  return $self
}

defmethod Matrix _action {column row action args} {

  if { $action != {} } {
    uplevel #0 [concat $action [list $self $column $row] $args]
  }
}

defmethod Matrix DeleteRow {i} {

  set ylabels [$self slot ylabels]
  set ny [llength $ylabels]
  if { $i < 0 || $i >= $ny } {
    return
  }

  set creators {}
  foreach creator [$self slot creators] {
    set range [split [lindex $creator 0] "," ]
    set x [lindex $range 0]
    set y [lindex $range 1]
    if { $i == $y } {
      continue
    }
    if { $y != "*" && $i < $y } {
      incr y -1
    }
    lappend creators [concat [list $x,$y] [lrange $creator 1 end]]
  }
  $self slot creators $creators

  $self slot ylabels [lreplace $ylabels $i $i]

  set values [$self slot values]
  if { [llength $values] > $i } {
    $self slot values [lreplace $values $i $i]
  }

  $self Redisplay 1
}

defmethod Matrix DeleteColumn {i} {

  set xlabels [$self slot xlabels]
  set nx [llength $xlabels]
  if { $i < 0 || $i >= $nx } {
    return
  }

  set creators {}
  foreach creator [$self slot creators] {
    set range [split [lindex $creator 0] "," ]
    set x [lindex $range 0]
    set y [lindex $range 1]
    if { $i == $x } {
      continue
    }
    if { $x != "*" && $i < $x } {
      incr x -1
    }
    lappend creators [concat [list $x,$y] [lrange $creator 1 end]]
  }
  $self slot creators $creators

  set width {}
  foreach wid [$self slot width] {
    set x [lindex $wid 0]
    if { $i == $x } {
      continue
    }
    if { $x != "label" && $x != "*" && $i < $x } {
      lappend width [concat [expr $x-1] [lrange $wid 1 end]]
    } {
      lappend width $wid
    }
  }
  $self slot width $width
  
  $self slot xlabels [lreplace $xlabels $i $i]

  set newvalues {}
  foreach list [$self slot values] {
    if { [llength $list] > $i } {
      set list [lreplace $list $i $i]
    }
    lappend newvalues $list
  }
  $self slot values $newvalues

  $self Redisplay 1
}

defmethod Matrix InsertRow {i label {newvalues {}}} {

  set ylabels [$self slot ylabels]
  set ny [llength $ylabels]
  if { $i < 0 || $i > $ny } {
    return
  }

  set creators {}
  foreach creator [$self slot creators] {
    set range [split [lindex $creator 0] "," ]
    set x [lindex $range 0]
    set y [lindex $range 1]
    if { $y != "*" && $i <= $y } {
      incr y
    }
    lappend creators [concat [list $x,$y] [lrange $creator 1 end]]
  }
  $self slot creators $creators

  $self slot ylabels [linsert $ylabels $i $label]

  set values [assertlength [$self slot values] $ny]
  $self slot values [linsert $values $i $newvalues]

  $self Redisplay 1
}

defmethod Matrix InsertColumn {i label {newlist {}}} {

  set xlabels [$self slot xlabels]
  set nx [llength $xlabels]
  if { $i < 0 || $i > $nx } {
    return
  }
  set ny [llength [$self slot ylabels]]

  set creators {}
  foreach creator [$self slot creators] {
    set range [split [lindex $creator 0] "," ]
    set x [lindex $range 0]
    set y [lindex $range 1]
    if { $x != "*" && $i <= $x } {
      incr x
    }
    lappend creators [concat [list $x,$y] [lrange $creator 1 end]]
  }
  $self slot creators $creators

  set width {}
  foreach wid [$self slot width] {
    set x [lindex $wid 0]
    if { $i == $x } {
      continue
    }
    if { $x != "label" && $x != "*" && $i <= $x } {
      lappend width [concat [expr $x+1] [lrange $wid 1 end]]
    } {
      lappend width $wid
    }
  }
  $self slot width $width
  
  $self slot xlabels [linsert $xlabels $i $label]

  # make rectangular
  set values {}
  foreach val [$self slot values] {
    lappend values [assertlength $val $nx]
  }
  set empty [assertlength {} $ny]
  for {set n [expr $ny-[llength $values]]} {$n > 0} {incr n -1} {
    lappend values $empty
  }

  # add new values
  set newvalues {}
  set n 0
  foreach list $values {
    lappend newvalues [linsert $list $i [lindex $newlist $n]]
    incr n
  }
  $self slot values $newvalues

  $self Redisplay 1
}

#---------------------------------------------------------------------------

defmethod Matrix get {args} {

  args	row column

  set xlabels [$self slot xlabels]
  set ylabels [$self slot ylabels]
  set ncol [llength $xlabels]
  set nrow [llength $ylabels]

  if { $row == "label" } {
    if { $column == {} } {
      return $xlabels
    } {
      return [lindex $xlabels $column]
    }
  }
  if { $column == "label" } {
    if { $row == {} } {
      return $ylabels
    } {
      return [lindex $ylabels $row]
    }
  }
  if { $row == {} && $column == {} } {
    # return all values: {ncol nrow ROW_0 ... ROW_N} (each row as a list)
    set list [list $ncol $nrow]
    for {set row 0} {$row < $nrow} {incr row} {
      set sublist {}
      for {set column 0} {$column < $ncol} {incr column} {
puts stdout "$column,$row {[$self slot $column,$row]}"
        lappend sublist [[$self slot $column,$row] get]
      }
      lappend list $sublist
    }
    return $list
  }
  if { $row == {} } {
    # return specific column: {nrow ELEM_0 .. ELEM_N}
    set list [list $nrow]
    for {set row 0} {$row < $nrow} {incr row} {
      lappend list [[$self slot $column,$row] get]
    }
    return $list
  }
  if { $column == {} } {
    # return specific row: {ncol ELEM_0 .. EbLEM_N}
    set list [list $ncol]
    for {set column 0} {$column < $ncol} {incr column} {
      lappend list [[$self slot $column,$row] get]
    }
    return $list
  }
  # return specific cell: ELEM
  return [[$self slot $column,$row] get]
}

defmethod Matrix set {args} {

  args	row column values

  set xlabels [$self slot xlabels]
  set ylabels [$self slot ylabels]
  set ncol [llength $xlabels]
  set nrow [llength $ylabels]

  if { $row == "label" } {
    if { $column == {} } {
      # set all elements of the label row (xlabels)
      set values [assertlength $values $ncol]
      for {set i 0} {$i < $ncol} {incr i} {
	[$self slot x$i] set [lindex $values $i]
      }
      $self slot xlabels $values
      return
    } {
      # set an element of the label row (xlabels)
      $self slot xlabels [lreplace $xlabels $column $column $values]
      [$self slot x$column] set $values
      return
    }
  }
  if { $column == "label" } {
    if { $row == {} } {
      # set all elements of the label column (ylabels)
      set values [assertlength $values $nrow]
      for {set i 0} {$i < $nrow} {incr i} {
	[$self slot y$i] set [lindex $values $i]
      }
      $self slot ylabels $values
      return
    } {
      # set an element of the label column (ylabels)
      $self slot ylabels [lreplace $ylabels $row $row $values]
      [$self slot y$row] set $values
      return
    }
  }
  if { $row == {} && $column == {} } {
    # set all elements of the matrix
    for {set row 0} {$row < $nrow} {incr row} {
      set sublist [lindex $values $row]
      for {set column 0} {$column < $ncol} {incr column} {
        [$self slot $column,$row] set [lindex $sublist $column]
      }
    }
    $self slot values $values
    return
  }
  if { $row == {} } {
    # set elements of a specific column
    set allvalues [$self slot values]
    set newvalues {}
    for {set row 0} {$row < $nrow} {incr row} {
      set rowvalues [assertlength [lindex $allvalues $row] $ncol]
      set newvalue [lindex $values $row]
      lappend newvalues [lreplace $rowvalues $column $column $newvalue]
      [$self slot $column,$row] set $newvalue
    }
    $self slot values $newvalues
    return
  }
  if { $column == {} } {
    # set elements of a specific row
    set allvalues [assertlength [$self slot values] $nrow]
    for {set column 0} {$column < $ncol} {incr column} {
      [$self slot $column,$row] set [lindex $values $column]
    }
    $self slot values [lreplace $allvalues $row $row $values]
    return
  }
  # set a specific cell
  set allvalues [assertlength [$self slot values] $nrow]
  set rowvalues [assertlength [lindex $allvalues $row] $ncol]
  set rowvalues [lreplace $rowvalues $column $column $values]
  $self slot values [lreplace $allvalues $row $row $rowvalues]
  [$self slot $column,$row] set $values
  return $values
}

#---------------------------------------------------------------------------

Window addDemo Matrix

defmethod Matrix demo {} {

  set t [Toplevel new * -resizable true]
  Matrix new $t.m \
	-layout {left fill expand} \
	-xlabels {Depth Comm line fill blt text arc cmplx xstones} \
	-ylabels {{Tatung 10/30 GX} {Sun 10/30 GX} {DecSt 5000/125 TX}
		  {Hyundai 10/30 Parallax} {DecSt 5000/200 PMAX}
		  {Sun 3/50 Standard}
		  {Tatung 10/30 GX} {Sun 10/30 GX} {DecSt 5000/125 TX}
		  {Hyundai 10/30 Parallax} {DecSt 5000/200 PMAX}
		  {Sun 3/50 Standard}} \
	-values {
		{8 socket 234208 46160 53773 586094 3542358 84379 102594}
		{8 socket 241948 46604 50668 596063 2775012 85359 101261}
		{8 socket 157243 16671 23059 160187 1904849 56993  42370}
		{8 socket 138048 17475 21156  42968 1872424 33529  33165}
		{8 socket 143070 17884  8395 224812 2174935 21437  25923}
		{1 socket  10000 10000 10000  10000   10000 10000  10000}
		{8 socket 234208 46160 53773 586094 3542358 84379 102594}
		{8 socket 241948 46604 50668 596063 2775012 85359 101261}
		{8 socket 157243 16671 23059 160187 1904849 56993  42370}
		{8 socket 138048 17475 21156  42968 1872424 33529  33165}
		{8 socket 143070 17884  8395 224812 2174935 21437  25923}
		{1 socket  10000 10000 10000  10000   10000 10000  10000}
		} \
	-creators {{0,* Entry -filter {_Filter allowNumber}} \
		   {1,* Entry} \
		   {2,* {} -fill red -justify right} \
		   {*,* Flatlabel -anchor e}} \
	-width {{label 22} {0 5} {1 6} {* 7}} \
	-grid true \
	-action "- showAction"

  Button new *$t -layout bottom -text "Dismiss" \
	-action "$t Dismiss"
  Button new *$t -layout bottom -text "Redisplay" \
	-action "$t.m slot _ok 0; $t.m Redisplay"
  Button new *$t -layout bottom -text "Delete Row 2" \
	-action "$t.m DeleteRow 2"
  Button new *$t -layout bottom -text "Delete Col 2" \
	-action "$t.m DeleteColumn 2"
  Button new *$t -layout bottom -text "Insert Row 2" \
	-action "$t.m InsertRow 2 \[anon\]"
  Button new *$t -layout bottom -text "Insert Col 2" \
	-action "$t.m InsertColumn 2 \[anon\] {a b c d}"
  Button new *$t -layout bottom -text "1,2 Set" \
	-action "$t.m set -column 1 -row 2 -values New"
  Button new *$t -layout bottom -text "*,2 Set" \
	-action "$t.m set -row 2 -values {a b c d e f g h i j}"
  Button new *$t -layout bottom -text "1,* Set" \
	-action "$t.m set -column 1 -values {r s t u v w x y z}"
  Button new *$t -layout bottom -text "*,* Set" \
	-action "$t.m set -values {{a b c} {d e f} {g i j k i u}}"
  Button new *$t -layout bottom -text "1,2 Get" \
	-action "puts stdout \[$t.m get -column 1 -row 2\]"
  Button new *$t -layout bottom -text "*,2 Get" \
	-action "puts stdout \[$t.m get -row 2\]"
  Button new *$t -layout bottom -text "1,* Get" \
	-action "puts stdout \[$t.m get -column 1\]"
  Button new *$t -layout bottom -text "*,* Get" \
	-action "puts stdout \[$t.m get\]"

  $t layout center
}
