#---------------------------------------------------------------------------
#
#	Network Drawing Widget
#
#	Juergen Wagner, FhG-IAO Stuttgart, May 1993
#
#---------------------------------------------------------------------------

defwidget Network Window {
	{width 1000}
	{height 750}
	{cw 300}
	{ch 300}
	{modes {}}
	{linecolor #606060}
	{nodecolor #b2c7ec}
	{areacolor #e0e8f0}
	{linkcolor black}
}

defmethod Network new {name args} {

  args	layout actions embedded text buttons title titlefont readonly \
	{grid 1} {linksfollow false} data links

  set actions [$self buildActions $actions $name]

  # Add default help action
  if { [assoc Help $actions] == {} } {
    lappend actions [list Help {} Library/network]
  }

  # Create toplevel window
  if { $embedded == "true" } {
    Frame new $name -relief flat
  } {
    Toplevel new $name \
	-title $text \
	-resizable true \
	-buttons $buttons \
	-handler [list $name _do] \
	-actions $actions
    Inputline new $name.name -layout {bottom padx 20 pady 20} \
	-label "Text: " -width 24
  }
  defsuper $name Network

  $name slot grid $grid
  $name slot linksfollow [expr {$linksfollow == "true"}]
  $name slot readonly [expr {$readonly == "true"}]
  $name slot action [lrange [assoc Double $actions] 2 end]

  set cw [$self slot cw]
  set ch [$self slot ch]

  Canvas new $name.c -layout {right padx 10 pady 10 expand fill} \
	-width $cw -height $ch \
	-scroll [list [$self slot width] [$self slot height]] \
	-title $title -titlefont $titlefont

  $name slot canvas [set canvas [$name.c canvas]]
  $name slot mode Pointer

  bind $canvas <Double-1> [list $name _event double %x %y]
  if { $readonly != "true" } {
    bind $canvas <1> [list $name _event start %x %y]
    bind $canvas <Button1-Motion> [list $name _event move %x %y]
    bind $canvas <ButtonRelease-1> [list $name _event stop %x %y]
  }

  $name load $data $links
  $name layout $layout
}

defmethod Network _do {button action} {
  global system

  case $button {
  {Help Dismiss Delete Link Unlink} {
	$self $button $action
    }
  {Print} {
	[$self.c canvas] print -printer $system(printer) -colormode gray
    }
  {Set} {
	if { $action == {} } {
	  set sel [$self slot selected]
	  if { $sel != {} } {
	    $self setText $sel [$self.name get]
	  }
	} {
	  uplevel #0 $action
	}
    }
  default {
	if { [position $button [Network slot modes]] >= 0 } {
	  $self setMode $button $action
	} {
	  uplevel #0 $action
	}
    }
  }
}

defmethod Network getText {id} {

  set descr [$self slot id$id]
  if { $descr != {} } {
    $self _get[assoc type $descr] [$self slot canvas] $id $text
  }
  return $id
}

defmethod Network setText {id text} {

  set descr [$self slot id$id]
  if { $descr != {} } {
    $self _edit[assoc type $descr] [$self slot canvas] $id $text
  }
  return $id
}

defmethod Network setMode {mode {action {}}} {

  if { [position $mode [Network slot modes]] >= 0 } {
    $self slot mode $mode
    if { $action != {} } {
      update #0 $action
    }
    return $mode
  }
}

#---------------------------------------------------------------------------
#
# Link:
#	A link connects a node with another node or line. Links are drawn
#	horizontal or vertical if possible. Otherwise, they are straight
#	lines from one object to another. Unfortunately, links have to know
#	about some objects in order to connect them correctly.
#

defmethod Network Link {action} {

  set sel1 [$self slot selected]
  set sel2 [$self slot selected2]
  if { $sel1 == {} || $sel2 == {} } {
    return
  }

  # Create a link from $sel1 to $sel2. This has to take into account the
  # different spatial extensions of objects and their reference points.
  $self doLink $sel1 $sel2

  if { $action != {} } {
    uplevel #0 $action
  }
}

defmethod Network updateLinks {id} {

  foreach spec [$self slot links] {
    if { [lindex $spec 0] == $id } {
      $self doLink $id [lindex $spec 1]
    } elseif { [lindex $spec 1] == $id } {
      $self doLink [lindex $spec 0] $id
    }
  }
  return
}

defmethod Network doLink {obj1 obj2} {

  # Ensure obj1 id is < obj2 id
  if { $obj1 > $obj2 } {
    set tmp $obj1; set obj1 $obj2; set obj2 $tmp
  }
  if { $obj1 == $obj2 } {
    return
  }

  # Ensure we have the object id of the group leaders
  set obj1spec [$self slot id$obj1]
  set obj2spec [$self slot id$obj2]
  if { $obj1spec == {} || $obj2spec == {} } {
    return
  }

  set canvas [$self slot canvas]

  # Check for existing link
  set id {}
  foreach spec [$self slot links] {
    if { [lindex $spec 0] == $obj1 && [lindex $spec 1] == $obj2 } {
      set id [lindex $spec 2]
      break
    }
  }

  # Determine coordinates of the group leader objects
  #
  set obj1type [assoc type $obj1spec]
  set obj2type [assoc type $obj2spec]

  set coords [$self _reference$obj1type $canvas $obj1]
  set x1o1 [lindex $coords 0]
  set y1o1 [lindex $coords 1]
  set x2o1 [lindex $coords 2]
  set y2o1 [lindex $coords 3]

  set coords [$self _reference$obj2type $canvas $obj2]
  set x1o2 [lindex $coords 0]
  set y1o2 [lindex $coords 1]
  set x2o2 [lindex $coords 2]
  set y2o2 [lindex $coords 3]

  # Determine relative positions in x direction
  #
  if { $x2o2 < $x1o1 } {
    # Disjoint: obj2 is left of obj1
    set x1 $x1o1
    set x2 $x2o2
  } elseif { $x2o1 < $x1o2 } {
    # Disjoint: obj2 is right of obj1
    set x1 $x2o1
    set x2 $x1o2
  } else {
    # Non-zero intersection: take center of intersection area
    set m1 [expr {($x1o1 < $x1o2) ? $x1o2 : $x1o1}]
    set m2 [expr {($x2o1 > $x2o2) ? $x2o2 : $x2o1}]
    set x1 [expr ($m1+$m2)/2]
    set x2 $x1
  }

  # Determine relative positions in x direction

  if { $y2o2 < $y1o1 } {
    # Disjoint: obj2 is left of obj1
    set y1 $y1o1
    set y2 $y2o2
  } elseif { $y2o1 < $y1o2 } {
    # Disjoint: obj2 is right of obj1
    set y1 $y2o1
    set y2 $y1o2
  } else {
    # Non-zero intersection: take center of intersection area
    set m1 [expr {($y1o1 < $y1o2) ? $y1o2 : $y1o1}]
    set m2 [expr {($y2o1 > $y2o2) ? $y2o2 : $y2o1}]
    set y1 [expr ($m1+$m2)/2]
    set y2 $y1
  }

  set coords [$self _connect$obj1type $canvas $obj1 $x1 $y1 $x2 $y2]
  set x1 [lindex $coords 0]
  set y1 [lindex $coords 1]

  set coords [$self _connect$obj2type $canvas $obj2 $x2 $y2 $x1 $y1]
  set x2 [lindex $coords 0]
  set y2 [lindex $coords 1]

  if { $id == {} } {
    set id [$canvas create line $x1 $y1 $x2 $y2 -tags NetworkLink \
		-width 1 -fill [Color pick black [Network slot linkcolor]]]
    $self slotappend links [list $obj1 $obj2 $id]
  } {
    $canvas coords $id $x1 $y1 $x2 $y2
  }

  return $id
}

defmethod Network Unlink {action} {

  set sel1 [$self slot selected]
  set sel2 [$self slot selected2]
  if { $sel1 == {} || $sel2 == {} } {
    return
  }

  # Create a link from $sel1 to $sel2. This has to take into account the
  # different spatial extensions of objects and their reference points.
  $self doUnlink $sel1 $sel2

  if { $action != {} } {
    uplevel #0 $action
  }
}

defmethod Network doUnlink {obj1 obj2} {

  # Ensure obj1 id is < obj2 id
  if { $obj1 > $obj2 } {
    set tmp $obj1; set obj1 $obj2; set obj2 $tmp
  }
  if { $obj1 == $obj2 } {
    return
  }

  # Ensure we have the object id of the group leaders
  set obj1spec [$self slot id$obj1]
  set obj2spec [$self slot id$obj2]
  if { $obj1spec == {} || $obj2spec == {} } {
    return
  }

  set canvas [$self slot canvas]

  set links [$self slot links]
  set i 0
  foreach link $links {
    if { [lindex $link 0] == $obj1 && [lindex $link 1] == $obj2 } {
      $canvas delete [lindex $link 2]
      $self slot links [lreplace $links $i $i]
      break
    }
    incr i
  } 
}

#---------------------------------------------------------------------------

defmethod Network Delete {action} {

  $self deleteItem [$self slot selected]
  if { $action != {} } {
    uplevel #0 $action
  }
}

defmethod Network deleteItem {id} {

  if { $id != {} && [$self slot id$id] != {} } {
    set canvas [$self slot canvas]
    eval [concat [list $canvas delete $id] [assoc other [$self slot id$id]]]
    $self unslot id$id
    set ids [$self slot ids]
    set pos [lsearch $ids $id]
    if { $pos >= 0 } {
      $self slot ids [lreplace $ids $pos $pos]
    }
    set links {}
    foreach link [$self slot links] {
      if { $id != [lindex $link 0] && $id != [lindex $link 1] } {
	lappend links $link
      } {
	$canvas delete [lindex $link 2]
      }
    }
    $self slot links $links
    $self _select $canvas
  }
}

#---------------------------------------------------------------------------
#
# Event handling:
#
#	$self _start$mode $canvas $x1 $y1 $x2 $y2 => ID
#		Initialization. Return id of created object.
#	$self _move$mode $canvas $id $x1 $y1 $x2 $y2 => VOID
#		Dynamic moving or resizing.
#	$self _stop$mode $canvas $id $x1 $y1 $x2 $y2 => DESCR
#		Fix the object in this position.
#	$self _edit$mode $canvas $id $text
#

defmethod Network _event {event x y} {

  set canvas [$self slot canvas]
  set grid [$self slot grid]
  set x [expr ([$canvas canvasx $x]+$grid/2)/$grid*$grid]
  set y [expr ([$canvas canvasy $y]+$grid/2)/$grid*$grid]
  if { $x < 0 || $y < 0 } {
    return
  }
  $self slot x $x
  $self slot y $y

  case $event {
  {double start} {
	# Register starting coordinates
	$self slot x0 [$self slot x1 [$self slot x2 $x]]
	$self slot y0 [$self slot y1 [$self slot y2 $y]]
	$self slot lastx {}
	$self slot lasty {}
	# Store id of group leader and current mode
	if { $event == "double" } {
	  set mode Pointer
	} {
	  $self slot curmode [set mode [$self slot mode]]
	}
	set id [$self _start$mode $canvas $x $y $x $y]
	$self slot id $id
	if { $event == "double" } {
	  set action [$self slot action]
	  if { $action != {} } {
	    uplevel #0 [concat $action [list $canvas [$self slot id$id]]]
	  }
	}
    }
  {move} {
	# Keep one point fixed, the other moves
	set y0 [$self slot y0]
	set x0 [$self slot x0]
	if { $x0 < $x } {
	  set x1 $x0; set x2 $x
	} {
	  set x1 $x; set x2 $x0
	}
	if { $y0 < $y } {
	  set y1 $y0; set y2 $y
	} {
	  set y1 $y; set y2 $y0
	}
	# Update position/size of current item
	$self slot x1 $x1
	$self slot y1 $y1
	$self slot x2 $x2
	$self slot y2 $y2
	set id [$self slot id]
	$self _move[$self slot curmode] $canvas $id $x1 $y1 $x2 $y2
	if { [$self slot linksfollow] && 
	     ($x != [$self slot lastx] || $y != [$self slot lasty]) } {
	  $self updateLinks $id
	}
	$self slot lastx $x
	$self slot lasty $y
    }
  {stop} {
	set mode [$self slot curmode]
	if { $mode == {} } {
	  return
	}
	if { $mode == "Pointer" } {
	  $self updateLinks [$self slot id]
	  return
	}
	# Get specification of group:
	#   {Leader {type Type} {other OtherIds} Attributes...}
	set x1 [$self slot x1]
	set y1 [$self slot y1]
	set x2 [$self slot x2]
	set y2 [$self slot y2]
	set spec [$self _$event$mode $canvas [$self slot id] $x1 $y1 $x2 $y2]
	# No object was created, remove marker
	if { $spec == {} } {
	  $self _select $canvas
	  return
	}
	set id [lindex $spec 0]
	$self register $canvas $id [lrange $spec 1 end]
	# Update all links to the new item
	$self updateLinks $id
	# Select the new item
	$self _select $canvas $id
    }
  }
}

defmethod Network register {canvas id other} {

  # Register all new ids in a slot "ids"
  if { [position $id [$self slot ids]] < 0 } {
    $self slotappend ids $id
  }
  $self slot id$id $other
  $canvas addtag Network withtag $id
  $canvas addtag [list NetworkId $id] withtag $id
  foreach oid [assoc other $other] {
    $canvas addtag Network withtag $oid
    $canvas addtag [list NetworkId $id] withtag $oid
  }
}

defmethod Network _select {canvas {id {}}} {

  set alert [Color slot bg,alert]
  set alert2 [Color slot bg,alert2]

  $canvas delete NetworkMarker NetworkMarker2
  if { $id == {} } {
    $self slot selected {}
    $self slot selected2 {}
    return
  }

  set oid [$self slot selected]
  if { $oid != {} && $oid != $id } {
    set coords [$self _reference[assoc type [$self slot id$oid]] $canvas $oid]
    set x1 [lindex $coords 0]
    set y1 [lindex $coords 1]
    set x2 [lindex $coords 2]
    set y2 [lindex $coords 3]

    $canvas create rectangle \
	[expr $x1-4] [expr $y1-4] [expr $x1+4] [expr $y1+4] \
	-fill $alert2 -outline black -width 1 \
	-tags {Network NetworkMarker2}
    $canvas create rectangle \
	[expr $x2-4] [expr $y2-4] [expr $x2+4] [expr $y2+4] \
	-fill $alert2 -outline black -width 1 \
	-tags {Network NetworkMarker2}

    $self slot selected2 $oid
  }

  set coords [$self _reference[assoc type [$self slot id$id]] $canvas $id]
  set x1 [lindex $coords 0]
  set y1 [lindex $coords 1]
  set x2 [lindex $coords 2]
  set y2 [lindex $coords 3]

  $canvas create rectangle \
	[expr $x1-4] [expr $y1-4] [expr $x1+4] [expr $y1+4] \
	-fill $alert -outline black -width 1 \
	-tags [list Network NetworkMarker [list NetworkOther 2]]
  $canvas create rectangle \
	[expr $x2-4] [expr $y2-4] [expr $x2+4] [expr $y2+4] \
	-fill $alert -outline black -width 1 \
	-tags [list Network NetworkMarker [list NetworkOther 0]]

  $self slot selected $id
}

#---------------------------------------------------------------------------
#
# Pointer:
#	The depression of the left mouse button over an object selects that
#	object and allows the user to move it. Selecting one of the reference
#	points of an object resizes it.
#
Network slotappend modes Pointer

defmethod Network _startPointer {canvas x1 y1 x2 y2} {

  set overlap [$canvas find overlapping $x1 $y1 $x2 $y2]

  for {set id [$canvas find withtag current]} {$id != {}} \
	{set id [$canvas find below $id]} {
    set alltags [$canvas gettags $id]
    set other [assoc NetworkOther $alltags]
    if { $other != {} } {
      set id [$self slot selected]
      set type [assoc type [$self slot id$id]]
      $self slot curmode $type
      set coords [$self _reference$type $canvas $id]
      $self slot x0 [lindex $coords $other]
      $self slot y0 [lindex $coords [expr $other+1]]
      return $id
    }
    set other [assoc NetworkId $alltags]
    if { $other != {} } {
      set id $other      
    }
    if { [$self slot id$id] != {} } {
      $self _select $canvas $id
      return $id
    }
  }

  $self _select $canvas
  return
}

defmethod Network _movePointer {canvas id x1 y1 x2 y2} {

  if { $id == {} } {
    return
  }
  set x0 [$self slot x0]
  set y0 [$self slot y0]
  set x [$self slot x]
  set y [$self slot y]
  $self slot x0 $x
  $self slot y0 $y

  $canvas move [list NetworkId $id] [expr $x-$x0] [expr $y-$y0]
  $canvas move NetworkMarker [expr $x-$x0] [expr $y-$y0]
}

#---------------------------------------------------------------------------
#
# Line:
#	A line is drawn from a starting point to an ending point. Only
#	horizontal and vertical lines are allowed.
#
Network slotappend modes Line

defmethod Network _referenceLine {canvas id} {

  $canvas coords $id
}

defmethod Network _connectLine {canvas id x y x2 y2} {

  return [list $x $y]
}

defmethod Network _startLine {canvas x1 y1 x2 y2} {

  set color [Color pick black [Network slot linecolor]]
  set id [$canvas create line $x1 $y1 $x2 $y2 -width 3 -fill $color]
  $canvas create rectangle \
	[expr $x1-3] [expr $y1-3] [expr $x1+3] [expr $y1+3] \
	-fill $color
  $canvas create rectangle \
	[expr $x2-3] [expr $y2-3] [expr $x2+3] [expr $y2+3] \
	-fill $color
  return $id
}

defmethod Network _moveLine {canvas id x1 y1 x2 y2} {

  set x0 [$self slot x0]
  set y0 [$self slot y0]

  if { [expr $x2-$x1] < [expr $y2-$y1] } {
    set x1 $x0
    set x2 $x0
  } {
    set y1 $y0
    set y2 $y0
  }
  $canvas coords $id $x1 $y1 $x2 $y2
  $canvas coords [expr $id+1] \
	[expr $x1-3] [expr $y1-3] [expr $x1+3] [expr $y1+3]
  $canvas coords [expr $id+2] \
	[expr $x2-3] [expr $y2-3] [expr $x2+3] [expr $y2+3]
}

defmethod Network _stopLine {canvas id x1 y1 x2 y2} {

  set x0 [$self slot x0]
  set y0 [$self slot y0]

  if { [expr $x2-$x1] < [expr $y2-$y1] } {
    set x1 $x0
    set x2 $x0
  } {
    set y1 $y0
    set y2 $y0
  }

  if { $x1 > [expr $x2-50] && $y1 > [expr $y2-50] } {
    $canvas delete $id [expr $id+1] [expr $id+2]
    return
  }
  return [list $id {type Line} [list other [expr $id+1] [expr $id+2]]]
}

defmethod Network _editLine {canvas id text} {
}

defmethod Network _getLine {canvas id} {
}

#---------------------------------------------------------------------------
#
# Area:
#	A rectangular area is drawn.
#
Network slotappend modes Area

defmethod Network _referenceArea {canvas id} {

  $canvas coords $id
}

defmethod Network _connectArea {canvas id x y x2 y2} {

  return [list $x $y]
}

defmethod Network _startArea {canvas x1 y1 x2 y2} {

  set id [$canvas create rectangle $x1 $y1 $x2 $y2 \
		-width 1 -fill [Color pick white [Network slot areacolor]]]
  $canvas create text [expr $x2-2] [expr $y2-2] \
		-anchor se -text "Area" -font [Font slot smallbold]
  catch {$canvas lower $id Network}
  return $id
}

defmethod Network _moveArea {canvas id x1 y1 x2 y2} {

  $canvas coords $id $x1 $y1 $x2 $y2
  $canvas coords [expr $id+1] [expr $x2-2] [expr $y2-2]
}

defmethod Network _stopArea {canvas id x1 y1 x2 y2} {

  if { [expr $x2-$x1] < 50 || [expr $y2-$y1] < 50 } {
    $canvas delete $id [expr $id+1]
    return
  }
  return [list $id {type Area} [list other [expr $id+1]]]
}

defmethod Network _editArea {canvas id text} {

  $canvas itemconfigure [assoc other [$self slot id$id]] -text $text
}

defmethod Network _getArea {canvas id} {

  lindex [$canvas itemconfigure [assoc other [$self slot id$id]] -text] 4
}

#---------------------------------------------------------------------------
#
# Oval:
#	Ovals can be connected to other Ovals or to lines. Note that without
#	trigonometric function support, the reference and connection points
#	for ovals will be identical. With atan/sin/cos available, lines will
#	connect to the line enclosing the oval.
#
Network slotappend modes Oval

defmethod Network _referenceOval {canvas id} {

  set coords [$canvas coords $id]
  set x [expr ([lindex $coords 0]+[lindex $coords 2])/2]
  set y [expr ([lindex $coords 1]+[lindex $coords 3])/2]

  return [list $x $y $x $y]
}

defmethod Network _connectOval {canvas id x y x2 y2} {

  if { [catch {expr sin(0)}] } {
    return [list $x $y]
  }

  set coords [$canvas coords $id]
  set rx [expr ([lindex $coords 2]-[lindex $coords 0])/2]
  set ry [expr ([lindex $coords 3]-[lindex $coords 1])/2]

  if { $x == $x2 } {
    if { $y < $y2 } {
      return [list $x [expr $y+$ry]]
    } {
      return [list $x [expr $y-$ry]]
    }
  }

  set phi [expr atan(1.0*($y2-$y)/($x2-$x))]

  if { $x < $x2 } {
    set x [expr $x+$rx*cos($phi)]
    set y [expr $y+$ry*sin($phi)]
  } {
    set x [expr $x-$rx*cos($phi)]
    set y [expr $y-$ry*sin($phi)]
  }

  return [list $x $y]
}

defmethod Network _startOval {canvas x1 y1 x2 y2} {

  $canvas create oval [expr $x1-10] [expr $y1-10] [expr $x1+10] [expr $y1+10] \
	-fill [Color pick white [Network slot nodecolor]]
}

defmethod Network _moveOval {canvas id x1 y1 x2 y2} {

  set x [$self slot x]
  set y [$self slot y]

  $canvas coords $id [expr $x-10] [expr $y-10] [expr $x+10] [expr $y+10]
}

defmethod Network _stopOval {canvas id x1 y1 x2 y2} {

  return [list $id {type Oval}]
}

defmethod Network _editOval {canvas id text} {
}

defmethod Network _getOval {canvas id} {
}

#---------------------------------------------------------------------------

Network slotappend modes Block

defmethod Network _referenceBlock {canvas id} {

  $canvas coords $id
}

defmethod Network _connectBlock {canvas id x y x2 y2} {

  return [list $x $y]
}

defmethod Network _startBlock {canvas x1 y1 x2 y2} {

  set id [$canvas create rectangle $x1 $y1 [expr $x1+80] [expr $y1+20] \
		-fill [Color pick white [Network slot nodecolor]]]
  $canvas create text [expr $x1+40] [expr $y1+10] \
		-font [Font slot small] -anchor c -text "Node"
  return $id
}

defmethod Network _moveBlock {canvas id x1 y1 x2 y2} {

  set x [$self slot x]
  set y [$self slot y]

  $canvas coords $id $x $y [expr $x+80] [expr $y+20]
  $canvas coords [expr $id+1] [expr $x+40] [expr $y+10]
}

defmethod Network _stopBlock {canvas id x1 y1 x2 y2} {

  return [list $id {type Block} [list other [expr $id+1]]]
}

defmethod Network _editBlock {canvas id text} {

  $canvas itemconfigure [assoc other [$self slot id$id]] -text $text
}

defmethod Network _getBlock {canvas id} {

  lindex [$canvas itemconfigure [assoc other [$self slot id$id]] -text] 4
}

#---------------------------------------------------------------------------

Network slotappend modes Text

defmethod Network _referenceText {canvas id} {

  set coords [$canvas coords $id]
  set x [lindex $coords 0]
  set y [lindex $coords 1]

  return [list $x $y $x $y]
}

defmethod Network _connectText {canvas id x y x2 y2} {

  return [list $x $y]
}

defmethod Network _startText {canvas x1 y1 x2 y2} {

  $canvas create text $x1 $y1 -font [Font slot small] -anchor c -text "Text"
}

defmethod Network _moveText {canvas id x1 y1 x2 y2} {

  set x [$self slot x]
  set y [$self slot y]

  $canvas coords $id $x $y
}

defmethod Network _stopText {canvas id x1 y1 x2 y2} {

  return [list $id {type Text}]
}

defmethod Network _editText {canvas id text} {

  $canvas itemconfigure $id -text $text
}

defmethod Network _getText {canvas id} {

  lindex [$canvas itemconfigure $id -text] 4
}

#---------------------------------------------------------------------------

defmethod Network Save {action} {

  set canvas [$self slot canvas]

  set data {}
  set i 0
  foreach id [$self slot ids] {
    set descr [$self slot id$id]
    set type [assoc type $descr]
    lappend data [list \
	$type [$self _get$type $canvas $id] \
	[$self _reference$type $canvas $id] \
	]
    set pos($id) $i
    incr i
  }

  set links {}
  foreach pair [$self slot links] {
    lappend links [list $pos([lindex $pair 0]) $pos([lindex $pair 1])]
  }

  if { $action != {} } {
    uplevel #0 [concat $action [list $data $links]]
  }
  return [list $data $links]
}

defmethod Network load {data links} {

  set canvas [$self slot canvas]
  set objects {}

  foreach item $data {
    set type [lindex $item 0]
    set id [eval [concat [list $self _start$type $canvas] [lindex $item 2]]]
    set spec [eval [concat [list $self _stop$type $canvas $id] \
			   [lindex $item 2]]]
    $self register $canvas $id $spec
    $self _edit$type $canvas $id [lindex $item 1]
    lappend objects $id
  }

  foreach item $links {
    $self doLink [lindex $objects [lindex $item 0]] \
	[lindex $objects [lindex $item 1]]
  }
}

defmethod Network clear {} {

  [$self slot canvas] delete Network NetworkLink
}

#---------------------------------------------------------------------------

Window addDemo Network

defmethod Network demo {} {

  set modes [Network slot modes]

  set t [anon]
  Network new $t \
	-grid 10 \
	-layout center \
	-embedded false \
	-buttons [concat $modes {Set Link Unlink Delete Transfer Print}] \
	-text "Network Editor" \
	-actions [concat $modes {Set Link Delete Unlink Print
		{Transfer {} - demoTransfer} {Double {} - showAction}}] \
	-data {
		{Line {} {100 40 100 340}}
		{Line {} {180 70 510 70}}
		{Oval {} {140 70 140 70}} 
		{Block Oropax {180 150 260 170}} 
		{Block Yen {300 150 380 170}} 
		{Block Linus {420 150 500 170}} 
		{Block Woodstock {240 210 320 230}} 
		{Block Solarix {360 210 440 230}} 
		{Area {Network Room 011} {70 20 560 350}}
	} \
	-links {{1 2} {0 2} {1 3} {1 4} {1 5} {1 7} {1 6}}

  Network new $t.n2 \
	-grid 10 \
	-layout 300x300-0-0 \
	-embedded false \
	-readonly true \
	-text "Network Display" \
	-actions {{Double {} - showAction Double}}
}

defmethod Network demoTransfer {} {

  set result [$self Save {}]
  set data [lindex $result 0]
  set links [lindex $result 1]
  $self.n2 clear
  $self.n2 load $data $links
}
