/*
 *=============================================================================
 *                                tSippInt.h
 *-----------------------------------------------------------------------------
 * Internal, global declarations for Tcl SIPP interface.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippInt.h,v 4.0 1993/11/27 21:28:57 markd Rel $
 *=============================================================================
 */

#include <malloc.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <memory.h>
#include <values.h>
#include <time.h>
#include <sys/types.h>

char*
ctime ();

#include "tclExtend.h"
#include "sipp.h"
#include "shaders.h"
#include "primitives.h"
#include "smalloc.h"
#include "xalloca.h"

/*
 * Version suffix for tsipp.  It is append to the SIPP version to identify
 * the version number for tsipp.  Also a variable that will contain the
 * full version string, including patch level.
 */
#define TSIPP_VERSION_SUFFIX "c"

extern char *TSIPP_VERSION;

/*
 * This is the maximum number of characters a floating point number can
 * occupy when formatted out to the maximum precision.  It includes the
 * '\0' byte.  It may have to be changed on some machines.
 */
#define TSIPP_DOUBLE_STR_SIZE 32

/*
 * Maximum number of image storage class that are supported.
 */
#define TSIPP_MAX_STORAGE_CLASSES  4

/*
 * Global Tcl SIPP structure.  It is the client data to all commands.
 */

typedef struct {
    Tcl_Interp    *interp;           /* Tcl interpreter pointer.             */
    bool           showBackFaces;    /* Render or draw backfacing polygons.  */
    bool           rendering;        /* Rendering in progress.               */
    Color          backgroundColor;  /* Current background color.            */
    Color          lineColor;        /* Color for lines in RLE LINE render   */
    void_pt        tkMainWindow;     /* Main window ptr for tksipp.          */

    /*
     * Routine called during rendering at each update period.  Under Tk, this
     * handles event updates and signals.  When not using Tk, this just handles
     * signals.  Delayed error is used when an error occurs while handling a
     * signal during rendering.  Its an error message to return at the end.
     */
    void          (*updateProc) ();
    char           *delayedError;

    /*
     * Table of image storage classes that images can be rendered to.
     * Terminated by a NULL entry.
     */
    struct tSippStorageClass_t
        *storageClassPtrs [TSIPP_MAX_STORAGE_CLASSES + 1];

    /*
     * Handle tables for the various tables we keep.
     */
    void_pt        surfaceTblPtr;    /* Surface table.            */
    void_pt        objectTblPtr;     /* Object table.             */
    void_pt        shaderTblPtr;     /* Shader table.             */
    void_pt        lightTblPtr;      /* Light table.              */
    void_pt        cameraTblPtr;     /* Camera table.             */
    void_pt        rleTblPtr;        /* RLE file table.           */
    void_pt        pbmTblPtr;        /* PBMPlus file table.       */
    void_pt        pixMapTblPtr;     /* Pixmap table.             */

    /*
     * Cleanup functions for optional modules.
     */
    void         (*rleCleanUpProc) ();  /* Cleanup RLE file table */

} tSippGlob_t, *tSippGlob_pt;

/*
 * Supported scanning direction.
 */
typedef enum {
    TSIPP_SCAN_BOTH,
    TSIPP_SCAN_TOP_DOWN,
    TSIPP_SCAN_BOTTOM_UP
} scanDirection_t;

/*
 * Used to set mode field when the mode is unknown 24 bit image.
 */
#define MODE_UNKNOWN24 255

/*
 * Structure used to describe the available image storage class.
 * The use of this data is described in detail in tSippRender.
 */

typedef struct tSippStorageClass_t {
    char             *handlePrefix;   /* Prefix identifying class handle.    */
    int               prefixSize;     /* Size of handle prefix string.       */
    scanDirection_t   scanDirection;  /* Supported scanning direction.       */
    bool              bitMapOptimal;  /* Bitmaps are optimized.              */

    void           *(*outputStart)();   /* Call at beginning of output.      */
    void            (*outputLine) ();   /* Call for each scan line.          */
    void            (*outputBitMap) (); /* Call for entire bitmap.           */
    bool            (*outputEnd) ();    /* Call at the end of output.        */
    bool            (*copyImage) ();    /* Call to copy an image.            */

} tSippStorageClass_t, *tSippStorageClass_pt;

/*
 * Information describing the image.
 */
typedef struct {
    /*
     * This describes the image being rendered and is saved with certain
     * types of image storage.
     */
    unsigned   xSize;                /* Size of image to output.      */
    unsigned   ySize;
    unsigned   mode;                 /* FLAT, GOURAUD, PHONG, LINE,   */
                                     /* MODE_PIXEL.                   */
    unsigned   overSampling;         /* Oversampling factor.          */
    int        field;                /* BOTH, ODD or EVEN scan lines. */
    u_char     backgroundColor [3];  /* Current background color.     */
    u_char     lineColor [3];        /* Color for lines in LINE mode. */

} tSippImageData_t, *tSippImageData_pt;

/*
 * Type containing data about an image that is going to be rendered or copied
 * as well as other attributes that influence rendering. This information is
 * passed to the image storage object.
 */
typedef struct {
    tSippImageData_t  imgData;        /* Image data.                         */

    char            **argv;           /* Render command, if available.       */
    int               update;         /* Update period. -1 is not specified. */
    int               maxDepth;       /* Maximum object tree depth.          */
    bool              clear;          /* Clear image before rendering.       */
    bool              bitMapOutput;   /* TRUE if a bitmap will be outputted. */
    scanDirection_t   scanDirection;  /* Scanning direction to use.          */

} tSippOutputParms_t, *tSippOutputParms_pt;

/*
 * Initial value for output parameters used to initialize the structure for
 * output.
 */
extern tSippOutputParms_t initOutputParms;

/*
 * Color indicies for the scanlines that are passed to the output object's
 * renderLine functions.
 */
#define TSIPP_RED    0
#define TSIPP_GREEN  1
#define TSIPP_BLUE   2

/*
 * Type used in initializing Tcl commands.
 */
typedef struct {
    char        *name;
    Tcl_CmdProc *proc;
} tSippTclCmdTbl_t;

/*
 * Type used to store a translated handle list.  A static array is used
 * unless the list is bigger, then it goes dynamic.
 */
#define HANDLE_LIST_STATIC_SIZE 16

typedef struct {
    void   **ptr;
    int      len;
    void    *staticArray [HANDLE_LIST_STATIC_SIZE];
} handleList_t, *handleList_pt;

/*
 * Macros to do string compares.  They pre-check the first character before
 * checking of the strings are equal.
 */

#define STREQU(str1, str2) \
    (((str1) [0] == (str2) [0]) && (strcmp (str1, str2) == 0))
#define STRNEQU(str1, str2, cnt) \
    (((str1) [0] == (str2) [0]) && (strncmp (str1, str2, cnt) == 0))

/*
 * Macro to add bytes to a pointer, no matter what the type.  Always returns
 * a void *.
 */
#define PTR_ADD(ptr, incr) ((void *) (((char *) ptr) + incr))

/*
 * Macros to handle dealing with numbers that may have encountered a round
 * off error:
 *   FZERO - Is the number zero, with in the tolerance.
 *   FZERO_ROUND - Round the number to zero, if its in the tolerance of zero.
 *   FEQUAL - Is the number equal, within the tolerance.
 *
 */
#define TOLERANCE 1e-13
#define FZERO(f) ((f) < TOLERANCE && (f) > -TOLERANCE)
#define FZERO_ROUND(f) (FZERO(f) ? 0.0 : (f))
#define FEQUAL(f1, f2) ((((f2) - TOLERANCE) < (f1)) && \
                        ((f1) < (((f2) + TOLERANCE))))

/*
 * WARNING: This is an internal Tcl data structure and function.  It might
 * change.  John needs to export this structure.????
 */
typedef struct OpenFile {
    FILE *f;
    FILE *f2;
    int readable;
    int writable;
    int numPids;
    int *pidPtr;
    int errorId;
} OpenFile;

extern int
TclGetOpenFile _ANSI_ARGS_((Tcl_Interp  *interp,
                            char        *string, 
                            OpenFile   **filePtrPtr));


/*
 * tSippUtil.c externals.
 */
char *
TSippFormatDouble  _ANSI_ARGS_((double  number,
                                char   *buffer));

bool
TSippConvertFraction _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                  char         *strBuf,
                                  double       *valuePtr));

bool
TSippConvertUnsignedDbl _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                     char         *strBuf,
                                     double       *valuePtr));

bool
TSippConvertPosUnsigned _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                     char         *strBuf,
                                     unsigned     *valuePtr));

bool
TSippConvertColor _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                               char         *listStr,
                               Color        *colorPtr));

bool
TSippConvertOpacity _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                 char         *listStr,
                                 Color        *opacityPtr));

bool
TSippConvertAngleRad _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                  char         *strBuf,
                                  double       *valuePtr));

bool
TSippConvertAngleDeg _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                  char         *strBuf,
                                  double       *valuePtr));

bool
TSippConvert2DPoint _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                 char         *listStr,
                                 double       *xCoordPtr,
                                 double       *yCoordPtr));

bool
TSippConvertVertex _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                char         *listStr,
                                Vector       *vertexPtr));

bool
TSippConvertVertexTex _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                   char         *listStr,
                                   Vector       *vertexPtr,
                                   Vector       *texturePtr));

bool
TSippConvertMatrix _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                                char         *listStr,
                                Transf_mat   *matrixPtr));

char *
TSippFormatMatrix _ANSI_ARGS_((Transf_mat   *matrixPtr));

bool
TSippHandleListConvert _ANSI_ARGS_((tSippGlob_pt     tSippGlobPtr,
                                    void            *handleTblPtr,
                                    char            *listPtr,
                                    handleList_pt    handleListPtr,
                                    handleList_pt    handleEntryListPtr,
                                    char          ***handleListArgvPtr));

void
TSippHandleListFree _ANSI_ARGS_((handleList_pt  handleListPtr));

void
TSippInitCmds _ANSI_ARGS_((tSippGlob_pt       tSippGlobPtr,
                           tSippTclCmdTbl_t  *cmdTablePtr));

void
TSippAddStorageClass _ANSI_ARGS_((tSippGlob_pt         tSippGlobPtr,
                                  tSippStorageClass_pt  storageClassPtr));

bool
TSippParseTextureMapping _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                      char           *textureStr,
                                      int            *texturePtr,
                                      int            *invalidList));

int
TSippNotWhileRendering _ANSI_ARGS_((Tcl_Interp *interp));

int
TSippAtEOF _ANSI_ARGS_((Tcl_Interp *interp,
                      char       *handle));

int
TSippNotReadable _ANSI_ARGS_((Tcl_Interp *interp,
                            char       *handle));

int
TSippNotWritable _ANSI_ARGS_((Tcl_Interp *interp,
                            char       *handle));

void
TSippPutCom _ANSI_ARGS_((char  ***commentsPtr,
                         char    *name,
                         char    *value));

char *
TSippGetCom _ANSI_ARGS_((char  **comments,
                         char    *name));

char *
TSippGetAllCom _ANSI_ARGS_((char  **comments));

void
TSippDelCom _ANSI_ARGS_((char  ***commentsPtr,
                         char    *name));

char **
TSippCopyCom _ANSI_ARGS_((char  **comments));

void
TSippMergeCom _ANSI_ARGS_((char   **oldComments,
                           char  ***newCommentsPtr));

void
TSippFreeCom _ANSI_ARGS_((char  ***commentsPtr));

void
TSippAddHist _ANSI_ARGS_((char   ***commentsPtr,
                          char    **argv));

/*
 * tSippMisc.c externals.
 */
int 
TSipp_Init _ANSI_ARGS_((Tcl_Interp *interp));

/*
 * tSippBezier.c externals.
 */
void
TSippBezierInit _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippPoly.c externals.
 */
Surface *
TSippSurfaceHandleToPtr _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                     char           *handle));

void
TSippPolyInit _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

void
TSippPolyCleanUp _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippObj.c externals.
 */
void
TSippBindObjectToHandle _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                    Object         *objectPtr));

Object *
TSippObjectHandleToPtr _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                    char           *handle));

void
TSippObjectInit _ANSI_ARGS_((tSippGlob_pt tclSippGlobP));

void
TSippObjectCleanUp _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippLight.c externals.
 */
void
TSippLightInit _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr));

void
TSippLightCleanUp _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippCamera.c externals.
 */
void
TSippCameraInit _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

void
TSippCameraCleanUp _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippShader.c externals.
 */
Shader *
TSippShaderHandleToPtr _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                    char           *handle,
                                    void          **surfDescHdrPtrPtr));

void
TSippShaderInit _ANSI_ARGS_((tSippGlob_pt  tclSippGlobP));

void
TSippShaderCleanUp _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippPrim.c externals.
 */
void
TSippPrimInit _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippPBM.c externals.
 */
void
TSippPBMInit _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr));

void
TSippPBMCleanUp _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippRLE.c externals.
 */
void
TSippRLEInit _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr));

/*
 * tSippRender.c externals.
 */
void
TSippRenderInit _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippTk.c externals.
 */
void
TkSippUpdate _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

void *
TSippInitialize _ANSI_ARGS_((Tcl_Interp *interp));

int
TkSipp_Init _ANSI_ARGS_((Tcl_Interp *interp));

/*
 * tSippPixMap.c externals.
 */
void
TSippPixMapInit _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

void
TSippPixMapCleanUp _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*
 * tSippGeom.c externals.
 */
void
TSippGeomInit _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

