/*
 *=============================================================================
 *                              tSippPixMap.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manipulate and render to memory resident pixel maps and
 * bit maps.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippPixMap.c,v 4.0 1993/11/27 21:29:43 markd Rel $
 *=============================================================================
 */

#include "tSippInt.h"
#include "sipp_pixmap.h"
#include "sipp_bitmap.h"

/*
 * PixMap table entry.  When an image is rendered in ODD or EVEN mode,
 * only half the scan lines are saved.
 */
typedef struct {
    tSippImageData_t  imgData;       /* Image data.                          */
    int               yImageSize;    /* Actual # of Y lines (for ODD/EVEN).  */
    Sipp_pixmap      *pixMapPtr;     /* Data: one or both ptrs will be NULL. */
    Sipp_bitmap      *bitMapPtr;
    char            **comments;      /* Array of comments.                   */
} pixMapDesc_t, *pixMapDesc_pt;

/*
 * Internal prototypes.
 */
static pixMapDesc_pt
PixMapHandleToPtr _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                               char           *handle));

static void
CleanPixMap _ANSI_ARGS_((pixMapDesc_pt pixMapPtr,
                         bool          cleanComments));

static pixMapDesc_pt
PixMapCommentCmdSetup _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                   char           *handle,
                                   char           *name));

static void *
PixMapOutputStart _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                               tSippOutputParms_pt   renderParmsPtr,
                               char                 *handle,
                               char                **comments));

static void
PixMapOutputLine _ANSI_ARGS_((pixMapDesc_pt  pixMapPtr,
                              int            y,
                              u_char        *rowPtr));

static void
PixMapOutputBitMap _ANSI_ARGS_((pixMapDesc_pt   pixMapPtr,
                               Sipp_bitmap    *bitMapPtr));

static bool
PixMapOutputEnd _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                             tSippOutputParms_pt   renderParmsPtr,
                             pixMapDesc_pt         pixMapPtr));

static bool
PixMapToPixMapCopy _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                pixMapDesc_pt   srcPixMapPtr,
                                char           *destHandle));

static void
PixMapToOtherImage _ANSI_ARGS_((tSippOutputParms_pt   outputParmsPtr,
                                pixMapDesc_pt         srcPixMapPtr,
                                void                 *destClientData,
                                tSippStorageClass_pt  storageClassPtr));

static bool
PixMapCopyImage _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                             char                 *srcHandle,
                             char                 *destHandle,
                             tSippStorageClass_pt  destClassPtr,
                             bool                  clear));

void
TSippPixMapCleanUp _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*=============================================================================
 * PixMapHandleToPtr --
 *   Utility procedure to convert a pixmap to a pixmap structure pointer.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A pixmap handle.
 * Returns:
 *   A pointer to the pixmap entry, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static pixMapDesc_pt
PixMapHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_pt    tSippGlobPtr;
    char           *handle;
{
    pixMapDesc_pt  pixMapPtr;

    pixMapPtr = (pixMapDesc_pt)
        Tcl_HandleXlate (tSippGlobPtr->interp,
                         tSippGlobPtr->pixMapTblPtr, handle);
    if (pixMapPtr == NULL)
        return NULL;
    return pixMapPtr;

}

/*=============================================================================
 * CleanPixMap --
 *   Release all dynamic memory in an pixmap.
 *
 * Parameters:
 *   o pixMapPtr (I) - Pointer to the pixmap.
 *   o cleanComments (I) - If TRUE comments are released, FALSE they are left
 *     alone.
 *-----------------------------------------------------------------------------
 */
static void
CleanPixMap (pixMapPtr, cleanComments)
    pixMapDesc_pt  pixMapPtr;
    bool           cleanComments;
{
    if (pixMapPtr->pixMapPtr != NULL)
        sipp_pixmap_destruct (pixMapPtr->pixMapPtr);

    if (pixMapPtr->bitMapPtr != NULL)
        sipp_bitmap_destruct (pixMapPtr->bitMapPtr);

    if (cleanComments)
        TSippFreeCom (&pixMapPtr->comments);

}

/*=============================================================================
 * SippPixMapCreate --
 *   Implements the command:
 *     SippPixMapCreate
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPixMapCreate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    pixMapDesc_pt   pixMapPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], (char *) NULL);
        return TCL_ERROR;
    }
    
    /*
     * Allocate and initialize the pixmap struct, defaulting all fields.
     */
    pixMapPtr = (pixMapDesc_pt) Tcl_HandleAlloc (tSippGlobPtr->pixMapTblPtr,
                                                 interp->result);
    pixMapPtr->imgData.xSize = 0;
    pixMapPtr->imgData.ySize = 0;
    pixMapPtr->yImageSize    = 0;
    pixMapPtr->pixMapPtr     = NULL;
    pixMapPtr->bitMapPtr     = NULL;
    pixMapPtr->comments      = NULL;

    return TCL_OK;

}

/*=============================================================================
 * SippPixMapUnref --
 *   Implements the command:
 *     SippPixMapUnref pixmapHandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPixMapUnref (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    pixMapDesc_pt   pixMapPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " pixmapHandle", (char *) NULL);
        return TCL_ERROR;
    }

    pixMapPtr = PixMapHandleToPtr (tSippGlobPtr, argv [1]);
    if (pixMapPtr == NULL)
        return TCL_ERROR;

    CleanPixMap (pixMapPtr, TRUE);  /* Clean comments */
    Tcl_HandleFree (tSippGlobPtr->pixMapTblPtr, pixMapPtr);

    return TCL_OK;

}

/*=============================================================================
 * PixMapCommentCmdSetup --
 *   Utility procedure to set up for one of the comments commands.  If converts
 * the pixmap handle and validates the comment name.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A pixmap handle.
 *   o name (I) - The name of the comment.  If NULL, then it is not validated
 * Returns:
 *   A pointer to the pixmap entry, or NULL if an error occured.  Errors
 * returned in tSippGlobPtr->interp->result.
 *-----------------------------------------------------------------------------
 */
static pixMapDesc_pt
PixMapCommentCmdSetup (tSippGlobPtr, handle, name)
    tSippGlob_pt    tSippGlobPtr;
    char           *handle;
    char           *name;
{
    pixMapDesc_pt  pixMapPtr;

    pixMapPtr = PixMapHandleToPtr (tSippGlobPtr, handle);
    if (pixMapPtr == NULL)
        return NULL;

    if ((name != NULL)  && (strpbrk (name, " \f\n\r\t\v=") != NULL)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "name may not contain the ",
                          "`=' or whitespace characters", (char *) NULL);
        return NULL;
    }
    return pixMapPtr;

}

/*=============================================================================
 * SippPixMapPutCom --
 *   Implements the command:
 *     SippPixMapPutCom pixmapHandle name [value]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPixMapPutCom (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    pixMapDesc_pt   pixMapPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 3) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " pixmapHandle name [value]", (char *) NULL);
        return TCL_ERROR;
    }

    pixMapPtr = PixMapCommentCmdSetup (tSippGlobPtr, argv [1], argv [2]);
    if (pixMapPtr == NULL)
        return TCL_ERROR;

    TSippPutCom (&pixMapPtr->comments, argv [2], argv [3]);
    return TCL_OK;

}

/*=============================================================================
 * SippPixMapGetCom --
 *   Implements the command:
 *     SippPixMapGetCom pixmaphandle [name] [retvar | {}]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPixMapGetCom (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    pixMapDesc_pt   pixMapPtr;
    char           *comStr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 2) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " pixmapHandle [name] [retvar | {}]", (char *) NULL);
        return TCL_ERROR;
    }

    pixMapPtr = PixMapCommentCmdSetup (tSippGlobPtr, argv [1], argv [2]);
    if (pixMapPtr == NULL)
        return TCL_ERROR;

    /*
     * If no name, return a keyed list of all comments.
     */
    if (argc == 2) {
        Tcl_SetResult (interp,
                       TSippGetAllCom (pixMapPtr->comments),
                       TCL_DYNAMIC);
        return TCL_OK;
    }

    /*
     * Search for a single comment.  If comment was not found, either return
     * false or leave result an empty string.  If its was found, either return
     * it in the variable or as the command result.
     */
    comStr = TSippGetCom (pixMapPtr->comments, argv [2]);

    if (comStr == NULL) {
        if (argc == 4)
            interp->result = "0";
    } else {
        if (argc == 4) {
            if (argv [3][0] != '\0') {
                if (Tcl_SetVar (interp, argv [3], comStr, 
                                TCL_LEAVE_ERR_MSG) == NULL)
                    return TCL_ERROR;
            }
            interp->result = "1";
        } else {
            Tcl_SetResult (interp, comStr, TCL_STATIC);
        }
    }
    return TCL_OK;

}

/*=============================================================================
 * SippPixMapDelCom --
 *   Implements the command:
 *     SippPixMapDelCom pixmapHandle [name]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPixMapDelCom (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    pixMapDesc_pt   pixMapPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " pixmapHandle [name]", (char *) NULL);
        return TCL_ERROR;
    }

    pixMapPtr = PixMapCommentCmdSetup (tSippGlobPtr, argv [1], argv [2]);
    if (pixMapPtr == NULL)
        return TCL_ERROR;

    if (argc == 2)
        TSippFreeCom (&pixMapPtr->comments);
    else
        TSippDelCom (&pixMapPtr->comments, argv [2]);
    return TCL_OK;

}

/*=============================================================================
 * SippPixMapInfo --
 *   Implements the command:
 *     SippPixMapInfo pixmapHandle attribute
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPixMapInfo (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    pixMapDesc_pt   pixMapPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " pixmapHandle attribute", (char *) NULL);
        return TCL_ERROR;
    }

    pixMapPtr = PixMapHandleToPtr (tSippGlobPtr, argv [1]);
    if (pixMapPtr == NULL)
        return TCL_ERROR;

    if (STREQU (argv [2], "XSIZE")) {
        sprintf (interp->result, "%d", pixMapPtr->imgData.xSize);
        return TCL_OK;
    }
    if (STREQU (argv [2], "YSIZE")) {
        sprintf (interp->result, "%d", pixMapPtr->imgData.ySize);
        return TCL_OK;
    }

    Tcl_AppendResult (interp, "unknown attribute \"", argv [2], 
                      "\", expected one of \"XSIZE\". or \"YSIZE\"",
                      (char *) NULL);
    return TCL_ERROR;

}

/*=============================================================================
 * PixMapOutputStart --
 *   Start output to an pixmap.  This routine is pointed to by the pixmap
 * image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handle (I) - The pixmap handle for pixmap to write.
 *   o comments (I) - If the source image has comments associated with it, they
 *     are passed here.  If the destination object can store them, it possible.
 *     NULL if no comments available.  Should contain standard comments when
 *     rendering.
 *
 * Returns:
 *   A pointer to be passed back into the other image output routines or NULL
 *   an a message in tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static void *
PixMapOutputStart (tSippGlobPtr, renderParmsPtr, handle, comments)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   renderParmsPtr;
    char                 *handle;
    char               **comments;
{
    pixMapDesc_pt  pixMapPtr;

    pixMapPtr = PixMapHandleToPtr (tSippGlobPtr, handle);
    if (pixMapPtr == NULL)
        return NULL;

    CleanPixMap (pixMapPtr, FALSE);  /* Don't clean comments */

    if (comments != NULL) {
        TSippMergeCom (comments, &pixMapPtr->comments);
    }

    /*
     * Determine Y size of image.  If only OLD or EVEN fields are being
     * rendered, its half the size (LINE mode is always both).
     */
    switch (renderParmsPtr->imgData.field) {
      case BOTH:
        pixMapPtr->yImageSize = renderParmsPtr->imgData.ySize;
        break;

      case EVEN:
        pixMapPtr->yImageSize = (renderParmsPtr->imgData.ySize & 1) ?
            (renderParmsPtr->imgData.ySize >> 1) + 1 :
            renderParmsPtr->imgData.ySize >> 1;
        break;

      case ODD:
        pixMapPtr->yImageSize = renderParmsPtr->imgData.ySize >> 1;
        break;

    }

    /*
     * Setup the pixmap header.
     */
    pixMapPtr->imgData = renderParmsPtr->imgData;

    if (renderParmsPtr->bitMapOutput) {
        pixMapPtr->bitMapPtr =
            sipp_bitmap_create (renderParmsPtr->imgData.xSize,
                                renderParmsPtr->imgData.ySize);
    } else {
        pixMapPtr->pixMapPtr =
            sipp_pixmap_create (renderParmsPtr->imgData.xSize,
                                pixMapPtr->yImageSize);
    }

    return pixMapPtr;

}

/*=============================================================================
 * PixMapOutputLine --
 *   Output a rendered line to a pixmap ISO.  This routine is pointed to
 * by the pixmap image storage class table.
 *
 * Parameters:
 *   o pixMapPtr (I) - A pointer to the pixmap structure.
 *   o y (I) - The scan line that was just rendered.
 *   o rowPtr (I) - The pixels for the scanline that was just rendered.
 *-----------------------------------------------------------------------------
 */
static void
PixMapOutputLine (pixMapPtr, y, rowPtr)
    pixMapDesc_pt  pixMapPtr;
    int          y;
    u_char      *rowPtr;
{
    u_char *pixelPtr;

    if (pixMapPtr->imgData.field != BOTH)
        y = y >> 1;

    pixelPtr = pixMapPtr->pixMapPtr->buffer +
        (3 * y * pixMapPtr->imgData.xSize);
    memcpy (pixelPtr, rowPtr, pixMapPtr->imgData.xSize * 3 * sizeof (u_char));

}

/*=============================================================================
 * PixMapOutputBitMap --
 *   Output a SIPP bit map to a pixmap.  This routine is pointed to by the
 * pixmap image storage class table.
 *
 * Parameters:
 *   o pixMapPtr (I) - A pointer to the pixmap structure.
 *   o bitMapPtr (I) - Pointer to the SIPP bit map structure.
 *-----------------------------------------------------------------------------
 */
static void
PixMapOutputBitMap (pixMapPtr, bitMapPtr)
    pixMapDesc_pt  pixMapPtr;
    Sipp_bitmap   *bitMapPtr;
{
    memcpy (pixMapPtr->bitMapPtr->buffer, bitMapPtr->buffer, 
            bitMapPtr->width * bitMapPtr->width_bytes);

}

/*=============================================================================
 * PixMapOutputEnd --
 *   Finish up output to a pixmap.  This routine is pointed to by the
 * pixmap image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o pixMapPtr (I) - A pointer to the pixmap structure.
 * Returns:
 *   TRUE, this routine really does nothing.
 *-----------------------------------------------------------------------------
 */
static bool
PixMapOutputEnd (tSippGlobPtr, renderParmsPtr, pixMapPtr)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   renderParmsPtr;
    pixMapDesc_pt         pixMapPtr;
{
   return TRUE;

}

/*=============================================================================
 * PixMapToPixMapCopy --
 *   Copy an image from one pixmap to another pixmap.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o srcPixMapPtr (I) - Pointer to the pixmap structure for the source image.
 *   o destHandle (I) - The handle for the destination pixmap.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PixMapToPixMapCopy (tSippGlobPtr, srcPixMapPtr, destHandle)
    tSippGlob_pt    tSippGlobPtr;
    pixMapDesc_pt   srcPixMapPtr;
    char           *destHandle;
{
    pixMapDesc_pt   destPixMapPtr;
    char          **saveComments;

    destPixMapPtr = PixMapHandleToPtr (tSippGlobPtr, destHandle);
    if (destPixMapPtr == NULL)
        return FALSE;

    /*
     * Copy header data to the pixmap.  Save the comments instead of releasing
     * them so they can be merged.
     */
    CleanPixMap (destPixMapPtr, FALSE);  /* Don't clean comments */

    saveComments = destPixMapPtr->comments;
    *destPixMapPtr = *srcPixMapPtr;
    destPixMapPtr->comments = saveComments;
    TSippMergeCom (srcPixMapPtr->comments, &destPixMapPtr->comments);

    /*
     * Copy the pixmap.
     */
    if (srcPixMapPtr->bitMapPtr != NULL) {
        destPixMapPtr->bitMapPtr =
            sipp_bitmap_create (srcPixMapPtr->imgData.xSize,
                                srcPixMapPtr->imgData.ySize);

        memcpy (destPixMapPtr->bitMapPtr->buffer, 
                srcPixMapPtr->bitMapPtr->buffer, 
                srcPixMapPtr->bitMapPtr->width * 
                srcPixMapPtr->bitMapPtr->width_bytes);
    } else {
        destPixMapPtr->pixMapPtr =
            sipp_pixmap_create (srcPixMapPtr->pixMapPtr->width,
                                srcPixMapPtr->pixMapPtr->height);

        memcpy (destPixMapPtr->pixMapPtr->buffer, 
                srcPixMapPtr->pixMapPtr->buffer, 
                3 * sizeof (u_char) *
                srcPixMapPtr->pixMapPtr->width *
                srcPixMapPtr->pixMapPtr->height);
    }

    return TRUE;

}

/*=============================================================================
 * PixMapToOtherImage --
 *   Copy an image from a pixmap to some other image type.  This assumes
 * all setup has been done, it just copies the scanlines.
 *
 * Parameters:
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o srcPixMapPtr (I) - Pointer to the pixmap structure for the source image.
 *   o destClientData (I) - The clientdata returned by the output start
 *     function for the image destination.
 *   o storageClassPtr (I) - Pointer to the image storage class description
 *     for the target class.
 *-----------------------------------------------------------------------------
 */
static void
PixMapToOtherImage (outputParmsPtr, srcPixMapPtr, destClientData, 
                   storageClassPtr)
    tSippOutputParms_pt   outputParmsPtr;
    pixMapDesc_pt         srcPixMapPtr;
    void                 *destClientData;
    tSippStorageClass_pt  storageClassPtr;
{
    int      xSize = outputParmsPtr->imgData.xSize;
    int      yStart, yLimit, yIncr, y;
    u_char  *rowPtr;
    void   (*outputLine) () = storageClassPtr->outputLine;

    /*
     * Set the scanline bounds based on the target scan direction.  If ODD or
     * EVEN fields are being outputed, only output those lines.
     */
    if (storageClassPtr->scanDirection == TSIPP_SCAN_BOTTOM_UP) {
        yStart = srcPixMapPtr->imgData.ySize - 1;
        yLimit = -1;
        yIncr  = -1;
    } else {
        yStart = 0;
        yLimit = srcPixMapPtr->imgData.ySize;
        yIncr  = 1;
    }
    if (srcPixMapPtr->imgData.field == ODD) {
        if ((yStart & 1) == 0)
            yStart += yIncr;   /* Make start ODD */
        if ((yLimit & 1) == 0)
            yLimit += yIncr;   /* Make limit ODD */
        yIncr *= 2;
    } else if (srcPixMapPtr->imgData.field == EVEN) {
        if ((yStart & 1) != 0)
            yStart += yIncr;   /* Make start EVEN */
        if ((yLimit & 1) != 0)
            yLimit += yIncr;   /* Make limit EVEN */
        yIncr *= 2;
    }

    /*
     * Copy the image.  Indexing is different for ODD or EVEN, as only 1/2
     * the lines are stored.
     */
    if (srcPixMapPtr->imgData.field == BOTH) {
        for (y = yStart; y != yLimit; y += yIncr) {
            rowPtr = srcPixMapPtr->pixMapPtr->buffer + (3 * y * xSize);
            (*outputLine) (destClientData, y, rowPtr);
        }
    } else {
        for (y = yStart; y != yLimit; y += yIncr) {
            rowPtr = srcPixMapPtr->pixMapPtr->buffer + (3 * (y >> 1) * xSize);
            (*outputLine) (destClientData, y, rowPtr);
        }
    }

}

/*=============================================================================
 * PixMapCopyImage --
 *   Copy an image from a pixmap to another image storage object.
 * This routine is pointed to by the pixmap image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o srcHandle (I) - The pixmap handle for the pixmap image to read.
 *   o destHandle (I) - The handle for the destination image storage object.
 *   o destClassPtr (I) - Pointer to the image storage class description
 *     for the destination class.
 *   o clear (I) - If TRUE, clear target image before copying, if FALSE,
 *     don't clear.  Ignored if the ISO does not support this.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PixMapCopyImage (tSippGlobPtr, srcHandle, destHandle, destClassPtr, clear)
    tSippGlob_pt          tSippGlobPtr;
    char                 *srcHandle;
    char                 *destHandle;
    tSippStorageClass_pt  destClassPtr;
    bool                  clear;
{
    pixMapDesc_pt        srcPixMapPtr;
    tSippOutputParms_t   outputParms;
    void                *destClientData;

    srcPixMapPtr = PixMapHandleToPtr (tSippGlobPtr, srcHandle);
    if (srcPixMapPtr == NULL)
        return FALSE;

    /*
     * If the target is a pixmap, optimize the copy.
     */
    if (destClassPtr->copyImage == PixMapCopyImage)
        return PixMapToPixMapCopy (tSippGlobPtr, srcPixMapPtr, destHandle);

    /*
     * Set up the general output parameters assoicated with this image.
     */
    outputParms.imgData = srcPixMapPtr->imgData;

    /*
     * Initialize other output parms fields.
     */
    outputParms.argv     = NULL;   
    outputParms.update   = 0;
    outputParms.maxDepth = 0;
    outputParms.clear    = clear;

    outputParms.bitMapOutput  = srcPixMapPtr->bitMapPtr != NULL;
    outputParms.scanDirection = destClassPtr->scanDirection;


    /*
     * Copy to other image source.
     */
    destClientData = (*destClassPtr->outputStart) (tSippGlobPtr,
                                                   &outputParms,
                                                   destHandle,
                                                   srcPixMapPtr->comments);
    if (destClientData == NULL)
        return FALSE;

    if (srcPixMapPtr->bitMapPtr != NULL) {
        (*destClassPtr->outputBitMap) (destClientData,
                                       srcPixMapPtr->bitMapPtr);
    } else {
        PixMapToOtherImage (&outputParms,
                            srcPixMapPtr,
                            destClientData,
                            destClassPtr);
    }

    return (*destClassPtr->outputEnd) (tSippGlobPtr,
                                       &outputParms,
                                       destClientData);

}

/*=============================================================================
 * TSippPixMapInit --
 *   Initialized the pixmap commands and the pixmap table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPixMapInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippPixMapCreate",  (Tcl_CmdProc *) SippPixMapCreate},
        {"SippPixMapUnref",   (Tcl_CmdProc *) SippPixMapUnref},
        {"SippPixMapPutCom",  (Tcl_CmdProc *) SippPixMapPutCom},
        {"SippPixMapGetCom",  (Tcl_CmdProc *) SippPixMapGetCom},
        {"SippPixMapDelCom",  (Tcl_CmdProc *) SippPixMapDelCom},
        {"SippPixMapInfo",    (Tcl_CmdProc *) SippPixMapInfo},
        {NULL,                NULL}
    };

    static tSippStorageClass_t storageClass = {
        "pixmap",              /* handlePrefix  */
        5,                     /* prefixSize    */
        TSIPP_SCAN_BOTH,       /* scanDirection */
        TRUE,                  /* bitMapOptimal */
        PixMapOutputStart,
        PixMapOutputLine,
        PixMapOutputBitMap,
        PixMapOutputEnd,
        PixMapCopyImage
    };

    tSippGlobPtr->pixMapTblPtr =
        Tcl_HandleTblInit ("pixmap", sizeof (pixMapDesc_t), 4);

    TSippInitCmds (tSippGlobPtr, cmdTable);
    TSippAddStorageClass (tSippGlobPtr, &storageClass);

}

/*=============================================================================
 * TSippPixMapCleanUp --
 *   Delete all pixmaps handles and release all unreferenced pixmaps.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPixMapCleanUp (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    int            walkKey = -1;
    pixMapDesc_pt  pixMapPtr;

    while (TRUE) {
        pixMapPtr = Tcl_HandleWalk (tSippGlobPtr->pixMapTblPtr, &walkKey);
        if (pixMapPtr == NULL)
            break;
        CleanPixMap (pixMapPtr, TRUE);  /* Clean comments */
        Tcl_HandleFree (tSippGlobPtr->pixMapTblPtr, pixMapPtr);
    }
    Tcl_HandleTblRelease (tSippGlobPtr->pixMapTblPtr);
    tSippGlobPtr->pixMapTblPtr = NULL;

}

