/*
 * t4command.cpp --
 *
 *	This file contains the implementation of procedures for
 *	creation of Tcl commands that invoke operations on T4Graph
 *	objects.
 *
 *	Authors: Jacob Levy and Jean-Claude Wippler.
 *		 jyl@best.com	jcw@equi4.com
 *
 * Copyright (c) 2000-2003, JYL Software Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE, EVEN IF
 * JYL SOFTWARE INC. IS MADE AWARE OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "t4graphrep.h"

/*
 * StorageCmdDeleteProc --
 *
 *	This procedure is called when the Tcl command for a storage
 *	is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 */

static void
StorageCmdDeleteProc(ClientData cd)
{
    /* Nothing to do. */
}

/*
 * T4Graph_MakeStorageCommand --
 *
 *	Given a T4Storage object, make a Tcl command that will
 *	invoke operations on that object.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	May create a new Tcl command. May create a new namespace for
 *	stored procedure commands.
 */

int
T4Graph_MakeStorageCommand(Tcl_Interp *interp, T4Storage *s)
{
    char *sn = s->GetName();
    Tcl_Command res;

    res = Tcl_CreateObjCommand(interp,
			       sn,
			       T4Graph_StorageCmdProc,
			       s,
			       StorageCmdDeleteProc);
    if (res != NULL) {
	Tcl_SetStringObj(Tcl_GetObjResult(interp), sn, -1);
	return TCL_OK;
    }
    return TCL_ERROR;
}

/*
 * T4Graph_DeleteStorageCommand --
 *
 *	Given a T4Storage object that is about to be deleted, delete the
 *	Tcl command created with MakeStorageCommand to operate on it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	May delete a Tcl command. May delete the namespace for stored
 *	procedure commands associated with this storage.
 */

void
T4Graph_DeleteStorageCommand(Tcl_Interp *interp, T4Storage *s)
{
    char *sn = s->GetName();

    Tcl_DeleteCommand(interp, sn);
    Tcl_ResetResult(interp);
}
