# Emacs -*- C -*-

# /*
##############################################################################
#
#   File: insertion.tcl
#   C statements to emulate Tcl bytecodes.
#
#   Note: despite the Emacs modeline above, this is really a Tcl file.
#
##############################################################################
#
#   Copyright (c) 2000 David Cuthbert / Kanga International Group.
#   All rights reserved.
#
#   Permission to use, copy, modify, distribute and sell this software
#   and its documentation for any purpose is hereby granted without fee,
#   provided that the above copyright notice appear in all copies and
#   that both that copyright notice and this permission notice appear
#   in supporting documentation.  The author(s) make no representations
#   about the suitability of this software for any purpose.  It is 
#   provided "as is" without express or implied warranty.
#
##############################################################################
#
#   This file contains code derived from tclExecute.c, which contains the
#   following notice:
#
#   Copyright (c) 1996-1997 Sun Microsystems, Inc.
#   Copyright (c) 1998-2000 by Scriptics Corporation.
#
#   This software is copyrighted by the Regents of the University of
#   California, Sun Microsystems, Inc., Scriptics Corporation,
#   and other parties.  The following terms apply to all files associated
#   with the software unless explicitly disclaimed in individual files.
#
#   The authors hereby grant permission to use, copy, modify, distribute,
#   and license this software and its documentation for any purpose, provided
#   that existing copyright notices are retained in all copies and that this
#   notice is included verbatim in any distributions. No written agreement,
#   license, or royalty fee is required for any of the authorized uses.
#   Modifications to this software may be copyrighted by their authors
#   and need not follow the licensing terms described here, provided that
#   the new terms are clearly indicated on the first page of each file where
#   they apply.
#
#   IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY
#   FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
#   ARISING OUT OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY
#   DERIVATIVES THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE
#   POSSIBILITY OF SUCH DAMAGE.
#
#   THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES,
#   INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
#   FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE
#   IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE
#   NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
#   MODIFICATIONS.
#
#   GOVERNMENT USE: If you are acquiring this software on behalf of the
#   U.S. government, the Government shall have only "Restricted Rights"
#   in the software and related documentation as defined in the Federal 
#   Acquisition Regulations (FARs) in Clause 52.227.19 (c) (2).  If you
#   are acquiring the software on behalf of the Department of Defense, the
#   software shall be classified as "Commercial Computer Software" and the
#   Government shall have only "Restricted Rights" as defined in Clause
#   252.227-7013 (c) (1) of DFARs.  Notwithstanding the foregoing, the
#   authors grant the U.S. Government and others acting in its behalf
#   permission to use and distribute the software in accordance with the
#   terms specified in this license. 
#
##############################################################################
# */

set __signature {
int ${function}(ClientData clientData, Tcl_Interp *interp, int objc,
    Tcl_Obj* const args[])
}

set __prologue {
    KangaProc   *kprocPtr = (KangaProc *)clientData;
    Interp      *iPtr = (Interp *) interp;
#if ($maxStackDepth > 0)
    Tcl_Obj     *stackData[$maxStackDepth];
#else
    Tcl_Obj     *stackData[1];
#endif
    Tcl_Obj    **stackPtr = stackData;
#if ($maxExceptDepth > 0)
    Tcl_Obj    **unwindPtrs[$maxExceptDepth];
#endif
    int          unwindTop = -1;
    int          result = TCL_OK;
#if ($numCompiledLocals > 0)
    Var          compiledLocals[$numCompiledLocals];
#else
    Var          compiledLocals[1];
#endif
    Var         *varPtr;
    CallFrame    frame;
    int          argCount;
    int          i;
    int          iter;
}

set __prologue_2 {
    assert(kprocPtr->nsPtr != NULL);

    /* Setup and push a new call frame. */
    result = Tcl_PushCallFrame(interp, (Tcl_CallFrame *)&frame, 
			       (Tcl_Namespace *)kprocPtr->nsPtr, 1);
    if(result != 0) return result;

    /* Initialise the frame */
    frame.objc = objc;
    frame.objv = args;
    frame.procPtr = kprocPtr->procPtr;
    frame.numCompiledLocals = $numCompiledLocals;
    frame.compiledLocals = compiledLocals;
    TclInitCompiledLocals(interp, &frame, frame.nsPtr);

    /* Match the passed arguments to the procedure's args. */
    varPtr = frame.compiledLocals;

    /* Argument detection */
    argCount = 1;
}

set __detect_anyarg {
    /* argument $argIndex: anyarg args */
    assert(argCount == $argIndex);
    varPtr->value.objPtr = Tcl_NewListObj(objc-$argIndex, args+$argIndex);
    Tcl_IncrRefCount(varPtr->value.objPtr);
    varPtr->flags &= ~VAR_UNDEFINED;
    argCount = $argLength;
    varPtr++;
}

set __detect_stdarg {
    /* argument $argIndex: stdarg $argName */
    if(objc > $argIndex) {
	varPtr->value.objPtr = args[$argIndex];
	Tcl_IncrRefCount(varPtr->value.objPtr);
	varPtr->flags &= ~VAR_UNDEFINED;
	argCount++;
	varPtr++;
    } else {
	goto detectEnd;
    }
}


set __detect_defarg_int {
    /* argument $argIndex: defarg int $argName $defValue */
    if(objc > $argIndex) {
	varPtr->value.objPtr = args[$argIndex];
	Tcl_IncrRefCount(varPtr->value.objPtr);
	varPtr->flags &= ~VAR_UNDEFINED;
    } else {
	varPtr->value.objPtr = Tcl_NewIntObj($defValue);
	Tcl_IncrRefCount(varPtr->value.objPtr);
	varPtr->flags &= ~VAR_UNDEFINED;
    }
    argCount++;
    varPtr++;
}


set __detect_defarg_double {
    /* argument $argIndex: defarg double $argName $defValue */
    if(objc > $argIndex) {
	varPtr->value.objPtr = args[$argIndex];
	Tcl_IncrRefCount(varPtr->value.objPtr);
	varPtr->flags &= ~VAR_UNDEFINED;
    } else {
	varPtr->value.objPtr = Tcl_NewDoubleObj($defValue);
	Tcl_IncrRefCount(varPtr->value.objPtr);
	varPtr->flags &= ~VAR_UNDEFINED;
    }
    argCount++;
    varPtr++;
}


set __detect_defarg_string {
    /* argument $argIndex: defarg double $argName $defValue */
    if(objc > $argIndex) {
	varPtr->value.objPtr = args[$argIndex];
	Tcl_IncrRefCount(varPtr->value.objPtr);
	varPtr->flags &= ~VAR_UNDEFINED;
    } else {
	varPtr->value.objPtr = Tcl_NewStringObj("$defValue", $defValueLen);
	Tcl_IncrRefCount(varPtr->value.objPtr);
	varPtr->flags &= ~VAR_UNDEFINED;
    }
    argCount++;
    varPtr++;
}


set __detect_end {
 detectEnd:
    /* Make sure we're not waiting for another argument */
    if(argCount < $argLength) {
	Tcl_ResetResult(interp);
	Tcl_AppendToObj(Tcl_GetObjResult(interp), 
			"wrong # args: should be \"$usage\"", -1);
	result = 1;
	goto procDone;
    }
}


set push1 {
    assert(stackPtr < stackData + $maxStackDepth);
    *stackPtr = kprocPtr->literals[$op1];
    Tcl_IncrRefCount(*stackPtr);
    stackPtr++;
}
set push4 $push1

set pop {
    stackPtr--;
    assert(stackPtr >= stackData);
    Tcl_DecrRefCount(*stackPtr);
}

set loadScalar1 {
    assert(stackPtr < stackData + $maxStackDepth);
    *stackPtr = TclGetIndexedScalar(interp, $op1, 1);
    assert(*stackPtr != NULL);
    Tcl_IncrRefCount(*stackPtr);
    stackPtr++;
}
set loadScalar4 $loadScalar1

set storeScalar1 {
    {
	Tcl_Obj *valuePtr1, *valuePtr2;
	
	stackPtr--;
	assert(stackPtr >= stackData);
	valuePtr1 = *stackPtr;
	valuePtr2 = TclSetIndexedScalar(interp, $op1, valuePtr1, 1);
	if(valuePtr2 == NULL) {
	    Tcl_DecrRefCount(valuePtr1);
	    $catchJump;
	}

	*stackPtr = valuePtr2;
	Tcl_IncrRefCount(*stackPtr);
	Tcl_DecrRefCount(valuePtr1);
	stackPtr++;
    }
}
set storeScalar4 $storeScalar1

set invokeStk1 {
    if(iPtr->flags & DELETED) {
	Tcl_ResetResult(interp);
	Tcl_AppendToObj(Tcl_GetObjResult(interp),
			"attempt to call eval in deleted interpreter", -1);
	Tcl_SetErrorCode(interp, "CORE", "IDELETE",
			 "attempt to call eval in deleted interpreter", 0);
	result = TCL_ERROR;

	
	/* insert exception handling code here */
	goto procDone;
    }

    {
	Command  *cmdPtr;
	Tcl_Obj **cmdStart = (stackPtr - $op1);
	int       cmdLength = $op1;

	assert(cmdStart >= stackData);
	assert(stackPtr <= stackData + $maxStackDepth);

	cmdPtr = (Command *)Tcl_GetCommandFromObj(interp, *cmdStart);
	if(cmdPtr == NULL) {
	    cmdPtr = (Command *)Tcl_FindCommand(interp, "unknown", 0,
						TCL_GLOBAL_ONLY);
	    if(cmdPtr == NULL) {
		Tcl_ResetResult(interp);
		Tcl_AppendStringsToObj(Tcl_GetObjResult(interp),
				       "invalid command name \"",
				       Tcl_GetString(*cmdStart), "\"", 0);
		result = TCL_ERROR;
		/* insert exception handling code here */
		goto procDone;
	    }
	
	    for(i = $op1 - 1; i >= 0; i--) {
		assert(cmdStart + i + 1 < stackData + $maxStackDepth);
		cmdStart[i + i] = cmdStart[i];
	    }
	    
	    *cmdStart = Tcl_NewStringObj("unknown", 7);
	    Tcl_IncrRefCount(*cmdStart);
	    stackPtr++;
	    cmdLength++;
	}

	Tcl_ResetResult(interp);
	iPtr->cmdCount++;
	result = (*cmdPtr->objProc)(cmdPtr->objClientData, interp, cmdLength,
				    cmdStart);

	if(Tcl_AsyncReady())
	    result = Tcl_AsyncInvoke(interp, result);

	if(*(iPtr->result) != 0) (void) Tcl_GetObjResult(interp);
	
	for(i = 0; i < cmdLength; i++) {
	    stackPtr--;
	    assert(stackPtr >= stackData);
	    Tcl_DecrRefCount(*stackPtr);
	}

	switch(result) {
	    case TCL_OK:
	    assert(stackPtr < stackData + $maxStackDepth);
	    *stackPtr = Tcl_GetObjResult(interp);
	    Tcl_IncrRefCount(*stackPtr);
	    stackPtr++;
	    break;

	    case TCL_BREAK:
	    $breakJump;

	    case TCL_CONTINUE:
	    $continueJump;

	    case TCL_ERROR:
	    case TCL_RETURN:
	    default:
	    $catchJump;
	}
    }
}

set concat1 {
    {
	Tcl_Obj *concatPtr = Tcl_NewObj();
	Tcl_IncrRefCount(concatPtr);
	assert(stackPtr - $op1 >= stackData);

	for(i = $op1; i > 0; i--) {
	    int oldlen;
	    char *otherString;
	    int   otherLen;

	    otherString = Tcl_GetStringFromObj(*(stackPtr - i), &otherLen);

	    Tcl_AppendToObj(concatPtr, otherString, otherLen);
	    Tcl_DecrRefCount(*(stackPtr - i));

	    assert(concatPtr->bytes != NULL);
	}

	stackPtr -= $op1;
	assert(stackPtr >= stackData);
	*stackPtr = concatPtr;
	stackPtr++;
	assert(stackPtr <= stackData + $maxStackDepth);
    }
}


set done {
    stackPtr--;
    assert(stackPtr == stackData);
    Tcl_SetObjResult(interp, *stackPtr);
    Tcl_DecrRefCount(*stackPtr);
    goto procDone;
}


set foreach_start4 {
    for(i = 0; i < foreach_numLists_$op1; ++i) {
	Tcl_Obj *listObj =
	    TclGetIndexedScalar(interp, foreach_firstList_$op1 + i, 1);
	
	if(listObj->typePtr != kprocPtr->listType) {
	    result = (*kprocPtr->listType->setFromAnyProc)(interp, listObj);
	    if(result) 
		$catchJump;
	}

	foreach_list_$op1[i] = (List *)listObj->internalRep.otherValuePtr;
	foreach_listIndex_$op1[i] = 0;
    }
}

set foreach_step4 {
    do {
	int continueForeach = 0;

	assert(stackPtr < stackData + $maxStackDepth);

	for(i = 0; i < foreach_numLists_$op1; ++i) {
	    if(foreach_listIndex_$op1[i] < foreach_list_$op1[i]->elemCount) {
		/* Ok, proceed with the foreach */
		continueForeach = 1;
		break;
	    }
	}

	*stackPtr = Tcl_NewIntObj(continueForeach);
	Tcl_IncrRefCount(*stackPtr);
	stackPtr++;

	if(!continueForeach) break;

	for(i = 0; i < foreach_numLists_$op1; ++i) {
	    int j;

	    for(j = 0; j < foreach_numVars_$op1[i]; ++j) {
		int scalarIndex = foreach_varList_$op1[i][j];
		Tcl_Obj *listElement;

		if(foreach_list_$op1[i]->elemCount > foreach_listIndex_$op1[i])
		{
		    listElement = foreach_list_$op1[i]->elements[
			foreach_listIndex_$op1[i]];
		    foreach_listIndex_$op1[i]++;
		} else {
		    listElement = Tcl_NewObj();
		}

		TclSetIndexedScalar(interp, scalarIndex, listElement, 1);
	    }
	}
    } while(0);
}

set jumpFalse1 {
    stackPtr--;
    assert(stackPtr >= stackData);
    
    result = Tcl_GetIntFromObj(interp, *stackPtr, &i);
    Tcl_DecrRefCount(*stackPtr);

    if(result) $catchJump;
    if(i == 0) goto $jtarget;
}
set jumpFalse4 $jumpFalse1


set jumpTrue1 {
    stackPtr--;
    assert(stackPtr >= stackData);
    
    result = Tcl_GetIntFromObj(interp, *stackPtr, &i);
    Tcl_DecrRefCount(*stackPtr);

    if(result) $catchJump;
    if(i != 0) goto $jtarget;
}
set jumpTrue4 $jumpTrue1

set jump1 {
    goto $jtarget;
}
set jump4 $jump1

set incrScalar1 {
    stackPtr--;
    assert(stackPtr >= stackData);
    
    if((*stackPtr)->typePtr != kprocPtr->intType) {
	result = kprocPtr->intType->setFromAnyProc(interp, *stackPtr);
	if(result) $catchJump;
    }

    {
	Tcl_Obj *resultObj = TclIncrIndexedScalar(
	    interp, $op1, (*stackPtr)->internalRep.longValue);
	Tcl_DecrRefCount(*stackPtr);

	if(resultObj == NULL) {
	    result = TCL_ERROR;
	    $catchJump;
	}

	*stackPtr = resultObj;
	Tcl_IncrRefCount(*stackPtr);
	stackPtr++;
    }
}

set incrScalar1Imm {
    assert(stackPtr < stackData + $maxStackDepth);

    {
	Tcl_Obj *resultObj = TclIncrIndexedScalar(interp, $op1, $op2);

	if(resultObj == NULL) {
	    result = TCL_ERROR;
	    $catchJump;
	}

	*stackPtr = resultObj;
	Tcl_IncrRefCount(*stackPtr);
	stackPtr++;
    }
}

set add {
    assert(stackPtr >= stackData + 2);
    {
	Tcl_Obj *op1,        *op2;
	char    *op1_string, *op2_string;
	int      op1_slen,    op2_slen;
	int      op1IsInt;
	int      op2IsInt;

	op2 = *(--stackPtr);
	op1 = *(--stackPtr);

	/* Use integer arithmetic if:
	 * - Both arguments have integral type, OR
	 * - Both arguments look like integers.
	 * Otherwise, use double arithmetic.
	 */

	if(op1->typePtr == kprocPtr->intType)
	    op1IsInt = 1;
	else if(op1->typePtr == kprocPtr->doubleType && op1->bytes == NULL)
	    op1IsInt = 0;
	else {
	    op1_string = Tcl_GetStringFromObj(op1, &op1_slen);
	    if(TclLooksLikeInt(op1_string, op1_slen))
		op1IsInt = 1;
	    else
		op1IsInt = 0;
	}

	if(op2->typePtr == kprocPtr->intType)
	    op2IsInt = 1;
	else if(op2->typePtr == kprocPtr->doubleType && op2->bytes == NULL)
	    op2IsInt = 0;
	else {
	    op2_string = Tcl_GetStringFromObj(op2, &op2_slen);
	    if(TclLooksLikeInt(op2_string, op2_slen))
		op2IsInt = 1;
	    else
		op2IsInt = 0;
	}

	if(op1IsInt && op2IsInt) {
	    long op1IntVal;
	    long op2IntVal;
	    
	    Tcl_GetLongFromObj(interp, op1, &op1IntVal);
	    Tcl_GetLongFromObj(interp, op2, &op2IntVal);

	    *stackPtr = Tcl_NewLongObj(op1IntVal + op2IntVal);
	} else {
	    double op1DoubleVal;
	    double op2DoubleVal;

	    Tcl_GetDoubleFromObj(interp, op1, &op1DoubleVal);
	    Tcl_GetDoubleFromObj(interp, op2, &op2DoubleVal);

	    *stackPtr = Tcl_NewDoubleObj(op1DoubleVal + op2DoubleVal);
	}

	Tcl_IncrRefCount(*stackPtr);
	stackPtr++;

    add_done_$ip:
	Tcl_DecrRefCount(op1);
	Tcl_DecrRefCount(op2);
    }
}


set sub {
    assert(stackPtr >= stackData + 2);
    {
	Tcl_Obj *op1,        *op2;
	char    *op1_string, *op2_string;
	int      op1_slen,    op2_slen;
	int      op1IsInt;
	int      op2IsInt;

	op2 = *(--stackPtr);
	op1 = *(--stackPtr);

	/* Use integer arithmetic if:
	 * - Both arguments have integral type, OR
	 * - Both arguments look like integers.
	 * Otherwise, use double arithmetic.
	 */

	if(op1->typePtr == kprocPtr->intType)
	    op1IsInt = 1;
	else if(op1->typePtr == kprocPtr->doubleType && op1->bytes == NULL)
	    op1IsInt = 0;
	else {
	    op1_string = Tcl_GetStringFromObj(op1, &op1_slen);
	    if(TclLooksLikeInt(op1_string, op1_slen))
		op1IsInt = 1;
	    else
		op1IsInt = 0;
	}

	if(op2->typePtr == kprocPtr->intType)
	    op2IsInt = 1;
	else if(op2->typePtr == kprocPtr->doubleType && op2->bytes == NULL)
	    op2IsInt = 0;
	else {
	    op2_string = Tcl_GetStringFromObj(op2, &op2_slen);
	    if(TclLooksLikeInt(op2_string, op2_slen))
		op2IsInt = 1;
	    else
		op2IsInt = 0;
	}

	if(op1IsInt && op2IsInt) {
	    long op1IntVal;
	    long op2IntVal;
	    
	    Tcl_GetLongFromObj(interp, op1, &op1IntVal);
	    Tcl_GetLongFromObj(interp, op2, &op2IntVal);

	    *stackPtr = Tcl_NewLongObj(op1IntVal - op2IntVal);
	} else {
	    double op1DoubleVal;
	    double op2DoubleVal;

	    Tcl_GetDoubleFromObj(interp, op1, &op1DoubleVal);
	    Tcl_GetDoubleFromObj(interp, op2, &op2DoubleVal);

	    *stackPtr = Tcl_NewDoubleObj(op1DoubleVal - op2DoubleVal);
	}

	Tcl_IncrRefCount(*stackPtr);
	stackPtr++;

    sub_done_$ip:
	Tcl_DecrRefCount(op1);
	Tcl_DecrRefCount(op2);
    }
}


set mult {
    assert(stackPtr >= stackData + 2);
    {
	Tcl_Obj *op1,        *op2;
	char    *op1_string, *op2_string;
	int      op1_slen,    op2_slen;
	int      op1IsInt;
	int      op2IsInt;

	op2 = *(--stackPtr);
	op1 = *(--stackPtr);

	/* Use integer arithmetic if:
	 * - Both arguments have integral type, OR
	 * - Both arguments look like integers.
	 * Otherwise, use double arithmetic.
	 */

	if(op1->typePtr == kprocPtr->intType)
	    op1IsInt = 1;
	else if(op1->typePtr == kprocPtr->doubleType && op1->bytes == NULL)
	    op1IsInt = 0;
	else {
	    op1_string = Tcl_GetStringFromObj(op1, &op1_slen);
	    if(TclLooksLikeInt(op1_string, op1_slen))
		op1IsInt = 1;
	    else
		op1IsInt = 0;
	}

	if(op2->typePtr == kprocPtr->intType)
	    op2IsInt = 1;
	else if(op2->typePtr == kprocPtr->doubleType && op2->bytes == NULL)
	    op2IsInt = 0;
	else {
	    op2_string = Tcl_GetStringFromObj(op2, &op2_slen);
	    if(TclLooksLikeInt(op2_string, op2_slen))
		op2IsInt = 1;
	    else
		op2IsInt = 0;
	}

	if(op1IsInt && op2IsInt) {
	    long op1IntVal;
	    long op2IntVal;
	    
	    Tcl_GetLongFromObj(interp, op1, &op1IntVal);
	    Tcl_GetLongFromObj(interp, op2, &op2IntVal);

	    *stackPtr = Tcl_NewLongObj(op1IntVal * op2IntVal);
	} else {
	    double op1DoubleVal;
	    double op2DoubleVal;

	    Tcl_GetDoubleFromObj(interp, op1, &op1DoubleVal);
	    Tcl_GetDoubleFromObj(interp, op2, &op2DoubleVal);

	    *stackPtr = Tcl_NewDoubleObj(op1DoubleVal * op2DoubleVal);
	}

	Tcl_IncrRefCount(*stackPtr);
	stackPtr++;

    mul_done_$ip:
	Tcl_DecrRefCount(op1);
	Tcl_DecrRefCount(op2);
    }
}


set div {
    assert(stackPtr >= stackData + 2);
    {
	Tcl_Obj *op1,        *op2;
	char    *op1_string, *op2_string;
	int      op1_slen,    op2_slen;
	int      op1IsInt;
	int      op2IsInt;

	op2 = *(--stackPtr);
	op1 = *(--stackPtr);

	/* Use integer arithmetic if:
	 * - Both arguments have integral type, OR
	 * - Both arguments look like integers.
	 * Otherwise, use double arithmetic.
	 */

	if(op1->typePtr == kprocPtr->intType)
	    op1IsInt = 1;
	else if(op1->typePtr == kprocPtr->doubleType && op1->bytes == NULL)
	    op1IsInt = 0;
	else {
	    op1_string = Tcl_GetStringFromObj(op1, &op1_slen);
	    if(TclLooksLikeInt(op1_string, op1_slen))
		op1IsInt = 1;
	    else
		op1IsInt = 0;
	}

	if(op2->typePtr == kprocPtr->intType)
	    op2IsInt = 1;
	else if(op2->typePtr == kprocPtr->doubleType && op2->bytes == NULL)
	    op2IsInt = 0;
	else {
	    op2_string = Tcl_GetStringFromObj(op2, &op2_slen);
	    if(TclLooksLikeInt(op2_string, op2_slen))
		op2IsInt = 1;
	    else
		op2IsInt = 0;
	}

	if(op1IsInt && op2IsInt) {
	    long op1IntVal;
	    long op2IntVal;
	    
	    Tcl_GetLongFromObj(interp, op1, &op1IntVal);
	    Tcl_GetLongFromObj(interp, op2, &op2IntVal);

	    if(op2IntVal == 0) {
		Tcl_DecrRefCount(op1);
		Tcl_DecrRefCount(op2);

		Tcl_ResetResult(interp);
		Tcl_AppendToObj(Tcl_GetObjResult(interp), "divide by zero", 
				-1);
		Tcl_SetErrorCode(interp, "ARITH", "DIVZERO", "divide by zero",
				 NULL);
		result = TCL_ERROR;
		$catchJump;
	    }

	    *stackPtr = Tcl_NewLongObj(op1IntVal / op2IntVal);
	} else {
	    double op1DoubleVal;
	    double op2DoubleVal;
	    double quotient;

	    Tcl_GetDoubleFromObj(interp, op1, &op1DoubleVal);
	    Tcl_GetDoubleFromObj(interp, op2, &op2DoubleVal);

	    if(op2DoubleVal == 0.0) {
		Tcl_DecrRefCount(op1);
		Tcl_DecrRefCount(op2);

		Tcl_ResetResult(interp);
		Tcl_AppendToObj(Tcl_GetObjResult(interp), "divide by zero", 
				-1);
		Tcl_SetErrorCode(interp, "ARITH", "DIVZERO", "divide by zero",
				 NULL);
		result = TCL_ERROR;
		$catchJump;
	    }
	    
	    quotient = op1DoubleVal / op2DoubleVal;

	    if(quotient != quotient || quotient > DBL_MAX || 
	       quotient < -DBL_MAX) 
	    {
		Tcl_DecrRefCount(op1);
		Tcl_DecrRefCount(op2);
		
		TclExprFloatError(interp, quotient);
		result = TCL_ERROR;
		$catchJump;
	    }
	    *stackPtr = Tcl_NewDoubleObj(quotient);
	}

	Tcl_IncrRefCount(*stackPtr);
	stackPtr++;

    div_done_$ip:
	Tcl_DecrRefCount(op1);
	Tcl_DecrRefCount(op2);
    }
}


set lt {
    {
	Tcl_Obj *op1,        *op2;
	char    *op1_string, *op2_string;
	int      op1_slen,    op2_slen;
	long     op1_lval,    op2_lval;
	double   op1_dval,    op2_dval;        
	int      op1_type;
	int      op2_type;
	int      compResult;

	op2 = *(--stackPtr);
	op1 = *(--stackPtr);

	/* Use integer comparison if:
	 * - Both arguments have integral type, OR
	 * - Both arguments look like integers.
	 * Otherwise, use double comparison if:
	 * - Both arguments have double or integer type, OR
	 * - Both arguments look like numbers.
	 * Otherwise, use string comparison;
	 */

	if(op1->typePtr == kprocPtr->intType)
	    op1_lval = op1->internalRep.longValue;
	else if(op1->typePtr == kprocPtr->doubleType && op1->bytes == NULL)
	    op1_dval = op1->internalRep.doubleValue;
	else {
	    op1_string = Tcl_GetStringFromObj(op1, &op1_slen);
	    if(TclLooksLikeInt(op1_string, op1_slen))
		Tcl_GetLongFromObj(NULL, op1, &op1_lval);
	    else
		Tcl_GetDoubleFromObj(NULL, op1, &op1_dval);
	}

	if(op2->typePtr == kprocPtr->intType)
	    op2_lval = op2->internalRep.longValue;
	else if(op2->typePtr == kprocPtr->doubleType && op2->bytes == NULL)
	    op2_dval = op2->internalRep.doubleValue;
	else {
	    op2_string = Tcl_GetStringFromObj(op2, &op2_slen);
	    if(TclLooksLikeInt(op2_string, op2_slen))
		Tcl_GetLongFromObj(NULL, op2, &op2_lval);
	    else
		Tcl_GetDoubleFromObj(NULL, op2, &op2_dval);
	}

	if(op1->typePtr == kprocPtr->intType && op2->typePtr == kprocPtr->intType) {
	    if(op1_lval < op2_lval) compResult = 1;
	    else                    compResult = 0;
	}
	else if((op1->typePtr == kprocPtr->intType || 
		 op1->typePtr == kprocPtr->doubleType) &&
		(op2->typePtr == kprocPtr->intType || 
		 op2->typePtr == kprocPtr->doubleType))
	{
	    if(op1->typePtr == kprocPtr->intType)
		op1_dval = (double)op1_lval;
	    else if(op2->typePtr == kprocPtr->intType)
		op2_dval = (double)op2_lval;

	    if(op1_lval < op2_lval) compResult = 1;
	    else                    compResult = 0;
	}
	else {
	    op1_string = Tcl_GetStringFromObj(op1, &op1_slen);
	    op2_string = Tcl_GetStringFromObj(op2, &op2_slen);

	    for(i = 0; ; i++) {
		if(i == op1_slen) {
		    if(i == op2_slen) compResult = 0;
		    else              compResult = 1;
		    break;
		}

		if(i == op2_slen) {
		    compResult = 0;
		    break;
		}

		if(op1_string[i] < op2_string[i]) {
		    compResult = 1;
		    break;
			
		}

		if(op1_string[i] > op2_string[i]) {
		    compResult = 0;
		    break;
		}
	    }
	}

	*stackPtr = Tcl_NewIntObj(compResult);
	Tcl_IncrRefCount(*stackPtr);
	stackPtr++;

	Tcl_DecrRefCount(op1);
	Tcl_DecrRefCount(op2);
    }
}

set eq {
    {
	Tcl_Obj *op1,        *op2;
	char    *op1_string, *op2_string;
	int      op1_slen,    op2_slen;
	long     op1_lval,    op2_lval;
	double   op1_dval,    op2_dval;        
	int      op1_type;
	int      op2_type;
	int      compResult;

	op2 = *(--stackPtr);
	op1 = *(--stackPtr);

	/* Use integer comparison if:
	 * - Both arguments have integral type, OR
	 * - Both arguments look like integers.
	 * Otherwise, use double comparison if:
	 * - Both arguments have double or integer type, OR
	 * - Both arguments look like numbers.
	 * Otherwise, use string comparison;
	 */

	if(op1->typePtr == kprocPtr->intType)
	    op1_lval = op1->internalRep.longValue;
	else if(op1->typePtr == kprocPtr->doubleType && op1->bytes == NULL)
	    op1_dval = op1->internalRep.doubleValue;
	else {
	    op1_string = Tcl_GetStringFromObj(op1, &op1_slen);
	    if(TclLooksLikeInt(op1_string, op1_slen))
		Tcl_GetLongFromObj(NULL, op1, &op1_lval);
	    else
		Tcl_GetDoubleFromObj(NULL, op1, &op1_dval);
	}

	if(op2->typePtr == kprocPtr->intType)
	    op2_lval = op2->internalRep.longValue;
	else if(op2->typePtr == kprocPtr->doubleType && op2->bytes == NULL)
	    op2_dval = op2->internalRep.doubleValue;
	else {
	    op2_string = Tcl_GetStringFromObj(op2, &op2_slen);
	    if(TclLooksLikeInt(op2_string, op2_slen))
		Tcl_GetLongFromObj(NULL, op2, &op2_lval);
	    else
		Tcl_GetDoubleFromObj(NULL, op2, &op2_dval);
	}

	if(op1->typePtr == kprocPtr->intType && op2->typePtr == kprocPtr->intType) {
	    if(op1_lval == op2_lval) compResult = 1;
	    else                    compResult = 0;
	}
	else if((op1->typePtr == kprocPtr->intType || 
		 op1->typePtr == kprocPtr->doubleType) &&
		(op2->typePtr == kprocPtr->intType || 
		 op2->typePtr == kprocPtr->doubleType))
	{
	    if(op1->typePtr == kprocPtr->intType)
		op1_dval = (double)op1_lval;
	    else if(op2->typePtr == kprocPtr->intType)
		op2_dval = (double)op2_lval;

	    if(op1_lval == op2_lval) compResult = 1;
	    else                     compResult = 0;
	}
	else {
	    op1_string = Tcl_GetStringFromObj(op1, &op1_slen);
	    op2_string = Tcl_GetStringFromObj(op2, &op2_slen);

	    for(i = 0; ; i++) {
		if(i == op1_slen) {
		    if(i == op2_slen) compResult = 1;
		    else              compResult = 0;
		    break;
		}

		if(i == op2_slen) {
		    compResult = 0;
		    break;
		}

		if(op1_string[i] != op2_string[i]) {
		    compResult = 1;
		    break;
			
		}
	    }
	}

	*stackPtr = Tcl_NewIntObj(compResult);
	Tcl_IncrRefCount(*stackPtr);
	stackPtr++;

	Tcl_DecrRefCount(op1);
	Tcl_DecrRefCount(op2);
    }
}



set beginCatch4 {
    unwindTop++;
    assert(unwindTop < $maxExceptDepth);
    unwindPtrs[unwindTop] = stackPtr;
}


set pushResult {
    assert(stackPtr < stackData + $maxExceptDepth);
    *stackPtr = Tcl_GetObjResult(interp);
    Tcl_IncrRefCount(*stackPtr);
    stackPtr++;
}


set pushReturnCode {
    assert(stackPtr < stackData + $maxExceptDepth);
    *stackPtr = Tcl_NewIntObj(result);
    Tcl_IncrRefCount(*stackPtr);
    stackPtr++;
}

set endCatch {
    unwindTop--;
    result = TCL_OK;
}


set tryCvtToNumeric {
    assert(stackPtr > stackData);
    {
	Tcl_ObjType *tPtr;
	long         ival;
	double       dval;

	tPtr = (*stackPtr)->typePtr;

	if((tPtr != kprocPtr->intType) && ((tPtr != kprocPtr->doubleType) ||
				     ((*stackPtr)->bytes != NULL)))
	{
	    if((tPtr == &tclBooleanType) && ((*stackPtr)->bytes == NULL))
		(*stackPtr)->typePtr = kprocPtr->intType;
	    else {
		char *objString;
		int   slen;

		objString = Tcl_GetStringFromObj(*stackPtr, &slen);

		if(TclLooksLikeInt(objString, slen))
		    result = Tcl_GetLongFromObj(NULL, *stackPtr, &ival);
		else
		    result = Tcl_GetDoubleFromObj(NULL, *stackPtr, &dval);

		result = TCL_OK;
	    }
	    
	    tPtr = (*stackPtr)->typePtr;
	}

	if((tPtr == kprocPtr->intType) || (tPtr == kprocPtr->doubleType)) {
	    if(Tcl_IsShared(*stackPtr)) {
		if((*stackPtr)->bytes != NULL) {
		    Tcl_Obj *newobj;

		    /* Make a copy */
		    if(tPtr == kprocPtr->intType)
			newobj = Tcl_NewLongObj((*stackPtr)->internalRep.
						longValue);
		    else
			newobj = Tcl_NewDoubleObj((*stackPtr)->internalRep.
						  doubleValue);
		    
		    Tcl_IncrRefCount(newobj);
		    Tcl_DecrRefCount(*stackPtr);
		    *stackPtr = newobj;
		}
	    } else
		Tcl_InvalidateStringRep(*stackPtr);
	 
	    /* If we have a double, make sure it's valid */
	    if(tPtr == kprocPtr->doubleType) {
		dval = (*stackPtr)->internalRep.doubleValue;
		if((dval != dval) || (dval > DBL_MAX) || (dval < -DBL_MAX)) {
		    TclExprFloatError(interp, dval);
		    result = TCL_ERROR;
		    $catchJump;
		}
	    }
	}
    }
}


set __epilogue {
 abnormalReturn:
    while(stackPtr > stackData) {
	stackPtr--;
	Tcl_DecrRefCount(*stackPtr);
    }

 procDone:
    Tcl_PopCallFrame(interp);

    /* Last minute changes to the result code */
    switch(result) {
	case TCL_RETURN:
	{
	    int code = iPtr->returnCode;
	    iPtr->returnCode = TCL_OK;

	    if(code == TCL_ERROR) {
		Tcl_SetVar2(interp, "errorCode", NULL, 
			    (iPtr->errorCode != NULL) ? 
			    iPtr->errorCode : "NONE", TCL_GLOBAL_ONLY);
		iPtr->flags |= ERROR_CODE_SET;
		if(iPtr->errorInfo != NULL) {
		    Tcl_SetVar2(interp, "errorInfo", NULL, iPtr->errorInfo,
				TCL_GLOBAL_ONLY);
		    iPtr->flags |= ERR_IN_PROGRESS;
		}
	    }

	    result = code;
	    break;
	}

	case TCL_BREAK:
	Tcl_ResetResult(interp);
	Tcl_AppendToObj(Tcl_GetObjResult(interp), 
			"invoked \"break\" outside of a loop", -1);
	result = TCL_ERROR;
	break;

	case TCL_CONTINUE:
	Tcl_ResetResult(interp);
	Tcl_AppendToObj(Tcl_GetObjResult(interp), 
			"invoked \"continue\" outside of a loop", -1);
	result = TCL_ERROR;
	break;

	default:
	/* do nothing */
	break;
    }
	
    return result;
}



set __declaration {
    /* Create ${procName}.
     * Be sure you've included tclInt.h and tcl.h.
     */

    int ${function}(ClientData, Tcl_Interp *, int, Tcl_Obj* const []);
    Interp    *iPtr = (Interp *)interp;
    int        result;
    char      *procName = "$procName";
    char      *simpleName;
    Namespace *nsPtr;
    Namespace *altNsPtr;
    Namespace *cxtNsPtr;
#if ($numCompiledLocals > 0)
    CompiledLocal *clArray[$numCompiledLocals];
#else
    CompiledLocal *clArray[1];
#endif
    Proc      *procPtr;
    KangaProc *kprocPtr;

    /* Implementation */
    TclGetNamespaceForQualName(interp, procName, 0, CREATE_NS_IF_UNKNOWN,
			       &nsPtr, &altNsPtr, &cxtNsPtr, &simpleName);

    if(nsPtr == NULL) {
	Tcl_ResetResult(interp);
	Tcl_AppendToObj(Tcl_GetObjResult(interp),
			"can't create procedure \"$procName\": unknown "
			"namespace", -1);
	return TCL_ERROR;
    }

    if(simpleName == NULL) {
	Tcl_ResetResult(interp);
	Tcl_AppendToObj(Tcl_GetObjResult(interp),
			"can't create procedure \"$procName\": bad "
			"procedure name", -1);
	return TCL_ERROR;
    }

    /* Fake a proc structure */
    procPtr = (Proc *)ckalloc(sizeof(Proc));
    procPtr->iPtr = iPtr;
    procPtr->refCount = 1;
    procPtr->bodyPtr = NULL;
    procPtr->numArgs = $argLength;
    procPtr->numCompiledLocals = $numCompiledLocals;

    $compiledLocalInitialisation;

    procPtr->cmdPtr = NULL;
    procPtr->firstLocalPtr = clArray[0];
    procPtr->lastLocalPtr = clArray[$numCompiledLocals - 1];

    /* Allocate the KangaProc structure */
    kprocPtr = (KangaProc *)ckalloc(sizeof(KangaProc));
    kprocPtr->nsPtr = nsPtr;
    kprocPtr->procPtr = procPtr;
    kprocPtr->literals = (Tcl_Obj **)ckalloc(sizeof(Tcl_Obj *) * $numLiterals);
    kprocPtr->intType = Tcl_GetObjType("int");
    kprocPtr->doubleType = Tcl_GetObjType("double");
    kprocPtr->listType = Tcl_GetObjType("list");

    /* Create the command */
    procPtr->cmdPtr = (Command *)
	Tcl_CreateObjCommand(interp, procName, $function,
			     (ClientData)kprocPtr, 0);
}

set __clallocate {
    clArray[$clIndex] = (CompiledLocal *)ckalloc(sizeof(CompiledLocal) + $clNameLength);}

set __clinitialise {
    clArray[$clIndex]->nextPtr = ($clIndex == $numCompiledLocals - 1 ? NULL : clArray[$clIndex + 1]);
    clArray[$clIndex]->nameLength = $clNameLength;
    clArray[$clIndex]->flags = $clFlags;
    clArray[$clIndex]->defValuePtr = NULL;
    clArray[$clIndex]->resolveInfo = NULL;
    strcpy(clArray[$clIndex]->name, "$clName");
}

set __literal_create {
    kprocPtr->literals[$literalIndex] = $createLiteral($literalIndex);
    Tcl_IncrRefCount(kprocPtr->literals[$literalIndex]);
}

