# i18n.tcl --
#
#       Demonstrates internationalization (i18n) features of Tix
#       widgets.
#
# This sample program is structured in such a way so that it can be
# executed from the Tix demo program "widget": it must have a
# procedure called "RunSample". It should also have the "if" statment
# at the end of this file so that it can be run as a standalone
# program using tixwish.
#
# $Id: i18n.tcl,v 1.1 2001/03/03 09:27:55 ioilam Exp $

namespace eval ::i18n {

}

set langdata {
   "Chinese Simplified"
    "\u7b80\u4f53\u4e2d\u6587"
    "\u4f60\u597d"
   "Chinese Traditional"
    "\u7e41\u9ad4\u4e2d\u6587"
    "\u4f60\u597d"
   Czech
    "\u010desky"
    "Dobr\u00fd den"
   Danish
    Dansk
    "Hej, Goddag"
   English
    English
    Hello
   Finnish
    Suomi
    Hei
   French
    "Fran\u00e7ais"
    "Bonjour, Salut"
   German
    Deutsch
    "Guten Tag"
   Greek
    "\u0395\u03bb\u03bb\u03b7\u03bd\u03b9\u03ba\u03ac"
    "\u0393\u03b5\u03b9\u03ac \u03c3\u03b1\u03c2"
   Italian
    Italiano
    "Ciao, Buon giorno"
   Japanese
    "\u65e5\u672c\u8a9e"
    "\u3053\u3093\u306b\u3061\u306f"
   Korean
    "\ud55c\uae00"
    "\uc548\ub155\ud558\uc138\uc694, \uc548\ub155\ud558\uc2ed\ub2c8\uae4c"
   Norwegian
    Norsk
    "Hei, God dag"
   Russian
    "\u0420\u0443\u0441\u0441\u043a\u0438\u0439"
    "\u0417\u0434\u0440\u0430\u0432\u0441\u0442\u0432\u0443\u0439\u0442\u0435!"
   Spanish
    "Espa\u00f1ol"
    "\u00a1Hola!"
   Swedish
    Svenska
    "Hej, Goddag"
   Turkish
    "T\u00fcrk\u00e7e"
    Merhaba
}

foreach {lang ln h} $langdata {
    set lang_native($lang) [subst $ln]
    set lang_hello($lang)  [subst $h]

}

# i18n::RunSample --
#
#       Create the main window for this sample program. This procedure
#       is called either by main demo program "widget", or by the
#       block at the end of this file.
#
# Arguments:
#   w           Name of the top-level (already created) to use as
#               the main window of this sample program
# Results:
#   Returns nothing.
#
proc i18n::RunSample {w} {
    global langdata lang_native lang_hello tcl_platform

    if {$tcl_platform(platform) == "windows"} {
        #
        # The Windows default font is too small to display Asian
        # characters. Let's choose a bigger font
        #

        option add *[winfo name $w]*font {Arial 12}
    }

    set notebook [tixNoteBook $w.nb]
    pack $w.nb -padx 4 -pady 4 -expand yes -fill both

    $notebook add p1 -label $lang_hello(English)
    $notebook add p2 -label $lang_hello(Chinese Simplified) \
        -createcmd [list i18n::mkPage2 $notebook p2]
    $notebook add p3 -label $lang_hello(Swedish) \
        -createcmd [list i18n::mkPage3 $notebook p3]

    i18n::mkPage1 $notebook p1
}

proc i18n::mkPage1 {notebook pageName} {
    set p [$notebook subwidget $pageName]
    tixScrolledText $p.st -options {
        text.width  40
        text.height 15
        text.wrap   word
    }
    pack $p.st -expand yes -fill both -padx 6 -pady 6

    set intro {
        This sample program creates Tix widgets that display the words
        for "hello" in many languages of the world. Please click on
        the tabs above to begin. <p>

        You may have to wait for a few seconds for the preparation of
        the fonts of the different language. Please be patient. <p>

        The fonts of some languages may not be supported in your
        system, so you may see that words of these languages are
        replaced by rectangles or Unicode values such as \u1234. <p>

    }

    regsub -all "\[\t \]*\n\[\t \]*" $intro "" intro
    regsub -all "<p>" $intro \n\n intro

    $p.st subwidget text insert 1.0 $intro
}

proc i18n::mkPage2 {notebook pageName} {
    global lang_native lang_hello

    set p [$notebook subwidget $pageName]
    set tree [tixTree $p.tree]
    pack $tree -side left -padx 6 -pady 6 -expand yes -fill both
    set st [tixScrolledTList $p.st]
    pack $st -side left -padx 6 -pady 6 -expand yes -fill both

    set hl [$tree subwidget hlist]
    set tl [$st subwidget tlist]

    foreach lang [lsort [array names lang_native]] {
        $hl add $lang -itemtype imagetext \
            -text $lang \
            -image [tix getimage openfold]
        $hl add $lang.1 -itemtype imagetext \
            -text $lang_native($lang) \
            -image [tix getimage textfile]
        $hl add $lang.2 -itemtype imagetext \
            -text $lang_hello($lang) \
            -image [tix getimage textfile]

        $tl insert end -itemtype imagetext -text $lang \
	    -image [tix getimage openfold]
        $tl insert end -itemtype imagetext -text $lang_native($lang)\
	    -image [tix getimage openfold]
        $tl insert end -itemtype imagetext -text $lang_hello($lang) \
	    -image [tix getimage openfold]
    }
    $tree autosetmode


}

proc i18n::mkPage3 {notebook pageName} {
    global lang_native lang_hello

    set p [$notebook subwidget $pageName]
}

# This "if" statement makes it possible to run this script file inside or
# outside of the main demo program "widget".
#
if {![info exists tix_demo_running]} {
    # This example uses other scripts auto-loaded from this
    # directory:
    lappend auto_path [file dirname [info script]]

    # This example uses bitmaps from ../bitmaps.
    set demo_dir [file join [file dirname [info script]] ..]
    tix addbitmapdir [file join $demo_dir bitmaps]

    wm withdraw .
    set w .demo
    toplevel $w; wm transient $w ""
    i18n::RunSample $w
    bind $w <Destroy> {after 10 exit}
}

