/*
 * tkMegawidget.c --
 *
 *	Implementation of a tk "megawidget" command, which provides Tcl-level
 *	access to the C level widget management features (such as option
 *	management).
 *
 * Copyright (c) 2000 by Scriptics Corporation.
 * Copyright (c) 2001 by Eric Melski.
 * Copyright (c) 2001 by Jeffrey Hobbs.
 *
 * RCS: @(#) $Id: tkMegawidget.c,v 1.2 2001/12/19 20:42:13 hobbs Exp $
 */

#include <tk.h>
#include <string.h>

static Tcl_HashTable *classes = NULL;
static Tcl_HashTable *instances = NULL;
static Tcl_HashTable *stringTables = NULL;

typedef struct Widget {
    char *name;			/* Dynamically allocated class name. */
    int numOptions;		/* Number of options in the option table. */
    Tk_OptionSpec *optionSpec;	/* Dynamically allocated option spec table. */
    Tk_OptionTable optionTable;	/* Option table. */
} Widget;

typedef struct Instance {
    char *name;			/* Dynamically allocated widget name. */
    char *class;		/* Dynamically allocated class name. */
    Tk_OptionTable optionTable;	/* Pointer to the class option table. */
    Tk_Window tkwin;		/* Tk window for this instance. */
    Tcl_Obj **options;		/* Dynamically allocated array of option
				 * values. */
} Instance;


/*
 *----------------------------------------------------------------------
 *
 * Tk_WidgetObjCmd --
 *
 *	The "megawidget" command procedure.  Used to create new classes
 *	of widgets.
 *
 * Results:
 *	Standard Tcl result.
 *
 * Side effects:
 *	Depends on subcommand; may allocate memory; may free memory.
 *
 *----------------------------------------------------------------------
 */

int
Tk_MegawidgetObjCmd(dummy, interp, objc, objv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int objc;				/* Number of arguments. */
    Tcl_Obj *CONST objv[];		/* Argument objects. */
{
    int index;
    int result;
    Tcl_HashEntry *entry, *stringTableEntry;
    Tcl_Obj **elements;
    Widget *mw;
    Instance *w;
    int numElements, i, new;
    unsigned int length;
    char *widgetClass, *window;
    Tcl_Obj *objPtr;
    static char *commands[] = {
	"configure", "cget", "create", "declare", "stringtable", (char *) NULL
    };
    enum commands {
	WIDGET_CONFIGURE, WIDGET_CGET, WIDGET_CREATE,
	WIDGET_DECLARE, WIDGET_STRINGTABLE
    };

    static char *optionTypes[] = {
	"anchor", "bitmap", "boolean", "border", "color", "cursor", "double",
	"font", "int", "justify", "pixels", "relief", "string",
	"stringtable", "synonym", "window", (char *) NULL
    };

    static int types[] = {
	TK_OPTION_ANCHOR, TK_OPTION_BITMAP, TK_OPTION_BOOLEAN,
	TK_OPTION_BORDER, TK_OPTION_COLOR, TK_OPTION_CURSOR,
	TK_OPTION_DOUBLE, TK_OPTION_FONT, TK_OPTION_INT, TK_OPTION_JUSTIFY,
	TK_OPTION_PIXELS, TK_OPTION_RELIEF, TK_OPTION_STRING,
	TK_OPTION_STRING_TABLE, TK_OPTION_SYNONYM, TK_OPTION_WINDOW
    };

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option arg ?arg ...?");
	return TCL_ERROR;
    }

    result = Tcl_GetIndexFromObj(interp, objv[1], commands, "option", 0,
	    &index);
    if (result != TCL_OK) {
	return TCL_ERROR;
    }

    if (classes == NULL) {
	classes = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
	Tcl_InitHashTable(classes, TCL_STRING_KEYS);

	instances = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
	Tcl_InitHashTable(instances, TCL_STRING_KEYS);

	stringTables = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
	Tcl_InitHashTable(stringTables, TCL_STRING_KEYS);
    }

    switch ((enum commands) index) {
	case WIDGET_CONFIGURE: {
	    Tk_SavedOptions savedOptions;
	    window = Tcl_GetString(objv[2]);
	    entry = Tcl_FindHashEntry(instances, window);
	    if (entry == NULL) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "bad window path name \"",
			window, "\"", (char *)NULL);
		return TCL_ERROR;
	    }
	    w = Tcl_GetHashValue(entry);

	    if (objc <= 4) {
		objPtr = Tk_GetOptionInfo(interp, (char *)w->options,
			w->optionTable,
			(objc == 4) ? objv[3] : (Tcl_Obj *) NULL, w->tkwin);
		if (objPtr == NULL) {
		    return TCL_ERROR;
		} else {
		    Tcl_SetObjResult(interp, objPtr);
		}
	    } else {
		objc -= 3;
		objv += 3;
		for (i = 0; i <= 1; i++) {
		    if (!i) {
			if (Tk_SetOptions(interp, (char *)w->options,
				w->optionTable, objc, objv, w->tkwin,
				&savedOptions, (int *)NULL) != TCL_OK) {
			    continue;
			}
		    } else {
			objPtr = Tcl_GetObjResult(interp);
			Tcl_IncrRefCount(objPtr);
			Tk_RestoreSavedOptions(&savedOptions);
			return TCL_ERROR;
		    }
		    break;
		}
	    }
	    break;
	}

	case WIDGET_CGET: {
	    if (objc != 4) {
		if (objc == 3) {
		    Tcl_WrongNumArgs(interp, 3, objv, "option");
		} else {
		    Tcl_WrongNumArgs(interp, 2, objv, "widget option");
		}
		return TCL_ERROR;
	    }

	    window = Tcl_GetString(objv[2]);
	    entry = Tcl_FindHashEntry(instances, window);
	    if (entry == NULL) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "bad window path name \"",
			window, "\"", (char *)NULL);
		return TCL_ERROR;
	    }
	    w = Tcl_GetHashValue(entry);

	    objPtr = Tk_GetOptionValue(interp, (char *)w->options,
		    w->optionTable, objv[3], w->tkwin);
	    if (objPtr == NULL) {
		return TCL_ERROR;
	    } else {
		Tcl_SetObjResult(interp, objPtr);
	    }
	    break;
	}

	case WIDGET_CREATE: {
	    /*
	     * Create a new widget by allocating an option value array
	     * for it, associating that with a window path, initializing the
	     * options, and doing any needed configuration.
	     */
	    if (objc != 4) {
		Tcl_WrongNumArgs(interp, 2, objv, "class window");
		return TCL_ERROR;
	    }

	    widgetClass = Tcl_GetString(objv[2]);
	    window = Tcl_GetString(objv[3]);

	    entry = Tcl_FindHashEntry(classes, widgetClass);
	    if (entry == NULL) {
		Tcl_AppendResult(interp, "unknown class \"", widgetClass,
			"\"", (char *) NULL);
		return TCL_ERROR;
	    }

	    mw = (Widget *) Tcl_GetHashValue(entry);
	    
	    entry = Tcl_CreateHashEntry(instances, window, &new);
	    if (!new) {
		Tcl_AppendResult(interp, "window \"", window,
			"\" already exists", (char *) NULL);
		return TCL_ERROR;
	    }

	    w = (Instance *) ckalloc(sizeof(Instance));
	    w->name = strdup(window);
	    w->class = strdup(widgetClass);
	    w->options = (Tcl_Obj **)
		ckalloc(sizeof(Tcl_Obj *) *mw->numOptions);
	    w->optionTable = mw->optionTable;
	    w->tkwin = Tk_NameToWindow(interp, window, Tk_MainWindow(interp));
	    for (i = 0; i < mw->numOptions; i++) {
		w->options[i] = NULL;
	    }

	    Tcl_SetHashValue(entry, (ClientData) w);

	    result = Tk_InitOptions(interp, (char *)w->options,
		    w->optionTable, w->tkwin);
	    break;
	}
	
	case WIDGET_DECLARE: {
	    Tcl_Obj *currentOption;
	    Tcl_Obj **optionFields;
	    char *str;
	    int numFields, optionType;
	    /*
	     * Declare a new widget class by declaring a Tk option table
	     * for it.  This will create the option table and store it in
	     * a hash table indexed by the class name.
	     *
	     * An optionSpec is a list of 8-tuples of the form:
	     *   { optionType optionName dbName dbClass
	     *       defaultValue flags clientData intTypeMask }
	     * 'flags' currently not used, but should handle:
	     *		nullok | nodefault
	     * synonyms are create like so:
	     *   { synonym optionName "" "" "" "" realOptionName 0 }
	     */
	    if (objc != 4) {
		Tcl_WrongNumArgs(interp, 2, objv, "classname optionspec");
		return TCL_ERROR;
	    }
	    widgetClass = Tcl_GetString(objv[2]);
	    entry = Tcl_CreateHashEntry(classes, widgetClass, &new);
	    if (!new) {
		Tcl_SetResult(interp, "class already defined", TCL_STATIC);
		return TCL_ERROR;
	    }
	    if (Tcl_ListObjLength(interp, objv[3], &numElements) != TCL_OK) {
		return TCL_ERROR;
	    }

	    if (Tcl_ListObjGetElements(interp, objv[3], &numElements,
		    &elements) != TCL_OK) {
		return TCL_ERROR;
	    }

	    mw = (Widget *) ckalloc(sizeof(Widget));
	    mw->name = strdup(widgetClass);
	    mw->numOptions = numElements;
	    numFields = sizeof(Tk_OptionSpec) * (numElements + 1);
	    mw->optionSpec = (Tk_OptionSpec *)
		ckalloc((unsigned int) numFields);
	    memset((VOID *) mw->optionSpec, 0, (unsigned int) numFields);

	    /*
	     * Put the option terminator on the end of the option spec.
	     * All others fields will be set to 0/NULL by memset.
	     */
	    mw->optionSpec[numElements].type		= TK_OPTION_END;
	    mw->optionSpec[numElements].internalOffset	= -1;

	    result = TCL_OK;
	    for (i = 0; i < numElements; i++) {
		currentOption = elements[i];
		/*
		 * Each element of the list should be a proper list itself.
		 */
		result = Tcl_ListObjLength(interp, currentOption, &numFields);
		if (result != TCL_OK) {
		    break;
		}
		if (numFields != 8) {
		    char buf[32 + TCL_INTEGER_SPACE];
		    sprintf(buf, "wrong # elements in option %d", i);
		    Tcl_SetResult(interp, buf, TCL_VOLATILE);
		    result = TCL_ERROR;
		    break;
		}
		Tcl_ListObjGetElements(interp, currentOption, &numFields,
			&optionFields);

		/*
		 * Extract the type of the current option.
		 */

		result = Tcl_GetIndexFromObj(interp, optionFields[0],
			optionTypes, "type", 0, &optionType);
		if (result != TCL_OK) {
		    break;
		}
		mw->optionSpec[i].type = types[optionType];

		/*
		 * Do a sanity check based on type and default value.
		 */
		objPtr = optionFields[4];
		switch (types[optionType]) {
		    case TK_OPTION_BOOLEAN:
			result = Tcl_GetBooleanFromObj(interp, objPtr, &new);
			break;
		    case TK_OPTION_INT:
			result = Tcl_GetIntFromObj(interp, objPtr, &new);
			break;
		    case TK_OPTION_DOUBLE:
			result = Tcl_GetDoubleFromObj(interp, objPtr, &new);
			break;
#ifdef NOT_YET
			/*
			 * We don't have a tkwin to operate on here.
			 */
		    case TK_OPTION_COLOR:
			if (Tk_AllocColorFromObj(interp, tkwin, objPtr)
				== NULL) {
			    result = TCL_ERROR;
			}
			break;
		    case TK_OPTION_FONT:
			if (Tk_AllocFontFromObj(interp, tkwin, objPtr)
				== NULL) {
			    result = TCL_ERROR;
			}
			break;
		    case TK_OPTION_BITMAP:
			if (Tk_AllocBitmapFromObj(interp, tkwin, objPtr)
				== None) {
			    result = TCL_ERROR;
			}
			break;
		    case TK_OPTION_BORDER:
			if (Tk_Alloc3DBorderFromObj(interp, tkwin, objPtr)
				== NULL) {
			    result = TCL_ERROR;
			}
			break;
		    case TK_OPTION_CURSOR:
			if (Tk_AllocCursorFromObj(interp, tkwin, objPtr)
				== None) {
			    result = TCL_ERROR;
			}
			break;
#endif
		    case TK_OPTION_RELIEF:
			result = Tk_GetReliefFromObj(interp, objPtr, &new);
			break;
		    case TK_OPTION_JUSTIFY:
			result = Tk_GetJustifyFromObj(interp, objPtr,
				(Tk_Justify *) &new);
			break;
		    case TK_OPTION_ANCHOR:
			result = Tk_GetAnchorFromObj(interp, objPtr,
				(Tk_Anchor *) &new);
			break;
		    case TK_OPTION_PIXELS:
			result = Tk_GetPixelsFromObj(interp, tkwin, objPtr,
				&new);
			break;
#if 0
		    case TK_OPTION_STRING: /* That's always OK */
		    case TK_OPTION_STRING_TABLE: /* Handled below */
		    case TK_OPTION_WINDOW: /* We let this pass */
#endif
		    default:
			break;
		}
		if (result != TCL_OK) {
		    break;
		}

		/*
		 * Extract the option name, database name and database class
		 * for the current option.
		 */

		str = Tcl_GetStringFromObj(optionFields[1], &length);
		mw->optionSpec[i].optionName = (char *) ckalloc(length + 1);
		strcpy(mw->optionSpec[i].optionName, str);

		str = Tcl_GetStringFromObj(optionFields[2], &length);
		mw->optionSpec[i].dbName = (char *) ckalloc(length + 1);
		strcpy(mw->optionSpec[i].dbName, str);

		str = Tcl_GetStringFromObj(optionFields[3], &length);
		mw->optionSpec[i].dbClass = (char *) ckalloc(length + 1);
		strcpy(mw->optionSpec[i].dbClass, str);
		
		/*
		 * Extract the default value field.
		 */
		
		str = Tcl_GetStringFromObj(optionFields[4], &length);
		mw->optionSpec[i].defValue = (char *) ckalloc(length + 1);
		strcpy(mw->optionSpec[i].defValue, str);

		/*
		 * Because this structure is dynamically created, we can't
		 * really have typed storage for the fields.  Instead they will
		 * all be stored in Tcl_Obj's.  Set the offset here.  The
		 * actual storage will be allocated later when a widget
		 * of the appropriate class is declared.
		 */
		mw->optionSpec[i].objOffset = i * sizeof(Tcl_Obj *);
		mw->optionSpec[i].internalOffset = -1;

		mw->optionSpec[i].flags = 0;
		
		/*
		 * Extract the ClientData field (may be used if the option is
		 * a synonym or a color).
		 */
		str = Tcl_GetStringFromObj(optionFields[6], &length);
		if (mw->optionSpec[i].type == TK_OPTION_STRING_TABLE) {
		    /*
		     * The client data field names a string table to use.
		     */
		    stringTableEntry = Tcl_FindHashEntry(stringTables, str);
		    if (stringTableEntry == NULL) {
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp, "string table \"",
				str, "\" does not exist", (char *)NULL);
			result = TCL_ERROR;
			break;
		    }
		    mw->optionSpec[i].clientData =
			(ClientData) Tcl_GetHashValue(stringTableEntry);
		} else {
		    if (length) {
			mw->optionSpec[i].clientData =
			    (ClientData) ckalloc(length + 1);
			strcpy(mw->optionSpec[i].clientData, str);
		    }
		}

		/*
		 * Extract the type mask field.
		 */
		result = Tcl_GetIntFromObj(interp, optionFields[7],
			&mw->optionSpec[i].typeMask);
		if (result != TCL_OK) {
		    break;
		}
	    }
	    if (result != TCL_OK) {
		/*
		 * We need to unwind whatever we have created, freeing
		 * any memory, and unregistering the class.
		 */
		for ( ; i >= 0 ; i--) {
		    if (mw->optionSpec[i].optionName != NULL) {
			ckfree(mw->optionSpec[i].optionName);
		    }
		    if (mw->optionSpec[i].dbName != NULL) {
			ckfree(mw->optionSpec[i].dbName);
		    }
		    if (mw->optionSpec[i].dbClass != NULL) {
			ckfree(mw->optionSpec[i].dbClass);
		    }
		    if (mw->optionSpec[i].defValue != NULL) {
			ckfree(mw->optionSpec[i].defValue);
		    }
		    if ((mw->optionSpec[i].type != TK_OPTION_STRING_TABLE)
			    && (mw->optionSpec[i].clientData != NULL)) {
			ckfree(mw->optionSpec[i].clientData);
		    }
		}

		ckfree((char *) mw->name);
		ckfree((char *) mw->optionSpec);
		ckfree((char *) mw);
		Tcl_DeleteHashEntry(entry);

		return TCL_ERROR;
	    }

	    mw->optionTable = Tk_CreateOptionTable(interp, mw->optionSpec);

	    /*
	     * Add this option table to the option tables hash table.
	     */
	    Tcl_SetHashValue(entry, (ClientData) mw);
	    break;
	}
	
	case WIDGET_STRINGTABLE: {
	    char *str;
	    char **table;
	    
	    /*
	     * Create a named string table for enumerations.
	     */

	    if (objc != 4 && objc != 3) {
		Tcl_WrongNumArgs(interp, 2, objv, "table ?stringlist?");
		return TCL_ERROR;
	    }

	    /*
	     * Get the table name.
	     */
	    str = Tcl_GetString(objv[2]);
	    if (objc == 3) {
		/*
		 * Just querying the table's contents.
		 */
		entry = Tcl_FindHashEntry(stringTables, str);
		if (entry == NULL) {
		    Tcl_ResetResult(interp);
		    Tcl_AppendResult(interp, "unknown string table \"",
			    str, "\"", (char *)NULL);
		    return TCL_ERROR;
		}
		objPtr = Tcl_NewObj();
		table = (char **)Tcl_GetHashValue(entry);
		while (*table != NULL) {
		    Tcl_ListObjAppendElement(interp, objPtr,
			    Tcl_NewStringObj(*table, -1));
		    table++;
		}
		Tcl_SetObjResult(interp, objPtr);
	    } else {
		/*
		 * Creating a new table.
		 */
		entry = Tcl_CreateHashEntry(stringTables, str, &new);
		if (!new) {
		    Tcl_ResetResult(interp);
		    Tcl_AppendResult(interp, "string table \"",
			    str, "\" already defined", (char *)NULL);
		    return TCL_ERROR;
		}
		
		result = Tcl_ListObjLength(interp, objv[3], &length);
		if (result != TCL_OK) {
		    return TCL_ERROR;
		}
		
		table = (char **) ckalloc(sizeof(char *) * (length + 1));
		table[length] = NULL;
		Tcl_ListObjGetElements(interp, objv[3], &length, &elements);
		for (i = 0; i < length; i++) {
		    table[i] = strdup(Tcl_GetString(elements[i]));
		}
		
		Tcl_SetHashValue(entry, table);
	    }
	    break;
	}
    }
    return TCL_OK;
}
