
/*
 * bltBgStyle.c --
 *
 * This module creates background styles for the BLT toolkit.
 *
 *	Copyright 1995-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"
#include "bltChain.h"
#include "bltHash.h"
#include "bltPicture.h"
#include "bltPainter.h"
#include <X11/Xutil.h>

#define BG_STYLE_THREAD_KEY	"BLT Background Style Data"

extern Pix32 Blt_XColorToPix32(XColor *colorPtr);

typedef struct BackgroundStruct Background;
typedef struct BgTokenStruct BgToken;

/* 
  bgstyle create tile 
	-relativeto self|toplevel|window
	-picture $image
	-bg $color
  bgstyle create picture 
        -picture $image
	-filter $filterName
	-bg $color
  bgstyle create gradient 
	-type radial|xlinear|ylinear|diagonal
	-low $color
	-high $color
	-bg $color
  bgstyle create border 
	-bg $color

  bgstyle create texture -type metal|wind|??? 
	-bg $color

  bgstyle names
  bgstyle configure $tile
  bgstyle delete $tile
*/


enum StyleTypes {
    STYLE_BORDER,		/* Standard 3D border. */
    STYLE_PICTURE,		/* Resizable color picture. */
    STYLE_TILE,			/* Color picture tiled. */
    STYLE_GRADIENT,		/* Color gradient. */
    STYLE_TEXTURE,		/* Procedural texture. */
};

static char *styleTypes[] = {
    "border",
    "picture",
    "tile",
    "gradient",
    "texture"
};

enum ReferenceTypes {
    REFERENCE_SELF,		/* Current window. */
    REFERENCE_TOPLEVEL,		/* Toplevel of current window. */
    REFERENCE_WINDOW,		/* Specifically named window. */
};

typedef struct {
    Blt_HashTable styleTable;	/* Hash table of tile structures keyed by 
				 * the name of the image. */

    Tcl_Interp *interp;		/* Interpreter associated with this
				 * set of background styles. */

    int nextId;			/* Serial number of the identifier to
				 * be used for next background style
				 * created.  */
} BackgroundInterpData;


struct BgTokenStruct {
    Background *bgPtr;		/* Pointer to master background
				 * style. */

    Blt_BackgroundChangedProc *notifyProc;

    ClientData clientData;	/* Data to be passed on notifier
				 * callbacks.  */

    Blt_ChainLink *linkPtr;	/* Link of this entry in notifier list. */

};


struct BackgroundStruct {
    char *name;			/* Generated name of background
				 * style. */

    int style;			/* Type of background style: border,
				 * tile, texture, or gradient. */

    Blt_ConfigSpec *configSpecs;


    BackgroundInterpData *dataPtr;

    Tk_Window tkwin;		/* Main window. Used to query
				 * background style options. */

    Display *display;		/* Display of this background style. */

    int flags;			/* See definitions below. */

    Blt_HashEntry *hashPtr;	/*  */

    Tk_3DBorder border;		/* 3D Border.  May be used for all
				 * background types. */

    
    Blt_HashTable cacheTable;	/* Table of cached pictures for each
				 * background reference. */

    char *origName;		/* Name of original picture. */

    Blt_Picture original;	/* Original (not resampled) picture
				 * for "picture" and "tile"
				 * backgrounds. */

    Blt_PictureFilter filter;	/* 1D image filter to use to when
				 * resampling original picture. */

    Blt_Gradient gradient;

    XColor *color1, *color2;	/* Gradient colors. */


    int reference;		/* "self", "toplevel", or "window". */

    int xOrigin, yOrigin;

    Tk_Window refWindow;	/* Refer to coordinates in this window
				 * when determining the tile
				 * origin. */
    
    Blt_Chain *chainPtr;	/* List of style tokens.  Used to
				 * register callbacks for each client
				 * of the background style. */
};

#define DEF_BORDER		STD_NORMAL_BACKGROUND
#define DEF_GRADIENT_PATH	"y"
#define DEF_GRADIENT_HIGH	"grey90"
#define DEF_GRADIENT_JITTER	"no"
#define DEF_GRADIENT_LOGSCALE	"yes"
#define DEF_GRADIENT_LOW	"grey50"
#define DEF_GRADIENT_MODE	"xlinear"
#define DEF_GRADIENT_SHAPE	"linear"
#define DEF_REFERENCE		"toplevel"
#define DEF_RESAMPLE_FILTER	"box"

static Blt_OptionParseProc ObjToPicture;
static Blt_OptionPrintProc PictureToObj;
static Blt_CustomOption pictureOption =
{
    ObjToPicture, PictureToObj, NULL, (ClientData)0
};

static Blt_OptionParseProc ObjToFilter;
static Blt_OptionPrintProc FilterToObj;
static Blt_CustomOption filterOption =
{
    ObjToFilter, FilterToObj, NULL, (ClientData)0
};

static Blt_OptionParseProc ObjToReference;
static Blt_OptionPrintProc ReferenceToObj;
static Blt_CustomOption referenceToOption =
{
    ObjToReference, ReferenceToObj, NULL, (ClientData)0
};

static Blt_OptionParseProc ObjToShape;
static Blt_OptionPrintProc ShapeToObj;
static Blt_CustomOption shapeOption =
{
    ObjToShape, ShapeToObj, NULL, (ClientData)0
};

static Blt_OptionParseProc ObjToPath;
static Blt_OptionPrintProc PathToObj;
static Blt_CustomOption pathOption =
{
    ObjToPath, PathToObj, NULL, (ClientData)0
};

static Blt_ConfigSpec borderConfigSpecs[] =
{
    {BLT_CONFIG_BORDER, "-background", "background", "Background", DEF_BORDER, 
	Blt_Offset(Background, border), 0},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL, (char *)NULL, 0, 0},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

static Blt_ConfigSpec tileConfigSpecs[] =
{
    {BLT_CONFIG_BORDER, "-background", "background", "Background", DEF_BORDER, 
	Blt_Offset(Background, border), 0},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL, (char *)NULL, 0, 0},
    {BLT_CONFIG_CUSTOM, "-picture", "picture", "Picture", (char *)NULL,
        Blt_Offset(Background, original), BLT_CONFIG_DONT_SET_DEFAULT, 
	&pictureOption},
    {BLT_CONFIG_CUSTOM, "-relativeto", "relativeTo", "RelativeTo", 
	DEF_REFERENCE, Blt_Offset(Background, reference), 
	BLT_CONFIG_DONT_SET_DEFAULT, &referenceToOption},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

static Blt_ConfigSpec pictureConfigSpecs[] =
{
    {BLT_CONFIG_BORDER, "-background", "background", "Background", DEF_BORDER, 
	Blt_Offset(Background, border), 0},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL, (char *)NULL, 0, 0},
    {BLT_CONFIG_CUSTOM, "-filter", "filter", "Filter", DEF_RESAMPLE_FILTER,
        Blt_Offset(Background, filter), 0, &filterOption},
    {BLT_CONFIG_CUSTOM, "-picture", "picture", "Picture", (char *)NULL,
        Blt_Offset(Background, original), BLT_CONFIG_DONT_SET_DEFAULT, 
	&pictureOption},
    {BLT_CONFIG_CUSTOM, "-relativeto", "relativeTo", "RelativeTo", 
	DEF_REFERENCE, Blt_Offset(Background, reference), 
	BLT_CONFIG_DONT_SET_DEFAULT, &referenceToOption},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

static Blt_ConfigSpec gradientConfigSpecs[] =
{
    {BLT_CONFIG_BORDER, "-background", "background", "Background", DEF_BORDER,
	Blt_Offset(Background, border), 0},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL, (char *)NULL, 0, 0},
    {BLT_CONFIG_CUSTOM, "-direction", "direction", "Direction", 
	DEF_GRADIENT_PATH, Blt_Offset(Background, gradient.path), 
	BLT_CONFIG_DONT_SET_DEFAULT, &pathOption},
    {BLT_CONFIG_COLOR, "-high", "high", "High", DEF_GRADIENT_HIGH,
        Blt_Offset(Background, color2), 0},
    {BLT_CONFIG_BOOLEAN, "-jitter", "jitter", "Jitter", 
	DEF_GRADIENT_JITTER, Blt_Offset(Background, gradient.jitter), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-logscale", "logscale", "Logscale", 
	DEF_GRADIENT_LOGSCALE, Blt_Offset(Background, gradient.logScale), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_COLOR, "-low", "low", "Low", DEF_GRADIENT_LOW,
        Blt_Offset(Background, color1), 0},
    {BLT_CONFIG_CUSTOM, "-relativeto", "relativeTo", "RelativeTo", 
	DEF_REFERENCE, Blt_Offset(Background, reference), 
	BLT_CONFIG_DONT_SET_DEFAULT, &referenceToOption},
    {BLT_CONFIG_CUSTOM, "-shape", "shape", "Shape", DEF_GRADIENT_SHAPE, 
	Blt_Offset(Background, gradient.shape), BLT_CONFIG_DONT_SET_DEFAULT, 
	&shapeOption},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

static Blt_ConfigSpec textureConfigSpecs[] =
{
    {BLT_CONFIG_BORDER, "-background", "background", "Background", DEF_BORDER,
	Blt_Offset(Background, border), 0},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL, (char *)NULL, 0, 0},
    {BLT_CONFIG_COLOR, "-high", "high", "High", DEF_GRADIENT_HIGH,
        Blt_Offset(Background, color2), 0},
    {BLT_CONFIG_COLOR, "-low", "low", "Low", DEF_GRADIENT_LOW,
        Blt_Offset(Background, color1), 0},
    {BLT_CONFIG_CUSTOM, "-relativeto", "relativeTo", "RelativeTo", 
	DEF_REFERENCE, Blt_Offset(Background, reference), 
	BLT_CONFIG_DONT_SET_DEFAULT, &referenceToOption},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/*
 *--------------------------------------------------------------------------
 *
 * ObjToFilter --
 *
 *	Given a string name, get the resample filter associated with it.
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToFilter(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_PictureFilter *filterPtr = (Blt_PictureFilter *)(widgRec + offset);
    char *string;

    string = Tcl_GetString(objPtr);
    if (string[0] == '\0') {
	*filterPtr = NULL;
	return TCL_OK;
    }
    return Blt_GetPictureFilterFromObj(interp, objPtr, filterPtr);
}

/*
 *--------------------------------------------------------------------------
 *
 * FilterToObj --
 *
 *	Convert the picture filter into a string Tcl_Obj.
 *
 * Results:
 *	The string representation of the filter is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
FilterToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_PictureFilter filter = *(Blt_PictureFilter *)(widgRec + offset);

    if (filter == NULL) {
	return Tcl_NewStringObj("", -1);
    }
    return Tcl_NewStringObj(Blt_NameOfPictureFilter(filter), -1);
}

/*
 *--------------------------------------------------------------------------
 *
 * ObjToPicture --
 *
 *	Given a string name, get the resample filter associated with it.
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToPicture(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Background *bgPtr = (Background *)(widgRec);
    Blt_Picture picture;

    if (Blt_GetPictureFromObj(interp, objPtr, &picture) != TCL_OK) {
	return TCL_ERROR;
    }
    if (bgPtr->origName != NULL) {
	Blt_Free(bgPtr->origName);
    }
    if (bgPtr->original != NULL) {
	Blt_FreePicture(bgPtr->original);
    }
    bgPtr->origName = Blt_Strdup(Tcl_GetString(objPtr));
    bgPtr->original = picture;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * PictureToObj --
 *
 *	Convert the picture filter into a string Tcl_Obj.
 *
 * Results:
 *	The string representation of the filter is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
PictureToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Background *bgPtr = (Background *)(widgRec);

    if (bgPtr->origName == NULL) {
	return Tcl_NewStringObj("", -1);
    }
    return Tcl_NewStringObj(bgPtr->origName, -1);
}

/*
 *--------------------------------------------------------------------------
 *
 * ObjToReference --
 *
 *	Given a string name, get the resample filter associated with it.
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToReference(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Background *bgPtr = (Background *)(widgRec);
    int *referencePtr = (int *)(widgRec + offset);
    char *string;
    char c;
    int type;

    string = Tcl_GetString(objPtr);
    c = string[0];
    if ((c == 's') && (strcmp(string, "self") == 0)) {
	type = REFERENCE_SELF;
    } else if ((c == 't') && (strcmp(string, "toplevel") == 0)) {
	type = REFERENCE_TOPLEVEL;
    } else if (c == '.') {
	Tk_Window tkwin, tkMain;

	tkMain = Tk_MainWindow(interp);
	tkwin = Tk_NameToWindow(interp, string, tkMain);
	if (tkwin == NULL) {
	    return TCL_ERROR;
	}
	type = REFERENCE_WINDOW;
	bgPtr->refWindow = tkwin;
    } else {
	Tcl_AppendResult(interp, "unknown reference type \"", string, "\"",
			 (char *)NULL);
	return TCL_ERROR;
    }
    *referencePtr = type;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * ReferenceToObj --
 *
 *	Convert the picture filter into a string Tcl_Obj.
 *
 * Results:
 *	The string representation of the filter is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
ReferenceToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    int reference = *(int *)(widgRec + offset);
    char *string;

    switch (reference) {
    case REFERENCE_SELF:
	string = "self";
	break;

    case REFERENCE_TOPLEVEL:
	string = "toplevel";
	break;

    case REFERENCE_WINDOW:
	{
	    Background *bgPtr = (Background *)(widgRec);

	    string = Tk_PathName(bgPtr->refWindow);
	}
	break;

    default:
	string = "???";
	break;
    }
    return Tcl_NewStringObj(string, -1);
}

/*
 *--------------------------------------------------------------------------
 *
 * ObjToShape --
 *
 *	Translate the given string to the gradient shape is
 *	represents.  Value shapes are "linear", "bilinear", "radial",
 *	and "rectangular".
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToShape(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_GradientShape *shapePtr = (Blt_GradientShape *)(widgRec + offset);
    char *string;

    string = Tcl_GetString(objPtr);
    if (strcmp(string, "linear") == 0) {
	*shapePtr = BLT_GRADIENT_SHAPE_LINEAR;
    } else if (strcmp(string, "bilinear") == 0) {
	*shapePtr = BLT_GRADIENT_SHAPE_BILINEAR;
    } else if (strcmp(string, "radial") == 0) {
	*shapePtr = BLT_GRADIENT_SHAPE_RADIAL;
    } else if (strcmp(string, "rectangular") == 0) {
	*shapePtr = BLT_GRADIENT_SHAPE_RECTANGULAR;
    } else {
	Tcl_AppendResult(interp, "unknown gradient type \"", string, "\"",
			 (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * ShapeToObj --
 *
 *	Returns the string representing the current gradiant shape.
 *
 * Results:
 *	The string representation of the shape is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
ShapeToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_GradientShape shape = *(Blt_GradientShape *)(widgRec + offset);
    char *string;

    switch (shape) {
    case BLT_GRADIENT_SHAPE_LINEAR:
	string = "linear";
	break;

    case BLT_GRADIENT_SHAPE_BILINEAR:
	string = "bilinear";
	break;

    case BLT_GRADIENT_SHAPE_RADIAL:
	string = "radial";
	break;

    case BLT_GRADIENT_SHAPE_RECTANGULAR:
	string = "rectangular";
	break;

    default:
	string = "???";
    }
    return Tcl_NewStringObj(string, -1);
}

/*
 *--------------------------------------------------------------------------
 *
 * ObjToPath --
 *
 *	Translates the given string to the gradient path it
 *	represents.  Valid paths are "x", "y", "xy", and "yx".
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToPath(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_GradientPath *pathPtr = (Blt_GradientPath *)(widgRec + offset);
    char *string;

    string = Tcl_GetString(objPtr);
    if (strcmp(string, "x") == 0) {
	*pathPtr = BLT_GRADIENT_PATH_X;
    } else if (strcmp(string, "y") == 0) {
	*pathPtr = BLT_GRADIENT_PATH_Y;
    } else if (strcmp(string, "xy") == 0) {
	*pathPtr = BLT_GRADIENT_PATH_XY;
    } else if (strcmp(string, "yx") == 0) {
	*pathPtr = BLT_GRADIENT_PATH_YX;
    } else {
	Tcl_AppendResult(interp, "unknown gradient path \"", string, "\"",
			 (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * PathToObj --
 *
 *	Convert the picture filter into a string Tcl_Obj.
 *
 * Results:
 *	The string representation of the filter is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
PathToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_GradientPath path = *(Blt_GradientPath *)(widgRec + offset);
    char *string;

    switch (path) {
    case BLT_GRADIENT_PATH_X:
	string = "x";
	break;

    case BLT_GRADIENT_PATH_Y:
	string = "y";
	break;

    case BLT_GRADIENT_PATH_XY:
	string = "xy";
	break;

    case BLT_GRADIENT_PATH_YX:
	string = "yx";
	break;

    default:
	string = "?? unknown path ??";
	break;
    }
    return Tcl_NewStringObj(string, -1);
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyClients --
 *
 *	Notify each client that's using the background style that it
 *	has changed.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
NotifyClients(Background *bgPtr)
{
    Blt_ChainLink *linkPtr;

    for (linkPtr = Blt_ChainFirstLink(bgPtr->chainPtr); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	BgToken *tokenPtr;

	/* Notify each client that the background style has
	 * changed. The client should schedule itself for
	 * redrawing.  */
	tokenPtr = Blt_ChainGetValue(linkPtr);
	if (tokenPtr->notifyProc != NULL) {
	    (*tokenPtr->notifyProc)(tokenPtr->clientData);
	}
    }
}

static void
ClearCache(Background *bgPtr)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    for (hPtr = Blt_FirstHashEntry(&bgPtr->cacheTable, &cursor); hPtr != NULL;
	 hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_Picture picture;

	picture = Blt_GetHashValue(hPtr);
	Blt_FreePicture(picture);
    }
    Blt_DeleteHashTable(&bgPtr->cacheTable);
    Blt_InitHashTable(&bgPtr->cacheTable, BLT_ONE_WORD_KEYS);
}

/*
 *----------------------------------------------------------------------
 *
 * NewBackground --
 *
 *	Creates a new background style.
 *
 * Results:
 *	Returns pointer to the new background style.
 *
 *---------------------------------------------------------------------- 
 */
static Background *
NewBackground(
    BackgroundInterpData *dataPtr,
    Tcl_Interp *interp, 
    int style)
{
    Background *bgPtr;
    char string[200];
    Blt_HashEntry *hPtr;
    int isNew;

    bgPtr = Blt_Calloc(1, sizeof(Background));
    if (bgPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate background style", 
		(char *)NULL);
	return NULL;
    }
    /* Generate a unique bgstyle name. */
    do  {
	sprintf(string, "bgstyle%d", dataPtr->nextId++);
	hPtr = Blt_CreateHashEntry(&dataPtr->styleTable, string, &isNew);
    } while (!isNew);
    
    bgPtr->hashPtr = hPtr;
    bgPtr->dataPtr = dataPtr;
    bgPtr->style = style;
    bgPtr->name = Blt_Strdup(string);
    bgPtr->chainPtr = Blt_ChainCreate();
    bgPtr->reference = REFERENCE_TOPLEVEL;
    bgPtr->gradient.shape = BLT_GRADIENT_SHAPE_LINEAR;
    bgPtr->gradient.path = BLT_GRADIENT_PATH_Y;
    bgPtr->gradient.jitter = FALSE;
    bgPtr->gradient.logScale = TRUE;
    Blt_InitHashTable(&bgPtr->cacheTable, BLT_ONE_WORD_KEYS);
    Blt_SetHashValue(hPtr, bgPtr);
    return bgPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyBackground --
 *
 *	Removes the client from the servers's list of clients and
 *	memory used by the client token is released.  When the last
 *	client is deleted, the server is also removed.
 *
 * Results:
 *	None.
 *
 *---------------------------------------------------------------------- 
 */
static void
DestroyBackground(Background *bgPtr)
{
    Blt_FreeOptions(bgPtr->configSpecs, (char *)bgPtr, bgPtr->display, 0);

    ClearCache(bgPtr);
    Blt_DeleteHashTable(&bgPtr->cacheTable);

    if (bgPtr->original != NULL) {
	Blt_FreePicture(bgPtr->original);
    }
    if (bgPtr->border != NULL) {
	Tk_Free3DBorder(bgPtr->border);
    }
    if (bgPtr->hashPtr != NULL) {
	Blt_DeleteHashEntry(&bgPtr->dataPtr->styleTable, bgPtr->hashPtr);
    }
    if (bgPtr->name != NULL) {
	Blt_Free(bgPtr->name);
    }
    Blt_ChainDestroy(bgPtr->chainPtr);
    Blt_Free(bgPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * GetBackgroundFromObj --
 *
 *	Retrieves the background style named by the given the Tcl_Obj.
 *
 * Results:
 *	None.
 *
 *---------------------------------------------------------------------- 
 */
static int
GetBackgroundFromObj(
    Tcl_Interp *interp,
    BackgroundInterpData *dataPtr,
    Tcl_Obj *objPtr,
    Background **bgPtrPtr)
{
    Blt_HashEntry *hPtr;
    char *string;

    string = Tcl_GetString(objPtr);
    hPtr = Blt_FindHashEntry(&dataPtr->styleTable, string);
    if (hPtr == NULL) {
	Tcl_AppendResult(dataPtr->interp, "can't find background style \"", 
		string, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    *bgPtrPtr = Blt_GetHashValue(hPtr);
    return TCL_OK;
}

/*
 * bgstyle create ?type? ?option values?...
 */
static int
CreateOp(    
    ClientData clientData,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Background *bgPtr;
    BackgroundInterpData *dataPtr = clientData;
    Blt_ConfigSpec *configSpecs;
    char *string;
    char c;
    int style;

    string = Tcl_GetString(objv[2]);
    c = string[0];
    if ((c == 't') && (strcmp(string, "tile") == 0)) {
	style = STYLE_TILE;
	configSpecs = tileConfigSpecs;
    } else if ((c == 'p') && (strcmp(string, "picture") == 0)) {
	style = STYLE_PICTURE;
	configSpecs = pictureConfigSpecs;
    } else if ((c == 'g') && (strcmp(string, "gradient") == 0)) {
	style = STYLE_GRADIENT;
	configSpecs = gradientConfigSpecs;
    } else if ((c == 'b') && (strcmp(string, "border") == 0)) {
	style = STYLE_BORDER;
	configSpecs = borderConfigSpecs;
    } else if ((c == 't') && (strcmp(string, "texture") == 0)) {
	style = STYLE_TEXTURE;
	configSpecs = textureConfigSpecs;
    } else {
	Tcl_AppendResult(interp, "unknown background style \"", string, "\"",
		(char *)NULL);
	return TCL_ERROR;
    }

    bgPtr = NewBackground(dataPtr, interp, style);
    bgPtr->configSpecs = configSpecs;
    bgPtr->tkwin = Tk_MainWindow(interp);
    bgPtr->display = Tk_Display(bgPtr->tkwin);

    if (Blt_ConfigureWidgetFromObj(interp, bgPtr->tkwin, configSpecs, 
		objc - 3, objv + 3, (char *)bgPtr, 0) != TCL_OK) {
	DestroyBackground(bgPtr);
	return TCL_ERROR;
    }
    Tcl_SetStringObj(Tcl_GetObjResult(interp), bgPtr->name, -1);
    return TCL_OK;
}    

/*
 * bgstyle cget $style ?option?...
 */
static int
CgetOp(    
    ClientData clientData,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    BackgroundInterpData *dataPtr = clientData;
    Background *bgPtr;

    if (GetBackgroundFromObj(interp, dataPtr, objv[2], &bgPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return Blt_ConfigureValueFromObj(interp, bgPtr->tkwin, bgPtr->configSpecs, 
	(char *)bgPtr, objv[3], 0);
}

/*
 * bgstyle configure $style ?option?...
 */
static int
ConfigureOp(    
    ClientData clientData,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    BackgroundInterpData *dataPtr = clientData;
    Background *bgPtr;
    int flags;

    if (GetBackgroundFromObj(interp, dataPtr, objv[2], &bgPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    flags = BLT_CONFIG_OBJV_ONLY;
    if (objc == 3) {
	return Blt_ConfigureInfoFromObj(interp, bgPtr->tkwin, 
		bgPtr->configSpecs, (char *)bgPtr, (Tcl_Obj *)NULL, flags);
    } else if (objc == 4) {
	return Blt_ConfigureInfoFromObj(interp, bgPtr->tkwin, 
		bgPtr->configSpecs, (char *)bgPtr, objv[3], flags);
    } else {
	if (Blt_ConfigureWidgetFromObj(interp, bgPtr->tkwin, bgPtr->configSpecs,
		 objc - 3, objv + 3, (char *)bgPtr, flags) != TCL_OK) {
	    return TCL_ERROR;
	}
	ClearCache(bgPtr);
	NotifyClients(bgPtr);
	return TCL_OK;
    }
}

/*
 * bgstyle delete $style... 
 */
static int
DeleteOp(    
    ClientData clientData,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 2; i < objc; i++) {
	BackgroundInterpData *dataPtr = clientData;
	Background *bgPtr;

	if (GetBackgroundFromObj(interp, dataPtr, objv[2], &bgPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Blt_ChainGetLength(bgPtr->chainPtr) <= 0) {
	    /* If no one is using the background style, remove it. */
	    DestroyBackground(bgPtr);
	} else {
	    /* Otherwise just take the entry out of the style hash
	     * table. When the last client is done with the style, it
	     * will be removed. */
	    Blt_DeleteHashEntry(&dataPtr->styleTable, bgPtr->hashPtr);
	    bgPtr->hashPtr = NULL;
	}
    }
    return TCL_OK;
}

/*
 * bgstyle type $style
 */
static int
TypeOp(    
    ClientData clientData,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    BackgroundInterpData *dataPtr = clientData;
    Background *bgPtr;

    if (GetBackgroundFromObj(interp, dataPtr, objv[2], &bgPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetStringObj(Tcl_GetObjResult(interp), styleTypes[bgPtr->style], -1);
    return TCL_OK;
}

static Blt_OpSpec styleOps[] =
{
    {"cget", 2, (Blt_Op)CgetOp, 4, 4, "style option",},
    {"configure", 2, (Blt_Op)ConfigureOp, 3, 0, "style ?option value?...",},
    {"create", 2, (Blt_Op)CreateOp, 3, 0, "type ?args?",},
    {"delete", 1, (Blt_Op)DeleteOp, 2, 0, "style...",},
    {"type", 1, (Blt_Op)TypeOp, 3, 3, "style",},
};
static int nStyleOps = sizeof(styleOps) / sizeof(Blt_OpSpec);

static int
BackgroundCmdProc(
    ClientData clientData,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;

    proc = Blt_GetOpFromObj(interp, nStyleOps, styleOps, BLT_OP_ARG1, 
	objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    return (*proc) (clientData, interp, objc, objv);
}

/*
 *----------------------------------------------------------------------
 *
 * GetBackgroundInterpData --
 *
 *---------------------------------------------------------------------- 
 */
static BackgroundInterpData *
GetBackgroundInterpData(Tcl_Interp *interp)
{
    BackgroundInterpData *dataPtr;
    Tcl_InterpDeleteProc *proc;

    dataPtr = (BackgroundInterpData *)
	Tcl_GetAssocData(interp, BG_STYLE_THREAD_KEY, &proc);
    if (dataPtr == NULL) {
	dataPtr = Blt_Malloc(sizeof(BackgroundInterpData));
	assert(dataPtr);
	dataPtr->interp = interp;
	dataPtr->nextId = 1;

	/* FIXME: Create interp delete proc to teardown the hash table
	 * and data entry.  Must occur after all the widgets have been
	 * destroyed (clients of the background style). */

	Tcl_SetAssocData(interp, BG_STYLE_THREAD_KEY, 
		(Tcl_InterpDeleteProc *)NULL, dataPtr);
	Blt_InitHashTable(&dataPtr->styleTable, BLT_STRING_KEYS);
    }
    return dataPtr;
}

/*LINTLIBRARY*/
int
Blt_BgStyleInit(Tcl_Interp *interp)
{
    static Blt_InitCmdSpec cmdSpec = {
	"bgstyle", BackgroundCmdProc 
    };
    cmdSpec.clientData = GetBackgroundInterpData(interp);
    return Blt_InitCmd(interp, "blt", &cmdSpec);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_GetBackground
 *
 *	Retrieves a new token of a background style from the named
 *	background style.
 *
 * Results:
 *	Returns a background style token.
 *
 * Side Effects:
 *	Memory is allocated for the new token.
 *
 *----------------------------------------------------------------------
 */
Blt_Background
Blt_GetBackground(
    Tcl_Interp *interp, 
    Tk_Window tkwin, 
    CONST char *styleName)
{
    Background *bgPtr;
    BackgroundInterpData *dataPtr;
    BgToken *tokenPtr;
    Blt_ChainLink *linkPtr;
    Blt_HashEntry *hPtr;
    int isNew;
    
    dataPtr = GetBackgroundInterpData(interp);
    hPtr = Blt_CreateHashEntry(&dataPtr->styleTable, styleName, &isNew);
    if (isNew) {
	Tk_3DBorder border;

	/* Style doesn't exist, see if it's a color name (i.e. 3D
	 * border). */
	border = Tk_Get3DBorder(interp, tkwin, styleName);
	if (border == NULL) {
	    return NULL;
	} 
	bgPtr = NewBackground(dataPtr, interp, STYLE_BORDER);
	if (bgPtr == NULL) {
	    return NULL;
	}
	bgPtr->border = border;
	bgPtr->hashPtr = hPtr;
	Blt_SetHashValue(hPtr, bgPtr);
    } else {
	bgPtr = Blt_GetHashValue(hPtr);
    }
    tokenPtr = Blt_Calloc(1, sizeof(BgToken));
    if (tokenPtr == NULL) {
	DestroyBackground(bgPtr);
	return NULL;
    }
    /* Create new token for the background style. */
    linkPtr = Blt_ChainAppend(bgPtr->chainPtr, tokenPtr);
    tokenPtr->linkPtr = linkPtr;
    tokenPtr->bgPtr = bgPtr;
    return tokenPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_GetBackgroundFromObj
 *
 *	Retrieves a new token of a background style from the named
 *	background style.
 *
 * Results:
 *	Returns a background style token.
 *
 * Side Effects:
 *	Memory is allocated for the new token.
 *
 *----------------------------------------------------------------------
 */
Blt_Background
Blt_GetBackgroundFromObj(Tcl_Interp *interp, Tk_Window tkwin, Tcl_Obj *objPtr)
{
    return Blt_GetBackground(interp, tkwin, Tcl_GetString(objPtr));
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_SetBackgroundChangedProc
 *
 *	Sets the routine to called when an image changes.  
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The designated routine will be called the next time the
 *	image associated with the tile changes.
 *
 *----------------------------------------------------------------------
 */
/*LINTLIBRARY*/
void
Blt_SetBackgroundChangedProc(
    BgToken *tokenPtr,		/* Background to register callback with. */
    Blt_BackgroundChangedProc *notifyProc, /* Function to call when
					    * style has changed. NULL
					    * indicates to unset the
					    * callback.*/
    ClientData clientData)
{
    tokenPtr->notifyProc = notifyProc;
    tokenPtr->clientData = clientData;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_FreeBackground
 *
 *	Removes the background style token.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Memory is freed.
 *
 *----------------------------------------------------------------------
 */
void
Blt_FreeBackground(BgToken *tokenPtr)
{
    Background *bgPtr = tokenPtr->bgPtr;

    if (tokenPtr->bgPtr != NULL) {
	Blt_ChainDeleteLink(bgPtr->chainPtr, tokenPtr->linkPtr);
	if (Blt_ChainGetLength(bgPtr->chainPtr) <= 0) {
	    if (bgPtr->hashPtr == NULL) { 
		/* Indicates background style entry has already been
		 * deleted. */
		DestroyBackground(bgPtr);
	    }
	}
    }
    Blt_Free(tokenPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_SetBackgroundOrigin
 *
 *	Returns the name of the master background style referenced
 *	by the token.
 *
 * Results:
 *	Return the name of the background style.
 *
 *----------------------------------------------------------------------
 */
void
Blt_SetBackgroundOrigin(BgToken *tokenPtr, int x, int y)
{
    tokenPtr->bgPtr->xOrigin = x;
    tokenPtr->bgPtr->yOrigin = y;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_NameOfBackground
 *
 *	Returns the name of the master background style referenced
 *	by the token.
 *
 * Results:
 *	Return the name of the background style.
 *
 *----------------------------------------------------------------------
 */
CONST char *
Blt_NameOfBackground(BgToken *tokenPtr)
{
    return tokenPtr->bgPtr->name;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_BackgroundBorderColor
 *
 *	Returns the name of the master background style referenced
 *	by the token.
 *
 * Results:
 *	Return the name of the background style.
 *
 *----------------------------------------------------------------------
 */
XColor *
Blt_BackgroundBorderColor(BgToken *tokenPtr)
{
    return Tk_3DBorderColor(tokenPtr->bgPtr->border);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_BackgroundBorder
 *
 *	Returns the name of the master background style referenced
 *	by the token.
 *
 * Results:
 *	Return the name of the background style.
 *
 *----------------------------------------------------------------------
 */
Tk_3DBorder
Blt_BackgroundBorder(BgToken *tokenPtr)
{
    return tokenPtr->bgPtr->border;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DrawBackgroundRectangle
 *
 *	Draws the background style in the designated window.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_DrawBackgroundRectangle(
    Tk_Window tkwin, 
    Drawable drawable, 
    BgToken *tokenPtr, 
    int x, int y, int width, int height, 
    int borderWidth, int relief)
{
    Tk_Draw3DRectangle(tkwin, drawable, tokenPtr->bgPtr->border, x, y, 
	width, height, borderWidth, relief);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_FillBackgroundRectangle
 *
 *	Draws the background style in the designated window.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_FillBackgroundRectangle(
    Tk_Window tkwin, 
    Drawable drawable, 
    BgToken *tokenPtr, 
    int x, int y, int width, int height, 
    int borderWidth, int relief)
{
    Background *bgPtr;
    Blt_HashEntry *hPtr;
    Blt_Painter painter;
    Blt_Picture picture;
    Tk_Window refWindow;
    int isNew;
    int refWidth, refHeight;
    int srcX, srcY;

    bgPtr = tokenPtr->bgPtr;
    if (bgPtr->style == STYLE_BORDER) {
	Tk_Fill3DRectangle(tkwin, drawable, bgPtr->border, x, y, width, height,
		borderWidth, relief);
	return;
    } 
    if (bgPtr->reference == REFERENCE_SELF) {
	refWindow = tkwin;
    } else if (bgPtr->reference == REFERENCE_TOPLEVEL) {
	refWindow = Blt_Toplevel(tkwin);
    } else if (bgPtr->reference == REFERENCE_WINDOW) {
	refWindow = bgPtr->refWindow;
    } else {
	return;			/* Unknown reference window. */
    }
    if (bgPtr->reference == REFERENCE_SELF) {
	srcX = x, srcY = y;
    } else {
	Tk_Window tkwin2;

	srcX = x + bgPtr->xOrigin, srcY = y + bgPtr->yOrigin;
	tkwin2 = tkwin;
	while ((tkwin2 != refWindow) && (tkwin2 != NULL)) {
	    srcX += Tk_X(tkwin2) + Tk_Changes(tkwin2)->border_width;
	    srcY += Tk_Y(tkwin2) + Tk_Changes(tkwin2)->border_width;
	    tkwin2 = Tk_Parent(tkwin2);
	}
	if (tkwin2 == NULL) {
	    /* 
	     * The window associated with the background style isn't
	     * an ancestor of the current window. That means we can't
	     * use the reference window as a guide to the size of the
	     * picture.  Simply convert to a self reference.
	     */
	    bgPtr->reference = REFERENCE_SELF;
	    refWindow = tkwin;
	    srcX = x, srcY = y;	
	}
    }
    refWidth = Tk_Width(refWindow);
    refHeight = Tk_Height(refWindow);

    picture = NULL;

    /* See if a picture has previously been generated. There will be a
     * picture for each reference window. */
    hPtr = Blt_CreateHashEntry(&bgPtr->cacheTable, (char *)refWindow, &isNew);
    if (!isNew) {
	picture = Blt_GetHashValue(hPtr);
    } 
    if ((picture == NULL) || (Blt_PictureWidth(picture) != refWidth) ||
	(Blt_PictureHeight(picture) != refHeight)) {

	/* 
	 * Either the size of the reference window has changed or one
	 * of the background style options has been reset. Resize the
	 * picture if necessary and regenerate the background.
	 */

	if (picture == NULL) {
	    picture = Blt_CreatePicture(refWidth, refHeight);
	    Blt_SetHashValue(hPtr, picture);
	} else {
	    Blt_ReallocatePicture(picture, refWidth, refHeight, 0);
	}
	if (bgPtr->style == STYLE_PICTURE) {
	    Blt_ResamplePicture(picture, bgPtr->original, bgPtr->filter, 
		bgPtr->filter);
	} else if (bgPtr->style == STYLE_GRADIENT) {
	    Pix32 low, high;
	    
	    low = Blt_XColorToPix32(bgPtr->color1);
	    high = Blt_XColorToPix32(bgPtr->color2);
	    Blt_GradientPicture(picture, &high, &low, &bgPtr->gradient);
	} else if (bgPtr->style == STYLE_TILE) {
	    Blt_TilePicture(picture, bgPtr->original, 0, 0, 0, 0, refWidth, 
		refHeight);
	} else if (bgPtr->style == STYLE_TEXTURE) {
	    Pix32 low, high;
	    
	    low = Blt_XColorToPix32(bgPtr->color1);
	    high = Blt_XColorToPix32(bgPtr->color2);
	    Blt_TexturePicture(picture, &high, &low, 0);
	}
    }
    painter = Blt_GetPainter(tkwin, 1.0);
    Blt_PaintPicture(painter, drawable, picture, srcX, srcY, width, height, 
	x, y, 0);
    /*
     * We don't free the painter to avoid the expense of generating a
     * color ramp each time we need to draw a background (this cost
     * doesn't pertain to TrueColor visuals). Painters are reference
     * counted, so there will be only one painter per visual used.  In
     * most cases, that's one or two.
     */
    if ((relief != TK_RELIEF_FLAT) && (borderWidth > 0)) {
	Tk_Draw3DRectangle(tkwin, drawable, bgPtr->border, x, y, width, height,
		borderWidth, relief);
    }
}

