
/*
 * bltGrLine.c --
 *
 * This module implements line graph and stripchart elements for the
 * BLT graph widget.
 *
 *	Copyright 1993-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltGraph.h"
#include "bltChain.h"
#include <X11/Xutil.h>
#include "bltGrElem.h"
#include "tkDisplay.h"
#include "bltTile.h"
#include "bltImage.h"
#include "bltBitmap.h"

#define COLOR_DEFAULT	(XColor *)1
#define PATTERN_SOLID	((Pixmap)1)

#define PEN_INCREASING  1	/* Draw line segments for only those
				 * data points whose abscissas are
				 * monotonically increasing in
				 * order */
#define PEN_DECREASING  2	/* Lines will be drawn between only
				 * those points whose abscissas are
				 * decreasing in order */

#define PEN_BOTH_DIRECTIONS	(PEN_INCREASING | PEN_DECREASING)
 /* Lines will be drawn between points regardless of the ordering of
  * the abscissas */

#define BROKEN_TRACE(dir,last,next) \
    (((((dir) & PEN_DECREASING) == 0) && ((next) < (last))) || \
     ((((dir) & PEN_INCREASING) == 0) && ((next) > (last))))

#define DRAW_SYMBOL(linePtr) \
	(((linePtr)->symbolCounter % (linePtr)->symbolInterval) == 0)

typedef enum { 
    PEN_SMOOTH_LINEAR,		/* Line segments */
    PEN_SMOOTH_STEP,		/* Step-and-hold */
    PEN_SMOOTH_NATURAL,		/* Natural cubic spline */
    PEN_SMOOTH_QUADRATIC,	/* Quadratic spline */
    PEN_SMOOTH_CATROM,		/* Catrom parametric spline */
    PEN_SMOOTH_LAST		/* Sentinel */
} Smoothing;

typedef struct {
    char *name;
    Smoothing value;
} SmoothingInfo;

static SmoothingInfo smoothingInfo[] = {
    { "none",		PEN_SMOOTH_LINEAR },
    { "linear",		PEN_SMOOTH_LINEAR },
    { "step",		PEN_SMOOTH_STEP },
    { "natural",	PEN_SMOOTH_NATURAL },
    { "cubic",		PEN_SMOOTH_NATURAL },
    { "quadratic",	PEN_SMOOTH_QUADRATIC },
    { "catrom",		PEN_SMOOTH_CATROM },
    { (char *)NULL,	PEN_SMOOTH_LAST }
};


typedef struct {
    Point2D *screenPts;		/* Array of transformed coordinates */
    int nScreenPts;		/* Number of coordinates */
    int *dataToStyle;		/* Index of pen styles  */
    int *indices;		/* Maps segments/traces to data points */

} MapInfo;

/*
 * Symbol types for line elements
 */
typedef enum {
    SYMBOL_NONE,
    SYMBOL_SQUARE,
    SYMBOL_CIRCLE,
    SYMBOL_DIAMOND,
    SYMBOL_PLUS,
    SYMBOL_CROSS,
    SYMBOL_SPLUS,
    SYMBOL_SCROSS,
    SYMBOL_TRIANGLE,
    SYMBOL_ARROW,
    SYMBOL_BITMAP
} SymbolType;

typedef struct {
    char *name;
    size_t minChars;
    SymbolType type;
} GraphSymbol;

static GraphSymbol graphSymbols[] = {
    { "arrow",	  1, SYMBOL_ARROW, },
    { "circle",	  2, SYMBOL_CIRCLE, },
    { "cross",	  2, SYMBOL_CROSS, }, 
    { "diamond",  1, SYMBOL_DIAMOND,}, 
    { "none",	  1, SYMBOL_NONE, }, 
    { "plus",	  1, SYMBOL_PLUS, }, 
    { "scross",	  2, SYMBOL_SCROSS, }, 
    { "splus",	  2, SYMBOL_SPLUS, }, 
    { "square",	  2, SYMBOL_SQUARE, }, 
    { "triangle", 1, SYMBOL_TRIANGLE, }, 
    { NULL, 0, 0 }, 
};

typedef struct {
    SymbolType type;		/* Type of symbol to be drawn/printed */

    int size;			/* Requested size of symbol in pixels */

    XColor *outlineColor;	/* Outline color */

    int outlineWidth;		/* Width of the outline */

    GC outlineGC;		/* Outline graphics context */

    XColor *fillColor;		/* Normal fill color */

    GC fillGC;			/* Fill graphics context */

    /* The last two fields are used only for bitmap symbols. */

    Pixmap bitmap;		/* Bitmap to determine foreground/background
				 * pixels of the symbol */

    Pixmap mask;		/* Bitmap representing the transparent
				 * pixels of the symbol */

} Symbol;

typedef struct {
    int start;			/* Index into the X-Y coordinate
				 * arrays indicating where trace
				 * starts. */

    int nScreenPts;		/* Number of points in the continuous
				 * trace */

    Point2D *screenPts;		/* Array of screen coordinates
				 * (malloc-ed) representing the
				 * trace. */

    int *symbolToData;		/* Reverse mapping of screen
				 * coordinate indices back to their
				 * data coordinates */
} Trace;

typedef struct {
    Pen base;		/* Must be the first field in line pens. */

    /* Symbol attributes. */
    Symbol symbol;		/* Element symbol type */

    /* Trace attributes. */
    int traceWidth;		/* Width of the line segments. If
				 * lineWidth is 0, no line will be
				 * drawn, only symbols. */

    Blt_Dashes traceDashes;	/* Dash on-off list value */

    XColor *traceColor;		/* Line segment color */

    XColor *traceOffColor;	/* Line segment dash gap color */

    GC traceGC;			/* Line segment graphics context */
    
    /* Error bar attributes. */
    int errorBarShow;		/* Describes which error bars to
				 * display: none, x, y, or * both. */

    int errorBarLineWidth;	/* Width of the error bar segments. */

    int errorBarCapWidth;	/* Width of the cap on error bars. */

    XColor *errorBarColor;	/* Color of the error bar. */

    GC errorBarGC;		/* Error bar graphics context. */

    /* Show value attributes. */
    int valueShow;		/* Indicates whether to display data
				 * value.  Values are x, y, both, or 
				 * none. */
    char *valueFormat;		/* A printf format string. */

    TextStyle valueStyle;	/* Text attributes (color, font,
				 * rotation, etc.) of the value. */

} LinePen;

typedef struct {
    PenStyle base;		/* Must be the first field in line styles. */

    Point2D *symbolPts;		/* Points to start of array for this pen. */

    Segment2D *strips;		/* Points to start of the line segments
				 * for this pen. */
    int nSymbolPts;		/* # of points for this pen. */
    int nStrips;		/* # of line segments for this pen. */

} LinePenStyle;

typedef struct {
    Element base;		/* Must be the first field. */

    LinePen builtinPen;

    /* Line smoothing */
    Smoothing reqSmooth;	/* Requested smoothing function to use
				 * for connecting the data points */

    Smoothing smooth;		/* Smoothing function used. */

    double rTolerance;		/* Tolerance to reduce the number of
				 * points displayed. */

    /*
     * Drawing related data structures.
     */

    /* Area-under-curve fill attributes. */
    XColor *fillFgColor;
    XColor *fillBgColor;
    GC fillGC;

    Blt_Tile fillTile;		/* Tile for fill area. */
    Pixmap fillStipple;		/* Stipple for fill area. */

    Point2D *fillPts;		/* Array of points used to draw
				 * polygon to fill area under the
				 * curve */

    /* Symbol points */
    Point2D *symbolPts;		/* Holds the screen coordinates of all
				 * the data points for the element. */

    int *symbolToData;		/* Contains indices of data points.
				 * It's first used to map pens to the
				 * visible points to sort them by pen
				 * style, and later to find data
				 * points from the index of a visible
				 * point. */

    /* Active symbol points */
    Point2D *activePts;		/* Array of indices representing the
				 * "active" points. */
    int *activeToData;		/* Contains indices of data points.
				 * It's first used to map pens to the
				 * visible points to sort them by pen
				 * style, and later to find data
				 * points from the index of a visible
				 * point. */

    int nActivePts;		/* Number of indices in the above array. */
    int nSymbolPts;		/* Number of points */
    int nFillPts;

    int reqMaxSymbols;
    int symbolInterval;
    int symbolCounter;

    /* X-Y graph-specific fields */

    int penDir;			/* Indicates if a change in the pen
				 * direction should be considered a
				 * retrace (line segment is not
				 * drawn). */

    Blt_Chain *traces;		/* List of traces (a trace is a series
				 * of contiguous line segments).  New
				 * traces are generated when either
				 * the next segment changes the pen
				 * direction, or the end point is
				 * clipped by the plotting area. */

    /* Stripchart-specific fields */

    Segment2D *strips;		/* Holds the the line segments of the
				 * element trace. The segments are
				 * grouped by pen style. */
    int nStrips;		/* Number of line segments to be drawn. */
    int *stripToData;		/* Pen to visible line segment mapping. */

} Line;

static Blt_OptionFreeProc  FreePattern;
static Blt_OptionPrintProc PatternToObj;
static Blt_OptionParseProc ObjToPattern;
static Blt_CustomOption patternOption =
{
    ObjToPattern, PatternToObj, FreePattern, (ClientData)0
};

static Blt_OptionParseProc ObjToSmooth;
static Blt_OptionPrintProc SmoothToObj;
static Blt_CustomOption smoothOption =
{
    ObjToSmooth, SmoothToObj, NULL, (ClientData)0
};

static Blt_OptionParseProc ObjToPenDir;
static Blt_OptionPrintProc PenDirToObj;
static Blt_CustomOption penDirOption =
{
    ObjToPenDir, PenDirToObj, NULL, (ClientData)0
};

static Blt_OptionFreeProc FreeSymbol;
static Blt_OptionParseProc ObjToSymbol;
static Blt_OptionPrintProc SymbolToObj;
static Blt_CustomOption symbolOption =
{
    ObjToSymbol, SymbolToObj, FreeSymbol, (ClientData)0
};

extern Blt_CustomOption bltLineStylesOption;
extern Blt_CustomOption bltColorOption;
extern Blt_CustomOption bltDataOption;
extern Blt_CustomOption bltDataPairsOption;
extern Blt_CustomOption bltLinePenOption;
extern Blt_CustomOption bltXAxisOption;
extern Blt_CustomOption bltYAxisOption;

#define DEF_LINE_ACTIVE_PEN		"activeLine"
#define DEF_LINE_AXIS_X			"x"
#define DEF_LINE_AXIS_Y			"y"
#define DEF_LINE_DASHES			(char *)NULL
#define DEF_LINE_DATA			(char *)NULL
#define DEF_LINE_FILL_COLOR    		"defcolor"
#define DEF_LINE_HIDE			"no"
#define DEF_LINE_LABEL			(char *)NULL
#define DEF_LINE_LABEL_RELIEF		"flat"
#define DEF_LINE_MAX_SYMBOLS		"0"
#define DEF_LINE_OFFDASH_COLOR    	(char *)NULL
#define DEF_LINE_OUTLINE_COLOR		"defcolor"
#define DEF_LINE_OUTLINE_WIDTH 		"1"
#define DEF_LINE_PATTERN		(char *)NULL
#define DEF_LINE_PATTERN_BG		"white"
#define DEF_LINE_PATTERN_FG		"black"
#define DEF_LINE_PATTERN_TILE		(char *)NULL
#define DEF_LINE_PEN_COLOR		RGB_NAVYBLUE
#define DEF_LINE_PEN_DIRECTION		"both"
#define DEF_LINE_PEN_WIDTH		"1"
#define DEF_LINE_PIXELS			"0.125i"
#define DEF_LINE_REDUCE			"0.0"
#define DEF_LINE_SCALE_SYMBOLS		"yes"
#define DEF_LINE_SMOOTH			"linear"
#define DEF_LINE_STATE			"normal"
#define DEF_LINE_STIPPLE		(char *)NULL
#define DEF_LINE_STYLES			""
#define DEF_LINE_SYMBOL			"circle"
#define DEF_LINE_TAGS			"all"
#define DEF_LINE_X_DATA			(char *)NULL
#define DEF_LINE_Y_DATA			(char *)NULL

#define DEF_LINE_ERRORBAR_COLOR		"defcolor"
#define DEF_LINE_ERRORBAR_LINE_WIDTH	"1"
#define DEF_LINE_ERRORBAR_CAP_WIDTH	"1"
#define DEF_LINE_SHOW_ERRORBARS		"both"

#define DEF_PEN_ACTIVE_COLOR		RGB_BLUE
#define DEF_PEN_DASHES			(char *)NULL
#define DEF_PEN_FILL_COLOR    		"defcolor"
#define DEF_PEN_LINE_WIDTH		"1"
#define DEF_PEN_NORMAL_COLOR		RGB_NAVYBLUE
#define DEF_PEN_OFFDASH_COLOR    	(char *)NULL
#define DEF_PEN_OUTLINE_COLOR		"defcolor"
#define DEF_PEN_OUTLINE_WIDTH 		"1"
#define DEF_PEN_PIXELS			"0.125i"
#define DEF_PEN_SYMBOL			"circle"
#define DEF_PEN_TYPE			"line"
#define	DEF_PEN_VALUE_ANCHOR		"s"
#define	DEF_PEN_VALUE_COLOR		RGB_BLACK
#define	DEF_PEN_VALUE_FONT		STD_FONT_SMALL
#define	DEF_PEN_VALUE_FORMAT		"%g"
#define	DEF_PEN_VALUE_ANGLE		(char *)NULL
#define DEF_PEN_SHOW_VALUES		"no"

static Blt_ConfigSpec lineElemConfigSpecs[] =
{
    {BLT_CONFIG_CUSTOM, "-activepen", "activePen", "ActivePen",
	DEF_LINE_ACTIVE_PEN, Blt_Offset(Line, base.activePenPtr),
	BLT_CONFIG_NULL_OK, &bltLinePenOption},
    {BLT_CONFIG_CUSTOM, "-areapattern", "areaPattern", "AreaPattern",
        DEF_LINE_PATTERN, Blt_Offset(Line, fillStipple), 0, &patternOption},
    {BLT_CONFIG_COLOR, "-areaforeground", "areaForeground", "areaForeground",
	DEF_LINE_PATTERN_FG, Blt_Offset(Line, fillFgColor), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_COLOR, "-areabackground", "areaBackground", "areaBackground",
	DEF_LINE_PATTERN_BG, Blt_Offset(Line, fillBgColor), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_TILE, "-areatile", "areaTile", "AreaTile",
	DEF_LINE_PATTERN_TILE, Blt_Offset(Line, fillTile), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags", DEF_LINE_TAGS, 
	Blt_Offset(Line, base.object.tags), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_COLOR, "-color", "color", "Color",
	DEF_LINE_PEN_COLOR, Blt_Offset(Line, builtinPen.traceColor), 0},
    {BLT_CONFIG_DASHES, "-dashes", "dashes", "Dashes", DEF_LINE_DASHES, 
	Blt_Offset(Line, builtinPen.traceDashes), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-data", "data", "Data", DEF_LINE_DATA, 0, 0, 
	&bltDataPairsOption},
    {BLT_CONFIG_CUSTOM, "-errorbarcolor", "errorBarColor", "ErrorBarColor",
	DEF_LINE_ERRORBAR_COLOR, Blt_Offset(Line, builtinPen.errorBarColor), 
	0, &bltColorOption},
    {BLT_CONFIG_PIXELS, "-errorbarwidth", "errorBarWidth", "ErrorBarWidth",
	DEF_LINE_ERRORBAR_LINE_WIDTH, 
	Blt_Offset(Line, builtinPen.errorBarLineWidth),
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS, "-errorbarcap", "errorBarCap", "ErrorBarCap", 
	DEF_LINE_ERRORBAR_CAP_WIDTH, 
	Blt_Offset(Line, builtinPen.errorBarCapWidth),
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-fill", "fill", "Fill", DEF_LINE_FILL_COLOR, 
	Blt_Offset(Line, builtinPen.symbol.fillColor), BLT_CONFIG_NULL_OK, 
	&bltColorOption},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide", DEF_LINE_HIDE, 
	Blt_Offset(Line, base.object.hidden), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STRING, "-label", "label", "Label", (char *)NULL, 
	Blt_Offset(Line, base.label), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_RELIEF, "-labelrelief", "labelRelief", "LabelRelief",
	DEF_LINE_LABEL_RELIEF, Blt_Offset(Line, base.labelRelief),
	BLT_CONFIG_DONT_SET_DEFAULT}, 
    {BLT_CONFIG_PIXELS, "-linewidth", "lineWidth", "LineWidth",
	DEF_LINE_PEN_WIDTH, Blt_Offset(Line, builtinPen.traceWidth),
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-mapx", "mapX", "MapX",
        DEF_LINE_AXIS_X, Blt_Offset(Line, base.axes.x), 0, &bltXAxisOption},
    {BLT_CONFIG_CUSTOM, "-mapy", "mapY", "MapY",
	DEF_LINE_AXIS_Y, Blt_Offset(Line, base.axes.y), 0, &bltYAxisOption},
    {BLT_CONFIG_INT_NONNEGATIVE, "-maxsymbols", "maxSymbols", "MaxSymbols",
	DEF_LINE_MAX_SYMBOLS, Blt_Offset(Line, reqMaxSymbols),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-offdash", "offDash", "OffDash", 
	DEF_LINE_OFFDASH_COLOR, Blt_Offset(Line, builtinPen.traceOffColor),
	BLT_CONFIG_NULL_OK, &bltColorOption},
    {BLT_CONFIG_CUSTOM, "-outline", "outline", "Outline", 
	DEF_LINE_OUTLINE_COLOR, 
	Blt_Offset(Line, builtinPen.symbol.outlineColor), 0, &bltColorOption},
    {BLT_CONFIG_PIXELS, "-outlinewidth", "outlineWidth", "OutlineWidth",
	DEF_LINE_OUTLINE_WIDTH, 
	Blt_Offset(Line, builtinPen.symbol.outlineWidth),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-pen", "pen", "Pen", (char *)NULL, 
	Blt_Offset(Line, base.normalPenPtr), BLT_CONFIG_NULL_OK, 
	&bltLinePenOption},
    {BLT_CONFIG_PIXELS, "-pixels", "pixels", "Pixels", DEF_LINE_PIXELS, 
	Blt_Offset(Line, builtinPen.symbol.size), GRAPH | STRIPCHART}, 
    {BLT_CONFIG_DOUBLE, "-reduce", "reduce", "Reduce",
	DEF_LINE_REDUCE, Blt_Offset(Line, rTolerance),
	GRAPH | STRIPCHART | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-scalesymbols", "scaleSymbols", "ScaleSymbols",
	DEF_LINE_SCALE_SYMBOLS, Blt_Offset(Line, base.scaleSymbols),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_FILL, "-showerrorbars", "showErrorBars", "ShowErrorBars",
	DEF_LINE_SHOW_ERRORBARS, Blt_Offset(Line, builtinPen.errorBarShow),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_FILL, "-showvalues", "showValues", "ShowValues",
	DEF_PEN_SHOW_VALUES, Blt_Offset(Line, builtinPen.valueShow),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-smooth", "smooth", "Smooth", DEF_LINE_SMOOTH, 
	Blt_Offset(Line, reqSmooth), BLT_CONFIG_DONT_SET_DEFAULT, 
	&smoothOption},
    {BLT_CONFIG_STATE, "-state", "state", "State", DEF_LINE_STATE, 
	Blt_Offset(Line, base.state), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-styles", "styles", "Styles", DEF_LINE_STYLES, 
	Blt_Offset(Line, base.stylePalette), 0, &bltLineStylesOption},
    {BLT_CONFIG_CUSTOM, "-symbol", "symbol", "Symbol", DEF_LINE_SYMBOL, 
	Blt_Offset(Line, builtinPen.symbol), BLT_CONFIG_DONT_SET_DEFAULT, 
	&symbolOption},
    {BLT_CONFIG_CUSTOM, "-trace", "trace", "Trace", DEF_LINE_PEN_DIRECTION, 
	Blt_Offset(Line, penDir), BLT_CONFIG_DONT_SET_DEFAULT, &penDirOption},
    {BLT_CONFIG_ANCHOR, "-valueanchor", "valueAnchor", "ValueAnchor",
	DEF_PEN_VALUE_ANCHOR, Blt_Offset(Line, builtinPen.valueStyle.anchor), 
        0},
    {BLT_CONFIG_COLOR, "-valuecolor", "valueColor", "ValueColor",
	DEF_PEN_VALUE_COLOR, Blt_Offset(Line, builtinPen.valueStyle.color), 0},
    {BLT_CONFIG_FONT, "-valuefont", "valueFont", "ValueFont",
	DEF_PEN_VALUE_FONT, Blt_Offset(Line, builtinPen.valueStyle.font), 0},
    {BLT_CONFIG_STRING, "-valueformat", "valueFormat", "ValueFormat",
	DEF_PEN_VALUE_FORMAT, Blt_Offset(Line, builtinPen.valueFormat),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_DOUBLE, "-valuerotate", "valueRotate", "ValueRotate",
	DEF_PEN_VALUE_ANGLE, Blt_Offset(Line, builtinPen.valueStyle.angle),0},
    {BLT_CONFIG_CUSTOM, "-weights", "weights", "Weights", (char *)NULL, 
	Blt_Offset(Line, base.w), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-x", "xData", "XData", (char *)NULL, 
        Blt_Offset(Line, base.x), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xdata", "xData", "XData", (char *)NULL, 
	Blt_Offset(Line, base.x), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xerror", "xError", "XError", (char *)NULL, 
	Blt_Offset(Line, base.xError), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xhigh", "xHigh", "XHigh", (char *)NULL, 
	Blt_Offset(Line, base.xHigh), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xlow", "xLow", "XLow", (char *)NULL, 
	Blt_Offset(Line, base.xLow), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-y", "yData", "YData", (char *)NULL, 
	Blt_Offset(Line, base.y), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-ydata", "yData", "YData", (char *)NULL, 
	Blt_Offset(Line, base.y), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-yerror", "yError", "YError", (char *)NULL, 
	Blt_Offset(Line, base.yError), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-yhigh", "yHigh", "YHigh", (char *)NULL, 
	Blt_Offset(Line, base.yHigh), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-ylow", "yLow", "YLow", (char *)NULL, 
	Blt_Offset(Line, base.yLow), 0, &bltDataOption},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};


static Blt_ConfigSpec stripElemConfigSpecs[] =
{
    {BLT_CONFIG_CUSTOM, "-activepen", "activePen", "ActivePen",
	DEF_LINE_ACTIVE_PEN, Blt_Offset(Line, base.activePenPtr), 
	BLT_CONFIG_NULL_OK, &bltLinePenOption},
    {BLT_CONFIG_CUSTOM, "-areapattern", "areaPattern", "AreaPattern",
        DEF_LINE_PATTERN, Blt_Offset(Line, fillStipple), 0, &patternOption},
    {BLT_CONFIG_COLOR, "-areaforeground", "areaForeground", "areaForeground",
	DEF_LINE_PATTERN_FG, Blt_Offset(Line, fillFgColor), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_COLOR, "-areabackground", "areaBackground", "areaBackground",
	DEF_LINE_PATTERN_BG, Blt_Offset(Line, fillBgColor), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_TILE, "-areatile", "areaTile", "AreaTile",
	DEF_LINE_PATTERN_TILE, Blt_Offset(Line, fillTile), 
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags", DEF_LINE_TAGS, 
	Blt_Offset(Line, base.object.tags), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_COLOR, "-color", "color", "Color",
	DEF_LINE_PEN_COLOR, Blt_Offset(Line, builtinPen.traceColor), 0},
    {BLT_CONFIG_DASHES, "-dashes", "dashes", "Dashes",
	DEF_LINE_DASHES, Blt_Offset(Line, builtinPen.traceDashes),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-data", "data", "Data", DEF_LINE_DATA, 0, 0, 
	&bltDataPairsOption},
    {BLT_CONFIG_CUSTOM, "-errorbarcolor", "errorBarColor", "ErrorBarColor",
	DEF_LINE_ERRORBAR_COLOR, Blt_Offset(Line, builtinPen.errorBarColor), 0,
	&bltColorOption},
    {BLT_CONFIG_PIXELS, "-errorbarwidth", "errorBarWidth", "ErrorBarWidth",
	DEF_LINE_ERRORBAR_LINE_WIDTH, 
	Blt_Offset(Line, builtinPen.errorBarLineWidth),
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS, "-errorbarcap", "errorBarCap", "ErrorBarCap", 
        DEF_LINE_ERRORBAR_CAP_WIDTH, 
	Blt_Offset(Line, builtinPen.errorBarCapWidth), 
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-fill", "fill", "Fill", DEF_LINE_FILL_COLOR, 
	Blt_Offset(Line, builtinPen.symbol.fillColor),
	BLT_CONFIG_NULL_OK, &bltColorOption},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide", DEF_LINE_HIDE, 
	Blt_Offset(Line, base.object.hidden), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STRING, "-label", "label", "Label", (char *)NULL, 
	Blt_Offset(Line, base.label), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_RELIEF, "-labelrelief", "labelRelief", "LabelRelief",
	DEF_LINE_LABEL_RELIEF, Blt_Offset(Line, base.labelRelief),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS, "-linewidth", "lineWidth", "LineWidth", 
	DEF_LINE_PEN_WIDTH, Blt_Offset(Line, builtinPen.traceWidth), 
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-mapx", "mapX", "MapX", DEF_LINE_AXIS_X, 
	Blt_Offset(Line, base.axes.x), 0, &bltXAxisOption},
    {BLT_CONFIG_CUSTOM, "-mapy", "mapY", "MapY", DEF_LINE_AXIS_Y, 
	Blt_Offset(Line, base.axes.y), 0, &bltYAxisOption},
    {BLT_CONFIG_INT_NONNEGATIVE, "-maxsymbols", "maxSymbols", "MaxSymbols",
	DEF_LINE_MAX_SYMBOLS, Blt_Offset(Line, reqMaxSymbols),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-offdash", "offDash", "OffDash",
	DEF_LINE_OFFDASH_COLOR, Blt_Offset(Line, builtinPen.traceOffColor),
	BLT_CONFIG_NULL_OK, &bltColorOption},
    {BLT_CONFIG_CUSTOM, "-outline", "outline", "Outline",
	DEF_LINE_OUTLINE_COLOR, 
	Blt_Offset(Line, builtinPen.symbol.outlineColor), 0, &bltColorOption},
    {BLT_CONFIG_PIXELS, "-outlinewidth", "outlineWidth", "OutlineWidth",
	DEF_LINE_OUTLINE_WIDTH, 
	Blt_Offset(Line, builtinPen.symbol.outlineWidth),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-pen", "pen", "Pen", (char *)NULL, 
	Blt_Offset(Line, base.normalPenPtr), BLT_CONFIG_NULL_OK, 
	&bltLinePenOption},
    {BLT_CONFIG_PIXELS, "-pixels", "pixels", "Pixels", DEF_LINE_PIXELS, 
	Blt_Offset(Line, builtinPen.symbol.size), 0},
    {BLT_CONFIG_BOOLEAN, "-scalesymbols", "scaleSymbols", "ScaleSymbols",
	DEF_LINE_SCALE_SYMBOLS, Blt_Offset(Line, base.scaleSymbols),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_FILL, "-showerrorbars", "showErrorBars", "ShowErrorBars",
	DEF_LINE_SHOW_ERRORBARS, Blt_Offset(Line, builtinPen.errorBarShow),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_FILL, "-showvalues", "showValues", "ShowValues",
	DEF_PEN_SHOW_VALUES, Blt_Offset(Line, builtinPen.valueShow),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-smooth", "smooth", "Smooth", DEF_LINE_SMOOTH, 
        Blt_Offset(Line, reqSmooth), BLT_CONFIG_DONT_SET_DEFAULT, 
	&smoothOption},
    {BLT_CONFIG_CUSTOM, "-styles", "styles", "Styles", DEF_LINE_STYLES, 
	Blt_Offset(Line, base.stylePalette), 0, &bltLineStylesOption},
    {BLT_CONFIG_CUSTOM, "-symbol", "symbol", "Symbol", DEF_LINE_SYMBOL, 
	Blt_Offset(Line, builtinPen.symbol), BLT_CONFIG_DONT_SET_DEFAULT, 
	&symbolOption},
    {BLT_CONFIG_ANCHOR, "-valueanchor", "valueAnchor", "ValueAnchor",
	DEF_PEN_VALUE_ANCHOR, 
        Blt_Offset(Line, builtinPen.valueStyle.anchor), 0},
    {BLT_CONFIG_COLOR, "-valuecolor", "valueColor", "ValueColor",
	DEF_PEN_VALUE_COLOR, Blt_Offset(Line, builtinPen.valueStyle.color), 0},
    {BLT_CONFIG_FONT, "-valuefont", "valueFont", "ValueFont",
	DEF_PEN_VALUE_FONT, Blt_Offset(Line, builtinPen.valueStyle.font), 0},
    {BLT_CONFIG_STRING, "-valueformat", "valueFormat", "ValueFormat",
	DEF_PEN_VALUE_FORMAT, Blt_Offset(Line, builtinPen.valueFormat),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_DOUBLE, "-valuerotate", "valueRotate", "ValueRotate",
	DEF_PEN_VALUE_ANGLE, Blt_Offset(Line, builtinPen.valueStyle.angle),0},
    {BLT_CONFIG_CUSTOM, "-weights", "weights", "Weights", (char *)NULL, 
	Blt_Offset(Line, base.w), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-x", "xData", "XData", (char *)NULL, 
	Blt_Offset(Line, base.x), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xdata", "xData", "XData", (char *)NULL, 
	Blt_Offset(Line, base.x), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-y", "yData", "YData", (char *)NULL, 
	Blt_Offset(Line, base.y), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xerror", "xError", "XError", (char *)NULL, 
	Blt_Offset(Line, base.xError), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-ydata", "yData", "YData", (char *)NULL, 
	Blt_Offset(Line, base.y), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-yerror", "yError", "YError", (char *)NULL, 
	Blt_Offset(Line, base.yError), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xhigh", "xHigh", "XHigh", (char *)NULL, 
	Blt_Offset(Line, base.xHigh), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xlow", "xLow", "XLow", (char *)NULL, 
	Blt_Offset(Line, base.xLow), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-yhigh", "yHigh", "YHigh", (char *)NULL, 
	Blt_Offset(Line, base.xHigh), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-ylow", "yLow", "YLow", (char *)NULL, 
	Blt_Offset(Line, base.yLow), 0, &bltDataOption},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

static Blt_ConfigSpec linePenConfigSpecs[] =
{
    {BLT_CONFIG_COLOR, "-color", "color", "Color", DEF_PEN_ACTIVE_COLOR, 
	Blt_Offset(LinePen, traceColor), ACTIVE_PEN},
    {BLT_CONFIG_COLOR, "-color", "color", "Color", DEF_PEN_NORMAL_COLOR, 
	Blt_Offset(LinePen, traceColor), NORMAL_PEN},
    {BLT_CONFIG_DASHES, "-dashes", "dashes", "Dashes", DEF_PEN_DASHES, 
	Blt_Offset(LinePen, traceDashes), BLT_CONFIG_NULL_OK | ALL_PENS},
    {BLT_CONFIG_CUSTOM, "-errorbarcolor", "errorBarColor", "ErrorBarColor",
	DEF_LINE_ERRORBAR_COLOR, Blt_Offset(LinePen, errorBarColor), 
	ALL_PENS, &bltColorOption},
    {BLT_CONFIG_PIXELS, "-errorbarwidth", "errorBarWidth", "ErrorBarWidth",
	DEF_LINE_ERRORBAR_LINE_WIDTH, Blt_Offset(LinePen, errorBarLineWidth),
        ALL_PENS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS, "-errorbarcap", "errorBarCap", "ErrorBarCap", 
	DEF_LINE_ERRORBAR_CAP_WIDTH, Blt_Offset(LinePen, errorBarCapWidth),
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-fill", "fill", "Fill", DEF_PEN_FILL_COLOR, 
	Blt_Offset(LinePen, symbol.fillColor), BLT_CONFIG_NULL_OK | ALL_PENS, 
	&bltColorOption},
    {BLT_CONFIG_PIXELS, "-linewidth", "lineWidth", "LineWidth", (char *)NULL, 
	Blt_Offset(LinePen, traceWidth), ALL_PENS| BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-offdash", "offDash", "OffDash", DEF_PEN_OFFDASH_COLOR,
	Blt_Offset(LinePen, traceOffColor), BLT_CONFIG_NULL_OK | ALL_PENS, 
	&bltColorOption},
    {BLT_CONFIG_CUSTOM, "-outline", "outline", "Outline", DEF_PEN_OUTLINE_COLOR,
	Blt_Offset(LinePen, symbol.outlineColor), ALL_PENS, &bltColorOption},
    {BLT_CONFIG_PIXELS, "-outlinewidth", "outlineWidth", "OutlineWidth",
	DEF_PEN_OUTLINE_WIDTH, Blt_Offset(LinePen, symbol.outlineWidth),
	BLT_CONFIG_DONT_SET_DEFAULT | ALL_PENS},
    {BLT_CONFIG_PIXELS, "-pixels", "pixels", "Pixels", DEF_PEN_PIXELS, 
	Blt_Offset(LinePen, symbol.size), ALL_PENS},
    {BLT_CONFIG_FILL, "-showerrorbars", "showErrorBars", "ShowErrorBars",
	DEF_LINE_SHOW_ERRORBARS, Blt_Offset(LinePen, errorBarShow),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_FILL, "-showvalues", "showValues", "ShowValues",
	DEF_PEN_SHOW_VALUES, Blt_Offset(LinePen, valueShow),
	ALL_PENS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-symbol", "symbol", "Symbol", DEF_PEN_SYMBOL, 
	Blt_Offset(LinePen, symbol), BLT_CONFIG_DONT_SET_DEFAULT | ALL_PENS, 
	&symbolOption},
    {BLT_CONFIG_STRING, "-type", (char *)NULL, (char *)NULL, DEF_PEN_TYPE, 
	Blt_Offset(Pen, typeId), ALL_PENS | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_ANCHOR, "-valueanchor", "valueAnchor", "ValueAnchor",
	DEF_PEN_VALUE_ANCHOR, Blt_Offset(LinePen, valueStyle.anchor), ALL_PENS},
    {BLT_CONFIG_COLOR, "-valuecolor", "valueColor", "ValueColor",
	DEF_PEN_VALUE_COLOR, Blt_Offset(LinePen, valueStyle.color), ALL_PENS},
    {BLT_CONFIG_FONT, "-valuefont", "valueFont", "ValueFont",
	DEF_PEN_VALUE_FONT, Blt_Offset(LinePen, valueStyle.font), ALL_PENS},
    {BLT_CONFIG_STRING, "-valueformat", "valueFormat", "ValueFormat",
	DEF_PEN_VALUE_FORMAT, Blt_Offset(LinePen, valueFormat),
	ALL_PENS | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_DOUBLE, "-valuerotate", "valueRotate", "ValueRotate",
	DEF_PEN_VALUE_ANGLE, Blt_Offset(LinePen, valueStyle.angle), ALL_PENS},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

typedef double (DistanceProc) _ANSI_ARGS_((int x, int y, Point2D *p, 
	Point2D *q, Point2D *t));

/* Forward declarations */
static PenConfigureProc ConfigurePen;
static PenDestroyProc DestroyPen;
static ElementClosestProc ClosestLine;
static ElementConfigProc ConfigureLine;
static ElementDestroyProc DestroyLine;
static ElementDrawProc DrawActiveLine;
static ElementDrawProc DrawNormalLine;
static ElementDrawSymbolProc DrawSymbol;
static ElementExtentsProc GetLineExtents;
static ElementToPostScriptProc ActiveLineToPostScript;
static ElementToPostScriptProc NormalLineToPostScript;
static ElementSymbolToPostScriptProc SymbolToPostScript;
static ElementMapProc MapLine;
static DistanceProc DistanceToY;
static DistanceProc DistanceToX;
static DistanceProc DistanceToLine;
static Blt_TileChangedProc TileChangedProc;

#ifdef WIN32

static int tkpWinRopModes[] =
{
    R2_BLACK,			/* GXclear */
    R2_MASKPEN,			/* GXand */
    R2_MASKPENNOT,		/* GXandReverse */
    R2_COPYPEN,			/* GXcopy */
    R2_MASKNOTPEN,		/* GXandInverted */
    R2_NOT,			/* GXnoop */
    R2_XORPEN,			/* GXxor */
    R2_MERGEPEN,		/* GXor */
    R2_NOTMERGEPEN,		/* GXnor */
    R2_NOTXORPEN,		/* GXequiv */
    R2_NOT,			/* GXinvert */
    R2_MERGEPENNOT,		/* GXorReverse */
    R2_NOTCOPYPEN,		/* GXcopyInverted */
    R2_MERGENOTPEN,		/* GXorInverted */
    R2_NOTMASKPEN,		/* GXnand */
    R2_WHITE			/* GXset */
};

#endif

INLINE static int
Round(double x)
{
    return (int) (x + ((x < 0.0) ? -0.5 : 0.5));
}

/*
 * ----------------------------------------------------------------------
 * 	Custom configuration option (parse and print) routines
 * ----------------------------------------------------------------------
 */

/*ARGSUSED*/
static Tcl_Obj *
PatternToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,
    char *widgRec,		/* Element information record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Pixmap pm = *(Pixmap *)(widgRec + offset);
    Tcl_Obj *objPtr;

    if (pm == None) {
	objPtr = Tcl_NewStringObj("", -1);
    } else if (pm == PATTERN_SOLID) {
	objPtr = Tcl_NewStringObj("solid", 5);
    } else {
	objPtr = Tcl_NewStringObj(Tk_NameOfBitmap(Tk_Display(tkwin), pm), -1);
    }
    return objPtr;
}

/*ARGSUSED*/
static void
FreePattern(
    ClientData clientData,
    Display *display,		/* Not used. */
    char *widgRec,
    int offset)
{
    Pixmap *stipplePtr = (Pixmap *)(widgRec + offset);

    if ((*stipplePtr != None) && (*stipplePtr != PATTERN_SOLID)) {
	Tk_FreeBitmap(display, *stipplePtr);
    }
    *stipplePtr = None;
}

/*ARGSUSED*/
static int
ObjToPattern(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representing field */
    char *widgRec,		/* Element information record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Pixmap *stipplePtr = (Pixmap *)(widgRec + offset);
    Pixmap stipple;
    char *string;

    string = Tcl_GetString(objPtr);
    if (string[0] == '\0') {
	stipple = None;
    } else if (strcmp(string, "solid") == 0) {
	stipple = PATTERN_SOLID;
    } else {
	stipple = Tk_AllocBitmapFromObj(interp, tkwin, objPtr);
	if (stipple == None) {
	    return TCL_ERROR;
	}
    }
    FreePattern(clientData, Tk_Display(tkwin), widgRec, offset);
    *stipplePtr = stipple;
    return TCL_OK;
}

static void
ReleaseSymbol(Display *display, Symbol *symbolPtr)
{
    if (symbolPtr->bitmap != None) {
	Tk_FreeBitmap(display, symbolPtr->bitmap);
    }
    symbolPtr->bitmap = None;
    if (symbolPtr->mask != None) {
	Tk_FreeBitmap(display, symbolPtr->mask);
    }
    symbolPtr->mask = None;
    symbolPtr->type = SYMBOL_NONE;
}

/*ARGSUSED*/
static void
FreeSymbol(
    ClientData clientData,
    Display *display,		/* Not used. */
    char *widgRec,
    int offset)
{
    Symbol *symbolPtr = (Symbol *)(widgRec + offset);

    ReleaseSymbol(display, symbolPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ObjToSymbol --
 *
 *	Convert the string representation of a line style or symbol name
 *	into its numeric form.
 *
 * Results:
 *	The return value is a standard Tcl result.  The symbol type is
 *	written into the widget record.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToSymbol(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representing symbol type */
    char *widgRec,		/* Element information record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Symbol *symbolPtr = (Symbol *)(widgRec + offset);
    size_t length;
    char c;
    char *string;
    Pixmap bitmap, mask;
    Tcl_Obj **objv;
    int objc;
    GraphSymbol *p;

    string = Tcl_GetString(objPtr);
    c = string[0];
    if (c == '\0') {
	ReleaseSymbol(Tk_Display(tkwin), symbolPtr);
	symbolPtr->type = SYMBOL_NONE;
	return TCL_OK;
    }
    length = strlen(string);
    for (p = graphSymbols; p->name != NULL; p++) {
	if (length < p->minChars) {
	    continue;
	}
	if ((c == p->name[0]) && (strncmp(string, p->name, length) == 0)) {
	    ReleaseSymbol(Tk_Display(tkwin), symbolPtr);
	    symbolPtr->type = p->type;
	    return TCL_OK;
	}
    }
    if ((Tcl_ListObjGetElements((Tcl_Interp *)NULL, objPtr, &objc, &objv) 
	 != TCL_OK) || (objc > 2)) {
	goto bitmapError;
    }
    bitmap = mask = None;
    if (objc > 0) {
	bitmap = Tk_AllocBitmapFromObj((Tcl_Interp *)NULL, tkwin, objv[0]);
	if (bitmap == None) {
	    goto bitmapError;
	}
    }
    if (objc > 1) {
	mask = Tk_AllocBitmapFromObj((Tcl_Interp *)NULL, tkwin, objv[1]);
	if (mask == None) {
	    goto bitmapError;
	}
    }
    ReleaseSymbol(Tk_Display(tkwin), symbolPtr);
    symbolPtr->bitmap = bitmap;
    symbolPtr->mask = mask;
    symbolPtr->type = SYMBOL_BITMAP;
    return TCL_OK;
 bitmapError:
    Tcl_AppendResult(interp, "bad symbol \"", string, 
	"\": should be \"none\", \"circle\", \"square\", \"diamond\", "
	"\"plus\", \"cross\", \"splus\", \"scross\", \"triangle\", "
	"\"arrow\" or the name of a bitmap", (char *)NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * SymbolToObj --
 *
 *	Convert the symbol value into a string.
 *
 * Results:
 *	The string representing the symbol type or line style is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
SymbolToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,
    char *widgRec,		/* Element information record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Symbol *symbolPtr = (Symbol *)(widgRec + offset);

    if (symbolPtr->type == SYMBOL_BITMAP) {
	Tcl_Obj *listObjPtr, *objPtr;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
	objPtr = Tcl_NewStringObj(Tk_NameOfBitmap(Tk_Display(tkwin), 
		symbolPtr->bitmap), -1);
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	if (symbolPtr->mask == None) {
	    objPtr = bltEmptyStringObjPtr;
	} else {
	    objPtr = Tcl_NewStringObj(Tk_NameOfBitmap(Tk_Display(tkwin), 
		symbolPtr->mask), 1);
	}
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	return listObjPtr;
    } else {
	GraphSymbol *p;

	for (p = graphSymbols; p->name != NULL; p++) {
	    if (p->type == symbolPtr->type) {
		return Tcl_NewStringObj(p->name, -1);
	    }
	}
	return Tcl_NewStringObj("?unknown symbol type?", -1);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * NameOfSmooth --
 *
 *	Converts the smooth value into its string representation.
 *
 * Results:
 *	The static string representing the smooth type is returned.
 *
 *----------------------------------------------------------------------
 */
static char *
NameOfSmooth(Smoothing value)
{
    SmoothingInfo *siPtr;

    for (siPtr = smoothingInfo; siPtr->name != NULL; siPtr++) {
	if (siPtr->value == value) {
	    return siPtr->name;
	}
    }
    return "unknown smooth value";
}

/*
 *----------------------------------------------------------------------
 *
 * ObjToSmooth --
 *
 *	Convert the string representation of a line style or smooth name
 *	into its numeric form.
 *
 * Results:
 *	The return value is a standard Tcl result.  The smooth type is
 *	written into the widget record.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToSmooth(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representing smooth type */
    char *widgRec,		/* Element information record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Smoothing *valuePtr = (Smoothing *)(widgRec + offset);
    SmoothingInfo *siPtr;
    char *string;

    string = Tcl_GetString(objPtr);
    for (siPtr = smoothingInfo; siPtr->name != NULL; siPtr++) {
	if (strcmp(string, siPtr->name) == 0) {
	    *valuePtr = siPtr->value;
	    return TCL_OK;
	}
    }
    Tcl_AppendResult(interp, "bad smooth value \"", string, "\": should be \
linear, step, natural, or quadratic", (char *)NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * SmoothToObj --
 *
 *	Convert the smooth value into a string.
 *
 * Results:
 *	The string representing the smooth type or line style is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
SmoothToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,
    char *widgRec,		/* Element information record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    int smooth = *(int *)(widgRec + offset);

    return Tcl_NewStringObj(NameOfSmooth(smooth), -1);
}

/*
 *----------------------------------------------------------------------
 *
 * ObjToPenDir --
 *
 *	Convert the string representation of a line style or symbol name
 *	into its numeric form.
 *
 * Results:
 *	The return value is a standard Tcl result.  The symbol type is
 *	written into the widget record.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToPenDir(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representing pen direction */
    char *widgRec,		/* Element information record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    int *penDirPtr = (int *)(widgRec + offset);
    size_t length;
    char c;
    char *string;

    string = Tcl_GetString(objPtr);
    c = string[0];
    length = strlen(string);
    if ((c == 'i') && (strncmp(string, "increasing", length) == 0)) {
	*penDirPtr = PEN_INCREASING;
    } else if ((c == 'd') && (strncmp(string, "decreasing", length) == 0)) {
	*penDirPtr = PEN_DECREASING;
    } else if ((c == 'b') && (strncmp(string, "both", length) == 0)) {
	*penDirPtr = PEN_BOTH_DIRECTIONS;
    } else {
	Tcl_AppendResult(interp, "bad trace value \"", string,
	    "\" : should be \"increasing\", \"decreasing\", or \"both\"",
	    (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NameOfPenDir --
 *
 *	Convert the pen direction into a string.
 *
 * Results:
 *	The static string representing the pen direction is returned.
 *
 *----------------------------------------------------------------------
 */
static char *
NameOfPenDir(penDir)
    int penDir;			/* Direction for pen drawing between points */
{
    switch (penDir) {
    case PEN_INCREASING:
	return "increasing";
    case PEN_DECREASING:
	return "decreasing";
    case PEN_BOTH_DIRECTIONS:
	return "both";
    default:
	return "unknown trace direction";
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PenDirToObj --
 *
 *	Convert the pen direction into a string.
 *
 * Results:
 *	The string representing the pen drawing direction is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
PenDirToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Element information record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    int penDir = *(int *)(widgRec + offset);

    return Tcl_NewStringObj(NameOfPenDir(penDir), -1);
}


/*
 * Reset the number of points and segments, in case there are no
 * segments or points
 */
static void
ResetStylePalette(Blt_Chain *stylePalette)
{
    Blt_ChainLink *linkPtr;

    for (linkPtr = Blt_ChainFirstLink(stylePalette); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	PenStyle *stylePtr;
	LinePenStyle *lineStylePtr;

	stylePtr = Blt_ChainGetValue(linkPtr);
	lineStylePtr = (LinePenStyle *)stylePtr;
	lineStylePtr->nStrips = lineStylePtr->nSymbolPts = 0;
	stylePtr->xErrorBarCnt = stylePtr->yErrorBarCnt = 0;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigurePen --
 *
 *	Sets up the appropriate configuration parameters in the GC.
 *      It is assumed the parameters have been previously set by
 *	a call to Blt_ConfigureWidget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information such as line width, line style, color
 *	etc. get set in a new GC.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigurePen(graphPtr, penPtr)
    Graph *graphPtr;
    Pen *penPtr;
{
    LinePen *lpPtr = (LinePen *)penPtr;
    unsigned long gcMask;
    GC newGC;
    XGCValues gcValues;
    XColor *colorPtr;

    /*
     * Set the outline GC for this pen: GCForeground is outline color.
     * GCBackground is the fill color (only used for bitmap symbols).
     */
    gcMask = (GCLineWidth | GCForeground);
    colorPtr = lpPtr->symbol.outlineColor;
    if (colorPtr == COLOR_DEFAULT) {
	colorPtr = lpPtr->traceColor;
    }
    gcValues.foreground = colorPtr->pixel;
    if (lpPtr->symbol.type == SYMBOL_BITMAP) {
	colorPtr = lpPtr->symbol.fillColor;
	if (colorPtr == COLOR_DEFAULT) {
	    colorPtr = lpPtr->traceColor;
	}
	/*
	 * Set a clip mask if either
	 *	1) no background color was designated or
	 *	2) a masking bitmap was specified.
	 *
	 * These aren't necessarily the bitmaps we'll be using for
	 * clipping. But this makes it unlikely that anyone else will
	 * be sharing this GC when we set the clip origin (at the time
	 * the bitmap is drawn).
	 */
	if (colorPtr != NULL) {
	    gcValues.background = colorPtr->pixel;
	    gcMask |= GCBackground;
	    if (lpPtr->symbol.mask != None) {
		gcValues.clip_mask = lpPtr->symbol.mask;
		gcMask |= GCClipMask;
	    }
	} else {
	    gcValues.clip_mask = lpPtr->symbol.bitmap;
	    gcMask |= GCClipMask;
	}
    }
    gcValues.line_width = LineWidth(lpPtr->symbol.outlineWidth);
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (lpPtr->symbol.outlineGC != NULL) {
	Tk_FreeGC(graphPtr->display, lpPtr->symbol.outlineGC);
    }
    lpPtr->symbol.outlineGC = newGC;

    /* Fill GC for symbols: GCForeground is fill color */

    gcMask = (GCLineWidth | GCForeground);
    colorPtr = lpPtr->symbol.fillColor;
    if (colorPtr == COLOR_DEFAULT) {
	colorPtr = lpPtr->traceColor;
    }
    newGC = NULL;
    if (colorPtr != NULL) {
	gcValues.foreground = colorPtr->pixel;
	newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    }
    if (lpPtr->symbol.fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, lpPtr->symbol.fillGC);
    }
    lpPtr->symbol.fillGC = newGC;

    /* Line segments */

    gcMask = (GCLineWidth | GCForeground | GCLineStyle | GCCapStyle |
	GCJoinStyle);
    gcValues.cap_style = CapButt;
    gcValues.join_style = JoinRound;
    gcValues.line_style = LineSolid;
    gcValues.line_width = LineWidth(lpPtr->traceWidth);

    colorPtr = lpPtr->traceOffColor;
    if (colorPtr == COLOR_DEFAULT) {
	colorPtr = lpPtr->traceColor;
    }
    if (colorPtr != NULL) {
	gcMask |= GCBackground;
	gcValues.background = colorPtr->pixel;
    }
    gcValues.foreground = lpPtr->traceColor->pixel;
    if (LineIsDashed(lpPtr->traceDashes)) {
	gcValues.line_width = lpPtr->traceWidth;
	gcValues.line_style = 
	    (colorPtr == NULL) ? LineOnOffDash : LineDoubleDash;
    }
    newGC = Blt_GetPrivateGC(graphPtr->tkwin, gcMask, &gcValues);
    if (lpPtr->traceGC != NULL) {
	Blt_FreePrivateGC(graphPtr->display, lpPtr->traceGC);
    }
    if (LineIsDashed(lpPtr->traceDashes)) {
	lpPtr->traceDashes.offset = lpPtr->traceDashes.values[0] / 2;
	Blt_SetDashes(graphPtr->display, newGC, &(lpPtr->traceDashes));
    }
    lpPtr->traceGC = newGC;

    gcMask = (GCLineWidth | GCForeground);
    colorPtr = lpPtr->errorBarColor;
    if (colorPtr == COLOR_DEFAULT) {
	colorPtr = lpPtr->traceColor;
    }
    gcValues.line_width = LineWidth(lpPtr->errorBarLineWidth);
    gcValues.foreground = colorPtr->pixel;
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (lpPtr->errorBarGC != NULL) {
	Tk_FreeGC(graphPtr->display, lpPtr->errorBarGC);
    }
    lpPtr->errorBarGC = newGC;

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyPen --
 *
 *	Release memory and resources allocated for the style.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the pen style is freed up.
 *
 *----------------------------------------------------------------------
 */
static void
DestroyPen(graphPtr, penPtr)
    Graph *graphPtr;
    Pen *penPtr;
{
    LinePen *lpPtr = (LinePen *)penPtr;

    Blt_TextFreeStyle(graphPtr->display, &lpPtr->valueStyle);
    if (lpPtr->symbol.outlineGC != NULL) {
	Tk_FreeGC(graphPtr->display, lpPtr->symbol.outlineGC);
    }
    if (lpPtr->symbol.fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, lpPtr->symbol.fillGC);
    }
    if (lpPtr->errorBarGC != NULL) {
	Tk_FreeGC(graphPtr->display, lpPtr->errorBarGC);
    }
    if (lpPtr->traceGC != NULL) {
	Blt_FreePrivateGC(graphPtr->display, lpPtr->traceGC);
    }
    if (lpPtr->symbol.bitmap != None) {
	Tk_FreeBitmap(graphPtr->display, lpPtr->symbol.bitmap);
	lpPtr->symbol.bitmap = None;
    }
    if (lpPtr->symbol.mask != None) {
	Tk_FreeBitmap(graphPtr->display, lpPtr->symbol.mask);
	lpPtr->symbol.mask = None;
    }
}


static void
InitPen(penPtr)
    LinePen *penPtr;
{
    Blt_TextInitStyle(penPtr->valueStyle);
    penPtr->errorBarLineWidth = 1;
    penPtr->errorBarShow = SHOW_BOTH;
    penPtr->base.configProc = ConfigurePen;
    penPtr->base.configSpecs = linePenConfigSpecs;
    penPtr->base.destroyProc = DestroyPen;
    penPtr->base.flags = NORMAL_PEN;
    penPtr->base.name = "";
    penPtr->symbol.bitmap = penPtr->symbol.mask = None;
    penPtr->symbol.outlineColor = penPtr->symbol.fillColor = COLOR_DEFAULT;
    penPtr->symbol.outlineWidth = penPtr->traceWidth = 1;
    penPtr->symbol.type = SYMBOL_CIRCLE;
    penPtr->valueShow = SHOW_NONE;
}

Pen *
Blt_LinePen(char *penName)
{
    LinePen *penPtr;

    penPtr = Blt_Calloc(1, sizeof(LinePen));
    assert(penPtr);
    InitPen(penPtr);
    penPtr->base.name = Blt_Strdup(penName);
    penPtr->base.classId = OBJECT_CLASS_LINE_ELEMENT;
    if (strcmp(penName, "activeLine") == 0) {
	penPtr->base.flags = ACTIVE_PEN;
    }
    return (Pen *)penPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 *	In this section, the routines deal with building and filling
 *	the element's data structures with transformed screen
 *	coordinates.  They are triggered from TranformLine which is
 *	called whenever the data or coordinates axes have changed and
 *	new screen coordinates need to be calculated.
 *
 * ----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 *
 * ScaleSymbol --
 *
 *	Returns the scaled size for the line element. Scaling depends
 *	upon when the base line ranges for the element were set and
 *	the current range of the graph.
 *
 * Results:
 *	The new size of the symbol, after considering how much the
 *	graph has been scaled, is returned.
 *
 *----------------------------------------------------------------------
 */
static int
ScaleSymbol(elemPtr, normalSize)
    Element *elemPtr;
    int normalSize;
{
    int maxSize;
    double scale;
    int newSize;

    scale = 1.0;
    if (elemPtr->scaleSymbols) {
	double xRange, yRange;

	xRange = (elemPtr->axes.x->max - elemPtr->axes.x->min);
	yRange = (elemPtr->axes.y->max - elemPtr->axes.y->min);
	if (elemPtr->flags & SCALE_SYMBOL) {
	    /* Save the ranges as a baseline for future scaling. */
	    elemPtr->xRange = xRange;
	    elemPtr->yRange = yRange;
	    elemPtr->flags &= ~SCALE_SYMBOL;
	} else {
	    double xScale, yScale;

	    /* Scale the symbol by the smallest change in the X or Y axes */
	    xScale = elemPtr->xRange / xRange;
	    yScale = elemPtr->yRange / yRange;
	    scale = MIN(xScale, yScale);
	}
    }
    newSize = Round(normalSize * scale);

    /*
     * Don't let the size of symbols go unbounded. Both X and Win32
     * drawing routines assume coordinates to be a signed short int.
     */
    maxSize = (int)MIN(elemPtr->object.graphPtr->hRange, 
	elemPtr->object.graphPtr->vRange);
    if (newSize > maxSize) {
	newSize = maxSize;
    }

    /* Make the symbol size odd so that its center is a single pixel. */
    newSize |= 0x01;
    return newSize;
}

/*
 *----------------------------------------------------------------------
 *
 * GetScreenPoints --
 *
 *	Generates a coordinate array of transformed screen coordinates
 *	from the data points.
 *
 * Results:
 *	The transformed screen coordinates are returned.
 *
 * Side effects:
 *	Memory is allocated for the coordinate array.
 *
 *----------------------------------------------------------------------
 */
static void
GetScreenPoints(Graph *graphPtr, Element *elemPtr, MapInfo *mapPtr)
{
    double *x, *y;
    int i, n;
    int count;
    Point2D *screenPts;
    int *indices;

    n = NUMBEROFPOINTS(elemPtr);
    x = elemPtr->x.valueArr;
    y = elemPtr->y.valueArr;
    screenPts = Blt_Malloc(sizeof(Point2D) * n);
    assert(screenPts);
    indices = Blt_Malloc(sizeof(int) * n);
    assert(indices);

    count = 0;			/* Count the valid screen coordinates */
    if (graphPtr->inverted) {
	for (i = 0; i < n; i++) {
	    if ((FINITE(x[i])) && (FINITE(y[i]))) {
 		screenPts[count].x = Blt_HMap(elemPtr->axes.y, y[i]);
		screenPts[count].y = Blt_VMap(elemPtr->axes.x, x[i]);
		indices[count] = i;
		count++;
	    }
	}
    } else {
	for (i = 0; i < n; i++) {
	    if ((FINITE(x[i])) && (FINITE(y[i]))) {
		screenPts[count].x = Blt_HMap(elemPtr->axes.x, x[i]);
		screenPts[count].y = Blt_VMap(elemPtr->axes.y, y[i]);
		indices[count] = i;
		count++;
	    }
	}
    }
    mapPtr->screenPts = screenPts;
    mapPtr->nScreenPts = count;
    mapPtr->indices = indices;
}

/*
 *----------------------------------------------------------------------
 *
 * ReducePoints --
 *
 *	Generates a coordinate array of transformed screen coordinates
 *	from the data points.
 *
 * Results:
 *	The transformed screen coordinates are returned.
 *
 * Side effects:
 *	Memory is allocated for the coordinate array.
 *
 *----------------------------------------------------------------------
 */
static void
ReducePoints(mapPtr, tolerance)
    MapInfo *mapPtr;
    double tolerance;
{
    int i, n;
    Point2D *screenPts;
    int *indices, *simple;

    simple	= Blt_Malloc(sizeof(int) * mapPtr->nScreenPts);
    indices	= Blt_Malloc(sizeof(int) * mapPtr->nScreenPts);
    screenPts	= Blt_Malloc(sizeof(Point2D) * mapPtr->nScreenPts);
    n = Blt_SimplifyLine(mapPtr->screenPts, 0, mapPtr->nScreenPts - 1, 
		 tolerance, simple);
    for (i = 0; i < n; i++) {
	int k;

	k = simple[i];
	screenPts[i] = mapPtr->screenPts[k];
	indices[i] = mapPtr->indices[k];
    }
#ifdef notdef
    if (n < mapPtr->nScreenPts) {
	fprintf(stderr, "reduced from %d to %d\n", mapPtr->nScreenPts, n);
    }
#endif
    Blt_Free(mapPtr->screenPts);
    Blt_Free(mapPtr->indices);
    Blt_Free(simple);
    mapPtr->screenPts = screenPts;
    mapPtr->indices = indices;
    mapPtr->nScreenPts = n;
}

/*
 *----------------------------------------------------------------------
 *
 * GenerateSteps --
 *
 *	Resets the coordinate and pen index arrays adding new points
 *	for step-and-hold type smoothing.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The temporary arrays for screen coordinates and pen indices
 *	are updated.
 *
 *---------------------------------------------------------------------- 
 */
static void
GenerateSteps(mapPtr)
    MapInfo *mapPtr;
{
    int newSize;
    int i, count;
    Point2D *screenPts;
    int *indices;

    newSize = ((mapPtr->nScreenPts - 1) * 2) + 1;
    screenPts = Blt_Malloc(newSize * sizeof(Point2D));
    assert(screenPts);
    indices = Blt_Malloc(sizeof(int) * newSize);
    assert(indices);

    screenPts[0] = mapPtr->screenPts[0];
    indices[0] = 0;

    count = 1;
    for (i = 1; i < mapPtr->nScreenPts; i++) {
	screenPts[count + 1] = mapPtr->screenPts[i];

	/* Hold last y-coordinate, use new x-coordinate */
	screenPts[count].x = screenPts[count + 1].x;
	screenPts[count].y = screenPts[count - 1].y;

	/* Use the same style for both the hold and the step points */
	indices[count] = indices[count + 1] = mapPtr->indices[i];
	count += 2;
    }
    Blt_Free(mapPtr->screenPts);
    Blt_Free(mapPtr->indices);
    mapPtr->indices = indices;
    mapPtr->screenPts = screenPts;
    mapPtr->nScreenPts = newSize;
}

/*
 *----------------------------------------------------------------------
 *
 * GenerateSpline --
 *
 *	Computes a spline based upon the data points, returning a new
 *	(larger) coordinate array or points.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The temporary arrays for screen coordinates and data indices
 *	are updated based upon spline.
 *
 * FIXME:  Can't interpolate knots along the Y-axis.   Need to break
 *	   up point array into interchangable X and Y vectors earlier.
 *	   Pass extents (left/right or top/bottom) as parameters.
 *
 *----------------------------------------------------------------------
 */
static void
GenerateSpline(
    Graph *graphPtr,
    Line *linePtr,
    MapInfo *mapPtr)
{
    Point2D *origPts, *intpPts;
    int *indices;
    int extra;
    int nIntpPts, nOrigPts;
    int result;
    int i, j, count;

    nOrigPts = mapPtr->nScreenPts;
    origPts = mapPtr->screenPts;
    assert(mapPtr->nScreenPts > 0);
    for (i = 0, j = 1; j < nOrigPts; i++, j++) {
	if (origPts[j].x <= origPts[i].x) {
	    return;		/* Points are not monotonically increasing */
	}
    }
    if (((origPts[0].x > (double)graphPtr->right)) ||
	((origPts[mapPtr->nScreenPts - 1].x < (double)graphPtr->left))) {
	return;			/* All points are clipped */
    }
    /*
     * The spline is computed in screen coordinates instead of data
     * points so that we can select the abscissas of the interpolated
     * points from each pixel horizontally across the plotting area.
     */
    extra = (graphPtr->right - graphPtr->left) + 1;
    if (extra < 1) {
	return;
    }
    nIntpPts = nOrigPts + extra + 1;
    intpPts = Blt_Malloc(nIntpPts * sizeof(Point2D));
    assert(intpPts);

    indices = Blt_Malloc(sizeof(int) * nIntpPts);
    assert(indices);

    /* Populate the x2 array with both the original X-coordinates and
     * extra X-coordinates for each horizontal pixel that the line
     * segment contains. */
    count = 0;
    for (i = 0, j = 1; j < nOrigPts; i++, j++) {

	/* Add the original x-coordinate */
	intpPts[count].x = origPts[i].x;

	/* Include the starting offset of the point in the offset array */
	indices[count] = mapPtr->indices[i];
	count++;

	/* Is any part of the interval (line segment) in the plotting
	 * area?  */
	if ((origPts[j].x >= (double)graphPtr->left) || 
	    (origPts[i].x <= (double)graphPtr->right)) {
	    double x, last;

	    x = origPts[i].x + 1.0;

	    /*
	     * Since the line segment may be partially clipped on the
	     * left or right side, the points to interpolate are
	     * always interior to the plotting area.
	     *
	     *           left			    right
	     *      x1----|--------------------------|---x2
	     *
	     * Pick the max of the starting X-coordinate and the
	     * left edge and the min of the last X-coordinate and
	     * the right edge.
	     */
	    x = MAX(x, (double)graphPtr->left);
	    last = MIN(origPts[j].x, (double)graphPtr->right);

	    /* Add the extra x-coordinates to the interval. */
	    while (x < last) {
		indices[count] = mapPtr->indices[i];
		intpPts[count++].x = x;
		x++;
	    }
	}
    }
    nIntpPts = count;
    result = FALSE;
    if (linePtr->smooth == PEN_SMOOTH_NATURAL) {
	result = Blt_NaturalSpline(origPts, nOrigPts, intpPts, nIntpPts);
    } else if (linePtr->smooth == PEN_SMOOTH_QUADRATIC) {
	result = Blt_QuadraticSpline(origPts, nOrigPts, intpPts, nIntpPts);
    }
    if (!result) {
	/* The spline interpolation failed.  We'll fallback to the
	 * current coordinates and do no smoothing (standard line
	 * segments).  */
	linePtr->smooth = PEN_SMOOTH_LINEAR;
	Blt_Free(intpPts);
	Blt_Free(indices);
    } else {
	Blt_Free(mapPtr->screenPts);
	Blt_Free(mapPtr->indices);
	mapPtr->indices = indices;
	mapPtr->screenPts = intpPts;
	mapPtr->nScreenPts = nIntpPts;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * GenerateParametricSpline --
 *
 *	Computes a spline based upon the data points, returning a new
 *	(larger) coordinate array or points.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The temporary arrays for screen coordinates and data indices
 *	are updated based upon spline.
 *
 * FIXME:  Can't interpolate knots along the Y-axis.   Need to break
 *	   up point array into interchangable X and Y vectors earlier.
 *	   Pass extents (left/right or top/bottom) as parameters.
 *
 *----------------------------------------------------------------------
 */
static void
GenerateParametricSpline(graphPtr, linePtr, mapPtr)
    Graph *graphPtr;
    Line *linePtr;
    MapInfo *mapPtr;
{
    Region2D exts;
    Point2D *origPts, *intpPts;
    int *indices;
    int nIntpPts, nOrigPts;
    int result;
    int i, j, count;

    nOrigPts = mapPtr->nScreenPts;
    origPts = mapPtr->screenPts;
    assert(mapPtr->nScreenPts > 0);

    Blt_GraphExtents(graphPtr, &exts);

    /* 
     * Populate the x2 array with both the original X-coordinates and
     * extra X-coordinates for each horizontal pixel that the line
     * segment contains. 
     */
    count = 1;
    for (i = 0, j = 1; j < nOrigPts; i++, j++) {
	Point2D p, q;

        p = origPts[i];
        q = origPts[j];
	count++;
        if (Blt_LineRectClip(&exts, &p, &q)) {
	    count += (int)(hypot(q.x - p.x, q.y - p.y) * 0.5);
	}
    }
    nIntpPts = count;
    intpPts = Blt_Malloc(nIntpPts * sizeof(Point2D));
    assert(intpPts);

    indices = Blt_Malloc(sizeof(int) * nIntpPts);
    assert(indices);

    /* 
     * FIXME: This is just plain wrong.  The spline should be computed
     *        and evaluated in separate steps.  This will mean breaking
     *	      up this routine since the catrom coefficients can be
     *	      independently computed for original data point.  This 
     *	      also handles the problem of allocating enough points 
     *	      since evaluation is independent of the number of points 
     *		to be evalualted.  The interpolated 
     *	      line segments should be clipped, not the original segments.
     */
    count = 0;
    for (i = 0, j = 1; j < nOrigPts; i++, j++) {
	Point2D p, q;
	double dist;

        p = origPts[i];
        q = origPts[j];

        dist = hypot(q.x - p.x, q.y - p.y);
        /* Add the original x-coordinate */
        intpPts[count].x = (double)i;
        intpPts[count].y = 0.0;

        /* Include the starting offset of the point in the offset array */
        indices[count] = mapPtr->indices[i];
        count++;

        /* Is any part of the interval (line segment) in the plotting
         * area?  */

        if (Blt_LineRectClip(&exts, &p, &q)) {
            double distP, distQ;

            distP = hypot(p.x - origPts[i].x, p.y - origPts[i].y);
            distQ = hypot(q.x - origPts[i].x, q.y - origPts[i].y);
            distP += 2.0;
            while(distP <= distQ) {
                /* Point is indicated by its interval and parameter t. */
                intpPts[count].x = (double)i;
                intpPts[count].y =  distP / dist;
                indices[count] = mapPtr->indices[i];
                count++;
                distP += 2.0;
            }
        }
    }
    intpPts[count].x = (double)i;
    intpPts[count].y = 0.0;
    indices[count] = mapPtr->indices[i];
    count++;
    nIntpPts = count;
    result = FALSE;
    if (linePtr->smooth == PEN_SMOOTH_NATURAL) {
        result = Blt_NaturalParametricSpline(origPts, nOrigPts, &exts, FALSE,
                            intpPts, nIntpPts);
    } else if (linePtr->smooth == PEN_SMOOTH_CATROM) {
        result = Blt_CatromParametricSpline(origPts, nOrigPts, intpPts,
                                            nIntpPts);
    }
    if (!result) {
        /* The spline interpolation failed.  We'll fallback to the
         * current coordinates and do no smoothing (standard line
         * segments).  */
        linePtr->smooth = PEN_SMOOTH_LINEAR;
        Blt_Free(intpPts);
        Blt_Free(indices);
    } else {
        Blt_Free(mapPtr->screenPts);
        Blt_Free(mapPtr->indices);
        mapPtr->indices = indices;
        mapPtr->screenPts = intpPts;
        mapPtr->nScreenPts = nIntpPts;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * MapSymbols --
 *
 *	Creates two arrays of points and pen indices, filled with
 *	the screen coordinates of the visible
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is freed and allocated for the index array.
 *
 *----------------------------------------------------------------------
 */
static void
MapSymbols(
    Graph *graphPtr,
    Element *elemPtr,
    MapInfo *mapPtr)
{
    Line *linePtr = (Line *)elemPtr;
    Region2D exts;
    Point2D *pp, *symbolPts;
    int *indices;
    int i, count;

    symbolPts = Blt_Malloc(sizeof(Point2D) * mapPtr->nScreenPts);
    assert(symbolPts);

    indices = Blt_Malloc(sizeof(int) * mapPtr->nScreenPts);
    assert(indices);

    Blt_GraphExtents(graphPtr, &exts);
    count = 0;			/* Count the number of visible points */

    for (pp = mapPtr->screenPts, i = 0; i < mapPtr->nScreenPts; i++, pp++) {
	if (PointInRegion(&exts, pp->x, pp->y)) {
	    symbolPts[count].x = pp->x;
	    symbolPts[count].y = pp->y;
	    indices[count] = mapPtr->indices[i];
	    count++;
	}
    }
    linePtr->symbolPts = symbolPts;
    linePtr->nSymbolPts = count;
    linePtr->symbolToData = indices;
}

/*
 *----------------------------------------------------------------------
 *
 * MapActiveSymbols --
 *
 *	Creates an array of points of the active graph coordinates.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is freed and allocated for the active point array.
 *
 *----------------------------------------------------------------------
 */
static void
MapActiveSymbols(Graph *graphPtr, Element *elemPtr)
{
    Line *linePtr = (Line *)elemPtr;
    Point2D *activePts;
    Region2D exts;
    int *activeToData;
    int count, i, nPoints;

    if (linePtr->activePts != NULL) {
	Blt_Free(linePtr->activePts);
	linePtr->activePts = NULL;
    }
    if (linePtr->activeToData != NULL) {
	Blt_Free(linePtr->activeToData);
	linePtr->activeToData = NULL;
    }
    Blt_GraphExtents(graphPtr, &exts);
    activePts = Blt_Malloc(sizeof(Point2D) * elemPtr->nActiveIndices);
    assert(activePts);
    activeToData = Blt_Malloc(sizeof(int) * elemPtr->nActiveIndices);
    nPoints = NUMBEROFPOINTS(elemPtr);
    count = 0;			/* Count the visible active points */
    for (i = 0; i < elemPtr->nActiveIndices; i++) {
	double x, y;
	int iPoint;

	iPoint = elemPtr->activeIndices[i];
	if (iPoint >= nPoints) {
	    continue;		/* Index not available */
	}
	x = elemPtr->x.valueArr[iPoint];
	y = elemPtr->y.valueArr[iPoint];
	activePts[count] = Blt_Map2D(graphPtr, x, y, &elemPtr->axes);
	activeToData[count] = iPoint;
	if (PointInRegion(&exts, activePts[count].x, activePts[count].y)) {
	    count++;
	}
    }
    if (count > 0) {
	linePtr->activePts = activePts;
	linePtr->activeToData = activeToData;
    } else {
	/* No active points were visible. */
	Blt_Free(activePts);
	Blt_Free(activeToData);	
    }
    linePtr->nActivePts = count;
    elemPtr->flags &= ~ACTIVE_PENDING;
}

/*
 *----------------------------------------------------------------------
 *
 * MapStrip --
 *
 *	Creates an array of line segments of the graph coordinates.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is  allocated for the line segment array.
 *
 *----------------------------------------------------------------------
 */
static void
MapStrip(graphPtr, linePtr, mapPtr)
    Graph *graphPtr;
    Line *linePtr;
    MapInfo *mapPtr;
{
    Region2D exts;
    Segment2D *strips;
    int *indices, *indexPtr;
    Point2D *pend, *pp;
    Segment2D *sp;
    int count;

    indices = Blt_Malloc(sizeof(int) * mapPtr->nScreenPts);
    assert(indices);

    /* 
     * Create array to hold points for line segments (not polyline 
     * coordinates).  So allocate twice the number of points.  
     */
    sp = strips = Blt_Malloc(mapPtr->nScreenPts * sizeof(Segment2D));
    assert(strips);

    Blt_GraphExtents(graphPtr, &exts);
    count = 0;			/* Count the number of segments. */
    indexPtr = mapPtr->indices;
    for (pp = mapPtr->screenPts, pend = pp + (mapPtr->nScreenPts - 1); 
	pp < pend; pp++, indexPtr++) {
	sp->p = pp[0], sp->q = pp[1];
	if (Blt_LineRectClip(&exts, &sp->p, &sp->q)) {
	    sp++;
	    indices[count] = *indexPtr;
	    count++;
	}
    }
    linePtr->stripToData = indices;
    linePtr->nStrips = count;
    linePtr->strips = strips;
}

/*
 *----------------------------------------------------------------------
 *
 * MergePens --
 *
 *	Reorders the both arrays of points and segments to merge pens.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The old arrays are freed and new ones allocated containing
 *	the reordered points and segments.
 *
 *----------------------------------------------------------------------
 */
static void
MergePens(Element *elemPtr, PenStyle **dataToStyle)
{
    Line *linePtr = (Line *)elemPtr;

    if (Blt_ChainGetLength(elemPtr->stylePalette) < 2) {
	Blt_ChainLink *linkPtr;
	LinePenStyle *stylePtr;

	linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette);
	stylePtr = Blt_ChainGetValue(linkPtr);
	stylePtr->nStrips = linePtr->nStrips;
	stylePtr->strips = linePtr->strips;
	stylePtr->nSymbolPts = linePtr->nSymbolPts;
	stylePtr->symbolPts = linePtr->symbolPts;
	stylePtr->base.xErrorBarCnt = elemPtr->xErrorBarCnt;
	stylePtr->base.yErrorBarCnt = elemPtr->yErrorBarCnt;
	stylePtr->base.xErrorBars = elemPtr->xErrorBars;
	stylePtr->base.yErrorBars = elemPtr->yErrorBars;
	stylePtr->base.errorBarCapWidth = elemPtr->errorBarCapWidth;
	return;
    }

    /* We have more than one style. Group line segments and points of
     * like pen styles.  */

    if (linePtr->nStrips > 0) {
	Blt_ChainLink *linkPtr;
	Segment2D *sp, *strips;
	int *ip;
	int *stripToData;

	strips = Blt_Malloc(linePtr->nStrips * sizeof(Segment2D));
	stripToData = Blt_Malloc(linePtr->nStrips * sizeof(int));
	assert(strips && stripToData);
	sp = strips, ip = stripToData;
	for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); 
	     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	    LinePenStyle *stylePtr;
	    int i;

	    stylePtr = Blt_ChainGetValue(linkPtr);
	    stylePtr->strips = sp;
	    for (i = 0; i < linePtr->nStrips; i++) {
		int iData;

		iData = linePtr->stripToData[i];
		if (dataToStyle[iData] == (PenStyle *)stylePtr) {
		    *sp++ = linePtr->strips[i];
		    *ip++ = iData;
		}
	    }
	    stylePtr->nStrips = sp - stylePtr->strips;
	}
	Blt_Free(linePtr->strips);
	linePtr->strips = strips;
	Blt_Free(linePtr->stripToData);
	linePtr->stripToData = stripToData;
    }
    if (linePtr->nSymbolPts > 0) {
	Blt_ChainLink *linkPtr;
	int *ip;
	Point2D *symbolPts, *pp;
	int *symbolToData;

	symbolPts = Blt_Malloc(linePtr->nSymbolPts * sizeof(Point2D));
	symbolToData = Blt_Malloc(linePtr->nSymbolPts * sizeof(int));
	assert(symbolPts && symbolToData);
	pp = symbolPts, ip = symbolToData;
	for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); 
	     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	    LinePenStyle *stylePtr;
	    int i;

	    stylePtr = Blt_ChainGetValue(linkPtr);
	    stylePtr->symbolPts = pp;
	    for (i = 0; i < linePtr->nSymbolPts; i++) {
		int iData;

		iData = linePtr->symbolToData[i];
		if (dataToStyle[iData] == (PenStyle *)stylePtr) {
		    *pp++ = linePtr->symbolPts[i];
		    *ip++ = iData;
		}
	    }
	    stylePtr->nSymbolPts = pp - stylePtr->symbolPts;
	}
	Blt_Free(linePtr->symbolPts);
	linePtr->symbolPts = symbolPts;
	Blt_Free(linePtr->symbolToData);
	linePtr->symbolToData = symbolToData;
    }
    if (elemPtr->xErrorBarCnt > 0) {
	Segment2D *xErrorBars, *sp;
	int *xErrorToData, *ip;
	Blt_ChainLink *linkPtr;

	xErrorBars = Blt_Malloc(elemPtr->xErrorBarCnt * sizeof(Segment2D));
	xErrorToData = Blt_Malloc(elemPtr->xErrorBarCnt * sizeof(int));
	assert(xErrorBars);
	sp = xErrorBars, ip = xErrorToData;
	for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); 
	     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	    PenStyle *stylePtr;
	    int i;

	    stylePtr = Blt_ChainGetValue(linkPtr);
	    stylePtr->xErrorBars = sp;
	    for (i = 0; i < elemPtr->xErrorBarCnt; i++) {
		int iData;

		iData = elemPtr->xErrorToData[i];
		if (dataToStyle[iData] == stylePtr) {
		    *sp++ = elemPtr->xErrorBars[i];
		    *ip++ = iData;
		}
	    }
	    stylePtr->xErrorBarCnt = sp - stylePtr->xErrorBars;
	}
	Blt_Free(elemPtr->xErrorBars);
	elemPtr->xErrorBars = xErrorBars;
	Blt_Free(elemPtr->xErrorToData);
	elemPtr->xErrorToData = xErrorToData;
    }
    if (elemPtr->yErrorBarCnt > 0) {
	Segment2D *errorBars, *sp;
	int *errorToData, *ip;
	Blt_ChainLink *linkPtr;

	errorBars = Blt_Malloc(elemPtr->yErrorBarCnt * sizeof(Segment2D));
	errorToData = Blt_Malloc(elemPtr->yErrorBarCnt * sizeof(int));
	assert(errorBars);
	sp = errorBars, ip = errorToData;
	for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); 
	     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	    PenStyle *stylePtr;
	    int i;

	    stylePtr = Blt_ChainGetValue(linkPtr);
	    stylePtr->yErrorBars = sp;
	    for (i = 0; i < elemPtr->yErrorBarCnt; i++) {
		int iData;

		iData = elemPtr->yErrorToData[i];
		if (dataToStyle[iData] == stylePtr) {
		    *sp++ = elemPtr->yErrorBars[i];
		    *ip++ = iData;
		}
	    }
	    stylePtr->yErrorBarCnt = sp - stylePtr->yErrorBars;
	}
	Blt_Free(elemPtr->yErrorBars);
	elemPtr->yErrorBars = errorBars;
	Blt_Free(elemPtr->yErrorToData);
	elemPtr->yErrorToData = errorToData;
    }
}

#define CLIP_TOP	(1<<0)
#define CLIP_BOTTOM	(1<<1)
#define CLIP_RIGHT	(1<<2)
#define CLIP_LEFT	(1<<3)

INLINE static int
OutCode(extsPtr, p)
    Region2D *extsPtr;
    Point2D *p;
{
    int code;

    code = 0;
    if (p->x > extsPtr->right) {
	code |= CLIP_RIGHT;
    } else if (p->x < extsPtr->left) {
	code |= CLIP_LEFT;
    }
    if (p->y > extsPtr->bottom) {
	code |= CLIP_BOTTOM;
    } else if (p->y < extsPtr->top) {
	code |= CLIP_TOP;
    }
    return code;
}

static int
ClipSegment(
    Region2D *extsPtr,
    int code1, int code2,
    Point2D *p, Point2D *q)
{
    int inside, outside;

    inside = ((code1 | code2) == 0);
    outside = ((code1 & code2) != 0);

    /*
     * In the worst case, we'll clip the line segment against each of
     * the four sides of the bounding rectangle.
     */
    while ((!outside) && (!inside)) {
	if (code1 == 0) {
	    Point2D *tmp;
	    int code;

	    /* Swap pointers and out codes */
	    tmp = p, p = q, q = tmp;
	    code = code1, code1 = code2, code2 = code;
	}
	if (code1 & CLIP_LEFT) {
	    p->y += (q->y - p->y) *
		(extsPtr->left - p->x) / (q->x - p->x);
	    p->x = extsPtr->left;
	} else if (code1 & CLIP_RIGHT) {
	    p->y += (q->y - p->y) *
		(extsPtr->right - p->x) / (q->x - p->x);
	    p->x = extsPtr->right;
	} else if (code1 & CLIP_BOTTOM) {
	    p->x += (q->x - p->x) *
		(extsPtr->bottom - p->y) / (q->y - p->y);
	    p->y = extsPtr->bottom;
	} else if (code1 & CLIP_TOP) {
	    p->x += (q->x - p->x) *
		(extsPtr->top - p->y) / (q->y - p->y);
	    p->y = extsPtr->top;
	}
	code1 = OutCode(extsPtr, p);

	inside = ((code1 | code2) == 0);
	outside = ((code1 & code2) != 0);
    }
    return (!inside);
}

/*
 *----------------------------------------------------------------------
 *
 * SaveTrace --
 *
 *	Creates a new trace and inserts it into the line's
 *	list of traces.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
SaveTrace(linePtr, start, length, mapPtr)
    Line *linePtr;
    int start;			/* Starting index of the trace in data point
				 * array.  Used to figure out closest point */
    int length;			/* Number of points forming the trace */
    MapInfo *mapPtr;
{
    Trace *tracePtr;
    Point2D *screenPts;
    int *indices;
    int i, j;

    tracePtr = Blt_Malloc(sizeof(Trace));
    assert(tracePtr);
    screenPts = Blt_Malloc(sizeof(Point2D) * length);
    assert(screenPts);
    indices = Blt_Malloc(sizeof(int) * length);
    assert(indices);

    /* Copy the screen coordinates of the trace into the point array */

    if (mapPtr->indices != NULL) {
	for (i = 0, j = start; i < length; i++, j++) {
	    screenPts[i].x = mapPtr->screenPts[j].x;
	    screenPts[i].y = mapPtr->screenPts[j].y;
	    indices[i] = mapPtr->indices[j];
	}
    } else {
	for (i = 0, j = start; i < length; i++, j++) {
	    screenPts[i].x = mapPtr->screenPts[j].x;
	    screenPts[i].y = mapPtr->screenPts[j].y;
	    indices[i] = j;
	}
    }
    tracePtr->nScreenPts = length;
    tracePtr->screenPts = screenPts;
    tracePtr->symbolToData = indices;
    tracePtr->start = start;
    if (linePtr->traces == NULL) {
	linePtr->traces = Blt_ChainCreate();
    }
    Blt_ChainAppend(linePtr->traces, tracePtr);
}

/*
 *----------------------------------------------------------------------
 *
 * FreeTraces --
 *
 *	Deletes all the traces for the line.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
FreeTraces(linePtr)
    Line *linePtr;
{
    Blt_ChainLink *linkPtr;

    for (linkPtr = Blt_ChainFirstLink(linePtr->traces); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	Trace *tracePtr;

	tracePtr = Blt_ChainGetValue(linkPtr);
	Blt_Free(tracePtr->symbolToData);
	Blt_Free(tracePtr->screenPts);
	Blt_Free(tracePtr);
    }
    Blt_ChainDestroy(linePtr->traces);
    linePtr->traces = NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * MapTraces --
 *
 *	Creates an array of line segments of the graph coordinates.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is  allocated for the line segment array.
 *
 *----------------------------------------------------------------------
 */
static void
MapTraces(graphPtr, linePtr, mapPtr)
    Graph *graphPtr;
    Line *linePtr;
    MapInfo *mapPtr;
{
    Point2D *p, *q;
    Region2D exts;
    int code1;
    int i;
    int start, count;

    Blt_GraphExtents(graphPtr, &exts);
    count = 1;
    code1 = OutCode(&exts, mapPtr->screenPts);
    p = mapPtr->screenPts;
    q = p + 1;
    for (i = 1; i < mapPtr->nScreenPts; i++, p++, q++) {
	Point2D s;
	int code2;
	int broken, offscreen;

	code2 = OutCode(&exts, q);
	if (code2 != 0) {
	    /* Save the coordinates of the last point, before clipping */
	    s = *q;
	}
	broken = BROKEN_TRACE(linePtr->penDir, p->x, q->x);
	offscreen = ClipSegment(&exts, code1, code2, p, q);
	if (broken || offscreen) {

	    /*
	     * The last line segment is either totally clipped by the plotting
	     * area or the x-direction is wrong, breaking the trace.  Either
	     * way, save information about the last trace (if one exists),
	     * discarding the current line segment
	     */

	    if (count > 1) {
		start = i - count;
		SaveTrace(linePtr, start, count, mapPtr);
		count = 1;
	    }
	} else {
	    count++;		/* Add the point to the trace. */
	    if (code2 != 0) {

		/*
		 * If the last point is clipped, this means that the trace is
		 * broken after this point.  Restore the original coordinate
		 * (before clipping) after saving the trace.
		 */

		start = i - (count - 1);
		SaveTrace(linePtr, start, count, mapPtr);
		mapPtr->screenPts[i] = s;
		count = 1;
	    }
	}
	code1 = code2;
    }
    if (count > 1) {
	start = i - count;
	SaveTrace(linePtr, start, count, mapPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MapFillArea --
 *
 *	Creates an array of points that represent a polygon that fills
 *	the area under the element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is  allocated for the polygon point array.
 *
 *----------------------------------------------------------------------
 */
static void
MapFillArea(Graph *graphPtr, Element *elemPtr, MapInfo *mapPtr)
{
    Line *linePtr = (Line *)elemPtr;
    Point2D *origPts, *clipPts;
    Region2D exts;
    int n;

    if (linePtr->fillPts != NULL) {
	Blt_Free(linePtr->fillPts);
	linePtr->fillPts = NULL;
	linePtr->nFillPts = 0;
    }
    if (mapPtr->nScreenPts < 3) {
	return;
    }
    n = mapPtr->nScreenPts + 3;
    Blt_GraphExtents(graphPtr, &exts);

    origPts = Blt_Malloc(sizeof(Point2D) * n);
    if (graphPtr->inverted) {
	double minX;
	int i;

	minX = (double)elemPtr->axes.y->screenMin;
	for (i = 0; i < mapPtr->nScreenPts; i++) {
	    origPts[i].x = mapPtr->screenPts[i].x + 1;
	    origPts[i].y = mapPtr->screenPts[i].y;
	    if (origPts[i].x < minX) {
		minX = origPts[i].x;
	    }
	}	
	/* Add edges to make (if necessary) the polygon fill to the
	 * bottom of plotting window */
	origPts[i].x = minX;
	origPts[i].y = origPts[i - 1].y;
	i++;
	origPts[i].x = minX;
	origPts[i].y = origPts[0].y; 
	i++;
	origPts[i] = origPts[0];
    } else {
	double maxY;
	int i;

	maxY = (double)elemPtr->axes.y->bottom;
	for (i = 0; i < mapPtr->nScreenPts; i++) {
	    origPts[i].x = mapPtr->screenPts[i].x + 1;
	    origPts[i].y = mapPtr->screenPts[i].y;
	    if (origPts[i].y > maxY) {
		maxY = origPts[i].y;
	    }
	}	
	/* Add edges to extend the fill polygon to the bottom of
	 * plotting window */
	origPts[i].x = origPts[i - 1].x;
	origPts[i].y = maxY;
	i++;
	origPts[i].x = origPts[0].x; 
	origPts[i].y = maxY;
	i++;
	origPts[i] = origPts[0];
    }

    clipPts = Blt_Malloc(sizeof(Point2D) * n * 3);
    assert(clipPts);
    n = Blt_PolyRectClip(&exts, origPts, n - 1, clipPts);

    Blt_Free(origPts);
    if (n < 3) {
	Blt_Free(clipPts);
    } else {
	linePtr->fillPts = clipPts;
	linePtr->nFillPts = n;
    }
}

static void
ResetLine(Element *elemPtr)
{
    Line *linePtr = (Line *)elemPtr;

    FreeTraces(linePtr);
    ResetStylePalette(elemPtr->stylePalette);
    if (linePtr->symbolPts != NULL) {
	Blt_Free(linePtr->symbolPts);
    }
    if (linePtr->symbolToData != NULL) {
	Blt_Free(linePtr->symbolToData);
    }
    if (linePtr->strips != NULL) {
	Blt_Free(linePtr->strips);
    }
    if (linePtr->stripToData != NULL) {
	Blt_Free(linePtr->stripToData);
    }
    if (linePtr->activePts != NULL) {
	Blt_Free(linePtr->activePts);
    }
    if (linePtr->activeToData != NULL) {
	Blt_Free(linePtr->activeToData);
    }
    if (elemPtr->xErrorBars != NULL) {
	Blt_Free(elemPtr->xErrorBars);
    }
    if (elemPtr->xErrorToData != NULL) {
	Blt_Free(elemPtr->xErrorToData);
    }
    if (elemPtr->yErrorBars != NULL) {
	Blt_Free(elemPtr->yErrorBars);
    }
    if (elemPtr->yErrorToData != NULL) {
	Blt_Free(elemPtr->yErrorToData);
    }
    elemPtr->xErrorBars = elemPtr->yErrorBars = linePtr->strips = NULL;
    linePtr->symbolPts = linePtr->activePts = NULL;
    linePtr->stripToData = linePtr->symbolToData = elemPtr->xErrorToData = 
	elemPtr->yErrorToData = linePtr->activeToData = NULL;
    linePtr->nActivePts = linePtr->nSymbolPts = linePtr->nStrips = 
	elemPtr->xErrorBarCnt = elemPtr->yErrorBarCnt = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * MapLine --
 *
 *	Calculates the actual window coordinates of the line element.
 *	The window coordinates are saved in an allocated point array.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is (re)allocated for the point array.
 *
 *----------------------------------------------------------------------
 */
static void
MapLine(
    Graph *graphPtr,		/* Graph widget record */
    Element *elemPtr)		/* Element component record */
{
    Line *linePtr = (Line *)elemPtr;
    MapInfo mapInfo;
    int size, nPoints;
    PenStyle **dataToStyle;
    Blt_ChainLink *linkPtr;
    
    ResetLine(elemPtr);
    nPoints = NUMBEROFPOINTS(elemPtr);
    if (nPoints < 1) {
	return;			/* No data points */
    }
    GetScreenPoints(graphPtr, elemPtr, &mapInfo);
    MapSymbols(graphPtr, elemPtr, &mapInfo);

    if ((elemPtr->flags & ACTIVE_PENDING) && (elemPtr->nActiveIndices > 0)) {
	MapActiveSymbols(graphPtr, elemPtr);
    }
    /*
     * Map connecting line segments if they are to be displayed.
     */
    linePtr->smooth = linePtr->reqSmooth;
    if ((nPoints > 1) && ((graphPtr->classId == OBJECT_CLASS_STRIP_ELEMENT) ||
	    (linePtr->builtinPen.traceWidth > 0))) {

	/*
	 * Do smoothing if necessary.  This can extend the coordinate array,
	 * so both mapInfo.points and mapInfo.nPoints may change.
	 */

	switch (linePtr->smooth) {
	case PEN_SMOOTH_STEP:
	    GenerateSteps(&mapInfo);
	    break;

	case PEN_SMOOTH_NATURAL:
	case PEN_SMOOTH_QUADRATIC:
	    if (mapInfo.nScreenPts < 3) {
		/* Can't interpolate with less than three points. */
		linePtr->smooth = PEN_SMOOTH_LINEAR;
	    } else {
		GenerateSpline(graphPtr, linePtr, &mapInfo);
	    }
	    break;

	case PEN_SMOOTH_CATROM:
	    if (mapInfo.nScreenPts < 3) {
		/* Can't interpolate with less than three points. */
		linePtr->smooth = PEN_SMOOTH_LINEAR;
	    } else {
		GenerateParametricSpline(graphPtr, linePtr, &mapInfo);
	    }
	    break;

	default:
	    break;
	}
	if (linePtr->rTolerance > 0.0) {
	    ReducePoints(&mapInfo, linePtr->rTolerance);
	}
	if ((linePtr->fillTile != NULL) || (linePtr->fillStipple != None)) {
	    MapFillArea(graphPtr, elemPtr, &mapInfo);
	}
	if (graphPtr->classId == OBJECT_CLASS_STRIP_ELEMENT) {
	    MapStrip(graphPtr, linePtr, &mapInfo);
	} else {
	    MapTraces(graphPtr, linePtr, &mapInfo);
	}
    }
    Blt_Free(mapInfo.screenPts);
    Blt_Free(mapInfo.indices);

    /* Set the symbol size of all the pen styles. */
    for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	PenStyle *stylePtr;
	LinePen *penPtr;

	stylePtr = Blt_ChainGetValue(linkPtr);
	penPtr = (LinePen *)stylePtr->penPtr;
	size = ScaleSymbol(elemPtr, penPtr->symbol.size);
	stylePtr->symbolSize = size;
	stylePtr->errorBarCapWidth = (penPtr->errorBarCapWidth > 0) 
	    ? penPtr->errorBarCapWidth : (int)(size * 0.6666666);
	stylePtr->errorBarCapWidth /= 2;
    }
    dataToStyle = Blt_StyleMap(elemPtr);
    if (((elemPtr->yHigh.nValues > 0) && (elemPtr->yLow.nValues > 0)) ||
	((elemPtr->xHigh.nValues > 0) && (elemPtr->xLow.nValues > 0)) ||
	(elemPtr->xError.nValues > 0) || (elemPtr->yError.nValues > 0)) {
	Blt_MapErrorBars(graphPtr, elemPtr, dataToStyle);
    }
    MergePens(elemPtr, dataToStyle);
    Blt_Free(dataToStyle);
}

static double
DistanceToLine(x, y, p, q, t)
    int x, y;			/* Sample X-Y coordinate. */
    Point2D *p, *q;		/* End points of the line segment. */
    Point2D *t;			/* (out) Point on line segment. */
{
    double right, left, top, bottom;

    *t = Blt_GetProjection(x, y, p, q);
    if (p->x > q->x) {
	right = p->x, left = q->x;
    } else {
	left = p->x, right = q->x;
    }
    if (p->y > q->y) {
	bottom = p->y, top = q->y;
    } else {
	top = p->y, bottom = q->y;
    }
    if (t->x > right) {
	t->x = right;
    } else if (t->x < left) {
	t->x = left;
    }
    if (t->y > bottom) {
	t->y = bottom;
    } else if (t->y < top) {
	t->y = top;
    }
    return hypot((t->x - x), (t->y - y));
}

static double
DistanceToX(
    int x, int y, 		/* Search X-Y coordinate. */
    Point2D *p, 
    Point2D *q,			/* End points of the line segment. */
    Point2D *t)			/* (out) Point on line segment. */
{
    double dx, dy;
    double dist;

    if (p->x > q->x) {
	if ((x > p->x) || (x < q->x)) {
	    return DBL_MAX; /* X-coordinate outside line segment. */
	}
    } else {
	if ((x > q->x) || (x < p->x)) {
	    return DBL_MAX; /* X-coordinate outside line segment. */
	}
    }
    dx = p->x - q->x;
    dy = p->y - q->y;
    t->x = (double)x;
    if (FABS(dx) < DBL_EPSILON) {
	double d1, d2;
	/* 
	 * Same X-coordinate indicates a vertical line.  Pick the
	 * closest end point. 
	 */
	d1 = p->y - y;
	d2 = q->y - y;
	if (FABS(d1) < FABS(d2)) {
	    t->y = p->y, dist = d1;
	} else {
	    t->y = q->y, dist = d2;
	}
    } else if (FABS(dy) < DBL_EPSILON) {
	/* Horizontal line. */
	t->y = p->y, dist = p->y - y;
    } else {
	double m, b;
		
	m = dy / dx;
	b = p->y - (m * p->x);
	t->y = (x * m) + b;
	dist = y - t->y;
    }
   return FABS(dist);
}

static double
DistanceToY(x, y, p, q, t)
    int x, y;			/* Search X-Y coordinate. */
    Point2D *p, *q;		/* End points of the line segment. */
    Point2D *t;			/* (out) Point on line segment. */
{
    double dx, dy;
    double dist;

    if (p->y > q->y) {
	if ((y > p->y) || (y < q->y)) {
	    return DBL_MAX;
	}
    } else {
	if ((y > q->y) || (y < p->y)) {
	    return DBL_MAX;
	}
    }
    dx = p->x - q->x;
    dy = p->y - q->y;
    t->y = y;
    if (FABS(dy) < DBL_EPSILON) {
	double d1, d2;

	/* Save Y-coordinate indicates an horizontal line. Pick the
	 * closest end point. */
	d1 = p->x - x;
	d2 = q->x - x;
	if (FABS(d1) < FABS(d2)) {
	    t->x = p->x, dist = d1;
	} else {
	    t->x = q->x, dist = d2;
	}
    } else if (FABS(dx) < DBL_EPSILON) {
	/* Vertical line. */
	t->x = p->x, dist = p->x - x;
    } else {
	double m, b;
	
	m = dy / dx;
	b = p->y - (m * p->x);
	t->x = (y - b) / m;
	dist = x - t->x;
    }
    return FABS(dist);
}

/*
 *----------------------------------------------------------------------
 *
 * ClosestTrace --
 *
 *	Find the line segment closest to the given window coordinate
 *	in the element.
 *
 * Results:
 *	If a new minimum distance is found, the information regarding
 *	it is returned via searchPtr.
 *
 *----------------------------------------------------------------------
 */
static int
ClosestTrace(
    Graph *graphPtr,		/* Graph widget record */
    Element *elemPtr,
    ClosestSearch *searchPtr,	/* Info about closest point in element */
    DistanceProc *distProc)
{
    Line *linePtr = (Line *)elemPtr; /* Line element record */
    Blt_ChainLink *linkPtr;
    Point2D closest;
    double minDist;
    int iClose;

    iClose = -1;		/* Suppress compiler warning. */
    minDist = searchPtr->dist;
    for (linkPtr = Blt_ChainFirstLink(linePtr->traces); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	Trace *tracePtr;
	Point2D *p, *pend;

	tracePtr = Blt_ChainGetValue(linkPtr);
	for (p = tracePtr->screenPts, pend = p + (tracePtr->nScreenPts - 1);
	     p < pend; p++) {
	    Point2D b;
	    double dist;

	    dist = (*distProc)(searchPtr->x, searchPtr->y, p, p + 1, &b);
	    if (dist < minDist) {
		closest = b;
		iClose = tracePtr->symbolToData[p - tracePtr->screenPts];
		minDist = dist;
	    }
	}
    }
    if (minDist < searchPtr->dist) {
	searchPtr->dist = minDist;
	searchPtr->elemPtr = elemPtr;
	searchPtr->index = iClose;
	searchPtr->point = Blt_InvMap2D(graphPtr, closest.x, closest.y,
	    &(elemPtr->axes));
	return TRUE;
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * ClosestStrip --
 *
 *	Find the line segment closest to the given window coordinate
 *	in the element.
 *
 * Results:
 *	If a new minimum distance is found, the information regarding
 *	it is returned via searchPtr.
 *
 *----------------------------------------------------------------------
 */
static int
ClosestStrip(
    Graph *graphPtr,		/* Graph widget record */
    Element *elemPtr,		/* Line element record */
    ClosestSearch *searchPtr,	/* Info about closest point in element */
    DistanceProc *distProc)
{
    Line *linePtr = (Line *)elemPtr; /* Line element record */
    Point2D closest;
    double minDist;
    int count;
    int iClose;
    Segment2D *sp;

    iClose = 0;
    minDist = searchPtr->dist;
    for (sp = linePtr->strips, count = 0; count < linePtr->nStrips; 
	count++, sp++) {
	double dist;
	Point2D b;

	dist = (*distProc)(searchPtr->x, searchPtr->y, &(sp->p), &(sp->q), &b);
	if (dist < minDist) {
	    closest = b;
	    iClose = linePtr->stripToData[count];
	    minDist = dist;
	}
    }
    if (minDist < searchPtr->dist) {
	searchPtr->dist = minDist;
	searchPtr->elemPtr = elemPtr;
	searchPtr->index = iClose;
	searchPtr->point = Blt_InvMap2D(graphPtr, closest.x, closest.y,
	    &elemPtr->axes);
	return TRUE;
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * ClosestPoint --
 *
 *	Find the element whose data point is closest to the given screen
 *	coordinate.
 *
 * Results:
 *	If a new minimum distance is found, the information regarding
 *	it is returned via searchPtr.
 *
 *----------------------------------------------------------------------
 */
static void
ClosestPoint(
    Element *elemPtr,		/* Line element that we are looking at */
    ClosestSearch *searchPtr)	/* Assorted information related to searching
				 * for the closest point */
{
    Line *linePtr = (Line *)elemPtr; /* Line element that we are looking at */
    double minDist;
    int count, iClose;
    Point2D *pp;

    minDist = searchPtr->dist;
    iClose = 0;

    /*
     * Instead of testing each data point in graph coordinates, look at
     * the array of mapped screen coordinates. The advantages are
     *   1) only examine points that are visible (unclipped), and
     *   2) the computed distance is already in screen coordinates.
     */

    for (pp = linePtr->symbolPts, count = 0; count < linePtr->nSymbolPts; 
	 count++, pp++) {
	double dx, dy;
	double dist;

	dx = (double)(searchPtr->x - pp->x);
	dy = (double)(searchPtr->y - pp->y);
	if (searchPtr->along == SEARCH_BOTH) {
	    dist = hypot(dx, dy);
	} else if (searchPtr->along == SEARCH_X) {
	    dist = dx;
	} else if (searchPtr->along == SEARCH_Y) {
	    dist = dy;
	} else {
	    /* This can't happen */
	    continue;
	}
	if (dist < minDist) {
	    iClose = linePtr->symbolToData[count];
	    minDist = dist;
	}
    }
    if (minDist < searchPtr->dist) {
	searchPtr->elemPtr = elemPtr;
	searchPtr->dist = minDist;
	searchPtr->index = iClose;
	searchPtr->point.x = elemPtr->x.valueArr[iClose];
	searchPtr->point.y = elemPtr->y.valueArr[iClose];
    }
}

/*
 *----------------------------------------------------------------------
 *
 * GetLineExtents --
 *
 *	Retrieves the range of the line element
 *
 * Results:
 *	Returns the number of data points in the element.
 *
 *----------------------------------------------------------------------
 */
static void
GetLineExtents(elemPtr, extsPtr)
    Element *elemPtr;
    Region2D *extsPtr;
{
    int nPoints;

    extsPtr->top = extsPtr->left = DBL_MAX;
    extsPtr->bottom = extsPtr->right = -DBL_MAX;

    nPoints = NUMBEROFPOINTS(elemPtr);
    if (nPoints < 1) {
	return;
    } 
    extsPtr->right = elemPtr->x.max;
    if ((elemPtr->x.min <= 0.0) && (elemPtr->axes.x->logScale)) {
	extsPtr->left = Blt_FindElemVectorMinimum(&elemPtr->x, DBL_MIN);
    } else {
	extsPtr->left = elemPtr->x.min;
    }
    extsPtr->bottom = elemPtr->y.max;
    if ((elemPtr->y.min <= 0.0) && (elemPtr->axes.y->logScale)) {
	extsPtr->top = Blt_FindElemVectorMinimum(&elemPtr->y, DBL_MIN);
    } else {
	extsPtr->top = elemPtr->y.min;
    }

    /* Correct the data limits for error bars */

    if (elemPtr->xError.nValues > 0) {
	int i;
	
	nPoints = MIN(elemPtr->xError.nValues, nPoints);
	for (i = 0; i < nPoints; i++) {
	    double x;

	    x = elemPtr->x.valueArr[i] + elemPtr->xError.valueArr[i];
	    if (x > extsPtr->right) {
		extsPtr->right = x;
	    }
	    x = elemPtr->x.valueArr[i] - elemPtr->xError.valueArr[i];
	    if (elemPtr->axes.x->logScale) {
		if (x < 0.0) {
		    x = -x;	/* Mirror negative values, instead
				 * of ignoring them. */
		}
		if ((x > DBL_MIN) && (x < extsPtr->left)) {
		    extsPtr->left = x;
		}
	    } else if (x < extsPtr->left) {
		extsPtr->left = x;
	    }
	}		     
    } else {
	if ((elemPtr->xHigh.nValues > 0) && 
	    (elemPtr->xHigh.max > extsPtr->right)) {
	    extsPtr->right = elemPtr->xHigh.max;
	}
	if (elemPtr->xLow.nValues > 0) {
	    double left;
	    
	    if ((elemPtr->xLow.min <= 0.0) && 
		(elemPtr->axes.x->logScale)) {
		left = Blt_FindElemVectorMinimum(&elemPtr->xLow, DBL_MIN);
	    } else {
		left = elemPtr->xLow.min;
	    }
	    if (left < extsPtr->left) {
		extsPtr->left = left;
	    }
	}
    }
    
    if (elemPtr->yError.nValues > 0) {
	int i;
	
	nPoints = MIN(elemPtr->yError.nValues, nPoints);
	for (i = 0; i < nPoints; i++) {
	    double y;

	    y = elemPtr->y.valueArr[i] + elemPtr->yError.valueArr[i];
	    if (y > extsPtr->bottom) {
		extsPtr->bottom = y;
	    }
	    y = elemPtr->y.valueArr[i] - elemPtr->yError.valueArr[i];
	    if (elemPtr->axes.y->logScale) {
		if (y < 0.0) {
		    y = -y;	/* Mirror negative values, instead
				 * of ignoring them. */
		}
		if ((y > DBL_MIN) && (y < extsPtr->left)) {
		    extsPtr->top = y;
		}
	    } else if (y < extsPtr->top) {
		extsPtr->top = y;
	    }
	}		     
    } else {
	if ((elemPtr->yHigh.nValues > 0) && 
	    (elemPtr->yHigh.max > extsPtr->bottom)) {
	    extsPtr->bottom = elemPtr->yHigh.max;
	}
	if (elemPtr->yLow.nValues > 0) {
	    double top;
	    
	    if ((elemPtr->yLow.min <= 0.0) && 
		(elemPtr->axes.y->logScale)) {
		top = Blt_FindElemVectorMinimum(&elemPtr->yLow, DBL_MIN);
	    } else {
		top = elemPtr->yLow.min;
	    }
	    if (top < extsPtr->top) {
		extsPtr->top = top;
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TileChangedProc
 *
 *	Rebuilds the designated GC with the new tile pixmap.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
TileChangedProc(clientData, tile)
    ClientData clientData;
    Blt_Tile tile;		/* Not used. */
{
    Element *elemPtr = clientData;
    Graph *graphPtr;

    graphPtr = elemPtr->object.graphPtr;
    if (graphPtr->tkwin != NULL) {
	graphPtr->flags |= REDRAW_WORLD;
	Blt_EventuallyRedrawGraph(graphPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureLine --
 *
 *	Sets up the appropriate configuration parameters in the GC.
 *      It is assumed the parameters have been previously set by
 *	a call to Blt_ConfigureWidget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information such as line width, line style, color
 *	etc. get set in a new GC.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigureLine(Graph *graphPtr, Element *elemPtr)
{
    Line *linePtr = (Line *)elemPtr;
    unsigned long gcMask;
    XGCValues gcValues;
    GC newGC;
    Blt_ChainLink *linkPtr;
    PenStyle *stylePtr;

    if (ConfigurePen(graphPtr, (Pen *)&linePtr->builtinPen) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * Point to the static normal/active pens if no external pens have
     * been selected.
     */
    linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette);
    if (linkPtr == NULL) {
	linkPtr = Blt_ChainAllocLink(sizeof(LinePenStyle));
	Blt_ChainLinkBefore(elemPtr->stylePalette, linkPtr, NULL);
    } 
    stylePtr = Blt_ChainGetValue(linkPtr);
    stylePtr->penPtr = NORMALPEN(elemPtr);

    if (linePtr->fillTile != NULL) {
	Blt_SetTileChangedProc(linePtr->fillTile, TileChangedProc, linePtr);
    }
    /*
     * Set the outline GC for this pen: GCForeground is outline color.
     * GCBackground is the fill color (only used for bitmap symbols).
     */
    gcMask = 0;
    if (linePtr->fillFgColor != NULL) {
	gcMask |= GCForeground;
	gcValues.foreground = linePtr->fillFgColor->pixel;
    }
    if (linePtr->fillBgColor != NULL) {
	gcMask |= GCBackground;
	gcValues.background = linePtr->fillBgColor->pixel;
    }
    if ((linePtr->fillStipple != None) && 
	(linePtr->fillStipple != PATTERN_SOLID)) {
	gcMask |= (GCStipple | GCFillStyle);
	gcValues.stipple = linePtr->fillStipple;
	gcValues.fill_style = (linePtr->fillBgColor == NULL) 
	    ? FillStippled : FillOpaqueStippled;
    }
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (linePtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, linePtr->fillGC);
    }
    linePtr->fillGC = newGC;

    if (Blt_ConfigModified(elemPtr->configSpecs, "-scalesymbols", 
			   (char *)NULL)) {
	elemPtr->flags |= (MAP_ITEM | SCALE_SYMBOL);
    }
    if (Blt_ConfigModified(elemPtr->configSpecs, "-pixels", "-trace", 
	"-*data", "-smooth", "-map*", "-label", "-hide", "-x", "-y", 
	"-areapattern", (char *)NULL)) {
	elemPtr->flags |= MAP_ITEM;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ClosestLine --
 *
 *	Find the closest point or line segment (if interpolated) to
 *	the given window coordinate in the line element.
 *
 * Results:
 *	Returns the distance of the closest point among other
 *	information.
 *
 *----------------------------------------------------------------------
 */
static void
ClosestLine(graphPtr, elemPtr, searchPtr)
    Graph *graphPtr;		/* Graph widget record */
    Element *elemPtr;		/* Element to examine */
    ClosestSearch *searchPtr;	/* Info about closest point in element */
{
    int mode;

    mode = searchPtr->mode;
    if (mode == SEARCH_AUTO) {
	LinePen *penPtr;

	penPtr = (LinePen *)NORMALPEN(elemPtr);
	mode = SEARCH_POINTS;
	if ((NUMBEROFPOINTS(elemPtr) > 1) && (penPtr->traceWidth > 0)) {
	    mode = SEARCH_TRACES;
	}
    }
    if (mode == SEARCH_POINTS) {
	ClosestPoint(elemPtr, searchPtr);
    } else {
	DistanceProc *distProc;
	int found;

	if (searchPtr->along == SEARCH_X) {
	    distProc = DistanceToX;
	} else if (searchPtr->along == SEARCH_Y) {
	    distProc = DistanceToY;
	} else {
	    distProc = DistanceToLine;
	}
	if (elemPtr->object.classId == OBJECT_CLASS_STRIP_ELEMENT) {
	    found = ClosestStrip(graphPtr, elemPtr, searchPtr, distProc);
	} else {
	    found = ClosestTrace(graphPtr, elemPtr, searchPtr, distProc);
	}
	if ((!found) && (searchPtr->along != SEARCH_BOTH)) {
	    ClosestPoint(elemPtr, searchPtr);
	}
    }
}

/*
 * XDrawLines() points: XMaxRequestSize(dpy) - 3
 * XFillPolygon() points:  XMaxRequestSize(dpy) - 4
 * XDrawSegments() segments:  (XMaxRequestSize(dpy) - 3) / 2
 * XDrawRectangles() rectangles:  (XMaxRequestSize(dpy) - 3) / 2
 * XFillRectangles() rectangles:  (XMaxRequestSize(dpy) - 3) / 2
 * XDrawArcs() or XFillArcs() arcs:  (XMaxRequestSize(dpy) - 3) / 3
 */

#define MAX_DRAWLINES(d)	Blt_MaxRequestSize(d, sizeof(XPoint))
#define MAX_DRAWPOLYGON(d)	Blt_MaxRequestSize(d, sizeof(XPoint))
#define MAX_DRAWSEGMENTS(d)	Blt_MaxRequestSize(d, sizeof(XSegment))
#define MAX_DRAWRECTANGLES(d)	Blt_MaxRequestSize(d, sizeof(XRectangle))
#define MAX_DRAWARCS(d)		Blt_MaxRequestSize(d, sizeof(XArc))

#ifdef WIN32

static void
DrawCircles(
    Display *display,
    Drawable drawable,
    Line *linePtr,
    LinePen *penPtr,
    int nSymbolPts,
    Point2D *symbolPts,
    int radius)
{
    HBRUSH brush, oldBrush;
    HPEN pen, oldPen;
    HDC dc;
    TkWinDCState state;

    if (drawable == None) {
	return;			/* Huh? */
    }
    if ((penPtr->symbol.fillGC == NULL) && 
	(penPtr->symbol.outlineWidth == 0)) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    /* SetROP2(dc, tkpWinRopModes[penPtr->symbol.fillGC->function]); */
    if (penPtr->symbol.fillGC != NULL) {
	brush = CreateSolidBrush(penPtr->symbol.fillGC->foreground);
    } else {
	brush = GetStockBrush(NULL_BRUSH);
    }
    if (penPtr->symbol.outlineWidth > 0) {
	pen = Blt_GCToPen(dc, penPtr->symbol.outlineGC);
    } else {
	pen = GetStockPen(NULL_PEN);
    }
    oldPen = SelectPen(dc, pen);
    oldBrush = SelectBrush(dc, brush);
    {
	Point2D *pp, *pend;

	for (pp = symbolPts, pend = pp + nSymbolPts; pp < pend; pp++) {
	    Ellipse(dc, (int)pp->x - radius, (int)pp->y - radius, 
		(int)pp->x + radius + 1, (int)pp->y + radius + 1);
	}
    }
    DeleteBrush(SelectBrush(dc, oldBrush));
    DeletePen(SelectPen(dc, oldPen));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

#else

static void
DrawCircles(display, drawable, linePtr, penPtr, nSymbolPts, symbolPts, radius)
    Display *display;
    Drawable drawable;
    Line *linePtr;
    LinePen *penPtr;
    int nSymbolPts;
    Point2D *symbolPts;
    int radius;
{
    int i;
    XArc *arcs;			/* Array of arcs (circle) */
    int reqSize;
    int s;
    int count;

    s = radius + radius;
    arcs = Blt_Malloc(nSymbolPts * sizeof(XArc));

    if (linePtr->symbolInterval > 0) {
	Point2D *pp, *pend;
	XArc *ap;

        ap = arcs;
	count = 0;
	for (pp = symbolPts, pend = pp + nSymbolPts; pp < pend; pp++) {
	    if (DRAW_SYMBOL(linePtr)) {
		ap->x = (short int)pp->x - radius;
		ap->y = (short int)pp->y - radius;
		ap->width = ap->height = (unsigned short)s;
		ap->angle1 = 0;
		ap->angle2 = 23040;
		ap++, count++;
	    }
	    linePtr->symbolCounter++;
	}
    } else {
	Point2D *pp, *pend;
	XArc *ap;

        ap = arcs;
	for (pp = symbolPts, pend = pp + nSymbolPts; pp < pend; pp++) {
	    ap->x = (short int)pp->x - radius;
	    ap->y = (short int)pp->y - radius;
	    ap->width = ap->height = (unsigned short)s;
	    ap->angle1 = 0;
	    ap->angle2 = 23040;
	    ap++;
	}
	count = nSymbolPts;
    }
    reqSize = MAX_DRAWARCS(display);
    for (i = 0; i < count; i += reqSize) {
	int nArcs;

	nArcs = ((i + reqSize) > count) ? (count - i) : reqSize;
	if (penPtr->symbol.fillGC != NULL) {
	    XFillArcs(display, drawable, penPtr->symbol.fillGC, arcs + i, 
		      nArcs);
	}
	if (penPtr->symbol.outlineWidth > 0) {
	    XDrawArcs(display, drawable, penPtr->symbol.outlineGC, arcs + i, 
		      nArcs);
	}
    }
    Blt_Free(arcs);
}

#endif

static void
DrawSquares(
    Display *display,
    Drawable drawable,
    Line *linePtr,
    LinePen *penPtr,
    int nSymbolPts,
    Point2D *symbolPts,
    int r)
{
    XRectangle *rectangles;
    int reqSize;
    int s, i, count;

    s = r + r;
    rectangles = Blt_Malloc(nSymbolPts * sizeof(XRectangle));
    if (linePtr->symbolInterval > 0) {
	Point2D *pp, *pend;
	XRectangle *rp;

	count = 0;
	rp = rectangles;
	for (pp = symbolPts, pend = pp + nSymbolPts; pp < pend; pp++) {
	    if (DRAW_SYMBOL(linePtr)) {
		rp->x = (short int)(pp->x - r);
		rp->y = (short int)(pp->y - r);
		rp->width = rp->height = (unsigned short)s;
		rp++, count++;
	    }
	    linePtr->symbolCounter++;
	}
    } else {
	Point2D *pp, *pend;
	XRectangle *rp;

	rp = rectangles;
	for (pp = symbolPts, pend = pp + nSymbolPts; pp < pend; pp++) {
	    rp->x = (short int)(pp->x - r);
	    rp->y = (short int)(pp->y - r);
	    rp->width = rp->height = (unsigned short)s;
	    rp++;
	}
	count = nSymbolPts;
    }
    reqSize = MAX_DRAWRECTANGLES(display);
    for (i = 0; i < count; i += reqSize) {
	int nRects;

	nRects = ((i + reqSize) > count) ? (count - i) : reqSize;
	if (penPtr->symbol.fillGC != NULL) {
	    XFillRectangles(display, drawable, penPtr->symbol.fillGC, 
			    rectangles + i, nRects);
	}
	if (penPtr->symbol.outlineWidth > 0) {
	    XDrawRectangles(display, drawable, penPtr->symbol.outlineGC, 
			    rectangles + i, nRects);
	}
    }
    Blt_Free(rectangles);
}

/*
 * -----------------------------------------------------------------
 *
 * DrawSymbols --
 *
 * 	Draw the symbols centered at the each given x,y coordinate
 *	in the array of points.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Draws a symbol at each coordinate given.  If active,
 *	only those coordinates which are currently active are
 *	drawn.
 *
 * -----------------------------------------------------------------
 */
static void
DrawSymbols(graphPtr, drawable, linePtr, penPtr, size, nSymbolPts, symbolPts)
    Graph *graphPtr;		/* Graph widget record */
    Drawable drawable;		/* Pixmap or window to draw into */
    Line *linePtr;
    LinePen *penPtr;
    int size;			/* Size of element */
    int nSymbolPts;		/* Number of coordinates in array */
    Point2D *symbolPts;		/* Array of x,y coordinates for line */
{
    XPoint pattern[13];		/* Template for polygon symbols */
    int r1, r2;
    int count;
#define SQRT_PI		1.77245385090552
#define S_RATIO		0.886226925452758

    if (size < 3) {
	if (penPtr->symbol.fillGC != NULL) {
	    Point2D *pp, *endp;
	    XPoint *points, *xpp;
	    
	    xpp = points = Blt_Malloc(nSymbolPts * sizeof(XPoint));
	    for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		xpp->x = (short int)pp->x;
		xpp->y = (short int)pp->y;
		xpp++;
	    }
	    XDrawPoints(graphPtr->display, drawable, penPtr->symbol.fillGC, 
			points, nSymbolPts, CoordModeOrigin);
	    Blt_Free(points);
	}
	return;
    }
    r1 = (int)ceil(size * 0.5);
    r2 = (int)ceil(size * S_RATIO * 0.5);

    switch (penPtr->symbol.type) {
    case SYMBOL_NONE:
	break;

    case SYMBOL_SQUARE:
	DrawSquares(graphPtr->display, drawable, linePtr, penPtr, nSymbolPts,
	    symbolPts, r2);
	break;

    case SYMBOL_CIRCLE:
	DrawCircles(graphPtr->display, drawable, linePtr, penPtr, nSymbolPts,
	    symbolPts, r1);
	break;

    case SYMBOL_SPLUS:
    case SYMBOL_SCROSS:
	{
	    XSegment *segments;	/* Array of line segments (splus, scross) */
	    int i;
	    int reqSize, nSegs;

	    if (penPtr->symbol.type == SYMBOL_SCROSS) {
		r2 = Round(r2 * M_SQRT1_2);
		pattern[3].y = pattern[2].x = pattern[0].x = pattern[0].y = -r2;
		pattern[3].x = pattern[2].y = pattern[1].y = pattern[1].x = r2;
	    } else {
		pattern[0].y = pattern[1].y = pattern[2].x = pattern[3].x = 0;
		pattern[0].x = pattern[2].y = -r2;
		pattern[1].x = pattern[3].y = r2;
	    }
	    segments = Blt_Malloc(nSymbolPts * 2 * sizeof(XSegment));
	    if (linePtr->symbolInterval > 0) {
		Point2D *pp, *endp;
		XSegment *sp;

		sp = segments;
		count = 0;
		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    if (DRAW_SYMBOL(linePtr)) {
			sp->x1 = pattern[0].x + (short int)pp->x;
			sp->y1 = pattern[0].y + (short int)pp->y;
			sp->x2 = pattern[1].x + (short int)pp->x;
			sp->y2 = pattern[1].y + (short int)pp->y;
			sp++;
			sp->x1 = pattern[2].x + (short int)pp->x;
			sp->y1 = pattern[2].y + (short int)pp->y;
			sp->x2 = pattern[3].x + (short int)pp->x;
			sp->y2 = pattern[3].y + (short int)pp->y;
			sp++;
			count++;
		    }
		    linePtr->symbolCounter++;
		}
	    } else {
		Point2D *pp, *endp;
		XSegment *sp;

		sp = segments;
		count = nSymbolPts;
		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    sp->x1 = pattern[0].x + (short int)pp->x;
		    sp->y1 = pattern[0].y + (short int)pp->y;
		    sp->x2 = pattern[1].x + (short int)pp->x;
		    sp->y2 = pattern[1].y + (short int)pp->y;
		    sp++;
		    sp->x1 = pattern[2].x + (short int)pp->x;
		    sp->y1 = pattern[2].y + (short int)pp->y;
		    sp->x2 = pattern[3].x + (short int)pp->x;
		    sp->y2 = pattern[3].y + (short int)pp->y;
		    sp++;
		}
	    }
	    nSegs = count * 2;
	    /* Always draw skinny symbols regardless of the outline width */
	    reqSize = MAX_DRAWSEGMENTS(graphPtr->display);
	    for (i = 0; i < nSegs; i += reqSize) {
		int chunk;

		chunk = ((i + reqSize) > nSegs) ? (nSegs - i) : reqSize;
		XDrawSegments(graphPtr->display, drawable, 
			penPtr->symbol.outlineGC, segments + i, chunk);
	    }
	    Blt_Free(segments);
	}
	break;

    case SYMBOL_PLUS:
    case SYMBOL_CROSS:
	{
	    XPoint *polygon;
	    int d;		/* Small delta for cross/plus thickness */

	    d = (r2 / 3);

	    /*
	     *
	     *          2   3       The plus/cross symbol is a closed polygon
	     *                      of 12 points. The diagram to the left
	     *    0,12  1   4    5  represents the positions of the points
	     *           x,y        which are computed below. The extra
	     *     11  10   7    6  (thirteenth) point connects the first and
	     *                      last points.
	     *          9   8
	     */

	    pattern[0].x = pattern[11].x = pattern[12].x = -r2;
	    pattern[2].x = pattern[1].x = pattern[10].x = pattern[9].x = -d;
	    pattern[3].x = pattern[4].x = pattern[7].x = pattern[8].x = d;
	    pattern[5].x = pattern[6].x = r2;
	    pattern[2].y = pattern[3].y = -r2;
	    pattern[0].y = pattern[1].y = pattern[4].y = pattern[5].y =
		pattern[12].y = -d;
	    pattern[11].y = pattern[10].y = pattern[7].y = pattern[6].y = d;
	    pattern[9].y = pattern[8].y = r2;

	    if (penPtr->symbol.type == SYMBOL_CROSS) {
		int i;

		/* For the cross symbol, rotate the points by 45 degrees. */
		for (i = 0; i < 12; i++) {
		    double dx, dy;

		    dx = (double)pattern[i].x * M_SQRT1_2;
		    dy = (double)pattern[i].y * M_SQRT1_2;
		    pattern[i].x = Round(dx - dy);
		    pattern[i].y = Round(dx + dy);
		}
		pattern[12] = pattern[0];
	    }
	    polygon = Blt_Malloc(nSymbolPts * 13 * sizeof(XPoint));
	    if (linePtr->symbolInterval > 0) {
		Point2D *pp, *endp;
		XPoint *xpp;

		count = 0;
		xpp = polygon;
		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    if (DRAW_SYMBOL(linePtr)) {
			int i;

			for (i = 0; i < 13; i++) {
			    xpp->x = pattern[i].x + (short int)pp->x;
			    xpp->y = pattern[i].y + (short int)pp->y;
			    xpp++;
			}
			count++;
		    }
		    linePtr->symbolCounter++;
		}
	    } else {
		Point2D *pp, *endp;
		XPoint *xpp;

		xpp = polygon;
		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    int i;

		    for (i = 0; i < 13; i++) {
			xpp->x = pattern[i].x + (short int)pp->x;
			xpp->y = pattern[i].y + (short int)pp->y;
			xpp++;
		    }
		}
		count = nSymbolPts;
	    }
	    if (penPtr->symbol.fillGC != NULL) {
		int i;
		XPoint *xpp;

		for (xpp = polygon, i = 0; i < count; i++, xpp += 13) {
		    XFillPolygon(graphPtr->display, drawable, 
			penPtr->symbol.fillGC, xpp, 13, Complex, 
			CoordModeOrigin);
		}
	    }
	    if (penPtr->symbol.outlineWidth > 0) {
		int i;
		XPoint *xpp;

		for (xpp = polygon, i = 0; i < count; i++, xpp += 13) {
		    XDrawLines(graphPtr->display, drawable, 
			penPtr->symbol.outlineGC, xpp, 13, CoordModeOrigin);
		}
	    }
	    Blt_Free(polygon);
	}
	break;

    case SYMBOL_DIAMOND:
	{
	    XPoint *polygon;

	    /*
	     *
	     *                      The plus symbol is a closed polygon
	     *            1         of 4 points. The diagram to the left
	     *                      represents the positions of the points
	     *       0,4 x,y  2     which are computed below. The extra
	     *                      (fifth) point connects the first and
	     *            3         last points.
	     *
	     */
	    pattern[1].y = pattern[0].x = -r1;
	    pattern[2].y = pattern[3].x = pattern[0].y = pattern[1].x = 0;
	    pattern[3].y = pattern[2].x = r1;
	    pattern[4] = pattern[0];

	    polygon = Blt_Malloc(nSymbolPts * 5 * sizeof(XPoint));
	    if (linePtr->symbolInterval > 0) {
		Point2D *pp, *endp;
		XPoint *xpp;

		xpp = polygon;
		count = 0;
		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    int i;

		    if (DRAW_SYMBOL(linePtr)) {
			for (i = 0; i < 5; i++) {
			    xpp->x = pattern[i].x + (short int)pp->x;
			    xpp->y = pattern[i].y + (short int)pp->y;
			    xpp++;
			}
			count++;
		    }
		    linePtr->symbolCounter++;
		}
	    } else {
		Point2D *pp, *endp;
		XPoint *xpp;

		xpp = polygon;
		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    int i;

		    for (i = 0; i < 5; i++) {
			xpp->x = pattern[i].x + (short int)pp->x;
			xpp->y = pattern[i].y + (short int)pp->y;
			xpp++;
		    }
		}
		count = nSymbolPts;
	    }
	    if (penPtr->symbol.fillGC != NULL) {
		XPoint *xpp;
		int i;

		for (xpp = polygon, i = 0; i < count; i++, xpp += 5) {
		    XFillPolygon(graphPtr->display, drawable, 
			penPtr->symbol.fillGC, xpp, 5, Convex, CoordModeOrigin);

		}
	    }
	    if (penPtr->symbol.outlineWidth > 0) {
		XPoint *xpp;
		int i;

		for (xpp = polygon, i = 0; i < count; i++, xpp += 5) {
		    XDrawLines(graphPtr->display, drawable, 
		       penPtr->symbol.outlineGC, xpp, 5, CoordModeOrigin);
		}
	    }
	    Blt_Free(polygon);
	}
	break;

    case SYMBOL_TRIANGLE:
    case SYMBOL_ARROW:
	{
	    XPoint *polygon;
	    double b;
	    int b2, h1, h2;
#define H_RATIO		1.1663402261671607
#define B_RATIO		1.3467736870885982
#define TAN30		0.57735026918962573
#define COS30		0.86602540378443871

	    b = Round(size * B_RATIO * 0.7);
	    b2 = Round(b * 0.5);
	    h2 = Round(TAN30 * b2);
	    h1 = Round(b2 / COS30);
	    /*
	     *
	     *                      The triangle symbol is a closed polygon
	     *           0,3         of 3 points. The diagram to the left
	     *                      represents the positions of the points
	     *           x,y        which are computed below. The extra
	     *                      (fourth) point connects the first and
	     *      2           1   last points.
	     *
	     */

	    if (penPtr->symbol.type == SYMBOL_ARROW) {
		pattern[3].x = pattern[0].x = 0;
		pattern[3].y = pattern[0].y = h1;
		pattern[1].x = b2;
		pattern[2].y = pattern[1].y = -h2;
		pattern[2].x = -b2;
	    } else {
		pattern[3].x = pattern[0].x = 0;
		pattern[3].y = pattern[0].y = -h1;
		pattern[1].x = b2;
		pattern[2].y = pattern[1].y = h2;
		pattern[2].x = -b2;
	    }
	    polygon = Blt_Malloc(nSymbolPts * 4 * sizeof(XPoint));
	    if (linePtr->symbolInterval > 0) {
		Point2D *pp, *endp;
		XPoint *xpp;

		xpp = polygon;
		count = 0;
		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    int i;

		    if (DRAW_SYMBOL(linePtr)) {
			for (i = 0; i < 4; i++) {
			    xpp->x = pattern[i].x + (short int)pp->x;
			    xpp->y = pattern[i].y + (short int)pp->y;
			    xpp++;
			}
			count++;
		    }
		    linePtr->symbolCounter++;
		}
	    } else {
		Point2D *pp, *endp;
		XPoint *xpp;

		xpp = polygon;
		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    int i;

		    for (i = 0; i < 4; i++) {
			xpp->x = pattern[i].x + (short int)pp->x;
			xpp->y = pattern[i].y + (short int)pp->y;
			xpp++;
		    }
		}
		count = nSymbolPts;
	    }
	    if (penPtr->symbol.fillGC != NULL) {
		XPoint *xpp;
		int i;

		xpp = polygon;
		for (xpp = polygon, i = 0; i < count; i++, xpp += 4) {
		    XFillPolygon(graphPtr->display, drawable, 
			penPtr->symbol.fillGC, xpp, 4, Convex, CoordModeOrigin);
		}
	    }
	    if (penPtr->symbol.outlineWidth > 0) {
		XPoint *xpp;
		int i;

		xpp = polygon;
		for (xpp = polygon, i = 0; i < count; i++, xpp += 4) {
		    XDrawLines(graphPtr->display, drawable, 
			penPtr->symbol.outlineGC, xpp, 4, CoordModeOrigin);
		}
	    }
	    Blt_Free(polygon);
	}
	break;
    case SYMBOL_BITMAP:
	{
	    Pixmap bitmap, mask;
	    int width, height, bmWidth, bmHeight;
	    double scale, sx, sy;
	    int dx, dy;

	    Tk_SizeOfBitmap(graphPtr->display, penPtr->symbol.bitmap,
		&width, &height);
	    mask = None;

	    /*
	     * Compute the size of the scaled bitmap.  Stretch the
	     * bitmap to fit a nxn bounding box.
	     */
	    sx = (double)size / (double)width;
	    sy = (double)size / (double)height;
	    scale = MIN(sx, sy);
	    bmWidth = (int)(width * scale);
	    bmHeight = (int)(height * scale);

	    XSetClipMask(graphPtr->display, penPtr->symbol.outlineGC, None);
	    if (penPtr->symbol.mask != None) {
		mask = Blt_ScaleBitmap(graphPtr->tkwin, penPtr->symbol.mask,
		    width, height, bmWidth, bmHeight);
		XSetClipMask(graphPtr->display, penPtr->symbol.outlineGC, 
			     mask);
	    }
	    bitmap = Blt_ScaleBitmap(graphPtr->tkwin, penPtr->symbol.bitmap,
		width, height, bmWidth, bmHeight);
	    if (penPtr->symbol.fillGC == NULL) {
		XSetClipMask(graphPtr->display, penPtr->symbol.outlineGC, 
			     bitmap);
	    }
	    dx = bmWidth / 2;
	    dy = bmHeight / 2;
	    if (linePtr->symbolInterval > 0) {
		Point2D *pp, *endp;

		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    if (DRAW_SYMBOL(linePtr)) {
			int x, y;
	    
			x = (int)pp->x - dx;
			y = (int)pp->y - dy;
			if ((penPtr->symbol.fillGC == NULL) || (mask !=None)) {
			    XSetClipOrigin(graphPtr->display,
				penPtr->symbol.outlineGC, x, y);
			}
			XCopyPlane(graphPtr->display, bitmap, drawable,
			    penPtr->symbol.outlineGC, 0, 0, bmWidth, bmHeight, 
				   x, y, 1);
		    }
		    linePtr->symbolCounter++;
		}
	    } else {
		Point2D *pp, *endp;

		for (pp = symbolPts, endp = pp + nSymbolPts; pp < endp; pp++) {
		    int x, y;

		    x = (int)pp->x - dx;
		    y = (int)pp->y - dy;
		    if ((penPtr->symbol.fillGC == NULL) || (mask != None)) {
			XSetClipOrigin(graphPtr->display, 
				penPtr->symbol.outlineGC, x, y);
		    }
		    XCopyPlane(graphPtr->display, bitmap, drawable,
			penPtr->symbol.outlineGC, 0, 0, bmWidth, bmHeight, 
			x, y, 1);
		}
	    }
	    Tk_FreePixmap(graphPtr->display, bitmap);
	    if (mask != None) {
		Tk_FreePixmap(graphPtr->display, mask);
	    }
	}
	break;
    }
}

/*
 * -----------------------------------------------------------------
 *
 * DrawSymbol --
 *
 * 	Draw the symbol centered at the each given x,y coordinate.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Draws a symbol at the coordinate given.
 *
 * -----------------------------------------------------------------
 */
static void
DrawSymbol(graphPtr, drawable, elemPtr, x, y, size)
    Graph *graphPtr;		/* Graph widget record */
    Drawable drawable;		/* Pixmap or window to draw into */
    Element *elemPtr;		/* Line element information */
    int x, y;			/* Center position of symbol */
    int size;			/* Size of symbol. */
{
    Line *linePtr = (Line *)elemPtr;
    LinePen *penPtr;

    penPtr = (LinePen *)NORMALPEN(elemPtr);
    if (penPtr->traceWidth > 0) {
	/*
	 * Draw an extra line offset by one pixel from the previous to
	 * give a thicker appearance.  This is only for the legend
	 * entry.  This routine is never called for drawing the actual
	 * line segments.
	 */
	XDrawLine(graphPtr->display, drawable, penPtr->traceGC, x - size, 
		y, x + size, y);
	XDrawLine(graphPtr->display, drawable, penPtr->traceGC, x - size,
		y + 1, x + size, y + 1);
    }
    if (penPtr->symbol.type != SYMBOL_NONE) {
	Point2D point;

	point.x = x, point.y = y;
	DrawSymbols(graphPtr, drawable, linePtr, penPtr, size, 1, &point);
    }
}

#ifdef WIN32

static void
DrawTraces(
    Graph *graphPtr,
    Drawable drawable,		/* Pixmap or window to draw into */
    Line *linePtr,
    LinePen *penPtr)
{
    Blt_ChainLink *linkPtr;
    HBRUSH brush, oldBrush;
    HDC dc;
    HPEN pen, oldPen;
    POINT *points;
    TkWinDCState state;
    int nPoints;

    /*  
     * Depending if the line is wide (> 1 pixel), arbitrarily break
     * the line in sections of 100 points.  This bit of weirdness has
     * to do with wide geometric pens.  The longer the polyline, the
     * slower it draws.  The trade off is that we lose dash and cap
     * uniformity for unbearably slow polyline draws.
     */
    if (penPtr->traceGC->line_width > 1) {
	nPoints = 100;
    } else {
	nPoints = Blt_MaxRequestSize(graphPtr->display, sizeof(POINT)) - 1;
    }
    points = Blt_Malloc((nPoints + 1) * sizeof(POINT));

    dc = TkWinGetDrawableDC(graphPtr->display, drawable, &state);

    pen = Blt_GCToPen(dc, penPtr->traceGC);
    oldPen = SelectPen(dc, pen);
    brush = CreateSolidBrush(penPtr->traceGC->foreground);
    oldBrush = SelectBrush(dc, brush);
    SetROP2(dc, tkpWinRopModes[penPtr->traceGC->function]);

    for (linkPtr = Blt_ChainFirstLink(linePtr->traces); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	POINT *p;
	Trace *tracePtr;
	int count, remaining;

	tracePtr = Blt_ChainGetValue(linkPtr);

	/*
	 * If the trace has to be split into separate XDrawLines
	 * calls, then the end point of the current trace is also the
	 * starting point of the new split.
	 */

	/* Step 1. Convert and draw the first section of the trace.
	 *	   It may contain the entire trace. */

	for (p = points, count = 0; count < MIN(nPoints, tracePtr->nScreenPts); 
	     count++, p++) {
	    p->x = (int)tracePtr->screenPts[count].x;
	    p->y = (int)tracePtr->screenPts[count].y;
	}
	Polyline(dc, points, count);

	/* Step 2. Next handle any full-size chunks left. */

	while ((count + nPoints) < tracePtr->nScreenPts) {
	    int j;

	    /* Start with the last point of the previous trace. */
	    points[0].x = points[nPoints - 1].x;
	    points[0].y = points[nPoints - 1].y;

	    for (p = points + 1, j = 0; j < nPoints; j++, count++, p++) {
		p->x = (int)tracePtr->screenPts[count].x;
		p->y = (int)tracePtr->screenPts[count].y;
	    }
	    Polyline(dc, points, nPoints + 1);
	}
	
	/* Step 3. Convert and draw the remaining points. */

	remaining = tracePtr->nScreenPts - count;
	if (remaining > 0) {
	    /* Start with the last point of the previous trace. */
	    points[0].x = points[nPoints - 1].x;
	    points[0].y = points[nPoints - 1].y;

	    for (p = points + 1; count < tracePtr->nScreenPts; count++, p++) {
		p->x = (int)tracePtr->screenPts[count].x;
		p->y = (int)tracePtr->screenPts[count].y;
	    }	    
	    Polyline(dc, points, remaining + 1);
	}
    }
    Blt_Free(points);
    DeletePen(SelectPen(dc, oldPen));
    DeleteBrush(SelectBrush(dc, oldBrush));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

#else

static void
DrawTraces(graphPtr, drawable, linePtr, penPtr)
    Graph *graphPtr;
    Drawable drawable;		/* Pixmap or window to draw into */
    Line *linePtr;
    LinePen *penPtr;
{
    Blt_ChainLink *linkPtr;
    XPoint *points;
    int nPoints;

    nPoints = Blt_MaxRequestSize(graphPtr->display, sizeof(XPoint)) - 1;
    points = Blt_Malloc((nPoints + 1) * sizeof(XPoint));
	    
    for (linkPtr = Blt_ChainFirstLink(linePtr->traces); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	XPoint *xpp;
	Trace *tracePtr;
	int remaining, count;
	int n;

	tracePtr = Blt_ChainGetValue(linkPtr);

	/*
	 * If the trace has to be split into separate XDrawLines
	 * calls, then the end point of the current trace is also the
	 * starting point of the new split.
	 */
	/* Step 1. Convert and draw the first section of the trace.
	 *	   It may contain the entire trace. */

	n = MIN(nPoints, tracePtr->nScreenPts); 
	for (xpp = points, count = 0; count < n; count++, xpp++) {
	    xpp->x = (short int)tracePtr->screenPts[count].x;
	    xpp->y = (short int)tracePtr->screenPts[count].y;
	}
	XDrawLines(graphPtr->display, drawable, penPtr->traceGC, points, 
	   count, CoordModeOrigin);

	/* Step 2. Next handle any full-size chunks left. */

	while ((count + nPoints) < tracePtr->nScreenPts) {
	    int j;

	    /* Start with the last point of the previous trace. */
	    points[0].x = points[nPoints - 1].x;
	    points[0].y = points[nPoints - 1].y;
	    
	    for (xpp = points + 1, j = 0; j < nPoints; j++, count++, xpp++) {
		xpp->x = (short int)tracePtr->screenPts[count].x;
		xpp->y = (short int)tracePtr->screenPts[count].y;
	    }
	    XDrawLines(graphPtr->display, drawable, penPtr->traceGC, points, 
		       nPoints + 1, CoordModeOrigin);
	}
	
	/* Step 3. Convert and draw the remaining points. */

	remaining = tracePtr->nScreenPts - count;
	if (remaining > 0) {
	    /* Start with the last point of the previous trace. */
	    points[0].x = points[nPoints - 1].x;
	    points[0].y = points[nPoints - 1].y;
	    for (xpp = points + 1; count < tracePtr->nScreenPts; count++, 
		     xpp++) {
		xpp->x = (short int)tracePtr->screenPts[count].x;
		xpp->y = (short int)tracePtr->screenPts[count].y;
	    }	    
	    XDrawLines(graphPtr->display, drawable, penPtr->traceGC, points, 
		remaining + 1, CoordModeOrigin);
	}
    }
    Blt_Free(points);
}
#endif /* WIN32 */

static void
DrawValues(
    Graph *graphPtr,
    Drawable drawable,
    Element *elemPtr,
    LinePen *penPtr,
    int nSymbolPts,
    Point2D *symbolPts,
    int *pointToData)
{
    Point2D *pp, *endp;
    char *fmt;
    char string[TCL_DOUBLE_SPACE * 2 + 2];
    int count;
    
    fmt = penPtr->valueFormat;
    if (fmt == NULL) {
	fmt = "%g";
    }
    count = 0;
    for (pp = symbolPts, endp = symbolPts + nSymbolPts; pp < endp; pp++) {
	double x, y;

	x = elemPtr->x.valueArr[pointToData[count]];
	y = elemPtr->y.valueArr[pointToData[count]];
	count++;
	if (penPtr->valueShow == SHOW_X) {
	    sprintf(string, fmt, x); 
	} else if (penPtr->valueShow == SHOW_Y) {
	    sprintf(string, fmt, y); 
	} else if (penPtr->valueShow == SHOW_BOTH) {
	    sprintf(string, fmt, x);
	    strcat(string, ",");
	    sprintf(string + strlen(string), fmt, y);
	}
	Blt_DrawText(graphPtr->tkwin, drawable, string, &(penPtr->valueStyle), 
		(int)pp->x, (int)pp->y);
    } 
}


/*
 *----------------------------------------------------------------------
 *
 * DrawActiveLine --
 *
 *	Draws the connected line(s) representing the element. If the
 *	line is made up of non-line symbols and the line width
 *	parameter has been set (linewidth > 0), the element will also
 *	be drawn as a line (with the linewidth requested).  The line
 *	may consist of separate line segments.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	X drawing commands are output.
 *
 *----------------------------------------------------------------------
 */
static void
DrawActiveLine(
    Graph *graphPtr,		/* Graph widget record */
    Drawable drawable,		/* Pixmap or window to draw into */
    Element *elemPtr)		/* Element to be drawn */
{
    Line *linePtr = (Line *)elemPtr;
    LinePen *penPtr = (LinePen *)elemPtr->activePenPtr;
    int symbolSize;

    if (penPtr == NULL) {
	return;
    }
    symbolSize = ScaleSymbol(elemPtr, penPtr->symbol.size);

    /* 
     * nActiveIndices 
     *	  > 0		Some points are active.  Uses activeArr.
     *	  < 0		All points are active.
     *    == 0		No points are active.
     */
    if (elemPtr->nActiveIndices > 0) {
	if (elemPtr->flags & ACTIVE_PENDING) {
	    MapActiveSymbols(graphPtr, elemPtr);
	}
	if (penPtr->symbol.type != SYMBOL_NONE) {
	    DrawSymbols(graphPtr, drawable, linePtr, penPtr, symbolSize,
		linePtr->nActivePts, linePtr->activePts);
	}
	if (penPtr->valueShow != SHOW_NONE) {
	    DrawValues(graphPtr, drawable, elemPtr, penPtr, linePtr->nActivePts,
		 linePtr->activePts, linePtr->activeToData);
	}
    } else if (elemPtr->nActiveIndices < 0) { 
	if (penPtr->traceWidth > 0) {
	    if (linePtr->nStrips > 0) {
		Blt_Draw2DSegments(graphPtr->display, drawable, 
			penPtr->traceGC, linePtr->strips, linePtr->nStrips);
	    } else if (Blt_ChainGetLength(linePtr->traces) > 0) {
		DrawTraces(graphPtr, drawable, linePtr, penPtr);
	    }
	}
	if (penPtr->symbol.type != SYMBOL_NONE) {
	    DrawSymbols(graphPtr, drawable, elemPtr, penPtr, symbolSize,
		linePtr->nSymbolPts, linePtr->symbolPts);
	}
	if (penPtr->valueShow != SHOW_NONE) {
	    DrawValues(graphPtr, drawable, elemPtr, penPtr, linePtr->nSymbolPts,
		       linePtr->symbolPts, linePtr->symbolToData);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DrawNormalLine --
 *
 *	Draws the connected line(s) representing the element. If the
 *	line is made up of non-line symbols and the line width parameter
 *	has been set (linewidth > 0), the element will also be drawn as
 *	a line (with the linewidth requested).  The line may consist of
 *	separate line segments.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	X drawing commands are output.
 *
 *----------------------------------------------------------------------
 */
static void
DrawNormalLine(graphPtr, drawable, elemPtr)
    Graph *graphPtr;		/* Graph widget record */
    Drawable drawable;		/* Pixmap or window to draw into */
    Element *elemPtr;		/* Element to be drawn */
{
    Line *linePtr = (Line *)elemPtr;
    Blt_ChainLink *linkPtr;
    unsigned int count;

    /* Fill area under the curve */
    if (linePtr->fillPts != NULL) {
	XPoint *points;
	Point2D *endp, *pp;

	points = Blt_Malloc(sizeof(XPoint) * linePtr->nFillPts);
	count = 0;
	for (pp = linePtr->fillPts, endp = pp + linePtr->nFillPts; 
	     pp < endp; pp++) {
	    points[count].x = (short int)pp->x;
	    points[count].y = (short int)pp->y;
	    count++;
	}
	if (linePtr->fillTile != NULL) {
	    Blt_SetTileOrigin(graphPtr->tkwin, linePtr->fillTile, 0, 0);
	    Blt_TilePolygon(graphPtr->tkwin, drawable, linePtr->fillTile, 
			    points, linePtr->nFillPts);
	} else if (linePtr->fillStipple != None) {
	    XFillPolygon(graphPtr->display, drawable, linePtr->fillGC, 
		 points, linePtr->nFillPts, Complex, CoordModeOrigin);
	}
	Blt_Free(points);
    }

    /* Lines: stripchart segments or graph traces. */
    if (linePtr->nStrips > 0) {
	for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); 
	     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	    LinePenStyle *stylePtr;
	    LinePen *penPtr;

	    stylePtr = Blt_ChainGetValue(linkPtr);
	    penPtr = (LinePen *)stylePtr->base.penPtr;
	    if ((stylePtr->nStrips > 0) && (penPtr->errorBarLineWidth > 0)) {
		Blt_Draw2DSegments(graphPtr->display, drawable, 
			penPtr->traceGC, stylePtr->strips, stylePtr->nStrips);
	    }
	}
    } else {
	LinePen *penPtr;

	penPtr = (LinePen *)NORMALPEN(elemPtr);
	if ((Blt_ChainGetLength(linePtr->traces) > 0) && 
	    (penPtr->traceWidth > 0)) {
	    DrawTraces(graphPtr, drawable, linePtr, penPtr);
	}
    }

    if (linePtr->reqMaxSymbols > 0) {
	int total;

	total = 0;
	for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); 
	     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	    LinePenStyle *stylePtr;

	    stylePtr = Blt_ChainGetValue(linkPtr);
	    total += stylePtr->nSymbolPts;
	}
	linePtr->symbolInterval = total / linePtr->reqMaxSymbols;
	linePtr->symbolCounter = 0;
    }

    /* Symbols, error bars, values. */

    count = 0;
    for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	LinePenStyle *lineStylePtr;
	PenStyle *stylePtr;
	LinePen *penPtr;

	stylePtr = Blt_ChainGetValue(linkPtr);
	lineStylePtr = (LinePenStyle *)stylePtr;
	penPtr = (LinePen *)stylePtr->penPtr;
	if ((stylePtr->xErrorBarCnt > 0) && (penPtr->errorBarShow & SHOW_X)) {
	    Blt_Draw2DSegments(graphPtr->display, drawable, penPtr->errorBarGC, 
			       stylePtr->xErrorBars, stylePtr->xErrorBarCnt);
	}
	if ((stylePtr->yErrorBarCnt > 0) && (penPtr->errorBarShow & SHOW_Y)) {
	    Blt_Draw2DSegments(graphPtr->display, drawable, penPtr->errorBarGC, 
		stylePtr->yErrorBars, stylePtr->yErrorBarCnt);
	}
	if ((lineStylePtr->nSymbolPts > 0) && 
	    (penPtr->symbol.type != SYMBOL_NONE)) {
	    DrawSymbols(graphPtr, drawable, elemPtr, penPtr, 
		stylePtr->symbolSize, lineStylePtr->nSymbolPts, 
		lineStylePtr->symbolPts);
	}
	if (penPtr->valueShow != SHOW_NONE) {
	    DrawValues(graphPtr, drawable, elemPtr, penPtr, 
		lineStylePtr->nSymbolPts, lineStylePtr->symbolPts, 
		linePtr->symbolToData + count);
	}
	count += lineStylePtr->nSymbolPts;
    }
    linePtr->symbolInterval = 0;
}

/*
 * -----------------------------------------------------------------
 *
 * GetSymbolPostScriptInfo --
 *
 *	Set up the PostScript environment with the macros and
 *	attributes needed to draw the symbols of the element.
 *
 * Results:
 *	None.
 *
 * -----------------------------------------------------------------
 */
static void
GetSymbolPostScriptInfo(
    Graph *graphPtr,
    Blt_PostScript ps,
    LinePen *penPtr,
    int size)
{
    XColor *outlineColor, *fillColor, *defaultColor;

    /* Set line and foreground attributes */
    outlineColor = penPtr->symbol.outlineColor;
    fillColor = penPtr->symbol.fillColor;
    defaultColor = penPtr->traceColor;

    if (fillColor == COLOR_DEFAULT) {
	fillColor = defaultColor;
    }
    if (outlineColor == COLOR_DEFAULT) {
	outlineColor = defaultColor;
    }
    if (penPtr->symbol.type == SYMBOL_NONE) {
	Blt_LineAttributesToPostScript(ps, defaultColor, penPtr->traceWidth + 2,
		 &penPtr->traceDashes, CapButt, JoinMiter);
    } else {
	Blt_LineWidthToPostScript(ps, penPtr->symbol.outlineWidth);
	Blt_LineDashesToPostScript(ps, (Blt_Dashes *)NULL);
    }

    /*
     * Build a PostScript procedure to draw the symbols.  For bitmaps,
     * paint both the bitmap and its mask. Otherwise fill and stroke
     * the path formed already.
     */
    Blt_AppendToPostScript(ps, "\n/DrawSymbolProc {\n", (char *)NULL);
    switch (penPtr->symbol.type) {
    case SYMBOL_NONE:
	break;			/* Do nothing */
    case SYMBOL_BITMAP:
	{
	    int width, height;
	    double sx, sy, scale;

	    /*
	     * Compute how much to scale the bitmap.  Don't let the
	     * scaled bitmap exceed the bounding square for the
	     * symbol.
	     */
	    Tk_SizeOfBitmap(graphPtr->display, penPtr->symbol.bitmap,
		&width, &height);
	    sx = (double)size / (double)width;
	    sy = (double)size / (double)height;
	    scale = MIN(sx, sy);

	    if ((penPtr->symbol.mask != None) && (fillColor != NULL)) {
		Blt_AppendToPostScript(ps, "\n  % Bitmap mask is \"",
		    Tk_NameOfBitmap(graphPtr->display, penPtr->symbol.mask),
		    "\"\n\n  ", (char *)NULL);
		Blt_BackgroundToPostScript(ps, fillColor);
		Blt_BitmapToPostScript(ps, graphPtr->display, 
			penPtr->symbol.mask, scale, scale);
	    }
	    Blt_AppendToPostScript(ps, "\n  % Bitmap symbol is \"",
		Tk_NameOfBitmap(graphPtr->display, penPtr->symbol.bitmap),
		"\"\n\n  ", (char *)NULL);
	    Blt_ForegroundToPostScript(ps, outlineColor);
	    Blt_BitmapToPostScript(ps, graphPtr->display, 
		penPtr->symbol.bitmap, scale, scale);
	}
	break;
    default:
	if (fillColor != NULL) {
	    Blt_AppendToPostScript(ps, "  ", (char *)NULL);
	    Blt_BackgroundToPostScript(ps, fillColor);
	    Blt_AppendToPostScript(ps, "  Fill\n", (char *)NULL);
	}
	if ((outlineColor != NULL) && (penPtr->symbol.outlineWidth > 0)) {
	    Blt_AppendToPostScript(ps, "  ", (char *)NULL);
	    Blt_ForegroundToPostScript(ps, outlineColor);
	    Blt_AppendToPostScript(ps, "  stroke\n", (char *)NULL);
	}
	break;
    }
    Blt_AppendToPostScript(ps, "} def\n\n", (char *)NULL);
}

/*
 * -----------------------------------------------------------------
 *
 * SymbolsToPostScript --
 *
 * 	Draw a symbol centered at the given x,y window coordinate
 *	based upon the element symbol type and size.
 *
 * Results:
 *	None.
 *
 * Problems:
 *	Most notable is the round-off errors generated when
 *	calculating the centered position of the symbol.
 *
 * -----------------------------------------------------------------
 */
static void
SymbolsToPostScript(
    Graph *graphPtr,
    Blt_PostScript ps,
    LinePen *penPtr,
    int size,
    int nSymbolPts,
    Point2D *symbolPts)
{
    double symbolSize;
    static char *symbolMacros[] =
    {
	"Li", "Sq", "Ci", "Di", "Pl", "Cr", "Sp", "Sc", "Tr", "Ar", "Bm", 
	(char *)NULL,
    };
    GetSymbolPostScriptInfo(graphPtr, ps, penPtr, size);

    symbolSize = (double)size;
    switch (penPtr->symbol.type) {
    case SYMBOL_SQUARE:
    case SYMBOL_CROSS:
    case SYMBOL_PLUS:
    case SYMBOL_SCROSS:
    case SYMBOL_SPLUS:
	symbolSize = (double)Round(size * S_RATIO);
	break;
    case SYMBOL_TRIANGLE:
    case SYMBOL_ARROW:
	symbolSize = (double)Round(size * 0.7);
	break;
    case SYMBOL_DIAMOND:
	symbolSize = (double)Round(size * M_SQRT1_2);
	break;

    default:
	break;
    }
    {
	Point2D *pp, *endp;

	for (pp = symbolPts, endp = symbolPts + nSymbolPts; pp < endp; pp++) {
	    Blt_FormatToPostScript(ps, "%g %g %g %s\n", pp->x, pp->y, 
		symbolSize, symbolMacros[penPtr->symbol.type]);
	}
    }
}

/*
 * -----------------------------------------------------------------
 *
 * SymbolToPostScript --
 *
 * 	Draw the symbol centered at the each given x,y coordinate.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Draws a symbol at the coordinate given.
 *
 * -----------------------------------------------------------------
 */
static void
SymbolToPostScript(
    Graph *graphPtr,		/* Graph widget record */
    Blt_PostScript ps,
    Element *elemPtr,		/* Line element information */
    double x, double y,		/* Center position of symbol */
    int size)			/* Size of element */
{
    LinePen *penPtr;

    penPtr = (LinePen *)NORMALPEN(elemPtr);
    if (penPtr->traceWidth > 0) {
	/*
	 * Draw an extra line offset by one pixel from the previous to
	 * give a thicker appearance.  This is only for the legend
	 * entry.  This routine is never called for drawing the actual
	 * line segments.
	 */
	Blt_LineAttributesToPostScript(ps, penPtr->traceColor,
	    penPtr->traceWidth + 2, &(penPtr->traceDashes), CapButt, JoinMiter);
	Blt_FormatToPostScript(ps, "%g %g %d Li\n", x, y, size + size);
    }
    if (penPtr->symbol.type != SYMBOL_NONE) {
	Point2D point;

	point.x = x, point.y = y;
	SymbolsToPostScript(graphPtr, ps, penPtr, size, 1, &point);
    }
}

static void
SetLineAttributes(Blt_PostScript ps, LinePen *penPtr)
{
    /* Set the attributes of the line (color, dashes, linewidth) */
    Blt_LineAttributesToPostScript(ps, penPtr->traceColor,
	penPtr->traceWidth, &penPtr->traceDashes, CapButt, JoinMiter);
    if ((LineIsDashed(penPtr->traceDashes)) && 
	(penPtr->traceOffColor != NULL)) {
	Blt_AppendToPostScript(ps, "/DashesProc {\n  gsave\n    ", 
		(char *)NULL);
	Blt_BackgroundToPostScript(ps, penPtr->traceOffColor);
	Blt_AppendToPostScript(ps, "    ", (char *)NULL);
	Blt_LineDashesToPostScript(ps, (Blt_Dashes *)NULL);
	Blt_AppendToPostScript(ps, "stroke\n  grestore\n} def\n", (char *)NULL);
    } else {
	Blt_AppendToPostScript(ps, "/DashesProc {} def\n", (char *)NULL);
    }
}

static void
TracesToPostScript(Blt_PostScript ps, Line *linePtr, LinePen *penPtr)
{
    Blt_ChainLink *linkPtr;

    SetLineAttributes(ps, penPtr);
    for (linkPtr = Blt_ChainFirstLink(linePtr->traces); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	Point2D *pp, *endp;
	Trace *tracePtr;
	int count;

	tracePtr = Blt_ChainGetValue(linkPtr);
	if (tracePtr->nScreenPts <= 0) {
	    continue;
	}
#define PS_MAXPATH	1500	/* Maximum number of components in a PostScript
				 * (level 1) path. */
	pp = tracePtr->screenPts;
	Blt_FormatToPostScript(ps, " newpath %g %g moveto\n", pp->x, pp->y);
	pp++;
	count = 0;
	for (endp = tracePtr->screenPts + (tracePtr->nScreenPts - 1);
	     pp < endp; pp++) {
	    Blt_FormatToPostScript(ps, " %g %g lineto\n", pp->x, pp->y);
	    if ((count % PS_MAXPATH) == 0) {
		Blt_FormatToPostScript(ps, 
		   "DashesProc stroke\n newpath  %g %g moveto\n", pp->x, pp->y);
	    }
	    count++;
	}
	Blt_FormatToPostScript(ps, " %g %g lineto\n", pp->x, pp->y);
	Blt_AppendToPostScript(ps, "DashesProc stroke\n", (char *)NULL);
    }
}


static void
ValuesToPostScript(
    Blt_PostScript ps,
    Element *elemPtr,
    LinePen *penPtr,
    int nSymbolPts,
    Point2D *symbolPts,
    int *pointToData)
{
    Point2D *pp, *endp;
    int count;
    char string[TCL_DOUBLE_SPACE * 2 + 2];
    char *fmt;
    
    fmt = penPtr->valueFormat;
    if (fmt == NULL) {
	fmt = "%g";
    }
    count = 0;
    for (pp = symbolPts, endp = symbolPts + nSymbolPts; pp < endp; pp++) {
	double x, y;

	x = elemPtr->x.valueArr[pointToData[count]];
	y = elemPtr->y.valueArr[pointToData[count]];
	count++;
	if (penPtr->valueShow == SHOW_X) {
	    sprintf(string, fmt, x); 
	} else if (penPtr->valueShow == SHOW_Y) {
	    sprintf(string, fmt, y); 
	} else if (penPtr->valueShow == SHOW_BOTH) {
	    sprintf(string, fmt, x);
	    strcat(string, ",");
	    sprintf(string + strlen(string), fmt, y);
	}
	Blt_TextToPostScript(ps, string, &penPtr->valueStyle, pp->x, pp->y);
    } 
}


/*
 *----------------------------------------------------------------------
 *
 * ActiveLineToPostScript --
 *
 *	Generates PostScript commands to draw as "active" the points
 *	(symbols) and or line segments (trace) representing the
 *	element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript pen width, dashes, and color settings are changed.
 *
 *----------------------------------------------------------------------
 */
static void
ActiveLineToPostScript(
    Graph *graphPtr,
    Blt_PostScript ps,
    Element *elemPtr)
{
    Line *linePtr = (Line *)elemPtr;
    LinePen *penPtr = (LinePen *)elemPtr->activePenPtr;
    int symbolSize;

    if (penPtr == NULL) {
	return;
    }
    symbolSize = ScaleSymbol(elemPtr, penPtr->symbol.size);
    if (elemPtr->nActiveIndices > 0) {
	if (elemPtr->flags & ACTIVE_PENDING) {
	    MapActiveSymbols(graphPtr, elemPtr);
	}
	if (penPtr->symbol.type != SYMBOL_NONE) {
	    SymbolsToPostScript(graphPtr, ps, penPtr, symbolSize,
		linePtr->nActivePts, linePtr->activePts);
	}
	if (penPtr->valueShow != SHOW_NONE) {
	    ValuesToPostScript(ps, elemPtr, penPtr, linePtr->nActivePts,
		       linePtr->activePts, linePtr->activeToData);
	}
    } else if (elemPtr->nActiveIndices < 0) {
	if (penPtr->traceWidth > 0) {
	    if (linePtr->nStrips > 0) {
		SetLineAttributes(ps, penPtr);
		Blt_2DSegmentsToPostScript(ps, linePtr->strips, 
			linePtr->nStrips);
	    }
	    if (Blt_ChainGetLength(linePtr->traces) > 0) {
		TracesToPostScript(ps, linePtr, (LinePen *)penPtr);
	    }
	}
	if (penPtr->symbol.type != SYMBOL_NONE) {
	    SymbolsToPostScript(graphPtr, ps, penPtr, symbolSize,
		linePtr->nSymbolPts, linePtr->symbolPts);
	}
	if (penPtr->valueShow != SHOW_NONE) {
	    ValuesToPostScript(ps, elemPtr, penPtr, linePtr->nSymbolPts, 
		       linePtr->symbolPts, linePtr->symbolToData);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * NormalLineToPostScript --
 *
 *	Similar to the DrawLine procedure, prints PostScript related
 *	commands to form the connected line(s) representing the element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript pen width, dashes, and color settings are changed.
 *
 *----------------------------------------------------------------------
 */
static void
NormalLineToPostScript(
    Graph *graphPtr,
    Blt_PostScript ps,
    Element *elemPtr)
{
    Line *linePtr = (Line *)elemPtr;
    Blt_ChainLink *linkPtr;
    unsigned int count;

    /* Draw fill area */
    if (linePtr->fillPts != NULL) {
	/* Create a path to use for both the polygon and its outline. */
	Blt_PathToPostScript(ps, linePtr->fillPts, linePtr->nFillPts);
	Blt_AppendToPostScript(ps, "closepath\n", (char *)NULL);

	/* If the background fill color was specified, draw the
	 * polygon in a solid fashion with that color.  */
	if (linePtr->fillBgColor != NULL) {
	    Blt_BackgroundToPostScript(ps, linePtr->fillBgColor);
	    Blt_AppendToPostScript(ps, "Fill\n", (char *)NULL);
	}
	Blt_ForegroundToPostScript(ps, linePtr->fillFgColor);
	if (linePtr->fillTile != NULL) {
	    /* TBA: Transparent tiling is the hard part. */
	} else if ((linePtr->fillStipple != None) &&
		   (linePtr->fillStipple != PATTERN_SOLID)) {
	    /* Draw the stipple in the foreground color. */
	    Blt_StippleToPostScript(ps, graphPtr->display, 
		linePtr->fillStipple);
	} else {
	    Blt_AppendToPostScript(ps, "Fill\n", (char *)NULL);
	}
    }

    /* Draw lines */
    if (linePtr->nStrips > 0) {
	for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); 
	     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	    LinePenStyle *lineStylePtr;
	    LinePen *penPtr;

	    lineStylePtr = Blt_ChainGetValue(linkPtr);
	    penPtr = (LinePen *)lineStylePtr->base.penPtr;
	    if ((lineStylePtr->nStrips > 0) && (penPtr->traceWidth > 0)) {
		SetLineAttributes(ps, penPtr);
		Blt_2DSegmentsToPostScript(ps, lineStylePtr->strips, 
			lineStylePtr->nStrips);
	    }
	}
    } else {
	LinePen *penPtr;

	penPtr = (LinePen *)NORMALPEN(elemPtr);
	if ((Blt_ChainGetLength(linePtr->traces) > 0) &&
	       (penPtr->traceWidth > 0)) {
	    TracesToPostScript(ps, linePtr, penPtr);
	}
    }

    /* Draw symbols, error bars, values. */

    count = 0;
    for (linkPtr = Blt_ChainFirstLink(elemPtr->stylePalette); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	PenStyle *stylePtr;
	LinePenStyle *lineStylePtr;
	LinePen *penPtr;
	XColor *colorPtr;

	stylePtr = Blt_ChainGetValue(linkPtr);
	lineStylePtr = (LinePenStyle *)stylePtr;
	penPtr = (LinePen *)stylePtr->penPtr;
	colorPtr = penPtr->errorBarColor;
	if (colorPtr == COLOR_DEFAULT) {
	    colorPtr = penPtr->traceColor;
	}
	if ((stylePtr->xErrorBarCnt > 0) && (penPtr->errorBarShow & SHOW_X)) {
	    Blt_LineAttributesToPostScript(ps, colorPtr,
		penPtr->errorBarLineWidth, NULL, CapButt, JoinMiter);
	    Blt_2DSegmentsToPostScript(ps, stylePtr->xErrorBars,
		stylePtr->xErrorBarCnt);
	}
	if ((stylePtr->yErrorBarCnt > 0) && (penPtr->errorBarShow & SHOW_Y)) {
	    Blt_LineAttributesToPostScript(ps, colorPtr,
		   penPtr->errorBarLineWidth, NULL, CapButt, JoinMiter);
	    Blt_2DSegmentsToPostScript(ps, stylePtr->yErrorBars,
		stylePtr->yErrorBarCnt);
	}
	if ((lineStylePtr->nSymbolPts > 0) &&
	    (penPtr->symbol.type != SYMBOL_NONE)) {
	    SymbolsToPostScript(graphPtr, ps, penPtr, stylePtr->symbolSize, 
		lineStylePtr->nSymbolPts, lineStylePtr->symbolPts);
	}
	if (penPtr->valueShow != SHOW_NONE) {
	    ValuesToPostScript(ps, elemPtr, penPtr, lineStylePtr->nSymbolPts, 
		lineStylePtr->symbolPts, linePtr->symbolToData + count);
	}
	count += lineStylePtr->nSymbolPts;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyLine --
 *
 *	Release memory and resources allocated for the line element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the line element is freed up.
 *
 *----------------------------------------------------------------------
 */
#define FREEVECTOR(v) \
    if ((v).clientId != NULL) { \
	Blt_FreeVectorId((v).clientId); \
    } else if ((v).valueArr != NULL) { \
	Blt_Free((v).valueArr); \
    } 

static void
DestroyLine(Graph *graphPtr, Element *elemPtr)
{
    Line *linePtr = (Line *)elemPtr;

    DestroyPen(graphPtr, (Pen *)&linePtr->builtinPen);
    if (elemPtr->activePenPtr != NULL) {
	Blt_FreePen((Pen *)elemPtr->activePenPtr);
    }
    FREEVECTOR(elemPtr->w);
    FREEVECTOR(elemPtr->x);
    FREEVECTOR(elemPtr->xHigh);
    FREEVECTOR(elemPtr->xLow);
    FREEVECTOR(elemPtr->xError);
    FREEVECTOR(elemPtr->y);
    FREEVECTOR(elemPtr->yHigh);
    FREEVECTOR(elemPtr->yLow);
    FREEVECTOR(elemPtr->yError);

    ResetLine(elemPtr);
    if (elemPtr->stylePalette != NULL) {
	Blt_FreeStylePalette(elemPtr->stylePalette);
	Blt_ChainDestroy(elemPtr->stylePalette);
    }
    if (elemPtr->activeIndices != NULL) {
	Blt_Free(elemPtr->activeIndices);
    }
    if (linePtr->fillPts != NULL) {
	Blt_Free(linePtr->fillPts);
    }
    if (linePtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, linePtr->fillGC);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_LineElement --
 *
 *	Allocate memory and initialize methods for the new line element.
 *
 * Results:
 *	The pointer to the newly allocated element structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the line element structure.
 *
 *----------------------------------------------------------------------
 */

static ElementProcs lineProcs =
{
    ClosestLine,		/* Finds the closest element/data point */
    ConfigureLine,		/* Configures the element. */
    DestroyLine,		/* Destroys the element. */
    DrawActiveLine,		/* Draws active element */
    DrawNormalLine,		/* Draws normal element */
    DrawSymbol,			/* Draws the element symbol. */
    GetLineExtents,		/* Find the extents of the element's data. */
    ActiveLineToPostScript,	/* Prints active element. */
    NormalLineToPostScript,	/* Prints normal element. */
    SymbolToPostScript,		/* Prints the line's symbol. */
    MapLine			/* Compute element's screen coordinates. */
};

Element *
Blt_LineElement(Graph *graphPtr, CONST char *name, int classId)
{
    Line *linePtr;

    linePtr = Blt_Calloc(1, sizeof(Line));
    assert(linePtr);
    linePtr->base.procsPtr = &lineProcs;
    if (classId == OBJECT_CLASS_LINE_ELEMENT) {
	linePtr->base.configSpecs = lineElemConfigSpecs;
    } else {
	linePtr->base.configSpecs = stripElemConfigSpecs;
    }
    linePtr->base.object.name = Blt_Strdup(name);
    Blt_GraphSetObjectClass(&linePtr->base.object, classId);
    linePtr->base.flags = SCALE_SYMBOL;
    linePtr->base.object.graphPtr = graphPtr;
    /* By default an element's name and label are the same. */
    linePtr->base.label = Blt_Strdup(name);
    linePtr->base.labelRelief = TK_RELIEF_FLAT;
    linePtr->penDir = PEN_BOTH_DIRECTIONS;
    linePtr->base.stylePalette = Blt_ChainCreate();
    linePtr->base.builtinPenPtr = (Pen *)&linePtr->builtinPen;
    linePtr->reqSmooth = PEN_SMOOTH_LINEAR;
    InitPen(&linePtr->builtinPen);
    bltLineStylesOption.clientData = (ClientData)sizeof(LinePenStyle);
    return (Element *)linePtr;
}
