/**
 * com.e4graph.VertexIterator:
 *
 *	This file contains the Java implementation of a class to iterate over
 *	the vertices of e4Graph node objects.
 *
 * Copyright (c) 2000-2003, JYL Software Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE, EVEN IF
 * JYL SOFTWARE INC. IS MADE AWARE OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.e4graph;

import java.util.Iterator;
import java.util.NoSuchElementException;

public final class VertexIterator implements Iterator
{
    /*
     * These are filters that restrict the name and type to visit.
     */

    private int vertexType	= Vertex.ILLEGAL;
    private String vertexName	= null;

    /*
     * State variables.
     */

    private Vertex current	= null;
    private Vertex next		= null;
    private Node node		= null;

    /**
     * Constructor that creates an iterator that visits every vertex in
     * this node.
     */
    public VertexIterator(Node n) {
	node = n;
	try {
	    computeCurrent();
	} catch (NoSuchElementException e) {
	    current = null;
	}
    }

    /**
     * Constructor that creates an iterator that visits only vertices with
     * the given type, in the given node.
     */
    public VertexIterator(Node n, int type) {
	node = n;
	vertexType = type;
	try {
	    computeCurrent();
	} catch (NoSuchElementException e) {
	    current = null;
	}
    }

    /**
     * Constructor that creates an iterator that visits only vertices with
     * the given name, in the given node.
     */
    public VertexIterator(Node n, String name) {
	node = n;
	vertexName = name;
	try {
	    computeCurrent();
	} catch (NoSuchElementException e) {
	    current = null;
	}
    }

    /**
     * Constructor that creates an iterator that visits only vertices with
     * the given name and type, in the given node.
     */
    public VertexIterator(Node n, String name, int type) {
	node = n;
	vertexName = name;
	vertexType = type;
	try {
	    computeCurrent();
	} catch (NoSuchElementException e) {
	    current = null;
	}
    }
    
    /**
     * Remove method -- always throws UnsupportedOperationException.
     */
    public void remove() 
	throws UnsupportedOperationException
    {
	throw new UnsupportedOperationException("remove() not supported");
    }

    /**
     * Does this iterator have a next element?
     */
    public boolean hasNext() {
	if (current == null) {
	    return false;
	}
	try {
	    computeNext();
	    return true;
	} catch (NoSuchElementException e) {
	    return false;
	}
    }

    /**
     * Get the next element of the iteration.
     */
    public Object next()
	throws NoSuchElementException
    {
	computeNext();
	current = next;
	next = null;
	return current;
    }

    /*
     * Compute the first element of the iteration.
     */
    private void computeCurrent()
	throws NoSuchElementException
    {
	try {
	    current = node.getVertexByRank(1);
	} catch (NoSuchVertexException e) {
	    throw new NoSuchElementException("node contains no vertices");
	}
	if (((vertexType != Vertex.ILLEGAL) &&
	     (vertexType != current.type())) ||
	    ((vertexName != null) &&
	     (!vertexName.equals(current.name())))) {
	    computeNext();
	    current = next;
	    next = null;
	}
    }
	    
    /*
     * Compute the next element of the iteration.
     */
    private void computeNext()
	throws NoSuchElementException
    {
	Vertex run;

	if (next != null) {
	    return;
	}
	try {
	    for (run = current.next(1); ; run = run.next(1)) {
		if ((vertexType != Vertex.ILLEGAL) &&
		    (vertexType != run.type())) {
		    continue;
		}
		if ((vertexName != null) &&
		    (!vertexName.equals(run.name()))) {
		    continue;
		}
		next = run;
		break;
	    }
	} catch (NoSuchVertexException e) {
	    throw new NoSuchElementException("no matching next vertex");
	} catch (NullPointerException n) {
	    throw new NoSuchElementException("node contains no vertices");
	}
    }
}

    
    
