/*****************************************************************************
 *
 *  File: kt2c.c
 *  Core of the Kanga Tcl-to-C converter.
 *
 *****************************************************************************
 *
 *  Copyright (c) 2000 David Cuthbert / Kanga International Group.
 *  All rights reserved.
 *
 *  Permission to use, copy, modify, distribute and sell this software
 *  and its documentation for any purpose is hereby granted without fee,
 *  provided that the above copyright notice appear in all copies and
 *  that both that copyright notice and this permission notice appear
 *  in supporting documentation.  The author(s) make no representations
 *  about the suitability of this software for any purpose.  It is 
 *  provided "as is" without express or implied warranty.
 *
 *****************************************************************************/

#include <assert.h>
#include <limits.h>
#include <time.h>

#if defined(_WIN32) || defined(__WIN32__)
#include <windows.h>
#else
#include <sys/times.h>
#endif

#include "tcl.h"
#include "tclInt.h"
#include "tclCompile.h"
#include "escape.h"
#include "kt2c.h"
#include "oprintf.h"

#define SUBST(a) "subst -nocommands -nobackslashes " a

static int AnalyseVmState(Kanga_CodeAccum *ca);
static int AnalyseInstructionEffect(Kanga_VmState *vm, int depth);
static int InitialiseCodeAccum(Kanga_CodeAccum *ca, Tcl_Interp *interp,
			       int objc, Tcl_Obj *const objv[]);
static int SetupVariables(Kanga_CodeAccum *ca);
static int SetForeachInitialiser(Kanga_CodeAccum *ca, int i);
static int CleanupCodeAccum(Kanga_CodeAccum *ca);
static int TransferError(Kanga_CodeAccum *ca, int result);
static int AddPrologue(Kanga_CodeAccum *ca);
static int AddPrologue2(Kanga_CodeAccum *ca);
static int AddAuxData(Kanga_CodeAccum *ca);
static int AddLiteralCreationCode(Kanga_CodeAccum *ca);
static int AddArgDetectionCode(Kanga_CodeAccum *ca);
static int AddCompiledBody(Kanga_CodeAccum *ca);
static int AddEpilogue(Kanga_CodeAccum *ca);
static int AddDeclaration(Kanga_CodeAccum *ca);
static int AddCompiledLocalInitialiser(Kanga_CodeAccum *ca);
static ExceptionRange *GetExceptRangeForPc (unsigned char *pc, int catchOnly,
					    ByteCode* codePtr);

/****************************************************************************/


int Kanga_Bytecode2C(ClientData dummy,      /* Not used. */
		     Tcl_Interp *interp,    /* Current interpreter. */
		     int objc,              /* Number of arguments. */
		     Tcl_Obj *const objv[]) /* Argument objects. */
{
    Kanga_CodeAccum *ca; /* Information about the code we're processing */
    int  i = 0;          /* Generic iteration variable */
    int result;
    Tcl_Obj *header_code[2];

    if (objc != 2) {
        Tcl_WrongNumArgs(interp, 1, objv, "procedure");
        return TCL_ERROR;
    }

    /* Create the code accumulator */
    ca = Kanga_CodeAccum_Create(interp, Tcl_GetString(objv[1]));
    if(ca == 0) return TCL_ERROR;

    /* Get information about the state of the Tcl VM at each instruction */
    result = AnalyseVmState(ca);         if(result) goto done;

    /* Perform some variable setup */
    result = SetupVariables(ca);         if(result) goto done;

    /* Write out the header for the procedure */
    result = AddPrologue(ca);            if(result) goto done;
    result = AddAuxData(ca);             if(result) goto done;
    result = AddPrologue2(ca);           if(result) goto done;
    result = AddArgDetectionCode(ca);    if(result) goto done;
    result = AddCompiledBody(ca);        if(result) goto done;
    result = AddEpilogue(ca);            if(result) goto done;
    result = AddDeclaration(ca);         if(result) goto done;

    header_code[0] = ca->header;
    header_code[1] = ca->code;

    Tcl_SetObjResult(interp, Tcl_NewListObj(2, header_code));

 done:
    if(result != 0) Tcl_SetObjResult(interp, Tcl_GetObjResult(ca->interp));

    Kanga_CodeAccum_Delete(ca);
    return result;
}


static int AnalyseVmState(Kanga_CodeAccum *ca)
{
    Kanga_VmState *vmstate;
    CompiledLocal *cl;
    int i;
    Tcl_Obj *var_base = Tcl_NewObj();
    AuxDataType *fetype;

    fetype = TclGetAuxDataType("ForeachInfo");
    assert(fetype != 0);

    Tcl_IncrRefCount(var_base);

    /* Set up enumeration values */
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("vsUnused", -1), 0, 
		   Tcl_NewIntObj(vsUnused), TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("vsInt", -1), 0,
		   Tcl_NewIntObj(vsInt), TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("vsDouble", -1), 0,
		   Tcl_NewIntObj(vsDouble), TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("vsNumeric", -1), 0,
		   Tcl_NewIntObj(vsNumeric), TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("vsUnknown", -1), 0,
		   Tcl_NewIntObj(vsUnknown), TCL_GLOBAL_ONLY);

    /* Set up foreach variables */
    for(i = 0; i < ca->bc->numAuxDataItems; i++) {
	ForeachInfo *foreach;
	
	if(ca->bc->auxDataArrayPtr[i].type != fetype) continue;

	foreach = (ForeachInfo *)ca->bc->auxDataArrayPtr[i].clientData;
	Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("foreach_scalar", -1),
		       Tcl_NewIntObj(i), Tcl_NewIntObj(foreach->loopCtTemp),
		       TCL_GLOBAL_ONLY);
    }

    /* Set up the literal type table */
    oprintf(var_base, "literal");
    for(i = 0; i < ca->bc->numLitObjects; ++i) {
	int     ival;
	double  dval;

	if(Tcl_GetIntFromObj(0, ca->bc->objArrayPtr[i], &ival) == 0)
	    ca->literalState[i] = vsInt;
	else if(Tcl_GetDoubleFromObj(0, ca->bc->objArrayPtr[i], &dval) == 0)
	    ca->literalState[i] = vsDouble;
	else
	    ca->literalState[i] = vsUnknown;

	Tcl_ObjSetVar2(ca->interp, var_base, Tcl_NewIntObj(i),
		       Tcl_NewIntObj(ca->literalState[i]), TCL_GLOBAL_ONLY);
    }

    /* Create a VM state for the first instruction */
    vmstate = Kanga_VmState_Create(&ca->instrState[0]);
    
    vmstate->valid = 1;
    vmstate->stackSize = 0;

    /* Set up the known state upon entering the procedure:
     *     All stack items are unused.
     *     All temporary compiled locals are unused.
     *     All other compiled locals are used and have an undetermined type.
     */
    for(i = 0; i <= ca->bc->maxStackDepth; i++) {
	vmstate->stackState[i] = vsUnused;
    }

    oprintf(var_base, "stack");
    Tcl_ObjSetVar2(ca->interp, var_base, 0, Tcl_NewObj(), TCL_GLOBAL_ONLY);

    oprintf(var_base, "scalar");
    cl = ca->procPtr->firstLocalPtr;
    for(i = 0; i < ca->procPtr->numArgs; i++) {
	assert(cl != NULL);
	vmstate->scalarState[i] = vsUnknown;

	Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("scalar", 6),
		       Tcl_NewIntObj(i),
		       Tcl_NewIntObj(vsUnknown), TCL_GLOBAL_ONLY);

	cl = cl->nextPtr;
    }

    for(; i < ca->procPtr->numCompiledLocals; i++) {
	assert(cl != NULL);

	if(cl->flags & VAR_TEMPORARY) 
	    vmstate->scalarState[i] = vsUnused;
	else
	    vmstate->scalarState[i] = vsUnknown;

	Tcl_ObjSetVar2(ca->interp, var_base, Tcl_NewIntObj(i),
		       Tcl_NewIntObj(vmstate->scalarState[i]),
		       TCL_GLOBAL_ONLY);

	cl = cl->nextPtr;
    }

    /* Add this state to the instruction */
    Kanga_InstrState_Imbue(&ca->instrState[0], vmstate);

    /* We should have iterated over every compiled local, so cl should now
     * be NULL.
     */
    assert(cl == NULL);

    Tcl_DecrRefCount(var_base);

    return AnalyseInstructionEffect(vmstate, 0);
}


static int AnalyseNormalInstruction(Kanga_VmState const *vm,
				    Kanga_VmState **nvmp,
				    Kanga_CodeAccum *ca,
				    InstructionDesc *idesc,
				    int index,
				    int offset)
{
    int            i;
    int            result;
    Tcl_Obj       *run;
    Tcl_Obj       *stack;
    Tcl_Obj       *scalar;
    Kanga_VmState *nvm;
    Tcl_Obj       *var_name = Tcl_NewObj();
    Tcl_Obj       *stack_name = Tcl_NewStringObj("stack", -1);
    Tcl_Obj       *scalar_name = Tcl_NewStringObj("scalar", -1);

    Tcl_IncrRefCount(var_name);
    Tcl_IncrRefCount(stack_name);
    Tcl_IncrRefCount(scalar_name);

    /* Set variables indicating the name, operands, etc. */
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("offset", -1), 0,
		   Tcl_NewIntObj(offset), TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("iname", -1), 0,
		   Tcl_NewStringObj(idesc->name, -1), TCL_GLOBAL_ONLY);
    
    for(i = 0; i < MAX_INSTRUCTION_OPERANDS; i++) {
	oprintf(var_name, "op%d", i+1);

	if(i < idesc->numOperands) {
	    Tcl_ObjSetVar2(ca->interp, var_name, 0,
			   Tcl_NewIntObj(vm->instr->op[i]),
			   TCL_GLOBAL_ONLY);
	} else {
	    Tcl_UnsetVar2(ca->interp, Tcl_GetString(var_name), 0,
			  TCL_GLOBAL_ONLY);
	}
    }

    oprintf(var_name, "vm::%s", idesc->name);

    /* Set variables for the stack and scalars */
    stack = Tcl_NewObj();
    Tcl_IncrRefCount(stack);
    for(i = 0; i < vm->stackSize; i++) {
	Tcl_ListObjAppendElement(ca->interp, stack, 
				 Tcl_NewIntObj(vm->stackState[i]));
    }
    Tcl_ObjSetVar2(ca->interp, stack_name, 0, stack, TCL_GLOBAL_ONLY);
    Tcl_DecrRefCount(stack);

    for(i = 0; i < vm->instr->ca->procPtr->numCompiledLocals; i++) {
	Tcl_ObjSetVar2(ca->interp, scalar_name, Tcl_NewIntObj(i),
		       Tcl_NewIntObj(vm->scalarState[i]), TCL_GLOBAL_ONLY);
    }

    /* Retrieve the code for the instruction */
    run = Tcl_ObjGetVar2(ca->interp, var_name, 0, 
			 TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);
    if(run == 0) {
	fprintf(stderr, "AnalyseNormalInstruction: couldn't read variable "
		"vm::%s\n", idesc->name);
	result = TCL_ERROR;
	goto done;
    }

    /* Execute it */
    Tcl_IncrRefCount(run);
    result = Tcl_EvalObjEx(ca->interp, run, TCL_EVAL_GLOBAL);
    Tcl_DecrRefCount(run);
    if(result) {
	fprintf(stderr, "AnalyseNormalInstruction: error while evaluating "
		"script for %s\n", idesc->name);
	goto done;
    }

    /* Now retrieve the state of the stack and scalars */
    stack = Tcl_ObjGetVar2(ca->interp, stack_name, 0,
			   TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);

    if(stack == 0) {
	fprintf(stderr, "AnalyseNormalInstruction: couldn't read variable "
		"stack.\n");
	result = TCL_ERROR;
	goto done;
    }

    Tcl_IncrRefCount(stack);

    nvm = Kanga_VmState_Create(&ca->instrState[index+1]);
    nvm->valid = 1;
		
    result = Tcl_ListObjLength(ca->interp, stack,
			       &nvm->stackSize);
    if(result) {
	fprintf(stderr, "AnalyseNormalInstruction: couldn't find length of "
		"stack.\n");
	Tcl_DecrRefCount(stack);
	Kanga_VmState_Delete(nvm);
	goto done;
    }

    for(i = 0; i <= ca->bc->maxStackDepth; i++) {
	if(i >= nvm->stackSize)
	    nvm->stackState[i] = vsUnused;
	else {
	    Tcl_Obj *ssptr;
			
	    result = Tcl_ListObjIndex(ca->interp, stack, i, &ssptr);
	    if(result) {
		fprintf(stderr, "AnalyseNormalInstruction: couldn't obtain "
			"element %d of stack (size = %d).\n", i,
			nvm->stackSize);
		Tcl_DecrRefCount(stack);
		Kanga_VmState_Delete(nvm);
		goto done;
	    }

	    assert(ssptr != 0);

	    Tcl_IncrRefCount(ssptr);
	    result = Tcl_GetIntFromObj(
		ca->interp, ssptr, (int *)&nvm->stackState[i]);
	    Tcl_DecrRefCount(ssptr);
			
	    if(result) {
		fprintf(stderr, "AnalyseNormalInstruction: couldn't convert "
			"\"%s\" to int.\n", Tcl_GetString(ssptr));
		Tcl_DecrRefCount(stack);
		Kanga_VmState_Delete(nvm);
		goto done;
	    }
	}
    }

    Tcl_DecrRefCount(stack);

    for(i = 0; i < ca->procPtr->numCompiledLocals; i++) {
	scalar = Tcl_ObjGetVar2(ca->interp, scalar_name,
				Tcl_NewIntObj(i), 
				TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);

	if(scalar == 0) {
	    fprintf(stderr, "AnalyseNormalInstruction: couldn't read "
		    "variable scalar(%d).\n", i);
	    result = TCL_ERROR;
	    goto done;
	}

	Tcl_IncrRefCount(scalar);
	result = Tcl_GetIntFromObj(
	    ca->interp, scalar, (int *)&nvm->scalarState[i]);
	Tcl_DecrRefCount(scalar);

	if(result) {
	    fprintf(stderr, "AnalyseNormalInstruction: couldn't convert \"%s\""
		    " to int.\n", Tcl_GetString(scalar));
	    goto done;
	}
    }

 done:
    if(result == 0) *nvmp = nvm;

    Tcl_DecrRefCount(var_name);
    Tcl_DecrRefCount(stack_name);
    Tcl_DecrRefCount(scalar_name);
    return result;
}


struct KVm_Linker {
    struct KVm_Linker *next;
    int                depth;
    Kanga_VmState     *vm;
};

typedef struct KVm_Linker KVm_Linker;

static int AnalyseInstructionEffect(Kanga_VmState *first_vm, int depth)
{
    int result = TCL_OK;
    Kanga_CodeAccum  *ca = first_vm->instr->ca;
    Tcl_Obj *var_name = Tcl_NewObj();
    KVm_Linker *chain, *chain_next;

    Tcl_IncrRefCount(var_name);

    chain = (KVm_Linker *)ckalloc(sizeof(KVm_Linker));
    chain->next = 0;
    chain->depth = 1;
    chain->vm = first_vm;

    while(chain != 0) {
	Kanga_VmState    *vm = chain->vm;
	Kanga_InstrState *istate = vm->instr;
	int               offset = istate->offset;
	unsigned int      instr = istate->instruction;
	InstructionDesc  *idesc = TclGetInstructionTable() + instr;
	int               index, nindex[2], brkindex, cntindex, errindex;
	Kanga_VmState    *nvm[2];
	Tcl_HashEntry    *he;
	int               i, j;	

#ifndef NDEBUG
	fprintf(stderr, "AnalyseInstructionEffect %04d depth %d %s", offset, 
		chain->depth, idesc->name);

	for(i = 0; i < idesc->numOperands; i++)
	    fprintf(stderr, " %d", istate->op[i]);

#endif

	/* Find the index of the instruction into the instrState array */
	he = Tcl_FindHashEntry(&ca->offset2Index, (char *)offset);
	assert(he != 0);
	index = (int)Tcl_GetHashValue(he);

	/* Assume that the next instruction to execute is the next instruction
	 * in the bytecode.
	 */
	nindex[0] = index + 1;

	/* Assume that we're not on a conditional branch */
	nindex[1] = -1;

	/* Assume that breaks, continues, and errors will go to 
	 * abnormalReturn */
	brkindex = -1;
	cntindex = -1;
	errindex = -1;

	/* Assume that we will produce no new vm states */
	nvm[0] = nvm[1] = 0;

	/* Act on the instruction */
	switch(instr) {
	    case INST_DONE:
	    break;

	    case INST_JUMP1:
	    case INST_JUMP4:
	    he = Tcl_FindHashEntry(&ca->offset2Index, 
				   (char *)(offset + istate->op[0]));
	    assert(he != 0);
	    nindex[0] = (int)Tcl_GetHashValue(he);
	    nvm[0] = Kanga_VmState_Create(&ca->instrState[index + 1]);
	    nvm[0]->valid = 1;
	    Kanga_VmState_CopyStack(nvm[0], vm);
	    Kanga_VmState_CopyScalar(nvm[0], vm);
	    nvm[0]->stackSize = vm->stackSize;
	    break;

	    case INST_JUMP_FALSE1:
	    case INST_JUMP_FALSE4:
	    case INST_JUMP_TRUE1:
	    case INST_JUMP_TRUE4:
	    /* Find the index of the jump target in the instrState array */
	    he = Tcl_FindHashEntry(&ca->offset2Index,
				   offset + (char *)(istate->op[0]));
	    assert(he != 0);
	    nindex[1] = (int)Tcl_GetHashValue(he);

	    nvm[0] = Kanga_VmState_Create(&ca->instrState[index + 1]);
	    nvm[0]->valid = 1;
	    Kanga_VmState_CopyStack(nvm[0], vm);
	    Kanga_VmState_CopyScalar(nvm[0], vm);
	    nvm[0]->stackSize = vm->stackSize - 1;
	    nvm[0]->stackState[vm->stackSize - 1] = vsUnused;

	    nvm[1] = Kanga_VmState_Create(&ca->instrState[nindex[1]]);
	    nvm[1]->valid = 1;
	    Kanga_VmState_CopyStack(nvm[1], vm);
	    Kanga_VmState_CopyScalar(nvm[1], vm);
	    nvm[1]->stackSize = vm->stackSize - 1;
	    nvm[1]->stackState[vm->stackSize - 1] = vsUnused;
	    break;

	    default:
	    result = AnalyseNormalInstruction(vm, &nvm[0], ca, idesc, index,
					      offset);
	    if(result) goto done;
	}

	for(i = 0; i < 2; i++) {
	    Kanga_InstrState *nistate;
	    Kanga_VmState    *existing_vm;
	    KVm_Linker *chain_iterator;
	    KVm_Linker *new_link;

	    if(nvm[i] == 0) break;
	    assert(nindex[i] != -1);
	
	    nistate = ca->instrState + nindex[i];

	    /* Has the next instruction already seen this vm state? */
	    existing_vm = Kanga_InstrState_VmStateExists(nistate, nvm[i]);
	    if(existing_vm != 0) {
#ifndef NDEBUG
		fprintf(stderr, " x%04d", nvm[i]->instr->offset);
#endif

		/* Yes, delete our duplicate. */
		Kanga_VmState_Delete(nvm[i]);

		/* Make sure our current VM state knows to follow on to this
		 * existing one.
		 */
		vm->nextState[i] = existing_vm;

		continue;
	    }

#ifndef NDEBUG
	    fprintf(stderr, " +%04d", nvm[i]->instr->offset);
#endif
	    nvm[i]->prev = vm;
	    nvm[i]->instr = nistate;

	    /* No; add it to that instruction */
	    Kanga_InstrState_Imbue(nistate, nvm[i]);

	    /* Add this state as a followup */
	    vm->nextState[i] = nvm[i];

	    /* And mark the next state for processing */
	    new_link = (KVm_Linker *)ckalloc(sizeof(KVm_Linker));
	    new_link->next = 0;
	    new_link->vm = nvm[i];

	    if(nvm[i]->instr->offset != vm->instr->offset + idesc->numBytes)
		new_link->depth = chain->depth + 1;
	    else
		new_link->depth = chain->depth;

	    for(chain_iterator = chain; chain_iterator->next != 0; 
		chain_iterator = chain_iterator->next)
	    {
		if(chain_iterator->next->depth > new_link->depth) break;
	    }

	    new_link->next = chain_iterator->next;
	    chain_iterator->next = new_link;
	}

#ifndef NDEBUG
	fprintf(stderr, "\n");
#endif

	chain_next = chain->next;
	ckfree((char *)chain);
	chain = chain_next;
    }

 done:
    Tcl_DecrRefCount(var_name);
    return result;
}


Kanga_CodeAccum *
Kanga_CodeAccum_Create(Tcl_Interp *interp, char *procName)
{
    int              result = 0;
    int              i;
    char            *libraryDir;
    char            *pathName;
    unsigned char   *code_ptr, *code_end;
    Kanga_CodeAccum *ca;
    Proc            *procPtr;
    Namespace       *nsPtr;
    Tcl_Interp      *slave;

    /* Find the procedure and the effective namespace in which to list
     * procs. */

    procPtr = TclFindProc((Interp *)interp, procName);
    if(procPtr == NULL) {
	Tcl_ResetResult(interp);
	oprintf(Tcl_GetObjResult(interp),
		"procedure %s does not exist", procName);
	return 0;
    }

    nsPtr = procPtr->cmdPtr->nsPtr;

    /* Compile the procedure if it isn't already compiled. */
    if(procPtr->bodyPtr->typePtr == NULL) {
	int result;
	result = TclProcCompileProc(interp, procPtr, procPtr->bodyPtr, 
				    nsPtr, "body of proc", procName);
	if(result != 0) return 0;
    }

    /* Find the library directory */
    libraryDir = Tcl_GetVar(interp, "::kt2c::LibraryDirectory",
			    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);
    if(libraryDir == 0) {
	Tcl_ResetResult(interp);
	oprintf(Tcl_GetObjResult(interp), 
		"cannot find kt2c library directory");
	return 0;
    }

    /* Create a new interpreter for evaluating the processed file */
    slave = Tcl_CreateInterp();

    pathName = (char *)ckalloc(strlen(libraryDir) + 32);

    /* Read the file that provides C equivalents for each instruction */
    strcpy(pathName, libraryDir);
    strcat(pathName, "/insertion.tcl");
    if(Tcl_EvalFile(slave, pathName) != 0) {
	Tcl_Obj *why = Tcl_GetObjResult(slave);
	Tcl_IncrRefCount(why);

	fprintf(stderr, "while reading %s: %s\n", pathName, 
		Tcl_GetString(why));
	Tcl_AddErrorInfo(interp, 
			 Tcl_GetVar(slave, "errorInfo", 
				    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG));
	Tcl_SetObjResult(interp, why);
	Tcl_DecrRefCount(why);
	Tcl_DeleteInterp(slave);
	ckfree(pathName);
	return 0;
    }

    /* Now read the file that knows how each instruction affects the VM */
    strcpy(pathName, libraryDir);
    strcat(pathName, "/effect.tcl");
    if(Tcl_EvalFile(slave, pathName) != 0) {
	Tcl_Obj *why = Tcl_GetObjResult(slave);
	Tcl_IncrRefCount(why);

	fprintf(stderr, "while reading %s: %s\n", pathName, 
		Tcl_GetString(why));
	Tcl_AddErrorInfo(interp, Tcl_GetVar(slave, "errorInfo", 
					    TCL_GLOBAL_ONLY));
	Tcl_SetObjResult(interp, why);
	Tcl_DecrRefCount(why);
	Tcl_DeleteInterp(slave);
	ckfree(pathName);
	return 0;
    }

    ckfree(pathName);

    /* Ok, we're in the clear.  Go ahead and create the CodeAccum structure */
    ca = (Kanga_CodeAccum *)ckalloc(sizeof(Kanga_CodeAccum));

    ca->original = interp;
    ca->interp   = slave;
    ca->code     = Tcl_NewObj();
    ca->header   = Tcl_NewObj();
    ca->bc       = (ByteCode *)procPtr->bodyPtr->internalRep.otherValuePtr;
    ca->procName = strdup(procName);
    ca->procNameLength = strlen(procName);
    ca->procPtr  = procPtr;
    ca->nsPtr    = nsPtr;

    Tcl_IncrRefCount(ca->code);
    Tcl_IncrRefCount(ca->header);

    /* Initialise the offset-to-instruction-index hash table */
    Tcl_InitHashTable(&ca->offset2Index, TCL_ONE_WORD_KEYS);

    /* Mangle the proc name so that we can use it as the name of a C function
     * and pass it as a C string.
     */
    ca->mangledProcName = (char *)ckalloc(strlen(ca->procName) + 2);
    MangleName(ca->mangledProcName, ca->procName);
    ca->mangledProcNameLength = strlen(ca->mangledProcName);
    
    ca->escapedProcNameLength =
	GetEscapeCStringSize(ca->procName, ca->procNameLength);
    ca->escapedProcName = (char *)ckalloc(ca->escapedProcNameLength + 1);
    EscapeCString(ca->escapedProcName, ca->procName, ca->procNameLength);

    /* Count the number of instructions contained in the bytecode */
    ca->numInstructions = 0;
    code_ptr = ca->bc->codeStart;
    code_end = code_ptr + ca->bc->numCodeBytes;
    for( ; code_ptr < code_end; ca->numInstructions++) {
	unsigned int     instr = (unsigned int)*code_ptr;
	InstructionDesc *idesc;
	Tcl_HashEntry   *he;
	int              is_new;
	int              offset = code_ptr - ca->bc->codeStart;

	assert(instr < LAST_INST_OPCODE);
	idesc = TclGetInstructionTable() + instr;
	code_ptr += idesc->numBytes;

	he = Tcl_CreateHashEntry(&ca->offset2Index, (char *)offset, &is_new);
	assert(is_new);

	Tcl_SetHashValue(he, ca->numInstructions);
    }

    assert(code_ptr == code_end);

    /* Create the VM state array */
    ca->instrState = Kanga_InstrState_CreateArray(ca, ca->numInstructions);

    ca->literalState = (Kanga_VarState *)
	ckalloc(sizeof(Kanga_VarState *) * ca->bc->numLitObjects);

    /* Run through again and initialise the instruction objects */
    for(code_ptr = ca->bc->codeStart, i = 0; code_ptr != code_end; i++) {
	unsigned int     instr = (unsigned int)*code_ptr;
	InstructionDesc *idesc;
	int              offset = code_ptr - ca->bc->codeStart;
	int              opoffset, opidx;

	ca->instrState[i].offset = offset;
	ca->instrState[i].instruction = instr;
	idesc = TclGetInstructionTable() + instr;

	assert(idesc->numOperands <= MAX_INSTRUCTION_OPERANDS);

	/* Get each operand */
	opoffset = 1;
	for(opidx = 0; opidx < idesc->numOperands; opidx++) {
	    switch(idesc->opTypes[opidx]) {
		case OPERAND_INT1:
		ca->instrState[i].op[opidx] = 
		    TclGetInt1AtPtr(code_ptr + opoffset);
		opoffset += 1;
		break;

		case OPERAND_INT4:
		ca->instrState[i].op[opidx] =
		    TclGetInt4AtPtr(code_ptr + opoffset);
		opoffset += 4;
		break;

		case OPERAND_UINT1:
		ca->instrState[i].op[opidx] =
		    TclGetUInt1AtPtr(code_ptr + opoffset);
		opoffset += 1;
		break;

		case OPERAND_UINT4:
		ca->instrState[i].op[opidx] =
		    TclGetUInt4AtPtr(code_ptr + opoffset);
		break;

		default:
		fprintf(stderr, "Error: Unknown operand type %d encountered "
			"for instruction %s at offset %d.\n",
			idesc->opTypes[opidx], idesc->name, offset);
		break;
	    }
	}

	code_ptr += idesc->numBytes;
    }

    return ca;
}


void
Kanga_CodeAccum_Delete(Kanga_CodeAccum *ca)
{
    Tcl_DecrRefCount(ca->code);
    Tcl_DecrRefCount(ca->header);

    Tcl_DeleteInterp(ca->interp);
    ckfree(ca->mangledProcName);
    ckfree(ca->escapedProcName);
    Tcl_DeleteHashTable(&ca->offset2Index);
    Kanga_InstrState_Delete(ca->instrState);
    ckfree((char *)ca->literalState);
    ckfree((char *)ca);

    return;
}


static int TransferError(Kanga_CodeAccum *ca, int result)
{
    Tcl_Obj *why = Tcl_GetObjResult(ca->interp);
    Tcl_IncrRefCount(why);
	
    Tcl_AddErrorInfo(ca->original, Tcl_GetVar(ca->interp, "errorInfo", 
					      TCL_GLOBAL_ONLY));
    Tcl_SetObjResult(ca->interp, why);
    Tcl_DecrRefCount(why);

    return result;
}

static int SetupVariables(Kanga_CodeAccum *ca)
{
    int result = TCL_OK;
    int i;

    Tcl_SetVar(ca->interp, "$", "$", TCL_GLOBAL_ONLY);
    Tcl_SetVar(ca->interp, "function", ca->mangledProcName, TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("maxStackDepth", -1), 0,
		   Tcl_NewIntObj(ca->bc->maxStackDepth + 1), TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("numCompiledLocals", -1), 0,
		   Tcl_NewIntObj(ca->procPtr->numCompiledLocals),
		   TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("procName", -1), 0,
		   Tcl_NewStringObj(ca->procName, ca->procNameLength),
		   TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("numLiterals", -1), 0,
		   Tcl_NewIntObj(ca->bc->numLitObjects), TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("maxExceptDepth", -1), 0,
		   Tcl_NewIntObj(ca->bc->maxExceptDepth), TCL_GLOBAL_ONLY);

    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("argLength", -1), 0,
		   Tcl_NewIntObj(ca->procPtr->numArgs), TCL_GLOBAL_ONLY);

 done:
    return result;
}


static int SetForeachInitialiser(Kanga_CodeAccum *ca, int i)
{
    ForeachInfo *foreach = (ForeachInfo *)ca->bc->auxDataArrayPtr[i].
	clientData;
    int maxVars;
    int j, k;

    oappendf(ca->code, "    int foreach_numLists_%d = %d;\n", i, 
	     foreach->numLists);
    oappendf(ca->code, "    List *foreach_list_%d[%d];\n", i, 
	     foreach->numLists);
    oappendf(ca->code, "    int foreach_listIndex_%d[%d];\n", i,
	     foreach->numLists);
    oappendf(ca->code, "    int foreach_firstList_%d = %d;\n", i,
	     foreach->firstValueTemp);
    oappendf(ca->code, "    int foreach_numVars_%d[] = { ", i);

    for(j = 0; j < foreach->numLists; ++j) {
	if(j == 0) {
	    oappendf(ca->code, "%d", foreach->varLists[j]->numVars);
	    maxVars = foreach->varLists[j]->numVars;
	}
	else {
	    oappendf(ca->code, ", %d", foreach->varLists[j]->numVars);
	    if(foreach->varLists[j]->numVars > maxVars) 
		maxVars = foreach->varLists[j]->numVars;
	}
    }

    oappendf(ca->code, " };\n");
    oappendf(ca->code, "    int foreach_varList_%d[%d][%d] = { ", i,
	     foreach->numLists, maxVars);

    for(j = 0; j < foreach->numLists; ++j) {
	ForeachVarList *varList = foreach->varLists[j];

	if(j > 0) oappendf(ca->code, ", ");
	oappendf(ca->code, "{ ");

	for(k = 0; k < maxVars; ++k) {
	    if(k == 0)
		oappendf(ca->code, "%d", varList->varIndexes[k]);
	    else if(k < varList->numVars)
		oappendf(ca->code, ", %d", varList->varIndexes[k]);
	    else
		oappendf(ca->code, ", -1");
	}

	oappendf(ca->code, " }");
    }

    oappendf(ca->code, " };\n");

    return TCL_OK;
}


static int AddPrologue(Kanga_CodeAccum *ca)
{
    int result;

    result = Tcl_Eval(ca->interp, SUBST("$__signature"));
    if(result) return TransferError(ca, result);

    Tcl_AppendObjToObj(ca->code, Tcl_GetObjResult(ca->interp));
    Tcl_AppendToObj(ca->code, "{\n", 2);

    result = Tcl_Eval(ca->interp, SUBST("$__prologue"));
    if(result != TCL_OK) return TransferError(ca, result);

    Tcl_AppendObjToObj(ca->code, Tcl_GetObjResult(ca->interp));

    return result;
}


static int AddPrologue2(Kanga_CodeAccum *ca)
{
    int result;

    result = Tcl_Eval(ca->interp, SUBST("$__prologue_2"));
    if(result != TCL_OK) return TransferError(ca, result);

    Tcl_AppendObjToObj(ca->code, Tcl_GetObjResult(ca->interp));

    return result;
}


static int AddAuxData(Kanga_CodeAccum *ca)
{
    Tcl_Obj *auxDataVar = Tcl_NewStringObj("auxDataType", -1);
    int i;
    int result = TCL_OK;
    AuxDataType *fetype;

    fetype = TclGetAuxDataType("ForeachInfo");
    assert(fetype != 0);

    Tcl_IncrRefCount(auxDataVar);

    /* Set up the auxdata structures */
    for(i = 0; i < ca->bc->numAuxDataItems; ++i) {
	Tcl_ObjSetVar2(ca->interp, auxDataVar, Tcl_NewIntObj(i), 
		       Tcl_NewStringObj(
			   ca->bc->auxDataArrayPtr[i].type->name, -1),
		       TCL_GLOBAL_ONLY);

	if(ca->bc->auxDataArrayPtr[i].type == fetype)
	    SetForeachInitialiser(ca, i);
	else {
	    Tcl_ResetResult(ca->interp);
	    oprintf(Tcl_GetObjResult(ca->interp), 
		    "unknown auxdata type %s at entry %d",
		    ca->bc->auxDataArrayPtr[i].type->name, i);
	    result = TCL_ERROR;
	    goto done;
	}
    }

 done:
    Tcl_DecrRefCount(auxDataVar);
    return result;
}


static int AddArgDetectionCode(Kanga_CodeAccum *ca)
{
    CompiledLocal *localPtr = ca->procPtr->firstLocalPtr;
    int nargs = ca->procPtr->numArgs;
    int i;
    int result;
    Tcl_Obj *usage = Tcl_NewStringObj(ca->procName, ca->procNameLength);
    Tcl_IncrRefCount(usage);

    for(i = 1; i <= nargs; i++) {
	Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("argIndex", -1), 0,
		       Tcl_NewIntObj(i), TCL_GLOBAL_ONLY);

	Tcl_SetVar(ca->interp, "argName", localPtr->name, TCL_GLOBAL_ONLY);

	/* Is the last argument args? */
	if(i == nargs && strcmp(localPtr->name, "args") == 0) {
	    result = Tcl_Eval(ca->interp, SUBST("$__detect_anyarg"));
	    oappendf(usage, " ...");
	} else if(localPtr->defValuePtr == NULL) {
	    result = Tcl_Eval(ca->interp, SUBST("$__detect_stdarg"));
	    oappendf(usage, " %s", localPtr->name);
	}
	else {
	    int    ival;
	    double dval;

	    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("defValue", -1), 0,
			   localPtr->defValuePtr, TCL_GLOBAL_ONLY);

	    /* Try to convert this to a number */
	    if(Tcl_GetIntFromObj(0, localPtr->defValuePtr, &ival) == 0)
		result = Tcl_Eval(ca->interp, SUBST("$__detect_defarg_int"));
	    else if(Tcl_GetDoubleFromObj(0, localPtr->defValuePtr, &dval) == 0)
		result = Tcl_Eval(ca->interp,
				  SUBST("$__detect_defarg_double"));
	    else
		result = Tcl_Eval(ca->interp,
				  SUBST("$__detect_defarg_string"));

	    oappendf(usage, " ?%s?", localPtr->name);
	}

	if(result) return TransferError(ca, result);

	Tcl_AppendObjToObj(ca->code, Tcl_GetObjResult(ca->interp));
	localPtr = localPtr->nextPtr;
    }

    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("usage", -1), 0,
		   usage, TCL_GLOBAL_ONLY);
    Tcl_DecrRefCount(usage);

    result = Tcl_Eval(ca->interp, SUBST("$__detect_end"));
    if(result) return TransferError(ca, result);

    Tcl_AppendObjToObj(ca->code, Tcl_GetObjResult(ca->interp));

    return result;
}


static void SetupExceptionJumps(Kanga_CodeAccum *ca, unsigned char *code_ptr)
{
    int offset = code_ptr - ca->bc->codeStart;
    ExceptionRange  *loopException;
    ExceptionRange  *miscException;
    Tcl_Obj         *breakJump = Tcl_NewObj();
    Tcl_Obj         *continueJump = Tcl_NewObj();
    Tcl_Obj         *catchJump = Tcl_NewObj();

    Tcl_IncrRefCount(breakJump);
    Tcl_IncrRefCount(continueJump);
    Tcl_IncrRefCount(catchJump);

    /* Find the exception ranges for this PC */
    loopException = GetExceptRangeForPc(code_ptr, 0, ca->bc);
	
    if(loopException == 0 || loopException->type == CATCH_EXCEPTION_RANGE)
	miscException = loopException;
    else
	miscException = GetExceptRangeForPc(code_ptr, 1, ca->bc);

    if(loopException == 0) {
	oprintf(breakJump, "goto abnormalReturn");
	oprintf(continueJump, "goto abnormalReturn");
    } else {
	oprintf(breakJump, "goto instruction_%04d", 
		loopException->breakOffset);
	oprintf(continueJump, "goto instruction_%04d",
		loopException->continueOffset);
    }

    if(miscException == 0) {
	oprintf(catchJump, "goto abnormalReturn");
    } else {
	oprintf(catchJump, 
		"{ while(stackPtr > unwindPtrs[unwindTop]) { "
		"stackPtr--; Tcl_DecrRefCount(*stackPtr); }; "
		"goto instruction_%04d; }",
		miscException->catchOffset);
    }

    /* Set the jump variables for these exceptions. */
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("breakJump", -1), 0,
		   breakJump, TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("continueJump", -1), 0,
		   continueJump, TCL_GLOBAL_ONLY);
    Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("catchJump", -1), 0,
		   catchJump, TCL_GLOBAL_ONLY);

    Tcl_DecrRefCount(breakJump);
    Tcl_DecrRefCount(continueJump);
    Tcl_DecrRefCount(catchJump);
    
    return;
}


static int AddCompiledBody(Kanga_CodeAccum *ca)
{
    unsigned char *code_base; /* Base of the bytecode */
    unsigned char *code_ptr;  /* Current piece of bytecode we're working on */
    unsigned char *code_end;  /* End+1 of the bytecode */
    int            i;         /* Generic iteration variable */
    int            depth = 1; /* Depth of exception/catch structures */
    char           cmd[256];
    int            result;
    int            count = 0;

    /* Ok, the procedure was found and is compiled.  Grab the basic information
       about the procedure, and allocate space for the returned C code. */
    code_base = ca->bc->codeStart;
    code_end  = code_base + ca->bc->numCodeBytes;

    /* Run through and convert each bytecode instruction to an
       equivalent set of C instructions. */
    for(code_ptr = code_base; code_ptr < code_end; count++) {
	int              offset = code_ptr - code_base;
	unsigned int     instr = (unsigned int)(*code_ptr);
	InstructionDesc *idesc = TclGetInstructionTable() + instr;
	int              j;
	Kanga_InstrState *istate = ca->instrState + count;
	Kanga_VmState *vm;

	assert(instr < LAST_INST_OPCODE);
	
	SetupExceptionJumps(ca, code_ptr);

	if(istate->firstVmState == 0) {

	    /* FIXME: We must always write out elided instructions for now 
	     * since the bytecode tracer does not yet follow breaks,
	     * continues, and catches.
	     */
#if defined(NDEBUG) || 1
	    fprintf(stderr, "Warning: Code at offset %d instruction %s is "
		    "unreachable.\n",
		    offset, idesc->name);
#else
	    oappendf(ca->code, "\n  /* Instruction %04d elided */\n", offset);
	    fprintf(stderr, "Warning: Code at offset %d instruction %s is "
		    "unreachable and will be elided.\n",
		    offset, idesc->name);

	    code_ptr += idesc->numBytes;
	    continue;
#endif
	}

	for(i = 0, vm = istate->firstVmState; vm != 0;
	    vm = vm->nextSibling, i++)
	{
	    oappendf(ca->code, "\n  /* VM State %d: prev = ", i);

	    if(vm->prev == 0)
		oappendf(ca->code, "none, stack = ");
	    else
		oappendf(ca->code, "%04d, stack = ", vm->prev->instr->offset);

	    for(j = 0; j <= ca->bc->maxStackDepth; j++) {
		if(j == vm->stackSize) oappendf(ca->code, "^");

		switch(vm->stackState[j]) {
		    case vsUnused:
		    oappendf(ca->code, "-");
		    break;
		    
		    case vsInt:
		    oappendf(ca->code, "i");
		    break;

		    case vsDouble:
		    oappendf(ca->code, "d");
		    break;

		    case vsNumeric:
		    oappendf(ca->code, "n");
		    break;

		    case vsUnknown:
		    oappendf(ca->code, "s");
		    break;

		    default:
		    oappendf(ca->code, "?");
		    break;
		}
	    }

	    if(vm->stackSize == ca->bc->maxStackDepth + 1)
		oappendf(ca->code, "^");

	    oappendf(ca->code, ", scalars = ");
	    for(j = 0; j < ca->procPtr->numCompiledLocals; j++) {
		switch(vm->scalarState[j]) {
		    case vsUnused:
		    oappendf(ca->code, "-");
		    break;
		    
		    case vsInt:
		    oappendf(ca->code, "i");
		    break;

		    case vsDouble:
		    oappendf(ca->code, "d");
		    break;

		    case vsNumeric:
		    oappendf(ca->code, "n");
		    break;

		    case vsUnknown:
		    oappendf(ca->code, "s");
		    break;

		    default:
		    oappendf(ca->code, "?");
		    break;
		}
	    }

	    oappendf(ca->code, " */");
	}

	/* Set the instruction pointer variable */
	Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("ip", -1), 0,
		       Tcl_NewIntObj(offset), TCL_GLOBAL_ONLY);

	/* Write out the current opcode as a comment */
	oappendf(ca->code, "\n  instruction_%04d: /* %s", offset, idesc->name);
	
	/* Write out each operand */
	j = 1; /* offset of the operand from code_ptr */
	for(i = 0; i < idesc->numOperands; ++i) {
	    int operand;
	    Tcl_Obj *opvar = Tcl_NewObj();
	    Tcl_IncrRefCount(opvar);
	    
	    switch(idesc->opTypes[i]) {
		case OPERAND_INT1:
       		operand = TclGetInt1AtPtr(code_ptr + j);
		oappendf(ca->code, " %d", operand);
		j += 1;
		break;

		case OPERAND_INT4:
       		operand = TclGetInt4AtPtr(code_ptr + j);
		oappendf(ca->code, " %d", operand);
		j += 4;
		break;

		case OPERAND_UINT1:
       		operand = TclGetUInt1AtPtr(code_ptr + j);
		oappendf(ca->code, " %u", operand);
		j += 1;
		break;

		case OPERAND_UINT4:
       		operand = TclGetUInt4AtPtr(code_ptr + j);
		oappendf(ca->code, " %u", operand);
		j += 4;
		break;

		default:
		oappendf(ca->code, " (unknown operand type %d)", 
			 idesc->opTypes[i]);
	    }

	    oprintf(opvar, "op%d", i+1);
	    Tcl_ObjSetVar2(ca->interp, opvar, 0, Tcl_NewIntObj(operand),
			   TCL_GLOBAL_ONLY);
	    Tcl_DecrRefCount(opvar);

	    if(i == 0) {
		Tcl_Obj *target = Tcl_NewObj();
		Tcl_IncrRefCount(target);
		oprintf(target, "instruction_%04d", offset + operand);

		Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("jtarget", -1),
			       0, target, TCL_GLOBAL_ONLY);

		Tcl_DecrRefCount(target);
	    }
	}

	oappendf(ca->code, " */ \n");

	sprintf(cmd, SUBST("$%s"), idesc->name);
	result = Tcl_Eval(ca->interp, cmd);
	if(result) return TransferError(ca, result);

	Tcl_AppendObjToObj(ca->code, Tcl_GetObjResult(ca->interp));
 	code_ptr += idesc->numBytes;
    }

#ifndef NDEBUG
    fprintf(stderr, "Emitted %d instructions\n", count);
#endif
    
    return TCL_OK;
}


static int AddEpilogue(Kanga_CodeAccum *ca)
{
    int result;

    result = Tcl_Eval(ca->interp, SUBST("$__epilogue"));
    if(result) return TransferError(ca, result);

    Tcl_AppendObjToObj(ca->code, Tcl_GetObjResult(ca->interp));
    Tcl_AppendToObj(ca->code, "}\n", 2);
    return result;
}


static int AddDeclaration(Kanga_CodeAccum *ca)
{
    int result;

    result = AddCompiledLocalInitialiser(ca);
    if(result) return result;

    result = Tcl_Eval(ca->interp, SUBST("$__declaration"));
    if(result) return TransferError(ca, result);
    
    Tcl_AppendToObj(ca->header, "    {\n", -1);
    Tcl_AppendObjToObj(ca->header, Tcl_GetObjResult(ca->interp));
    AddLiteralCreationCode(ca);
    Tcl_AppendToObj(ca->header, "    }\n", -1);

    return TCL_OK;
}


static int AddLiteralCreationCode(Kanga_CodeAccum *ca)
{
    int      i;
    int      result;
    char     cmd[256];
    Tcl_Obj *createVar = Tcl_NewStringObj("createLiteral", -1);
    Tcl_IncrRefCount(createVar);

    for(i = 0; i < ca->bc->numLitObjects; ++i) {
	int     ival;
	double  dval;
	Tcl_Obj *creationCmd = Tcl_NewObj();
	Tcl_IncrRefCount(creationCmd);

	/* Try to convert this to a number */
	if(Tcl_GetIntFromObj(0, ca->bc->objArrayPtr[i], &ival) == 0)
	    oprintf(creationCmd, "Tcl_NewIntObj(%d)", ival);
	else if(Tcl_GetDoubleFromObj(0, ca->bc->objArrayPtr[i], &dval) == 0)
	    oprintf(creationCmd, "Tcl_NewDoubleObj(%.14lf)", dval);
	else {
	    char *origString;
	    int   origStringSize;
	    char *escString;
	    int   escStringSize;

	    origString = Tcl_GetString(ca->bc->objArrayPtr[i]);
	    origStringSize = Tcl_GetCharLength(ca->bc->objArrayPtr[i]);

	    escStringSize = GetEscapeCStringSize(origString, origStringSize);
	    escString = (char *)ckalloc(escStringSize + 1);
	    
	    EscapeCString(escString, origString, origStringSize);

	    oprintf(creationCmd, "Tcl_NewStringObj(\"%s\", %d)",
		    escString, escStringSize);

	    ckfree(escString);
	}

	Tcl_ObjSetVar2(ca->interp, createVar, Tcl_NewIntObj(i),
		       creationCmd, TCL_GLOBAL_ONLY);
	Tcl_DecrRefCount(creationCmd);

	Tcl_ObjSetVar2(ca->interp, Tcl_NewStringObj("literalIndex", -1), 0,
		       Tcl_NewIntObj(i), TCL_GLOBAL_ONLY);

	result = Tcl_Eval(ca->interp, SUBST("$__literal_create"));
	if (result) break;

	Tcl_AppendObjToObj(ca->header, Tcl_GetObjResult(ca->interp));
    }

    Tcl_DecrRefCount(createVar);

    if (result) TransferError(ca, result);
    return result;
}


static int AddCompiledLocalInitialiser(Kanga_CodeAccum *ca)
{
    int result;
    int i;
    CompiledLocal *cl;
    Tcl_Obj *clIndex = Tcl_NewStringObj("clIndex", -1);
    Tcl_Obj *clName = Tcl_NewStringObj("clName", -1);
    Tcl_Obj *clNameLength = Tcl_NewStringObj("clNameLength", -1);
    Tcl_Obj *clFlags = Tcl_NewStringObj("clFlags", -1);
    Tcl_Obj *clcode = Tcl_NewObj();
    Tcl_Obj *clinit = Tcl_NewObj();

    Tcl_IncrRefCount(clIndex);
    Tcl_IncrRefCount(clName);
    Tcl_IncrRefCount(clNameLength);
    Tcl_IncrRefCount(clFlags);
    Tcl_IncrRefCount(clcode);
    Tcl_IncrRefCount(clinit);

    cl = ca->procPtr->firstLocalPtr;
    for(i = 0; i < ca->procPtr->numCompiledLocals; i++) {
	Tcl_ObjSetVar2(ca->interp, clName, 0,
		       Tcl_NewStringObj(cl->name, cl->nameLength),
		       TCL_GLOBAL_ONLY);
	Tcl_ObjSetVar2(ca->interp, clNameLength, 0,
		       Tcl_NewIntObj(cl->nameLength), TCL_GLOBAL_ONLY);
	Tcl_ObjSetVar2(ca->interp, clFlags, 0,
		       Tcl_NewIntObj(cl->flags), TCL_GLOBAL_ONLY);
	Tcl_ObjSetVar2(ca->interp, clIndex, 0,
		       Tcl_NewIntObj(i), TCL_GLOBAL_ONLY);

	result = Tcl_Eval(ca->interp, SUBST("$__clallocate"));
	if(result) return TransferError(ca, result);

	Tcl_AppendObjToObj(clcode, Tcl_GetObjResult(ca->interp));

	result = Tcl_Eval(ca->interp, SUBST("$__clinitialise"));
	if(result) return TransferError(ca, result);
	Tcl_AppendObjToObj(clinit, Tcl_GetObjResult(ca->interp));

	cl = cl->nextPtr;
    }

    Tcl_AppendObjToObj(clcode, clinit);

    Tcl_ObjSetVar2(ca->interp, 
		   Tcl_NewStringObj("compiledLocalInitialisation", -1), 0,
		   clcode, TCL_GLOBAL_ONLY);

    Tcl_DecrRefCount(clIndex);
    Tcl_DecrRefCount(clName);
    Tcl_DecrRefCount(clNameLength);
    Tcl_DecrRefCount(clFlags);
    Tcl_DecrRefCount(clcode);
    Tcl_DecrRefCount(clinit);
    
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * GetExceptRangeForPc --
 *
 *	Given a program counter value, return the closest enclosing
 *	ExceptionRange.
 *
 * Results:
 *	In the normal case, catchOnly is 0 (false) and this procedure
 *	returns a pointer to the most closely enclosing ExceptionRange
 *	structure regardless of whether it is a loop or catch exception
 *	range. This is appropriate when processing a TCL_BREAK or
 *	TCL_CONTINUE, which will be "handled" either by a loop exception
 *	range or a closer catch range. If catchOnly is nonzero, this
 *	procedure ignores loop exception ranges and returns a pointer to the
 *	closest catch range. If no matching ExceptionRange is found that
 *	encloses pc, a NULL is returned.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static ExceptionRange *
GetExceptRangeForPc(pc, catchOnly, codePtr)
    unsigned char *pc;		/* The program counter value for which to
				 * search for a closest enclosing exception
				 * range. This points to a bytecode
				 * instruction in codePtr's code. */
    int catchOnly;		/* If 0, consider either loop or catch
				 * ExceptionRanges in search. If nonzero
				 * consider only catch ranges (and ignore
				 * any closer loop ranges). */
    ByteCode* codePtr;		/* Points to the ByteCode in which to search
				 * for the enclosing ExceptionRange. */
{
    ExceptionRange *rangeArrayPtr;
    int numRanges = codePtr->numExceptRanges;
    register ExceptionRange *rangePtr;
    int pcOffset = (pc - codePtr->codeStart);
    register int i, level;

    if (numRanges == 0) {
	return NULL;
    }
    rangeArrayPtr = codePtr->exceptArrayPtr;

    for (level = codePtr->maxExceptDepth;  level >= 0;  level--) {
	for (i = 0;  i < numRanges;  i++) {
	    rangePtr = &(rangeArrayPtr[i]);
	    if (rangePtr->nestingLevel == level) {
		int start = rangePtr->codeOffset;
		int end   = (start + rangePtr->numCodeBytes);
		if ((start <= pcOffset) && (pcOffset < end)) {
		    if ((!catchOnly)
			    || (rangePtr->type == CATCH_EXCEPTION_RANGE)) {
			return rangePtr;
		    }
		}
	    }
	}
    }
    return NULL;
}


#if defined(_WIN32) || defined(__WIN32__)
int Kanga_CpuTime(ClientData dummy,      /* Not used. */
		  Tcl_Interp *interp,    /* Current interpreter. */
		  int objc,              /* Number of arguments. */
		  Tcl_Obj *const objv[]) /* Argument objects. */
{
    register Tcl_Obj *objPtr;
    register int i, result;
    int count;
    double totalMicroSec;
    char buf[100];
    __int64 ignored1, ignored2, ktime_begin, ktime_end, utime_begin, utime_end;
    OSVERSIONINFO ovi;

    if (objc == 2) {
	count = 1;
    } else if (objc == 3) {
	result = Tcl_GetIntFromObj(interp, objv[2], &count);
	if (result != TCL_OK) {
	    return result;
	}
    } else {
	Tcl_WrongNumArgs(interp, 1, objv, "command ?count?");
	return TCL_ERROR;
    }

    ovi.dwOSVersionInfoSize = sizeof(ovi);
    GetVersionEx(&ovi);
    if(ovi.dwPlatformId != VER_PLATFORM_WIN32_NT) {
	Tcl_ResetResult(interp);
	oprintf(Tcl_GetObjResult(interp), 
		"cputime not supported on non-NT systems");
	return TCL_ERROR;
    }
    
    objPtr = objv[1];
    i = count;
    
    /* Not exactly ethical, but it works */
    GetProcessTimes(GetCurrentProcess(), (FILETIME *)&ignored1, 
		    (FILETIME *)&ignored2, (FILETIME *)&ktime_begin,
		    (FILETIME *)&utime_begin);

    while (i-- > 0) {
	result = Tcl_EvalObjEx(interp, objPtr, 0);
	if (result != TCL_OK) {
	    return result;
	}
    }

    GetProcessTimes(GetCurrentProcess(), (FILETIME *)&ignored1, 
		    (FILETIME *)&ignored2, (FILETIME *)&ktime_end,
		    (FILETIME *)&utime_end);
    
    totalMicroSec = 
	((ktime_end - ktime_begin) + (utime_end - utime_begin)) * 0.1;

    sprintf(buf, "%.0f microseconds per iteration",
	((count <= 0) ? 0 : totalMicroSec/count));
    Tcl_ResetResult(interp);
    Tcl_AppendToObj(Tcl_GetObjResult(interp), buf, -1);
    return TCL_OK;

}
#else
int Kanga_CpuTime(ClientData dummy,      /* Not used. */
		  Tcl_Interp *interp,    /* Current interpreter. */
		  int objc,              /* Number of arguments. */
		  Tcl_Obj *const objv[]) /* Argument objects. */
{
    register Tcl_Obj *objPtr;
    register int i, result;
    int count;
    double totalMicroSec;
    char buf[100];
    struct tms start, stop;

    if (objc == 2) {
	count = 1;
    } else if (objc == 3) {
	result = Tcl_GetIntFromObj(interp, objv[2], &count);
	if (result != TCL_OK) {
	    return result;
	}
    } else {
	Tcl_WrongNumArgs(interp, 1, objv, "command ?count?");
	return TCL_ERROR;
    }
    
    objPtr = objv[1];
    i = count;
    
    times(&start);

    while (i-- > 0) {
	result = Tcl_EvalObjEx(interp, objPtr, 0);
	if (result != TCL_OK) {
	    return result;
	}
    }

    times(&stop);
    
    totalMicroSec =
	((stop.tms_utime - start.tms_utime) +
	 (stop.tms_stime - start.tms_stime) +
	 (stop.tms_cutime - start.tms_cutime) +
	 (stop.tms_cstime - start.tms_cstime)) * 1000000 / CLK_TCK;

    sprintf(buf, "%.0f microseconds per iteration",
	((count <= 0) ? 0 : totalMicroSec/count));
    Tcl_ResetResult(interp);
    Tcl_AppendToObj(Tcl_GetObjResult(interp), buf, -1);
    return TCL_OK;

}
#endif

