#!/bin/sh
# The next line is executed by /bin/sh, but not tcl \
        exec wish8.4 "$0" ${1+"$@"}

#############################################################################
#############################################################################
## This script implements only drop targets, that utilise our cross-platform
## formats Text, Files, Image. This is a simple demo showing the ability to
## understand a wide range of Windows platform specific formats...
##
## Georgios Petasis, 03/01/2000
#############################################################################
#############################################################################

## Check Tk version:
package require Tk 8.3

if {$::tcl_version == "8.3" && ![package vsatisfies $::tcl_patchLevel 8.3.3]} {
    tk_messageBox -type ok -icon error \
            -message "  =====> TkDND requires at least tk8.3.3! <====="
    exit 1
}

if {[string equal $tcl_platform(platform) unix]} {
    option add *font {arial 12}
    option add *borderWidth 1
}

## Load tkdnd package...
set DIR [file dirname [file normalize [info script]]]
## Make sure that we can find the tkdnd package even if the user has not yet
## installed the package.
lappend auto_path [file dirname $DIR] [file dirname $DIR]/lib
package require tkdnd

##
## Create our main window...
##
wm overrideredirect . 1
wm geometry . 1x1-10-10
set win .demo_window
toplevel $win
wm title $win "Cross-Platform Drag and Drop Demo..."
wm protocol $win WM_DELETE_WINDOW exit
bind $win <Destroy> {exit}

## Add drag sources...
set dwin $win.drag_sources
frame $dwin
label $dwin.text  -bg red -fg white -relief raised -bd 1 -text Text

label $dwin.files  -bg green -fg black -relief raised -bd 1 -text Files

label $dwin.images -bg green -fg black -relief raised -bd 1 -text Image
## Select the image to be transfered...
radiobutton $dwin.i1 -variable tImage -value itk
radiobutton $dwin.i2 -variable tImage -value iteapot
set tImage itk
## Select the canvas background...
button $dwin.white -text {    } -bg white   -bd 1 -command \
  "$win.frame.dropped_data configure -bg white"
button $dwin.green -text {    } -bg #8fbc8f -bd 1 -command \
  "$win.frame.dropped_data configure -bg #8fbc8f"
button $dwin.black -text {    } -bg black   -bd 1 -command \
  "$win.frame.dropped_data configure -bg black"

grid $dwin.text     -padx 2 -pady 2 -sticky snew
grid $dwin.files    -padx 2 -pady 2 -sticky snew
grid $dwin.images   -padx 2 -pady 2 -sticky snew
grid $dwin.i1       -padx 2 -pady 2 -sticky snew
grid $dwin.i2       -padx 2 -pady 2 -sticky snew
grid $dwin.white    -padx 2 -pady 2 -sticky snew
grid $dwin.green    -padx 2 -pady 2 -sticky snew
grid $dwin.black    -padx 2 -pady 2 -sticky snew

## Add drop targets...
label  $win.text  -height 3 -bg red    -text {Drop any textual format here!}
label  $win.files -height 3 -bg green  -text {Drop any file(s) here!}
label  $win.image -height 3 -bg yellow -text {Drop any image format here!}
frame $win.frame
canvas $win.frame.dropped_data -bg white -relief sunken -bd 2 \
  -xscrollcommand "$win.frame.x set" -yscrollcommand "$win.frame.y set"
scrollbar $win.frame.x -orient horizontal \
  -command "$win.frame.dropped_data xview"
scrollbar $win.frame.y -orient vertical \
  -command "$win.frame.dropped_data yview"
$win.frame.dropped_data create text  150 150 -tags {text}  -width 500 -justify left
$win.frame.dropped_data create image 150 150 -tags {image} -anchor n
grid $win.frame.dropped_data $win.frame.y -sticky snew
grid $win.frame.x -sticky snew
grid columnconfigure $win.frame 0 -weight 1
grid rowconfigure    $win.frame 0 -weight 1

## Pack widgets...
grid $win.drag_sources -row 0 -column 0 -sticky new  -padx 2 -pady 4 -rowspan 4
grid $win.text         -row 0 -column 1 -sticky snew -padx 2 -pady 4
grid $win.files        -row 1 -column 1 -sticky snew -padx 2 -pady 4
grid $win.image        -row 2 -column 1 -sticky snew -padx 2 -pady 4
grid $win.frame        -row 3 -column 1 -sticky snew -padx 2 -pady 4
grid rowconfigure    $win 3 -weight 1
grid columnconfigure $win 0 -weight 0
grid columnconfigure $win 1 -weight 1

## Register our drop targets. Do not forget that widgets must have been
## created in order to be drag&drop aware :-)
update idletasks
dnd bindtarget $win.text  Text  <Drop> {HandleTextDrop  %D %T}
dnd bindtarget $win.files Files <Drop> {HandleFileDrop  %D %T}
dnd bindtarget $win.image Image <Drop> {HandleImageDrop %D %T}

## Register our drag sources...
dnd bindsource $dwin.text     Text \
        {return "Greek Text: \u03b1\u03b2\u03ac\u03c1\u03b5\u03c4\u03bf\u03c2"}
dnd bindsource $dwin.files    Files       {return [pwd]}
dnd bindsource $dwin.images   Image       {return $::tImage}
foreach widget {text files images} {
    bind $dwin.$widget <1> {dnd drag %W}
}

proc HandleTextDrop {data type} {
    global win
    
    $win.frame.dropped_data itemconfigure image -image {}
    $win.frame.dropped_data itemconfigure text  -text $data
    $win.frame.dropped_data configure -scrollregion [$win.frame.dropped_data bbox all]
}

proc HandleFileDrop {data type} {
    global win
    
    $win.frame.dropped_data itemconfigure image -image {}
    $win.frame.dropped_data itemconfigure text  -text $data
    $win.frame.dropped_data configure -scrollregion [$win.frame.dropped_data bbox all]
}

proc HandleImageDrop {data type} {
    global win

    ## Destroy any previous image...
    catch {
      set image [$win.frame.dropped_data itemcget image -image]
      $win.frame.dropped_data itemconfigure image -image {}
      if {![image inuse $image]} {image delete $image}
    }
    ## Is the data an image name?
    if {[catch {image inuse $data}]} {
      $win.frame.dropped_data itemconfigure text  -text $data
    } else {
      $win.frame.dropped_data itemconfigure image -image $data
      $win.frame.dropped_data itemconfigure text  -text {}
    }
    $win.frame.dropped_data configure -scrollregion [$win.frame.dropped_data bbox all]
}

#############################################################################
## What follows is a sample tk image (tk logo :-))

image create photo itk -data {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===}

image create photo iteapot -data {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}
$dwin.i1 configure -image itk
$dwin.i2 configure -image iteapot


