#include <tkgs.h>
#include <tkgsInt.h>

#include "tkgsPS.h"

#include <string.h>

/*
 * Prototypes for procedures defined later in this file:
 */

/* Driver */

static TkGS_GetDrawableProc		PSTkGS_GetDrawable;
static TkGSUpdateDrawableStateProc	PSTkGSUpdateDrawableState;

static TkGS_DrawRectangleProc		PSTkGS_DrawRectangle;
static TkGS_DrawRectanglesProc		PSTkGS_DrawRectangles;
static TkGS_FillRectangleProc		PSTkGS_FillRectangle;
static TkGS_FillRectanglesProc		PSTkGS_FillRectangles;
static TkGS_DrawEllipseProc		PSTkGS_DrawEllipse;
static TkGS_DrawEllipsesProc		PSTkGS_DrawEllipses;

static TkGS_EnumerateFontFamiliesProc	PSTkGS_EnumerateFontFamilies;

/* PSDrawable */

static void		PSDrawableFreeIntRep _ANSI_ARGS_((TkGS_InternalRep *intRepPtr));

/*
 * Implementation note:
 *
 * PostScript origin is at the bottom left of the page, and y axis is 
 * oriented to the top of the page.
 * TkGS origin is at the top left of the page, and y axis to the bottom
 * of the page (as with most display models). 
 * This driver defines the origin to be at the top left of the drawing
 * area, however we cannot reverse the y axis since this flips the text
 * upside down. So every drawing primitives must negate the y coordinates
 * they use, and possibly the angular values. This eases coordinates
 * translations a lot. Tk PostScript commands has to substract the y 
 * coordinate from the total height of the area, which makes the code a 
 * bit harder to read.
 */

#include "prolog.h"

/*
 * Driver
 */

TkGS_DeviceDriver PSDeviceDriver = {
    "PostScript",

    PSTkGS_GetDrawable,

    PSTkGSUpdateDrawableState,

    /* Line & shape primitives */
    PSTkGS_DrawRectangle,
    PSTkGS_DrawRectangles,
    PSTkGS_FillRectangle,
    PSTkGS_FillRectangles,
    PSTkGS_DrawEllipse,
    PSTkGS_DrawEllipses,

    /* Fonts and text primitives: */
    NULL,
    NULL,
    PSTkGS_EnumerateFontFamilies,
    NULL,

    /*  - Unicode */
    NULL,
    NULL,
    NULL,
    NULL,

    /*  - UTF-8 */
    NULL,
    NULL,
    NULL,
    NULL,

    /*  - System-specific */
    NULL,
    NULL,
    NULL,
    NULL,

    /*  - Multi-font Unicode system */
    0,
    0,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
};



/*
 * PSDrawable
 */

TkGS_ObjType PSDrawableType = {
    "PostScriptDrawable",
    NULL,				/* Base type, initialized at runtime */

    PSDrawableFreeIntRep,		/* freeIntRepProc */
    NULL				/* setFromAnyProc */
					/* NULL means immutable */
};

/* Accessors */
#define PSDrawable_AppendResultProc(intRepPtr) \
    ((PSDrawableAppendResultProc*) intRepPtr->value.twoValue.value1.otherValuePtr)
#define PSDrawable_ClientData(intRepPtr) \
    ((ClientData) intRepPtr->value.twoValue.value2.longValue)

#define PSDrawable_Append(intRepPtr,string,length) \
    (PSDrawable_AppendResultProc(intRepPtr))( \
	PSDrawable_ClientData(intRepPtr), (string), (length))



/* Object type procs */

static void
PSDrawableFreeIntRep(intRepPtr)
    TkGS_InternalRep *intRepPtr;
{
    char *string;
    int length;

    /*
     *---------------------------------------------------------------------
     * Output page-end information, such as commands to print the page
     * and document trailer stuff.
     *---------------------------------------------------------------------
     */

    string = "restore showpage\n\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    string = "%%Trailer\nend\n%%EOF\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
}






int
PSTkGSUpdateDrawableState(d, valueMask)
    TkGS_Drawable d;
    unsigned long valueMask;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    register unsigned long *gcValueMask = &TkGSDrawable_GCValueMask(d);
    register TkGS_InternalRep 
	*intRepPtr = TkGS_FindInternalRep((TkGS_Obj *) (d), &PSDrawableType);

    char string[200];
    int length;

    /*
     * Update GC state if needed
     */

    valueMask &= *gcValueMask;
    if (valueMask == 0) {
	return TKGS_OK;
    }
    
    /* foreground */
    if (valueMask & TkGS_GCForeground) {
	/* Code ripped off tkCanvPs.c */

	/*
	 * No color map entry for this color.  Grab the color's intensities
	 * and output Postscript commands for them.  Special note:  X uses
	 * a range of 0-65535 for intensities, but most displays only use
	 * a range of 0-255, which maps to (0, 256, 512, ... 65280) in the
	 * X scale.  This means that there's no way to get perfect white,
	 * since the highest intensity is only 65280 out of 65535.  To
	 * work around this problem, rescale the X intensity to a 0-255
	 * scale and use that as the basis for the Postscript colors.  This
	 * scheme still won't work if the display only uses 4 bits per color,
	 * but most diplays use at least 8 bits.
	 */

	unsigned short r, g, b;
	double red, green, blue;

	TkGS_GetRGBColorValues(gcValues->foreground, &r, &g, &b);

	red   = ((double) (r >> 8))/255.0;
	green = ((double) (g >> 8))/255.0;
	blue  = ((double) (b >> 8))/255.0;

	sprintf(string, "%.3f %.3f %.3f setrgbcolor AdjustColor\n",
		red, green, blue);
	length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
    }

    /* background */
    if (valueMask & TkGS_GCBackground) {
	/* TODO: what? */
	/* Background will become useful when we allow stippling or text */
    }

    /* lineWidth */
    if (valueMask & TkGS_GCLineWidth) {
	sprintf(string, "%d setlinewidth\n",
		gcValues->lineWidth);
	length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
    }

    /* font */
    if (valueMask & TkGS_GCFont) {
	/* TODO */
    }

    *gcValueMask = 0;
    return TKGS_OK;
}




/*
 * Drawable-related driver procs
 */

TkGS_Drawable
PSTkGS_GetDrawable(clientData)
    ClientData clientData;
{
    PSDrawableCreateData *cd = (PSDrawableCreateData *) clientData;
    TkGS_Drawable d = TkGSNewDrawable(&PSDeviceDriver);
    TkGS_InternalRep intRep;
    register TkGS_InternalRep *intRepPtr = &intRep;
    char *string, buf[256];
    int length;
    CONST char * CONST *chunk;

    /* 
     * Initialize internal rep
     */

    PSDrawable_AppendResultProc(intRepPtr) = cd->appendResultProc;
    PSDrawable_ClientData(intRepPtr) = cd->clientData;

    /* Code mostly ripped off tkCanvPs.c */

    /*
     *--------------------------------------------------------
     * Generate the header and prolog for the Postscript.
     *--------------------------------------------------------
     */

    string = "%!PS-Adobe-3.0 EPSF-3.0\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    string = "%%Creator: TkGS PostScript driver\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    /* TODO: bounding box, creation date, etc. */

    string = "%%Pages: 1\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
    string = "%%DocumentData: Clean7Bit\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
    string = "%%Orientation: Portrait\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    string = "%%EndComments\n\n";length = strlen(string);
    PSDrawable_Append(intRepPtr,string,length);

    /*
     * Insert the prolog
     */
    for (chunk=prolog; *chunk; chunk++) {
	length = strlen(*chunk); PSDrawable_Append(intRepPtr,(char *) *chunk,length);
    }

    /*
     *-----------------------------------------------------------
     * Document setup:  set the color level and include fonts.
     *-----------------------------------------------------------
     */

    string = "%%BeginSetup\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    /* TODO: manage color level. Default to 2 (ie full color) */

    string = "/CL 2 def\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    string = "%%EndSetup\n\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    /*
     *-----------------------------------------------------------
     * Page setup:  move to page positioning point, rotate if
     * needed, set scale factor, offset for proper anchor position,
     * and set clip region.
     *-----------------------------------------------------------
     */

    string = "%%Page: 1 1\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    string = "save\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    /* TODO: handle pagex, pagey, rotation, etc. */

    /* Go to center of page */
    sprintf(buf, "%.1f %.1f translate\n", 72*4.25, 72*5.5);
    length = strlen(buf); PSDrawable_Append(intRepPtr,buf,length);

    /* Set the pixel width */
    sprintf(buf, "%.4g %.4g scale\n", cd->pixelWidth, cd->pixelHeight);
    length = strlen(buf); PSDrawable_Append(intRepPtr,buf,length);

    /* Center the display box */
    sprintf(buf, "%.4g %.4g translate\n", -(double)cd->width/2.0, (double)cd->height/2.0);
    length = strlen(buf); PSDrawable_Append(intRepPtr,buf,length);

    /* Set clipping rectangle */
    sprintf(buf, "0 0 moveto %.15g 0 rlineto 0 %.15g rlineto %.15g 0 rlineto closepath clip newpath\n",
    		(double)cd->width, -(double)cd->height, - (double)cd->width);
    length = strlen(buf); PSDrawable_Append(intRepPtr,buf,length);

    intRepPtr->typePtr = &PSDrawableType;

    /*
     * Add internal rep to object
     */

    TkGS_PushInternalRep((TkGS_Obj *) d, intRepPtr);

    return d;
}



/*
 * Drawing primitives
 */

#define DECLAREDRAWINGVARS(d, intRepPtr, gcValues) \
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d); \
    register TkGS_InternalRep *intRepPtr = TkGS_FindInternalRep((TkGS_Obj *) (d), &PSDrawableType);

 
void
PSTkGS_DrawRectangle(d, filled, x, y, width, height)
    TkGS_Drawable	d;
    int			filled;
    int			x;
    int			y;
    unsigned int	width;
    unsigned int	height;
{
    DECLAREDRAWINGVARS(d, intRepPtr, gcValues)
    char *string, buf[500];
    int length;

    string = "gsave\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    sprintf(buf, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto %.15g 0 rlineto closepath\n",
    		(double)x, -(double)y, 
		(double)width, -(double)height, - (double)width);
    length = strlen(buf); PSDrawable_Append(intRepPtr,buf,length);

    if (filled) {
	string = "fill\n";
	length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
    } else {
	string = "stroke\n";
	length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
    }

    string = "grestore\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
}

void
PSTkGS_DrawRectangles(d, filled, rectangles, nbRectangles) 
    TkGS_Drawable	d;
    int			filled;
    TkGS_Rectangle*	rectangles;
    int			nbRectangles;
{
    int i;
    for (i=0; i<nbRectangles; i++) {
	PSTkGS_DrawRectangle(d, filled, rectangles[i].x, rectangles[i].y,
			     rectangles[i].width, rectangles[i].height);
    }
}

void
PSTkGS_FillRectangle(d, color, x, y, width, height)
    TkGS_Drawable	d;
    TkGS_Color		color;
    int			x;
    int			y;
    unsigned int	width;
    unsigned int	height;
{
    DECLAREDRAWINGVARS(d, intRepPtr, gcValues)
    char *string, buf[500];
    int length;
    unsigned short r, g, b;
    double red, green, blue;

    TkGS_GetRGBColorValues(color, &r, &g, &b);

    string = "gsave\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    red   = ((double) (r >> 8))/255.0;
    green = ((double) (g >> 8))/255.0;
    blue  = ((double) (b >> 8))/255.0;

    sprintf(buf, "%.3f %.3f %.3f setrgbcolor AdjustColor\n",
	    red, green, blue);
    length = strlen(buf); PSDrawable_Append(intRepPtr,buf,length);

    sprintf(buf, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto %.15g 0 rlineto closepath\n",
    		(double)x, -(double)y, 
		(double)width, -(double)height, - (double)width);
    length = strlen(buf); PSDrawable_Append(intRepPtr,buf,length);

    string = "fill\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    string = "grestore\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
}

void
PSTkGS_FillRectangles(d, color, rectangles, nbRectangles) 
    TkGS_Drawable	d;
    TkGS_Color		color;
    TkGS_Rectangle*	rectangles;
    int			nbRectangles;
{
    int i;
    for (i=0; i<nbRectangles; i++) {
	PSTkGS_FillRectangle(d, color, rectangles[i].x, rectangles[i].y,
				 rectangles[i].width, rectangles[i].height);
    }
}

void
PSTkGS_DrawEllipse(d, filled, x, y, width, height)
    TkGS_Drawable	d;
    int			filled;
    int			x;
    int			y;
    unsigned int	width;
    unsigned int	height;
{
    DECLAREDRAWINGVARS(d, intRepPtr, gcValues)
    char *string, buf[500];
    int length;

    string = "gsave\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);

    sprintf(buf, "matrix currentmatrix\n%.15g %.15g translate %.15g %.15g scale 1 0 moveto 0 0 1 0 360 arc\nsetmatrix\n",
		(double)x+((double)width/2.0), -(double)y-((double)height/2.0),
		(double)width/2.0, -(double)height/2.0);
    length = strlen(buf); PSDrawable_Append(intRepPtr,buf,length);

    if (filled) {
	string = "fill\n";
	length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
    } else {
	string = "stroke\n";
	length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
    }

    string = "grestore\n";
    length = strlen(string); PSDrawable_Append(intRepPtr,string,length);
}

void
PSTkGS_DrawEllipses(d, filled, ellipses, nbEllipses) 
    TkGS_Drawable	d;
    int			filled;
    TkGS_Rectangle*	ellipses;
    int			nbEllipses;
{
    int i;
    for (i=0; i<nbEllipses; i++) {
	PSTkGS_DrawEllipse(d, filled, ellipses[i].x, ellipses[i].y,
			   ellipses[i].width, ellipses[i].height);
    }
}


void
PSTkGS_EnumerateFontFamilies(d, name, enumProc, clientData)
    TkGS_Drawable		d;
    CONST char			*name;
    TkGS_FontFamiliesEnumProc	*enumProc;
    ClientData			clientData;
{
    /* FIXME */
    static char *families[] = {
	"Helvetica", "Times", "Courier", "Symbol", "AvantGarde", "Bookman",
	"NewCenturySchlbk", "Palatino", "ZapfChancery", "ZapfDingbats",

	NULL
    };
    char **familyPtr;
    int result;

    /* TODO: take name into account. */
    for (familyPtr = families; *familyPtr != NULL; familyPtr++) {
	result = (enumProc)(d, clientData, *familyPtr);
	if (!result) {
	    break;
	}
    }
}





/*
 * Initialization code
 */

static void
PSTkGSInit()
{
    PSDrawableType.baseTypePtr = TkGS_GetDrawableBaseType();

    TkGS_RegisterDeviceDriver(&PSDeviceDriver);
}

void
PSTkGS_StaticPackage(interp)
    Tcl_Interp *interp;
{  
    Tcl_StaticPackage(interp, "Pstkgs", Pstkgs_Init, Pstkgs_SafeInit);
}


/*
 *----------------------------------------------------------------------
 *
 * Pstkgs_Init --
 *
 *	This function is used to initialize the package. Here we check
 *	dependencies with other packages (eg Tcl), create new commands,
 *	and register our packages
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	New commands created, and package registered (see documentation).
 *
 *----------------------------------------------------------------------
 */

int Pstkgs_Init(interp)
    Tcl_Interp *interp;
{  
#ifdef USE_TCL_STUBS
    if (Tcl_InitStubs(interp, "8.3", 0) == NULL) {
	return TCL_ERROR;
    }
#endif

#ifdef USE_TK_STUBS
    if (Tk_InitStubs(interp, "8.3", 0) == NULL) {
	return TCL_ERROR;
    }
#endif

    /* Dependencies */
    if (Tcl_PkgRequire(interp, "TkGS", "1.0", 0) == NULL)
	return TCL_ERROR;

    /* Package provided */
    if (Tcl_PkgProvide(interp, "PSTkGS", "1.0") == TCL_ERROR)
	return TCL_ERROR;

    PSTkGSInit();

    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Pstkgs_SafeInit --
 *
 *	This function is used to initialize the package in the special case
 *	of safe interpreters (like the Tcl Plugin). Here we check
 *	dependencies with other packages (eg Tcl), create new commands,
 *	and register our packages
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	New commands created, and package registered (see documentation).
 *
 *----------------------------------------------------------------------
 */

int Pstkgs_SafeInit(interp)
    Tcl_Interp *interp;
{  
    /* For now, do the same as with unsafe interpreters */
    return Pstkgs_Init(interp);
}

