/* -*- C -*-
 *
 * Program:	ximap
 * File:        mime.c -- Fns for getting type information about body parts
 *                        when composing a message.  The name is deceptive.  It
 *                        will eventually have all of the MIME specific ximap
 *                        fns.
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <stdio.h>
#include <string.h>

#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>
#include <Client/rfc822.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Box.h>

#include <Xb/BSBMnBt.h>
#include <Xb/Browser.h>
#include <Xb/CascadeMenu.h>

#include "globals.h"
#include "structures.h"
#include "mailcap.h"
#include "util.h"
#include "mime.h"
#include "buttons.h"
#include "ximap.h"

static void setType();
static void setSubtype();

char *tis_typeNames[] =
{
    "text",
    "multipart",
    "message",
    "application",
    "audio",
    "image",
    "video",
    "other",
    NULL
};

ButtonStruct typeMenuEntries[] =
{
    {"text",        setType,    NULL, NEVER},
    {"image",       setType,    NULL, NEVER},
    {"audio",       setType,    NULL, NEVER},
    {"application", setType,    NULL, NEVER},
    NULL
};

ButtonStruct textMenuEntries[] =
{
    {"Plain",    setSubtype, NULL, NEVER},
    {"Richtext", setSubtype, NULL, NEVER},
    {"OTHER",    setSubtype, NULL, NEVER},
    NULL
};

ButtonStruct imageMenuEntries[] =
{
    {"GIF",    setSubtype, NULL, NEVER},
    {"TIFF",   setSubtype, NULL, NEVER},
    {"PS",     setSubtype, NULL, NEVER},
    {"OTHER",  setSubtype, NULL, NEVER},
    NULL
};

ButtonStruct audioMenuEntries[] =
{
    {"sun-audio", setSubtype, NULL, NEVER},
    {"OTHER",     setSubtype, NULL, NEVER},
    NULL
};

ButtonStruct applicationMenuEntries[] =
{
    {"octet-stream",  setSubtype, NULL, NEVER},
    NULL
};

ButtonStruct *subtypeMenus[] =
{
    textMenuEntries,
    NULL,
    NULL,
    applicationMenuEntries,
    audioMenuEntries,
    imageMenuEntries,
    NULL,
    NULL
};

typedef struct ti_structure 
{
    int  type;
    char *typename;
    char *subtype;
    char *description;

    Widget shell;
    Widget form;
    Widget panes;
    Widget buttons;
    
    Widget typemenu;
    Widget subtypemenu;
    Widget *subMenus;

    int done;
} TIStructure;

static void initMenus();

static void createTIWindow();

static void tiOk();
static void tiCancel();

static int getTypeFromName(/* char* */);

ButtonStruct typeInfoButtons[] = 
{
    { "Ok",      tiOk,     NULL, NEVER },
    { "Cancel" , tiCancel, NULL, NEVER },
    NULL
};

Arg tiarg[ARGLISTSIZE];
int tin = 0;

/*
   This function gets MIME type information from the
   user and sets the fields of the body part, using the 
   the information acquired and the data in the BinaryBuffer.

   It returns 1 if it successfully got the info and set the body part,
   and 0 if it failed for some reason.  On failure, the calling
   program is expected to set the information.
*/
int getTypeInfo(w, body, bb, name)
     Widget w;
     BODY *body;
     BinaryBuffer *bb;
     char *name;
{
    int gotInfo = 0;
    TIStructure *tis = (TIStructure*) malloc (sizeof(TIStructure));

    tis->type = TYPEAPPLICATION;
    tis->typename = cpystr(tis_typeNames[tis->type]);
    tis->subtype = cpystr("octet-stream");
    tis->description = cpystr(name);

    tis->done = 0;

    /* Pop up a window and get some info... */
    createTIWindow(w, tis);

    /* This will handle events properly until tis->done is set */
    XimapMainLoop(&tis->done);

    /* If < 0 then it was 'cancel'ed */
    if(tis->done > 0)
    {
	gotInfo = 1;

	/* put info into body here */
	body->type = tis->type;
	if(tis->subtype && !strcmp(tis->subtype,"OTHER"))
	    body->subtype = cpystr(tis->subtype);

	switch(body->type)
	{
	  case TYPETEXT:
	    body->contents.text = (unsigned char*)fs_get(bb->bufferSize + 1);
	    bcopy(bb->bufferContents,body->contents.text,bb->bufferSize);
	    body->contents.text[bb->bufferSize] = '\0';
	    /* If the buffer is not just one zero terminated string...*/
	    if(strlen(bb->bufferContents) != bb->bufferSize)
	    {
		/* ...We don't allow any encoding but the default */
		gotInfo = 0;
	    }
	    else
	    {
		body->size.bytes = bb->bufferSize;
		body->encoding = ENC7BIT;
	    }
	    break;
	  case TYPEIMAGE:
	  case TYPEAUDIO:
	  case TYPEAPPLICATION:
	    body->contents.text = rfc822_binary (bb->bufferContents,
						 bb->bufferSize,
						 &body->size.bytes);
		    
	    body->encoding = ENCBASE64;
	    body->description = cpystr(tis->description);
	    break;
	  default:
	    gotInfo = 0;
	    break;
	}
    }

    XtPopdown(tis->shell);
    XtDestroyWidget(tis->shell);

    return(gotInfo);
}

/* 
   This function creates the window for setting the 
   MIME type information, and pops up the window.
*/
static void createTIWindow(w, tis)
     Widget w;
     TIStructure *tis;
{
    Widget type_label, subtype_label;

    XtSetArg(tiarg[tin], XtNallowShellResize, TRUE); tin++;
    tis->shell = XtCreatePopupShell("tis_shell",
				    topLevelShellWidgetClass,
				    w, tiarg, tin); tin = 0;

    tis->form = XtCreateManagedWidget("tis_form",
				      formWidgetClass,
				      tis->shell, tiarg, tin); tin = 0;

    XtSetArg(tiarg[tin], XtNborderWidth, 0); tin++;
    XtSetArg(tiarg[tin], XtNleft, XawChainLeft); tin++;
    XtSetArg(tiarg[tin], XtNright, XawChainLeft); tin++;
    XtSetArg(tiarg[tin], XtNresizable, FALSE); tin++;
    XtSetArg(tiarg[tin], XtNlabel, "Type:   "); tin++;
    type_label = XtCreateManagedWidget("type_label",
				       labelWidgetClass,
				       tis->form, tiarg, tin); tin = 0;
    
    XtSetArg(tiarg[tin], XtNfromHoriz, type_label); tin++;
    XtSetArg(tiarg[tin], XtNleft, XawChainLeft); tin++;
    XtSetArg(tiarg[tin], XtNright, XawChainRight); tin++;
    XtSetArg(tiarg[tin], XtNresizable, TRUE); tin++;
    XtSetArg(tiarg[tin], XtNlabel, tis->typename); tin++;
    tis->typemenu = XtCreateManagedWidget("type_menu",
					  bsbMnBtWidgetClass,
					  tis->form, tiarg, tin); tin = 0;

    makeMenu(tis->typemenu, "menu", tis, typeMenuEntries);
    
    XtSetArg(tiarg[tin], XtNborderWidth, 0); tin++;
    XtSetArg(tiarg[tin], XtNfromVert, tis->typemenu); tin++;
    XtSetArg(tiarg[tin], XtNleft, XawChainLeft); tin++;
    XtSetArg(tiarg[tin], XtNright, XawChainLeft); tin++;
    XtSetArg(tiarg[tin], XtNresizable, FALSE); tin++;
    XtSetArg(tiarg[tin], XtNlabel, "Subtype:"); tin++;
    subtype_label = XtCreateManagedWidget("subtype_label",
					  labelWidgetClass,
					  tis->form, tiarg, tin); tin = 0;

    XtSetArg(tiarg[tin], XtNfromHoriz, subtype_label); tin++;
    XtSetArg(tiarg[tin], XtNfromVert, tis->typemenu); tin++;
    XtSetArg(tiarg[tin], XtNleft, XawChainLeft); tin++;
    XtSetArg(tiarg[tin], XtNright, XawChainRight); tin++;
    XtSetArg(tiarg[tin], XtNresizable, TRUE); tin++;
    XtSetArg(tiarg[tin], XtNmenuName, tis->typename); tin++;
    XtSetArg(tiarg[tin], XtNlabel, tis->subtype); tin++;
    tis->subtypemenu = XtCreateManagedWidget("subtype_menu",
					     bsbMnBtWidgetClass,
					     tis->form, tiarg, tin); tin = 0;
    initMenus(tis);

    XtSetArg(tiarg[tin], XtNborderWidth, 0); tin++;
    XtSetArg(tiarg[tin], XtNfromVert, tis->subtypemenu); tin++;
    XtSetArg(tiarg[tin], XtNorientation, XtorientHorizontal); tin++;
    tis->buttons = XtCreateManagedWidget("tis_buttons",
					 boxWidgetClass,
					 tis->form, tiarg, tin); tin = 0;

    createButtons(tis->buttons,
		  tis,
		  typeInfoButtons);

    XtPopup(tis->shell, XtGrabNone);
}

/*
   Callback for clicking the 'Ok' button in the
   TypeInfo window.
*/
static void tiOk(w, tis, e)
     Widget w;
     TIStructure *tis;
     XEvent *e;
{
    /* Yes, we want to use the current info. */
    tis->done = 1;
}

/* 
   Callback for clicking the 'Cancel' button
   in the TypeInfo window.
*/
static void tiCancel(w, tis, e)
     Widget w;
     TIStructure *tis;
     XEvent *e;
{
    /* A negative return value means no info */
    tis->done = -1;
}

/* 
   Callback for selecting a menu entry on 
   the Type menu.
*/
static void setType(w, tis, e)
     Widget w;
     TIStructure *tis;
     XEvent *e;
{
    char *typename = NULL;
    Widget menu = NULL;

    /* Get label from menu entry           */
    XtSetArg(tiarg[tin], XtNlabel, &typename); tin++;
    XtGetValues(w, tiarg, tin); tin = 0;

    /* 
       We only do the reat of the stuff if
       a new type was selected.
    */
    if(strcmp(typename, tis->typename))
    {
	menu = XtNameToWidget(tis->typemenu, "menu");

	/* Set label of MenuButton             */
	XtSetArg(tiarg[tin], XtNlabel, cpystr(typename)); tin++;
	XtSetValues(tis->typemenu, tiarg, tin); tin = 0;

	/* Set Popup on entry for type */
	XtSetArg(tiarg[tin], XtNpopupOnEntry, w); tin++;
	XtSetValues(menu, tiarg, tin); tin = 0;

	/* Set TIStructure->type               */
	fs_give(&tis->typename);
	tis->typename = cpystr(typename);
	tis->type = getTypeFromName(typename);

	/* Set SubtypeMenu to appropriate menu */
	XtSetArg(tiarg[tin], XtNmenuName, cpystr(tis->typename)); tin++;
	XtSetArg(tiarg[tin], XtNlabel, cpystr("")); tin++;
	XtSetValues(tis->subtypemenu, tiarg, tin); tin = 0;

	/* NULL out TIStructure->subtype */
	if(tis->subtype)
	{
	    fs_give(&tis->subtype);
	    tis->subtype = NULL;
	}
    }
}

/* 
   Callback for selecting a menu entry on 
   the Type menu.
*/
static void setSubtype(w, tis, e)
     Widget w;
     TIStructure *tis;
     XEvent *e;
{
    Widget menu = NULL;
    char *typename = NULL;

    /* Get label from menu entry           */
    XtSetArg(tiarg[tin], XtNlabel, &typename); tin++;
    XtGetValues(w, tiarg, tin); tin = 0;

    if(!tis->subtype 
       || strcmp(tis->subtype, typename))
    {
	/* Set label of MenuButton             */
	XtSetArg(tiarg[tin], XtNlabel, cpystr(typename)); tin++;
	XtSetValues(tis->subtypemenu, tiarg, tin); tin = 0;

	menu = XtNameToWidget(tis->subtypemenu, tis->typename);

	/* Set Popup on entry                  */
	XtSetArg(tiarg[tin], XtNpopupOnEntry, w); tin++;
	XtSetValues(menu, tiarg, tin); tin = 0;

	/* Set TIStructure->subtype            */
	fs_give(&tis->subtype);
	tis->subtype = cpystr(typename);
    }
}

/*
   Returns the integer representing the 
   type named by 'name'
*/
static int getTypeFromName(name)
     char *name;
{
    int retval = 0;

    while((strcasecmp(name, tis_typeNames[retval])) 
	  && (tis_typeNames[retval] != NULL))
	retval++;
    
    return((tis_typeNames[retval] == NULL) ? 0 : retval);
}

/* 
   Creates the menus of subtypes for the various types
*/
static void initMenus(tis)
     TIStructure *tis;
{
    int i = 0;

    i = sizeof(subtypeMenus)/sizeof(ButtonStruct*);
    tis->subMenus = (Widget*)malloc(i*sizeof(Widget));
    while(i--)
    {
	if(subtypeMenus[i])
	    tis->subMenus[i] = makeMenu(tis->subtypemenu, 
					tis_typeNames[i], 
					tis, 
					subtypeMenus[i]);
	else
	    tis->subMenus[i] = NULL;
    }	    
}
